/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/crgc.c                  */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Sun Sep 13 11:58:32 1998                          */
/*    Last change :  Fri Jul 28 17:26:43 2006 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Rgc runtime (mostly port handling).                              */
/*=====================================================================*/
#include <stdio.h>
#include <errno.h> 
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#ifndef _MSC_VER
#   include <dirent.h>
#   include <sys/file.h>
#   include <sys/time.h>
#else
#   include <io.h>
#   include <windows.h>
#endif
#if( !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   ifdef HAVE_TERMIO
#      include <termio.h>
#   endif
#endif
#if !defined( sony_news ) && \
    !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) && \
    !defined( _MSC_VER )
#   include <unistd.h>
#endif
#include <bigloo.h>
#if( defined( sony_news ) || (defined( NeXT ) && defined( mc68000 )) )
#   include <ctype.h>
#endif
#if POSIX_FILE_OPS
#   include <unistd.h>
#endif

/*---------------------------------------------------------------------*/
/*    isascii                                                          */
/*---------------------------------------------------------------------*/
#if( !defined( isascii ) )
#   define isascii( c ) (!((c) & ~0177))
#endif

#define RGC_DEBUG
#undef RGC_DEBUG

/*---------------------------------------------------------------------*/
/*    C importations                                                   */
/*---------------------------------------------------------------------*/
extern obj_t bigloo_case_sensitive;
extern obj_t string_to_keyword( char * );

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    rgc_enlarge_buffer ...                                           */
/*    -------------------------------------------------------------    */
/*    This function augments the size of a port's buffer. An error is  */
/*    raised if there is not enough room for the allocation.           */
/*    The size must be given as parameter.                             */
/*---------------------------------------------------------------------*/
static void
rgc_enlarge_buffer( obj_t port, long n ) {
   long bufsize = INPUT_PORT( port ).bufsiz;
   long bufpos = INPUT_PORT( port ).bufpos;

   if( n <= bufsize ) return;

   // FIXME where does 2 come from?
   if( bufsize == 2 ) {
      C_SYSTEM_FAILURE( BGL_IO_READ_ERROR,
			"read",
			"Can't enlarge buffer for non bufferized port (see the user manual for details)",
			port );
   } else {
#if defined( RGC_DEBUG )
      printf( "***** rgc_enlarge_buffer: bufsize: %d new: %d\n", bufsize, n );
#endif
      if( !RGC_BUFFER( port ) )
	 C_SYSTEM_FAILURE( BGL_IO_READ_ERROR,
			   "read",
			   "Can't enlarge buffer",
			   port );

      RGC_BUFFER( port ) =
	 (unsigned char *)GC_REALLOC( (obj_t)RGC_BUFFER( port ), n );
      INPUT_PORT( port ).bufsiz = n;
   }
}
  
/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    rgc_double_buffer ...                                            */
/*    -------------------------------------------------------------    */
/*    This function doubles the size of a port's buffer. An error is   */
/*    raised if there is not enough room for the allocation.           */
/*---------------------------------------------------------------------*/
static void
rgc_double_buffer( obj_t port ) {
   long bufsize = INPUT_PORT( port ).bufsiz;
   rgc_enlarge_buffer( port, bufsize * 2 );
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    shift_buffer ...                                                 */
/*---------------------------------------------------------------------*/
static void
shift_buffer( obj_t port ) {
   long bufsize = INPUT_PORT( port ).bufsiz;
   long bufpos = INPUT_PORT( port ).bufpos;
   long matchstart = INPUT_PORT( port ).matchstart;
   unsigned char *buffer = RGC_BUFFER( port );
   long movesize = bufpos - matchstart;

   assert( bufpos > 0 );

   /* we shift the buffer left and we fill the buffer */
   memmove( (char *)&buffer[ 0 ], (char *)&buffer[ matchstart ], movesize );

   INPUT_PORT( port ).bufpos    -= matchstart;
   INPUT_PORT( port ).matchstop -= matchstart;
   INPUT_PORT( port ).forward   -= matchstart;
   INPUT_PORT( port ).lastchar   = RGC_BUFFER( port )[ matchstart - 1 ];
   INPUT_PORT( port ).matchstart = 0;
}
 
/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_size_fill_buffer ...                                         */
/*---------------------------------------------------------------------*/
static bool_t
rgc_size_fill_buffer( obj_t port, char *buf, int bufpos, int size ) {
   long r;
   int fb = INPUT_PORT( port ).fillbarrier;
   
#if defined( RGC_DEBUG )
   assert( bufpos >= 1 );
   assert( (bufpos + size) == INPUT_PORT( port ).bufsiz );
   printf( "~ ~ ~ rgc_size_fill_buffer (%p) ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~\n", port );

   printf( "rgc_size_fill_file_buffer: bufpos=%d  size=%d\n", bufpos, size );
   assert( size > 0 );
#endif

   if( fb == 0 ) {
#if defined( RGC_DEBUG )
      printf( "fb == 0\n" );
      printf( "~ ~ ~ (%p) ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~\n", port );
#endif
      return 0;
   }
   
   /* we start reading at BUFPOS - 1 because we have */
   /* to remove the '\0' sentinel that ends the buffer */
   if( (fb > 0) && (size > fb) ) size = fb;

   r = INPUT_PORT( port ).sysread( &buf[ bufpos - 1 ], 1, size, port );

#ifdef POSIX_FILE_OPS
   if( r < 0 ) {
      C_SYSTEM_FAILURE( BGL_IO_READ_ERROR, "read", strerror( errno ), port );
   }

   if( r == 0 )
      INPUT_PORT( port ).eof = 1;
#else
   if( ferror( (FILE *)INPUT_PORT( port ).file ) ) {
      C_SYSTEM_FAILURE( BGL_IO_READ_ERROR, "read", strerror( errno ), port );
   }

   if( INPUT_PORT( port ).syseof( port )
      INPUT_PORT( port ).eof = 1;
#endif
   
   buf[ bufpos - 1 + r ] = 0;
#if defined( RGC_DEBUG )
   printf( "rgc_size_fill_file_buffer, sysread r=%d size=%d\n", r, size );
   printf( "sysread [%s]\n",  &buf[ bufpos - 1 ] );
#endif

   if( fb > 0 ) {
      INPUT_PORT( port ).fillbarrier = (fb - r);
   }
   
   bufpos += r;

   INPUT_PORT( port ).bufpos = bufpos;

   assert( INPUT_PORT( port ).bufpos <= INPUT_PORT( port ).bufsiz );

#if defined( RGC_DEBUG )
   printf( "FIN de fill: size=%d  asize=%d  forward=%d  mstart=%d  mstop=%d\n",
	   INPUT_PORT( port ).bufsiz,
	   INPUT_PORT( port ).bufpos,
	   INPUT_PORT( port ).forward, 
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
   printf( "buffer: [%s]\n", buf );
#endif

   if( bufpos > 0 ) {
      buf[ bufpos - 1 ] = '\0';

#if defined( RGC_DEBUG )
      printf( "return 1\n" );
      printf( "~ ~ ~ (%p) ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~\n", port );
#endif
      return 1;
   } else {
#if defined( RGC_DEBUG )
      printf( "return 0\n" );
      printf( "~ ~ ~ (%p) ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~\n", port );
#endif
      return 0;
   }
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_fill_buffer ...                                              */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
bool_t
rgc_fill_buffer( obj_t port ) {
   long bufsize = INPUT_PORT( port ).bufsiz;
   long bufpos = INPUT_PORT( port ).bufpos;
   long matchstart = INPUT_PORT( port ).matchstart;
   unsigned char *buf = RGC_BUFFER( port );

#if defined( RGC_DEBUG )
   printf( "rgc_fill_buffer: %s bufsize=%d  bufpos=%d  forward=%d  mstart=%d  mstop=%d\n",
	   BSTRING_TO_STRING( INPUT_PORT( port ).name ),
	   bufsize, bufpos, INPUT_PORT( port ).forward,
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
   printf( "~~~~~ rgc_fill_buffer (%p) ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n", port );
   printf( "eof=%d  mstart=%d  mstop=%d bufpos=%d\n",
	   INPUT_PORT( port ).eof,
	   INPUT_PORT( port ).matchstart,
	   INPUT_PORT( port ).matchstop,
	   INPUT_PORT( port ).bufpos );
#endif
   /* In every case, forward has to be unwinded */
   /* because forward has reached the sentinel  */
   INPUT_PORT( port ).forward--;

   /* an input port that has seen its eof       */
   /* cannot be filled anymore                  */
   if( INPUT_PORT( port ).eof ) {
      return 0;
   } else {
      if( bufpos < bufsize )
	 /* the buffer is not full, we fill it */
	 return rgc_size_fill_buffer( port, buf, bufpos, bufsize - bufpos );
      else {
	 if( matchstart > 0 ) {
	    shift_buffer( port );
#if defined( RGC_DEBUG )
      printf( "buffer shifted [%s]\n", buf );
      printf( ". . . (%p) . . . . . . . . . . . . . . . . . . . . . . . .\n", port );
#endif
	    bufpos = INPUT_PORT( port ).bufpos;
	    
	    return rgc_size_fill_buffer( port, buf, bufpos, bufsize - bufpos );
	 } else {
	    /* the current token is too large for the buffer */
	    /* we have to enlarge it.                        */
	    /* Note: see rgc_size_fil_buffer for other       */
	    /* enlarge_buffer                                */
	    rgc_double_buffer( port );

	    /* undo forward--                                */
	    INPUT_PORT( port ).forward++;

	    return rgc_fill_buffer( port );
	 }
      }
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_substring ...                                         */
/*    -------------------------------------------------------------    */
/*    This function makes no bound checks because these tests have     */
/*    already been performed in the grammar.                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_substring( obj_t ip, long offset, long end ) {
   long start = INPUT_PORT( ip ).matchstart;
   long len   = end - offset;

#if defined( RGC_DEBUG )
   printf( "buffer-substring: start=%d  stop=%d  forward=%d  bufpos=%d\n",
	   start, INPUT_PORT( ip ).matchstop,
	   INPUT_PORT( ip ).forward, INPUT_PORT( ip ).bufpos );
#endif

   return string_to_bstring_len( (char *)&RGC_BUFFER( ip )[ start + offset ],
				 len );
}

/*---------------------------------------------------------------------*/
/*    CHEAT_BUFFER_AT                                                  */
/*---------------------------------------------------------------------*/
#define CHEAT_BUFFER_AT( s ) \
   long stop  = s; \
   char bck; \
   bck = RGC_BUFFER( ip )[ stop ]; \
   RGC_BUFFER( ip )[ stop ] = '\0';

/*---------------------------------------------------------------------*/
/*    CHEAT_BUFFER                                                     */
/*---------------------------------------------------------------------*/
#define CHEAT_BUFFER() \
   CHEAT_BUFFER_AT( INPUT_PORT( ip ).matchstop )

/*---------------------------------------------------------------------*/
/*    RESTORE_BUFFER                                                   */
/*---------------------------------------------------------------------*/
#define RESTORE_BUFFER() \
   RGC_BUFFER( ip )[ stop ] = bck;

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    rgc_buffer_fixnum ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
long
rgc_buffer_fixnum( obj_t ip ) {
   long res;
   
   CHEAT_BUFFER();
   
   res = atol( (const char *)&RGC_BUFFER(ip)[ INPUT_PORT( ip ).matchstart ] );
   
   RESTORE_BUFFER();
   
   return res;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_integer ...                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_integer( obj_t ip ) {
   long stop = INPUT_PORT( ip ).matchstop;
   long start = INPUT_PORT( ip ).matchstart;
   long res = 0;
   int sign = 1;

   /* the sign */
   if( RGC_BUFFER(ip)[ start ] == '+' ) {
      start++;
   } else {
      if( RGC_BUFFER(ip)[ start ] == '-' ) {
	 start++;
	 sign = -1;
      }
   }

   /* skip the 0 padding */
   while( (start < stop) && RGC_BUFFER(ip)[ start ] == '0' )
      start++;

   /* the real number */
   while( start < stop ) {
      long res2;
      res2 = res * 10 + (RGC_BUFFER(ip)[ start ] - '0');

      if( res2 < res )
	 goto llong;

      res = res2;
      start++;
   }

#define BGL_MAX_FIXNUM_BITS ((((long)1 << PTR_ALIGNMENT) << 3) - PTR_ALIGNMENT - 1)
#define BGL_MAX_FIXNUM (((long)1 << BGL_MAX_FIXNUM_BITS) - 1)
   if( res > BGL_MAX_FIXNUM )
      return LLONG_TO_BLLONG( (BGL_LONGLONG_T)(sign > 0 ? res : -res) );
   else
      return BINT( sign > 0 ? res : -res );
      
#undef BGL_MAX_FIXNUM
#undef BGL_MAX_FIXNUM_BITS   
llong:
   {
      BGL_LONGLONG_T lres = (BGL_LONGLONG_T)res;

      while( start < stop ) {
	 lres = lres * 10 + (RGC_BUFFER(ip)[ start ] - '0');
	 start++;
      }

      return LLONG_TO_BLLONG( sign > 0 ? lres : -lres );
   }
}

/*---------------------------------------------------------------------*/
/*    double                                                           */
/*    rgc_buffer_flonum ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
double
rgc_buffer_flonum( obj_t ip ) {
   double res;
   
   CHEAT_BUFFER();
  
   res = strtod( (const char *)&RGC_BUFFER(ip)[ INPUT_PORT(ip).matchstart ], 0 );
   
   RESTORE_BUFFER();
   
   return res;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_symbol ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_symbol( obj_t ip ) {
   unsigned char *aux;
   obj_t sym;
   long start = INPUT_PORT( ip ).matchstart;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   sym = string_to_symbol( (char *)aux );

   RESTORE_BUFFER();
   
   return sym;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_subsymbol ...                                         */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_subsymbol( obj_t ip, long offset, long end ) {
   unsigned char *aux;
   obj_t sym;
   long start = INPUT_PORT( ip ).matchstart;
   long len = end - offset;
   
   CHEAT_BUFFER_AT( start + len );
   
   aux = &RGC_BUFFER( ip )[ start + offset ];
   
   sym = string_to_symbol( (char *)aux );

   RESTORE_BUFFER();
   
   return sym;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_upcase_symbol ...                                     */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_upcase_symbol( obj_t ip ) {
   unsigned char *aux;
   obj_t sym;
   long start = INPUT_PORT( ip ).matchstart;
   unsigned char *walk;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   for( walk = aux; *walk; walk++ )
      if( isascii( *walk ) )
	 *walk = toupper( *walk );

   sym = string_to_symbol( (char *)aux );

   RESTORE_BUFFER();
   
   return sym;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_downcase_symbol ...                                   */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_downcase_symbol( obj_t ip ) {
   unsigned char *aux;
   obj_t sym;
   long start = INPUT_PORT( ip ).matchstart;
   unsigned char *walk;
   
   CHEAT_BUFFER();
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   for( walk = aux; *walk; walk++ )
      if( isascii( *walk ) )
	 *walk = tolower( *walk );

   sym = string_to_symbol( (char *)aux );

   RESTORE_BUFFER();
   
   return sym;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_keyword ...                                           */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_keyword( obj_t ip ) {
   unsigned char *aux;
   obj_t key;
   long start = INPUT_PORT( ip ).matchstart;
   long cheat;
   
   aux = &RGC_BUFFER( ip )[ start ];

   if( *aux == ':' ) {
      aux++;
      cheat = INPUT_PORT( ip ).matchstop;
   } else {
      cheat = INPUT_PORT( ip ).matchstop - 1;
   }

   {
      CHEAT_BUFFER_AT( cheat );
      key = string_to_keyword( (char *)aux );
      RESTORE_BUFFER();
   }

   return key;
}
 
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_downcase_keyword ...                                  */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_downcase_keyword( obj_t ip ) {
   unsigned char *aux;
   obj_t key;
   long start = INPUT_PORT( ip ).matchstart;
   long cheat;
   unsigned char *walk;
   
   aux = &RGC_BUFFER( ip )[ start ];

   if( *aux == ':' ) {
      aux++;
      cheat = INPUT_PORT( ip ).matchstop;
   } else {
      cheat = INPUT_PORT( ip ).matchstop -1;
   }

   {
      CHEAT_BUFFER_AT( cheat );
      
      for( walk = aux; *walk; walk++ )
	 if( isascii( *walk ) ) *walk = tolower( *walk );

      key = string_to_keyword( (char *)aux );
      
      RESTORE_BUFFER();
   }

   return key;
}
 
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_upcase_keyword ...                                    */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
rgc_buffer_upcase_keyword( obj_t ip ) {
   unsigned char *aux;
   obj_t key;
   long start = INPUT_PORT( ip ).matchstart;
   long cheat;
   unsigned char *walk;
   
   aux = &RGC_BUFFER( ip )[ start ];

   if( *aux == ':' ) {
      aux++;
      cheat = INPUT_PORT( ip ).matchstop;
   } else {
      cheat = INPUT_PORT( ip ).matchstop -1;
   }

   {
      CHEAT_BUFFER_AT( cheat );
	 
      for( walk = aux; *walk; walk++ )
	 if( isascii( *walk ) ) *walk = toupper( *walk );

      key = string_to_keyword( (char *)aux );

      RESTORE_BUFFER();
   }

   return key;
}
 
/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    rgc_buffer_unget_char ...                                        */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
int
rgc_buffer_unget_char( obj_t ip, int c ) {
   INPUT_PORT( ip ).filepos--;
   
   if( INPUT_PORT( ip ).matchstop > 0 ) {
      INPUT_PORT( ip ).matchstop--;
   } else {
      RGC_BUFFER( ip )[ 0 ] = c;
      if( INPUT_PORT( ip ).bufpos == 0 ) {
	 INPUT_PORT( ip ).bufpos = 1;
	 RGC_BUFFER( ip )[ 1 ] = '\0';
      }
   }

   return c;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_bol_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the matchstart position located at the beginning of a line?   */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
bool_t
rgc_buffer_bol_p( obj_t ip ) {
#if( defined( RGC_DEBUG ) )
   printf( "RGC_BUFFER_BOL_P: mstart: %d  [mstart]: %d  lastchar: %d  --> %d\n",
	   INPUT_PORT( ip ).matchstart, 
	   RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart - 1 ],
	   INPUT_PORT( ip ).lastchar,
	   INPUT_PORT( ip ).lastchar == '\n' );
#endif
   
   if( INPUT_PORT( ip ).matchstart > 0 )
      return RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart - 1 ] == '\n';
   else
      return INPUT_PORT( ip ).lastchar == '\n';
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_eol_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Does the buffer contain, at its first non match position, a `\n' */
/*    character?                                                       */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
bool_t
rgc_buffer_eol_p( obj_t ip ) {
   int c = RGC_BUFFER_GET_CHAR( ip );
   
#if( defined( RGC_DEBUG ) )   
   long f = INPUT_PORT( ip ).forward;
#endif
   
#if( defined( RGC_DEBUG ) )   
   printf( "RGC_BUFFER_EOL_P: forward: %d %d", f, c );
#endif
   
   if( !c ) {
      if( !RGC_BUFFER_EMPTY( ip ) ) {
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   not empty --> 0" );
#endif
	 return 0;
      }

      if( PORT( ip ).kindof == KINDOF_CONSOLE ) {
#if( defined( RGC_DEBUG ) )   
	 puts( "  kindof == CONSOLE --> 1" );
#endif
	 return 1;
      }
      if( rgc_fill_buffer( ip ) )
	 return rgc_buffer_eol_p( ip );
      else {
#if( defined( RGC_DEBUG ) )   
	 puts( "   not rgc_fill_buffer --> 0" );
#endif
	 return 0;
      }
   } else {
      INPUT_PORT( ip ).forward--;
#if( defined( RGC_DEBUG ) )   
      printf( "   --> %d\n", c == '\n' );
#endif
      return c == '\n';
   }
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_bof_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the match position at the beginning of the file?              */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
bool_t
rgc_buffer_bof_p( obj_t ip ) {
   return INPUT_PORT( ip ).filepos == 0;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_eof_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the input port at its end-of-file position?                   */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
bool_t
rgc_buffer_eof_p( obj_t ip ) {
   int c = RGC_BUFFER_GET_CHAR( ip );

#if( defined( RGC_DEBUG ) )   
   long f = INPUT_PORT( ip ).forward;
#endif
   
#if( defined( RGC_DEBUG ) )   
   printf( "RGC_BUFFER_EOF_P: forward: %d %d", f, c );
#endif
   
   if( !c ) {
      if( !RGC_BUFFER_EMPTY( ip ) ) {
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   not empty --> 0" );
#endif
	 return 0;
      }
      else
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   --> 1" );
#endif
	 return 1;
   } else {
      INPUT_PORT( ip ).forward--;
#if( defined( RGC_DEBUG ) )   
      puts( "   not empty --> 0" );
#endif
      return 0;
   }
}

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    file_charready ...                                               */
/*---------------------------------------------------------------------*/
static int
file_charready( FILE *f ) {
#ifndef _MSC_VER
#   if( BGL_HAVE_SELECT )
       fd_set readfds;
       struct timeval timeout;
       int fno = fileno( f ) + 1;

       FD_ZERO( &readfds );
       FD_SET( fileno( f ), &readfds );
       timeout.tv_sec = 0; timeout.tv_usec = 0;

       return (select( fno, &readfds, NULL, NULL, &timeout ) > 0);
#   else
       return 0;
#   endif
#else
    HANDLE hFile = (HANDLE)_get_osfhandle( _fileno( f ) );

    return ((WaitForSingleObject( hFile, 0) == WAIT_OBJECT_0) ? 1 : 0);
#endif
}

/*---------------------------------------------------------------------*/
/*    boot_t                                                           */
/*    bgl_rgc_charready ...                                            */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF bool_t
bgl_rgc_charready( obj_t port ) {
   switch( (long)PORT( port ).kindof ) {
      case (long)KINDOF_CLOSED:
	 return 0;
	 
      case (long)KINDOF_STRING:
	 return ((INPUT_PORT( port ).forward+1) < INPUT_PORT( port ).bufpos);
	 
      case (long)KINDOF_FILE:
	 return ((INPUT_PORT( port ).forward+1) < INPUT_PORT( port ).bufpos)
	    || (!feof( (FILE *)PORT( port ).stream )
		&& !INPUT_PORT( port ).eof);
	 
      case (long)KINDOF_PROCPIPE:
      case (long)KINDOF_PIPE:
      case (long)KINDOF_CONSOLE:
      case (long)KINDOF_SOCKET:
	 return ((INPUT_PORT( port ).forward+1) < INPUT_PORT( port ).bufpos)
	    || file_charready( PORT( port ).stream );
	 
      case (long)KINDOF_PROCEDURE:
      case (long)KINDOF_GZIP:
	 /* to know if a char is available we only could call the procedure */
	 /* this could block, so we just return true                        */
	 return 1;

      default:
	 return 0;
   }
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    rgc_blit_string ...                                              */
/*---------------------------------------------------------------------*/
int
rgc_blit_string( obj_t p, obj_t bs, int o, int l ) {
   char *s = BSTRING_TO_STRING( bs );
   int bsz = INPUT_PORT( p ).bufsiz;

   RGC_START_MATCH( p );
   
   if( bsz == 2 ) {
      /* we are reading from a non bufferized port, we have to read */
      /* each character at a time. */
      int i;

      for( i = 0; i < l; i++ ) {
	 char c;
	 RGC_START_MATCH( p );
	 if( !(c = RGC_BUFFER_GET_CHAR( p )) ) {
	    rgc_fill_buffer( p );
	    c = RGC_BUFFER_GET_CHAR( p );
	 }
	 RGC_STOP_MATCH( p );
	 s[ o + i ] = c;
      }
      s[ o + i ] = 0;

      return l;
   } else {
      int o0 = o;
      int bufl = INPUT_PORT( p ).bufpos - INPUT_PORT( p ).matchstart - 1;
      int ml = (l <= bufl ? l : bufl);

      if( ml > 0 ) {
	 memmove( &s[ o ],
		  &RGC_BUFFER( p )[ INPUT_PORT( p ).matchstart ],
		  ml );
	 
	 INPUT_PORT( p ).forward = INPUT_PORT( p ).matchstart + ml;
	 RGC_STOP_MATCH( p );
	 RGC_SET_FILEPOS( p );
      }

      if( ml == l ) {
	 return l;
      } else {
	 /* adjust the destination string position and length */
	 o += ml;
	 l -= ml;
	 
	 /* reset the buffer cursors */
	 RGC_START_MATCH( p );
      
	 while( (l > 0) && !(INPUT_PORT( p ).eof) ) {
	    int m = (bsz <= l ? bsz : l);
	    int r; 
	    
	    rgc_size_fill_buffer( p, &s[ o ], 1, m );
	    r = INPUT_PORT( p ).bufpos - 1;

	    INPUT_PORT( p ).filepos += r;
	    
	    l -= r;
	    o += r;

/* 	    {* we have failed to read m characters *}                  */
/* 	    if( r < m ) break;                                         */
	    /* MS 28 march 2006: we cannot read more */
	    if( r <= 0 ) break;
	 }

	 /* the following settings replace the call to RGC_STOP_MATCH */
	 INPUT_PORT( p ).matchstart = 0;
	 INPUT_PORT( p ).matchstop = 0;
	 INPUT_PORT( p ).bufpos = 1;
	 INPUT_PORT( p ).buffer[ 0 ] = '\0';
	 INPUT_PORT( p ).lastchar = '\n';

	 return o - o0;
      }
   }
}
