//
// Author: 
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio HB
// 

using Rss;
using Atom.Core;
using Atom.Core.Collections;
using System;
using System.IO;
using System.Net;

//using System.Web;

namespace Imendio.Blam {
    public class FeedUpdater {
	public static bool Update (Channel channel)
	{
	    //Console.WriteLine("Updating: " + channel.Url + " ...");

	    channel.LastRefreshed = DateTime.Now;

	    bool updated = false;
	    WebResponse res = null;

	    if (channel.Type == "") {
	        try {
		    res = GetChannelData(channel);

		    if (res == null) {
		        // Unable to connect, or feed not changed.
		        return false;
		    }

		    RssReader reader = new RssReader(res.GetResponseStream());
		    RssChannel rssChannel = null;
		    RssElement e;

		    do {
		    	e = reader.Read();
		    	if (e is RssRedirect){
		    		channel.Url = ((RssRedirect)e).Location.ToString();
		    		return Update(channel);
		    	} else if (e is RssChannel) {
			    	rssChannel = (RssChannel)e;
				}
		    } while (e != null);

		    if (rssChannel != null) {
		        reader.Close();
		        channel.Type = "RSS";
		    } else {
		        reader.Close();
		        res.Close();

		        res = GetChannelData(channel);

		        if (res == null) {
		            return false;
		        }

		        try {
			    Stream stream = res.GetResponseStream ();
			    AtomFeed feed = AtomFeed.Load (stream);

			    if (feed != null) {
		                if (feed.Version == "0.3") {
		                    channel.Type = "Atom";
		                }
			    } 
			} catch (Exception ex) {
			    Console.WriteLine ("Exception when parsing: {0}", ex.Message);
		        }
		    }

		    res.Close();
	        } catch (Exception) {
	        }

	        /* The feed type is still unknown.  The feed is an unsupported 
	         * type or possibly malformed.
	         */
	        if (channel.Type == "") {
		    return false;
	        }
	    }

	    try {
	        res = GetChannelData(channel);

	        if (res == null) {
		    return false;
	        }

	        if (channel.Type == "RSS") {
		    updated = UpdateRssChannel(channel, res.GetResponseStream());
	        }
	        else if (channel.Type == "Atom") {
		    updated = UpdateAtomChannel(channel, res.GetResponseStream());
	        }

	        if (res.Headers.Get("Last-Modified") != null) {
		    channel.LastModified = res.Headers.Get("Last-Modified"); 
	        }
	        if (res.Headers.Get("ETag") != null) {
		    channel.ETag = res.Headers.Get("ETag");
	        }

	        res.Close ();
	    } catch (Exception) {
	    }

	    return updated;
	}

	public static WebResponse GetChannelData (Channel channel)
	{
	    HttpWebRequest req = null;
	    WebResponse res = null;

	    try {
	        req = (HttpWebRequest) WebRequest.Create(channel.Url);

	        if (channel.LastModified != "") {
	            req.IfModifiedSince = DateTime.Parse(channel.LastModified);
	        }

	        if (channel.ETag != "") {
	            req.Headers.Add("If-None-Match", channel.ETag);
	        }

	        req.Timeout = 20000;

	        WebProxy proxy = Proxy.GetProxy();
	        if (proxy != null) {
	            req.Proxy = proxy;
	        }

	        try {
	            res = req.GetResponse();
	        } catch (WebException wE) {
	            switch (wE.Status) {
	            case WebExceptionStatus.ProtocolError:
		        if (((HttpWebResponse)wE.Response).StatusCode == 
			    HttpStatusCode.NotModified) {
		            //Console.WriteLine("No changes to feed.");
		        }
		        break;
	            case WebExceptionStatus.Timeout:
		        //Console.WriteLine("Timed out");
		        //Console.WriteLine("Exception: " + wE.ToString());
		        break;
	            default:
		        //Console.WriteLine("Exception: " + wE.ToString());
		        break;
	            }
	        }
	    } catch (Exception) {
	    }

	    return res;
	}

	private static bool UpdateRssChannel (Channel channel, Stream stream)
	{
	    bool channelUpdated = false;

	    RssReader reader = new RssReader (stream);
	    RssChannel rssChannel = null;
	    RssElement e;

	    // FIXME: Handle multiple channels per feed?
	    do {
	        e = reader.Read();
	        if (e is RssChannel) {
	            rssChannel = (RssChannel)e;
	        }
	    } while (e != null);

	    reader.Close ();

	    if (rssChannel != null) {
	        /* Check if this channel is updated */
	        channel.StartRefresh ();

	        /*if (channel.Image.Equals ("") &&
                      rssChannel.Image != null) {
                      channel.Image = rssChannel.Image.Url.ToString ();
	        }*/

	        if (channel.Name.Equals ("") &&
		    rssChannel.Title != null) {
	            channel.Name = HtmlUtils.StripHtml(rssChannel.Title);
	            channelUpdated = true;
	        }

	        foreach (RssItem item in rssChannel.Items) {
	            string id;
	            bool   itemUpdated;

	            id = GenerateItemId(item);
	            if (item.Title == null || item.Title.Equals("")) {
		        if (!item.PubDate.Equals (DateTime.MinValue)) {
		            item.Title = item.PubDate.ToString("d MMM yyyy");
		        } else {
		            item.Title = channel.Name;
		        }
	            }

	            // Will add, update or do nothing.
	            itemUpdated = channel.UpdateItem (id, item);
	            if (itemUpdated) {
		        channelUpdated = true;
	            }
	        }
	        channel.FinishRefresh ();
	    }

	    return channelUpdated;
	}

	private static bool UpdateAtomChannel(Channel channel, Stream stream)
	{
	    bool channelUpdated = false;

	    try {
	        AtomFeed feed = AtomFeed.Load(stream);

	        if (feed != null) {
	            channel.StartRefresh();

	            if (channel.Name == "" && 
		        feed.Title.Content != null) {
		        channel.Name = HtmlUtils.StripHtml(feed.Title.Content.Trim());
		        channelUpdated = true;
	            }

	            foreach (AtomEntry entry in feed.Entries) {
		        string id;
		        bool   entryUpdated;

		        id = GenerateItemId(entry);
		        if (entry.Title.Content == "") {
		            if (!entry.Modified.DateTime.Equals(DateTime.MinValue)) {
		                entry.Title.Content = 
		                    entry.Modified.DateTime.ToString("d MMM yyyy");
		            } else {
		                entry.Title.Content = channel.Name;
		            }
		        }

		        entryUpdated = channel.UpdateItem (id, entry);
		        if (entryUpdated) {
		            channelUpdated = true;
		        }
	            }

	            channel.FinishRefresh();
	        }
	    } catch (Exception) {
	    }

	  return channelUpdated;
	}

	public static string GenerateItemId (RssItem item)
	{
	    if (item.Guid != null) {
		return item.Guid.Name;
	    } 
	    else if (item.Link != RssDefault.Uri) {
		return item.Link.ToString ();
	    } else {
		return item.Title;
	    }
	}

	public static string GenerateItemId (AtomEntry entry)
	{
	    return entry.Id.ToString();
	}
    }
}
