/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2008  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <gtk/gtktreemodel.h>
#include <gtk/gtktreesortable.h>
#include <gtk/gtkstock.h>

#include "device-store.h"

#define BLUETOOTH_DEVICE_STORE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), \
			BLUETOOTH_TYPE_DEVICE_STORE, BluetoothDeviceStorePrivate))

typedef struct _BluetoothDeviceStorePrivate BluetoothDeviceStorePrivate;

struct _BluetoothDeviceStorePrivate {
	GList *list;
	gint stamp;
};

struct device_data {
	gchar *address;
	gchar *name;
	guint32 class;
	gint16 rssi;
	gboolean valid;
};

static GtkTreeModelFlags bluetooth_device_store_get_flags(GtkTreeModel *model)
{
	g_return_val_if_fail(BLUETOOTH_IS_DEVICE_STORE(model), 0);

	return (GTK_TREE_MODEL_LIST_ONLY | GTK_TREE_MODEL_ITERS_PERSIST);
}

static gint bluetooth_device_store_get_n_columns(GtkTreeModel *model)
{
	g_return_val_if_fail(BLUETOOTH_IS_DEVICE_STORE(model), 0);

        return 6;
}

static GType bluetooth_device_store_get_column_type(GtkTreeModel *model,
								gint index)
{
	g_return_val_if_fail(BLUETOOTH_IS_DEVICE_STORE(model), G_TYPE_INVALID);
	g_return_val_if_fail(index >= 0 && index < 7, G_TYPE_INVALID);

	if (index == 2 || index == 3)
		return G_TYPE_INT;

	if (index == 5)
		return G_TYPE_BOOLEAN;

	return G_TYPE_STRING;
}

static gboolean bluetooth_device_store_get_iter(GtkTreeModel *model,
					GtkTreeIter *iter, GtkTreePath *path)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(model);
	gint *indices, depth;
	guint num;

	indices = gtk_tree_path_get_indices(path);
	depth   = gtk_tree_path_get_depth(path);

	g_return_val_if_fail(depth == 1, FALSE);

	num = g_list_length(priv->list);

	if (indices[0] < 0 || indices[0] >= num)
		return FALSE;

	iter->stamp = priv->stamp;
	iter->user_data = g_list_nth_data(priv->list, indices[0]);

	return TRUE;
}

static GtkTreePath *bluetooth_device_store_get_path(GtkTreeModel *model,
							GtkTreeIter *iter)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(model);
	GtkTreePath *path;

	g_return_val_if_fail(BLUETOOTH_IS_DEVICE_STORE(model), NULL);
	g_return_val_if_fail(iter != NULL, NULL);

	path = gtk_tree_path_new();
	gtk_tree_path_append_index(path,
				g_list_index(priv->list, iter->user_data));

	return path;
}

static gchar *class_to_icon_name(guint32 class)
{
	switch ((class & 0x1f00) >> 8) {
	case 0x01:
		return "computer";
	case 0x02:
		switch ((class & 0xfc) >> 2) {
		case 0x01:
		case 0x02:
		case 0x03:
			return "stock_cell-phone";
		case 0x04:
			return "modem";
		case 0x05:
			return "stock_landline-phone";
		default:
			return "bluetooth";
		}
	case 0x03:
		return "network-wireless";
	case 0x04:
		return "stock_headphones";
	case 0x05:
		switch ((class & 0xc0) >> 6) {
		case 0x01:
			return "input-keyboard";
		case 0x02:
			return "input-mouse";
		default:
			return "bluetooth";
		}
	case 0x06:
		switch ((class & 0xf0) >> 4) {
		case 0x02:
			return "camera-photo";
		case 0x08:
			return "printer";
		default:
			return "bluetooth";
		}
	default:
		return "bluetooth";
	}
}

static void bluetooth_device_store_get_value(GtkTreeModel *model,
			GtkTreeIter *iter, gint column, GValue *value)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(model);
	struct device_data *data;

	g_return_if_fail(BLUETOOTH_IS_DEVICE_STORE(model));
	g_return_if_fail(iter != NULL);
	g_return_if_fail(column < 7);

	data = g_list_find(priv->list, iter->user_data)->data;
	if (!data)
		return;

	switch (column) {
	case 1:
		g_value_init(value, G_TYPE_STRING);
		g_value_set_string(value, data->name ? data->name : data->address);
		break;
	case 2:
		g_value_init(value, G_TYPE_INT);
		g_value_set_int(value, data->class);
		break;
	case 3:
		g_value_init(value, G_TYPE_INT);
		g_value_set_int(value, data->rssi);
		break;
	case 4:
		g_value_init(value, G_TYPE_STRING);
		if (data->valid == TRUE)
			g_value_set_string(value, class_to_icon_name(data->class));
		break;
	case 5:
		g_value_init(value, G_TYPE_BOOLEAN);
		g_value_set_boolean(value, data->valid);
		break;
	default:
		break;
	}
}

static gboolean bluetooth_device_store_iter_next(GtkTreeModel *model,
							GtkTreeIter *iter)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(model);
	GList *list;

	g_return_val_if_fail(BLUETOOTH_IS_DEVICE_STORE(model), FALSE);

	if (iter == NULL || iter->user_data == NULL)
		return FALSE;

	list = g_list_find(priv->list, iter->user_data);

	if (g_list_next(list) == NULL)
		return FALSE;

	iter->stamp = priv->stamp;
	iter->user_data = g_list_next(list)->data;

	return TRUE;
}

static gboolean bluetooth_device_store_iter_children(GtkTreeModel *model,
					GtkTreeIter *iter, GtkTreeIter *parent)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(model);

	if (parent)
		return FALSE;

	if (g_list_length(priv->list) == 0)
		return FALSE;

	iter->stamp = priv->stamp;
	iter->user_data = g_list_nth_data(priv->list, 0);

	return TRUE;
}

static gboolean bluetooth_device_store_iter_has_child(GtkTreeModel *model,
							GtkTreeIter *iter)
{
	return FALSE;
}

static gint bluetooth_device_store_iter_n_children(GtkTreeModel *model,
							GtkTreeIter *iter)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(model);

	g_return_val_if_fail(BLUETOOTH_IS_DEVICE_STORE(model), -1);
	g_return_val_if_fail(iter == NULL || iter->user_data != NULL, -1);

	if (!iter)
		return g_list_length(priv->list);

	return 0;
}

static gboolean bluetooth_device_store_iter_nth_child(GtkTreeModel *model,
				GtkTreeIter *iter, GtkTreeIter *parent, gint n)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(model);

	g_return_val_if_fail(BLUETOOTH_IS_DEVICE_STORE(model), FALSE);

	if (parent)
		return FALSE;

	if (n >= g_list_length(priv->list))
		return FALSE;

	iter->stamp = priv->stamp;
	iter->user_data = g_list_nth_data(priv->list, n);

	return TRUE;
}

static gboolean bluetooth_device_store_iter_parent(GtkTreeModel *model,
					GtkTreeIter *iter, GtkTreeIter *child)
{
	return FALSE;
}

static void bluetooth_device_store_tree_model_init(GtkTreeModelIface *iface)
{
	iface->get_flags       = bluetooth_device_store_get_flags;
	iface->get_n_columns   = bluetooth_device_store_get_n_columns;
	iface->get_column_type = bluetooth_device_store_get_column_type;
	iface->get_iter        = bluetooth_device_store_get_iter;
	iface->get_path        = bluetooth_device_store_get_path;
	iface->get_value       = bluetooth_device_store_get_value;
	iface->iter_next       = bluetooth_device_store_iter_next;
	iface->iter_children   = bluetooth_device_store_iter_children;
	iface->iter_has_child  = bluetooth_device_store_iter_has_child;
	iface->iter_n_children = bluetooth_device_store_iter_n_children;
	iface->iter_nth_child  = bluetooth_device_store_iter_nth_child;
	iface->iter_parent     = bluetooth_device_store_iter_parent;
}

G_DEFINE_TYPE_WITH_CODE(BluetoothDeviceStore, bluetooth_device_store, G_TYPE_OBJECT, \
		G_IMPLEMENT_INTERFACE(GTK_TYPE_TREE_MODEL, \
				bluetooth_device_store_tree_model_init))

static void bluetooth_device_store_init(BluetoothDeviceStore *self)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(self);

	priv->list = NULL;

	priv->stamp = g_random_int();
}

static void bluetooth_device_store_finalize(GObject *object)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(object);

	g_list_free(priv->list);
}

static void bluetooth_device_store_class_init(BluetoothDeviceStoreClass *klass)
{
	g_type_class_add_private(klass, sizeof(BluetoothDeviceStorePrivate));

	G_OBJECT_CLASS(klass)->finalize = bluetooth_device_store_finalize;
}

BluetoothDeviceStore *bluetooth_device_store_new(void)
{
	return g_object_new(BLUETOOTH_TYPE_DEVICE_STORE, NULL);
}

static GList *device_find(GList *list, const gchar *address)
{
	gint do_compare(gconstpointer a, gconstpointer b)
	{
		const struct device_data *data = a;
		const char *address = b;

		return g_ascii_strcasecmp(data->address, address);
	}

	GList *link;
	gchar *key;

	key = g_strdup(address);

	link = g_list_find_custom(list, key, do_compare);

	g_free(key);

	return link;
}

static void device_add(GtkTreeModel *model, struct device_data *data,
						gint stamp, gboolean update)
{
	GtkTreeIter iter;
	GtkTreePath *path;

	iter.stamp = stamp;
	iter.user_data = data;

	path = bluetooth_device_store_get_path(model, &iter);

	if (update == TRUE)
		gtk_tree_model_row_changed(model, path, &iter);
	else
		gtk_tree_model_row_inserted(model, path, &iter);

	gtk_tree_path_free(path);
}

void bluetooth_device_store_add_device(BluetoothDeviceStore *self,
			const gchar *address, guint32 class, gint16 rssi)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(self);
	GList *list;
	struct device_data *data;

	list = device_find(priv->list, address);
	if (list) {
		data = list->data;

		data->class = class;
		data->rssi  = rssi;
		data->valid = TRUE;

		device_add(GTK_TREE_MODEL(self), list->data, priv->stamp, TRUE);

		return;
	}

	data = g_slice_new0(struct device_data);

	data->address = g_strdup(address);
	data->class = class;
	data->rssi  = rssi;
	data->valid = TRUE;

	priv->list = g_list_append(priv->list, data);

	device_add(GTK_TREE_MODEL(self), data, priv->stamp, FALSE);
}

void bluetooth_device_store_add_name(BluetoothDeviceStore *self,
				const gchar *address, const gchar *name)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(self);
	GList *list;
	struct device_data *data;

	list = device_find(priv->list, address);
	if (!list)
		return;

	data = list->data;
	data->name  = g_strdup(name);
	data->valid = TRUE;

	device_add(GTK_TREE_MODEL(self), list->data, priv->stamp, TRUE);
}

void bluetooth_device_store_del_device(BluetoothDeviceStore *self,
						const gchar *address)
{
	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(self);
	GList *list;
	struct device_data *data;

	list = device_find(priv->list, address);
	if (!list)
		return;

	data = list->data;
	data->rssi  = 0;
	data->valid = FALSE;

	device_add(GTK_TREE_MODEL(self), list->data, priv->stamp, TRUE);
}

void bluetooth_device_store_invalidate_devices(BluetoothDeviceStore *self)
{
	void make_invalid(gpointer data, gpointer user_data)
	{
		BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(user_data);

		((struct device_data *) data)->rssi  = 0;
		((struct device_data *) data)->valid = FALSE;

		device_add(GTK_TREE_MODEL(user_data), data, priv->stamp, TRUE);
	}

	BluetoothDeviceStorePrivate *priv = BLUETOOTH_DEVICE_STORE_GET_PRIVATE(self);

	g_list_foreach(priv->list, make_invalid, self);
}

#define fabs(x) ((x) < 0 ? -(x) : (x))

gint bluetooth_device_store_rssi_compare_func(GtkTreeModel *model,
			GtkTreeIter *a, GtkTreeIter *b, gpointer user_data)
{
	struct device_data *a_data = a->user_data;
	struct device_data *b_data = b->user_data;

	if (a_data->valid == FALSE && b_data->valid == TRUE)
		return 1;

	if (a_data->valid == TRUE && b_data->valid == FALSE)
		return -1;

	return fabs(a_data->rssi) - fabs(b_data->rssi);
}
