//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <revision.h>
#include <build.h>

#include <glibmm.h>
#include <glibmm/wrap_init.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <glib/gstdio.h>

#include <gtkmm.h>
#include <gtk/gtk.h>
#include <gdkmm/wrap_init.h>
#include <gtkmm/wrap_init.h>
#include <pangomm/wrap_init.h>

#include <dbus/dbus-gtype-specialized.h>

#ifdef HAVE_MOOD
#  include <moo/driver.hh>
#endif

#include <mcs/mcs.h>
#include <gst/gst.h>
#include <cstdlib>
#include <cstring>

#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#ifdef HAVE_SM
#  include "sm.hh"
#endif

#ifdef HAVE_STARTUP_NOTIFICATION
#    define SN_API_NOT_YET_FROZEN
#    include <libsn/sn-launchee.h>
#    include <gdk/gdkx.h>
namespace
{
  SnLauncheeContext *sn_context = 0;
  SnDisplay *sn_display = 0;
}
#endif // HAVE_STARTUP_NOTIFICATION

#include "audio.hh"
#include "bookmarks.hh"
#ifdef HAVE_HAL
# include "hal.hh"
#endif
#include "lastfm.hh"
#include "monitor.hh"
#include "play.hh"
#include "service_core.hh"
#include "vfs.hh"

#include "debug.hh"
#include "library.hh"
#include "logger.hh"
#include "network.hh"
#include "paths.hh"
#include "sanity.h"
#include "signals.hh"
#include "util.hh"
#include "util_file.hh"

#include "splash-screen.hh"
#include "ui_toolbox.hh"

#include "main.hh"

using namespace Bmp;

FileMonitor::Monitor   *monitor           = 0;
Play                   *play              = 0;
Logger                 *logger            = 0;
Library::Library       *library           = 0;
VFS::VFS               *vfs               = 0;
ServiceCore            *core              = 0;
Mcs::Mcs               *mcs               = 0;
LastFM::Scrobbler      *lastfm_scrobbler  = 0;
LastFM::Radio          *lastfm_radio      = 0;

#ifdef HAVE_HAL
Library::HAL           *hal               = 0;
#endif //HAVE_HAL

#ifdef HAVE_MOOD
Moo::Driver            *mood              = 0;
#endif //HAVE_MOOD

#ifdef HAVE_SM
namespace { char *session_id = 0; }
#endif //HAVE_SM

namespace { bool show_version  = false; }

char * bmp_paths[BMP_N_PATHS] = { 0 };
bool   startup = false;

namespace
{
  bool log_stdout = false;
  bool no_remote = false;
  bool no_network = false;

  SplashWindow *splash = 0;

  void
  on_startup_complete ()
  {
    if (splash)
      {
        delete splash; 
        splash = 0;
      }
  }

  GOptionEntry bmp_options[] =
    {
      {"version", 'v', 0, G_OPTION_ARG_NONE, &show_version,
       N_("Display version"), 0},
      {"no-log", 0, 0, G_OPTION_ARG_NONE, &log_stdout,
       N_("Disable logfile, log messages to shell"), 0},
      {"no-remote", 0, 0, G_OPTION_ARG_NONE, &no_remote,
       N_("Disable DBUS remote interface"), 0},
      {"no-network", 0, 0, G_OPTION_ARG_NONE, &no_network,
       N_("Disable Network dependent services"), 0},
#ifdef HAVE_SM
      {"sm-client-id", 0, 0, G_OPTION_ARG_STRING, &session_id,
       N_("Specify session management ID"), N_("ID")},
#endif // HAVE_SM
      { 0 }
    };

  static const char *features[] =
    {

#ifdef HAVE_SM
      N_("X11 Session Management"),
#endif

      N_("D-BUS support"),

#ifdef HAVE_HAL
      N_("HAL support"),
#endif

#ifdef HAVE_ALSA
      N_("ALSA Support (Linux)"),
#endif

#ifdef HAVE_SUN
      N_("SUN Audio support")
#endif
    };

  void
  print_version ()
  {
    g_print ("%s %s",
             PACKAGE,
             VERSION);

    if (std::strlen (RV_REVISION))
      g_print (" (%s-R%s)",
               RV_LAST_CHANGED_DATE,
               RV_REVISION);

    g_print (_("\nCopyright (c) 2005-2006 BMP Project <http://www.beep-media-player.org>\n\n"));

    g_print (_("Built on %s for %s with:\n"),
             BUILD_DATE,
             BUILD_ARCH);

    for (unsigned int i = 0; i < G_N_ELEMENTS (features); i++)
      g_print ("* %s\n", _(features[i]));

    g_print ("\n");
  }

  int
  create_path (const char *name)
  {
    const mode_t mode755 = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;

    if (Glib::file_test (name, Glib::FILE_TEST_IS_DIR))
      return 0;

    int mkdir_rv = g_mkdir_with_parents (name, mode755);

    if (mkdir_rv)
      {
        int errsv = errno;
        g_critical ("%s: Unable to create %s: %s", G_STRFUNC, name, strerror(errsv));
      }

    return mkdir_rv;
  }

  void
  create_paths ()
  {
    const mode_t mode755 = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
    int          sanity  = 0;

    std::string cache_dir = Glib::build_filename (g_get_user_cache_dir (), BMP_XDG_BASEDIR);

    if (!g_file_test (cache_dir.c_str (), G_FILE_TEST_IS_DIR))
      {

        sanity |= g_mkdir_with_parents (cache_dir.c_str (), mode755);

        if (sanity != 0)
          {
            int errsv = errno;
            g_critical ("%s: Unable to create %s: %s", G_STRFUNC, cache_dir.c_str (), strerror(errsv));
          }
      }

    sanity |= create_path (BMP_PATH_USER_DIR);
    sanity |= create_path (BMP_PATH_DATA_DIR);

    sanity |= create_path (BMP_PATH_LOGS_DIR);

    sanity |= create_path (BMP_PATH_COVER_THUMB_DIR);
    sanity |= create_path (BMP_PATH_LYRIC_CACHE_DIR);
    sanity |= create_path (BMP_PATH_PODCAST_CACHE_DIR);

    if (sanity)
      {
        g_error (_("%s: Unable to create one or more user directories used by BMP (please see log above). "
                   "Aborting startup; please check permissions in your home directory"), G_STRFUNC);
        exit (EXIT_FAILURE);
      }
  }

  void
  free_paths ()
  {
    for (int i = 0; i < BMP_N_PATHS; i++)
      {
        g_free (bmp_paths[i]);
        bmp_paths[i] = 0;
      }
  }

#define USER_PATH(path) \
  g_build_filename (BMP_PATH_USER_DIR, path, NULL);

#define DATA_PATH(path) \
  g_build_filename (BMP_PATH_DATA_DIR, path, NULL);

#define XDG_CACHE_PATH(path) \
  g_build_filename (g_get_user_cache_dir(), BMP_XDG_BASEDIR, path, NULL);

  void
  paths_init ()
  {
    std::string culprit = Glib::build_filename (Glib::get_home_dir(), ".config");

    if (Glib::file_test (culprit, Glib::FILE_TEST_EXISTS) && 
        Glib::file_test (culprit, Glib::FILE_TEST_IS_REGULAR))
      {
        std::string destination = Glib::build_filename (Glib::get_home_dir(), ".config-moved-by-bmpx");

        try
          {
            Util::copy_file (culprit.c_str(), destination.c_str());  
          }
        catch (std::exception& cxe)
          {
            g_critical ("%s: Unable to copy file to destination. Reason: '%s'. Source: '%s'. Destination: '%s'",
                             G_STRLOC, cxe.what(), culprit.c_str(), destination.c_str());
          }

        g_unlink (culprit.c_str());
        g_message ("The file %s has been moved to %s to accomodate for the XDG user home config dir", culprit.c_str(), destination.c_str());

      }

    BMP_PATH_USER_DIR =
      g_build_filename (g_get_user_config_dir (), BMP_XDG_BASEDIR, NULL);

    BMP_PATH_DATA_DIR =
      g_build_filename (g_get_user_data_dir (),   BMP_XDG_BASEDIR, NULL);

    BMP_PATH_LOGS_DIR =
      USER_PATH(BMP_BASENAME_LOGS_DIR);

    BMP_PATH_COVER_THUMB_DIR =
      XDG_CACHE_PATH(BMP_BASENAME_COVER_THUMB_DIR);
    BMP_PATH_LYRIC_CACHE_DIR =
      XDG_CACHE_PATH(BMP_BASENAME_LYRIC_CACHE_DIR);
    BMP_PATH_PODCAST_CACHE_DIR =
      XDG_CACHE_PATH(BMP_BASENAME_PODCAST_CACHE_DIR);

    create_paths ();
  }

  void
  i18n_init ()
  {
    gtk_set_locale ();
    bindtextdomain (PACKAGE, LOCALE_DIR);
    bind_textdomain_codeset (PACKAGE, "UTF-8");
    textdomain (PACKAGE);
  }

#ifdef HAVE_STARTUP_NOTIFICATION
  void
  sn_error_trap_push(SnDisplay *display, Display *xdisplay)
  {
    gdk_error_trap_push();
  }

  void
  sn_error_trap_pop(SnDisplay *display, Display *xdisplay)
  {
    gdk_error_trap_pop();
  }

  void
  startup_notification_complete()
  {
    Display *xdisplay;

    xdisplay = GDK_DISPLAY();
    sn_display = sn_display_new(xdisplay,
                                sn_error_trap_push,
                                sn_error_trap_pop);
    sn_context =
      sn_launchee_context_new_from_environment(sn_display,
                                               DefaultScreen(xdisplay));

    if (sn_context != 0)
      {
        sn_launchee_context_complete(sn_context);
        sn_launchee_context_unref(sn_context);

        sn_display_unref(sn_display);
      }
  }
#endif // HAVE_STARTUP_NOTIFICATION

  void
  register_keys (void)
  {
    using namespace Mcs;

    mcs->register_domain ("playlist-window");
    mcs->register_key ("playlist-window", "visible", false);
    mcs->register_key ("playlist-window", "width", 0); //FIXME:
    mcs->register_key ("playlist-window", "height",0); //FIXME:
    mcs->register_key ("playlist-window", "pos_x", 0);
    mcs->register_key ("playlist-window", "pos_y", 0);

    mcs->register_domain ("main-window");
    mcs->register_key ("main-window", "width", 0); //FIXME:
    mcs->register_key ("main-window", "height", 0); //FIXME:
    mcs->register_key ("main-window", "pos_x", 0);
    mcs->register_key ("main-window", "pos_y", 0);

    mcs->register_domain ("bmp");
    mcs->register_key ("bmp", "no-ui", false);
    mcs->register_key ("bmp", "ui-esc-trayconify", false);
    mcs->register_key ("bmp", "keep-above", false);
    mcs->register_key ("bmp", "file-chooser-close-on-open", true);
    mcs->register_key ("bmp", "file-chooser-close-on-add", false);
    mcs->register_key ("bmp", "file-chooser-path", std::string (g_get_home_dir ()));
    mcs->register_key ("bmp", "file-chooser-path-relocate-album", std::string (g_get_home_dir ()));
    mcs->register_key ("bmp", "font", std::string ("Sans 12"));
    mcs->register_key ("bmp", "icon-theme", std::string ("tango"));
    mcs->register_key ("bmp", "use-custom-cursors", true);
    mcs->register_key ("bmp", "display-tracklist-numbers", true);
    mcs->register_key ("bmp", "library-view-size", 0);
    mcs->register_key ("bmp", "force-rgba-disable", false);
    mcs->register_key ("bmp", "strip-mpeg-files", false);
    mcs->register_key ("bmp", "display-notifications", true);
    mcs->register_key ("bmp", "playlist-show-dnd-menu", false);
    mcs->register_key ("bmp", "no-remote", false);
    mcs->register_key ("bmp", "physically-delete-files", false);
    mcs->register_key ("bmp", "time-remaining", false);
    mcs->register_key ("bmp", "volume", 50);
    mcs->register_key ("bmp", "follow-current-track", false);
    mcs->register_key ("bmp", "resume-on-startup", false);
    mcs->register_key ("bmp", "resume-on-startup-track", 0);
    mcs->register_key ("bmp", "resume-on-startup-time", 0);
    mcs->register_key ("bmp", "shuffle", false);
    mcs->register_key ("bmp", "repeat", false);
    mcs->register_key ("bmp", "send-statistics", true);
    mcs->register_key ("bmp", "enable-autoplay", false);

    mcs->register_domain ("audio");
    mcs->register_key ("audio", "cdrom-device", std::string ("/dev/cdrom"));
    mcs->register_key ("audio", "sink", std::string (DEFAULT_SINK));
#ifdef HAVE_ALSA
    mcs->register_key ("audio", "alsa-buffer-time",200000);
    mcs->register_key ("audio", "device-alsa", std::string (DEFAULT_DEVICE_ALSA));
#endif //HAVE_ALSA
#ifdef HAVE_SUN
    mcs->register_key ("audio", "sun-buffer-time", 200000);
    mcs->register_key ("audio", "device-sun", std::string (DEFAULT_DEVICE_SUN));
#endif //HAVE_SUN
    mcs->register_key ("audio", "oss-buffer-time",  200000);
    mcs->register_key ("audio", "device-oss", std::string (DEFAULT_DEVICE_OSS));
    mcs->register_key ("audio", "esd-buffer-time",  200000);
    mcs->register_key ("audio", "device-esd", std::string (DEFAULT_DEVICE_ESD));

    mcs->register_domain ("lastfm");
    mcs->register_key ("lastfm", "queue-enable", false);
    mcs->register_key ("lastfm", "submit-enable", false);
    mcs->register_key ("lastfm", "username", std::string (""));
    mcs->register_key ("lastfm", "password", std::string (""));
    mcs->register_key ("lastfm", "record-to-profile", false);

#ifdef HAVE_MOOD
    mcs->register_domain ("mood");
    mcs->register_key ("bmp", "enable-mood", false);
    mcs->register_key ("mood", "hostname", std::string ("localhost"));
    mcs->register_key ("mood", "port", std::string ("2240"));
    mcs->register_key ("mood", "password", std::string (""));
#endif //HAVE_MOOD

#ifdef HAVE_OFA
    mcs->register_domain ("musicbrainz");
    mcs->register_key ("musicbrainz", "username", std::string (""));
    mcs->register_key ("musicbrainz", "password", std::string (""));
#endif //HAVE_OFA

    mcs->register_domain ("albums");
    mcs->register_key ("albums", "continuous-play", false); 

    mcs->register_domain ("podcasts");
    mcs->register_key ("podcasts", "download-dir", std::string(g_get_home_dir())); 
    mcs->register_key ("podcasts", "update-on-startup", bool(false)); 

    mcs->register_domain ("playlist");
    mcs->register_key ("playlist", "rootpath", std::string(g_get_home_dir())); 
  }

  Gtk::Main *kit = 0;

  void
  app_start () { kit->run(); }

  void
  app_quit  () { kit->quit(); }

  /// Splash screen stuff
  char const* splash_messages[] =
  {
    _("Initializing Core"),    
#ifdef HAVE_HAL
    _("Initializing HAL Subsystem"),
#endif
#ifdef HAVE_MOOD
    _("Initializing MooDriver"),
#endif
    _("Initializing VFS"),
    _("Initializing File Monitoring"),
    _("Initializing Playback System"),
    _("Initializing Library"),
    _("Initializing LastFM Scrobbling"),
    _("Initializing LastFM Radio"),
    _("Setting up User Interface") 
  };

  unsigned int message_id = 0;

  char const*
  splash_get_message ()
  {
    return _(splash_messages[message_id++]);
  }

  double
  splash_get_percent ()
  {
    return double((message_id*1.)/((G_N_ELEMENTS(splash_messages)-1)*1.));
  } 

} //!<unnamed>:: 

int
main (int    argc,
      char** argv)
{
  char* sanity_error = sanity_check_libs ();
  if (sanity_error)
    {
      g_printerr ("%s\n", sanity_error);
      g_free (sanity_error);
      std::exit (EXIT_FAILURE);
    }

  Glib::init ();
  Glib::thread_init (0);
  Glib::wrap_init();
  i18n_init ();
  paths_init ();
  GtkWidget *w = 0; 

  GOptionContext *_oc = g_option_context_new (_(" - Run BMP"));
  g_option_context_add_main_entries (_oc, bmp_options, PACKAGE);
  g_option_context_add_group (_oc, gst_init_get_option_group ());
  Glib::OptionContext oc (_oc, true);

  try { 
    kit = new Gtk::Main (argc, argv, oc);
    gtk_rc_parse_string ("style \"bmp-menu-bar-style\"\n"
                         "{\n"
                         "  GtkMenuBar::shadow-type = GTK_SHADOW_NONE\n"
                         "}\n"
                         "class \"GtkWidget\" style \"bmp-menu-bar-style\"\n");
    }
  catch (Glib::OptionError& cxe)
    {
      std::cerr << "BMP"  << " (" << argv[0] << "): " << cxe.what() << "\n";
      std::exit (EXIT_FAILURE);
    }

  if (show_version)
    {
      print_version ();
      goto main_early_exit;
    }

  debug_init ();
  startup = true;

  if (!log_stdout)
    {
      logger = new Logger;
    }

  if (!no_remote)
    {
      dbus_g_type_specialized_init ();
    }

  try {
      mcs = new Mcs::Mcs (Glib::build_filename (BMP_PATH_USER_DIR, "config.xml"), "bmp", 0.30);
      register_keys ();
      mcs->load (Mcs::Mcs::VERSION_IGNORE);
    }
  catch (Mcs::Mcs::Exceptions& cxe)
    {
      if (cxe == Mcs::Mcs::PARSE_ERROR)
      {
        g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, _("Unable to parse configuration file"));
      }
    }

  // Workaround for gtk+ < 2.10.4
  w = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_realize (w);
  Util::style_save (GTK_WIDGET (w));

  Util::check_alpha ();

  splash = new SplashWindow();
  splash->show ();

  while (gtk_events_pending()) gtk_main_iteration();

  Session::start (argc, argv, session_id);

#ifdef HAVE_STARTUP_NOTIFICATION
  gtk_window_set_auto_startup_notification (false);
#endif //HAVE_STARTUP_NOTIFICATION

  splash->set_message (splash_get_message(), splash_get_percent());

  core = new ServiceCore;

  core->signal_startup_complete().connect
      (sigc::ptr_fun (&on_startup_complete)); 
  core->register_application_control
      (sigc::ptr_fun (&app_start), sigc::ptr_fun (&app_quit));

  if (no_network)
    Network::disable ();
  else
    Network::check_connected (true);

  Signals::handlers_init ();

#ifdef HAVE_HAL
  splash->set_message (splash_get_message(), splash_get_percent());
  hal = new Library::HAL;
#endif //HAVE_HAL

#ifdef HAVE_MOOD
  splash->set_message (splash_get_message(), splash_get_percent());
  mood = new Moo::Driver (mcs->key_get<std::string>("mood", "hostname"),
                          mcs->key_get<std::string>("mood", "port"),
                          mcs->key_get<std::string>("mood", "password"));
                          
#endif //HAVE_MOOD

  splash->set_message (splash_get_message(), splash_get_percent());
  vfs = new VFS::VFS;

  splash->set_message (splash_get_message(), splash_get_percent());
  monitor = new FileMonitor::Monitor;

  splash->set_message (splash_get_message(), splash_get_percent());
  play = new Play;

  splash->set_message (splash_get_message(), splash_get_percent());
  library = new Library::Library;

  splash->set_message (splash_get_message(), splash_get_percent());
  lastfm_scrobbler = new LastFM::Scrobbler;

  splash->set_message (splash_get_message(), splash_get_percent());
  lastfm_radio = new LastFM::Radio;

  splash->set_message (splash_get_message(), splash_get_percent());
  core->ui_start ();

#ifdef HAVE_STARTUP_NOTIFICATION
  startup_notification_complete ();
#endif //HAVE_STARTUP_NOTIFICATION

  startup = false;
  core->run ();

  ////////// Shutdown

  Session::end ();

  delete monitor;
  delete play;
  delete lastfm_scrobbler;
  delete lastfm_radio;

#ifdef HAVE_MOOD
  mood->disconnect ();
  delete mood;
#endif

  core->ui_stop ();

#ifdef HAVE_HAL
  delete hal;
#endif

  delete vfs;
  delete library;
  delete mcs;
  delete logger;
  delete core;

 main_early_exit:

  free_paths ();
  std::exit (EXIT_SUCCESS);
}
