/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#include <errno.h>
#include "compat/strerror.h"
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strdup.h"
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include "hexdump.h"
#include "channel.h"
#include "game.h"
#include "server.h"
#include "eventlog.h"
#include "account.h"
#include "connection.h"
#include "game.h"
#include "version.h"
#include "eventlog.h"
#include "prefs.h"
#include "ladder.h"
#include "adbanner.h"


static void usage(char const * progname);


static void usage(char const * progname)
{
    fprintf(stderr,
	    "usage: %s [<options>]\n\n"
	    "    -c FILE, --config=FILE   use FILE as configuration file\n"
	    "    -d FILE, --hexdump=FILE  do hex dump of packets into FILE\n"
	    "    -f, --foreground         don't daemonize\n"
	    "    -h, --help, --usage      show this information and exit\n"
	    "    -v, --version            print version number and exit\n",
	    progname);
    exit(1);
}


extern int main(int argc, char * argv[])
{
    int          a;
    char const * pidfile;
    char const * hexfile=NULL;
    int          foreground=0;
    
    if (argc<1 || !argv || !argv[0])
    {
        fprintf(stderr,"bad arguments\n");
        return 1;
    }
    
    preffile = NULL;
    for (a=1; a<argc; a++)
        if (strncmp(argv[a],"--config=",9)==0)
        {
            if (preffile)
            {
                fprintf(stderr,"%s: configuration file was already specified as \"%s\"\n",argv[0],preffile);
                usage(argv[0]);
            }
	    preffile = &argv[a][9];
	}
        else if (strcmp(argv[a],"-c")==0)
        {
            if (a+1>argc)
            {
                fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
                usage(argv[0]);
            }
            if (preffile)
            {
                fprintf(stderr,"%s: configuration file was already specified as \"%s\"\n",argv[0],preffile);
                usage(argv[0]);
            }
            a++;
	    preffile = argv[a];
        }
        else if (strncmp(argv[a],"--hexdump=",10)==0)
        {
            if (hexfile)
            {
                fprintf(stderr,"%s: configuration file was already specified as \"%s\"\n",argv[0],hexfile);
                usage(argv[0]);
            }
            hexfile = &argv[a][10];
        }
        else if (strcmp(argv[a],"-d")==0)
        {
            if (a+1>argc)
            {
                fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
                usage(argv[0]);
            }
            if (hexfile)
            {
                fprintf(stderr,"%s: configuration file was already specified as \"%s\"\n",argv[0],hexfile);
                usage(argv[0]);
            }
            a++;
            hexfile = argv[a];
        }
        else if (strcmp(argv[a],"-f")==0 || strcmp(argv[a],"--foreground")==0)
            foreground = 1;
        else if (strcmp(argv[a],"-v")==0 || strcmp(argv[a],"--version")==0)
	{
            printf("version "BNETD_VERSION"\n");
            return 0;
	}
        else if (strcmp(argv[a],"-h")==0 || strcmp(argv[a],"--help")==0 || strcmp(argv[a],"--usage")==0)
            usage(argv[0]);
	else if (strcmp(argv[a],"--config")==0 || strcmp(argv[a],"--hexdump")==0)
	{
            fprintf(stderr,"%s: option \"%s\" requires and argument.\n",argv[0],argv[a]);
            usage(argv[0]);
	}
	else
        {
            fprintf(stderr,"%s: bad option \"%s\"\n",argv[0],argv[a]);
            usage(argv[0]);
        }
    
    eventlog_set(stderr);
    /* errors to eventlog from here on... */
    
    if (preffile)
    {
	if (prefs_load(preffile)<0)
	{
	    eventlog(eventlog_level_error,"main","could not parse configuration file (exiting)");
	    return 1;
	}
    }
    else
	if (prefs_load(BNETD_DEFAULT_CONF_FILE)<0)
	    eventlog(eventlog_level_warn,"main","using default configuration");
    
    {
	char const * levels;
	char * temp;
	char * tok;
	
	eventlog_clear_level();
	if ((levels = prefs_get_loglevels()))
	{
	    if (!(temp = strdup(levels)))
	    {
		eventlog(eventlog_level_error,"main","could not allocate memory for temp (exiting)");
		return 1;
	    }
	    tok = strtok(temp,","); /* strtok modifies the string it is passed */
	    while (tok)
	    {
		if (eventlog_add_level(tok)<0)
		    eventlog(eventlog_level_error,"main","could not add log level \"%s\"",tok);
		tok = strtok(NULL,",");
	    }
	    pfree(temp,strlen(levels)+1);
	}
    }
    
    if (eventlog_open(prefs_get_logfile())<0)
    {
	if (prefs_get_logfile())
	    eventlog(eventlog_level_error,"main","could not use file \"%s\" for the eventlog (exiting)",prefs_get_logfile());
	else
	    eventlog(eventlog_level_error,"main","no logfile specified in configuration file \"%s\" (exiting)",preffile);
	return 1;
    }
    
    /* eventlog must go to log file from here on... */
    
    if (!foreground)
    {
	if (chdir("/")<0)
	{
	    eventlog(eventlog_level_error,"main","could not change working directory to / (chdir: \"%s\")",strerror(errno));
	    return 1;
	}
	
	switch (fork())
	{
	case -1:
	    eventlog(eventlog_level_error,"main","could not fork (fork: \"%s\")",strerror(errno));
	    return 1;
	case 0: /* child */
	    break;
	default: /* parent */
	    return 0;
	}
	
	close(0);
	close(1);
	close(2);
	
	if (setpgid(0,0)<0)
	{
	    eventlog(eventlog_level_error,"main","could not create new process group (setpgid: \"%s\")",strerror(errno));
	    return 1;
	}
    }
    
    pidfile = prefs_get_pidfile();
    if (pidfile[0]!='\0')
    {
	FILE * fp;
	
	if (!(fp = fopen(pidfile,"w+")))
	{
	    eventlog(eventlog_level_error,"main","unable to open pid file \"%s\" for writing (fopen: \"%s\")",pidfile,strerror(errno));
	    pidfile = NULL;
	}
	else
	{
	    fprintf(fp,"%u",(unsigned int)getpid());
	    fclose(fp);
	}
    }
    eventlog(eventlog_level_info,"main","bnetd version "BNETD_VERSION" process %u",(unsigned int)getpid());
    eventlog(eventlog_level_info,"main","logging event levels: %s",prefs_get_loglevels());
    
    if (hexfile)
	if (!(hexstrm = fopen(hexfile,"w")))
	    eventlog(eventlog_level_error,"main","could not open file \"%s\" for writing the hexdump (fopen: %s)",hexfile,strerror(errno));
    
    connlist_init();
    gamelist_init();
    
    channellist_init();
    if (adbannerlist_load(prefs_get_adfile())<0)
	eventlog(eventlog_level_error,"server_process","could not load adbanner list");
    
    accountlist_load_default();
    accountlist_load();
    
    ladder_init();
    
    /* now process connections and network traffic */
    if (server_process()<0)
    {
        eventlog(eventlog_level_info,"main","failed to initialize network (exiting)");
	if (pidfile[0]!='\0')
	    if (unlink(pidfile)<0)
		eventlog(eventlog_level_error,"main","could not remove pid file \"%s\" (unlink: %s)",pidfile,strerror(errno));
        return 1;
    }
    
    eventlog(eventlog_level_info,"main","unloading user accounts");
    accountlist_unload();
    accountlist_unload_default();
    
    channellist_unload();
    adbannerlist_unload();

    if (hexstrm)
	fclose(hexstrm);
    
    eventlog(eventlog_level_info,"main","server has shut down");
    if (pidfile[0]!='\0')
	if (unlink(pidfile)<0)
	    eventlog(eventlog_level_error,"main","could not remove pid file \"%s\" (unlink: %s)",pidfile,strerror(errno));
    
    prefs_unload();
    
    return 0;
}
