/***************************************************************************
                          dlgmain.cpp  -  description
                             -------------------
    begin                : Sat Dec 28 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kenschenke at yahoo dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, as a special exception, Ken Schenke gives permission to  *
 *   link the code of this program with the Qt non-commercial edition (or  *
 *   with modified versions of the Qt non-commercial edition that use the  *
 *   same license as the Qt non-commercial edition, and distribute linked  *
 *   combinations including the two.  You must obey the GNU General Public *
 *   License in all respects for all of the code used other than the Qt    *
 *   Non-Commercial edition.  If you modify this file, you may extend this *
 *   exception to your version of the file, but you are not obligated to   *
 *   do so.  If you do not wish to do so, delete this exception statement  *
 *   from your version.                                                    *
 *                                                                         *
 ***************************************************************************/

#include "dlgmain.h"
#include "dlgsettings.h"
#include "dlgaddbrowser.h"
#include "dlgviewbookmarks.h"
#include "dlgabout.h"
#include "dlghelp.h"
#include "dlgexcept.h"
#include "version.h"

#include "bkutil.h"

#include <qmessagebox.h>
#include <qlistbox.h>
#include <qpushbutton.h>
#include <qlabel.h>

#include <algorithm>
#include <vector>
#include <functional>

using namespace std;

/***************************************************************************
 *                                                                         *
 *   This class (derived from QListBoxText) represents an item in the      *
 *   sources or destinations list boxes.  All it really does is act as a   *
 *   container for the ordinal (the browser).                              *
 *                                                                         *
 ***************************************************************************/

class SDItem : public QListBoxText
{
public:
	SDItem(QListBox *listbox, const QString & text = QString::null)
		: QListBoxText(listbox, text), ordinal(0) { };

	BRWSNUM ordinal;
};

/***************************************************************************
 *                                                                         *
 *   This class is a function object.  It's basically an object that acts  *
 *   like a function by over-riding the call operator (the ()'s).  It is   *
 *   used when the user wants to add a browser as a source.  This function *
 *   object is called by the remove_copy_if() C++ STL algorithm.  It is    *
 *   called for each browser as it is added from the configured browsers   *
 *   vector.  It returns non-zero if the browser is already listed in the  *
 *   sources vector.  This prevents the browser from being added a second  *
 *   time as a source by the user.                                         *
 *                                                                         *
 ***************************************************************************/

class SourceAddIf
{
public:
	SourceAddIf(BridgeCfg &cfg) : m_cfg(cfg) { };

	bool operator()(const BrowserCfg & browser) const
	{
		return (std::find(
			m_cfg.sources.begin(),
			m_cfg.sources.end(),
			browser.ordinal()) != m_cfg.sources.end());
	}

private:
	BridgeCfg & m_cfg;
};

/***************************************************************************
 *                                                                         *
 *   This class is a function object to prevent duplicate destination      *
 *   browsers.  See the explanation above for the SourceAddIf class.       *
 *                                                                         *
 ***************************************************************************/

class DestinationAddIf
{
public:
	DestinationAddIf(BridgeCfg &cfg) : m_cfg(cfg) { };

	bool operator()(const BrowserCfg & browser) const
	{
		return (std::find(
			m_cfg.destinations.begin(),
			m_cfg.destinations.end(),
			browser.ordinal()) != m_cfg.destinations.end()
		 || browser.readOnly() == true);
	}

private:
	BridgeCfg & m_cfg;
};

/***************************************************************************
 *                                                                         *
 *   DlgMain::DlgMain()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      QWidget *parent                                                    *
 *      const char *name                                                   *
 *      bool modal                                                         *
 *      WFlags f                                                           *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      Class constructor                                                  *
 *                                                                         *
 ***************************************************************************/

DlgMain::DlgMain( QWidget* parent, const char* name, bool modal, WFlags f )
	: DlgMainBase( parent, name, modal, f )
{
	QString version;

	version.sprintf("BookmarkBridge %s", BMB_VERSION);
	setCaption(version);

	m_dlgHelp = NULL;

	try
	{
		m_cfg.readConfig();
	}
	catch(BkException e)
	{
		DlgExcept dlg(e, this, NULL, true);

		dlg.exec();
		return;
	}

	refreshLists();
	refreshButtons();
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::~DlgMain()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      Class destructor                                                   *
 *                                                                         *
 ***************************************************************************/

DlgMain::~DlgMain()
{
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonAboutClicked()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks the About button.  All it does is display the      *
 *      DlgAbout dialog box.                                               *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonAboutClicked(void)
{
	DlgAbout	dlg(this, NULL, true);

	dlg.exec();
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonAddSourceClicked()                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Add (source) button.  It presents a list of *
 *      browsers to the user that have not already been selected as a      *
 *      source.  If the user picks one, it is added to the sources vector. *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonAddSourceClicked(void)
{
	// Special case: check to see if the user has not configured ANY browsers

	if(m_cfg.browsers.size() < 1)
	{
		QMessageBox::warning(this, "Notice",
			"You have not configured any browsers yet.\n"
			"Please click the Settings button in the main\n"
			"window of BookmarkBridge.");
		return;
	}

	vector<BrowserCfg> browsers;

	// Make a copy of the browsers from the configuration settings
	// to the browsers vector, declared above, that are not already
	// selected as source browsers.  The SourceAddIf function object
	// checks the sources list and returns true for each browser
	// already in the sources list.

	remove_copy_if(
		m_cfg.browsers.begin(), m_cfg.browsers.end(),
		inserter(browsers,browsers.end()), SourceAddIf(m_cfg));

	// If no available browsers were found, bail out.

	if(browsers.empty())
	{
		QMessageBox::information(this, "No Browsers Available",
			"All available browsers have already\n"
			"been selected as a source browser.");
		return;
	}

	// Configure and display the dialog box to add a source browser

	DlgAddBrowser   dlg(browsers, this, NULL, true);

	dlg.setCaption("Add Source Browser");
	dlg.labelSelect->setText("Select a Source Browser to Add");
	if(dlg.exec() != Accepted)
		return;

	// The user selected a browser and hit the "Add" button.  Add it
	// to the sources list and refresh the display.

	m_cfg.sources.push_back(dlg.ordinal);
	refreshLists();
	refreshButtons();

	try
	{
		m_cfg.writeConfig();
	}
	catch(BkException e)
	{
		DlgExcept dlg(e, this, NULL, true);

		dlg.exec();
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonAddDestinationClicked()                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Add (destination) button.  It presents a    *
 *      list of browsers to the user that have not already been selected   *
 *      as a destination or ones that are not configured as read only.  If *
 *      the user picks one, it is added to the destinations vector.        *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonAddDestinationClicked(void)
{
	// Special case: check to see if the user has not configured ANY browsers

	if(m_cfg.browsers.size() < 1)
	{
		QMessageBox::warning(this, "Notice",
			"You have not configured any browsers yet.\n"
			"Please click the Settings button in the main\n"
			"window of BookmarkBridge.");
		return;
	}

	vector<BrowserCfg> browsers;

	// Make a copy of the browsers from the configuration settings
	// to the browsers vector, declared above, that are not already
	// selected as destination browsers.  The DestinationAddIf function object
	// checks the destinations list and returns true for each browser
	// already in the destinations list or for browsers that are
	// declared as read-only.

	remove_copy_if(
		m_cfg.browsers.begin(), m_cfg.browsers.end(),
		inserter(browsers,browsers.end()), DestinationAddIf(m_cfg));

	// If no available browsers were found, bail out.

	if(browsers.empty())
	{
		QMessageBox::information(this, "No Browsers Available",
			"All available browsers have either, already been selected\n"
			"as a destination browser or, are marked as read-only.");
		return;
	}

	// Configure and display the dialog box to add a destination browser

	DlgAddBrowser   dlg(browsers, this, NULL, true);

	dlg.setCaption("Add Destination Browser");
	dlg.labelSelect->setText("Select a Destination Browser to Add");
	if(dlg.exec() != Accepted)
		return;

	// The user selected a browser and hit the "Add" button.  Add it
	// to the destinations list and refresh the display.

	m_cfg.destinations.push_back(dlg.ordinal);
	refreshLists();
	refreshButtons();

	try
	{
		m_cfg.writeConfig();
	}
	catch(BkException e)
	{
		DlgExcept dlg(e, this, NULL, true);

		dlg.exec();
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonDeleteSourceClicked()                                  *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Delete (source) button.  It confirms this   *
 *      action with the user then removes the browser from the sources     *
 *      vector.                                                            *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonDeleteSourceClicked(void)
{
	// make sure a source browser is actually selected

	int nItem = listSources->currentItem();
	if(nItem < 0)
		return;

	// confirm this with the user
	
	if(QMessageBox::warning(this, "Are You Sure?",
		"Are you sure you wish to remove\n"
		"this browser as a source?",
		QMessageBox::Yes, QMessageBox::No) != QMessageBox::Yes)
		return;
	SDItem *item = static_cast<SDItem *>(listSources->item(nItem));

	// this messy looking piece of code removes any entries from
	// the sources vector that match the ordinal to be deleted.
	
	m_cfg.sources.erase(remove_if(
		m_cfg.sources.begin(),
		m_cfg.sources.end(),
		bind2nd(equal_to<int>(),item->ordinal)), m_cfg.sources.end());

	refreshLists();
	refreshButtons();

	// save the change to the configuration file

	try
	{
		m_cfg.writeConfig();
	}
	catch(BkException e)
	{
		DlgExcept dlg(e, this, NULL, true);

		dlg.exec();
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonDeleteDestinationClicked()                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Delete (destination) button.  It confirms   *
 *      this action with the user then removes the browser from the        *
 *      destinations vector.                                               *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonDeleteDestinationClicked(void)
{
	// make sure a destination browser is actually selected

	int nItem = listDestinations->currentItem();
	if(nItem < 0)
		return;

	// confirm this with the user

	if(QMessageBox::warning(this, "Are You Sure?",
		"Are you sure you wish to remove\n"
		"this browser as a destination?",
		QMessageBox::Yes, QMessageBox::No) != QMessageBox::Yes)
		return;
	SDItem *item = static_cast<SDItem *>(listDestinations->item(nItem));

	// this messy looking piece of code removes any entries from
	// the destinations vector that match the ordinal to be deleted.

	m_cfg.destinations.erase(remove_if(
		m_cfg.destinations.begin(),
		m_cfg.destinations.end(),
		bind2nd(equal_to<int>(),item->ordinal)), m_cfg.destinations.end());

	refreshLists();
	refreshButtons();

	// save the change to the configuration file

	try
	{
		m_cfg.writeConfig();
	}
	catch(BkException e)
	{
		DlgExcept dlg(e, this, NULL, true);

		dlg.exec();
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonHelpClicked()                                          *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Help button.  It displays the help dialog.  *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonHelpClicked(void)
{
	// Since the Help dialog is modeless, meaning it doesn't block user
	// interaction with BookmarkBridge while open, it could already be
	// open.  If it is not, open it.

	if(m_dlgHelp == NULL)
		m_dlgHelp = new DlgHelp(m_cfg);

	// show the dialog and bring it to the front

	m_dlgHelp->setSource("index.html");
	m_dlgHelp->show();
	m_dlgHelp->setActiveWindow();
	m_dlgHelp->raise();
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonSettingsClicked()                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Settings button.  It gives the settings     *
 *      dialog box its own copy of the BridgeCfg class.  If the user hits  *
 *      Cancel in the dialog the no changes are made.  If the user hits Ok *
 *      instead then the changes are copied back from the dialog's copy of *
 *      the BridgeCfg class then saved to the configuration file.          *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonSettingsClicked(void)
{
	DlgSettings dlg(m_cfg, this, NULL, true);

	// show the dialog then return if the user hit Cancel
	
	if(dlg.exec() != Accepted)
		return;

	// copy the configuration changes back from the dialog's copy
	// of the BridgeCfg class
	
	m_cfg = dlg.cfg();

	// save the configuration and update the sources and destinations list

	try
	{
		m_cfg.writeConfig();
	}
	catch(BkException e)
	{
		DlgExcept dlg(e, this, NULL, true);

		dlg.exec();
		return;
	}
	refreshLists();
	refreshButtons();
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::refreshButtons()                                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called when the state of the dialog changes that  *
 *      could affect which buttons should be enabled.  This usually occurs *
 *      when the user selects or de-selects an item in the sources or      *
 *      destinations lists.                                                *
 *                                                                         *
 ***************************************************************************/

void DlgMain::refreshButtons(void)
{
	buttonSourceDelete->setEnabled(listSources->currentItem() != -1);
	buttonDestDelete->setEnabled(listDestinations->currentItem() != -1);
	buttonView->setEnabled(m_cfg.sources.size() > 0);
	buttonMerge->setEnabled(m_cfg.sources.size()>0 && m_cfg.destinations.size()>0);
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::refreshLists()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called when the state of the dialog changes that  *
 *      could affect the soures or destinations lists.  This usually       *
 *      occurs when the dialog box first opens or when the user changes    *
 *      the configuration.                                                 *
 *                                                                         *
 ***************************************************************************/

void DlgMain::refreshLists(void)
{
	vector<BRWSNUM>::const_iterator it;
	vector<BrowserCfg>::iterator fit;

	// clear the lists
	
	listSources->clear();
	listDestinations->clear();

	SDItem *item;

	// loop through the configured sources, adding them to the list
	
	for(it=m_cfg.sources.begin(); it!=m_cfg.sources.end(); ++it)
	{
		fit = std::find(m_cfg.browsers.begin(), m_cfg.browsers.end(), *it);
		if(fit != m_cfg.browsers.end())
		{
			item = new SDItem(listSources, fit->descrip().c_str());
			item->ordinal = *it;
		}
	}

	// loop through the configured destinations, adding them to the list
	
	for(it=m_cfg.destinations.begin(); it!=m_cfg.destinations.end(); ++it)
	{
		fit = std::find(m_cfg.browsers.begin(), m_cfg.browsers.end(), *it);
		if(fit != m_cfg.browsers.end())
		{
			item = new SDItem(listDestinations, fit->descrip().c_str());
			item->ordinal = *it;
		}
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonMergeClicked()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Merge button.                               *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonMergeClicked(void)
{
	try
	{
		BkUtil	util(m_cfg);

		// read BookmarkBridge's copy of the bookmark tree from the XBEL file
		
		util.readBookmarkTree();

		// read the bookmarks from each of the browsers, adding them to the
		// bookmark tree.
		
		util.readBookmarksFromBrowsers();

		// If any changes were found between a browser's copy of a bookmark
		// and the bookmark tree's copy, they are recorded in the tree in a
		// "difference list".  Each bookmark node in the tree has one of these
		// lists.  This function looks at each bookmark where this is list is
		// not empty and decides which version of the bookmark to keep.  It
		// then moves the changes up to the bookmark node and emptys the list.
		
		util.root()->rollUpDifferences();

		// This function saves the bookmark tree out to each browser, thereby
		// saves changes detected during the merge.
		
		util.saveBookmarksToBrowsers();

		// This function saves BookmarkBridge's copy of the bookmark tree.
		
		m_cfg.writeBookmarksXbel(*util.root());
	}
	catch(BkException e)
	{
		DlgExcept dlg(e, this, NULL, true);

		dlg.exec();
		return;
	}

	QMessageBox::information(this, "Done", "Bookmark Merge is Complete");
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::buttonViewBookmarksClicked()                                 *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the View button.                                *
 *                                                                         *
 ***************************************************************************/

void DlgMain::buttonViewBookmarksClicked(void)
{
	try
	{
		BkUtil	util(m_cfg);

		// read BookmarkBridge's copy of the bookmark tree from the XBEL file

		util.readBookmarkTree();

		// read the bookmarks from each of the browsers, adding them to the
		// bookmark tree.

		util.readBookmarksFromBrowsers();

		// If any changes were found between a browser's copy of a bookmark
		// and the bookmark tree's copy, they are recorded in the tree in a
		// "difference list".  Each bookmark node in the tree has one of these
		// lists.  This function looks at each bookmark where this is list is
		// not empty and decides which version of the bookmark to keep.  It
		// then moves the changes up to the bookmark node and emptys the list.

		util.root()->rollUpDifferences();

		// Show a dialog box allowing the user to view the bookmarks tree.  The
		// dialog box also allows the user to tell BookmarkBridge to ignore certain
		// bookmarks or even an entire folder.
		
		DlgViewBookmarks dlg(m_cfg, util.root(), this, NULL, true);

		dlg.exec();

		// This function saves BookmarkBridge's copy of the bookmark tree, including
		// any changes the user made.

		m_cfg.writeBookmarksXbel(*util.root());
	}
	catch(BkException e)
	{
		DlgExcept dlg(e, this, NULL, true);

		dlg.exec();
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::listDestinationsSelectionChanged()                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks anywhere in the destinations list.  All it really  *
 *      does is enable or disable buttons as appropriate by calling the    *
 *      refreshButtons() function.                                         *
 *                                                                         *
 ***************************************************************************/

void DlgMain::listDestinationsSelectionChanged(void)
{
	refreshButtons();
}

/***************************************************************************
 *                                                                         *
 *   DlgMain::listSourcesSelectionChanged()                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks anywhere in the sources list.  All it really does  *
 *      is enable or disable buttons as appropriate by calling the         *
 *      refreshButtons() function.                                         *
 *                                                                         *
 ***************************************************************************/

void DlgMain::listSourcesSelectionChanged(void)
{
	refreshButtons();
}
