#! /bin/bash

# env. vars you can set, and what they mean
#   tmpdir     -- where to build
#   arch       -- architecture, otherwise, autosense
#   verbose    -- be more verbose in output
#   debug      -- the above, and set -x
#   ignorefail -- ignore failures in subshells

# you can set tmpdir
tmpdir=${tmpdir:-/var/tmp}

# architecture is set in top-level Makefile's config; use that if we have it
arch=${architecture-$(dpkg --print-architecture)}

# abort on any non-zero exit value
set -e

# trap exit to make errors more obvious
settrapzero () {
	trap "if [ \$? -ne 0 ]; then echo E: $0 abort >&2; fi" 0
}

settrapzero

# you can turn on debugging with 'debug'
if [ ${debug} ]; then
    set -x
    set -v
fi

set_file_perms () {
	local file="$1"
	chmod 0644 $file
	chown root.root $file
}

last_by_version () {
	local a="$1"
	local b="$2"
	

	if [ -z "$b" ]; then
		echo $a
		exit 0
	fi

	shift 2

	if dpkg --compare-versions `dpkg -f $a Version` gt `dpkg -f $b Version`; then
		last_by_version $a $@
	else
		last_by_version $b $@
	fi
}

make_paths() {
	local PATHS=""
	local i
	local exit

	exit=false

	for i in $*; do
		local f=""
		local p
		local j
		for j in $archive/*/${i}_[^a-zA-Z]*.deb \
				`pwd`/updates/${i}_[^a-zA-Z]*.deb; do
			if [ -f "$j" ]; then
				f="$f $j"
			fi
		done
		p=`last_by_version $f`
		if [ ! -f "$p" ]; then
			echo "Can't find package" $i 1>& 2
			exit=true
		fi
		PATHS=$PATHS" "$p
	done
	if $exit; then
		exit -1
	fi
	echo $PATHS
}

# sync and sleep, avoiding weird failures
relax () {
    sync
    sleep 1
    sync
    sleep 1
}

# check that we are root
testroot () {
	[ `id -u` = 0 ] || error "must be run as root"
}

make_tmpdir() {
    test -d $1 || mkdir $1
}

# subshells failure doesn't cause script failure, so we use this 
# after using subshelles
testexit () {
	local status=$?
	if [ $status -ne 0 ]; then
		if [ ${ignorefail} ]; then
			warn "ignoring failure ($status)"
		else
			warn "abort ($status)"
			exit $status
		fi
	fi
}

# different internal logging facilities
warn () {
    echo "W:" $* 1>&2
}

error () {
    echo "E:" $* 1>&2
    exit -1
}

info () {
    echo "I:" $*
}

debug () {
    # either debug or the special verbose var can turn this on
    [ ${debug} ${verbose} ] && echo "D: " $* || true
}

# Pad the last floppy image with zeroes
# pad_last_image basename size
pad_last_image () {
  last=`ls ${1}*|tac|head -1`

  mv ${last} ${last}.tmp

  dd if=/dev/zero of=${last} bs=1k count=$2
  dd if=${last}.tmp of=${last} bs=1k count=$2 conv=notrunc

  rm -f ${last}.tmp
}

if [ ! -d ${tmpdir}/boot-floppies ]
then
    mkdir -m 700 ${tmpdir}/boot-floppies
fi
