#include <assert.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <linux/fdreg.h>
#include <linux/fd.h>
#include <syslog.h> 
#include "dbootstrap.h"
#include "lang.h"
#include "util.h"
#include <stdlib.h>
#include <dirent.h>

/* ED: floppy filesystem type is not the same for all architectures */
#if #cpu(sparc) || #cpu(powerpc) || #cpu(alpha)
static char *fs_type = "ext2";
#else
static char *fs_type = "msdos";
#endif

#if #cpu(sparc)
# define USE_SILO
#elif #cpu(m68k)
# define USE_LILOM68K
#elif #cpu(powerpc)
# define USE_PLILO
#elif #cpu(alpha)
# define USE_ABOOT
#elif #cpu(arm)
/* netwinder doesn't usr, riscpc? */
#else
# define USE_LILO
#endif

#ifdef _TESTING_
#define BC_DIR_PREFIX  ""
#define BC_LILO_CONF   "etc_lilo.conf_test"
#define BC_INST_MBR    "echo pretending to install-mbr device=%s"
#define BC_BOOT_DFLT   "echo pretending to activate device=%s, partition=%s"
#define BC_RUN_LILO    "echo pretend running lilo..."
#define BC_SFORMAT     "/usr/bin/superformat"
#define BC_RUN_SFORMAT "echo pretending to superformat device=%s, size=%s"

#define BC_ABOOT_CONF  "etc_aboot.conf_test"
#define BC_MILO_CONF   "etc_milo.conf_test"
#define BC_APB_CFG     "dos_abp.cfg_test"
#define BC_RUN_SWRITEBOOT "/bin/echo /sbin/swriteboot -v %s /boot/bootlx"
#define BC_RUN_ABOOTCONF "/bin/echo /sbin/abootconf %s %s"
#define BC_RUN_MILOCTL "/bin/echo /sbin/miloctl cmd=\"boot %s:\" /instmnt/milo"
#else

#define BC_DIR_PREFIX  "/target"
#define BC_LILO_CONF   "/target/etc/lilo.conf"
#define BC_INST_MBR    "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/install-mbr %s"
#define BC_BOOT_DFLT   "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/activate %s %s"
#define BC_RUN_LILO    "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/lilo -r /target >/dev/null"

#define BC_SILO_CONF   "/target/etc/silo.conf"
#define BC_RUN_SILO    "chroot /target /sbin/silo -f >/dev/null"

#define BC_QUIK_CONF  "/target/etc/quik.conf"
#define BC_RUN_QUIK   "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/quik -f -r /target >/dev/null"

#define BC_RUN_CATSBOOT "chroot /target /usr/sbin/catsboot /vmlinuz /vmlinuz"

/* Technically, LD_LIBRARY_PATH is not needed here, since we have
   swriteboot and abootconf on the root disk (for rescue use) but oh well */
#define BC_ABOOT_CONF "/target/etc/aboot.conf"
#define BC_RUN_SWRITEBOOT "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/swriteboot -f3 %s /target/boot/bootlx"
#define BC_RUN_ABOOTCONF "export LD_LIBRARY_PATH\"=/target/lib:/target/usr/lib\"; /target/sbin/abootconf %s %s"

#define BC_MILO_CONF  "/target/etc/milo.conf"
#define BC_RUN_MILOCTL "/sbin/miloctl cmd=\"boot %s:\" /instmnt/milo"

#define BC_APB_CFG "/target/dos/apb.cfg"

#define BC_VMELILO_CONF  "/target/etc/vmelilo.conf"
#define BC_RUN_VMELILO    "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/vmelilo -f -w /target >/dev/null"

#define BC_SFORMAT     "/target/usr/bin/superformat"
#define BC_RUN_SFORMAT "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; "BC_SFORMAT" %s %s"

#define BC_PREPDD      "/target/bin/dd"
#define BC_RUN_PREPDD  "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; "BC_PREPDD" if=%s of=%s 2>/dev/null"

#endif

static int write_boot_floppy (char *device)
{
  struct stat tmp;
  FILE* filep;
  int size, status;

  int fd;
  struct floppy_struct floppy_type;

  snprintf(prtbuf, sizeof(prtbuf), "umount %s", device);
  execlog(prtbuf, LOG_DEBUG);
  execlog("umount /floppy", LOG_DEBUG);

#if #cpu (powerpc)
  if (strcmp(Arch2, "PReP") == 0) {
      pleaseWaitBox(_("Copying the operating system kernel..."));
      snprintf(prtbuf, sizeof(prtbuf), BC_RUN_PREPDD, "/target/vmlinuz", device);
      status = execlog(prtbuf, LOG_INFO);
      boxPopWindow();
      if (status) {
	  eject_floppy(device);
	  return 1;
      }
      else return 0;
  }
#endif

#ifdef SCSI_FLOPPY
  if (strncmp(device, "/dev/sfd", 8)) /* not scsi floppy */
#endif
#if 0 /*PETA*/
    fdflush(device);
#else
    {
       int fd = open(device, 0);

       if (fd) {
	 ioctl(fd, FDFLUSH, 0);
	 close(fd);
       }
    }
#endif

  /*
   * get floppy geometry
   *
   * This code is taken from getfdprm, fdutils-4.3 by Alain Knaff
   * Alain.Knaff@inrialpes.fr, released under GPL.
   */

  fd = open(device, 3);
  if ( fd < 0 )
    fd = open(device, O_RDONLY);
  if ( fd < 0 ) {
    ERRMSG("error opening %s (%d)", device, fd);
    return 1;
  }
  
  if (ioctl(fd, FDGETPRM, &floppy_type) < 0) {
    close(fd);
    ERRMSG("error getting floppy geometry from %s", device);
    return 1; 
  }
  close(fd);

  /*
   * End of the code taken from getfdprm.c in fdutils.
   */

  size = floppy_type.size / 2;
  INFOMSG("write_boot_floppy: found floppy of size %d", size);

  /* We don't have (yet) 2.88 MB bootable floppy images to use with syslinux */
  if (size == 2880) size = 1440;


  if (stat(BC_SFORMAT, &tmp) == 0) {
    boxSuspend();
    printf(CLEAR);
    printf(_("Formatting the floppy...\n"));
    /* We don't have (yet) 2.88 MB bootable floppy ... */
    snprintf(prtbuf, sizeof(prtbuf), BC_RUN_SFORMAT, device,
             (size > 720) ? "hd" : "dd");
    /* DO NOT USE execlog() */
    INFOMSG("formatting floppy with cmd '%s'", prtbuf);
    status = system(prtbuf);
    boxResume();
    if (status) return 1;
  }
    
  pleaseWaitBox(_("Creating a filesystem on the floppy..."));

#if #cpu (m68k)
  if (strcmp(Arch2, "Atari") == 0)
     snprintf(prtbuf, sizeof(prtbuf), "zcat %s/usr/lib/atari-bootstrap/img%d%s.gz > %s",
	     BC_DIR_PREFIX,size,Arch2,device);
#elif #cpu (i386)
  snprintf(prtbuf, sizeof(prtbuf), "zcat %s/usr/lib/syslinux/img%dk.gz > %s",
	  BC_DIR_PREFIX,size,device);
#elif #cpu (sparc)
  /* read it from miniroot fs */
  snprintf(prtbuf, sizeof(prtbuf), "zcat /usr/lib/dbootstrap/silo%dk-%s.gz > %s",
	  size,Arch2,device);
#elif #cpu (alpha)
  /* XXX: maybe we should do the same thing as Sparc? (i.e. make a "template" image) */
  snprintf(prtbuf, sizeof(prtbuf), "/sbin/mkfs.ext2 -Onone %s >/dev/null 2>&1 && "
	   "/sbin/e2writeboot %s /boot/bootlx >/dev/null",
	   device, device);
#endif
  status = execlog(prtbuf, LOG_INFO);

  boxPopWindow();
  if (status)
    return 1;

  snprintf(prtbuf, sizeof(prtbuf), "mount -t %s %s /floppy", fs_type, device);
  status = execlog(prtbuf, LOG_INFO);
  if (status)
    return 1;

  pleaseWaitBox(_("Copying the operating system kernel..."));

  snprintf(prtbuf, sizeof(prtbuf), "cp %s/vmlinuz /floppy/linux", BC_DIR_PREFIX);
  status = execlog(prtbuf, LOG_INFO);

  if (status) {
    boxPopWindow();
    return 1;
  }

  sync();

  boxPopWindow();
  
  pleaseWaitBox(_("Writing the configuration files...\n"));
#if #cpu (m68k)
  if (strcmp(Arch2, "Atari") == 0) {
    if ((filep = fopen("/floppy/bootargs", "w")) == NULL) {
      execlog("umount /floppy", LOG_INFO);
      boxPopWindow();
      return 1;
    }
    fprintf(filep, "-s -k a:\\linux root=%s %s", Root->name, append_opts);
    fclose(filep);
  }
#elif #cpu (i386)
  if ((filep = fopen("/floppy/syslinux.cfg", "w")) == NULL) {
    execlog("umount /floppy", LOG_INFO);
    boxPopWindow();
    return 1;
  }
  fprintf(filep, "DISPLAY message.txt\n");
  fprintf(filep, "TIMEOUT 40\n");
  fprintf(filep, "PROMPT 1\n");
  fprintf(filep, "DEFAULT linux\n");
  fprintf(filep, "APPEND root=%s ro %s\n", Root->name, append_opts);
  fclose(filep);
#elif #cpu (sparc)
  if ((filep = fopen("/floppy/etc/silo.conf", "w")) == NULL) {
    execlog("umount /floppy", LOG_INFO);
    boxPopWindow();
    eject_floppy(device);
    return 1;
  }
  fprintf(filep, "partition=3\n");
  fprintf(filep, "root=%s\n", device);
  fprintf(filep, "timeout=100\n");
  fprintf(filep, "image=/linux\n");
  fprintf(filep, "label=linux\n");
  fprintf(filep, "read-only\n");
  fprintf(filep, "append=\"root=%s %s\"\n", Root->name, append_opts);
  fclose(filep);
  /* no need to run silo at this time; the floppy disk is already bootable */
#elif #cpu (alpha)
  if (mkdir("/floppy/etc", 0755) == -1) {
    execlog("umount /floppy", LOG_INFO);
    boxPopWindow();
    return 1;
  }
  if ((filep = fopen("/floppy/etc/aboot.conf", "w")) == NULL) {
    execlog("umount /floppy", LOG_INFO);
    boxPopWindow();
    return 1;
  }
  fprintf(filep, "0:linux ro root=%s %s\n", Root->name, append_opts);
  fclose(filep);
#endif
  
  if ((filep = fopen("/floppy/message.txt", "w")) != NULL) {
  /* FIXME The next message has to be DOS formatted for syslinux. Instead of
     printing it, what about copying it from /, like release_notes ? 
   */
  /* TRANS: The next message has to be DOS formatted for syslinux. */
    fprintf(filep, _("\nLinux will be started automatically using the kernel on this floppy disk.\nThe root filesystem will be mounted from %s .\n\nWhen you start to type a command at the \"boot:\" prompt, the automatic\nbootstrap process will stop. You may then type the name of the kernel to\nuse followed by any kernel options in the form option=value.\n\nThe kernel must be on the floppy disk in the first floppy drive. A kernel\nfile called \"linux\" was installed on this floppy when it was created. If you\nwish to use a kernel on the hard disk, remove the floppy and press RESET\n\n"), Root->name);
    fclose(filep);
    status = 0;
  } else 
    status = 1;
  
  boxPopWindow();
  execlog("umount /floppy", LOG_INFO);

  /* eject the floppy disk if avail (see dbootstrap.h) */
  eject_floppy(device);

  return status;
}

int make_boot_floppy (void)
{
  char* device = "/dev/fd0";
  char* designation = _("first");
  char* formatted = "";
  int result;
  struct stat tmp;
  int notCreatedBootFloppy_save;

  /* In case any of the problem boxes below return */
  notCreatedBootFloppy_save = notCreatedBootFloppy;
  notCreatedBootFloppy = 0;
#if #cpu (m68k)
  if (strcmp(Arch2, "Amiga") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for Amiga."),
               _("Problem"));
    return 0;
  }
  if (strcmp(Arch2, "VME") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for VME."),
               _("Problem"));
    return 0;
  }
  if (strcmp(Arch2, "Macintosh") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for the Macintosh."),
               _("Problem"));
    return 0;
  }
#endif

#if #cpu (powerpc)
  if (strcmp(Arch2, "apus") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for PowerPC/apus."),
               _("Problem"));
    return 0;
  }
  if (strcmp(Arch2, "CHRP") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for PowerPC/CHRP."),
               _("Problem"));
    return 0;
  }
  if (strcmp(Arch2, "PowerMac") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for PowerPC/PowerMac."), 
               _("Problem"));
    return 0;
  }
#endif

#if #cpu (alpha)
  if (! srm_boot) {
    /* Not SRM, no boot floppy */
    problemBox(_("Creating a boot floppy is still not possible for Alphas that boot from MILO"),
	       _("Problem"));
    return 0;
  }
#endif

  notCreatedBootFloppy = notCreatedBootFloppy_save;

  if (strcmp(InstallationRootDevice, "/dev/fd0") == 0) {
    snprintf(prtbuf, sizeof(prtbuf),
             _("You are not using the RAM-disk, and thus the Installation Root Disk in the first floppy drive cannot be removed.\nIf your second floppy drive is the same size as the first one, you can use it to write a boot floppy which you will place in the first drive when you boot the system. Otherwise, you can skip this step and use the Debian Rescue floppy to boot your hard disk by typing \"linux root=%s\" at the \"boot: \" prompt, or you may be able to boot directly from the hard disk.\nUse the second floppy drive to create a boot floppy?"),
             Root->name);
    if (yesNoBox(prtbuf, _("Use Second Floppy Drive?")) == DLG_NO)
      return 1;
    device = "/dev/fd1";
    designation = _("second");
  }

  if (stat(BC_SFORMAT, &tmp) != 0) {
    formatted = _(", formatted");
  }

  vaproblemBox(_("Change Disk"), 
               _("Please place a blank%s floppy disk in the %s floppy "
                 "disk drive, and press ENTER."),
               formatted, designation);

  result = write_boot_floppy(device);
  if (result != 0) {
      vaproblemBox(_("Problem"), 
                   _("Creation of the boot floppy failed. Please make sure that "
                     "the floppy was not write-protected, and that you put it "
                     "in the %s drive. Try another floppy if the problem persists."),
                   designation);
  } else {
    notCreatedBootFloppy = 0;
  }  
  return result;
}

static int firstnumber(const char* boot)
{
  int idx, ch;
  idx = strlen(boot);
  do {
    idx--;
    ch = boot[idx];
  } while ((ch >= '0') && (ch <= '9'));
  idx++;
  return idx;
}

#if #cpu (i386)
#define DEV_HDA      "/dev/hda"
#define DEV_SDA      "/dev/sda"
#define DEV_RD_C0D0  "/dev/rd/c0d0"
#define DEV_IDA_C0D0 "/dev/ida/c0d0"

/* part_disk_name: return name of partition's disk device, if available */
char *part_disk_name(struct fdisk_partition *part)
{
  return (part->disk ? part->disk->name : NULL);
}

/* is_first_scsi: return true if partition is on the first SCSI disk */
int is_first_scsi(struct fdisk_partition *part)
{
  char *dname = part_disk_name(part);
  return dname && (strcmp(dname, DEV_SDA)      == 0 ||
		   strcmp(dname, DEV_RD_C0D0)  == 0 ||
		   strcmp(dname, DEV_IDA_C0D0) == 0);
}

/* is_first_ide: return true if partition is on the first IDE disk */
int is_first_ide(struct fdisk_partition *part)
{
  char *dname = part_disk_name(part);
  return dname && strcmp(dname, DEV_HDA) == 0;
}

/* is_logical_part: return true if partition is an MS-DOS logical partition */
int is_logical_part(struct fdisk_partition *part)
{
  if (part->disk && strcmp(part->disk->partition_format, "msdos") == 0) {
    char *s, *t;

    assert(*part->name);
    /* find the first digit in the part->name, set t to the rest of the string */
    for (s = part->name, t = NULL; *s; s++) {
      if (isdigit(*s)) {
	t = s;
	break;
      }
    }
    if (t) {
      assert(strlen(t) <= 5);
      return(atoi(t) > 4);
    }
  }
  return 0;
}

/* !!!! strlen and sizeof are not the same thing. Please don't change this
 * again !!!!!! -randolph
 */
/* is_ide: returns true if partition is on an IDE disk */
int is_ide(const char *name)
{
  return strncmp(name, DEV_HDA, strlen(DEV_HDA) - 1) == 0;
}

/* is_scsi: returns true if partition is on a SCSI disk */
int is_scsi(const char *name)
{
  return strncmp(name, DEV_SDA, strlen(DEV_SDA) - 1) == 0;
}

/* is_scsird: returns true if partition is on a SCSI disk (/dev/rd/c*) */
int is_scsird(const char *name)
{
  return strncmp(name, DEV_RD_C0D0, strlen(DEV_RD_C0D0) - 3) == 0;
}
  
/* is_scsiida: returns true if partition is on a SCSI disk (/dev/rd/c*) */
int is_scsiida(const char *name)
{
  return strncmp(name, DEV_IDA_C0D0, strlen(DEV_IDA_C0D0) - 3) == 0;
}
#endif /* #cpu (i386) */


#ifdef USE_SILO
/* SILO is a special boot loader. It can actually read ext2, UFS and
 * iso9660 partitions. This means that it uses actual paths at boot time
 * as opposed to sectors/blocks like lilo does. Because of this we have to
 * take special care when "/boot" is not on the same partion as "/". So
 * how do we do this you say? We always set "partition" to Boot->name,
 * then if we find that /boot is a seperate partition, we muck around with
 * the vmlinuz symlinks to make everything happy. */
static int run_silo(void)
{
  FILE* filep;
  int idx;
  char *boot_part, *root_part;
  int retcode;

  if ((Boot = fdisk_find_partition_by_mntpoint ("/target/boot")) == NULL)
      Boot = Root;
  
  idx=firstnumber(Boot->name);
  boot_part=Boot->name+idx;
  idx=firstnumber(Root->name);
  root_part=Root->name+idx;

  pleaseWaitBox(_("Running SILO to make the kernel bootable from the hard disk without a boot floppy..."));
  if((filep=fopen(BC_SILO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }

  fprintf(filep,"partition=%s\n",boot_part);
  fprintf(filep,"root=%s\n",Root->name);
  fprintf(filep,"timeout=100\n");
  /* This is where it get's tricky */
  if (Boot != Root) {
      unlink("/target/vmlinuz"); /* does us no good */
      unlink("/target/boot/vmlinuz"); /* make sure this is gone */
      symlink("vmlinuz-" KVER, "/target/boot/vmlinuz");
  }
  fprintf(filep,"image=%s/vmlinuz\n", boot_part); /* just to make sure the partition is right */
  fprintf(filep,"label=linux\nread-only\n");
  fclose(filep);
  sleep(1);		/* wait some amount of time to see the message box */
  retcode = (execlog(BC_RUN_SILO, LOG_INFO) == 0) ? 0 : 1;
  boxPopWindow();
  return retcode;
}
#endif

#ifdef USE_PLILO  /* PowerPC-LILO: for the moment it is QUIK */
static int run_plilo(char *boot)
{
  FILE* filep;
  int idx;
  char *device, *partition;
  char vmlinux[256];
  struct stat statbuf;

  idx=firstnumber(boot);
  device=strdup(boot);
  device[idx]='\0';
  partition=strdup(boot+idx);

  /* potato run_plilo doesn't work on NewWorld Macs. this is gross, who cares -eb */
  if (!execlog("cat /proc/cpuinfo | grep -q pmac-generation", LOG_INFO))
      if (!execlog("cat /proc/cpuinfo | grep -q NewWorld", LOG_INFO)) {
	  problemBox(_("Please read the installation documentation.\n\n"
		       "yaboot cannot be installed in potato.\n\n"
		       "The installation documentation explains how to run mkofboot "
		       "from a shell so you may install yaboot manually."),
		       _("Cannot Install Yaboot"));
	  return 1;
      }

  /* 
     Translate /vmlinux symbolic link to real kernel 
     since apparently quik doesn't like the link.  backported from woody -eb
  */
  if (NAME_ISLNK("/target/vmlinux", &statbuf)) {
      int count;

      if ((count = readlink("/target/vmlinux", vmlinux, sizeof(vmlinux))) != -1) {
          vmlinux[count] = '\0';
      } else {
          ERRMSG("Unable to resolve /target/vmlinux: %s", strerror(errno));
          boxPopWindow();
          problemBox(_("The requested operation cannot be performed because "
                       "the /target/vmlinux symlink could not be resolved."), 
                     _("Need Valid Symlink"));
          return 1;
      }
  } else
      strcpy(vmlinux, "vmlinux");

  pleaseWaitBox(_("Running QUIK to make the kernel able to boot from the hard disk without a boot floppy..."));
  if((filep=fopen(BC_QUIK_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"partition=%s\n",partition);
  fprintf(filep,"root=%s\n",Root->name);
  fprintf(filep,"timeout=100\n\n");
  fprintf(filep,"image=/%s\n", vmlinux);
  fprintf(filep,"\tlabel=Linux\n");
  fprintf(filep,"\tread-only\n");
  fclose(filep);
  sleep(3);		/* wait some amount of time to see the message box */
  if(strlen(boot)==0) return 1;
  if(execlog(BC_RUN_QUIK, LOG_INFO)) return 1;
  boxPopWindow();
  free(partition);
  free(device);
  problemBox(_("You must still configure OpenFirmware yourself to boot QUIK.\n\n"
	       "Your system will not yet boot Debian directly from the hard disk."), 
	     _("Notice"));
  return 0;
}
#endif

#ifdef USE_LILOM68K
static int run_vmelilo(const char *boot)
{
  FILE* filep;

  pleaseWaitBox(_("Running VMELILO to make the kernel able to boot from the hard disk without a boot floppy..."));
  if((filep=fopen(BC_VMELILO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"default   = Linux\n");
  fprintf(filep,"boot      = %s\n", boot);
  fprintf(filep,"delay     = 2\n");
  fprintf(filep,"\n");
  fprintf(filep,"[boot]\n");
  fprintf(filep,"label     = Linux\n");
  fprintf(filep,"image     = /vmlinuz\n");
  fprintf(filep,"root      = %s\n", Root->name);
  fprintf(filep,"read-only\n");
  fclose(filep);
  sleep(1);		/* wait some amount of time to see the message box */
  boxPopWindow();
  if(strlen(boot)==0) return 1;
  if(execlog(BC_RUN_VMELILO, LOG_INFO)) return 1;
  return 0;
}
#endif

#ifdef USE_ABOOT /* Alpha equivalent to the above (for SRM only though) */
static int run_aboot(const char *boot)
{
  FILE *filep;
  int  idx = firstnumber(boot);
  const char *partition = boot + idx;

  /* write aboot.conf anyway since it's basically harmless */
  if ((filep = fopen(BC_ABOOT_CONF, "w")) == NULL) {
    problemBox(_("Could not write to /etc/aboot.conf"), _("Problem"));
    return 1;
  }
  /* our aboot handles symbolic links so 'boot/vmlinuz' is just fine */
  if (serialConsole >= 0) {
    fprintf(filep, "0:%s/vmlinuz ro root=%s console=ttyS%d\n",
	    partition, Root->name, serialConsole);
  } else {
    fprintf(filep, "0:%s/vmlinuz ro root=%s\n", partition, Root->name);
  }
  fclose(filep);

  if (strlen(boot) == 0) return 1;
  /* Since running aboot can be harmful, ask first */
  if (! bootargs.isquiet ) {
    sprintf(prtbuf,_("You have chosen to write the aboot bootloader to the boot sectors of %s.  This will destroy your ability to boot any other operating systems on this disk. Are you sure you want to do this?"),
	    boot);
    if (yesNoBox(prtbuf, _("Are You Sure?")) == DLG_NO)
      return 0;
  }
  pleaseWaitBox(_("Writing aboot to the first sector of the hard disk..."));
  sleep(1);
  boxPopWindow();
  {
    char *device = strdup(boot);
    device[idx] = '\0';
    snprintf(prtbuf, sizeof(prtbuf), BC_RUN_SWRITEBOOT, device);
    if (execlog(prtbuf, LOG_INFO)) return 1;
    snprintf(prtbuf, sizeof(prtbuf), BC_RUN_ABOOTCONF, device, partition);
    if (execlog(prtbuf, LOG_INFO)) return 1;
  }
  return 0;
}

static int mount_dos_fs(const char* dev, const char* mount)
{
  int status;

  do {
    sprintf(prtbuf, "cat /proc/mounts | grep -q %s", dev);
    if (!system(prtbuf)) {
      sprintf(prtbuf, "umount %s", dev);
      execlog(prtbuf, LOG_DEBUG);
    }
    sprintf(prtbuf, "cat /proc/mounts | grep -q %s", mount);
    if (!system(prtbuf)) {
      sprintf(prtbuf, "umount %s", mount);
      execlog(prtbuf, LOG_DEBUG);
    }
    sprintf(prtbuf, "mount -t msdos %s %s", dev, mount);
    status = execlog(prtbuf, LOG_INFO);
    if (status) {
      int answer;
      /* Offer to initialize it if the mount fails */
      sprintf(prtbuf, _("Failed to mount %s as a FAT partition.  "
			"This could simply be because it is not yet "
			"formatted as one.  Would you like to format "
			"it now?"), dev);
      answer = twoButtonBox(prtbuf, _("Initialize FAT partition?"),
			    _("Format"), _("Cancel"));
      if (answer == DLG_NO)
	return -1;
      sprintf(prtbuf, "mkdosfs %s", dev);
      if (execlog(prtbuf, LOG_INFO)) {
	sprintf(prtbuf, "Failed to make a FAT filesystem on %s", dev);
	problemBox(prtbuf, _("FAT filesystem init failed"));
	return -1;
      }
    }
  } while (status);

  return 0;
}

/* mount_and_check_floppy() imposes too much policy on us */
static int just_mount_the_damn_floppy(char *message)
{
  const char **fs_type;

  if (!system("cat /proc/mounts | grep -q /floppy"))
    execlog("umount /floppy", LOG_DEBUG);
  if (twoButtonBox(message, _("Please Insert Disk"),
		   _("Continue"), _("Cancel")) == DLG_NO)
    return -2;
  for (fs_type = fs_type_tab; *fs_type; fs_type++) {
    sprintf(prtbuf, "mount -r -t %s %s /floppy",
	    *fs_type, Archive_Dir);
    if (execlog(prtbuf, LOG_INFO) == 0) /* success */
      return 0;
  }
  return -1;
}

static int install_milo(const char* boot, struct fdisk_partition* p)
{
  FILE *filep;
  const char* bootdev;
  char *milo_path = NULL, *linload_path = NULL;
  struct stat statbuf;
  int status;

  /* skip past the "/dev/" */
  bootdev = strrchr(boot, '/');
  if (bootdev == NULL) {
    /* panic! */
    return 1;
  }
  bootdev++;

  /* write milo.conf */
  if ((filep = fopen(BC_MILO_CONF, "w")) == NULL) {
    problemBox(_("Could not write to /etc/milo.conf"), _("Problem"));
    return 1;
  }
  fprintf(filep,
	  "# MILO configuration, created by the Debian installer\n"
	  "timeout=100\n"
	  "image=%s:vmlinuz\n"
	  "root=%s\n",
	  bootdev, Root->name);
  if (serialConsole >= 0) {
    fprintf(filep, "append=\"console=ttyS%d %s\"\n",
	    serialConsole, append_opts);
  } else if (strlen (append_opts) > 0) {
    fprintf(filep, "append=\"%s\"\n", append_opts);
  }
  fclose(filep);

  /* This is totally fucking broken global variable abuse */
  disqtype = milo;
  if (choose_medium())
    return 1;

  if (strcmp(Archive_Dir, "netfetch") == 0) {
    /* punt for now, because netfetch can't do what we want */
    problemBox(_("Installing MILO from the network is not yet supported"),
	       _("Problem"));
    return 1;
  } else if (strncmp(Archive_Dir, "/dev/fd", 7) == 0) {
    /* again, mount_and_check_floppy() can't do what we want */
    const char **path;
    const char *paths[] = { "/floppy/%s",
			    "/floppy/milo_%s",
			    "/floppy/milo", NULL };

    for (;;) {
      sprintf(prtbuf,
	      _("Place the MILO disk you used to boot, or "
		"a floppy disk containing `linload.exe' and a MILO binary "
		/* XXX bad internationalization (word order) */
		"named `%s', in the %s floppy drive."),
	      Arch2, ((Archive_Dir[7] == '0')
		      ? _("first") : _("second")));
      if (just_mount_the_damn_floppy(prtbuf)) {
	problemBox(_("Unable to mount the floppy.  You may have "
		     "inserted the wrong floppy.  Please try again.")
		   , _("Mount Failed"));
	continue;
      }
      /* Try to find the MILO binary */
      for (path = paths; *path; path++) {
	sprintf(prtbuf, *path, Arch2);
	DEBUGMSG("looking for %s", prtbuf);
	if (NAME_ISREG(prtbuf, &statbuf)) {
	  milo_path = strdup(prtbuf);
	  break;
	}
      }
      if (!*path) {
	sprintf(prtbuf, 
		_("Unable to find a MILO binary on the floppy "
		  "(tried `%s', `milo_%s', and `milo').  Please "
		  "try again."), Arch2, Arch2);
	problemBox(prtbuf, _("MILO not found"));
	continue;
      }
      if (!NAME_ISREG("/floppy/linload.exe", &statbuf)) {
	problemBox(_("Unable to find `linload.exe' on the floppy.  "
		     "Please try again."), _("linload.exe not found"));
	continue;
      }
      DEBUGMSG("found milo and linload.exe");
      break;
    }
    linload_path = strdup("/floppy/linload.exe");
  } else {
    /* This is all mostly useless, since choose_medium() takes the
       liberty (GAR GAR GAR) of doing most of it for us in a horrible
       and broken way.  But we still need to do it because the MILO
       could be in ${Arch2} or in MILO/${Arch2}, and we don't know
       which. */
    const char **path;
    const char *paths[] = { "%s/MILO/%s", /* CD or NFS */
			    "%s/%s",
			    "%s/milo_%s",
			    "%s/milo", NULL };
    const char *linload_paths[] = { "%s/MILO/linload.exe",
				    "%s/linload.exe", NULL };

    /* Try to find the MILO binary */
    for (path = paths; *path; path++) {
      sprintf(prtbuf, *path, Archive_Dir, Arch2);
      DEBUGMSG("looking for %s", prtbuf);
      if (NAME_ISREG(prtbuf, &statbuf)) {
	milo_path = strdup(prtbuf);
	break;
      }
    }
    if (!*path) {
      sprintf(prtbuf, 
	      _("Unable to find a MILO binary in %s "
		"(tried `MILO/%s', `%s', `milo_%s', and `milo')."),
	      Archive_Dir, Arch2, Arch2);
	problemBox(prtbuf, _("MILO not found"));
	goto error_out;
    }
    /* and linload.exe */
    for (path = linload_paths; *path; path++) {
      sprintf(prtbuf, *path, Archive_Dir);
      DEBUGMSG("looking for %s", prtbuf);
      if (NAME_ISREG(prtbuf, &statbuf)) {
	linload_path = strdup(prtbuf);
	break;
      }
    }
    if (!*path) {
      sprintf(prtbuf, 
	      _("Unable to find `linload.exe' in %s "
		"(tried `MILO/linload.exe' and `linload.exe')."),
	      Archive_Dir);
	problemBox(prtbuf, _("linload.exe not found"));
	goto error_out;
    }
  }
  /* Mount the filesystem */
  if (mount_dos_fs(p->name, "/instmnt"))
    goto error_out;

  /* Now copy MILO and linload.exe */
  pleaseWaitBox(_("Copying MILO and linload.exe..."));
  sprintf(prtbuf, "cp %s /instmnt/milo", milo_path);
  status = execlog(prtbuf, LOG_INFO);
  if (status) {
    problemBox(_("Couldn't copy MILO to the FAT partition"),
	       _("MILO install failed"));
    goto error_out;
  }
  sprintf(prtbuf, "cp %s /instmnt", linload_path);
  status = execlog(prtbuf, LOG_INFO);
  if (status) {
    problemBox(_("Couldn't copy linload.exe to the FAT partition"),
	       _("MILO install failed"));
    goto error_out;
  }

#if MILOCTL_ON_ROOTDISK /* The MILO utilities are not packaged yet,
			   and I don't want to kludge it in */
  /* Now run miloctl on the MILO binary */
  sprintf(prtbuf, BC_RUN_MILOCTL, bootdev);
  status = execlog(prtbuf, LOG_INFO);
  if (status) {
    problemBox(_("Couldn't run miloctl on the installed MILO.  "
		 "MILO will not boot automatically when started."),
	       _("miloctl failed"));
    goto error_out;
  }
#endif
  
  return 0;

 error_out:
  if (milo_path) free(milo_path);
  return 1;
}

static int install_apb(const char *boot, struct fdisk_partition *p)
{
  struct stat statbuf;
  FILE *f;

  /* This is kind of suboptimal, as we really want to mount the DOS
     partition as /boot or something like that.  Anyhow we will try to
     mount the filesystem first, then write apb.cfg and up1000.pal,
     then copy the kernel to it.  */
  if (mkdir("/target/dos", 0755)) {
    if (errno != EEXIST)
      return 1;
    if (!NAME_ISDIR("/target/dos", &statbuf))
      return 1;
  }
  if (mount_dos_fs(p->name, "/target/dos"))
    return 1;
  if ((f = fopen("/target/etc/fstab", "a")) == NULL)
    return 1;
  fprintf(f, "%s /dos vfat defaults 0 0\n", p->name);
  fclose(f);

  if ((f = fopen(BC_APB_CFG, "w")) == NULL) {
    problemBox(_("Could not write to /dos/apb.cfg"), _("Problem"));
    return 1;
  }
  if (serialConsole >= 0)
    fprintf(f, "debian|bootl up1000.pal vmlinuz root=%s console=ttyS%d\n",
	    Root->name, serialConsole);
  else
    fprintf(f, "debian|bootl up1000.pal vmlinuz root=%s\n", Root->name);
  fclose(f);

  disqtype = apb;
  if (choose_medium())
    return 1;

  if (strcmp(Archive_Dir, "netfetch") == 0) {
    /* punt for now, because netfetch can't do what we want */
    problemBox(_("Installing APB from the network is not yet supported"),
	       _("Problem"));
    return 1;
  } else if (strncmp(Archive_Dir, "/dev/fd", 7) == 0) {
    /* this is easy, thankfully */
    mount_and_check_floppy(Archive_Dir, "rescue", _("Rescue Floppy"));
    strcpy(Archive_Dir, "/floppy");
  } else {
    /* ?? */
  }

  pleaseWaitBox(_("Copying APB and up1000.pal..."));
  sprintf(prtbuf, "cp %s/apb.exe /target/dos", Archive_Dir);
  if (execlog(prtbuf, LOG_INFO)) {
    problemBox(_("Couldn't copy APB to the FAT partition"),
	       _("APB install failed"));
    return 1;
  }
  sprintf(prtbuf, "cp %s/up1000.pal /target/dos", Archive_Dir);
  if (execlog(prtbuf, LOG_INFO)) {
    problemBox(_("Couldn't copy up1000.pal to the FAT partition"),
	       _("APB install failed"));
    return 1;
  }
  if (execlog("cp /target/boot/vmlinuz /target/dos", LOG_INFO)) {
    problemBox(_("Couldn't copy the installed kernel to the FAT partition"),
	       _("APB install failed"));
    return 1;
  }

  return 0;
}
#endif /* USE_ABOOT */

#ifdef USE_LILO
/*
 * Note:  To modify this template, please use `bootconfig-convert-
 *        lilotemplate', and follow the instructions in the docstring.
 */

/*
 * # /etc/lilo.conf - See: `lilo(8)' and `lilo.conf(5)',
 * # ---------------       `install-mbr(8)', `/usr/share/doc/lilo/',
 * #                       and `/usr/share/doc/mbr/'.
 *
 * # +---------------------------------------------------------------+
 * # |                        !! Reminder !!                         |
 * # |                                                               |
 * # | Don't forget to run `lilo' after you make changes to this     |
 * # | conffile, `/boot/bootmess.txt', or install a new kernel.  The |
 * # | computer will most likely fail to boot if a kernel-image      |
 * # | post-install script or you don't remember to run `lilo'.      |
 * # |                                                               |
 * # +---------------------------------------------------------------+
 *
 * # Support LBA for large hard disks.
 * #
 * lba32
 *
 * # Specifies the boot device.  This is where Lilo installs its boot
 * # block.  It can be either a partition, or the raw device, in which
 * # case it installs in the MBR, and will overwrite the current MBR.
 * #
 * boot=%s
 *
 * # Specifies the device that should be mounted as root. (`/')
 * #
 * root=%s
 *
 * # Enable map compaction:
 * # Tries to merge read requests for adjacent sectors into a single
 * # read request. This drastically reduces load time and keeps the
 * # map smaller.  Using `compact' is especially recommended when
 * # booting from a floppy disk.  It is disabled here by default
 * # because it doesn't always work.
 * #
 * # compact
 *
 * # Installs the specified file as the new boot sector
 * #
 * install=/boot/boot.b
 *
 * # Specifies the location of the map file
 * #
 * map=/boot/map
 *
 * # You can set a password here, and uncomment the `restricted' lines
 * # in the image definitions below to make it so that a password must
 * # be typed to boot anything but a default configuration.  If a
 * # command line is given, other than one specified by an `append'
 * # statement in `lilo.conf', the password will be required, but a
 * # standard default boot will not require one.
 * #
 * # This will, for instance, prevent anyone with access to the
 * # console from booting with something like `Linux init=/bin/sh',
 * # and thus becoming `root' without proper authorization.
 * #
 * # Note that if you really need this type of security, you will
 * # likely also want to use `install-mbr' to reconfigure the MBR
 * # program, as well as set up your BIOS to disallow booting from
 * # removable disk or CD-ROM, then put a password on getting into the
 * # BIOS configuration as well.  Please RTFM `install-mbr(8)'.
 * #
 * # password=tatercounter2000
 *
 * # Specifies the number of deciseconds (0.1 seconds) LILO should
 * # wait before booting the first image.
 * #
 * delay=20
 *
 * # You can put a customized boot message up if you like.  If you use
 * # `prompt', and this computer may need to reboot unattended, you
 * # must specify a `timeout', or it will sit there forever waiting
 * # for a keypress.  `single-key' goes with the `alias' lines in the
 * # `image' configurations below.  eg: You can press `1' to boot
 * # `Linux', `2' to boot `LinuxOLD', if you uncomment the `alias'.
 * #
 * # message=/boot/bootmess.txt
 * #	prompt
 * #	single-key
 * #	delay=100
 * #	timeout=100
 *
 * # Specifies the VGA text mode at boot time. (normal, extended, ask, <mode>)
 * #
 * # vga=ask
 * # vga=9
 * #
 * vga=normal
 *
 * # Kernel command line options that apply to all installed images go
 * # here.  See: The `boot-prompt-HOWO' and `kernel-parameters.txt' in
 * # the Linux kernel `Documentation' directory.
 * #
 * # append=""
 * %s
 * # Boot up Linux by default.
 * #
 * default=Linux
 *
 * image=/vmlinuz
 * 	label=Linux
 * 	read-only
 * #	restricted
 * #	alias=1
 *
 * image=/vmlinuz.old
 * 	label=LinuxOLD
 * 	read-only
 * 	optional
 * #	restricted
 * #	alias=2
 *
 * # If you have another OS on this machine to boot, you can uncomment the
 * # following lines, changing the device name on the `other' line to
 * # where your other OS' partition is.
 * #
 * # other=/dev/hda4
 * #	label=HURD
 * #	restricted
 * #	alias=3
 */

#define BC_LILOTEMPLATE1 \
"# /etc/lilo.conf - See: `lilo(8)' and `lilo.conf(5)',\n" \
"# ---------------       `install-mbr(8)', `/usr/share/doc/lilo/',\n" \
"#                       and `/usr/share/doc/mbr/'.\n" \
"\n" \
"# +---------------------------------------------------------------+\n" \
"# |                        !! Reminder !!                         |\n" \
"# |                                                               |\n" \
"# | Don't forget to run `lilo' after you make changes to this     |\n" \
"# | conffile, `/boot/bootmess.txt', or install a new kernel.  The |\n" \
"# | computer will most likely fail to boot if a kernel-image      |\n" \
"# | post-install script or you don't remember to run `lilo'.      |\n" \
"# |                                                               |\n" \
"# +---------------------------------------------------------------+\n" \
"\n" \
"# Support LBA for large hard disks.\n" \
"#\n" \
"lba32\n" \
"\n" \
"# Specifies the boot device.  This is where Lilo installs its boot\n" \
"# block.  It can be either a partition, or the raw device, in which\n" \
"# case it installs in the MBR, and will overwrite the current MBR.\n" \
"#\n" \
"boot=%s\n" \
"\n" \
"# Specifies the device that should be mounted as root. (`/')\n" \
"#\n" \
"root=%s\n" \
"\n" \
"# Enable map compaction:\n" \
"# Tries to merge read requests for adjacent sectors into a single\n" \
"# read request. This drastically reduces load time and keeps the\n" \
"# map smaller.  Using `compact' is especially recommended when\n" \
"# booting from a floppy disk.  It is disabled here by default\n" \
"# because it doesn't always work.\n" \
"#\n" \
"# compact\n" \
"\n" \
"# Installs the specified file as the new boot sector\n" \
"#\n" \
"install=/boot/boot.b\n" \
"\n" \
"# Specifies the location of the map file\n" \
"#\n" \
"map=/boot/map\n" \
"\n" \
"# You can set a password here, and uncomment the `restricted' lines\n" \
"# in the image definitions below to make it so that a password must\n" \
"# be typed to boot anything but a default configuration.  If a\n" \
"# command line is given, other than one specified by an `append'\n" \
"# statement in `lilo.conf', the password will be required, but a\n" \
"# standard default boot will not require one.\n" \
"#\n" \
"# This will, for instance, prevent anyone with access to the\n" \
"# console from booting with something like `Linux init=/bin/sh',\n" \
"# and thus becoming `root' without proper authorization.\n" \
"#\n" \
"# Note that if you really need this type of security, you will\n" \
"# likely also want to use `install-mbr' to reconfigure the MBR\n" \
"# program, as well as set up your BIOS to disallow booting from\n" \
"# removable disk or CD-ROM, then put a password on getting into the\n" \
"# BIOS configuration as well.  Please RTFM `install-mbr(8)'.\n" \
"#\n" \
"# password=tatercounter2000\n" \
"\n" \
"# Specifies the number of deciseconds (0.1 seconds) LILO should\n" \
"# wait before booting the first image.\n" \
"#\n" \
"delay=20\n" \
"\n" \
"# You can put a customized boot message up if you like.  If you use\n" \
"# `prompt', and this computer may need to reboot unattended, you\n" \
"# must specify a `timeout', or it will sit there forever waiting\n" \
"# for a keypress.  `single-key' goes with the `alias' lines in the\n" \
"# `image' configurations below.  eg: You can press `1' to boot\n" \
"# `Linux', `2' to boot `LinuxOLD', if you uncomment the `alias'.\n" \
"#\n" \
"# message=/boot/bootmess.txt\n" \
"#	prompt\n" \
"#	single-key\n" \
"#	delay=100\n" \
"#	timeout=100\n" \
"\n" \
"# Specifies the VGA text mode at boot time. (normal, extended, ask, <mode>)\n" \
"#\n" \
"# vga=ask\n" \
"# vga=9\n" \
"#\n" \
"vga=normal\n" \
"\n" \
"# Kernel command line options that apply to all installed images go\n" \
"# here.  See: The `boot-prompt-HOWO' and `kernel-parameters.txt' in\n" \
"# the Linux kernel `Documentation' directory.\n" \
"#\n" \
"# append=\"\"\n" \
"%s\n" \
"# Boot up Linux by default.\n" \
"#\n" \
"default=Linux\n" \
"\n" \
"image=/vmlinuz\n" \
"	label=Linux\n" 

#define BC_LILOTEMPLATE2 \
"   initrd=/boot/initrd.debinstall\n"

#define BC_LILOTEMPLATE3 \
"	read-only\n" \
"#	restricted\n" \
"#	alias=1\n" \
"\n" \
"image=/vmlinuz.old\n" \
"	label=LinuxOLD\n" \
"	read-only\n" \
"	optional\n" \
"#	restricted\n" \
"#	alias=2\n" \
"\n" \
"# If you have another OS on this machine to boot, you can uncomment the\n" \
"# following lines, changing the device name on the `other' line to\n" \
"# where your other OS' partition is.\n" \
"#\n" \
"# other=/dev/hda4\n" \
"#	label=HURD\n" \
"#	restricted\n" \
"#	alias=3\n"



static char * get_ramdisk_size(char *files[25])
{
	DIR * dir;
	struct dirent * file;
	struct stat statbuf;
	unsigned int size = 0;
	int i = 0;
	char * ascii_size;
	char * tmp = NULL;
	
	dir = opendir(MOD_SAVE_DIR);
	/* hopefully only one or two modules in this dir */
	file = readdir(dir);
	while ( file != NULL )
	{
		if ( *(file->d_name) == '.' )
		{
			file = readdir(dir);
			continue;
		}
		tmp = realloc(tmp,sizeof(MOD_SAVE_DIR) + sizeof("/") + sizeof(file->d_name));
		strcpy(tmp,MOD_SAVE_DIR "/");
		strcat(tmp,file->d_name);
		stat(tmp,&statbuf);
		size += statbuf.st_size;
		file = readdir(dir);
	}
	free(tmp);
	closedir(dir);
	size += 4096; /* big buffer for the linuxrc shell script */
	while ( strcmp(files[i],"") != 0 )
	{
		stat(files[i],&statbuf);
		size += statbuf.st_size;
		++i;
	}
	/* for directories and device*/
	size += (4096 * 8);
	size /= 1024;
	size += 30; /* trust no one */
	ascii_size = malloc(5 * sizeof(char));
	snprintf(ascii_size,5,"%u",size);
	return(ascii_size);
}

static char * get_device_number(char * device)
{
	unsigned char major, minor;
	char *number;
#if 1
	unsigned int temp;
	char *command;
	FILE *pipe;

	/* FIXME: Can we do something better ? */
	/* we get the major number of the device */
	command = malloc(sizeof("expr substr \"`ls -l ") + strlen(device) + sizeof("`\" 35 2"));
	if ( command == NULL )
	{
		problemBox(_("Can't allocate memory"),_("Critical Error"));
		exit(1);
	}
	strcpy(command,"expr substr \"`ls -l ");
	strcat(command,device);
	strcat(command,"`\" 35 2");
	pipe = popen(command,"r");
	if ( pipe == NULL )
	{
		problemBox(_("Can't open pipe"),_("Critical Error"));
		exit(1);
	}
	fscanf(pipe,"%u",&temp);
	major = (unsigned char) temp;
	pclose(pipe);
	/* we change the magic 35 into magic 40 
	   to get the minor number */
	command[strlen(command) - 3 ] = '0';
	command[strlen(command) - 4 ] = '4';
	pipe = popen(command,"r");
	if ( pipe == NULL )
	{
		problemBox(_("Can't open pipe"),_("Critical Error"));
		exit(1);
	}
	fscanf(pipe,"%u",&temp);
	minor = (unsigned char) temp;
	pclose(pipe);
	number = malloc(sizeof("0x") + 4);
	if ( number == NULL )
	{
		problemBox(_("Can't allocate memory"),_("Critical Error"));
		exit(1);
	}
    free (command); /* Free what we do not need any more */
#else
    struct stat buf;
	
    if (stat (device, &buf) == 0)
        if (S_ISCHR (buf.st_mode) || S_ISBLK (buf.st_mode))
        {
            major = MAJOR (buf.st_rdev);
            minor = MINOR (buf.st_rdev);
        }
        else
        {
            /* FIXME: should the file name be printed here? */
            problemBox (_("Specified file is not device", _("Critical Error")));

            exit (1);
        }
    else
    {
        problemBox (_("Cannot stat device", _("Critical Error")));

        exit (1);
    }
#endif
	sprintf(number,"0x%02u%02u",(unsigned int)major,(unsigned int)minor);

	return number;
}

#define FLOP_TMP_DIR "/tmp/mnt-flop-ramdisk"
#define FLOP_RAMDISK "/dev/ram8"

int create_ramdisk(char * device_number,char * size)
{
	 DIR *dir;
	 struct dirent * dirent;
	 char * tmp;
	 FILE * fd;
	 FILE * fd2;
	 char smp = 0;
	 char * command;
	 char buffer[512];
	 if ( yesNoBox(_("Do you plan to use SMP on this machine?"),_("Question")) == DLG_YES )
	 {
		 smp = 1;
	 }
	 
	 if ( mkdir(FLOP_TMP_DIR,0777) != 0 )
	 {
		 perrorBox(_("Can't mkdir"));
		 return 1;
	 }
	 /* we "zero" the device to enhance the compression */
	 command = malloc(sizeof("dd < /dev/zero > " FLOP_RAMDISK " bs=1k count=") + strlen(size));
	 if ( command == NULL )
	 {
		 problemBox(_("Cannot allocate memory. Stop"),_("Critical Error"));
	 	 return 1;
	 }
	 strcpy(command,"dd < /dev/zero > " FLOP_RAMDISK " bs=1k count=");
	 strcat(command,size);
	 if (execlog(command,LOG_INFO) )
	 {
		 problemBox(_("An error has occured during dd. Cannot continue"),_("Critical Error"));
		 return 1;
	 }
	 /* a minix fs would be better. But we don't have the tools available */
	 command = realloc(command,sizeof("/sbin/mkfs.ext2 -F -i 4096 -m 0 " FLOP_RAMDISK " ") + strlen(size));
	 strcpy(command,"/sbin/mkfs.ext2 -F -i 4096 -m 0 " FLOP_RAMDISK " ");
	 strcat(command,size);
	 if ( execlog(command,LOG_INFO) )
	 {
		 problemBox(_("An error has occured during mkfs.ext2. Cannot continue"),_("Critical Error"));
		 return 1;
	 }
	 /* FIXME:
	  	It would be better to do this with a system call
	 if ( mount(FLOP_RAMDISK,FLOP_TMP_DIR,"ext2",MS_MGC_VAL,"loop") ) does not work */
	 if ( execlog("mount -t ext2 -o loop " FLOP_RAMDISK " " FLOP_TMP_DIR,LOG_DEBUG ) )

	 {
		 perrorBox(_("Cannot mount:"));
		 return 1;
	 }
	 /* we create our tree */
	 rmdir(FLOP_TMP_DIR "/lost+found");
	 mkdir(FLOP_TMP_DIR "/dev",0777);
	 mkdir(FLOP_TMP_DIR "/proc",0777);
	 mkdir(FLOP_TMP_DIR "/lib",0777);
	 mkdir(FLOP_TMP_DIR "/bin",0777);
	 mkdir(FLOP_TMP_DIR "/sbin",0777);
	 mkdir(FLOP_TMP_DIR "/lib/modules",0777);
	 execlog("mknod " FLOP_TMP_DIR "/dev/tty1 c 4 1",LOG_DEBUG);
	 /* chmod 555 /FLOP_TMP_DIR/proc */
	 chmod(FLOP_TMP_DIR "/proc",S_IRUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
	 /* linuxrc is a file that is directly launched by initrd at boot
	  * so we gonna make a script that simply loads the modules */
	 fd = fopen(FLOP_TMP_DIR "/linuxrc","w");
	 if ( fd == NULL )
	 {
		perrorBox(_("Can't open file for writing:"));
		return 1;
	 }
	 fprintf(fd,"#!/bin/sh\n");

	 
	 dir = opendir(MOD_SAVE_DIR);
	 dirent = readdir(dir);
	 while ( dirent != NULL )
	 {
	 	tmp = strstr(dirent->d_name,".o");
		if ( (tmp) && tmp[2] == '\0' )
		{
			command = realloc(command,sizeof("cp ") + sizeof(MOD_SAVE_DIR) + sizeof("/")+ strlen(dirent->d_name) 
					+ sizeof(" " FLOP_TMP_DIR "/lib/modules"));
			strcpy(command,"cp ");
			strcat(command,MOD_SAVE_DIR);
			strcat(command,"/");
			strcat(command,dirent->d_name);
			strcat(command," " FLOP_TMP_DIR "/lib/modules");
			if ( execlog(command,LOG_DEBUG ) )
			{
				problemBox(_("Can't copy. Stop"),_("Critical Error"));
				return 1;
			}
			/*
			if ( smp )
			{
				fprintf(fd,"/sbin/insmod -f -P smp /lib/modules/%s\n",dirent->d_name);
			} else {
				fprintf(fd,"/sbin/insmod -f /lib/modules/%s\n",dirent->d_name);
			}
			*/
		}
		dirent = readdir(dir);
	 }
	 closedir(dir);
	 
	 fd2 = fopen(MOD_SAVE_FILE,"r");
	 if ( fd2 == NULL )
	 {
		 problemBox(_("Can't open the 'mod-save' file. Please report this bug"),_("Critical Error"));
		 return 1;
	 }
	 while (fgets(buffer,sizeof(buffer),fd2))
	 {
		if ( smp )
		{
				fprintf(fd,"/sbin/insmod -f -P smp /lib/modules/%s\n",buffer);
		} else {
				fprintf(fd,"/sbin/insmod -f /lib/modules/%s\n",buffer);
		}
	 } 
	 fclose(fd2);
	 fprintf(fd,"/bin/mount -t proc proc /proc\n");
	 fprintf(fd,"echo %s > /proc/sys/kernel/real-root-dev\n",device_number);
	 fprintf(fd,"/bin/umount /proc\n");
	 fclose(fd);
	 /* chmod 755 */
	 chmod(FLOP_TMP_DIR "/linuxrc",S_IRUSR | S_IXUSR | S_IWUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);

	 if (execlog("cp /sbin/insmod " FLOP_TMP_DIR "/sbin",LOG_DEBUG) )
	 {
		problemBox(_("Can't copy needed files"),_("Critical Error"));
		return 1;
	 }
	 if (execlog("cp /bin/mount /bin/umount /bin/sh " FLOP_TMP_DIR "/bin",LOG_DEBUG) )
	 {
		problemBox(_("Can't copy needed files"),_("Critical Error"));
		return 1;
	 }
	 if (execlog("cp /lib/ld-linux.so.2 /lib/libc.so.6 " FLOP_TMP_DIR "/lib",LOG_DEBUG) )
	 {
		problemBox(_("Can't copy needed files"),_("Critical Error"));
		return 1;
	 }
	 execlog("umount " FLOP_TMP_DIR,LOG_DEBUG);
	 rmdir(FLOP_TMP_DIR);
	 command = realloc(command,sizeof("dd < " FLOP_RAMDISK " > /target/boot/initrd.debinstall bs=1k count=") +
			 strlen(size));
	 strcpy(command,"dd < " FLOP_RAMDISK " > /target/boot/initrd.debinstall bs=1k count=");
	 strcat(command,size);
	 if ( execlog(command,LOG_DEBUG) )
	 {
		 problemBox("can't dd","Critical error");
		 return 1;
	 }
	 execlog("/target/bin/gzip -9f /target/boot/initrd.debinstall",LOG_DEBUG);
	 execlog("mv /target/boot/initrd.debinstall.gz /target/boot/initrd.debinstall",LOG_DEBUG);
	 sync(); /* I do not know if we need this, but this can't harm */
	 free(command);
	 return 0;
}	 
	 

static int run_lilo(char *boot)
{
  FILE* filep;
  int status = 0;
#ifndef _TESTING_
  struct stat statbuf;
  char * devnumber;
  char * size;
  /* files that we need */
  char  *files[]={
		"/sbin/insmod",
		"/bin/sh",
		"/bin/mount",
		"/bin/umount",
		"/lib/ld-linux.so.2",
		"/lib/libc.so.6",
		"" /* we use the NULL string to find the end */
  };
#endif
  char * append;
  
  pleaseWaitBox(_("Running LILO to make the kernel able to boot "
		  "from the hard disk without a boot floppy..."));

  if ((filep = fopen(BC_LILO_CONF, "w")) == NULL) {
    boxPopWindow();
#ifdef _TESTING_
    fprintf(stderr, "Cannot open %s: %s\n", BC_LILO_CONF, strerror(errno));
    sleep(2);
#endif
    return 1;
  }

  /* shouldn't be possible for name to be blank, but would explain Bug #53807 */
  assert(Root->name && Root->name[0]);
  assert(boot && boot[0]);

  INFOMSG("writing lilo.conf with boot of '%s' and root of '%s'",
          boot, Root->name);

  if (append_opts && strlen (append_opts)) {
      append = (char *)malloc (strlen (append_opts) + 1 + 10); /* "append=\"" + "\"\n"*/
      sprintf (append, "append=\"%s\"\n", append_opts);
      fprintf(filep, BC_LILOTEMPLATE1, boot, Root->name, append);
      free (append);
  } else {
      fprintf(filep, BC_LILOTEMPLATE1, boot, Root->name, "");
  }
#ifndef _TESTING_
  if ( NAME_ISDIR(MOD_SAVE_DIR,&statbuf) )
  {
	 devnumber = get_device_number(Root->name);
	 size = get_ramdisk_size(files);
	 if ( create_ramdisk(devnumber,size) )
	 {
		 return 1;
	 }
	 free(size);
	 free(devnumber);
	 fprintf(filep,BC_LILOTEMPLATE2);
  }
#endif
  fprintf(filep, BC_LILOTEMPLATE3);
  fclose(filep);
  chmod(BC_LILO_CONF, 0640);
  boxPopWindow();
#if !defined _TESTING_
  status = execlog(BC_RUN_LILO, LOG_INFO);
#endif
  return status;
}



static int install_mbr(const char* boot, int *installed_mbr)
{
  int status = 0;
  char *device = "";

  /* FIXME:  Check to make sure this is correct */
#if #cpu (i386)
  if (is_scsird(boot))
    device = DEV_RD_C0D0;
  else if (is_scsiida(boot)) 
    device = DEV_IDA_C0D0;
  else 
#endif
  {
    int idx;
    idx=firstnumber(boot);
    device=strdup(boot);
    device[idx]='\0';
  }

  snprintf(prtbuf, sizeof(prtbuf), _("A master boot record is required to boot the system.\nIf you are already using a boot manager, and want to keep it, answer \"No\" to the following question. If you don't know what a boot manager is or whether you have one, answer \"Yes\".\n\nInstall a master boot record on %s?"),device);
  if (bootargs.isquiet || 
      yesNoBox(prtbuf, _("Create Master Boot Record?")) == DLG_YES) {
    snprintf(prtbuf, sizeof(prtbuf), BC_INST_MBR, device);
    INFOMSG("installing mbr on %s", device);
#ifndef _TESTING_
    status = execlog(prtbuf, LOG_INFO);
#else
    status = 0;
#endif
    if (status == 0)
      *installed_mbr = 1;

    if (! bootargs.isquiet)
	problemBox(_("The master boot record program that was just installed has features not found in standard MBR programs.  A prompt can be triggered on startup by holding down the shift key. It shows the active partition, followed by the letters `F', for floppy, and `A', for advanced. You can press a number to boot the active partition, or `F' to boot from the floppy drive. Pressing `A' gets you a new prompt, this time listing all of the primary partitions on the drive, not just the active ones. By default, if no key was pressed, it boots into the first active partition on your harddisk. For more information about this MBR program, and how you can customize its configuration, please read the 'install-mbr' manual page."), _("Important Information about the installed MBR"));
  }
  return status;
}

static int set_boot_default(const char* boot)
{
  int idx, status = 0;
  int rootnum;
  char *device, *partition;

  idx         = firstnumber(boot);
  device      = strdup(boot);
  rootnum     = atoi(&device[idx]);
  device[idx] = '\0';
  partition   = strdup(boot + idx);
  snprintf(prtbuf, sizeof(prtbuf), _("If you want the Debian system to boot automatically from the hard disk when you turn your system on, answer \"Yes\" to the following question.\nIf you have another operating system that you'd prefer to be the one that boots automatically, answer \"No\".\nBoot the Debian system on %s as the default?"), boot);
  if (yesNoBox(prtbuf,_("Make Linux the Default Boot Partition?")) == DLG_YES) {
    INFOMSG("making linux partition '%s' the default for device '%s'",
            partition, device);
    snprintf(prtbuf, sizeof(prtbuf), BC_BOOT_DFLT, device, partition);
    status = execlog(prtbuf, LOG_INFO);
  } else {
    snprintf(prtbuf, sizeof(prtbuf), _("You have selected not to boot Debian as the default. To boot into Debian, you will need to hold down the SHIFT key on startup. When you see a prompt that says 123F: (the numbers may be different, depending on your configuration) enter %d (the root partition of your Debian installation). This will only work if you have previously chosen to install the master boot record."), rootnum);
    problemBox(prtbuf, _("How to boot into Debian"));
  }
  free(device);
  free(partition);
  return status;
}
#endif /* USE_LILO */


int make_bootable(void) 
{
  int status = 0;
  char *boot;

#if #cpu (i386)
  int installed_mbr = 0;
  char *device = "";
  struct d_choices opt[2];
  struct stat statbuf;
  int res;

  boot = Boot->name;

  if (! boot) {
    /* problem earlier and user responded "no": lilo.conf will be broken */
    problemBox(_("Please boot the system using the rescue boot method "
                 "and configure LILO manually."), _("Problem"));
    return 1;
  }

  /* set device to what we think is the boot disk */
  if (is_ide(boot)) device = DEV_HDA;
  else if (is_scsi(boot)) device = DEV_SDA;
  else if (is_scsird(boot)) device = DEV_RD_C0D0;
  else if (is_scsiida(boot)) device = DEV_IDA_C0D0;
  else {
    INFOMSG("couldn't determine drive type for new boot partition %s", boot);
    device = inputBox(_("The attempt to deduce what your boot disk is failed.  "
                        "Please enter the name of your boot disk, for instance, "
                        "'/dev/hda' or '/dev/sda'."),
                      _("Choose Boot Disk"), DEV_HDA);
    if (! NAME_ISBLK(device, &statbuf) ) {
      vaproblemBox(_("Invalid Boot Disk"), 
                   _("There is no block device '%s'."), device);
      return 1;
    }
  }
  
  opt[0].tag = device;
  opt[0].string = _("Install Lilo in the MBR (use this if unsure).");
  opt[1].tag = Boot->name;
  opt[1].string = _("Install LILO in the root partition's boot sector.");
  snprintf(prtbuf, sizeof(prtbuf), _("LILO can be installed either into the master boot record (MBR), or into the %s boot block. If installed into the MBR, LILO will take control of the boot process. If you choose not to install LILO into the MBR, you will have the opportunity later on to install an alternative MBR program (for bootstrapping LILO).\n"),
	   Boot->name);
  res = menuBox(prtbuf, _("Where should the Lilo boot loader be installed?"), opt, 2, 0);
  boot = opt[res].tag;
  installed_mbr = (res == 0);

  if (! NAME_ISBLK(boot, &statbuf) ) {
    vaproblemBox(_("Invalid Boot Disk"),
                 _("There is no block device '%s'.  You need to try this step again "
                   "with different choices, or perhaps make the device by hand."),
                 boot);
    return 1;
  }

  /* make sure this is first IDE or SCSI device  */
  if ( ! ( is_first_ide(Boot) || is_first_scsi(Boot) ||
	   yesNoBox(_("With your current configuration, it will try to boot "
		      "from a disk that is not your first IDE or SCSI disk. "
		      "It's possible that your BIOS doesn't support that.\n\n"
		      "Do you want to install the MBR anyway?"), _("Problem")) ) ) {
      return 1;
  }

  /* check for booting from SCSI but we also have /dev/hda */
  if ( is_first_scsi(Boot) && fdisk_find_disk(DEV_HDA) &&
	 ! yesNoBox(_("With your current configuration, it will try to boot "
		      "from your first SCSI disk, but your system has also a "
		      "master IDE disk at the primary IDE controller. "
		      "It's possible that your BIOS doesn't support that.\n\n"
		      "Do you want to install the MBR anyway?"), _("Problem")) ) {
      return 1;
  }

#if defined _TESTING_
  fprintf(stderr, "About to run_lilo(), boot is %s\n", boot);
  sleep(2);
#endif

  status = run_lilo(boot);

#if defined _TESTING_
  fprintf(stderr, "Just ran_lilo(), status is %d\n", status);
  sleep(2);
#endif

  if (status) {
    problemBox(_("LILO wasn't able to install. You'll still be able to boot your system if you create a boot floppy, but it won't be able to boot without a floppy.\nThe most common reason why LILO fails is trying to boot a kernel that resides at a location on the disk higher than the 1023rd cylinder. With older BIOS, LILO is unable to load any block of the kernel from a disk cylinder numbered higher than 1023. This is often a problem with disks larger than 524MB or so. One way to solve the problem is to make a small (5-10Mb) partition at the beginning of the disk and mount it on \"/boot\", so that the entire \"/boot\" partition will be below the 1023rd cylinder. Please refer to the installation manual for further information.\n"), _("Problem"));
    return 1;
  } else {
    if (!installed_mbr) {
      status = install_mbr(boot, &installed_mbr);
      if (status == 0 && installed_mbr) {
	  status = set_boot_default(boot);
#if !defined _TESTING_
	  execlog(BC_RUN_LILO, LOG_INFO); /* Must run `lilo' again, otherwise it will not reboot. (MBR prompt is all I get without this line - karlheg) */
#endif
      }
    }
  }
  notInstalledLILO = 0;
  
#elif #cpu (m68k)
  boot = Root->name;
  if (strcmp(Arch2, "VME") == 0) {
    boot=strdup(Root->name);
    boot[firstnumber(boot)] = '\0';
    status=run_vmelilo(boot);
    if(status!=0) {
      problemBox(_("VMELILO wasn't able to install. You'll still be able to boot your system if you create a boot floppy, but it won't be able to boot without a floppy."),_("Problem"));
      free(boot);
      return 1;
    }
    notInstalledLILO=0;
    free(boot);
  } else {
    problemBox(_("Installing a boot loader is not yet possible for Debian/m68k"),
	       _("Problem"));
  }
#elif #cpu (powerpc)
  struct fdisk_partition *boot_part;

  boot = Root->name;
  if (strcmp(Arch2, "PReP") == 0) {
    if ((boot_part = fdisk_find_partition_by_type(PTYPE_PREP_BOOT)) != NULL) {
      /* Copy a kernel to the PReP boot partition */
      pleaseWaitBox(_("Copying the operating system kernel..."));
      snprintf(prtbuf, sizeof(prtbuf), BC_RUN_PREPDD, "/target/vmlinuz", boot_part->name);
      status=execlog(prtbuf, LOG_INFO);
      /* FIXME MDP: Handle the error cases */
      boxPopWindow();
      notInstalledLILO=0;
      return 0;
    }
    else {
        /* FIXME MDP: Add in error message, must learn pointerize */
/*      problemBox(_("No PReP boot partition found"),_("Problem")); */
      return 1;
    }
  }
  else {
    status=run_plilo(boot);
    if(status!=0) {
      problemBox(_("Installation of QUIK has failed."),
		 _("Problem"));
      return 1;
    }
  }
  notInstalledLILO=0;
#elif #cpu (sparc)
  status=run_silo();
  if(status!=0) {
    problemBox(_("SILO wasn't able to install. You'll still be able to boot your system if you create a boot floppy, but it won't be able to boot without a floppy."),_("Problem"));
    return 1;
  }
  notInstalledLILO=0;
#elif #cpu (alpha)
  boot = Root->name;
  if (srm_boot) {
    status = run_aboot(boot);
    if (status) {
      problemBox(_("aboot wasn't able to install.  "
		   "Most likely this is because your first partition overlaps "
		   "the area where aboot was to be written. "
		   "Either repartition your disk and try again, or "
		   "boot the system using the rescue boot method "
		   "and install aboot manually."), _("Problem"));
      return 1;
    }
  } else {
    /* Try to find a FAT partition and copy MILO/APB to it */
    struct d_choices *choices;
    struct fdisk_partition **plist, *p;
    int r, i = 0;
    int is_apb = 0;

    choices = malloc(4 * sizeof(struct d_choices));
    plist   = malloc(4 * sizeof(struct fdisk_partition *));

    for (p = fdisk_partitions_by_type[FSTYPE_MSDOS];
	 p; p = p->next_by_type) {
      /* add it to the list */
      plist[i]          = p;
      choices[i].string = p->name;
      choices[i].tag    = NULL;
      choices[i].state  = 0;
      i++;
      if ((i % 4) == 0) {
	choices = realloc(choices,
			  (i+4) * sizeof(struct d_choices));
	plist   = realloc(plist,
			  (i+4) * sizeof(struct fdisk_partition *));
      }
    }
    if (i == 0) {
      problemBox(_("The MILO and APB bootloaders must reside on a FAT "
		   "partition in order to be read by ARC/AlphaBIOS firmware.  "
		   "You appear not to have any DOS FAT partitions.  "
		   "This means you will not be able to boot directly "
		   "from disk.\n"), _("No FAT partitions detected"));
      return 1;
    }
    r = menuBox(_("Select a FAT partition to install MILO or APB on"),
		_("Select Partition"), choices, i, 1);
    if (r < 0)
      p = NULL;
    else
      p = plist[r];

    free(plist);
    free(choices);
    if (p == NULL) return 1;
    if (strcmp(Arch2, "nautilus") == 0) {
      is_apb = 1;
      status = install_apb(boot, p);
    } else
      status = install_milo(boot, p);
    if (status) {
      sprintf(prtbuf, _("%s wasn't able to install. You'll still be "
			"able to boot your system from the disk you "
			"used to boot the Debian installer."),
	      is_apb ? "APB" : "MILO");
      problemBox(prtbuf, _("Problem"));
      return 1;
    }
  }
  notInstalledLILO=0;
#elif #cpu (arm)
  if (!strcmp(Arch2, "riscpc")) {
      problemBox(_("This option has no effect.  Debian GNU/Linux is "
		   "launched from RISC OS."),
		 _("Problem"));
  }
  else if (!strcmp(Arch2, "netwinder")) {
      extern const char *Arch3;
      if (!strcasecmp(Arch3, "rebel-netwinder")) {
	  problemBox(_("This option has no effect.  You need to configure "
		       "NeTTrom manually."),
		     _("Problem"));
      }
      else if (!strcasecmp(Arch3, "chalice-cats")) {
	  status = execlog(BC_RUN_CATSBOOT, LOG_INFO);
      }
      else {
	  problemBox(_("Don't know how to boot this system.  You must "
		       "configure your loader manually."),
		     _("Problem"));
      }
  }
  notInstalledLILO=0;
#else
# warning no boot loader to configure
#endif
  return status; 
}

#ifdef _TESTING_
/*
 * To test, compile using: make bootconfig_test
 */

void readcmdline (void);
void setup_image_names(void);
int main(int argc, char** argv)
{
  LOAD_TRMFILE("test.trm");
  readcmdline();
  get_kver();
  Root = calloc(1, sizeof(struct fdisk_partition));
  fdisk_reread();
#if #cpu(alpha)
  srm_boot = get_alpha_boot_method();
  Arch2 = get_alpha_model();
#endif
  setup_image_names();
  Root->name = strdup("/dev/hdc1");
  Root->disk = fdisk_disks;
  InstallationRootDevice = strdup("/dev/hdc");

  append_opts = strdup ("aha152x=0x200");

  if (argc != 2) {
      fprintf(stderr, "Boot Config test program\n%s (floppy|lilo)\n", argv[0]);
      exit(1);
  }
  
  if (strcmp(argv[1], "floppy") == 0) {
    boxInit();
    make_boot_floppy();
    boxFinished();
  }
  else if (strcmp(argv[1], "lilo") == 0) {
    printf("About to begin test number 1.\n");
    sleep(2);

    Boot = Root = calloc(1, sizeof(struct fdisk_partition));

    fdisk_disks = calloc(1, sizeof(struct fdisk_disk));
    fdisk_disks->name = strdup("/dev/hdc");
    fdisk_disks->partition_format = "msdos";

    Root->name = strdup("/dev/hdc1");
    Root->mount_point = strdup("/");
    Root->disk = fdisk_disks;

    InstallationRootDevice = strdup("/dev/hdc");

    boxInit();
    make_bootable();
    boxFinished();
    printf("Test number one complete...\n");
    sleep(2);

    printf("About to begin test number two.\n");
    sleep(2);

    Boot = calloc(1, sizeof(struct fdisk_partition));
    Root = calloc(1, sizeof(struct fdisk_partition));

    fdisk_disks = calloc(1, sizeof(struct fdisk_disk));
    fdisk_disks->name = strdup("/dev/hdc");
    fdisk_disks->partition_format = "msdos";

    Boot->name = strdup("/dev/hdc1");
    Boot->mount_point = strdup("/boot");
    Boot->disk = fdisk_disks;

    Root->name = strdup("/dev/hdc3");
    Root->mount_point = strdup("/");
    Root->disk = fdisk_disks;

    InstallationRootDevice = strdup("/dev/hdc");

    boxInit();
    make_bootable();
    boxFinished();
    printf("Test number two complete.\n");
    sleep(2);
  }
  else {
    fprintf(stderr, "Boot Config test program\n%s (floppy|lilo)\n", argv[0]);
    exit(-1);
  }
  exit (0);
}
#endif

/*
 * Local variables:
 *   c-basic-offset: 4
 *   indent-tabs-mode: t
 * End:
 */
