#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <zlib.h> 
#include <string.h>
#include <sys/mount.h>
#include <asm/page.h>
#include <sys/swap.h>
#include <sys/reboot.h>
#include <sys/stat.h>
#include "../libfdisk/fdisk.h"

#define CLEAR "[H[J"
#define MSG_ENTER "\nPress <ENTER> to continue...\n"
#define MSG_TYPEYN "\nType \"y\" to continue, type \"n\" to cancel:"

#define BUFSIZE 256
char prtbuf[BUFSIZE];
char *swappartition=NULL,*minixpartition=NULL;

/* functions defined in fdisk.c */
extern char read_char(char *mesg);
extern int read_line(void);
extern int fdisk_main(const char *device);
/* functions defined in mkswap.c */
extern int mkswap_main(const char *device_name, int pages, int do_check);

int zcat(const char *source, const char *target) {
  gzFile *fsrc;
  FILE *ftgt;
  int c;

  fsrc=gzopen(source,"rb");
  if ( fsrc == NULL ) {
    printf(CLEAR "Error opening %s!" MSG_ENTER, source);
    read_line();
    return 1;
  }
  ftgt=fopen(target,"wb");
  if ( ftgt == NULL ) {
    printf(prtbuf,CLEAR "Error opening %s!" MSG_ENTER, target);
    read_line();
    return 1;
  }
  while ( (c = gzgetc(fsrc)) != -1 )
    putc(c, ftgt);
  gzclose(fsrc);
  fclose(ftgt);
  return 0;
}

char *select_drive(void){ 
  struct fdisk_disk *disk;
  int i;
  if (NULL == fdisk_disks) {
    printf(CLEAR
"No hard disk drives could be found. Make sure they are cabled correctly and
are turned on before the system is started. You may have to change driver
settings when you start the system with a command at the \"boot:\" prompt,
or you may have to load a driver that is in a loadable module to solve this
problem." MSG_ENTER);
    read_line();
    return NULL;
  }
  printf(CLEAR
"Select the drive to partition. SCSI drives are listed in disk ID number
order. Only drives that were connected and operating when the system was
started will show up in this display. CD-ROM drives may be mis-identified as
writable disk drives by this menu.\n"); 
  disk = fdisk_disks;
  while (disk) {
    printf("%s ",disk->name);
    disk = disk->next;
  }
  printf("\nSelect Disk Drive: /dev/");
  sprintf(prtbuf,"/dev/");
  fgets(prtbuf+5,BUFSIZE,stdin); 
  for (i=0;prtbuf[i]!='\0';i++) if (prtbuf[i]=='\n') prtbuf[i]='\0';
  disk=fdisk_find_disk(prtbuf);
  if (disk) {
    return disk->name; 
  } else {
    printf(CLEAR "No valid drive was selected!" MSG_ENTER);
    read_line();
    return NULL;
  }
}

char *select_not_mounted( const char *message, const int fstype) { 
  struct fdisk_partition *p;
  char *name;
  int i,it=0;

  printf(CLEAR "%s\n",message);
  name = fstype_name[fstype];
  p = fdisk_partitions_by_type[fstype];
  while (p) {
    if (! p->in_use){
      printf("%s ",p->name);
      it++;
    }
    p = p->next_by_type;
  }

  if (!it) {
    printf(prtbuf,"\nNo %s partitions were detected!" MSG_ENTER,name);
    read_line();
    return NULL;
  }
  printf("\nSelect Partition: /dev/");
  sprintf(prtbuf,"/dev/");
  fgets(prtbuf+5,BUFSIZE,stdin); 
  for (i=0;prtbuf[i]!='\0';i++) if (prtbuf[i]=='\n') prtbuf[i]='\0';
  p=fdisk_find_partition_by_name(prtbuf);
  if ((p) && (fdisk_fstype_of(p->type)==fstype) && (! p->in_use)) {
    return (char *)strdup(p->name); 
  } else {
    return NULL;
  }
}

int partition_disk(void){
  char *drive,option;

  sync();
  if (swappartition) {
  /* Deactivate the swap partition before re-partitioning */
    while (1) {
      sprintf(prtbuf,CLEAR "You already activated the swap partition %s. 
It will be deactivated before running the partitioning program."
MSG_TYPEYN, swappartition);
      option=read_char(prtbuf);
      switch (option) {
	case 'y': 
	  swapoff(swappartition);
	  free(swappartition);
	  swappartition=NULL;
	  break;
	case 'n': return 1; break;
      }
    }
  }
  drive = select_drive();
  if (drive) {
    printf(CLEAR);
    if (fdisk_main(drive)) read_line();
    return 0;
  }
  return 1;
}

int init_swap_fn(void){
  printf(CLEAR "Initializing %s as a swap partition...\n",swappartition);
  if (mkswap_main(swappartition,0,1)) {
    printf("\nError initializing swap partition!" MSG_ENTER);
    read_line();
    return 1;
  }
  printf(CLEAR "Activating swap partition...\n");
  if (swapon(swappartition,0)) {
    printf("\nError activating swap partition!" MSG_ENTER);
    read_line();
    return 1;
  }
  return 0;
}

int init_minix_fn(void){
  printf(CLEAR "Put the rescue floppy on the first floppy drive." MSG_ENTER);
  read_line();
  if ( mount("/dev/fd0","/mnt","msdos",MS_RDONLY|MS_MGC_VAL,"\0") ) {
    printf("Unable to mount the rescue floppy disk!" MSG_ENTER);
    read_line();
    return 1;
  }
  printf(CLEAR 
    "Copying root filesystem from floppy to %s ...\n",minixpartition);
  if (zcat("/mnt/root.bin",minixpartition)){
    umount("/mnt");
    printf("Error copying root filesystem!" MSG_ENTER);
    read_line();
    return 1;
  }
  umount("/mnt");
  return 0;
}

int config_minix(void) {
  FILE* filep;
  struct stat info;
  printf(CLEAR "Configuring root filesystem ...\n");
  if ( mount(minixpartition,"/mnt","minix",MS_MGC_VAL,"\0") ) {
    printf("Unable to mount the temporary root partition!" MSG_ENTER);
    read_line();
    return 1;
  }
  if((filep=fopen("/mnt/etc/swappartition","w"))==NULL) {
    printf("Unable to open /mnt/etc/swappartition!" MSG_ENTER);
    read_line();
    return 1;
  }
  fprintf(filep,"%s\n",swappartition);
  fclose(filep);
#ifdef OLDKERNEL	/* For 2.0.x kernels */
  if((filep=fopen("/mnt/etc/swaps","w"))==NULL) {
    printf("Unable to open /mnt/etc/swaps" MSG_ENTER);
    read_line();
    return 1;
  }
  fprintf(filep,"%s none swap rw 0 0",swappartition);
  fclose(filep);
#endif
  umount("/mnt");

  mount(0,"/proc","proc",MS_MGC_VAL,"\0");
  if((filep=fopen("/proc/sys/kernel/real-root-dev","w"))==NULL) {
    printf("Unable to open /proc/sys/kernel/real-root-dev!" MSG_ENTER);
    read_line();
    return 1;
  }
  stat(minixpartition,&info);
  fprintf(filep,"%d\n",(int)info.st_rdev);
  fclose(filep);
  umount("/proc");
  return 0;
}

int init_partition(char **partition, const char *descrip, 
		const int fstype, int (*init_fn)(void)){
  char option;
  sync();
  sprintf(prtbuf,"Please select the partition to initialize as %s",descrip);
  (*partition)=select_not_mounted(prtbuf,fstype);
  if (!(*partition)) return 1;
  while (1) {
    sprintf(prtbuf,CLEAR "You have chosen to initialize %s as %s.
This will permanently erase any data on this partition." MSG_TYPEYN, 
(*partition),descrip);
    option=read_char(prtbuf);
    switch (option) {
      case 'n': return 1; break;
      case 'y': 
	if((*init_fn)()) {
	  free(*partition);
	  (*partition)=NULL; 
	  return 1;
	} else {
	  return 0;
	}
	break;
    }
  }
}

int reboot_system (void) {
  char option;
  while (1) {
    option=read_char(CLEAR 
	    "Preparing to reboot the system. Are you sure?" MSG_TYPEYN);
    switch (option) {
      case 'n': return 1; break;
      case 'y': 
        sync();
	if (swappartition) swapoff(swappartition);
        sync();
        reboot(RB_AUTOBOOT);
	break;
    }
  }
}

void print_menu(void) {
  printf(CLEAR 
"This system has relatively little memory. For a system like this, special 
procedures are required, because there is not enough RAM available
to run the graphical program without the virtual memory provided by the swap
partition. Thus, you should:

1. Run the disk partitioning program to add a \"Linux swap\" (type 82)
   partition and the \"Linux native\" (type 83) partition. You have to create
   an extra temporary root partition that has at least 2MB and type 81 
    (\"Linux/MINIX\").
   You will not be able to re-partition the drive while the
   swap partition on it is active.
2. Then initialize the swap partition.
3. Then initialize the temporary root partition
4. Then exit. The system will resume the installation from the temporary
   root partition.

Please select an action from the menu below:\n");

  if (!minixpartition) 
    printf("1. Run the disk partitioning program.\n");
  if ((!swappartition)) 
    printf("2. Initialize and activate the swap partition.\n");
  if ((swappartition)&&(!minixpartition)) 
    printf("3. Initialize the temporary root partition.\n");
  if (minixpartition) 
    printf("4. Exit.\n");
  printf("5. Reboot.\n");
}

void post_exit(int stat, void *foo);

int main(void){
  char option;

  swappartition=NULL;
  minixpartition=NULL;
  on_exit(post_exit,NULL);
  while (1) {
    fdisk_reread();
    print_menu();
    option=read_char("\nAction: ");
    switch (option) {
      case '1': 
	if (!minixpartition) 
	  partition_disk();
	break;
      case '2': 
	if ((!swappartition)) {
	  init_partition (&swappartition,"a swap device", FSTYPE_SWAP, init_swap_fn);
	}
	break;
      case '3':
	if ((swappartition)&&(!minixpartition)) {
	  init_partition (&minixpartition,"the temporary root partition",
			FSTYPE_MINIX, init_minix_fn);
	  config_minix();
	}
	break;
      case '4': if (minixpartition) return 0; break;
      case '5': reboot_system(); break;
    }
  }
}

void post_exit(int stat, void *foo){
  if (stat) {
    if (swappartition) {
      free(swappartition);
      swappartition=NULL;
    }
    if (minixpartition) {
      free(minixpartition);
      minixpartition=NULL;
    }
    main();
  }
}
