/*
 * PCMCIA Installation module
 * Luis Francisco Gonzalez
 *
 * The idea is to develop some modules that enable the user to create
 * the PCMCIA configuration.
 * 
 * There will be a special routine that will handle communication with
 * cardmgr so that at any one time the state is known. The rationale
 * is that some people need to access pcmcia drivers to install the
 * system even before the later configuration stages are reached.
 *
 * Files:
 * /etc/pcmcia.conf:        Controller type and options.
 * /etc/pcmcia/config.opts: Allows to exclude irq, I/O ports
 *                          and memory window for probing.
 */

#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "dbootstrap.h"
#include "lang.h"
#include <syslog.h> 
#include "util.h"

#ifdef PCMCIA

/*
 * Module variables
 */
#ifdef _TESTING_
static char pcmcia_id[]="$Id: pcmcia.c,v 1.29 2001/12/27 15:14:51 ender Exp $";
#endif
static char*  controller        = NULL;
static char*  pcic_opts         = NULL;
static char*  core_opts         = NULL;
static char*  cardmgr_opts      = NULL;
static char*  serial_link       = NULL;
static char*  serial_opts       = NULL;
static char*  inittab           = NULL;
static int    use_inittab       = 0;
static char*  cdrom_do_fstab    = NULL;
static char*  cdrom_do_mount    = NULL;
static char*  cdrom_fstype      = NULL;
static char*  cdrom_opts        = NULL;
static char*  cdrom_mountpt     = NULL;
static char** pcmcia_choices    = NULL;
static char*  pcmcia_values     = NULL;
static char   pcmcia_configured = 0;

/*
 * PCMCIA Check list
 */
#define _WIDTH  (65)   /* Width of the choices menu.     */
#define _HEIGHT (18)   /* Height of the choices menu.    */
#define OPTS (3)       /* Number of choices in the menu. */

/*
 * Function prototypes
 */
int         configure_pcmcia (void);
int         activate_pcmcia (void);

static int  input_parameter (char*, char*, char**);
static int  _activate_pcmcia (void);
static int  _cardmgr (char);
static int  pcmcia_serial_file (void);
static int  pcmcia_cdrom_file (void);
static int  pcmcia_conf_file (void);
static int  write_files (void);
static int  write_lowlevel_files (void);
static int  write_serial_opts (void);
static int  write_cdrom_opts (void);
#if 0
static int  write_network_config (void);
#endif
static int  select_components (void);

/*
 * Directories
 */
#ifndef _TESTING_
#define PCMCIA_ETC_DIR       "/target/etc"
#define PCMCIA_CONF_FILE     "/target/etc/pcmcia.conf"
#define PCMCIA_CONF_DIR      "/target/etc/pcmcia"
#define PCMCIA_SERIAL_OPTS   "/target/etc/pcmcia/serial.opts"
#define PCMCIA_CDROM_OPTS    "/target/etc/pcmcia/cdrom.opts"
#define PCMCIA_HOSTS_FILE    "/target/etc/hosts"
#define PCMCIA_NETWORK_OPTS  "/target/etc/pcmcia/network.opts"
#define PCMCIA_VAR_RUN       "/var/run"
#else
#define PCMCIA_ETC_DIR       "targetdir_test/etc"
#define PCMCIA_CONF_FILE     "targetdir_test/etc/pcmcia.conf"
#define PCMCIA_CONF_DIR      "targetdir_test/etc/pcmcia"
#define PCMCIA_SERIAL_OPTS   "targetdir_test/etc/pcmcia/serial.opts"
#define PCMCIA_CDROM_OPTS    "targetdir_test/etc/pcmcia/cdrom.opts"
#define PCMCIA_HOSTS_FILE    "targetdir_test/etc/hosts"
#define PCMCIA_NETWORK_OPTS  "targetdir_test/etc/pcmcia/network.opts"
#define PCMCIA_VAR_RUN       "targetdir_test/var/run"
#endif

#define MODULES_INSMOD 1
#define MODULES_RMMOD  2

#define CARDMGR_START 1
#define CARDMGR_STOP  2

#define PCMCIA_NOT_CFG   1
#define PCMCIA_MOD_ERROR 2

/*
 * Messages
 */

#define HEADER_TXT "#\n# This file was automatically generated during the Debian installation\n#\n"
#define WRT_HDR(FILEP) fprintf ((FILEP), "%s", HEADER_TXT)

#define SERIAL_TXT  "#\n# Serial device configuration.\n#\n"
#define CDROM_TXT   "#\n# CDROM device configuration.\n#\n"
#define NETWORK_TXT "#\n# Network adapter configuration.\n#\n"

static int
_modules (char action)
{
	switch (action)
	{
	case MODULES_INSMOD:
		snprintf (prtbuf, PRTBUFSIZE,
			  "/sbin/insmod -s "
#ifndef _TESTING_
			  "/target"
#endif
			  "/lib/modules/%s/pcmcia/pcmcia_core.o %s",
			  kver,
			  core_opts);
		if (execlog (prtbuf, LOG_INFO))
			return PCMCIA_MOD_ERROR;
		
		snprintf (prtbuf, PRTBUFSIZE,
			  "/sbin/insmod -s "
#ifndef _TESTING_
			  "/target"
#endif
			  "/lib/modules/%s/pcmcia/%s.o %s",
			  kver,
			  controller,
			  pcic_opts);
		if (execlog (prtbuf, LOG_INFO)) {
			execlog ("/sbin/rmmod pcmcia_core", LOG_INFO);
			return PCMCIA_MOD_ERROR;
		}
		
		snprintf (prtbuf, PRTBUFSIZE,
			  "/sbin/insmod -s "
#ifndef _TESTING_
			  "/target"
#endif
			  "/lib/modules/%s/pcmcia/ds.o ",
			  kver);
		if (execlog (prtbuf, LOG_INFO)) {
			snprintf (prtbuf, PRTBUFSIZE, "/sbin/rmmod %s", controller);
			execlog (prtbuf, LOG_INFO);
			execlog ("/sbin/rmmod pcmcia_core", LOG_INFO);
			return PCMCIA_MOD_ERROR;
		}
		break;

	case MODULES_RMMOD:
		{
			/* Get the list of modules that are using ds,
			   and unload them all */
			FILE *fp = fopen("/proc/modules", "r");
			if (fp) {
				char buf[256];
				while (fgets(buf, sizeof(buf), fp)) {
					if (!strncmp(buf, "ds ", 3)) {
						char *p = strchr(buf, '[');
						char *q = strchr(buf, ']');
						if (p && q) {
							p++;
							*q = 0;
							while (p) {
								q = strchr(p, ' ');
								if (q) {
									*q = 0;
									q++;
								}
								snprintf (prtbuf, PRTBUFSIZE, "/sbin/rmmod %s", p);
								execlog (prtbuf, LOG_INFO);
								p = q;
							}
						}
						break;
					}
				}
				fclose(fp);
			}
		}
		execlog ("/sbin/rmmod ds", LOG_INFO);
		snprintf (prtbuf, PRTBUFSIZE, "/sbin/rmmod %s", controller);
		execlog (prtbuf, LOG_INFO);
		execlog ("/sbin/rmmod pcmcia_core", LOG_INFO);
		break;
	}

	return 0;
}

static int
_cardmgr (char action)
{
	struct stat statbuf;
	int rtn = 1;

	switch (action)
	{
	case CARDMGR_START:
		snprintf (prtbuf, PRTBUFSIZE, "/sbin/cardmgr %s ; sleep 3", cardmgr_opts);
		INFOMSG ("Starting 'cardmgr %s'", cardmgr_opts);
		rtn = execlog (prtbuf, LOG_INFO);
		break;
		
	case CARDMGR_STOP:
		if ( NAME_ISREG ("/var/run/cardmgr.pid", &statbuf) ) {
			INFOMSG ("Stopping cardmgr");
			rtn = execlog ("kill -TERM $(cat /var/run/cardmgr.pid) 2>/dev/null; "
				       "sleep 6", LOG_INFO);
		} else {
			INFOMSG ("No /var/run/cardmgr.pid, cardmgr not needing to be stopped");
		}
		break;

	}
	return rtn;
}

static int
_activate_pcmcia (void)
{
	if (pcmcia_configured) {

		struct stat statbuf;

		/*
		 * Stop cardmgr and unload the modules.
		 *
		 * Since activate_pcmcia() is called after
		 * netconfig(), it must be shut down and restarted in
		 * order to get the network configuration read and the
		 * ifconfig done.  Otherwise, the user will be
		 * required to pop the card out of the slot then back
		 * in, to make it work.  Must unload the modules also,
		 * then reinsert them, to make sure that the hardware
		 * and drivers get properly initialized.  One person
		 * who helped test boot-floppies (calc) has a network
		 * card that is recognized as a memory card after
		 * cardmgr is restarted if cardmgr is shut down and
		 * restarted without also removing the modules and
		 * reinserting them.
		 */
		(void) _cardmgr (CARDMGR_STOP);
		(void) _modules (MODULES_RMMOD);

		/*
		 * Creates the /var/run directory if not there
		 */
		if ( ! NAME_ISDIR (PCMCIA_VAR_RUN, &statbuf) ) {
			if (mkdir(PCMCIA_VAR_RUN, 0755))
				return PCMCIA_MOD_ERROR;
		}
      
#ifndef _TESTING_
		/*
		 * Link /target/etc/pcmcia to /etc/pcmcia
		 */
		if ( ! NAME_ISREG("/etc/pcmcia", &statbuf) &&
		     ! NAME_ISLNK("/etc/pcmcia", &statbuf) ) {
			symlink ("/target/etc/pcmcia", "/etc/pcmcia");
		}
#endif
		/*
		 * Load the driver modules if not loaded.
		 */
		if (_modules (MODULES_INSMOD)) {
			return PCMCIA_MOD_ERROR;
		}

		/*
		 * Start the cardservices.
		 */
		if (_cardmgr (CARDMGR_START))
			return PCMCIA_MOD_ERROR;
	}
	else {
		return PCMCIA_NOT_CFG;
	}
	return 0;
}

/* public */ int
activate_pcmcia (void)
{
	int rtn = 0;

	boxSuspend();
	rtn = _activate_pcmcia();
	boxResume();

	switch (rtn)
	{
	case PCMCIA_NOT_CFG:
		problemBox (_("You should configure the PCMCIA support before activating it."),
			    _("PCMCIA Error"));
		break;
	case PCMCIA_MOD_ERROR:
		problemBox (_("An error encountered while trying to load and configure the PCMCIA modules."),
			    _("PCMCIA Error"));
		break;
	default:
	}

	/*
	 * Write to the user settings file to indicate pcmcia is
	 * configured.  #### This gets done twice; there are two call
	 * sites for activate_pcmcia().
	 */
	if (!rtn) write_userconfig ("PCMCIA", "yes");
 
	return rtn;
}

static int
input_parameter (char *text, char *title, char **variable)
{
	char *def = NULL;

	if (*variable) {
		def = strdup (*variable);
		free (*variable);
	}
	else
		def = strdup("");

	(*variable) = inputBox (text, title, def);
	free (def);

	if ((*variable) == NULL)
		return -1;

	return 0;
}

static int
pcmcia_serial_file (void)
{
	for (;;) {
		if (input_parameter(_("Link used to configure dialout devices"),
				    _("Serial Device Link"),
				    &serial_link) < 0)
			return -1;

		if (input_parameter(_("Options passed onto setserial"),
				    _("Setserial Options"),
				    &serial_opts) < 0)
			return -1;

		if (yesNoBox (_("Should we create an inittab entry for this port?"),
			      _("Inittab")) != 0 &&
		    input_parameter (_("Which command should be used for inittab?"),
				     _("Inittab"),
				     &inittab) == 0) {
			use_inittab = 1;
		}
		else {
			use_inittab = 0;
		}
	
		/*
		 * Confirm values
		 */
		snprintf (prtbuf, PRTBUFSIZE,
			  _("Serial Device:      %s\nOptions:            %s\n"),
			  serial_link,
			  serial_opts);
		if (use_inittab)
			snprintf (prtbuf + strlen (prtbuf), PRTBUFSIZE,
				  _("Inittab command:    %s\n"),
				  inittab);
		snprintf (prtbuf + strlen (prtbuf), PRTBUFSIZE,
			  _("\n\nCorrect?\n"));

		if (yesNoBox (prtbuf, _("Confirmation")) == DLG_YES)
			break;
	}

	return 0;
}

static int
pcmcia_cdrom_file (void)
{
	for (;;) {
		if (yesNoBox (_("Would you like to have the entry for this CD-ROM automatically added to the filesystem table?"),
			      _("Filesystem Table"))
		    == DLG_YES) {
			free (cdrom_do_fstab);
			cdrom_do_fstab = strdup ("y");
		}
		else {
			free (cdrom_do_fstab);
			cdrom_do_fstab = strdup ("n");
		}

		if (yesNoBox (_("Shall I mount the CDROM on connection?"),
			      _("Automatically Mount"))
		    == DLG_YES) {
			free (cdrom_do_mount);
			cdrom_do_mount = strdup ("y");
		}
		else {
			free (cdrom_do_mount);
			cdrom_do_mount = strdup ("n");
		}

		if (input_parameter (_("CD-ROMs normally have an iso9660 filesystem.\nWhich filesystem type should I use to mount the CD-ROM?"),
				     _("Filesystem Type"),
				     &cdrom_fstype) < 0)
			return -1;
		if (input_parameter (_("The CD-ROM should be mounted using which options?"),
				     _("Mount options"),
				     &cdrom_opts) < 0)
			return -1;
		if (input_parameter (_("The Debian filesystem structure uses /cdrom as the directory to mount any CD-ROMs. "
				       "Where would you like to have your CD-ROM mounted?"),
				     _("Mount point"),
				     &cdrom_mountpt) < 0)
			return -1;

		/*
		 * Confirm values
		 */
		snprintf (prtbuf, PRTBUFSIZE,
			  _("Update fstab:       %s\nMount:              %s\nMount options:      %s\nMount point:        %s\n\n\nCorrect?\n"),
			  cdrom_do_fstab,
			  cdrom_do_mount,
			  cdrom_opts,
			  cdrom_mountpt);

		if (yesNoBox (prtbuf, _("Confirmation")) == DLG_YES)
			break;
	}

	return 0;
}

static int
pcmcia_conf_file (void)
{
	for (;;) {
		/*
		 * Type of controller
		 */
		struct d_choices choices[2];
		int choice;

		choices[0].tag    = "i82365";
		choices[0].string = _("Intel i82365SL or compatible.");
		choices[0].state  = 0;
		choices[1].tag    = "tcic";
		choices[1].string = _("Databook TCIC-2 or compatible.");
		choices[1].state  = 0;

		if ((choice = menuBox (_("There are two major types of PCMCIA controllers, the Intel i82365SL-compatible type (option i82365) and the Databook TCIC-2 type (option tcic).\n\nWhat PCMCIA controller chip do you have?"),
				       _("Controller Type"), choices, 2, 0)
			) < 0) {choice = 0;}

		if (controller != NULL) free (controller);
		controller = strdup (choices[choice].tag);

		/*
		 * PCIC_OPTS, CORE_OPTS, CARDMGR_OPTS
		 */
		if (input_parameter (_("In systems with particularly fast processors, timing parameters need to be adjusted using \"cmd_time=<time>\" to determine the length of the PCMCIA bus cycles. The Cirrus PD6729 PCI controller needs \"fast_pci\" if the PCI speed is greater than 25 MHz. Also, unusual settings from manufacturers require that you provide special options in some cases. If you have a Halikan NBD 486 system, it has a TCIC-2 controller at an unusual location: you'll need add \"tcic_base=0x02c0\". In some systems using Cirrus controllers, including the NEC Versa M, the BIOS put the i82365 controller in a special suspended state at system startup time. On these systems, you will need to add the option \"wakeup=1\".\n\nYou probably don't need to add any option at all.\nWould you like to specify any controller setting?"),
				     _("Controller Options"),
				     &pcic_opts) < 0)
			return -1;

		input_parameter (_("To change the memory speed used for accessing a card's Card Information Structure (CIS), some systems with fast bus clocks will need to increase \"cis_speed=<speed>\".\n\nIt is highly unlikely that you should need to use this.\nWould you like to specify any core options?"),
				 _("Core Options"),
				 &core_opts);

                                /*  FIXME: why not skip this unless verbose ? */
		if (input_parameter (_("\"cardmgr\" is the daemon in charge of managing PCMCIA devices. Sometimes you want it to run with special settings, for example to enable debugging functionality. You can specify the options here.\n\nAs before you shouldn't worry about this at all.\nWould you like to specify any cardmgr options?"),
				     _("Card Manager Options"),
				     &cardmgr_opts) < 0)
			return -1;

		/*
		 * Confirm values
		 */
		snprintf (prtbuf, PRTBUFSIZE, 
			  _("Controller:         %s\nController options: %s\nCore options:       %s\nCardmgr options:    %s\n\n\nCorrect?\n"),
			  controller,
			  pcic_opts,
			  core_opts,
			  cardmgr_opts);

		if (yesNoBox (prtbuf, _("Confirmation")) == DLG_YES)
			break;
	}

	return 0;
}

static int
write_files (void)
{
	if (write_lowlevel_files() != 0)
	{
		problemBox (_("Error while writing configuration files."),
			    _("PCMCIA Error"));
		return 1;
	}
	else if (write_serial_opts() != 0)
	{
		problemBox(_("Error while writing configuration files."),
			   _("PCMCIA Error"));
		return 1;
	}
	else if (write_cdrom_opts() != 0)
	{
		problemBox(_("Error while writing configuration files."),
			   _("PCMCIA Error"));
		return 1;
	}
	/*  else if (write_network_config() != 0) return 1; */

	pcmcia_configured = 1;

	return 0;
}

static int
write_lowlevel_files (void)
{
	FILE *p_file;
	struct stat statbuf;

	/*
	 * Creates the /etc directory
	 */
	if (! NAME_ISDIR (PCMCIA_ETC_DIR, &statbuf)) {
		if (mkdir (PCMCIA_ETC_DIR, 0755)) {
			ERRMSG ("error mkdir " PCMCIA_ETC_DIR);
			return 1;
		}
	}

	/*
	 * /etc/pcmcia.conf
	 */
	if ((p_file = fopen (PCMCIA_CONF_FILE, "w")) == NULL) {
		ERRMSG("error opening " PCMCIA_CONF_FILE " for write");
		return 1;
	}
	WRT_HDR (p_file);
	fprintf (p_file, "PCMCIA='yes'\n");
	fprintf (p_file, 
                 "#\n"
                 "# Should be either i82365 or tcic\n"
                 "#\n"
                 "PCIC=\"%s\"\n"
                 "\n"
                 "#\n"
                 "# Put socket driver timing parameters here.\n"
                 "#\n"
                 "PCIC_OPTS=\"%s\"\n"
                 "\n"
                 "#\n"
                 "# Put pcmcia_core options here.\n"
                 "#\n"
                 "CORE_OPTS=\"%s\"\n"
                 "\n"
                 "#\n"
                 "# Put cardmgr options here.\n"
                 "#\n"
                 "CARDMGR_OPTS=\"%s\"\n",
		 controller,
		 pcic_opts,
		 core_opts,
		 cardmgr_opts);
	fclose (p_file);

	/*
	 * Creates /etc/pcmcia
	 */
	if (! NAME_ISDIR (PCMCIA_CONF_DIR, &statbuf)) {
		if (mkdir (PCMCIA_CONF_DIR, 0755)) {
			ERRMSG ("error mkdir " PCMCIA_CONF_DIR);
			return 1;
		}
	}

	return 0;
}

static int
write_serial_opts (void)
{
	FILE *p_file;

	/*
	 * serial.opts
	 */
	if ((p_file = fopen (PCMCIA_SERIAL_OPTS, "w")) == NULL) {
		ERRMSG ("error opening " PCMCIA_SERIAL_OPTS " for writing");
		return 1;
	}

	WRT_HDR (p_file);

	fprintf (p_file, SERIAL_TXT);

	fprintf (p_file,
		 "#\n"
                 "# The address format is \"scheme,socket,instance\".\n"
                 "#\n"
		 "case \"$ADDRESS\" in\n"
		 "*,*,*)\n"
		 "\t#\n"
                 "\t#Symbolic link to dialout device\n"
                 "\t#\n"
                 "\tLINK=\"%s\"\n"
                 "\n"
                 "\t#\n"
                 "\t# Options for 'setserial'\n"
                 "\t#\n"
                 "\tSERIAL_OPTS=\"%s\"\n"
                 "\n"
                 "\t#\n"
                 "\t# Should we create an inittab entry for this port?\n"
                 "\t#\n",
		 serial_link,
		 serial_opts);

	if (use_inittab)
		fprintf (p_file, "\tINITTAB=\"%s\"\n", inittab);
	else
		fprintf (p_file, "\t# INITTAB=\"%s\"\n", inittab);
  
	fprintf (p_file,
		 "\t;;\n"
		 "esac\n");

	fclose (p_file);
  
  return 0;
}

static int
write_cdrom_opts (void)
{
	FILE *p_file;
	/*
	 * cdrom.opts
	 */
	if ((p_file = fopen (PCMCIA_CDROM_OPTS, "w")) == NULL)
		return 1;

	WRT_HDR (p_file);

	fprintf (p_file, CDROM_TXT);

	fprintf(p_file,
                "#\n"
                "# The address format is \"scheme,socket\".\n"
                "#\n"
                "case \"$ADDRESS\" in\n"
                "*,*)\n"
                "\tDO_FSTAB=\"%s\" ; DO_MOUNT=\"%s\"\n"
                "\tFSTYPE=\"%s\"\n"
                "\tOPTS=\"%s\"\n"
                "\tMOUNTPT=\"%s\"\n"
                "\t;;\n"
                "esac\n",
                cdrom_do_fstab,
                cdrom_do_mount,
                cdrom_fstype,
                cdrom_opts,
                cdrom_mountpt);

	fclose(p_file);

	return 0;
}

/*
 * Mark what we need to configure
 */
static int
select_components (void)
{
	return checkBox (_("The following is a list of all drivers that can be configured automatically from within the Debian installation system. You don't need to select them all, as most will already provide reasonable defaults. To accept all the defaults, just press \"Ok\" without selecting anything.\nTo configure your networking, please use the \"Configure the Network\" option from the main menu after this. If using PPP/SLIP, configure the PCMCIA modem here but use the \"Configure the Network\" option from the main menu to configure your networking parameters.\nPlease mark the drivers you would like to configure."),
			 _("Debian PCMCIA Configuration"),
			 _HEIGHT, _WIDTH, pcmcia_choices, &pcmcia_values, OPTS);
}


int
configure_pcmcia (void)
{
	int ix, rtn = 0;
#ifdef _TESTING_
	problemBox (pcmcia_id, _("Version"));
#endif
        controller     = strdup("i82365");
        pcic_opts      = strdup("");
        core_opts      = strdup("");
        cardmgr_opts   = strdup("");
        serial_link    = strdup("/dev/modem");
        serial_opts    = strdup("");
        inittab        = strdup("/sbin/getty");
        cdrom_do_fstab = strdup("n");
        cdrom_do_mount = strdup("n");
        cdrom_fstype   = strdup("iso9660");
        cdrom_opts     = strdup("ro");
        cdrom_mountpt  = strdup("/cdrom");

	if (! pcmcia_values) {
		pcmcia_values = calloc (OPTS + 1, sizeof (char));
		for (ix = 0; ix < OPTS; ix++)
			pcmcia_values[ix] = ' ';
		pcmcia_values[ix] = '\0';
		pcmcia_values[0]  = '*';
	}

	pcmcia_choices = calloc (OPTS, sizeof (char*));
	pcmcia_choices[0] = strdup (_("PCMCIA controller"));
	pcmcia_choices[1] = strdup (_("Serial devices"));
	pcmcia_choices[2] = strdup (_("CD-ROM devices"));
#if 0
	pcmcia_choices[3] = strdup (_("Network devices"));
#endif

	if (select_components() != DLG_OKAY) {
		return 255;
	} else {
		for (ix = 0 ; ix < OPTS ; ix++) {
			if (pcmcia_values[ix] != ' ')
				switch (ix)
				{
				case 0:
					if (pcmcia_conf_file() < 0)
						return 255;
					break;
				case 1:
					if (pcmcia_serial_file() < 0)
						return 255;
					break;
				case 2:
					if (pcmcia_cdrom_file() < 0)
						return 255;
					break;
#if 0
				case 3:
					pcmcia_network_file(); break;
#endif
				}
		}
	}
  
	for (ix = 0 ; ix < OPTS ; ix++) {
		free (pcmcia_choices[ix]);
	}
	free (pcmcia_choices);

	if ((rtn = write_files()) != 0) return rtn;
  
	rtn = activate_pcmcia();
  
	return rtn;
}

/*
 * To test, compile using: make pcmcia_test
 */
#ifdef _TESTING_
void
main (void)
{
	LOAD_TRMFILE ("test.trm");
	get_kver();
	boxInit();

	configure_pcmcia();

	free (controller);
	free (pcic_opts);
	free (core_opts);
	free (cardmgr_opts);
	free (serial_link);
	free (serial_opts);
	free (inittab);
	free (cdrom_do_fstab);
	free (cdrom_do_mount);
	free (cdrom_fstype);
	free (cdrom_opts);
	free (cdrom_mountpt);
  
	boxFinished();
}
#endif
/*
 * Local Variables:
 * c-file-style: "linux"
 * End:
 */

#endif /* PCMCIA */
