/*
 * Description  Net-fetch module implementation.
 * CVS          $Id: net-fetch.c,v 1.61 2002/03/17 03:48:15 dwhedon Exp $
 * Author       Marcel Harkema <marcel@debian.org>
 *
 * Copyright (C) 1999, 2000 Marcel Harkema
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/param.h>
#include <time.h>
#include <assert.h>
#include <regex.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <newt.h>

#include "dbootstrap.h"
#include "net-fetch.h"
#include "lang.h"
#include "util.h"

#define WGET "/usr/bin/wget"

/*
 * Private data-structures and variables.
 */

struct _nf_state nf_state;

struct _nf_transport {
  char                *url_prefix;
  unsigned short int   default_port;
};


struct _nf_transport nf_transports[] = {
  { "http",  80 },
  { "ftp",   21 },
  { NULL,    0  }
};

#ifdef _TESTING_
extern char * get_device        (const char *);
extern void   release_device    (const char *);
extern int    install_floppy    (char *, const char *, const char *);
extern int    install_from_file (const char *, const char *);
extern int    extract_from_file (const char *, const char *);
extern void   setup_image_names (void);
extern void   get_kver          (void);
extern void   get_subarch_name  (void);
#endif


/*
 * Installation series.
 */

#define SERIES_OS    1

static int nf_install(int series);

static int
nf_extract_kernel(const char *filename_)
{
  char *device;
  int   r;
  char  filename[MAXPATHLEN];
  char *old_Archive_Dir;

  /* get_device() puts the Archive_Dir in front the the diskimage
   * filename, so we point Archive_Dir to an empty string. */
  old_Archive_Dir = Archive_Dir;
  Archive_Dir = "";

  snprintf(filename, sizeof(filename), "%s", target_path(filename_));

  if ((device = get_device(filename)) != NULL) {
    r = install_floppy(device, "rescue", _("Rescue Floppy"));
    release_device(device);
  }
  else {
    r = 1;
  }

  Archive_Dir = old_Archive_Dir;

  if (r) {
    vaproblemBox(_("Floppy Error"),
                 _("The attempt to extract the %s failed."), _("Rescue Floppy"));
  }

  return r;
}


static int
nf_extract_drivers(const char *filename_)
{
  char filename[MAXPATHLEN];
  snprintf(filename, sizeof(filename), "%s", target_path(filename_));
  return install_from_file(filename, _("Drivers"));
}

int
nf_install_os(void)
{
  return nf_install(SERIES_OS);
}

static int
nf_method_lookup(char *method)
{
  int ti;
  for (ti = 0; nf_transports[ti].url_prefix != NULL; ti++)
    if (!strcmp(nf_state.method, nf_transports[ti].url_prefix))
      break;
  return (nf_transports[ti].url_prefix == NULL) ? -1 : ti;
}


/*
 * Widget for selecting the server to download from.
 */

static int
nf_parse_server_url(char *url)
{
  /* this should be replaced with a proper parse fun.  also, this
   * one has minimal error checking. */
  char *buf = strdup(url);
  char *p, *q;
  int method;

  /* parse 'method://hostname:port/path' */

  p = strstr(buf, "://");
  if (p) {
    *p = 0;
    q = p+3; /* start of next part */
    p = buf; 
  }
  else {
    q = buf; /* start of next part */
    p = "http"; /* default */
  }
  free(nf_state.method);
  nf_state.method = strdup(p);

  free(nf_state.server.hostname);
  free(nf_state.path);
  nf_state.path = NULL;

  p = strchr(q, ':');
  if (p) {
    *p = 0;
    nf_state.server.hostname = strdup(q);
    p++;
    nf_state.server.port = strtoul(p, &q, 10);
    if (q) nf_state.path = strdup((*q == 0) ? "" : q+1);
  }
  else {
    nf_state.server.port = 0; /* use method's default port */
    p = strchr(q, '/');
    if (p) {
      *p = 0;
      nf_state.server.hostname = strdup(q);
      nf_state.path = strdup(p+1);
    }
    else {
      nf_state.server.hostname = strdup(q);
    }
    if (nf_state.path == NULL)
      nf_state.path = strdup("/");
  }

  if ((method = nf_method_lookup(nf_state.method)) < 0) {
    snprintf(prtbuf, sizeof(prtbuf) - 1,
	     _("Unknown protocol '%s' used in the URL."), nf_state.method);
    strcat(prtbuf, "\n");
    problemBox(prtbuf, "Unknown Protocol");
    free(buf);
    return 1;
  }

  if (nf_state.server.port == 0)
    nf_state.server.port = nf_transports[method].default_port;

  free(buf);
  return 0;
}


int
nf_select_server(void)
{
  char          *s[3];
  newtComponent  l[3];
  newtComponent  e[3];
  newtComponent  form, b1, b2, b3, textbox, ans;
  int            height;
  char           txtbuf[2048];
  int            ti;

  textbox = newtTextbox(1, 1, 72, 6, NEWT_FLAG_WRAP);
  newtTextboxSetText(textbox, _("Please provide the URL from which to download the installation files.  If you have a proxy server, fill that out as well; otherwise, leave it set to 'none'.\n"));
  height = newtTextboxGetNumLines(textbox);
  newtTextboxSetHeight(textbox, height);
  newtCenteredWindow(73, height + 7, _("Select Installation Server"));

  ti = nf_method_lookup(nf_state.method);
  if (ti >= 0
   && nf_transports[ti].url_prefix != NULL
   && nf_transports[ti].default_port == nf_state.server.port) {
    snprintf(txtbuf, sizeof txtbuf - 1, "%s://%s/%s",
             nf_state.method, nf_state.server.hostname, nf_state.path);
  }
  else {
    snprintf(txtbuf, sizeof txtbuf - 1, "%s://%s:%d/%s",
             nf_state.method, nf_state.server.hostname, nf_state.server.port, nf_state.path);
  }

  {
    const char *l1 = _("Download URL");
    const char *l2 = _("Proxy");
    const char *l3 = _("Proxy Port");
    int x = max(strwidth(l1), strwidth(l2));
    
    l[0] = newtLabel(1, height + 2, l1);
    e[0] = newtEntry(x + 2, height + 2, txtbuf, 70 - x, &s[0], NEWT_FLAG_SCROLL);
    
    l[1] = newtLabel(1, height + 4, l2);
    e[1] = newtEntry(x + 2, height + 4, nf_state.proxy.hostname, 60 - strwidth(l3) - x, &s[1], NEWT_FLAG_SCROLL);
    l[2] = newtLabel(65 - strwidth(l3), height + 4, l3);
    snprintf(txtbuf, sizeof txtbuf - 1, "%d", nf_state.proxy.port);
    e[2] = newtEntry(66, height + 4, txtbuf, 6, &s[2], NEWT_FLAG_SCROLL);
  }

  b1 = newtCompactButton(19, height + 6, _("OK"));
  b2 = newtCompactButton(29, height + 6, _("Cancel"));
  b3 = newtCompactButton(42, height + 6, _("Help"));

  form = newtForm(NULL, NULL, 0);
  newtFormAddComponents(form, textbox, l[0], e[0], l[1], e[1], l[2], e[2], b1, b2, b3, NULL);

  do {
    ans = newtRunForm(form);
    free(nf_state.proxy.hostname);
    nf_state.proxy.hostname = strdup(s[1]);
    nf_state.proxy.port = atoi(s[2]);
    if (ans == b3) {
      problemBox(_("Debian supports installing drivers and the base system over the Internet.  Currently, only HTTP retrieval is supported.\nPlease enter some information so that the files can be retrieved.\n\nDownload URL: the URL of the directory containing the file to be downloaded.  The default location should work on Debian standard servers.\nProxy: a proxy server to use to fetch the URL, or \"none\" if you don't need one.\nProxy Port: the port that the proxy server listens on."),
                 _("Help For Network Installation"));
    }
    else if (ans == b2 || nf_parse_server_url(s[0]) == 0) {
      break;
    } 
  }
  while (1);

  newtPopWindow();

  /* We cannot destroy the form until after we've used the value
   * from the entry widget. */
  newtFormDestroy(form);

  if (ans == b2)  /* cancel */
    return 1;

/* 
 * write the HTTP_PROXY user config, as used by base-config 
 * This may get written several times, but the last entry is
 * likely to be correct and that's what base-config will 
 * end up using.
 * 
 * Also set some environmental vars since wget grabs 
 * http_proxy/ftp_proxy if necessary 
 */
  if ( strcmp(nf_state.proxy.hostname, "none") != 0 ) {
      int rv;
      char *ptr;
     
      /* maybe user put a http:// or ftp:// in the proxy hostname,
       * silently remove it if necessary. */
      ptr = strstr(nf_state.proxy.hostname, "://");
      if (ptr) {
	  char *temp;
	  temp = nf_state.proxy.hostname;
	  nf_state.proxy.hostname = strdup(ptr+3); /* the actual hostname */
	  free (temp);
      }

      snprintf(txtbuf, sizeof txtbuf, "%s://%s:%d",
	      nf_state.method,
	      nf_state.proxy.hostname,
	      nf_state.proxy.port);
      if (strcmp (nf_state.method, "ftp") == 0) {
	  rv = setenv("ftp_proxy", txtbuf, 1);
	  write_userconfig("FTP_PROXY", txtbuf);
      } else {
	  rv = setenv("http_proxy", txtbuf, 1);
	  write_userconfig("HTTP_PROXY", txtbuf);
      }
      if (rv == -1){
	  problemBox(_("failure"), _("Could not set http/ftp proxy."));
	  return 1;
      }
  }
  else {
      unsetenv("http_proxy");
      unsetenv("ftp_proxy");
      write_userconfig("FTP_PROXY", NULL);
      write_userconfig("HTTP_PROXY", NULL);
  }
  
  return 0;
}


/*
 * Initialize net-fetch.
 */

void
nf_initialize(void)
{
  if (nf_state.initialized)
    return;

  nf_state.initialized = 1;

  // http://http.us.debian.org/debian/dists/potato/main/disks-$arch/current/
  nf_state.method = strdup("http");
  nf_state.server.hostname = strdup("http.us.debian.org");
#ifdef USE_LANGUAGE_CHOOSER
  find_opt_server(nf_state.server.hostname);
#endif  
  nf_state.server.port = 80;
  nf_state.proxy.hostname = strdup("none");
  nf_state.proxy.port = 8080;
  snprintf(prtbuf, sizeof(prtbuf) - 1, "debian/" ARCHIVE_LOCATION "/", ARCHNAME);
  nf_state.path = strdup(prtbuf);

#if defined (_TESTING_)
  // nf_state.proxy.hostname = strdup("zuma.kawa.dhs.org");
  // nf_state.proxy.port = 8080;
  // nf_state.server.hostname = strdup("zuma.kawa.dhs.org");
  // nf_state.server.port = 7070;
  // nf_state.path = strdup("debian-boot/feb18");
#endif
}


/*
  exec wget as indicated by argv

  read the progress bar on stderr and convert it into a pretty GUI
  progress bar like the rest of dbootstrap.

  Do we want ETA as well, I removed it to save space since the
  progress bar already gives a good idea.
  if ((rv = regcomp (&eta, "[[:digit:]]*:[[:digit:]]*[[:space:]]*ETA", 0)) != 0)
*/
static int
nf_fetchfile (char *remote_filename, char *local_filename)
{  
  int from_wget[2];
  char buf[256];
  pid_t pid;
  int bytes, status;
  regex_t percent;
  regmatch_t pmatch_percent[1];
  char *argv[5];
  char url[128];
  char msg[128];
  char *full_remote_filename;

  full_remote_filename = concat_paths (nf_state.path, remote_filename);

  argv[0] = WGET;
  argv[1] = "-O";
  argv[2] = local_filename;
  snprintf (url, sizeof url, "%s://%s:%d/%s",
	    nf_state.method, 
	    nf_state.server.hostname, 
	    nf_state.server.port,
	    full_remote_filename);
  argv[3] = url;
  argv[4] = NULL;
  
  snprintf(msg, sizeof msg, 
           "Host:  %s://%s:%d\nFile:  %s\n", 
	   nf_state.method,
	   nf_state.server.hostname, nf_state.server.port,
	   full_remote_filename);

  free(full_remote_filename);
  
  if ( strcmp(nf_state.proxy.hostname, "none") != 0 ) {
      snprintf(msg + strlen(msg), sizeof msg - strlen(msg),
	      "Proxy: %s:%d", nf_state.proxy.hostname, 
	      nf_state.proxy.port);
  }
  pipe (from_wget);
  if ((pid = fork ()) == 0)
    {
      close (from_wget[0]);
      if (dup2 (from_wget[1], 2) == -1)
	perror ("dup2");

      if (execv (argv[0], argv) != 0)
	perror ("execv");
      return -1;
    }
  else if (pid == -1)
    {
      perror ("fork");
      return -1;
    }

  close (from_wget[1]);

  if (regcomp (&percent, "[^[:space:]]*%", 0) != 0) {
      fprintf(stderr, "recomp failed.\n");
      return -1;
  }

  boxProgress (PROGRESS_INIT, _("Downloading"));
  boxProgress (PROGRESS_SETTEXT, msg);
  
  while ((bytes = read (from_wget[0], buf, sizeof (buf))) > 0)
    {
      buf[bytes] = '\0';
      if (regexec (&percent, buf, 1, pmatch_percent, 0) != REG_NOMATCH)
	{
	  snprintf (buf, sizeof buf, "%d 100",
		    atoi (buf + pmatch_percent[0].rm_so));
	  boxProgress (PROGRESS_UPDATE, buf);
	}
    }

  boxProgress (PROGRESS_CLOSE, NULL);
  regfree (&percent);

 if (waitpid(pid, &status, WNOHANG)  && (WIFEXITED(status) != 0 ))
   {
    int rv = WEXITSTATUS(status);
    if (rv != 0 ) 
      {
        /* the last line from wget should contain some
	 * diagnostic info */
	vaproblemBox (_("An Error occured"), "%s%s\n", msg, buf);
        return rv;
       }
     return 0;
   }
 else 
   {
     /* no child available, try to kill it */
     kill(SIGKILL, pid);
     return 1;
    }

  return 0;

}

static int
nf_install(int series)
{
  char        txtbuf[2048];
  int         i;
  char        filename[MAXPATHLEN];
  struct stat statbuf;
  int         len;

  struct {
    char *local;
    char *remote;
    int   (*install_fn) (const char *);
    int   series;
  }
  files_to_fetch[] = {
    { "rescue.bin",   kernel_image_path, nf_extract_kernel,  SERIES_OS   },
    { "drivers.tgz",  drivers_path,      nf_extract_drivers, SERIES_OS   },
    { NULL,           NULL,              NULL,               0           }
  };

  if (!NAME_ISDIR(target_path("/tmp"), &statbuf)) {
    snprintf(txtbuf, sizeof txtbuf - 1, "mkdir %s", target_path("/tmp"));
    if (execlog(txtbuf, LOG_INFO)) {
      vaproblemBox(_("Directory creation failed"),
                   _("There was a problem creating the directory %s."),
                   target_path("/tmp"));
      return 1;
    }
  }

  len = snprintf(txtbuf, sizeof txtbuf - 1,
                 _("\nGoing to download the following files over a HTTP connection:\n"));
  for (i = 0; files_to_fetch[i].remote != NULL; i++) {
    if (series != files_to_fetch[i].series)
      continue;
    len += snprintf(txtbuf + len, sizeof txtbuf - len - 1, "  %s\n", files_to_fetch[i].remote);
  }
  wideMessageBox(txtbuf, _("Fetching installation files over the network"));

  nf_initialize();

  switch (nf_select_server()) {
    case 0:
      break;
    default:
    case 1:
      return 1;
      /* NOTREACHED */
  }

  /*
   * First fetch the required files.
   */
  for (i = 0; files_to_fetch[i].remote != NULL; i++) {
    if (series != files_to_fetch[i].series)
      continue;

    snprintf(filename, sizeof filename - 1, target_path("/tmp/%s"), 
      files_to_fetch[i].local);

    INFOMSG("retrieving %s from %s", filename, files_to_fetch[i].remote);

    if (nf_fetchfile(files_to_fetch[i].remote, filename) != 0) {
      vaproblemBox(_("Problem"), _("Download failed for the file '%s'."),
                   files_to_fetch[i].remote);
      return 1;
    }

    INFOMSG("successfully retrieved %s", filename);
  }

  /*
   * Now install the required files.
   */
  for (i = 0; files_to_fetch[i].remote != NULL; i++) {
    if (series != files_to_fetch[i].series)
      continue;

    snprintf(filename, sizeof filename - 1, "/tmp/%s", files_to_fetch[i].local);

    INFOMSG("installing %s", filename);

    if (files_to_fetch[i].install_fn(filename) != 0) {
      vaproblemBox(_("Problem"), _("Installation of the file '%s' failed."), filename);
      return 1;
    }

    INFOMSG("successfully installed %s", filename);
  }

  return 0;
}



#ifdef _TESTING_
int
main ()
{
  LOAD_TRMFILE ("test.trm");
  get_kver ();
  boxInit ();

  nf_initialize();
  nf_select_server ();  

  nf_fetchfile ("ls-lR.gz", "foo");

  boxFinished();

  return 0;
}
#endif /* _TESTING_ */



#if 0

/* I didn't remove this because I'm nice, but it doesnt' compile anymore.
 * dwhedon@debian.org */

#ifdef _TESTING_
/*
 * Source code for testing net-fetch.  This may require some tweaking on
 * your system.  Please don't remove this code.  (marcel@debian.org)
 */

int
install_from_file (const char *filename, const char *descr)
  {
    int          status;
    struct stat  statbuf;
    const char   tmpdir [] = "/tmp/notarget/tmp";
    char         instdir [128];

    snprintf (instdir, sizeof instdir - 1, "%s/drivers", tmpdir);
    if (!NAME_ISDIR (tmpdir, &statbuf))
      {
        if (!mkdir (tmpdir, 0777))
          {
            chown (tmpdir, 0, 3); /* root_uid= 0 , sys_gid= 3 */
	    chmod (tmpdir, 01777);
          }
      }

    if (!NAME_ISDIR (instdir, &statbuf))
      {
        if (!mkdir (instdir, 0777))
          {
            chown (instdir, 0, 3); /* root_uid= 0 , sys_gid= 3 */
          }
      }

    if (!NAME_ISDIR (instdir, &statbuf))
      {
        vaproblemBox("Directory Error",
                     "Creation of the temporary directory '%s' failed.",
                     instdir);
        return 1;
      }

    sprintf (prtbuf, "Installing %s from %s...", descr, filename);
    pleaseWaitBox (prtbuf);
    chdir (instdir);

    sprintf (prtbuf, "zcat %s | tar -tf -", filename); /* list files, see system logs. */
    status = execlog (prtbuf, LOG_INFO);

    if (status)
      {
        boxPopWindow ();
        chdir ("/");
        vaproblemBox("File Error",
                     "There was a problem extracting the %s from the file '%s'.",
                     descr, filename);
        return 1;
      }
    else
      {
        status = 0;  // status = execlog ("./install.sh /target", LOG_INFO);
        chdir ("/");
        boxPopWindow ();
        if (status)
          {
            vaproblemBox("Install Script Error",
                         "There was a problem installing the %s: the install script, 'install.sh', failed.",
                         descr, filename);
            return 1;
          }
      }

    return 0;
  }


char *
get_device (const char *diskimage)
  {
    // char *device = find_unused_loop_device();
    // int   ro     = 1;
    //
    // sprintf (prtbuf, "%s/%s", Archive_Dir, diskimage);
    // if (set_loop(device,prtbuf,0,&ro))
    //   {
    //     syslog (LOG_ERR, "set_loop failed for %s on %s", prtbuf, device);
    //     return NULL;
    //   }
    // else
    //   {
    //     syslog (LOG_INFO, "set_loop for %s on %s", prtbuf, device);
    //     return device;
    //   }
    return "/dev/loop0";
  }


void
release_device (const char *device)
  {
    // del_loop(device);
  }


int
install_floppy (char *device, const char *type, const char *text)
  {
    int status = 0;

    // status = mount_and_check_floppy (device, type, text);
    // if (status == DLG_CANCEL)
    //   {
    //     return DLG_CANCEL;
    //   }
    //   else
    //   {
    //     return 1;
    //   }

    sprintf (prtbuf, "Installing the %s ", text);
    if (strncmp (device, "/dev/loop", 8))
      {
        strcat (prtbuf,"...");
      }
    else
      {
        strcat (prtbuf, "from floppy images on mounted medium ...");
      }

    pleaseWaitBox (prtbuf);

    sleep (3);
    // chdir ("/floppy");
    // status = execlog ("./install.sh /target", LOG_INFO);
    // chdir ("/");
    // execlog ("umount /floppy", LOG_DEBUG);

    boxPopWindow ();

    if (status)
      {
        return 1;
      }

    return 0;
  }


int
extract_from_file (const char *fil, const char *descr)
  {
    int status;

    sprintf (prtbuf, "Extracting %s from %s...", descr, fil);
    pleaseWaitBox (prtbuf);

    chdir (target_path (""));
    sprintf (prtbuf, "zcat %s | tar -tf -", fil); /* list files, see system logs. */
    status = execlog (prtbuf, LOG_INFO);
    chdir ("/");

    boxPopWindow ();

    if (status)
      {
        vaproblemBox("File error!",
                     "There was a problem extracting the %s from the file '%s'.",
                     descr, fil);
        return 1;
      }

    return 0;
  }


int
main (int argc, char *argv [])
  {
    struct stat statbuf;
    char        txtbuf [2048];

    assert (LOAD_TRMFILE ("test.trm") != 0);

    get_kver ();
    get_subarch_name ();

    if (bootargs.disksize == NULL)
      {
#if #cpu(mips) && defined (MIPSEL)
        bootargs.disksize = "2.88";
#else
        bootargs.disksize = "1.44";
#endif
      }
    bootargs.flavor = "compact"; // compact, safe, standard
    bootargs.isquiet = 0;

    setup_image_names ();

    Archive_Dir = strdup ("netfetch");
    if (!NAME_ISDIR (target_path (""), &statbuf))
      {
        snprintf (txtbuf, sizeof txtbuf - 1, "mkdir %s", target_path (""));
        system (txtbuf);
        /* @@ Marcel TODO: check for errors here. */
      }

    boxInit ();

    wideMessageBox ("\nInstallation of operating system and modules via net-fetch.\n", "TEST");
    nf_install_os ();

    wideMessageBox ("\nInstallation of the base system via net-fetch.\n", "TEST");
    nf_install_base ();

    wideMessageBox ("\nNet-fetch tests completed.\n", "TEST");
    boxFinished ();

    return EXIT_SUCCESS;
  }
#endif /* _TESTING_ */

#endif
