#include <ctype.h>
#include <fcntl.h>
#include <glob.h>
#include <limits.h>
#include <linux/fd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h> 
#include <string.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <syslog.h>
#include <termios.h>
#include <unistd.h>
#include <slang.h>
#include <newt.h>

/* for __NR_sysfs */
#include <asm/unistd.h>

#ifdef USE_LANGUAGE_CHOOSER
#include <wchar.h>

char *mirror_list[] = { 
"at", "au", "bg", "br", "cz", "de", "dk", "ee", "es", "fi", "fr", "uk", "hk",
"hr", "hu", "it", "jp", "nl", "no", "nz", "pl", "ru", "se", "si", "tr", "us",
NULL };


char *mirror_country_map[] = { 
/*   "en", "us", */
   "ca", "es", /*
   "cs", "us",
   "da", "us", */
   "de", "de",
/*   "eo", "us", */
   "es", "es",
   "fi", "fi",
   "fr", "fr",
   "gl", "es",
   "hr", "hr",
   "hu", "hu",
   "it", "it",
   "ja", "jp",
   "pl", "pl",
/*   "pt", "us", */
   "ru", "ru",
/*   "sk", "us",
   "sv", "us", */
   "tr", "tr",
   NULL };

#endif

#include "dbootstrap.h"
#include "lang.h"
#include "util.h"

/* for documentation see util.h */

int execlog (const char *incmd, int priority)
{ 
  FILE *output; 
  char line[MAXLINE]; 
  char cmd[strlen(incmd) + 6]; 
  strcpy(cmd, incmd); 

  if (bootargs.isdebug) {
    openlog(SYSLOG_IDENT, LOG_PID, LOG_USER);
    syslog(LOG_DEBUG, "running cmd '%s'", cmd);
  }
  else {
    openlog(SYSLOG_IDENT, LOG_PID, LOG_USER);
  }

  /* FIXME: this can cause the shell command if there's redirection
            already in the passed string */
  strcat(cmd, " 2>&1"); 
  output = popen(cmd, "r"); 
  while (fgets(line, MAXLINE, output) != NULL ) { 
    syslog(priority, line); 
  }
  closelog();
  return(pclose(output)); 
}

int
vexeclog(int priority, const char *fmt, ...) {
    va_list ap;

    va_start (ap, fmt);
    vsnprintf (prtbuf, sizeof (prtbuf) - 1, fmt, ap);
    va_end (ap);

    return execlog (prtbuf, priority);
}

int fullscreen_execlog  (const char *cmd)
{
  int ret;

  DEBUGMSG("running cmd fullscreen '%s'", cmd);

  newtCls();
  SLsmg_gotorc(0, 0);
  boxSuspend();
  ret = system(cmd);
  boxResume();

  if (ret != 0) {
    ERRMSG("command %s returned %d", cmd, ret);
  }

  return(ret);
}

int execlog_to_string(const char *cmd, int *exitcode,
		      char *result, int len, int flag)
{
  int retval = 0;
  int stdoutpipe[2];
  int stderrpipe[2];
  int childpid;
  int newlen = 0;
  char buf[1024];
  int stdout_eof = 0;
  int stderr_eof = 0;
  
  fd_set readfdset;
  
  if (pipe(stdoutpipe))
    {
      ERRMSG(_("Couldn't create pipe: %s"), strerror(errno));
      return -1;
    }

  if (flag != 0)
    if (pipe(stderrpipe))
      {
	ERRMSG(_("Couldn't create pipe: %s"), strerror(errno));
	return -1;
      }
  
  if ((childpid = fork()) < 0)
    {
      ERRMSG(_("Couldn't fork: %s"), strerror(errno));
      return -1;
    }
  else if (childpid == 0)
    {
      /* child */
      close(stdoutpipe[0]);
      if (flag != 0)
	close(stderrpipe[0]);
      dup2(stdoutpipe[1], 1);
      close(stdoutpipe[1]);
      if (flag != 0)
	{
	  dup2(stderrpipe[1], 2);
	  close(stderrpipe[1]);
	}
      else
	{
	  int fd;
	  if ((fd = open("/dev/null", O_WRONLY)) < 0)
	    {
	      ERRMSG(_("Couldn't open /dev/null: %s"), strerror(errno));
	      exit(1);
	    }
	  dup2(fd, 2);
	  close(fd);
	}
      if (execl("/bin/sh", "/bin/sh", "-c", cmd, NULL))
	{
	  ERRMSG(_("Couldn't run /bin/sh -c '%s': %s"), cmd, strerror(errno));
	  exit(1);
	}
    }
  /* parent */
  close(stdoutpipe[1]);
  if (flag != 0)
    close(stderrpipe[1]);

  if (bootargs.isdebug) {
    openlog(SYSLOG_IDENT, LOG_PID, LOG_USER);
    syslog(LOG_DEBUG, "running cmd '%s'", cmd);
  }
  else {
    openlog(SYSLOG_IDENT, LOG_PID, LOG_USER);
  }

  /* save space for the null terminator */
  if (--len <= 0)
    {
      retval = -1;
      goto done;
    }

  while (1)
    {
      int status;
      int cur;

      FD_ZERO(&readfdset);
      if (!stdout_eof)
	FD_SET(stdoutpipe[0], &readfdset);
      if (flag != 0 && !stderr_eof)
	FD_SET(stderrpipe[0], &readfdset);

      /* If there is nothing to read, and the process has exited, just
	 return the exit status. */
      if ((stdout_eof && (stderr_eof || flag == 0))
	  || (select(FD_SETSIZE, &readfdset, NULL, NULL, NULL) == 0
	      && errno != EINTR))
	{
	  waitpid(childpid, &status, 0);
	  if (WIFEXITED(status))
	    *exitcode = WEXITSTATUS(status);
	  else if (WIFSIGNALED(status) || WCOREDUMP(status))
	    *exitcode = 1;
	  result[newlen] = '\0';
	  retval = 0;
	  goto done;
	}

      /* Check stdout */
      if (FD_ISSET(stdoutpipe[0], &readfdset))
	{
	  if ((cur = read(stdoutpipe[0], result+newlen,
			  len-newlen < 0 ? 0 : len-newlen)) < 0)
	    {
	      retval = -1;
	      goto done;
	    }
	  else if (cur == 0)
	    {
	      stdout_eof = 1;
	      close(stdoutpipe[0]);
	    }
	  else
	    newlen += cur;
	}
      /* Check stderr */
      if (flag != 0 && FD_ISSET(stderrpipe[0], &readfdset))
	{
	  if (flag == 1)
	    {
	      if ((cur = read(stderrpipe[0], buf, sizeof(buf)-1)) < 0)
		{
		  retval = -1;
		  goto done;
		}
	      else if (cur == 0)
		{
		  stderr_eof = 1;
		  close(stderrpipe[0]);
		}
	      else
		{
		  buf[cur] = '\0';
		  syslog(LOG_ERR, buf[cur-1] == '\n' ? "%s" : "%s\n", buf);
		}
	    }
	  else if (flag == 2)
	    {
	      if ((cur = read(stderrpipe[0], result+newlen,
			      len-newlen < 0 ? 0 : len-newlen)) < 0)
		{
		  retval = -1;
		  goto done;
		}
	      else if (cur == 0)
		{
		  stderr_eof = 1;
		  close(stderrpipe[0]);
		}
	      else
		newlen += cur;
	    }
	}
    }
 done:
  closelog();
  return retval;
}


int check_target_path(void)
{
  if (!Root) {
    /* need to create the temp dir */
    return mkdir(TARGET_NOPREFIX, 0755);
  }
  return 0;
}

const char* target_path(const char* s)
{
  static char buf[PATH_MAX + 1];
  const char* prefix;
#ifndef _TESTING_
  if (Root)
    prefix = TARGET_PREFIX;
  else
    prefix = TARGET_NOPREFIX;
#else
  prefix = TARGET_TESTDIR;
#endif
  strcpy(buf, prefix);
  strcat(buf, s);
  return buf;
}


int check_dir(const char* dirname)
{
  struct stat check;
  if(stat(dirname, &check) == -1)
    return -1;
  else
    return S_ISDIR(check.st_mode);
}


/* for documentation see util.h */
int write_userconfig (const char *key, const char *value)
{
  FILE *conffile;
  const char *path;
  
  if ( check_target_path() != 0 ) {
    ERRMSG("unable to write settings, target root not mounted yet?");
  }

  path = target_path(USER_SETTINGS_DIR);
  if (check_dir(path) == -1)
    mkdir(path, 0700);
  
  if ((conffile = fopen(target_path(USER_SETTINGS_FILE), "a")) == NULL)
      return 1;

  if (value)
      fprintf(conffile, "%s='%s'\n", key, value);
  else
      fprintf(conffile, "%s=''\n", key);

  fclose(conffile);

  return 0;
}

/* for documentation see util.h */
int write_userconfig_va (const char *key, const char *fmt, ...)
{
  char *p;
  va_list ap;
  if ((p = malloc(MAXLINE)) == NULL)
    return -1;
  
  va_start(ap, fmt);
  (void) vsnprintf(p, MAXLINE, fmt, ap);
  va_end(ap);

  return write_userconfig(key, p);
}


/* for documentation see util.h */
char *normalize_dir (const char *dir, char *buf, size_t bufsize)
{
    char *result;

    if (dir[0] == '/')
    {
        struct stat ds;

        if (stat (dir, &ds) == 0 && S_ISDIR (ds.st_mode))
        {
            strncpy (buf, dir, bufsize - 1);
            buf[bufsize - 1] = '\0';

            result = buf;
        }
        else
            result = NULL;
    }
    else
        for (;;)
        {
            result = NULL;

            {
                int dd = open (dir, O_RDONLY);
                struct stat ds;

                if (dd == -1)
                    break;

                if (fstat (dd, &ds) == -1 || !S_ISDIR (ds.st_mode))
                {
                    close (dd);

                    break;
                }

                {
                    char *tempo = malloc (PATH_MAX);

                    getcwd (tempo, PATH_MAX);   /* Store the current directory for later reuse */

                    fchdir (dd);

                    getcwd (buf, bufsize);

                    chdir (tempo);

                    free (tempo);
                }
            }

            result = buf;

            break;
        }

    return result;
}

int
is_network_configured(void)
{
    struct stat statbuf;

    if ( (NAME_ISREG(target_path("/etc/network/interfaces"),&statbuf)) &&
	((NAME_ISEXE("/sbin/ifconfig",&statbuf)) ||
	 (NAME_ISEXE(target_path("/sbin/ifconfig"),&statbuf))))
	return 1;
    return 0;
}

inline int
is_filesystem_supported(const char *fstype)
{
  return (syscall(__NR_sysfs, 1, fstype) < 0);
}
/*
char *
fs_modules()
{
  FILE *proc;
  int i=0;
  char *space, *list, line[32];
  
  proc=fopen("/proc/modules","r");
  if ( proc ==  NULL )
    {
      perrorBox(_("Can't read /proc/modules"));
      return NULL;
    }
  while(!feof(proc)) {
    fgets(line, 31, proc);
    space = (strchr(line, ' '));
    if(space != NULL) 
    {
      *space = '\0'; 
      if(!strcmp(line, "reiserfs"))
          addtolist(&list, line);
      if(!strcmp(line, "ext3"))
          addtolist(&list, line);
      if(!strcmp(line, "xfs"))
          addtolist(&list, line);
      if(!strcmp(line, "umsdos"))
          addtolist(&list, line);
      if(!strcmp(line, "ext2"))
          addtolist(&list, line);
      if(!strcmp(line, "minix"))
          addtolist(&list, line);
    }
  }
  return list;
}
*/

int
is_fstype (const char *mount, const char *fs)
{
  FILE *proc;
  int match=0;
  char dev[80], dir[80], type[80];

  proc = fopen ("/proc/mounts", "r");
  if ( proc ==  NULL )
    {
      perrorBox(_("Can't read /proc/mounts"));
      return -1;
    }

  while ( EOF != fscanf (proc, "%s %s %s %*s %*i %*i\n",
			 dev, dir, type))
    {
      if (!strncmp (mount, dev, 79))
	{
	  match = strncmp (fs, type, 79) ? 0 : 1;
	  break;
	}

      if (!strncmp (mount, dir, 79))
	{
	  match = strncmp (fs, type, 79) ? 0 : 1;
	  break;
	}

    }
  fclose (proc);

  return match;
}

int
is_interface_up(char *inter)
{
    struct ifreq ifr;
    int sfd = -1, ret = -1;

    if ((sfd = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
	goto int_up_done;

    strncpy(ifr.ifr_name, inter, sizeof(ifr.ifr_name));

    if (ioctl(sfd, SIOCGIFFLAGS, &ifr) < 0)
	goto int_up_done;

    ret = (ifr.ifr_flags & IFF_UP) ? 1 : 0;

int_up_done:
    if (sfd != -1)
	close(sfd);
    return ret;
}

int
is_network_up(int all) {
    int ret = 0;
    char *buf;

    getif_start();
    while ((buf = getif(1)) != NULL)
      if (all || is_interface_up(buf))
	    ret++;
        
    getif_end();

    return ret;
}

static FILE *ifs = NULL;
static char ibuf[512];

void
getif_start(void) {
    if (ifs != NULL) {
	fclose(ifs);
	ifs = NULL;
    }
    if ((ifs = fopen("/proc/net/dev", "r")) != NULL) {
	fgets(ibuf, sizeof(ibuf), ifs); /* eat header */
	fgets(ibuf, sizeof(ibuf), ifs); /* ditto */
    }
    return;
}

void
getif_end(void) {
    if (ifs != NULL) {
	fclose(ifs);
	ifs = NULL;
    }
    return;
}

char *
getif(int all) {
    char rbuf[512];
    if (ifs == NULL)
	return NULL;

    if (fgets(rbuf, sizeof(rbuf), ifs) != NULL ) {
	get_name(ibuf, rbuf);
	if (!strcmp(ibuf,"lo")) /* ignore the loopback */
	    return getif(all); /* seriously doubt there is an infinite number of lo devices */
	if (all || is_interface_up(ibuf) == 1)
	    return ibuf;
    }
    return NULL;
}

void
get_name(char *name, char *p)
{
    while (isspace(*p))
	p++;
    while (*p) {
	if (isspace(*p))
	    break;
	if (*p == ':') {        /* could be an alias */
	    char *dot = p, *dotname = name;
	    *name++ = *p++;
	    while (isdigit(*p))
		*name++ = *p++;
	    if (*p != ':') {    /* it wasn't, backup */
		p = dot;
		name = dotname;
	    }
	    if (*p == '\0')
		return;
	    p++;
	    break;
	}
	*name++ = *p++;
    }
    *name++ = '\0';
    return;
}

int
initchroot_cap(void)
{
    struct stat statbuf;
    return 0; /* default for now */
    if (NAME_ISREG("/proc/sys/kernel/init-chroot",&statbuf))
	return 1;
    else
	return 0;
}

/* This may get more complex...leave it simple for now */
int
is_live_cd(void)
{
   struct stat statbuf;
   if (NAME_ISDIR("/dists",&statbuf))
      return 2;
   else
      return 0;
}


int
check_pending_config(void)
{
  const char *path;
  struct stat statbuf;

  if (check_dir(TARGET_NOPREFIX)!=-1) {
    /*
     * /tmp/notarget exists: network was previously configured with no
     * target system mounted.
     * Just move the notarget tree to /target
     */
    execlog("cp -pR " TARGET_NOPREFIX "/. " TARGET_PREFIX, LOG_INFO);
    /* also, readjust the symlink if needed for cdrom filesystem */
    if (is_live_cd() == 2) {
	unlink("/tmp/resolv.conf");
	symlink(target_path("/etc/resolv.conf"), "/tmp/resolv.conf");
    }
  }

  /* 
   * first create our config file, and the USER_SETTINGS_DIR too, incidentally
   */
  if ( write_userconfig("BUILDTIME", BUILDTIME) != 0 ) { /* check that we can write config  */
    ERRMSG("failed to create " USER_SETTINGS_FILE);
    vaproblemBox(_("Failed while saving dbootstrap settings in %s"), USER_SETTINGS_FILE);
    return 1;                   /* might as well give up now */
  }

  /* 
   * copy settings if any from the kbdconfig step
   */
  if (NAME_ISREG(KEYBD_SETTINGS_FILE, &statbuf)) {
    path = target_path(USER_SETTINGS_FILE);
    snprintf(prtbuf, sizeof(prtbuf) - 1, "cat " KEYBD_SETTINGS_FILE " >> %s", path);
    execlog(prtbuf, LOG_INFO);
  }

  /* 
   * note any significant boot flags
   */
  if ( bootargs.isverbose )
    write_userconfig("VERBOSE", "yes");
  if ( bootargs.isquiet )
    write_userconfig("VERBOSE", "quiet");
  if ( bootargs.isdebug )
    write_userconfig("DEBUG", "true");
/*fixme later, was ist dieses write_userconfig  if ( is_live_cd() > 0 )
    write_userconfig("CDROM", "true");
*/

#ifdef USE_LANGUAGE_CHOOSER
  if ( lang != NULL ) {
    /* The locale should be generated by base-config if needed, and ignored
       if it can't be used. */
    write_userconfig("LANG_INST", lang->locale);

    /* Specify priority list of translations to use when displaying messages */
    write_userconfig("LANGUAGE_INST", lang->msgcat);
  }
#endif

#if #cpu(powerpc)
  if ( ( 0 == strcmp( Arch2, "PowerMac" ) ) && ( NULL != Arch3 ) && 
	( 0 == strcmp( Arch3, "NewWorld" ) ) ) {
    struct fdisk_partition *bootstrap;
    char *message = NULL, *title = NULL;

    bootstrap = fdisk_find_partition_by_type( PTYPE_MAC_BOOT );
    if ( NULL == bootstrap ) {
	message = _("No Apple_Bootstrap partition was detected.");
	title = _("Missing Bootstrap Partition");
    }
    else if ( 800 > bootstrap->size ) {
	message = _("The Apple_Bootstrap partition must be at least 800K.");
	title = _("Bootstrap Partition Too Small");
    }
    else if ( 0 != strcmp( bootstrap->disk->name, Root->disk->name ) ) {
	message = _("The Apple_Bootstrap partition must reside on the same disk "
		      "as the root partition.");
	title = _("No Bootstrap Partition On Root Disk");
    }

    if (message != NULL && title != NULL)
    {
        snprintf (prtbuf, sizeof (prtbuf), _("%s  You should repartition your disk.\n\n"
		      "If you do not repartition and add an 800K Apple_Bootstrap "
		      "partition you will not be able to make the disk bootable."), message);

        problemBox (prtbuf, title);
    }
  } 
  else if ( 0 == strcmp( Arch2, "chrp" ) ) {
    struct fdisk_partition *bootstrap;
    char *message = NULL, *title = NULL;

    bootstrap = fdisk_find_partition_by_type( PTYPE_CHRP_BOOT );
    if ( NULL == bootstrap ) {
	message = _("No CHRP bootstrap partition was detected.");
	title = _("Missing Bootstrap Partition");
    }
    else if ( 800 > bootstrap->size ) {
	message = _("The CHRP bootstrap partition must be at least 800K.");
	title =_("Bootstrap Partition Too Small");
    }
    else if ( 0 != strcmp( bootstrap->disk->name, Root->disk->name ) ) {
        message = _("The CHRP bootstrap partition must reside on the same disk as the root partition.");
        title = _("No Bootstrap Partition On Root Disk");
    }

    if (message != NULL && title != NULL)
    {
        snprintf (prtbuf, sizeof (prtbuf), _("%s  You should repartition your disk.\n\n"
		      "If you do not repartition and add an 800K type 0x41 "
		      "partition you will not be able to make the disk bootable."), message);

        problemBox (prtbuf, title);
    }
  }
#endif

  return 0;
}

/*
 * Used, so far, by http-fetch.c:nf_http_fetchfile() and
 * choose_medium.c:choose_NFS_server().
 *
 * See "util.h" for documentation.
 */
sigjmp_buf env;
struct termios term_settings;
void (*old_sigint) (int);

sig_atomic_t threw;

static void
sigint (int signum)
{
  threw = 1;
  siglongjmp (env, 1);
}

/*
 * Potentially the `KDSIGACCEPT' console ioctl could be used for
 * something like this also.  One or several keys could be temporarily
 * rebound so that they generate the `KeyboardSignal' keysym, with an
 * "unwind-protect" type mechanism in the signal handler that resets
 * them to their normal behaviour before it throws, or something like
 * that. - karlheg
 *
 */

int
do_interruptable (const char *message,
		  void (*thunk) (void *result),
		  void *result)
{
  int saved_errno = 0;
  const char *press_cc = _("  Press Ctrl-C to interrupt.");
  char msg_buf[ strlen (message) + strlen (press_cc) + 1 ];
  snprintf (msg_buf, sizeof (msg_buf), "%s%s", message, press_cc);
  if (! sigsetjmp (env, 1)) {
    struct termios new_term_settings;
    tcgetattr (0, &term_settings);
    memcpy (&new_term_settings, &term_settings, sizeof (struct termios));
    new_term_settings.c_lflag |= NOFLSH;
    new_term_settings.c_lflag |= ISIG;
    /* Allow only VINTR (C-c) */
    new_term_settings.c_cc[VINTR]  = '\003';
    new_term_settings.c_cc[VQUIT]  = _POSIX_VDISABLE;
    new_term_settings.c_cc[VSUSP]  = _POSIX_VDISABLE;
    tcsetattr (0, TCSANOW, &new_term_settings);
    threw = 0;
    old_sigint = signal (SIGINT, sigint);
    pleaseWaitBox (msg_buf);
    thunk (result);
    saved_errno = errno;
  }
  tcsetattr (0, TCSANOW, &term_settings);
  signal (SIGINT, old_sigint);
  boxPopWindow();
  errno = saved_errno;
  return (threw);
}

#ifdef _UTIL_TEST_
int main (int argc, char **argv) { 
  int i=0; 
  int ret; 
  openlog ("execlog test", LOG_CONS | LOG_PID, LOG_USER);
  i++; 
  ret=execlog("echo 'I AM the pumpkin king'", LOG_INFO); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("cat /no_such_goddamn_file", LOG_INFO); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("echo 'P1i1p1e1s 1w1o1r1k' | sed -e 's/1//g'", LOG_INFO); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("echo 'LOG_INFO message'", LOG_INFO); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("echo 'LOG_NOTICE message'", LOG_NOTICE); 
  printf("Command %d returned %d\n", i, ret); 
  i++; 
  ret=execlog("echo 'LOG_ALERT message'", LOG_ALERT); 
  i++; 
  ret=execlog("echo 'Command with stderr redirected' 2> /tmp/test.out", LOG_INFO); 
  i++;
  ret=execlog("cat /etc/motd command_with_stdout_redirected > /tmp/test.out", LOG_INFO); 
}; 

#endif 
  
 


/*
 * ---------------------------------------------------------
 * Utility routines swiped from busybox, because I'm sick of
 * my changes to busybox internals breaking dbootstrap.
 * That is just stupid.
 *  -Erik <andersee@debian.org>
 */



/*
 * Walk down all the directories under the specified 
 * location, and do something (something specified
 * by the fileAction and dirAction function pointers).
 *
 * Unfortunatly, while nftw(3) could replace this and reduce 
 * code size a bit, nftw() wasn't supported before GNU libc 2.1, 
 * and so isn't sufficiently portable to take over since glibc2.1
 * is so stinking huge.
 *
 * Code written by Erik Andersen <andersee@debian.org> and
 * swiped from busybox.
 */
int
recursiveAction(const char *fileName, int recurse, int followLinks, int depthFirst,
		int (*fileAction) (const char *fileName, struct stat* statbuf),
		int (*dirAction) (const char *fileName, struct stat* statbuf))
{
    int status;
    struct stat statbuf, statbuf1;
    struct dirent *next;

    if (followLinks == TRUE)
	status = stat(fileName, &statbuf);
    else
	status = lstat(fileName, &statbuf);

    if (status < 0) {
	ERRMSG("Cannot stat %s", fileName);
	return (FALSE);
    }

    if ( (followLinks == FALSE) && (S_ISLNK(statbuf.st_mode)) ) {
	if (fileAction == NULL)
	    return (TRUE);
	else
	    return (fileAction(fileName, &statbuf));
    }

    if (recurse == FALSE) {
	if (S_ISDIR(statbuf.st_mode)) {
	    if (dirAction != NULL)
		return (dirAction(fileName, &statbuf));
	    else
		return (TRUE);
	} 
    }
    
    status = lstat(fileName, &statbuf1);
    if (status < 0) {
	ERRMSG("Cannot stat %s", fileName);
	return (FALSE);
    }

    if (S_ISDIR(statbuf.st_mode) && S_ISDIR(statbuf1.st_mode)) {
	DIR *dir;
	dir = opendir(fileName);
	if (!dir) {
	    ERRMSG("Cannot open %s", fileName);
	    return (FALSE);
	}
	if (dirAction != NULL && depthFirst == FALSE) {
	    status = dirAction(fileName, &statbuf);
	    if (status == FALSE) {
		ERRMSG("DirAction not successful with %s", fileName);
		return (FALSE);
	    }
	}
	while ((next = readdir(dir)) != NULL) {
	    char nextFile[NAME_MAX];
	    if ((!strcmp(next->d_name, ".."))
		|| (!strcmp(next->d_name, "."))
		|| (!strcmp(next->d_name, ".resource")) /* .resource and .finderinfo are HFS cruft */
		|| (!strcmp(next->d_name, ".finderinfo"))) {
		continue;
	    }
	    sprintf(nextFile, "%s/%s", fileName, next->d_name);
	    status =
		recursiveAction(nextFile, TRUE, followLinks, depthFirst, 
			fileAction, dirAction);
	    if (status < 0) {
		closedir(dir);
		return (FALSE);
	    }
	}
	status = closedir(dir);
	if (status < 0) {
	    ERRMSG("Cannot close directory");
	    return (FALSE);
	}
	if (dirAction != NULL && depthFirst == TRUE) {
	    status = dirAction(fileName, &statbuf);
	    if (status == FALSE) {
		ERRMSG("DirAction failed for %s", fileName);
		return (FALSE);
	    }
	}
    } else {
	if (fileAction == NULL)
	    return (TRUE);
	else
	    return (fileAction(fileName, &statbuf));
    }
    return (TRUE);
}


/* Is the target an NFS mountpoint? */
int
is_nfs_partition( const char* target )
{
    struct fdisk_partition* p;
    p = fdisk_find_partition_by_mntpoint(target);
    if ((p) && (p->type == PTYPE_NFS))
	return 1;
    return 0;
}

/* Is the root filesystem on a floppy drive (not cd or initrd) */
int is_root_a_floppy (void) {
    if (   !strcmp(InstallationRootDevice,"/dev/fd0") 
	|| !strcmp(InstallationRootDevice,"/dev/fd1") 
#ifdef SCSI_FLOPPY
	|| !strcmp(InstallationRootDevice,"/dev/sfd0") 
#endif
       ) {
	problemBox(_("You have bootstrapped the Installation Root Disk without using the RAM disk. If this is a low-memory installation, that's OK, but the installation is going to run a whole lot slower this way. If you didn't intend to boot without the RAM disk, it probably happened because you didn't have the root floppy inserted when the system first asked for it at boot time. You might want to reboot with the RAM disk if that's the case."),_("No RAM Disk?"));
    }
    return 0;
}

/* flush out the buffers */
inline int 
fdflush(const char *filename) 
{
	int	fd = open(filename, 0);
	if ( fd < 0 ) {
		return 1;
	}
	ioctl(fd, FDFLUSH, 0);
	close(fd);
	return 0;
}

/* safely add item to a list */
int addtolist(char **list, const char *element){
  char *tmp= (char*)malloc(strlen(*list)+3+strlen(element));
  if (NULL == tmp) {
    fprintf(stderr,_("Memory full!"));
    return (-1);
  }
  tmp[0]='\0';
  strcat(tmp,*list);
  strcat(tmp,", ");
  strcat(tmp,element);
  free(*list);
  *list=tmp;
  return (0);
}

#ifdef USE_LANGUAGE_CHOOSER
/* Helper functions */
size_t
strwidth (const char *what)
{
    size_t res;
    int k;
    const char *p;
    wchar_t c;

    for (res = 0, p = what ; (k = mbtowc (&c, p, MB_LEN_MAX)) > 0 ; p += k)
        res += wcwidth (c);

    return res;
}

void
wpadit (char *dest, const char *source, size_t width)
{
    wchar_t c;
    int k;
    const char *p;
    char *q;
    size_t len;

    for (len = 0, p = source, q = dest ; (k = mbtowc (&c, p, MB_LEN_MAX)) > 0 && len < width ; p += k)
    {
        len += wcwidth (c);

        q += wctomb (q, c);
    }

    if (k == 0)
        for (; len < width ; ++len)
        {
            *q++ = ' '; /* I hope this is ok */
        }

    *q = '\0';
}
#endif

void
frob_lib_modules (void)
{
  struct stat statbuf;
  char buf[PATH_MAX];

  /*
   * make a symlink from /lib/modules to /target/lib/modules
   * so that 'depmod' will function properly
   */

  strncpy(buf, "/lib/modules", sizeof(buf));

  if (! NAME_ISDIR("/target/lib/modules", &statbuf)) {
    INFOMSG("can't make symlink from /lib/modules to /target/lib/modules, the latter does not exist");
    return;
  }
  if (NAME_ISDIR(buf, &statbuf) ) {
    snprintf(prtbuf, sizeof(prtbuf), "%s.old", buf);
    INFOMSG("moving away %s to %s", buf, prtbuf);
    /* deal with existing /lib/modules.old, rather imperiously */
    execlog("rm -rf /lib/modules.old", LOG_DEBUG);/* uh, I should use prtbuf bug I'm lazy */
    rename(buf, prtbuf);
  } else {
    DEBUGMSG("non-directory %s, unlinking", buf)
    unlink(buf);
  }
  INFOMSG("making %s a link to /target/lib/modules", buf);
  symlink("/target/lib/modules", buf);

}

int
max (int a, int b)
{
    return a > b ? a : b;
}

#define GFST_ext2 "ext2"
#define GFST_ext3 "ext3"
#define GFST_reiserfs "reiserfs"
#define GFST_xfs "xfs"

const char *
get_fs_type(const char *name)
{
	FILE *fd;
	char xfsmagic[6];
	char e2fsmagic[4];
	char shfsmagic[11];

	bzero(xfsmagic, sizeof(xfsmagic));
	bzero(e2fsmagic, sizeof(e2fsmagic));
	bzero(shfsmagic, sizeof(shfsmagic));
	fd = fopen(name, "r");
	if (!fd)
		return NULL;

	fread (xfsmagic, 5, 1, fd);
	fseek (fd, 65588, SEEK_SET);
	fread (shfsmagic, 10, 1, fd);
	fseek (fd, 1116, SEEK_SET);
	fread (e2fsmagic, 1, 4, fd);



	if (is_xfs && (!(strcmp(xfsmagic, "XFSB")))) /* XFS */
		return GFST_xfs;
	else if (is_reiserfs && (!(strcmp(shfsmagic, "ReIsErFs") && strcmp(shfsmagic, "ReIsEr2Fs")))) /* reiserfs */
		return GFST_reiserfs;
	else if (is_ext3 && (*e2fsmagic & 0x0004)) /* ext3 */
		return GFST_ext3;

	return GFST_ext2;
}


/*
 * "/proc/sys/dev/cdrom/info", on bittersweet, contains:
 * 8<------------------------------------------------->8
 * CD-ROM information, Id: cdrom.c 2.56 1999/09/09
 *
 * drive name:             sr0     hdd
 * drive speed:            6       4
 * drive # of slots:       1       0
 * Can close tray:         1       1
 * Can open tray:          1       1
 * Can lock tray:          1       1
 * Can change speed:       1       1
 * Can select disk:        0       0
 * Can read multisession:  1       1
 * Can read MCN:           1       1
 * Reports media changed:  1       1
 * Can play audio:         1       1
 *
 *
 * 8<------------------------------------------------->8
 *
 * Note that there are actually tab characters in the file; I've
 * untabified so it stays lined up in this comment.  I have verified
 * that this file does not exist on a machine with no CDROM attached.
 * - karlheg 2000.03.20
 * Linux bittersweet 2.2.14 #9 Fri Mar 17 00:50:19 PST 2000 i586 unknown
 */

int
have_cdrom ()
{
    FILE *cd_info;
    char *line = NULL;
    size_t line_size = 0;
    if ((cd_info = fopen ("/proc/sys/dev/cdrom/info", "r")) != NULL) {
       /* Read the file and build a list of available CD devices. */
       getline (&line, &line_size, cd_info);
       getline (&line, &line_size, cd_info);
       getline (&line, &line_size, cd_info);
       if(strstr(line, "\t\t") != NULL) {
          /* leading tabs exist so the device names must follow in the line */
          fclose(cd_info);
          return 1;
       }
       fclose(cd_info);
    }
    return 0;
}

#ifdef USE_LANGUAGE_CHOOSER
void find_opt_server (char * hostname)
{
   if (NULL != lang && NULL != lang->msgcat && NULL != lang->locale)
   {
      int i=0;
      if(strlen(lang->locale) < 5)
         goto otherway;
      while(mirror_list[i] != NULL) {
         if(strncasecmp((lang->locale)+3,mirror_list[i], 2) == 0) {
            sprintf(hostname, "ftp.%s.debian.org", mirror_list[i]);
            return;
         }
         else
            i++;
      }
otherway:
      /* okay, no country matching, trying the another mapping */
      i=0;
      while(mirror_country_map[i] != NULL) {
         if(strcmp(lang->msgcat,mirror_country_map[i]) == 0) {
            sprintf(hostname, "ftp.%s.debian.org", mirror_country_map[i+1]);
            return;
         }
         else
            i+=2;
      }
   }
}
#endif

char *
concat_paths (const char *path1, const char *path2)
{
    char *path = malloc(strlen(path1) + 1 + strlen(path2) + 1);
    size_t len;

    strcpy(path, path1);

    len = strlen (path1);
    if (len > 0 && path1[len-1] != '/' && path2[0] != '/')
	path[len++] = '/';

    strcpy(path + len, path2);

    return path;
}
