/***************************************************************************
 *            recorder.c
 *
 *  dim jan 22 17:31:49 2006
 *  Copyright  2006  Rouquier Philippe
 *  brasero-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include <nautilus-burn-drive.h>

#include "brasero-marshal.h"
#include "burn-basics.h"
#include "burn-recorder.h"
#include "burn-job.h"

static void brasero_recorder_base_init (gpointer g_class);

GType
brasero_recorder_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BraseroRecorderIFace),
			brasero_recorder_base_init,
			NULL,
			NULL,
		};

		type = g_type_register_static (G_TYPE_INTERFACE, 
					       "BraseroRecorder",
					       &our_info,
					       0);
	}

	return type;
}

static void
brasero_recorder_base_init (gpointer g_class)
{
	static gboolean initialized = FALSE;

        if (initialized)
		return;

	initialized = TRUE;
}

BraseroBurnResult
brasero_recorder_set_drive (BraseroRecorder *recorder,
			    NautilusBurnDrive *drive,
			    GError **error)
{
	BraseroRecorderIFace *iface;

	g_return_val_if_fail (BRASERO_IS_RECORDER (recorder), BRASERO_BURN_ERR);
	g_return_val_if_fail (NAUTILUS_BURN_IS_DRIVE (drive), BRASERO_BURN_ERR);

	if (!nautilus_burn_drive_can_write (drive)) {
		g_set_error (error,
			     BRASERO_BURN_ERROR,
			     BRASERO_BURN_ERROR_GENERAL,
			     _("the drive cannot write CDs or DVDs"));
		return BRASERO_BURN_ERR;
	}

	if (brasero_job_is_running (BRASERO_JOB (recorder)))
		return BRASERO_BURN_RUNNING;

	iface = BRASERO_RECORDER_GET_IFACE (recorder);
	if (!iface->set_drive)
		return BRASERO_BURN_NOT_SUPPORTED;

	brasero_job_debug_message (BRASERO_JOB (recorder),
				   "recorder (%s) set_drive",
				   G_OBJECT_TYPE_NAME (recorder));
	return  (* iface->set_drive) (recorder,
				       drive,
				       error);
}

BraseroBurnResult
brasero_recorder_set_flags (BraseroRecorder *recorder,
			    BraseroRecorderFlag flags,
			    GError **error)
{
	BraseroRecorderIFace *iface;

	g_return_val_if_fail (BRASERO_IS_RECORDER (recorder), BRASERO_BURN_ERR);

	if (brasero_job_is_running (BRASERO_JOB (recorder)))
		return BRASERO_BURN_RUNNING;

	iface = BRASERO_RECORDER_GET_IFACE (recorder);
	if (!iface->set_flags)
		return BRASERO_BURN_NOT_SUPPORTED;

	brasero_job_debug_message (BRASERO_JOB (recorder),
				   "recorder (%s) set_flags",
				   G_OBJECT_TYPE_NAME (recorder));
	return  (*iface->set_flags) (recorder, flags, error);
}

BraseroBurnResult
brasero_recorder_blank (BraseroRecorder *recorder,
			GError **error)
{
	BraseroRecorderIFace *iface;

	g_return_val_if_fail (BRASERO_IS_RECORDER (recorder), BRASERO_BURN_ERR);

	if (brasero_job_is_running (BRASERO_JOB (recorder)))
		return BRASERO_BURN_RUNNING;

	iface = BRASERO_RECORDER_GET_IFACE (recorder);
	if (!iface->blank)
		return BRASERO_BURN_NOT_SUPPORTED;

	brasero_job_debug_message (BRASERO_JOB (recorder),
				   "recorder (%s) blank",
				   G_OBJECT_TYPE_NAME (recorder));
	return  (* iface->blank) (recorder, error);
}

BraseroBurnResult
brasero_recorder_record (BraseroRecorder *recorder,
			 GError **error)
{
	BraseroRecorderIFace *iface;
	BraseroBurnResult result;

	g_return_val_if_fail (BRASERO_IS_RECORDER (recorder), BRASERO_BURN_ERR);

	if (brasero_job_is_running (BRASERO_JOB (recorder)))
		return BRASERO_BURN_RUNNING;

	iface = BRASERO_RECORDER_GET_IFACE (recorder);
	if (!iface->record)
		return BRASERO_BURN_NOT_SUPPORTED;

	brasero_job_debug_message (BRASERO_JOB (recorder),
				   "recorder (%s) record",
				   G_OBJECT_TYPE_NAME (recorder));
	result = (*iface->record) (recorder, error);

	if (result == BRASERO_BURN_OK) {
		brasero_job_progress_changed (BRASERO_JOB (recorder), 
					      1.0,
					      -1);
	}

	return result;
}

BraseroBurnResult
brasero_recorder_get_fifo (BraseroRecorder *recorder, gint *fifo)
{
	BraseroRecorderIFace *iface;

	g_return_val_if_fail (BRASERO_IS_RECORDER (recorder), BRASERO_BURN_ERR);

	if (!brasero_job_is_running (BRASERO_JOB (recorder)))
		return BRASERO_BURN_NOT_RUNNING;

	iface = BRASERO_RECORDER_GET_IFACE (recorder);
	if (!iface->get_fifo)
		return BRASERO_BURN_NOT_SUPPORTED;

	if (!fifo)
		return BRASERO_BURN_OK;

	brasero_job_debug_message (BRASERO_JOB (recorder),
				   "recorder (%s) get_fifo",
				   G_OBJECT_TYPE_NAME (recorder));
	return  (* iface->get_fifo) (recorder, fifo);
}
