/*
buildtorrent -- torrent file creation program
Copyright (C) 2007,2008 Claude Heiland-Allen


This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "config.h"

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <inttypes.h>
#include <dirent.h>
#include <time.h>
/* for correct behaviour ensure this is the POSIX and not the GNU version */
#include <libgen.h>

#include "sha1.h"
#include "md5.h"


/******************************************************************************
program version string
******************************************************************************/
const char* __version__ = "0.7";

/******************************************************************************
torrent data structure declarations
******************************************************************************/

/* structure for torrent file data */
struct _bt_data;
typedef struct _bt_data *bt_data;

/* error flags */
enum _bt_error {
  BT_ERROR_NONE = 0,
  BT_ERROR_NULL,
  BT_ERROR_TYPE,
  BT_ERROR_MEMORY,
  BT_ERROR_IO,
  BT_ERROR_OVERFLOW
};
typedef enum _bt_error bt_error;

/* attempt to support large files */
typedef int64_t integer;

/* constructors */
bt_data bt_string(const unsigned int length, const char *data);
bt_data bt_integer(const integer number);
bt_data bt_list(void);
bt_data bt_dictionary(void);

/* append to a list */
bt_error bt_list_append(bt_data list, const bt_data data);

/* insert into a dictionary, maintaining sorted keys */
bt_error bt_dictionary_insert(
  bt_data dictionary,
  const unsigned int keylength, const char *keyname,
  const bt_data data
);

/* deep-copy a structure */
bt_data bt_copy(const bt_data data);

/* destructor */
void bt_free(bt_data bd);

/* write as a torrent file */
bt_error bt_write(FILE *f, const bt_data bd);

/* types of structures */
enum _bt_type {
  BT_TYPE_STRING = 1000,
  BT_TYPE_INTEGER,
  BT_TYPE_LIST,
  BT_TYPE_DICTIONARY
};
typedef enum _bt_type bt_type;

/* string structure */
struct _bt_string {
  unsigned int length;
  char *data;
};

/* integer structure */
struct _bt_integer {
  integer number;
};

/* list structure */
struct _bt_list {
  struct _bt_data *item;
  struct _bt_list *next;
};

/* dictionary structure */
struct _bt_dictionary {
  struct _bt_string *key;
  struct _bt_data *value;
  struct _bt_dictionary *next;
};

/* general structure */
struct _bt_data {
  bt_type type;
  union {
    struct _bt_string *string;
    struct _bt_integer *integer;
    struct _bt_list *list;
    struct _bt_dictionary *dictionary;
  } b;
};

/******************************************************************************
allocate a new string structure
******************************************************************************/
bt_data bt_string(const unsigned int length, const char *data) {
  bt_data bd = malloc(sizeof(struct _bt_data));
  if (!bd) {
    return NULL;
  }
  {
  struct _bt_string *s = malloc(sizeof(struct _bt_string));
  if (!s) {
    free(bd);
    return NULL;
  }
  s->data = malloc(length);
  if (!s->data) {
    free(s);
    free(bd);
    return NULL;
  }
  s->length = length;
  memcpy(s->data, data, length);
  bd->type = BT_TYPE_STRING;
  bd->b.string = s;
  }
  return bd;
}

/******************************************************************************
allocate a new integer structure
******************************************************************************/
bt_data bt_integer(const integer number) {
  bt_data bd = malloc(sizeof(struct _bt_data));
  if (!bd) {
    return NULL;
  }
  {
  struct _bt_integer *n = malloc(sizeof(struct _bt_integer));
  if (!n) {
    free(bd);
    return NULL;
  }
  n->number = number;
  bd->type = BT_TYPE_INTEGER;
  bd->b.integer = n;
  }
  return bd;
}

/******************************************************************************
allocate a new empty list structure
invariant: bd->b.list != NULL && last(bd->b.list).{item, next} == NULL
******************************************************************************/
bt_data bt_list(void) {
  bt_data bd = malloc(sizeof(struct _bt_data));
  if (!bd) {
    return NULL;
  }
  {
  struct _bt_list *l = malloc(sizeof(struct _bt_list));
  if (!l) {
    free(bd);
    return NULL;
  }
  l->item = NULL;
  l->next = NULL;
  bd->type = BT_TYPE_LIST;
  bd->b.list = l;
  }
  return bd;
}

/******************************************************************************
allocate a new empty dictionary structure
invariant: bd->b.dictionary != NULL &&
           last(bd->b.list).{key, value, next} == NULL &&
	   ordered ascending by key
******************************************************************************/
bt_data bt_dictionary(void) {
  bt_data bd = malloc(sizeof(struct _bt_data));
  if (!bd) {
    return NULL;
  }
  {
  struct _bt_dictionary *d = malloc(sizeof(struct _bt_dictionary));
  if (!d) {
    free(bd);
    return NULL;
  }
  d->key = NULL;
  d->value = NULL;
  d->next = NULL;
  bd->type = BT_TYPE_DICTIONARY;
  bd->b.dictionary = d;
  }
  return bd;
}

/******************************************************************************
append an item to a list
the item is not copied
returns an error flag
******************************************************************************/
bt_error bt_list_append(bt_data list, const bt_data data) {
  if (!list) {
    return BT_ERROR_NULL;
  }
  if (BT_TYPE_LIST != list->type) {
    return BT_ERROR_TYPE;
  }
  if (!list->b.list) {
    return BT_ERROR_NULL;
  }
  {
  struct _bt_list *current;
  struct _bt_list *end = malloc(sizeof(struct _bt_list));
  if (!end) {
    return BT_ERROR_MEMORY;
  }
  end->item = NULL;
  end->next = NULL;
  current = list->b.list;
  while (current->next) {
    current = current->next;
  }
  current->item = data;
  current->next = end;
  }
  return BT_ERROR_NONE;
}

/******************************************************************************
insert an item into a dictionary
the value is not copied, the key is copied
returns an error flag
maintains an ascending ordering of key names
FIXME: assumes key names are null terminated
******************************************************************************/
bt_error bt_dictionary_insert(
  bt_data dictionary,
  const unsigned int keylength, const char *keyname,
  const bt_data data
) {
  if (!dictionary) {
    return BT_ERROR_NULL;
  }
  if (BT_TYPE_DICTIONARY != dictionary->type) {
    return BT_ERROR_TYPE;
  }
  if (!dictionary->b.dictionary) {
    return BT_ERROR_NULL;
  }
  {
  struct _bt_dictionary *current;
  struct _bt_dictionary *lastcurrent = NULL;
  struct _bt_dictionary *insertee;
  struct _bt_string *key = malloc(sizeof(struct _bt_string));
  if (!key) {
    return BT_ERROR_MEMORY;
  }
  key->data = malloc(keylength);
  if (!key->data) {
    free(key);
    return BT_ERROR_MEMORY;
  }
  if (!(insertee = malloc(sizeof(struct _bt_dictionary)))) {
    free(key->data);
    free(key);
    return BT_ERROR_MEMORY;
  }
  memcpy(key->data, keyname, keylength);
  key->length = keylength;
  insertee->key = key;
  insertee->value = data;
  insertee->next = NULL;
  current = dictionary->b.dictionary;
  while (
    current->next && current->key && (0 < strcmp(keyname, current->key->data))
  ) {
    lastcurrent = current;
    current = current->next;
  }
  if (lastcurrent) {
    insertee->next = current;
    lastcurrent->next = insertee;
  } else {
    insertee->next = dictionary->b.dictionary;
    dictionary->b.dictionary = insertee;
  }
  }
  return BT_ERROR_NONE;
}

/******************************************************************************
deep copy a data structure
FIXME: doesn't handle dictionaries yet
******************************************************************************/
bt_data bt_copy(const bt_data data) {
  if (!data) {
    return NULL;
  }
  switch (data->type) {
  case (BT_TYPE_STRING): {
    return bt_string(data->b.string->length, data->b.string->data);
  }
  case (BT_TYPE_INTEGER): {
    return bt_integer(data->b.integer->number);
  }
  case (BT_TYPE_LIST): {
    struct _bt_list *current;
    bt_error err;
    bt_data list = bt_list();
    if (!list) {
      return NULL;
    }
    current = data->b.list;
    while (current && current->item) {
      if ((err = bt_list_append(list, bt_copy(current->item)))) {
        bt_free(list);
        return NULL;
      }
      current = current->next;
    }
    return list;
  }
  default: {
    return NULL;
  }
  }
}

/******************************************************************************
free a string
******************************************************************************/
void bt_free_string(struct _bt_string *s) {
  if (!s) {
    return;
  }
  if (s->data) {
    free(s->data);
  }
  free(s);
}

/******************************************************************************
deep free a data structure
******************************************************************************/
void bt_free(bt_data bd) {
  if (!bd) {
    return;
  }
  switch (bd->type) {
  case (BT_TYPE_STRING): {
    if (bd->b.string) {
      bt_free_string(bd->b.string);
    }
    break;
  }
  case (BT_TYPE_INTEGER): {
    if (bd->b.integer) {
      free(bd->b.integer);
    }
    break;
  }
  case (BT_TYPE_LIST): {
    struct _bt_list *current;
    struct _bt_list *next;
    current = bd->b.list;
    while (current) {
      next = current->next;
      if (current->item) {
        bt_free(current->item);
      }
      free(current);
      current = next;
    }
    break;
  }
  case (BT_TYPE_DICTIONARY): {
    struct _bt_dictionary *current;
    struct _bt_dictionary *next;
    current = bd->b.dictionary;
    while (current) {
      next = current->next;
      if (current->key) {
        bt_free_string(current->key);
      }
      if (current->value) {
        bt_free(current->value);
      }
      free(current);
      current = next;
    }
    break;
  }
  default:
    break;
  }
  free(bd);
}

/******************************************************************************
write a string in torrent encoding
******************************************************************************/
bt_error bt_write_string(FILE *f, const struct _bt_string *s) {
  if (!s) {
    return BT_ERROR_NULL;
  }
  if (!s->data) {
    return BT_ERROR_NULL;
  }
  if (0 > fprintf(f, "%d:", s->length)) {
    return BT_ERROR_IO;
  }
  if (1 != fwrite(s->data, s->length, 1, f)) {
    return BT_ERROR_IO;
  }
  return BT_ERROR_NONE;
}

/******************************************************************************
write a data structure in torrent encoding
******************************************************************************/
bt_error bt_write(FILE *f, const bt_data bd) {
  if (!bd) {
    return BT_ERROR_NULL;
  }
  switch (bd->type) {
  case (BT_TYPE_STRING): {
    return bt_write_string(f, bd->b.string);
    break;
  }
  case (BT_TYPE_INTEGER): {
    if (!bd->b.integer) {
      return BT_ERROR_NULL;
    }
    if (0 > fprintf(f, "i%" PRId64 "e", bd->b.integer->number)) {
      return BT_ERROR_IO;
    }
    break;
  }
  case (BT_TYPE_LIST): {
    struct _bt_list *current;
    bt_error err;
    current = bd->b.list;
    if (0 > fprintf(f, "l")) {
      return BT_ERROR_IO;
    }
    while (current->next) {
      if (current->item) {
        if ((err = bt_write(f, current->item))) {
          return err;
        }
      }
      current = current->next;
    }
    if (0 > fprintf(f, "e")) {
      return BT_ERROR_IO;
    }
    break;
  }
  case (BT_TYPE_DICTIONARY): {
    struct _bt_dictionary *current;
    bt_error err;
    current = bd->b.dictionary;
    if (0 > fprintf(f, "d")) {
      return BT_ERROR_IO;
    }
    while (current->next) {
      if (current->key) {
        if ((err = bt_write_string(f, current->key))) {
          return err;
        }
        if (!current->value) {
          return BT_ERROR_NULL;
        }
        if ((err = bt_write(f, current->value))) {
          return err;
        }
      }
      current = current->next;
    }
    if (0 > fprintf(f, "e")) {
      return BT_ERROR_IO;
    }
    break;
  }
  default: {
    return BT_ERROR_TYPE;
    break;
  }
  }
  return BT_ERROR_NONE;
}

/******************************************************************************
pretty print torrent data
******************************************************************************/
void bt_show(bt_data bd, int piecestoo, int indent, int indentstep, int comma) {
  int i;
  if (!bd) {
    for(i = 0; i < indent; i++) {
      printf(" ");
    }
    printf("NULL%s\n", comma ? "," : "");
    return;
  }
  switch (bd->type) {
  case (BT_TYPE_STRING): {
    char strbuf[512];
    int len = bd->b.string->length > 500 ? 500 : bd->b.string->length;
    memcpy(strbuf, bd->b.string->data, len);
    strbuf[len] = '\0';
    for(i = 0; i < indent; i++) {
      printf(" ");
    }
    printf(
      "\"%s\"%s%s\n", strbuf, comma ? "," : "", len == 500 ? " // truncated!" : ""
    );
    break;
  }
  case (BT_TYPE_INTEGER): {
    for(i = 0; i < indent; i++) {
      printf(" ");
    }
    printf("%" PRId64 "%s\n", bd->b.integer->number, comma ? "," : "");
    break;
  }
  case (BT_TYPE_LIST): {
    struct _bt_list *current;
    for(i = 0; i < indent; i++) {
      printf(" ");
    }
    printf("[\n");
    current = bd->b.list;
    while (current->next) {
      bt_show(
        current->item, piecestoo,
        indent + indentstep, indentstep,
        current->next->next != NULL
      );
      current = current->next;
    }
    for(i = 0; i < indent; i++) {
      printf(" ");
    }
    printf("]%s\n", comma ? "," : "");
    break;
  }
  case (BT_TYPE_DICTIONARY): {
    struct _bt_dictionary *current;
    char strbuf[512];
    int len;
    for(i = 0; i < indent; i++) {
      printf(" ");
    }
    printf("{\n");
    current = bd->b.dictionary;
    while (current->next) {
      len = current->key->length > 500 ? 500 : current->key->length;
      memcpy(strbuf, current->key->data, len);
      strbuf[len] = '\0';
      for(i = 0; i < indent + indentstep; i++) {
        printf(" ");
      }
      printf("\"%s\" =>%s\n", strbuf, len == 500 ? " // truncated!" : "");
      if (strcmp("pieces", strbuf) == 0) {
        if (piecestoo) {
          printf("----------------------------------------");
          char *hexdigits = "0123456789abcdef";
          for (i = 0; i < current->value->b.string->length; i++) {
            if ((i % 20) == 0) {
              printf("\n");
            }
            printf("%c%c",
              hexdigits[(current->value->b.string->data[i] & 0xF0) >> 4],
              hexdigits[(current->value->b.string->data[i] & 0x0F)]
            );
          }
          printf("\n----------------------------------------\n");
        } else {
          for(i = 0; i < indent + indentstep + indentstep; i++) {
            printf(" ");
          }
          printf("\"...\"%s // pieces not shown\n", current->next->next ? "," : "");
        }
      } else {
        bt_show(
          current->value, piecestoo, indent + indentstep + indentstep,
          indentstep, current->next->next != NULL
        );
      }
      current = current->next;
    }
    for(i = 0; i < indent; i++) {
      printf(" ");
    }
    printf("}%s\n", comma ? "," : "");
    break;
  }
  }
}


/******************************************************************************
hash queue data structure
the first in the list is a dummy
******************************************************************************/
struct _bt_hash_queue {
  integer size;
  char *file;
  struct _bt_hash_queue *next;
};
typedef struct _bt_hash_queue *bt_hash_queue;

/******************************************************************************
create the dummy first list element
******************************************************************************/
bt_hash_queue bt_create_hash_queue() {
  bt_hash_queue queue = malloc(sizeof(struct _bt_hash_queue));
  if (!queue) {
    return NULL;
  }
  queue->size = 0;
  queue->file = NULL;
  queue->next = NULL;
  return queue;
}

/******************************************************************************
append to the hash queue
copies the arguments
******************************************************************************/
bt_error bt_hash_enqueue(
  bt_hash_queue queue, const char *file, integer size, int verbose
) {
  if (!queue) {
    return BT_ERROR_NULL;
  }
  if (!file) {
    return BT_ERROR_NULL;
  }
  {
  bt_hash_queue current;
  bt_hash_queue node = malloc(sizeof(struct _bt_hash_queue));
  if (!node) {
    return BT_ERROR_MEMORY;
  }
  if (!(node->file = malloc(strlen(file) + 1))) {
    free(node);
    return BT_ERROR_MEMORY;
  }
  memcpy(node->file, file, strlen(file) + 1);
  node->size = size;
  node->next = NULL;
  current = queue;
  while (current->next) {
    current = current->next;
  }
  current->next = node;
  if (verbose) {
    fprintf(stderr, "%24" PRId64 " : %s\n", size, file);
  }
  return BT_ERROR_NONE;
  }
}

/******************************************************************************
append a file to a path string
path must be at least length maxlength bytes
returns an error flag
******************************************************************************/
bt_error bt_joinpath(size_t maxlength, char *path, const char *file) {
  if (strlen(path) + strlen(file) + 1 + 1 > maxlength) {
    return BT_ERROR_OVERFLOW;
  } else {
    size_t oldlength = strlen(path);
    if (oldlength > 0) {
      path[oldlength] = '/';
      memcpy(path + oldlength + 1, file, strlen(file) + 1);
    } else {
      memcpy(path, file, strlen(file) + 1);
    }
    return BT_ERROR_NONE;
  }
}


/******************************************************************************
md5sum a file
******************************************************************************/

bt_data bt_md5sum_file(const char *filename, integer length) {
  struct MD5Context context;
  char *hexdigits = "0123456789abcdef";
  unsigned char *buffer = NULL;
  unsigned char digest[16];
  char hexdump[33];
  integer buflen = 262144;
  integer size = 0;
  integer left = length;
  integer i;
  FILE *file = NULL;
  if (!filename) {
    return NULL;
  }
  if (!(buffer = malloc(buflen))) {
    return NULL;
  }
  if (!(file = fopen(filename, "rb"))) {
    free(buffer);
    return NULL;
  }
  MD5Init(&context);
  while (left > 0) {
    if (left > buflen) {
       size = buflen;
    } else {
       size = left;
    }
    if (1 != fread(buffer, size, 1, file)) {
      free(buffer);
      fclose(file);
      return NULL;
    }
    MD5Update(&context, buffer, size);
    left -= size;
  }
  MD5Final(digest, &context);
  for (i = 0; i < 16; ++i) {
    hexdump[i+i+0] = hexdigits[(digest[i] & 0xF0) >> 4];
    hexdump[i+i+1] = hexdigits[(digest[i] & 0x0F)];
  }
  free(buffer);
  fclose(file);
  return bt_string(32, hexdump);
}



/******************************************************************************
find files recursively
directory is the initial directory
path is a buffer of maxlength bytes to store the accumulated path
pathlist is the accumulated path exploded as a list
files is a list to add the found files to
returns an error flag
******************************************************************************/
bt_error bt_findfiles(
  DIR *directory, size_t maxlength, char *path, bt_data pathlist,
  bt_data files, bt_hash_queue queue, int domd5sum, int verbose
) {
  struct dirent *entry;
  bt_data file = NULL;
  bt_data filename = NULL;
  bt_data filesize = NULL;
  bt_data filepath = NULL;
  bt_data md5sum = NULL;
  bt_error err = BT_ERROR_NONE;
  while ((entry = readdir(directory))) {
    if (strcmp(entry->d_name, ".") && strcmp(entry->d_name, "..")) {
      struct stat s;
      size_t oldlength = strlen(path);
      if ((err = bt_joinpath(maxlength, path, entry->d_name))) {
        return err;
      }
      if (!stat(path, &s)) {
        if (S_ISREG(s.st_mode)) {
          if ((file = bt_dictionary())) {
            if ((filesize = bt_integer(s.st_size))) {
              if ((
                filename = bt_string(strlen(entry->d_name), entry->d_name)
              )) {
                if ((filepath = bt_copy(pathlist))) {
                  if (!(err = bt_list_append(filepath, filename))) {
                    if (!(err = bt_dictionary_insert(
                      file, strlen("length"), "length", filesize
                    ))) {
    if (domd5sum) {
      if ((md5sum = bt_md5sum_file(path, s.st_size))) {
        err = bt_dictionary_insert(file, strlen("md5sum"), "md5sum", md5sum);
      } else {
        err = BT_ERROR_IO;
      }
    }
    if (!err) {
    
                      if (!(err = bt_dictionary_insert(
                        file, strlen("path"), "path", filepath
                      ))) {
                        if (!(err = bt_list_append(files, file))) {
                          err = bt_hash_enqueue(queue, path, s.st_size, verbose);
                        }
                      }
                    }
    }
                  }
                } else {
                  err = BT_ERROR_MEMORY;
                }
              } else {
                err = BT_ERROR_MEMORY;
              }
            } else {
              err = BT_ERROR_MEMORY;
            }
          } else {
            err = BT_ERROR_MEMORY;
          }
        } else if (S_ISDIR(s.st_mode)) {
          DIR* dir;
          if ((dir = opendir(path))) {
            if ((
              filename = bt_string(strlen(entry->d_name), entry->d_name)
            )) {
              if ((filepath = bt_copy(pathlist))) {
                if (!(err = bt_list_append(filepath, filename))) {
                  err = bt_findfiles(
                    dir, maxlength, path, filepath, files, queue, domd5sum, verbose
                  );
                  closedir(dir);
                }
              } else {
                err = BT_ERROR_MEMORY;
              }
            } else {
              err = BT_ERROR_MEMORY;
            }
          } else {
            err = BT_ERROR_IO;
          }
        }
      } else {
        err = BT_ERROR_IO;
      }
      path[oldlength] = '\0';
    }
  }
  return err;
}


/******************************************************************************
draw progressbar
******************************************************************************/

void bt_progressbar(integer piece, integer count) {
  integer oldblocks = ((piece - 1) * 50) / count;
  integer blocks = (piece * 50) / count;
  integer i;
  char s[53];
  if (blocks != oldblocks) {
    s[0] = '[';
    for (i = 0; i < 50; i++) {
      s[i+1] = (i < blocks) ? '=' : ((i == blocks) ? '>' : ' ');
    }
    s[51] = ']';
    s[52] = '\0';
    fprintf(
      stderr,
      "\b\b\b\b\b\b\b\b\b\b"
      "\b\b\b\b\b\b\b\b\b\b"
      "\b\b\b\b\b\b\b\b\b\b"
      "\b\b\b\b\b\b\b\b\b\b"
      "\b\b\b\b\b\b\b\b\b\b"
      "\b\b%s",
      s
    );
  }
}


/******************************************************************************
hash files
******************************************************************************/

bt_data bt_hash_pieces(bt_hash_queue queue, integer size, int verbose) {
  bt_hash_queue node = NULL;
  char *hashdata = NULL;
  integer total = 0;
  unsigned char *buffer = NULL;
  unsigned char *bufptr = NULL;
  integer remain = size;
  integer left;
  FILE *file = NULL;
  integer piececount;
  integer i;
  sha1_byte digest[SHA1_DIGEST_LENGTH];
  if (!queue) {
    return NULL;
  }
  if (!(buffer = malloc(size))) {
    return NULL;
  }
  node = queue;
  while (node->next) {
    node = node->next;
    total += node->size;
  }
  piececount = (total + size - 1) / size; /* ceil(total/size) */
  if (piececount <= 0) { /* FIXME: no idea what to do if there's no data */
    free(buffer);
    fprintf(stderr, "torrent has no data, aborting!\n");
    return NULL;
  }
  if (verbose) {
    fprintf(stderr, "hashing %" PRId64 " pieces\n", piececount);
    fprintf(stderr, "["
                    "          "
                    "          "
                    "          "
                    "          "
                    "          "
                    "]");
  }
  if (!(hashdata = malloc(piececount * SHA1_DIGEST_LENGTH))) {
    free(buffer);
    return NULL;
  }
  node = queue->next;
  file = fopen(node->file, "rb");
  if (!file) {
    free(buffer);
    return NULL;
  }
  left = node->size;
  bufptr = buffer;
  for (i = 0; i < piececount; ++i) {
    do {
      if (left <= remain) {
        /* take all */
        if (left != 0) { /* don't fail on empty files */
          if (1 != fread(bufptr, left, 1, file)) {
            fclose(file);
            free(buffer);
            return NULL;
          }
          bufptr += left;
          remain -= left;
          fclose(file);
          file = NULL;
        }
        node = node->next;
        if (node) {
          file = fopen(node->file, "rb");
          if (!file) {
            free(buffer);
            return NULL;
          }
          left = node->size;
        }
      } else { /* left > remain */
        /* take as much as we can */
        if (remain != 0) { /* don't fail on empty files */
          if (1 != fread(bufptr, remain, 1, file)) {
            free(buffer);
            return NULL;
          }
          bufptr += remain;
          left -= remain;
          remain = 0;
        }
      }
    } while (remain != 0 && node);
    if (!node && i != piececount - 1) {
      /* somehow the pieces don't add up */
      if (file) {
        fclose(file);
      }
      free(buffer);
      return NULL;
    }
    /* remain == 0 || i == piececount - 1 */
    if (verbose) {
      bt_progressbar(i, piececount);
    }
    SHA1(buffer, size - remain, digest);
    memcpy(hashdata + i * SHA1_DIGEST_LENGTH, digest, SHA1_DIGEST_LENGTH);
    bufptr = buffer;
    remain = size;
  }
  if (verbose) {
    bt_progressbar(piececount, piececount);
    fprintf(stderr, "\n");
  }
  return bt_string(SHA1_DIGEST_LENGTH * piececount, hashdata);
}

/******************************************************************************
parse bencoded data, eg a torrent file
******************************************************************************/
/*
bt_data bt_parse_torrent(FILE *in) {
  
}
*/

/******************************************************************************
parse an announce list
format = "url|url|url,url|url|url,url|url|url"
******************************************************************************/

bt_data bt_parse_announcelist(const char *urls) {
  bt_data announcelist;
  bt_data tier;
  const char *s;
  const char *t;
  const char *t1;
  const char *t2;
  if (!urls) {
    return NULL;
  }
  if (strcmp("", urls) == 0) {
    return NULL;
  }
  announcelist = bt_list();
  if (!announcelist) {
    return NULL;
  }
  s = urls;
  tier = bt_list();
  do {
    t = NULL;
    t1 = strchr(s, '|');
    t2 = strchr(s, ',');
    if (!(t1 || t2)) {
      t = s + strlen(s);
    } else {
      t = (t1 - t2) > 0 ? t2 : t1;
    }
    if (t - s == 0) {
      return NULL;
    }
    if (bt_list_append(tier, bt_string(t - s, s))) {
      return NULL;
    }
    if (t[0] == ',' || t[0] == '\0') {
      if (bt_list_append(announcelist, tier)) {
        return NULL;
      };
      if (t[0] != '\0') {
        tier = bt_list();
      } else {
        tier = NULL;
      }
    }
    s = t + 1;
  } while (t[0] != '\0');
  return announcelist;
}

/******************************************************************************
show usage message
******************************************************************************/
void bt_usage(void) {
  printf(
    "Usage: buildtorrent [OPTIONS] -a announceurl input output\n"
    "\n"
    "options:\n"
    "--announce      -a  <announce>   : announce url (required)\n"
    "--announcelist  -A  <announces>  : announce url list\n"
    "--piecelength   -l  <length>     : piece length in bytes, default 262144\n"
    "--comment       -c  <comment>    : user comment, omitted by default\n"
    "--private       -p  <private>    : private flag, either 0 or 1\n"
    "--nodate        -D               : omit 'creation date' field\n"
    "--nocreator     -C               : omit 'created by' field\n"
    "--md5sum        -m               : add an 'md5sum' field for each file\n"
    "--show          -s               : show generated torrent structure\n"
    "--showall       -S               : show pieces too (implies '-s')\n"
    "--quiet         -q               : quiet operation\n"
    "--version       -V               : show version of buildtorrent\n"
    "--help          -h               : show this help screen\n"
  );
}

/******************************************************************************
main program
******************************************************************************/
int main(int argc, char **argv) {

  char *url = NULL;
  char *urls = NULL;
  char *inname = NULL;
  char *namebase = NULL;
  char *outfile = NULL;
  char *commentstr = NULL;
  unsigned int plen = 262144;
  int verbose = 1;
  int nodate = 0;
  int nocreator = 0;
  int privated = 0;
  int privateopt = 0;
  int domd5sum = 0;
  int show = 0;
  int slen;
  int i;

  DIR *dir;
  FILE *output;
  bt_data torrent;
  bt_data announce;
  bt_data announcelist = NULL;
  bt_data info;
  bt_data piecelength;
  bt_data pieces;
  bt_data files;
  bt_data name;
  bt_data length;
  bt_data pathlist;
  bt_data creator;
  bt_data creationdate;
  bt_data comment;
  bt_data private;
  bt_data md5sum;
  bt_hash_queue queue;

  struct stat s;
  char path[8192];
  char nametemp[8192];

  while (1) {
    int optidx = 0;
    static struct option options[] = {
      { "announce", 1, 0, 'a' },
      { "announcelist", 1, 0, 'A' },
      { "piecelength", 1, 0, 'l' },
      { "comment", 1, 0, 'c' },
      { "private", 1, 0, 'p' },
      { "nodate", 0, 0, 'D' },
      { "nocreator", 0, 0, 'C' },
      { "md5sum", 0, 0, 'm' },
      { "show", 0, 0, 's' },
      { "showpieces", 0, 0, 'S' },
      { "quiet", 0, 0, 'q' },
      { "version", 0, 0, 'V' },
      { "help", 0, 0, 'h' },
      { 0, 0, 0, 0 }
    };
    char c = getopt_long(argc, argv, "hVqSsmCDa:A:l:c:p:", options, &optidx );
    if (c == -1) {
      break;
    }
    switch (c) {
    case ('?'):
      return 1;
    case ('a'):
      url = optarg;
      break;
    case ('A'):
      urls = optarg;
      break;
    case ('l'):
      plen = atoi(optarg);
      break;
    case ('c'):
      commentstr = optarg;
      break;
    case ('p'):
      privated = 1;
      privateopt = (strcmp(optarg, "0") == 0) ? 0 : 1;
      break;
    case ('D'):
      nodate = 1;
      break;
    case ('C'):
      nocreator = 1;
      break;
    case ('m'):
      domd5sum = 1;
      break;
    case ('s'):
      show = 1;
      break;
    case ('S'):
      show = 2;
      break;
    case ('q'):
      verbose = 0;
      break;
    case ('V'):
      printf(
        "buildtorrent %s (GPL) 2007-8 "
        "Claude Heiland-Allen <claudiusmaximus@goto10.org>\n",
        __version__
      );
      return 0;
    case ('h'):
      bt_usage();
      return 0;
    }
  }
  if (!url) {
    fprintf(stderr, "buildtorrent: announce url required\n");
    return 1;
  }
  if (plen <= 0) { /* avoid division by zero */
    fprintf(stderr, "buildtorrent: piece length must be greater than 0\n");
    return 1;
  }
  if (optind + 2 < argc) {
    fprintf(stderr, "buildtorrent: too many arguments\n");
    return 1;
  }
  if (optind + 2 > argc) {
    fprintf(stderr, "buildtorrent: too few arguments\n");
    return 1;
  }

  inname  = argv[optind];
  outfile = argv[optind + 1];

  /* handle paths correctly (note: requires POSIX basename(), not GNU) */
  strncpy(nametemp, inname, 8191);
  nametemp[8191] = '\0';
  namebase = basename(nametemp);
  slen = strlen(namebase);
  for (i = 0; i < slen; ++i) {
    if (namebase[i] == '/') {
      fprintf(
        stderr,
        "buildtorrent: BUG! input (\"%s\") munged (\"%s\") contains '/'.\n",
        inname,
        namebase
      );
      return 1;
    }
  }

  if (stat(inname, &s)) {
    fprintf(stderr, "buildtorrent: could not stat \"%s\"\n", inname);
    return 1;
  }

  if (!(torrent = bt_dictionary())) {
    fprintf(stderr, "buildtorrent: couldn't allocate torrent dictionary\n");
    return 1;
  }
  if (!(info = bt_dictionary())) {
    fprintf(stderr, "buildtorrent: couldn't allocate info dictionary\n");
    return 1;
  }
  if (!(announce = bt_string(strlen(url), url))) {
    fprintf(stderr, "buildtorrent: couldn't allocate announce string\n");
    return 1;
  }
  if (!(piecelength = bt_integer(plen))) {
    fprintf(stderr, "buildtorrent: couldn't allocate piece length integer\n");
    return 1;
  }
  if (!(name = bt_string(strlen(namebase), namebase))) {
    fprintf(stderr, "buildtorrent: couldn't allocate name string\n");
    return 1;
  }
  if (bt_dictionary_insert(info, strlen("name"), "name", name)) {
    fprintf(stderr, "buildtorrent: couldn't insert name into info\n");
    return 1;
  }
  if (bt_dictionary_insert(
    info, strlen("piece length"), "piece length", piecelength
  )) {
    fprintf(stderr, "buildtorrent: couldn't insert piece length into info\n");
    return 1;
  }
  if (urls) {
    if (!(announcelist = bt_parse_announcelist(urls))) {
      fprintf(stderr, "buildtorrent: error parsing announce-list argument\n");
      return 1;
    }
  }

  if (!(queue = bt_create_hash_queue())) {
    fprintf(stderr, "buildtorrent: couldn't allocate hash queue\n");
    return 1;
  }

  if (S_ISDIR(s.st_mode)) {

    if (!(dir = opendir(inname))) {
      fprintf(stderr, "buildtorrent: couldn't open directory\n");
      return 1;
    }
    if (!(files = bt_list())) {
      fprintf(stderr, "buildtorrent: couldn't allocate files list\n");
      return 1;
    }
    if (!(pathlist = bt_list())) {
      fprintf(stderr, "buildtorrent: couldn't path list\n");
      return 1;
    }
    if (strlen(inname) > 8190) {
      fprintf(stderr, "buildtorrent: 'input' argument too long\n");
      return 1;
    }
    memcpy(path, inname, strlen(inname) + 1);
    if (bt_findfiles(
      dir, 8192, path, pathlist, files, queue, domd5sum, verbose
    )) {
      fprintf(stderr, "buildtorrent: error finding files\n");
      return 1;
    }
    closedir(dir);
    if (bt_dictionary_insert(info, strlen("files"), "files", files)) {
      fprintf(stderr, "buildtorrent: couldn't insert files into info\n");
      return 1;
    }

  } else if (S_ISREG(s.st_mode)) {

    if (!(length = bt_integer(s.st_size))) {
      fprintf(stderr, "buildtorrent: couldn't allocate length integer\n");
      return 1;
    }
    if (bt_dictionary_insert(info, strlen("length"), "length", length)) {
      fprintf(stderr, "buildtorrent: couldn't insert length into info\n");
      return 1;
    }
    if (domd5sum) {
      if (!(md5sum = bt_md5sum_file(inname, s.st_size))) {
        fprintf(stderr, "buildtorrent: couldn't md5sum \"%s\"\n", inname);
        return 1;
      }
      if (bt_dictionary_insert(info, strlen("md5sum"), "md5sum", md5sum)) {
        fprintf(stderr, "buildtorrent: couldn't insert md5sum into info\n");
        return 1;
      }
    }
    if ((bt_hash_enqueue(queue, inname, s.st_size, verbose))) {
      fprintf(stderr, "buildtorrent: error enqueueing file for hashing\n");
      return 1;
    }

  } else {
    fprintf(
      stderr, "buildtorrent: \"%s\" is neither file nor directory\n", inname
    );
    return 1;
  }

  if (!(pieces = bt_hash_pieces(queue, plen, verbose))) {
    fprintf(stderr, "buildtorrent: error hashing files\n");
    return 1;
  }
  if (bt_dictionary_insert(info, strlen("pieces"), "pieces", pieces)) {
    fprintf(stderr, "buildtorrent: couldn't insert pieces into info\n");
    return 1;
  }
  if (privated) {
    if (!(private = bt_integer(privateopt))) {
      fprintf(stderr, "buildtorrent: couldn't allocate private integer\n");
      return 1;
    }
    if (bt_dictionary_insert(info, strlen("private"), "private", private)) {
      fprintf(stderr, "buildtorrent: couldn't insert private into info\n");
      return 1;
    }
  }  
  if (bt_dictionary_insert(torrent, strlen("announce"), "announce", announce)) {
    fprintf(stderr, "buildtorrent: couldn't insert announce into torrent\n");
    return 1;
  }
  if (urls) {
    if (bt_dictionary_insert(torrent, strlen("announce-list"), "announce-list", announcelist)) {
      fprintf(stderr, "buildtorrent: couldn't insert announce-list into torrent\n");
      return 1;
    }
  }
  if (bt_dictionary_insert(torrent, strlen("info"), "info", info)) {
    fprintf(stderr, "buildtorrent: couldn't insert info into torrent\n");
    return 1;
  }
  if (!nodate) {
    if (!(creationdate = bt_integer(time(NULL)))) {
      fprintf(stderr, "buildtorrent: couldn't allocate creation date integer\n");
      return 1;
    }
    if (bt_dictionary_insert(
      torrent, strlen("creation date"), "creation date", creationdate
    )) {
      fprintf(stderr, "buildtorrent: couldn't insert creation date into torrent\n");
      return 1;
    }
  }
  if (!nocreator) {
    if (!(creator = bt_string(strlen("buildtorrent/0.7"), "buildtorrent/0.7"))) {
      fprintf(stderr, "buildtorrent: couldn't allocate created by string\n");
      return 1;
    }
    if (bt_dictionary_insert(
      torrent, strlen("created by"), "created by", creator
    )) {
      fprintf(stderr, "buildtorrent: couldn't insert created by into torrent\n");
      return 1;
    }
  }
  if (commentstr) {
    if (!(comment = bt_string(strlen(commentstr), commentstr))) {
      fprintf(stderr, "buildtorrent: couldn't allocate comment string\n");
      return 1;
    }
    if (bt_dictionary_insert(
      torrent, strlen("comment"), "comment", comment
    )) {
      fprintf(stderr, "buildtorrent: couldn't insert comment into torrent\n");
      return 1;
    }
  }
  if (!(output = fopen(outfile, "wb"))) {
    fprintf(stderr, "buildtorrent: couldn't open \"%s\" for writing\n", outfile);
    return 1;
  }
  if (bt_write(output, torrent)) {
    fprintf(stderr, "buildtorrent: error writing \"%s\"\n", outfile);
    return 1;
  }
  if (show) {
    printf("torrent =>\n");
    bt_show(torrent, show == 2, 2, 2, 0);
    printf("}\n");
  }
  bt_free(torrent);
  fclose(output);
  return 0;
}

/* EOF */
