/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2001 The Caudium Group
 * Based on IMHO  Stefan Wallstrm and Bosse Lincoln.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas.pike,v 1.113.2.25.2.29 2001/11/26 17:00:35 kiwi Exp $
 */

#define IMHO this_object ()

#include <module.h>
inherit "module";
inherit "caudiumlib";

#include <camas/globals.h>                  // Global definitions
#include <camas/functions.h>                // Some useful functions
#include <camas/smtp.h>                     // SMTP client

// This is the real Camas version. It should changed before each
// release by make bump_version
constant __camas_version__ = "1.0";
constant __camas_build__ = "30";
constant __camas_state__ = "REL";

constant camas_version = __camas_version__+"."+__camas_build__+"-"+__camas_state__;

//
//! module: CAMAS: Main module
//!  Caudium Mail Access System, the official Webmail for Caudium Webserver
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_LOCATION|MODULE_PROVIDER
//! cvs_version: $Id: camas.pike,v 1.113.2.25.2.29 2001/11/26 17:00:35 kiwi Exp $
//

constant cvs_version   = "$Id: camas.pike,v 1.113.2.25.2.29 2001/11/26 17:00:35 kiwi Exp $";
constant module_type   = MODULE_LOCATION|MODULE_PROVIDER;
constant module_name   = "CAMAS: Main module ("+camas_version+")";
constant module_doc    = "CAudium Mail Access System, the official Webmail "
                         "for Caudium Webserver <br />"
			 "<i>CAMAS Version : </i>"+camas_version+"<br />"
			 "Please see Status for more informations...";
constant thread_safe   = 1;
constant module_unique = 1;


#define MSG(m) sessobj->lang_module->msg (sessobj, m)
#define MSGA(m, a) sessobj->lang_module->msg (sessobj, m, a)

string global_addressbook = "";
int plugin_loaded = 0;
object plugin = 0;
int global_kludges = 0, global_ok = 0; //oliv3: until we are sure the XML Parser has no problems anymore
mixed gc_call_out = 0;

// These properties are saved and loaded for each user. To add a new, change
// here and in the SETUP menu.
mapping prefproperties=([ 
  "name" : "", 
  "address" : "",
  "mailpath" : "", 
  "replymsgprefix" : "> ",
  "extraheader" : "",
  "signature" : "",
  "autologout" : "20",
  "visiblemail" : "20",
  "sortorder" : "backward",
  "sortcolumn" : "num",
  "trashfolder" : "Trash",
  "sentfolder" : "sent-mail",
  "draftsfolder" : "Drafts",
  "autobcc" : "",
  "filterbook" : "",
  "language" : "",
  "addressbook" : "",
  "organization" : "",

  // oliv3: these properties _MUST_ be queried with
  // QUERY (x)=="1" or QUERY (x)=="0",
  // *NOT* with QUERY (x) and !QUERY (x)
  "replyincludemsg" : "1",
  "saveattachments" : "0", 
  "showhiddenheaders": "1",
  "showhtml": "0",
  "showtext": "1"
]);

mapping (string:array) prefprop_att = ([
  /* These are the attribute used to generate the customization interface
   * in Caudium. This mapping should have the same keys as prefproperties above
   * and those the values should be a 3 (at least) element array :
   * ({ title, type, help [, stringlist] })
   * where title is a string, indexing a message in the configuration menus,
   * type is the type of widget used (an int : TYPE_INT, TYPE_STRING...) and
   * help is the html text displayed as an help message in those menus.
   * stringlist is a list of choices (for TYPE_STRING_LIST).
   */
  "name"             : ({ "Name", TYPE_STRING, "no help available" }),
  "address"          : ({ "Mail address", TYPE_STRING, "The address handled by this imho." }),
  "mailpath"         : ({ "Mail searchpath", TYPE_STRING, "no help available" }),
  "replymsgprefix"   : ({ "Prefix", TYPE_STRING,
			  "The bit of text prepended to each line of an "
			  "included message text into a reply." }),
  "replyincludemsg"  : ({ "Include mail when replying", TYPE_FLAG, 
			  "Toggles the auto inclusion of a message text into a reply." }),
  "extraheader"      : ({ "Extra headers", TYPE_STRING, "no help available" }),
  "signature"        : ({ "Signature", TYPE_TEXT_FIELD, "no help available" }),
  "autologout"       : ({ "Inactive logout (minutes)", TYPE_STRING, "no help available" }),
  "visiblemail"      : ({ "Messages shown in mailbox", TYPE_STRING, "no help available" }),
  "sortorder"        : ({ "Mail sort order", TYPE_STRING_LIST,
			  "no help available", ({ "forward", "backward" }) }),
  "trashfolder"      : ({ "Trash folder", TYPE_STRING, "no help available" }),
  "sentfolder"       : ({ "Sent mail folder", TYPE_STRING, "no help available" }),
  "draftsfolder"     : ({ "Drafts folder", TYPE_STRING, "no help available" }),
  "autobcc"          : ({ "Default Bcc address", TYPE_STRING, "no help available" }),
  "saveattachments"  : ({ "Include attachments", TYPE_FLAG, "no help available" }),
  "language"         : ({ "Language", TYPE_STRING, "no help available" }),
  "layout"           : ({ "User interface", TYPE_STRING, "no help available" }),
  "showhtml"  : ({ "Show HTML messages", TYPE_FLAG, "Show HTML messages." }),
  "showtext"  : ({ "Show text messages", TYPE_FLAG, "Show text messages." }),
  "organization" : ({ "Organization", TYPE_STRING, "Organization." })
]);

// Setup screen
constant SETUP_FIELDS = ({ "$USERNAME", "$USERADDRESS", "$MAILPATH", "$LANGUAGE", "$LAYOUT",
			   "$REPLY", "$PREFIX", "$SIGNATURE", "$HEADERS", "$LOGOUT", "$VISIBLE",
			   "$SORTORDER", "$SORTCOLUMN", "$SHOWHTML", "$SHOWTEXT",
			   "$TRASH", "$DRAFTS", "$SENT", "$ATTACHMENTS", "$BCCCOPY",
			   "$SHOWHEADERS", "$ORGANIZATION" });
constant SETUP_MFIELDS = ({ "$MUSERNAME", "$MUSERADDRESS", "$MMAILPATH", "$MLANGUAGE", "$MLAYOUT", 
			    "$MREPLY", "$MPREFIX", "$MSIGNATURE", "$MHEADERS", "$MLOGOUT", "$MVISIBLE",
			    "$MSORTORDER", "$MSORTCOLUMN", "$MSHOWHTML", "$MSHOWTEXT",
			    "$MTRASH", "$MDRAFTS", "$MSENT", "$MATTACHMENTS", "$MBCCCOPY",
			    "$MSHOWHEADERS", "$MORGANIZATION" });
#define SF_UN 0
#define SF_UA 1
#define SF_MP 2
#define SF_LA 3
#define SF_LY 4
#define SF_RI 5
#define SF_RP 6
#define SF_SI 7
#define SF_UH 8
#define SF_LO 9
#define SF_VM 10
#define SF_SO 11
#define SF_SC 12
#define SF_SH 13
#define SF_ST 14
#define SF_FT 15
#define SF_FD 16
#define SF_FS 17
#define SF_SA 18
#define SF_BC 19
#define SF_DH 20
#define SF_OR 21

// inline display, oliv3 FIXME
mapping stats = ([ ]);
multiset (string) image_types = (< >);

void init_image_types () {
  image_types = (< >);
  string q = QUERY (displayinlineimages);
  if (sizeof (q))
    foreach (q/"\n", string i)
      if (sizeof (i)) {
	string imgtype = lower_case (i);
	if (!image_types[imgtype])
	  image_types[imgtype] = 1;
      }
}

#if constant(thread_create)
// Lock for sessions
object global_lock = Thread.Mutex();
// Lock for logfile
object log_lock = Thread.Mutex();
#endif

int feature (int f) {
  switch(f) {
   case FEAT_MAILFILTER:
    return (QUERY (filterbook) && feature (FEAT_USERSETUP));
    break;
   case FEAT_EDITFILTERBOOK:
    return (QUERY (filterbook) && feature (FEAT_USEREDITSETUP));
    break;
   case FEAT_EXTENDEDABOOK:
    return (QUERY (extendedabook) && feature (FEAT_USEREDITSETUP));
    break;
   case FEAT_USERHEADERS:
    return (QUERY (userheaders) && feature (FEAT_USEREDITSETUP));
    break;
   case FEAT_ADDRESSBOOK:
    return (QUERY (addressbook) && feature (FEAT_USERSETUP));
    break;
   case FEAT_EDITADDRESSBOOK:
    return (QUERY (addressbook) && feature (FEAT_USEREDITSETUP));
    break;
   case FEAT_USERSETUP:
    return (QUERY (usersetup) && !(!QUERY (mailboxes) && !sizeof(QUERY (prefsdir))));
    break;
   case FEAT_USEREDITSETUP:
     //return (QUERY (usersetup) && QUERY (usereditsetup) && !(!QUERY (mailboxes) && !sizeof(QUERY (prefsdir))));
    return (feature (FEAT_USERSETUP) && QUERY (usereditsetup));
    break;
   case FEAT_MAILBOXES:
    return (QUERY (mailboxes));
    break;
   case FEAT_USERMAILPATH:
    return (QUERY (usermailpath) && feature (FEAT_USEREDITSETUP));
    break;
   case FEAT_ATTACHMENTS:
    return (QUERY (attachments));
    break;
   case FEAT_INBOXES:
    return (QUERY (inboxes));
   case FEAT_USERADDRESS:
    return (QUERY (useraddress) && feature (FEAT_USEREDITSETUP));
  case FEAT_LDAP:
#if constant(Protocols.LDAP)
    return (QUERY (ldap));
#else
    return 0;
#endif
  case FEAT_DSN:
    return (QUERY (dsn));
  case FEAT_MAILNOTIFY:
    return (QUERY (mailnotify));
  case FEAT_SHOWHTML:
    return (QUERY (showhtml));
   case FEAT_USERLANGUAGE:
    return (QUERY (userlanguage) && feature (FEAT_USEREDITSETUP));
   case FEAT_USERTRASHFOLDER:
    return (QUERY (usertrashfolder) && feature (FEAT_USEREDITSETUP));
   case FEAT_USERSENTFOLDER:
    return (QUERY (usersentfolder) && feature (FEAT_USEREDITSETUP));
   case FEAT_USERDRAFTSFOLDER:
    return (QUERY (userdraftsfolder) && feature (FEAT_USEREDITSETUP));
   case FEAT_USERSENTSAVEATTACHMENTS:
    return (QUERY (usersentsaveattachments) && feature (FEAT_USERSENTFOLDER) && feature (FEAT_USEREDITSETUP));
   case FEAT_USERBCCCOPY:
    return (QUERY (userbcccopy) && feature (FEAT_USEREDITSETUP));
   case FEAT_USERCANCHANGENAME:
    return (QUERY (usercanchangename) && feature (FEAT_USEREDITSETUP));
   case FEAT_SAVEMAILCOPY:
    return (QUERY (composesavemail));
  case FEAT_USERSETREPLYPREFIX:
    return (QUERY (chgreplymsgprefix) && feature (FEAT_USEREDITSETUP));
  case FEAT_USERSETUPSHOWHIDDENHEADERS:
    return (QUERY (showhiddenheaders) && feature (FEAT_USEREDITSETUP));
  case FEAT_FORWARD:
    break;
  case FEAT_SHOWTEXT:
    return (QUERY (showtext));
  case FEAT_STAYINATTACHMENTS:
    return (feature (FEAT_ATTACHMENTS) && QUERY (stayinattachments));
  case FEAT_ORGANIZATION:
    return (QUERY (organization)) && feature (FEAT_USEREDITSETUP);
    
  default:
    return 0;
  } 
}

static object logfile = 0;

void imho_log (string event, mapping data) {
  string now;

  if (!logfile && sizeof(QUERY (logfile)) > 0) {
    logfile = Stdio.File();
    if (!(logfile->open(QUERY (logfile),"wca"))) {
      logfile = 0;
    }
  }
#if constant(thread_create)
  object lock = log_lock->lock();
#endif

  if(logfile)
    now=replace(ctime(time()),"\n","");

  switch(event) {
   case "startup":
    stats->starttime = time();
    if (logfile)
      logfile->write(now+" CAMAS Starting up.\n");
    break;
   case "sendmail":
    stats->nomailsent++;
    stats->nobytessent+=data->size;
    if (logfile && QUERY (logsendmail))
      logfile->write(now+
		     sprintf(" Sent mail from %s to %s, size %d\n",
			     data->from, data->to, data->size));
    break;
   case "login":
    stats->nologins++;
    if (logfile && QUERY (loglogin))
      logfile->write(now+" Login:"+data->login+"\n");
    break;
   case "loginfailed":
    stats->nofailedlogins++;
    if (logfile && QUERY (loglogin))
      logfile->write(now+" Login failed:"+data->login+"\n");
    break;
   case "logout":
    stats->nologouts++;
    if (logfile && QUERY (loglogin))
      logfile->write(now+" Logout:"+data->login+"\n");
    break;
   case "autologout":
    stats->noautologouts++;
    if (logfile && QUERY (loglogin))
      logfile->write(now+" Auto logout:"+data->login+"\n");
    break;
   case "imapfail":
    stats->noimapfailures++;
    if (logfile && QUERY (logerrors))
      logfile->write(now+" IMAP Error, command: "+data->command+", response: "+data->line+"\n- History:\n- "+replace(data->history,"\n","\n- ")+"\n");
    break;
   case "smtpfail":
    stats->nosmtpfailures++;
    if (logfile && QUERY (logerrors))
      logfile->write(now+" SMTP Error, command: "+data->command+", response: "+data->line+"\n");
    break;
  }
#if constant(thread_create)
  destruct(lock);
#endif
}

void load_global_addressbook() {
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  global_addressbook = "";
  array (string) entries = replace(QUERY (globaladdressbook),"\r\n","\n") / "\n";
  string fn = QUERY (globaladdressbookfile);
  string filebook = "";
  if (sizeof(fn) > 0)
    filebook=Stdio.read_bytes(fn) || "";
  filebook = import_addressbook(filebook);
  entries += replace(filebook,"\r\n","\n") / "\n";
  entries = sort(entries);
  foreach(entries, string entry) {
    if (sizeof(entry) > 0)
      global_addressbook += (sizeof(global_addressbook)==0?"":"\n")+entry;
  }

#if constant(thread_create)
  destruct(lock);
#endif
}


void load_plugin() {
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  if(!plugin_loaded) {
    mixed err;
    master()->set_inhibit_compile_errors("");
    err=catch(plugin=compile_string(Stdio.read_bytes(QUERY (plugin)))());
    master()->set_inhibit_compile_errors(0);
  }
  plugin_loaded=1;
#if constant(thread_create)
  destruct(lock);
#endif
}

constant rfc822_timezones = ([ "ut":0, "utc":0, "gmt":0, "z":0,
			       "edt":14400, "est":18000, "cdt":18000, 
			       "cst":21600, "mdt":21600, "mst":25200,
			       "pdt":25200, "pst":28800, "a":3600,
			       "m":43200, "n":-3600, "y":-43200 ]);

constant gmt_times = ([ "GMT+13":13, "GMT+12":12, "GMT+11":11, "GMT+10":10,
                        "GMT+9":9, "GMT+8":8, "GMT+7":7, "GMT+6":6, "GMT+5":5,
                        "GMT+4":4, "GMT+3":3, "GMT+2":2, "GMT+1":1, "GMT":0,
                        "GMT-1":-1, "GMT-2":-2, "GMT-3":-3, "GMT-4":-4,
                        "GMT-5":-5, "GMT-6":-6, "GMT-7":-7, "GMT-8":-8,
                        "GMT-9":-9, "GMT-10":-10, "GMT-11":-11, "GMT-12":-12 ]);

#define ZERO2D(x) sprintf ("%02d", x)

string fix_timezone(int tz, int isdst) {
  /* Takes a localtime() timezone and isdst and
   * produces a typical RFC822 timezone string.
   * Adjusts for weird time zones (e.g., Newfoundland: -0330/-0230)
   */
  int timezone = (tz - (3600 * isdst))*100/3600;
  if (timezone%100) {
    int q = abs(timezone)/100;
    int r = abs(timezone)%100;
    int s = sgn(timezone);
    timezone = s*(100*q+r*60/100);
  }
  return sprintf("%s%04d", (timezone>=0 ? "-":"+"), abs(timezone));
}

void|array
parse_date(string s) {
  /*
    Date-formats parsed

    Non-RFC822 conformant:
    26 Jul 2000 21:51:41 -0000,  2000-Jul-26 21:51:41 -0000
    2000-07-26 21:51:41 -0000,  Tue, 27 Jul 2000 15:31:30
    Thu, 27 Jul 2000 17:31:30 +0200 (CEST)

    RFC822 conformant:
    Thu, 27 Jul 2000 17:31:30 +0200,  Thu, 27 Jul 2000 17:31 +0200
    Tue, 30 May 2000 01:13:42 PDT,  Tue, 30 May 2000 01:13 PDT

    Output:
    Localized time as an array, see DD_INDEX, MM_INDEX, etc..
    => ({ "31", "08", "00", "14", "47", "53", "2000" , "+0100" , "0", "2451545" });
  */

  array pdate = ({ });
  int year, day, hour, minute, second, timezone, z;
  string wday, month, tmp;

  if (s && (
	    (sscanf (s, "%s,%*[ ]%d%*[ ]%s%*[ ]%d%*[ ]%d:%d:%d%s", wday, day, month, year, hour, minute, second, tmp) == 12) ||
	    (sscanf (s, "%d%*[ ]%s%*[ ]%d%*[ ]%d:%d:%d%s", day, month, year, hour, minute, second, tmp) == 10) ||
	    (sscanf (s, "%d-%d-%d%*[ ]%d:%d:%d%s", day, z, year, hour, minute, second, tmp) == 8) ||
	    (sscanf (s, "%d-%s-%d%*[ ]%d:%d:%d%s", day, month, year, hour, minute, second, tmp) == 8) ||
	    (sscanf (s, "%d%*[ ]%s%*[ ]%d%*[ ]%d:%d:%d%s", day, month, year, hour, minute, second) == 9) ||
	    (sscanf (s, "%s%*[ ]%s%*[ ]%d%*[ ]%d:%d:%d%*[ ]%d%s", wday, month, day, hour, minute, second, year, tmp) == 12)
	    )) 
  {
#if constant(Calendar.Language)
    month = Calendar.Language.English.month_shortname_from_number(z);
#else
    if (z<12 && z>1) month=(Calendar.Gregorian.month_names[z])[0..2];
#endif
    // get timezone -- else assume UTC/GMT
    if (sscanf(tmp,"%d", timezone)==1) {  
      int q = abs(timezone)/100;
      int r = abs(timezone)%100;
      int s = sgn(timezone);
      timezone = s*(q*60+r)*60;
      timezone=-timezone;
    } else
      timezone=(rfc822_timezones[lower_case((tmp/" ")[0]||"")])||0;
    
    if (year<50) year+=2000; // y2k safety/stupidity
    if (year<1000) year+=1900; // years starts a 1900, ie 1900 + 117 = 2017
    // localize time
    mapping l = localtime(mktime((["year":(year - 1900), "mon":(month?MONTHS[lower_case(month)]:1||1),
				   "mday":day, "hour":hour - gmt_times[QUERY(systz)], "min":minute, "sec":second,
				   "timezone":timezone, "isdst":0 ])));
    
    pdate += ({ ZERO2D (l->mday), ZERO2D(l->mon+1), ZERO2D (l->year % 100) });
    pdate += ({ ZERO2D (l->hour), ZERO2D (l->min), ZERO2D (l->sec) });    
    pdate += ({ sprintf("%04d", l->year + 1900), fix_timezone(l->timezone, l->isdst) });
    pdate += ({ sprintf("%d", l->wday) });
#if constant(Calendar.Coptic)
    // Glue to handle Pike 7.2/7.3 Calenday
    pdate += ({ sprintf("%d", Calendar.Gregorian.Year(l->year + 1900)->julian_day()+l->yday)  });
#else
    pdate += ({ sprintf("%d", Calendar.Gregorian.Year(l->year + 1900)->julian_day(l->yday)) });
#endif
    return pdate;
  }
  else {
    if (QUERY (debug))
      write ("parse_date: could not parse \"" + s + "\"\n");
    return;
  }
}

void|mapping
parse_delay(string s) {
  /*
    Date-formats parsed

    Non-RFC822 conformant:
    26 Jul 2000 21:51:41 -0000,  2000-Jul-26 21:51:41 -0000
    2000-07-26 21:51:41 -0000,  Tue, 27 Jul 2000 15:31:30
    Thu, 27 Jul 2000 17:31:30 +0200 (CEST)

    RFC822 conformant:
    Thu, 27 Jul 2000 17:31:30 +0200,  Thu, 27 Jul 2000 17:31 +0200
    Tue, 30 May 2000 01:13:42 PDT,  Tue, 30 May 2000 01:13 PDT

    Output:
    Localized time as an array, see DD_INDEX, MM_INDEX, etc..
    => ({ "31", "08", "00", "14", "47", "53", "2000" , "+0100" , "0", "2451545" });
  */

  int year, day, hour, minute, second, timezone, z;
  string wday, month, tmp;

  if (s && (
	    (sscanf (s, "%s,%*[ ]%d%*[ ]%s%*[ ]%d%*[ ]%d:%d:%d%s", wday, day, month, year, hour, minute, second, tmp) == 12) ||
	    (sscanf (s, "%d%*[ ]%s%*[ ]%d%*[ ]%d:%d:%d%s", day, month, year, hour, minute, second, tmp) == 10) ||
	    (sscanf (s, "%d-%d-%d%*[ ]%d:%d:%d%s", day, z, year, hour, minute, second, tmp) == 8) ||
	    (sscanf (s, "%d-%s-%d%*[ ]%d:%d:%d%s", day, month, year, hour, minute, second, tmp) == 8) ||
	    (sscanf (s, "%d%*[ ]%s%*[ ]%d%*[ ]%d:%d:%d%s", day, month, year, hour, minute, second) == 9) ||
	    (sscanf (s, "%s%*[ ]%s%*[ ]%d%*[ ]%d:%d:%d%*[ ]%d%s", wday, month, day, hour, minute, second, year, tmp) == 12)
	    )) 
  {
#if constant(Calendar.Language)
    month = Calendar.Language.English.month_shortname_from_number(z);
#else
    if (z<12 && z>1) month=(Calendar.Gregorian.month_names[z])[0..2];
#endif
    // get timezone -- else assume UTC/GMT
    if (sscanf(tmp,"%d", timezone)==1)
      timezone=-timezone;
    else
      timezone=(rfc822_timezones[lower_case((tmp/" ")[0]||"")])||0;
     
    if (year<50) year+=2000; // y2k safety/stupidity
    if (year<1000) year+=1900; // years starts a 1900, ie 1900 + 117 = 2017
    // localize time
    int delay = time() - mktime((["year":(year - 1900), "mon":(month?MONTHS[lower_case(month)]:1||1),
				   "mday":day, "hour":hour - gmt_times[QUERY(systz)], "min":minute, "sec":second,
				   "timezone":timezone, "isdst":0 ]));
    mapping pdelay = ( [ "day":0, "hour":0, "min":0, "sec":0 ]);
    
    float day, hour, min, sec;

    day  = (float)delay/86400;
    pdelay->day     = (int)floor(day);
    hour = (day - pdelay->day) * 24;
    pdelay->hour    = (int)floor(hour);
    min  = (hour - pdelay->hour) * 60;
    pdelay->min     = (int)floor(min);
    sec  = (min - pdelay->min) * 60; 
    pdelay->sec = (int)floor(sec);
    return pdelay;
  }
  else {
    if (QUERY (debug))
      write ("parse_delay: could not parse \"" + s + "\"\n");
    return;
  }
}

string sortable_date(void|string rfc822date, void|string imapdate) {
  string d = ctime(time(1)); // fall-back
  if (rfc822date && rfc822date != "")
    d = rfc822date;
  else
    if (imapdate && imapdate != "")
      d = imapdate;

  void|array p = parse_date(d);
  if (p) 
    return sprintf("%s%s%s%s%s%s%s",
		   p[YYYY_INDEX], p[MM_INDEX], p[DD_INDEX],
		   p[HOURS_INDEX], p[MINUTES_INDEX], p[SECONDS_INDEX],
		   p[TZ_INDEX]);
  else {
    if (QUERY (debug))
      write ("sortable_date: unparseable date(s): \"" + (rfc822date||"") +
	     "\", \"" + (imapdate||"") + "\"\n");
    return "???????????????????";
  }
}

string mailindex_date(mapping sessobj, string|string rfc822date, void|string imapdate) {
  string d = ctime(time(1)); // fall-back
  array imho_from = ({ });
  array imho_to   = ({ });
  mapping lt;
  int z;

  if (rfc822date && rfc822date != "")
    d = rfc822date;
  else
    if (imapdate && imapdate != "")
      d = imapdate;

  void|array pdate = parse_date(d);
  if (!pdate) {
    if (QUERY (debug))
      write ("mailindex_date: unparseable date(s): \"" + (rfc822date||"") +
	     "\", \"" + (imapdate||"") + "\"\n");
    pdate = ({ "??", "??", "??", "??", "??", "??", "????", "+????", "?", "?" });
  }
  if (sizeof(QUERY (indexdateshortformat))) {
    if (pdate[YDAY_INDEX] && pdate[YDAY_INDEX]!="?") {
      lt = localtime(time(1));
#if constant(Calendar.Coptic)
      // Glue to handle Pike 7.2/7.3 Calenday
      z = (Calendar.Gregorian.Year(lt->year + 1900)->julian_day()+lt->yday) - ((int) pdate[YDAY_INDEX]);
#else
      z = (Calendar.Gregorian.Year(lt->year + 1900)->julian_day(lt->yday)) - ((int) pdate[YDAY_INDEX]);
#endif
      if ((z < 7) && (z >= 0)) {
	imho_from = ({
	  "$WDAY", "$WEEKDAY", "$TZ",
	  "$HOURS", "$MINUTES", "$SECONDS"
	});

	string wdayname = "?????";
	if (z) {
	  switch (((int)pdate[WDAY_INDEX]) % 7) {
	  case 0: wdayname = MSG(M_MONDAY); break;
	  case 1: wdayname = MSG(M_TUESDAY); break;
	  case 2: wdayname = MSG(M_WEDNESDAY); break;
	  case 3: wdayname = MSG(M_THURSDAY); break;
	  case 4: wdayname = MSG(M_FRIDAY); break;
	  case 5: wdayname = MSG(M_SATURDAY); break;
	  case 6: wdayname = MSG(M_SUNDAY); break;
	  default: break;
	  }
	}
	else
	  wdayname = MSG(M_TODAY);
	
        imho_to   = ({ 
	  (wdayname[0..2]), wdayname, pdate[TZ_INDEX],
	  pdate[HOURS_INDEX], pdate[MINUTES_INDEX], pdate[SECONDS_INDEX]
	});
	return replace (QUERY (indexdateshortformat), imho_from, imho_to);
      }
    }
  }
  // if we are here, it's the longer format
  imho_from = ({ 
    "$DD", "$MM", "$YY", "$TZ",
    "$HOURS", "$MINUTES", "$SECONDS", "$YYYY"
  });
  imho_to   = ({ 
    pdate[DD_INDEX], pdate[MM_INDEX], pdate[YY_INDEX], pdate[TZ_INDEX],
    pdate[HOURS_INDEX], pdate[MINUTES_INDEX], pdate[SECONDS_INDEX], pdate[YYYY_INDEX]
  });
  return replace (QUERY (indexdatelongformat), imho_from, imho_to);
}

string mailindex_delay(mapping sessobj, string|string rfc822date, void|string imapdate) {
  string d = ctime(time(1)); // fall-back
  string delay = "&nbsp;";
  string days = "" , hours = "", min = "", sec = "";
  array imho_from = ({ });
  array imho_to   = ({ });

  if (rfc822date && rfc822date != "")
    d = rfc822date;
  else
    if (imapdate && imapdate != "")
      d = imapdate;

  void|array pdelay = parse_delay(d);
  if (!pdelay) {
    if (QUERY (debug))
      write ("mailindex_delay: unparseable date(s): \"" + (rfc822date||"") +
	     "\", \"" + (imapdate||"") + "\"\n");
    pdelay = ({ "??", "??", "??", "??" });
  }
  switch ((int)pdelay["day"]) {
    case 0: delay += "<div class=\"delay\">"; break;
    case 1: delay += "<div class=\"delay1\">"; break;
    case 2: delay += "<div class=\"delay2\">"; break;
    case 3: delay += "<div class=\"delay3\">"; break;
    case 4: delay += "<div class=\"delay4\">"; break;
    case 5: delay += "<div class=\"delay5\">"; break;
    case 6: delay += "<div class=\"delay6\">"; break;
    default: delay += "<div class=\"delay6\">"; break;
  }
  days = (string)pdelay["day"];
  hours = (string)pdelay["hour"];
  min = (string)pdelay["min"];
  sec = (string)pdelay["sec"];
  imho_from = ({ "$DDAYS", "$DHOURS", "$DMINUTES", "$DSECONDS" });
  imho_to = ({ days, hours, min, sec });
  delay += replace (QUERY (indexdelayformat), imho_from, imho_to);
  delay += "</div>";
  return delay;
}


string make_links(object id, string text){
  string newstr="";
  //FIXME: 10000 ?
  string instr=text[0..(10000<sizeof(text)?10000:sizeof(text)-1)];

  array (string)keys=({ "http://" , "ftp://" , "telnet://","gopher://","wais://", "@","https://" });
  string stoptkn=" \n\r\"()[]{}<>'`;";

  int lenght=sizeof(instr);
  int ready=0;
  int position=0;
  int position2=0;

  while(!ready){
    int minpos=-1;
    int index;
    int foo,tmp;

    for(foo=0;foo<sizeof(keys);foo++){
      tmp=search(instr,keys[foo],position);
      if ((tmp!=-1) && ( minpos==-1 || tmp<minpos)){
	minpos=tmp;
	index=foo;
      }
    }
    
    if(minpos!=-1){
      if(keys[index]!="@"){
	if(minpos>position)
	  newstr+=instr[position..minpos-1];
	newstr+="<a target=\""+
	  (QUERY (linkwindow)?"_blank":"_top")+
	  "\" href=\"";
	position=minpos;
	int amp = 0;
	while((position<lenght) && (amp || search(stoptkn,instr[position..position])==-1)) {
	  if (amp && instr[position] == ';')
	    amp = 0;
	  if (instr[position] == '&')
	    amp = 1;
	  position++;
	}
	newstr+=replace(instr[minpos..position-1],"&amp;","&")+"\">";
	newstr+=instr[minpos..position-1]+"</a>";
      }
      else{
	position2=minpos;
	while((position2>=0) && ((search(stoptkn,instr[position2..position2]))==-1))
	  position2--;
	newstr+=instr[position..position2];
	position=minpos;
	while((position<lenght) && ((search(stoptkn+"&",instr[position..position]))==-1))
	  position++;
	if((position2<minpos-1)&&(position>minpos+2)){
	  newstr+="<a target=\""+id->misc->imho->uptarget+"\" href=\""+id->misc->imho->nextuptarget+"?actioncompose=1&to="+
	    http_encode_url(instr[position2+1..position-1])+"\">";
	  newstr+=instr[position2+1..position-1]+"</a>";
	}
	else
	  newstr+=instr[position2+1..position-1];
      }
    }
    ready=(minpos==-1);
  }
  newstr+=instr[position..];
  //FIXME: 10000 ?
  if (sizeof(text) > 10001)
    newstr += text[10001..sizeof(text)-1];
  return newstr;
}

inline string strip (string s) {
  if (!s)
    return "";
  return replace (s, ({ "\r" }), ({ "" })); 
}

inline string strip_path (string s) {
  if (!s)
    return "";
  return replace (s, ({ ".." }), ({ "" })); 
}

string filetype (string fname) {
  return my_configuration ()->type_from_filename (fname, 0);
}

string
mb_displayname_from_name(string mbox,mapping sessobj) {
  foreach(sessobj->mailboxes,array mbox_a)
    if(mbox_a[MB_FOLDERNAME_IDX]==mbox)
      return mbox_a[MB_DISPLAYNAME_IDX];
  return "";
}

string server_url (object id, void|int no_prestate) {
  string s1, s2, server_url = "";
  int port = 443;

  if (sizeof (QUERY (urloverride)) > 0)
    return QUERY (urloverride);

  if (QUERY (ssl3redir) && id->conf->server_ports){
    if ((!QUERY (sslfaulty)) || (!id->supports->ssl_faulty)) {
      if (QUERY (debug))
	write ("SSL enabled...\n");
      foreach (indices (id->conf->server_ports), string s) {
	if (sscanf (s, "ssl3://%s/%s", s1, s2) == 2) {
	  sscanf (s1, "%s:%d", s1, port);
	  if (s1 == "ANY") {
	    // FIXME: does caudium work w.o. gethostname()?
	    // Yeah, but I don't of of a case where it would happen,
	    // at least not on UNIX. This test is safe in any case. -- david
#if constant(gethostname)
	    s1 = gethostname ();
#else
	    s1 = "localhost";
#endif
	  }
	  server_url = "https://" + s1 + (port == 443 ? "" : (":" + port));
	}
      }
    }
    else {
      if (QUERY (debug))
	write ("SSL disabled...\n");
      server_url = id->conf->query ("MyWorldLocation");
      server_url = server_url[0..sizeof (server_url)-2];
    }
  }

  if ((!id->cookies["SessionID"] || !sizeof (id->cookies["SessionID"])) && (!no_prestate) && id->misc->session_id) {
    server_url += "/(SessionID=" + id->misc->session_id;
    if (sizeof (id->prestate) > 1) {
      foreach (indices (id->prestate), string p) {
	//write ("considering prestate: " + p + " ");
	if (p[0..8] == "SessionID") {
	  //write ("-> already set.\n");
	}
	else {
	  server_url += "," + p;
	  //write ("-> added.\n");
	}
      }
    }
    server_url += ")";
  }

  if (QUERY (debug))
    write ("server_url: " + server_url + "\n");

  return server_url;
}

mapping redirect (object id, string nextpage) {
  string redir = server_url (id, 1) + QUERY (location);
  redir += "newbie" + (nextpage == "" ? "" : ("/" + nextpage));
  //write ("redir= " + redir + "\n");
  return http_redirect (redir);
}

string smtp_format(string s, int keepbcc)
{ // Make sure each line is max 1000 bytes and that no lines begins with "."
  // Also remove BCC address
  string ret = "";
  int headers = 1;
  foreach(s/"\r\n", string l) {
    string bcc = "";
    if (!(headers && !keepbcc &&(sscanf(lower_case(l), "bcc%*[ ]:%s",bcc) > 0) && sizeof(bcc) > 0)) {
      if (sizeof(l) == 0)
	headers = 0;
      while (sizeof(l) > 1000) {
	string first = l[0..999];
	if (headers)
	  l = " "+l[1000..];
	else
	l = l[1000..];
	if (first[0] == '.')
	  ret+= ".";
	ret+=first+"\r\n";
      }
      if (sizeof(l) > 0 && l[0] == '.')
	ret+= ".";
      ret+=l+"\r\n";
    } 
  }
  return(ret);
}

string wrap_lines(string s, int len)
{ // Wrap lines in a text to max len chars (i.e insert CRLFs at
  // appropriate positions). Also make sure LF is really CRLF.
  // And: Remove spaces in the end by replacing them by "\0" and then
  // by nothing.
  s=replace(s,"\r","");
  return (replace(len>0 ? sprintf("%-=*~s",len,"\0",s):s, 
		  ({ "\n", "\0" }), 
		  ({ "\r\n", "" })));
}

// Find a mail with a certain key and return the array index.
int find_mail (array mails, int value)
{ 
  int nr = Array.search_array (mails,
			       lambda (mapping mail, int value) 
			       {
				 if (mail->imap->UID == value)
				   return 1;
				 return 0;
			       }, value);
  return (nr);
}


string mailtime() {
  int t = time();
  mapping lt = localtime(t);
  return sprintf("%s, %02d %s %d %02d:%02d:%02d %s",
#if constant(Calendar.Language)
		 Calendar.Language.English.
		 gregorian_week_day_shortname_from_number(lt->wday),
		 lt->mday,
		 Calendar.Language.English.
		 month_shortname_from_number(lt->mon+1),
#else
		 (Calendar.Gregorian.week_day_names[lt->wday])[0..2],
		 lt->mday, (Calendar.Gregorian.month_names[lt->mon])[0..2],
#endif		 
		 lt->year+1900, lt->hour, lt->min, lt->sec,
		 fix_timezone(lt->timezone, lt->isdst));
}

string
decode_fname(string fname) {
  return replace(fname,({ "@0","@1" }),({ "@","_" }) );
}

string
encode_fname(string fname) {
  return replace(fname,({ "@","_" }),({ "@0","@1" }) );
}


//
// Modified UTF-7 encoding/decoding
//

string decode_mboxname (string s) {
  string out="";
  int coded=0;
  string codeds="";
  string decodeds="";
  foreach(s/"",string s1) { 

    switch(s1) {
     case "&":
      coded=1;
      break;
 
     case "-":
      if(coded) {
	if(sizeof(codeds)) {
	  codeds+=({ "","===","==","=" })[sizeof(codeds)%4];
	  decodeds=MIME.decode_base64(replace(codeds,",","/"));
	  if(sizeof(decodeds)&1)
	    decodeds=decodeds[..sizeof(decodeds)-2];
	  out+=unicode_to_string(decodeds);
	} 
	else
	  out+="&";
	coded=0;
	codeds="";
      }
      else
	out+="-";
      break;
      
     default:
      if(coded)
	codeds+=s1;
      else
	out+=s1;
      break;
      
    }
  }

  if(coded) {
    if(sizeof(codeds)) {
      codeds+({ "","===","==","=" })[sizeof(codeds)%4];
      decodeds=MIME.decode_base64(replace(codeds,",","/"));
      if(sizeof(decodeds)&1)
	decodeds=decodeds[..sizeof(decodeds)-2];
      out+=unicode_to_string(decodeds);
    }
    else
      out+="&";
  }


  return out;
}

int allowed_mbox_char (string s) {
  int i=s[0];
  return (i!='&' && i>=0x20 && i<=0x7e);
}

string encode_mboxname (string s) {
  string out="";
  int coded=0;
  string codeds="";
  foreach(s/"",string s1) {
    
    if(s1=="&") {
      if(coded) {
	out+="&"+replace(MIME.encode_base64(string_to_unicode(codeds)),({"/","="}),({",",""}) )+"-";
	coded=0;
	codeds="";
      }
      out+="&-";
    }
    else {
      if(allowed_mbox_char(s1)) {
	if(coded) {
	  out+="&"+replace(MIME.encode_base64(string_to_unicode(codeds)),({"/","="}),({",",""}) )+"-";
	  coded=0;
	  codeds="";
	}
	out+=s1;
      }
      else {
	coded=1;
	codeds+=s1;
      }
    }
  }

  if(coded)
    out+="&"+replace(MIME.encode_base64(string_to_unicode(codeds)),({"/","="}),({",",""}) )+"-";

  return out;
}

string encode_header (string s,string cs) {
  //FIXME: doesn't honour the length limits, 75,76 chars.
  //       it would be very difficult due to the integral constraint
  //       for multibyte character sets
  array a1=(Locale.Charset.encoder(cs,"")->feed(s)->drain())/" ";
  array a2=Array.map(a1,lambda(string s,string cs){return MIME.encode_word(({s,cs}),"quoted-printable");},cs);
  array a3=Array.map(a2,lambda(string s){return (s/"?");} );

  for(int i=0;i<sizeof(a1);i++){
    if(a1[i]!=a3[i][3]){
      if(i && a1[i-1][0..1]=="=?") {
	a3[i][3]="_"+a3[i][3];
	a1[i]=a3[i]*"?";
      }
      else
	a1[i]=a2[i];
    }
  }
  return a1*" ";
}


string
encode_addresses(array a,string cs) {
  array a2=({ });
  string s1,s2;
  foreach(a,string s) {
    if(sscanf(s, "%s<%s>",s1,s2)==2)
      a2 += ({ encode_header(s1,cs)+" <"+s2+">"  });
    else 
      a2 += ({ s });
  }
  
  return a2*",";
}




void
sendmail(mapping sessobj,int createonly){
  // Make a string of the mail in the sessobj, and send it if "sendmail" 
  // is to be used. 
  // If createonly==1, the mail will be converted to a string, but no
  // other actions are taken. The mail string is found in
  // sessobj->sendmail
  int parts=1;
  if (sessobj->attachments)
    parts=sizeof(sessobj->attachments)+1;
  string charset="";
  string header_charset="";
  string message="";
  mixed err=0;
  array charsets=({ });
  //FIXME:  seems to pass for us-ascii...
  if(sessobj->replytoidx !=-1 && sessobj->replytocharset && !(<"us-ascii","us_ascii","ascii" >)[lower_case(sessobj->replytocharset)])
    charsets += ({ sessobj->replytocharset });
  charsets +=({ "iso-8859-1",sessobj->lang_module->lang_charsets[sessobj->language] });
  charsets+= ( Array.map(QUERY (morecharsets)/",",String.trim_whites) - ({ "" })  );
  charsets += ({ "utf-8" });
  foreach( charsets,charset) {
    err=0;
    err=catch { message=Locale.Charset.encoder(charset)->feed(sessobj->message)->drain();};
    if(!err)
      break;
  }

  object mail = MIME.Message("",
			     ([ "MIME-Version" : "1.0",
				"Content-Type" : ((parts==1)?"text/plain; charset="+charset:"multipart/mixed"),
				"User-Agent" : "CAMAS/"+camas_version+" (CAudium Mail Access System)" ]) );

  if (sessobj["message-id"])
    mail->headers["in-reply-to"] = sessobj["message-id"];

  string headers=sessobj->to+sessobj->cc+sessobj->bcc+sessobj["name"]+" <"+sessobj["address"]+">"+sessobj->subject;
  foreach( charsets,header_charset) {
    err=0;
    err=catch { Locale.Charset.encoder(header_charset)->feed(headers)->drain();};
    if(!err)
      break;
  }
  array addresses;
  sessobj->sendrecipients = ({ });
  addresses = address_split(sessobj->to);
  mail->headers["to"]  = encode_addresses(addresses,header_charset);
  sessobj->sendrecipients += addresses;
  if (strlen(sessobj["cc"]) > 0) {
    addresses = address_split(sessobj->cc);
    mail->headers["cc"]  = encode_addresses(addresses,header_charset);
    sessobj->sendrecipients += addresses;
  }
  if (strlen(sessobj["bcc"]) > 0) {
    addresses = address_split(sessobj->bcc);
    mail->headers["bcc"]  = encode_addresses(addresses,header_charset);
    sessobj->sendrecipients += addresses;
  }

  foreach(indices(sessobj->sendrecipients), int i) {
    sscanf(sessobj->sendrecipients[i], "%*s<%s>", sessobj->sendrecipients[i]);
    sscanf(sessobj->sendrecipients[i], "%*[ ]%s", sessobj->sendrecipients[i]);
    if(sscanf(reverse(sessobj->sendrecipients[i]), "%*[ ]%s", sessobj->sendrecipients[i])==2)
      sessobj->sendrecipients[i]=reverse(sessobj->sendrecipients[i]);
    if(sizeof(sessobj->sendrecipients[i]) && QUERY (addmaildomain) && !has_value (sessobj->sendrecipients[i], "@")) {
      sessobj->sendrecipients[i]=sessobj->sendrecipients[i]+"@"+sessobj->maildomain;
    }
  }
    
  // Do not encode the mail address. Right or wrong? Problem with "_".
  mail->headers["from"] = encode_header(sessobj["name"],header_charset)+" <"+sessobj["address"]+">";
  mail->headers["subject"] = encode_header(sessobj["subject"],header_charset);
  mail->headers["date"] = mailtime();

  if (QUERY (includeipnumber))
    mail->headers["X-Originating-IP"] = "["+sessobj->ip+"]";

  if (sizeof(sessobj->extraheader)) {
    string key   = "";
    string value = "";
    foreach(sessobj->extraheader/"\n", string l) {
      string k = "";
      string v = "";
      if (sscanf(l," %s",v) && sizeof(v))
        value += v;
      else if (sscanf(l,"%s:%*[ \t]%s",k,v) > 0 &&
               sizeof(k) && sizeof(v)) {
        if (sizeof(key) && !(mail->headers[key]))
	  mail->headers[key] = value;
        key = k;
        value = v;
      }
    }
    if (sizeof(key) && !(mail->headers[key]))
      mail->headers[key] = value;
  }

  if (sessobj->mdn) {
    if (QUERY (mdntype) == "Both")
      mail->headers["disposition-notification-to"] = mail->headers["return-receipt-to"] = mail->headers->from;
    else {
      if (QUERY (mdntype) == "Netscape")
        mail->headers["disposition-notification-to"] = mail->headers->from;
      else // Microsoft
        mail->headers["return-receipt-to"] = mail->headers->from;
    }
  }

  if (sizeof (QUERY (siteorg)))
    mail->headers["organization"] = QUERY (siteorg);
  else
    if (sizeof (sessobj->organization))
      mail->headers["organization"] = sessobj->organization;

  mail->setencoding("8bit");

  if(parts==1)
    mail->setdata(createonly?message:wrap_lines(message,sessobj->wrapcolumn)); 
  else {
    mail["body_parts"]=({ });
    object mailpart = MIME.Message("",
				   ([ "MIME-Version" : "1.0",
				      "Content-Type" : "text/plain; charset="+charset ]) );
    mailpart->setencoding("8bit");
    mailpart->setdata(createonly?message:wrap_lines(message,sessobj->wrapcolumn));
    mail->body_parts+=({ mailpart });

    foreach(sessobj->attachments, mapping file) {
      object mailpart = MIME.Message("",
				     ([ "MIME-Version" : "1.0",
					"Content-Type" : file->type ]) );
      string fname=file->fname;
      if( !CAMAS.Tools.is_7bitsafe( fname ) ) {
        string enc=(sessobj->lang_module->lang_charsets[sessobj->language] 
                                       || "iso-8859-1" );
        fname=Locale.Charset.encoder(enc)->feed(fname)->drain();
        fname=MIME.encode_word(({ fname, enc }),"b");
      }
      mailpart->setdisp_param("filename", fname);
      mailpart->setparam("name", fname);
      mailpart->setencoding((file->type=="text/plain")?"8bit":"base64");
      if (!(file->data))
	mailpart->setdata(Stdio.read_bytes(QUERY (uploaddir)+"/"+sessobj->login+"_"+encode_fname(file->fname)));
      else
	mailpart->setdata(file->data);
      mail->body_parts+=({ mailpart });
    }	    
  }

  sessobj["sendmail"]=(string)mail;
  if (!createonly && QUERY (sendmethod) == "sendmail") {
    object in = Stdio.File("stdout");
    object out=in->pipe();
    Process.spawn(QUERY (sendmail)+" -t -f "+sessobj["address"],out,0,out);
    in->write(smtp_format(sessobj->sendmail,1));
    in->close();
    sessobj->sendmail = 0;
  }

  if (sessobj->saveattachments == "0") {
    if(parts>1) {
      mail->body_parts=({ mail->body_parts[0] });
    }
  }
  //FIXME: optimize: don't cast twice
  sessobj["sentmaildata"]=(string)mail;
}

void get_files(mapping sessobj) {
  if(sizeof(QUERY (uploaddir))) {
    array (string) files;
    array (int) filestat;
    files=get_dir(QUERY (uploaddir));
    if(files) {
      string fname;
      foreach(files,string file){
	if(sscanf(file,lower_case(sessobj["login"])+"_%s",fname)) {
	  if (!has_value (fname, "_")) {
	    fname=decode_fname(fname);
	    filestat=file_stat(QUERY (uploaddir)+"/"+file);
	    sessobj["files"]+= ({ ([ "fname":fname, "size":filestat[1], "type": filetype(fname)  ]) });
	  }
	}
      }
    }
  }
}



void 
default_session(mapping sessobj,object id) {
  //write ("++default_session\n");
  sessobj["smtpserver"]=QUERY (smtpserver);
  sessobj["smtpport"]=QUERY (smtpport);
  sessobj["debug"]=QUERY (debug);
  sessobj["mailbox"] = ({ "INBOX","INBOX",MB_NOINFERIORS,0,({ "INBOX" }) });
  sessobj["files"] = ({ });

  foreach(indices(prefproperties), string prop) {
    sessobj[prop] = prefproperties[prop]; // += hmm, += wasn't destruktive
    sessobj["usersetup"+prop] = 1; // User is allowed to change all (so far)
    // write ("setting properties: " + prop + "= " + prefproperties[prop] + "\n");
  }

  sessobj["name"] = "John Doe";
  sessobj["lastpath"] = "";
  sessobj["usersetupmailpath"] = feature (FEAT_USERMAILPATH);
  sessobj["hidemboxes"] = QUERY (hidemboxes);
  if(!sizeof(sessobj["hidemboxes"]))
    sessobj["hidemboxes"]=0;
  sessobj["usersetup"] = feature (FEAT_USERSETUP);
  sessobj["usersetupaddress"] = feature (FEAT_USERADDRESS);
  sessobj["usersetupextraheader"] = feature (FEAT_USERHEADERS);
  sessobj["usersetupreplymsgprefix"] = feature (FEAT_USERSETREPLYPREFIX);
  sessobj["usersetuptrashfolder"] = feature (FEAT_USERTRASHFOLDER);
  sessobj["usersetupsentfolder"] = feature (FEAT_USERSENTFOLDER);
  sessobj["usersetupsaveattachments"] = feature (FEAT_USERSENTSAVEATTACHMENTS);
  sessobj["usersetupautobcc"] = feature (FEAT_USERBCCCOPY);
  sessobj["usersetupname"] = feature (FEAT_USERCANCHANGENAME);
  sessobj["usersetuplanguage"] = feature (FEAT_USERLANGUAGE);
  sessobj["usersetupshowhiddenheaders"] = feature (FEAT_USERSETUPSHOWHIDDENHEADERS);
  sessobj["showprefsbox"] = QUERY (showprefsbox);
  sessobj["prefsdir"] = QUERY (prefsdir);
  sessobj["wrapcolumn"] = (int) QUERY (wrapcolumn);
  sessobj["imapidlelogout"] = (int) QUERY (imapidlelogout);
  sessobj["displayimaperrors"] = QUERY (displayimaperrors);
  sessobj["displaysmtperrors"] = QUERY (displaysmtperrors);
#if constant(thread_create)
  sessobj["lock"] = Thread.Mutex();
#endif
  sessobj["language"] = 0;
  sessobj->filters_have_changed = 1;
  
  array(string) layouts = id->conf->get_provider ("camas_layout_manager")->list_layouts ();
  if (sizeof (layouts) == 1)
    sessobj->overridelayout = layouts[0];

  sessobj->lang_module = id->conf->get_provider ("camas_lang");

  if (sessobj->lang_module->query ("browserlang") && (id->request_headers["accept-language"])) {
    foreach (id->request_headers["accept-language"] / ",", string foo) {
      foo = (foo / ";")[0];
      if (sessobj->lang_module->lang_progs_short[foo]) {
	sessobj["language"] = sessobj->lang_module->lang_progs_short[foo];
	break;
      }
    }
  }
  if(!sessobj["language"])
    sessobj["language"] = sessobj->lang_module->query ("defaultlang");
  //FIXME: better UTF-8 detection?
  sessobj["utf-8"]=0;
  if (QUERY (useutf-8)) {
    if(search(lower_case(id->request_headers["accept-charset"]||""),"utf-8")!=-1)
      sessobj["utf-8"]=1;
    else {
      int i;
      if((i=search(id->client,"MSIE"))!=-1)
	if(sizeof(id->client)>i)
	  sessobj["utf-8"]=((int)(id->client[i+1]))>=4;
    }
  }
    
  sessobj["charset"]=sessobj["utf-8"]?"utf-8":(sessobj->lang_module->lang_charsets[sessobj->language]||"iso-8859-1");
  sessobj["mails"] = ({ });
  sessobj["cmail"] = 0;
  sessobj["cache"] = ([ ]);
  sessobj["cmailidx"] = -1;
  sessobj["cmailuid"] = -1;
  sessobj["imapclient"] = CAMAS.imapclient.imapclient ();
  sessobj["prefsloaded"]=0;
  sessobj["mboxencode"] = (QUERY (mboxencode) == "encode");
  sessobj["ldap"] = feature (FEAT_LDAP);
#if constant(Protocols.LDAP)
  //oliv3 FIXME is this ok ? (ldap functions do the queries)
  sessobj["ldapserver"] = QUERY (ldapserver);
  sessobj["ldapsearchroot"] = QUERY (ldapsearchroot);
#endif
}


string query_provides () {
  return "camas_main";
}

// stene FIXME: into its own .h-file?
// This maybe a good idea - xavier
#define DEFAULT_IMAPSERVERS "your.domain.com\tlocalhost"
#define DEFAULT_LAYEDITADDRESSFILTER 	 \
	"<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"660\">" \
	"<tr>" \
	"<td width=\"310\" valign=\"middle\"><font face=\"Arial\" size=\"3\">" \
	"<b>Email field:</b>$INPUTFIELD</font></td>" \
	"<td width=\"310\" valign=\"middle\"><font face=\"Arial\" size=\"3\">" \
	"<b>Filtre:</b> $INPUTFILTER</font></td>" \
	"<td width=\"310\" valign=\"middle\"><font face=\"Arial\" size=\"3\">" \
	"<b>Folder:</b> $INPUTFOLDER</font></td>" \
	"</tr><tr>" \
	"<td colspan=4 align=\"left\">$BUTTONFILTERSAVE $BUTTONFILTERCANCEL $BUTTONFILTERDELETE</td>" \
	"</tr>" \
	"</table>"
#define DEFAULT_LAYLOGOUT "$LOGOUTMSG"
#define DEFAULT_LAYLOGIN1 \
	 "<define name=jscript>on</define>\n" \
	"<table>\n <tr>\n  <td><imho_text nfont='Arial Rounded' scale=0.3>$MSGLOGIN</imho_text></td>\n" \
        "  <td><imho_text nfont='Arial Rounded' scale=0.3>$MSGPASSWORD</imho_text></td>" \
	"  <td></td>\n" \
	" </tr>\n" \
	" <tr>\n" \
	"  <td>$INPUTLOGIN</td>\n" \
	"  <td>$INPUTPASSWORD</td>\n" \
	"  <td>$INPUTOK</td>\n" \
	" </tr>\n" \
	"</table>"
#define DEFAULT_LAYLOGIN2 \
	"<center><table><tr><td>\n" \
        " <imho_text nfont='Arial Rounded' scale=0.3>$MSGUSRINFACE</imho_text></td></tr>\n" \
	" <tr><td>$INPUTSELECTINFACE</td></tr></table></center>"
#define DEFAULT_LAYCOMPOSE1 "$INPUTSEND $INPUTCANCEL $INPUTSPELLCHECK $INPUTSPELLCHOOSE $INPUTSAVEDRAFT"
#define DEFAULT_LAYCOMPOSE2 \
	"<TABLE BORDER=0 WIDTH=\"100%\"><tr><td>$MSGFROM</td>\n" \
	"<td>$MSGEMAILFROM $MSGORGANIZATION</td></tr>\n" \
	"<tr><td>$MSGTO</td><td>$INPUTTO $INPUTADDRBOOKTO $INPUTLDAPTO</td></tr>\n" \
	"<tr><td>$MSGCC</td><td>$INPUTCC $INPUTADDRBOOKCC $INPUTLDAPCC</td></tr>\n" \
	"<tr><td>$MSGBCC</td><td>$INPUTBCC $INPUTADDRBOOKBCC $INPUTLDAPBCC</td></tr>\n" \
	"<tr><td>$MSGSUBJECT</td><td>$INPUTSUBJECT</td></tr>\n</table>"
#define DEFAULT_LAYCOMPOSE3 "$MSGATTACHMENTS $INPUTFILEATT $INPUTADD $INPUTDEL"
#define DEFAULT_LAYCOMPOSE4 \
        "$INPUTMSGAREA <br />\n" \
	"$INPUTSAVECOPY $INPUTDSNDELAY $INPUTDSNSUCCESS $INPUTMDN"
#define DEFAULT_LAYFOLDERSROW \
	"<tr>\n<td>$MARK</td><td>$NAME</td><td>$MESSAGES</td>\n<td>$UNSEEN</td><td>$SIZE</td>\n</tr>"
#define DEFAULT_LAYFOLDERSTOP \
	"Total size: <imho_totalsize> kB (of <imho_quota> Kb, <imho_use>% used)<hr />" \
        "<table width=\"100%\"><tr><th></th><th>Folder</th>" \
	"<th>Messages</th><th>Unseen</th><th>Size</th></tr>"
#define DEFAULT_LAYFOLDERSBOTTOM \
	"</table><hr />Create folder: $NEWFOLDERNAME $CREATE<br />" \
	"Rename folder: $OLDFOLDERNAME $RENAMEFOLDERNAME $RENAME<br />$DELETE"
#define DEFAULT_SELECTADDRESSES "From, To, Cc"
#define DEFAULT_INDEXDATELONGFORMAT "$YY$MM$DD $HOURS:$MINUTES"
#define DEFAULT_INDEXDATESHORTFORMAT "$WDAY $HOURS:$MINUTES"
#define DEFAULT_INDEXDELAY "($DDAYS $DHOURS:$DMINUTES)" 
#define DEFAULT_REPLYSEPARATOR "-------------------"
#define DEFAULT_FORWARDSEPARATOR "------ Forwarded message -------"
#define DEFAULT_LAYSETUP "<tr><td>$MUSERNAME</td><td>$USERNAME</td></tr>\n" \
"<tr><td>$MUSERADDRESS</td><td>$USERADDRESS</td></tr>\n" \
"<tr><td>$MMAILPATH</td><td>$MAILPATH</td></tr>\n" \
"<tr><td>$MLANGUAGE</td><td>$LANGUAGE</td></tr>\n" \
"<tr><td>$MLAYOUT</td><td>$LAYOUT</td></tr>\n" \
"<tr><td>$MREPLY</td><td>$REPLY</td></tr>\n" \
"<tr><td>$MPREFIX</td><td>$PREFIX</td></tr>\n" \
"<tr><td>$MSIGNATURE</td><td>$SIGNATURE</td></tr>\n" \
"<tr><td>$MHEADERS</td><td>$HEADERS</td></tr>\n" \
"<tr><td>$MLOGOUT</td><td>$LOGOUT</td></tr>\n" \
"<tr><td>$MVISIBLE</td><td>$VISIBLE</td></tr>\n" \
"<tr><td>$MSORTORDER</td><td>$SORTORDER</td></tr>\n" \
"<tr><td>$MSORTCOLUMN</td><td>$SORTCOLUMN</td></tr>\n" \
"<tr><td>$MSHOWHTML</td><td>$SHOWHTML</td></tr>\n" \
"<tr><td>$MSHOWTEXT</td><td>$SHOWTEXT</td></tr>\n" \
"<tr><td>$MTRASH</td><td>$TRASH</td></tr>\n" \
"<tr><td>$MDRAFTS</td><td>$DRAFTS</td></tr>\n" \
"<tr><td>$MSENT</td><td>$SENT</td></tr>\n" \
"<tr><td>$MATTACHMENTS</td><td>$ATTACHMENTS</td></tr>\n" \
"<tr><td>$MBCCCOPY</td><td>$BCCCOPY</td></tr>\n" \
"<tr><td>$MSHOWHEADERS</td><td>$SHOWHEADERS</td></tr>\n" \
"<tr><td>$MORGANIZATION</td><td>$ORGANIZATION</td></tr>\n"

/*
  hiding the defvars
*/
int hide_ebook () {
  return !QUERY (extendedabook); 
}

int hide_buttonpathcaudium () {
  return !sizeof (QUERY (buttonpath));
}

int hide_stayinattachments () {
  return !QUERY (attachments);
}
/*
  end functions hiding the defvars
*/

void create (mixed...foo) {
  if(sizeof(foo)==12){
    mapping g=foo[0];
    variables=foo[1];

    prefproperties=g["prefproperties"];
#if constant(thread_create)
    global_lock=g["global_lock"];
#endif
    plugin_loaded=g["plugin_loaded"];
    plugin=g["plugin"];
    global_addressbook=g["global_addressbook"];

    return;
  }

  defvar ("location", "/mail/", "Mountpoint", TYPE_LOCATION, "Mailreader server location.");

  foreach ( indices (prefproperties), string prop ) {
    if (prefprop_att && prefprop_att[prop]) {
      defvar("pref_" + prop, prefproperties[prop],
	     "Default properties:" + prefprop_att[prop][0],
	     prefprop_att[prop][1], prefprop_att[prop][2],
	     (sizeof(prefprop_att[prop])>3)?prefprop_att[prop][3]:0,
	     (sizeof(prefprop_att[prop])>4)?prefprop_att[prop][4]:0);
    }
  }

  defvar("newbiesetup", 1, "New users:Enter setup", TYPE_FLAG,
	 "Choose whether new users (to CAMAS) will enter preferences screen "
	 "when logged in for the first time.");
  defvar("newbiecreatefolder", "", "New users:Create folders",
	 TYPE_STRING, "Comma separated list of foldernames which will be "
	 "created when logging in for the first time.");
  defvar("newbiecreatesentfolder", 1, "New users:Create sent folder",
	 TYPE_FLAG, "Auto create the folder where the outgoing mail is copied.");
  defvar("newbiecreatetrashfolder", 1, "New users:Create trash folder",
	 TYPE_FLAG, "Auto create the folder where are the deleted mail is saved.");
  defvar("newbiecreatedraftsfolder", 1, "New users:Create drafts folder",
	 TYPE_FLAG, "Auto create the folder where are the draft mail is stored.");
  defvar("sitesig", "", "Outgoing mail:Site-wide signature",
	 TYPE_TEXT_FIELD, "Piece of text added to every outgoing message.");
  defvar("siteorg", "", "Outgoing mail:Site-wide organization",
	 TYPE_STRING, "Site-wide organization.");
  defvar("featuredelay", 0, "User interface dialogs:Mail index - Delay",
         TYPE_FLAG, "Add a column delay to mailindex");
  defvar("indexdelayformat", DEFAULT_INDEXDELAY ,
         "User interface dialogs:Mail index - Delay format",  
         TYPE_TEXT_FIELD,
         "Set a format for the delay.<br />"
         "You can use the following replacements :<ul>"
         "<li><b>$DDAYS</b>: Day</li>"
         "<li><b>$DHOURS</b>: Hour</li>"
         "<li><b>$DMINUTES</b>: Minute</li>"
         "<li><b>$DSECONDS</b>: Second</li>"
         "</ul>"
         );

  defvar("layeditaddress",
	 "",
	 "User interface dialogs:Pages - Edit address", TYPE_TEXT_FIELD,
	 "Layout to edit an addressbook entry.\n"
	 "Leave empty to use the default layout (autogenerated).\n"
	 "Otherwise, you must use in your <tt>input</tt> tags :\n"
	 "<lu>\n"
	 "<li><b>$NAME</b> for the name value (primary key)</li>\n"
	 "<li><b>$ADDRESS</b> for the address value</li>\n"
	 "<li><b>$BUTTONSAVE</b> for the validation button</li>\n"
	 "<li><b>$BUTTONDELETE</b> for the deletion button</li>\n"
	 "<li><b>$BUTTONCANCEL</b> for the cancel button</li>\n"
	 "</lu>\n"
	 "You can also use, according to your setting of extended "
	 "addressbook fields :\n"
	 "<lu>\n"
	 "<li><b>%<i>field</i>%</b> as the name of the <tt>input</tt> "
	 "for the extended addressbook field named <i>field</i> "
	 "(case sensitive)</li>\n"
	 "<li><b>$<i>field</i>$</b> as it's value</li>\n"
	 "</lu><br />\n"
	 "e.g.:<br />\n"
	 "Let's suppose we have two extended address book fields named "
	 "<it>Phone</it> and <it>Postal</it>, you could write here :<p><tt>"
	 + html_encode_string(
			      "<input name=\"name\" value=\"$NAME\"> is\n"
			      "<input size=60 name=\"address\" value=\"$ADDRESS\">\n"
			      "<hr />Ring at :\n"
			      "<input size=20 name=\"%Phone%\" value=\"$Phone$\"><br />\n"
			      "Write at:\n"
			      "<textarea name=\"%Postal%\" rows=6 cols=80>"
			      "$Postal$"
			      "</textarea><br />"
			      "$BUTTONSAVE $BUTTONDELETE $BUTTONCANCEL")
	 + "</tt>");
  defvar("attachwindow", 1, "Features:Open attachments in new window",
	 TYPE_FLAG, "Enables multi-window view of attachments.");
  defvar("linkwindow", 1, "Features:Open links in new window",
	 TYPE_FLAG, "Enables opening of links in a new window.");
  defvar("extendedabook", 1, "Features:Extended addressbook",
	 TYPE_FLAG, "Enables extended addressbook.");
  defvar("extendedabookfields", "Firstname, Lastname, Phone, Snailmail",
	 "Features:Extended addressbook fields.",
	 TYPE_STRING, "Comma-separated list of field names.",0,hide_ebook);
  defvar("deleteinebook", 1, "Features:Also delete in extended addressbook",
	 TYPE_FLAG, "When deleting an entry in the addresbook, also delete "
	 "a matching one in the extended addressbook. You probably want "
	 "to say yes unless you plan your users to use either plain or "
	 "extended addressbook.",0,hide_ebook);

  defvar("globaladdressbook", "", "Global address book: Addresses",
	 TYPE_TEXT_FIELD, 
	 "List of addresses which will appear in all users' address books. "
	 "Format: \"&lt;name&gt;:&lt;address&gt;\", e.g \"CAMAS:camas@caudium.net\".\n");

  defvar("globaladdressbookfile", "", "Global address book: File",
	 TYPE_STRING, 
	 "A file containing global addressbook addresses. Can be in LDIF or "
	 "Pine format. (File in real file system or empty.)");

  defvar("displayimaperrors", 0, "User interface:Display IMAP errors to user", TYPE_FLAG,
         "If this is set to 'yes', the IMAP error string from the server "
	 "will be shown to the user in the error dialog. The text comes from "
	 "the server and cannot be altered nor translated.");

  defvar("displaysmtperrors", 0, "User interface:Display SMTP errors to user", TYPE_FLAG,
         "If this is set to 'yes', the SMTP error string from the server will be "
	 "shown to the user in the error dialog. The text comes from the server "
	 "and cannot be altered nor translated.");

  defvar("useutf-8", 1, "User interface:Output UTF-8 (if possible)", TYPE_FLAG,
         "UTF-8 is an encoding for 16-bit characters, which is used in CAMAS internally. "
	 "If this option is enabled, CAMAS will output UTF-8 data to the browser if the "
	 "browser supports it. If it is disabled, the character set will be taken from "
	 "the current language file.");

  defvar("newusermsg", "", "User interface:New user message", TYPE_STRING,
	 "New users are first presented with the preferences screen. This message is shown then.");

  defvar("imapservers", DEFAULT_IMAPSERVERS , "Incoming mail:IMAP servers", TYPE_TEXT_FIELD,
         "List of servers to use. "
         "<i><b>Format:</b> Domain Server[:port]\t [options]</i><br />"
         "Example:\n"
         "<pre>acc.umu.se	imap.acc.umu.se<br />"
         "some.where	mail.some.where</pre>"
         "Option 'mailpath:path' to override the default IMAP mail folder path<br />"
         "Option 'prefsbox:name' to override the default Prefs foldername<br />"
	 "Option 'folderstyle:mailandfolders' indicates that folders can contain both folders and mail<br />"
         "Example:"
         "<pre>acc.umu.se	imap.acc.umu.se	prefsbox:imhoprefs,"
         "mailpath:INBOX</pre>"
         "<br />");

  defvar("layeditaddressfilter", DEFAULT_LAYEDITADDRESSFILTER ,
         "User interface dialogs:Pages - Edit filter", TYPE_TEXT_FIELD,
         "Layout to edit an filterbook entry.\n"
         "Leave empty to use the default layout (autogenerated).\n"
         "Otherwise, you can use :\n"
         "<lu>\n"
         "<li><b>$INPUTNAMEFIELD</b> for the name select option</li>\n"
         "<li><b>$INPUTFIELD</b> for the field select option</li>\n"
         "<li><b>$INPUTNAMEFILTER</b> for the name textbox (primary key)</li>\n"
         "<li><b>$INPUTFILTER</b> for the filter textbox</li>\n"
         "<li><b>$INPUTFOLDER</b> for the select option associated to "
         "extended filterbook field named "
         "(case sensitive)</li>\n"
         "<li><b>$BUTTONFILTERSAVE</b> for the validation button</li>\n"
         "<li><b>$BUTTONFILTERDELETE</b> for the deletion button</li>\n"
         "<li><b>$BUTTONFILTERCANCEL</b> for the cancel button</li>\n"
         "</lu>\n"
         );

  // Logout Screen
  defvar("laylogout", DEFAULT_LAYLOGOUT ,
	 "User interface dialogs:Pages - Logout page", 
	 TYPE_TEXT_FIELD,
         "Replace the content of the logout page by some RXML contents.<br />Use <b>$LOGOUTMSG</b> to use imho"
	 " standard language file messages.");

  // Login Screen - part 1
  defvar("laylogin1", DEFAULT_LAYLOGIN1 ,
	 "User interface dialogs:Pages - Login page, part 1",
	 TYPE_TEXT_FIELD,
	 "Content of the login page with some RXML contents.<br />Use the following to replace with internal imho"
	 " controls:<ul><li><b>$MSGLOGIN</b>: Login message from imho languages files</li>"
	 "<li><b>$MSGPASSWORD</b>: Password message from imho language files</li>"
	 "<li><b>$INPUTLOGIN</b>: Input box for login</li>"
	 "<li><b>$INPUTPASSWORD</b>: Input box for password</li>"
	 "<li><b>$INPUTOK</b>: The Ok button from imho languages files.</li></ul>"
	 "You can set the define \"jscript\" to \"on\" if you wishe to use some "
	 "\"magic\" javascript to help users to login...");
  
  // Login screen - part 2
  defvar("laylogin2", DEFAULT_LAYLOGIN2 ,
	 "User interface dialogs:Pages - Login page, part2",
	 TYPE_TEXT_FIELD,
	 "Content of the login page <i>(selection of layouts if the is more than one layout)</i>.<br />"
	 "As usual you can use the following to replace with controls/messages from imho:<ul>"
	 "<li><b>$MSGUSRINFACE</b>:The message from imho language file to choose the user interface</li>"
	 "<li><b>$INPUTSELECTINFACE</b>:Popup with the differents users interfaces in ;-)</li></ul>.");

  // Compose screen - part 1 : actions buttons
  defvar("laycompose1", DEFAULT_LAYCOMPOSE1 ,
	 "User interface dialogs:Pages - Compose page, Actions buttons",
         TYPE_TEXT_FIELD,
	 "Action button layout e.g. Send, Cancel and Spell check buttons.<br /> As usual you can use<ul>"
	 "<li><b>$INPUTSEND</b>: for the Send button</li>"
	 "<li><b>$INPUTCANCEL</b>: for the cancel button</li>"
	 "<li><b>$INPUTSPELLCHECK</b>: for the spell button</li>"
	 "<li><b>$INPUTSPELLCHOOSE</b>: for the popup list of languages for the spelling</li>"
	 "<li><b>$INPUTSAVEDRAFT</b>: allow the user to save a draft</li></ul>");

  // Compose screen - part 2 : headers
  defvar("laycompose2", DEFAULT_LAYCOMPOSE2 ,
	 "User interface dialogs:Pages - Compose page, Headers actions",
	 TYPE_TEXT_FIELD,
	 "Header actions e.g. the input boxes, buttons to fill the mail headers.<br />"
	 "As usual, some replacements"
	 " fields :<ul><li><b>$MSGFROM</b>: Imho language 'from' text</li>"
	 "<li><b>$MSGEMAILFROM</b>: The email entered in the configuration dialogs</li>"
	 "<li><b>$MSGORGANIZATION</b>: The organization entered in the configuration dialogs</li>"
	 "<li><b>$MSGTO</b>: Imho language 'To' text</li>"
	 "<li><b>$MSGCC</b>: Imho language 'CC' text</li>"
	 "<li><b>$MSGBCC</b>: Imho language 'Bcc' text</li>"
	 "<li><b>$MSGSUBJECT</b>: Imho language 'Subject' text</li>"
	 "<li><b>$INPUTTO</b>: The input box for 'To'</li>"
	 "<li><b>$INPUTCC</b>: The input box for 'CC'</li>"
	 "<li><b>$INPUTBCC</b>: The input box for 'Bcc'</li>"
	 "<li><b>$INPUTSUBJECT</b>: The input box for the 'subject'</li>"
	 "<li><b>$INPUTADDRBOOKTO</b>: Button to add address from addressbook for 'To'</li>"
	 "<li><b>$INPUTADDRBOOKCC</b>: Button to add address from addressbook for 'Cc'</li>"
	 "<li><b>$INPUTADDRBOOKBCC</b>: Button to add address from addressbook for 'Bcc'</li>"
	 "<li><b>$INPUTADDRBOOK</b>: Button to add addresses from addressbook ('To', 'Cc', and 'Bcc')</li>"
	 "<li><b>$INPUTLDAPTO</b>: Button to add address from LDAP addressbook for 'To'</li>"
	 "<li><b>$INPUTLDAPCC</b>: Button to add address from LDAP addressbook for 'Cc'</li>"
	 "<li><b>$INPUTLDAPBCC</b>: Button to add address from LDAP addressbook for 'Bcc'</li>"
	 "<li><b>$INPUTLDAP</b>: Button to add addresses from LDAP addressbook ('To', 'Cc', and 'Bcc')</li>"
	 "</ul>");
	  
  defvar("headersize", 45,
	 "User interface dialogs:Pages - Compose page, Headers size", TYPE_INT,
	 "Width of the (to, cc, bcc) input fields.");

  defvar("subjectsize", 50,
	 "User interface dialogs:Pages - Compose page, Subject size", TYPE_INT,
	 "Width of the subject field.");

  // Compose screen - part 3 : attachment toolbar
  defvar("laycompose3", DEFAULT_LAYCOMPOSE3 ,
	 "User interface dialogs:Pages - Compose page, Attachments actions",
         TYPE_TEXT_FIELD,
	 "Attachments actions for composing mail.<br />As usual there is options/replacements :<ul>"
	 "<li><b>$MSGATTACHMENTS</b>: Imho language for adding attachements files</li>"
	 "<li><b>$INPUTFILEATT</b>: List of attached files</li>"
	 "<li><b>$INPUTADD</b>: Button to add new files</li>"
	 "<li><b>$INPUTDEL</b>: Button to del selected files</li></ul>");

  // Compose screen - part 4 : compose mail screen
  defvar("laycompose4", DEFAULT_LAYCOMPOSE4 , 
	 "User interface dialogs:Pages - Compose page, Compose screen", 
	 TYPE_TEXT_FIELD,
	 "Compose screen.<br />As usual some options/replacements :<ul>"
	 "<li><b>$INPUTMSGAREA</b>: The input box for message writting</li>"
	 "<li><b>$INPUTSAVECOPY</b>: Check box to save a copy into sent mail</li>"
	 "<li><b>$INPUTDSNDELAY</b> & <b>$INPUTDSNSUCCESS</b>: DSN options.</li>"
	 "<li><b>$INPUTMDN</b>: Request a MDN (Mail Delivery Notification)</li></ul>");

  defvar("msgrows", 20,
	 "User interface dialogs:Pages - Compose page, Input box rows", TYPE_INT,
	 "Rows in the input box.");

  defvar("msgcols", 65,
	 "User interface dialogs:Pages - Compose page, Input box columns", TYPE_INT,
	 "Columns in the input box.");

  // Folders list
  defvar("layfoldersrow", DEFAULT_LAYFOLDERSROW,
	 "User interface dialogs:Folders List - Table row", 
	 TYPE_TEXT_FIELD,
	 "The layout for the folders list.<br />As usual some options/replacements :<ul>"
	 "<li><b>$MARK</b>: Button to select the folder for deletion</li>"
	 "<li><b>$NAME</b>: The name of the folder</li>"
	 "<li><b>$MESSAGES</b>: The number of messages</li>"
	 "<li><b>$RECENT</b>: The number of recent messages</li>"
	 "<li><b>$UNSEEN</b>: The number of unseen messages</li>"
	 "<li><b>$SIZE</b>: The size of the folder (in kb).</li></ul>");

  defvar("layfolderstop", DEFAULT_LAYFOLDERSTOP ,
	 "User interface dialogs:Folders List - Table header",
	 TYPE_TEXT_FIELD,
	 "The layout for the top of the table. "
	 "Provided rxml tag: <b>&lt;imho_totalsize&gt;</b> for the size of all the folders.<br />"
	 "(also in &amp;session.allfolderssize;).");

  defvar("layfoldersbottom", DEFAULT_LAYFOLDERSBOTTOM ,
         "User interface dialogs:Folders List - Table footer",
	 TYPE_TEXT_FIELD,
	 "The layout for the bottom of the table.<br />As usual some options/replacements :<ul>"
	 "<li><ul>Creating folders :<li><b>$NEWFOLDERNAME</b>: Text area for the name of the new folder</li>"
	 "<li><b>$CREATE</b>: Button to create the folder</li></ul></li>"
	 "<li><ul>Renaming folders :<li><b>$OLDFOLDERNAME</b>: Selects the folder to rename</li>"
	 "<li><b>$RENAMEFOLDERNAME</b>: Text area for the name of the new folder</li>"
	 "<li><b>$RENAME</b>: Button to rename the folder </li></ul></li>"
	 "<li><ul>Deleting folders :<li><b>$DELETE</b>: Button to delete marked folders</li></ul></li></ul>");

  // Setup screen
  defvar ("laysetup", DEFAULT_LAYSETUP,
	  "User interface dialogs:Pages - Setup screen",
	  TYPE_TEXT_FIELD,
	  "The layout for the setup screen."
	  "<br />As usual some replacements :<ul>"
	  "<li><b>" + SETUP_FIELDS[SF_UN] + "</b>: Field to change the user name.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_UA] + "</b>: Field to change the user address.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_MP] + "</b>: The mailpath.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_LA] + "</b>: The language.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_LY] + "</b>: The layout.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_RI] + "</b>: Include the message when replying.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_RP] + "</b>: The reply prefix.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_SI] + "</b>: The signature.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_UH] + "</b>: User specified headers.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_LO] + "</b>: Auto-logout delay.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_VM] + "</b>: The number of visible messages.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_SO] + "</b>: The mail sorting order.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_SC] + "</b>: The mail sorting column.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_SH] + "</b>: Display HTML mails.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_ST] + "</b>: Display text mails.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_FT] + "</b>: User trash folder.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_FD] + "</b>: User drafts folder.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_FS] + "</b>: User sent-mail folder.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_SA] + "</b>: Save attachments.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_BC] + "</b>: Default BCC.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_DH] + "</b>: Show/Hide headers.</li>"
	  "<li><b>" + SETUP_FIELDS[SF_OR] + "</b>: Organization.</li>"
	  "</ul><br />"
	  "Add a leading '<b>M</b>' to get the field name eg. $USERNAME -> $<b>M</b>USERNAME.");

  defvar("rfc822dateformat", "", "User interface dialogs:Readmail - Date format", TYPE_TEXT_FIELD,
 	  "Set a format for the date.<br />"
	  "You can use the following replacements :<ul>"
          "<li><b>$DD</b>: Day</li>"
          "<li><b>$MM</b>: Month</li>"
          "<li><b>$YY</b>: Short year</li>"
          "<li><b>$YYYY</b>: Long year</li>"
          "<li><b>$HOURS</b>: Hours</li>"
          "<li><b>$MINUTES</b>: Minutes</li>"
          "<li><b>$SECONDS</b>: Seconds</li>"
          "<li><b>$TZ</b>: Timezone</li>"
	  "</ul>"
	  );

  defvar("selectaddresses", DEFAULT_SELECTADDRESSES ,
	 "User interface dialogs:Readmail - Select addresses", 
	 TYPE_STRING,
	 "Allow the user to add individual addresses from the mail headers.<br />"
	 "Allowed values are: From, To, Cc");

  defvar("multipleselect",1,"User interface dialogs:Readmail - Select multiple addresses", TYPE_FLAG,
         "If set, allow the user to add all addresses from the mail headers.");

  defvar("rfc822readdate",0,"User interface dialogs:Readmail - Use RFC822 standard date", TYPE_FLAG,
         "When set, standard RFC822 date is used instead of parsed date as in the folder index.");

  defvar("imapidlelogout", 300,  "Incoming mail:Idle IMAP connection timeout", TYPE_INT,
	 "Close an open IMAP connection after this many seconds. The user session may still be "
	 "active, and will not be disturbed. The IMAP connection will be reopened when the user "
	 "issues an IMAP command.");

  defvar("mboxencode", "encode", "Incoming mail:Mailbox addressing", TYPE_STRING_LIST,
         "Choose whether CAMAS should use IMAPs modified UTF-7 coding when refering "
	 "to mailboxes or if it should send mailbox names unmodified. If you're having trouble "
	 "accessing mailboxes with non-USASCII characters, try changing this option.",
         ({ "encode", "8-bit" }));

  defvar("defaultmailpath", "INBOX", "Incoming mail:Default IMAP mail folder path", TYPE_STRING,
	 "The default mail folder path. Is usually \"INBOX\" (WU-IMAP) or \"INBOX.\" (Courier-IMAP). "
	 "Can be changed by the user, although the user's preferences is always saved here.");

  defvar("hidemboxes", ".", "Incoming mail:Hide mailboxes starting with", TYPE_STRING,
	 "Optionally hide mailboxes starting with the specified string. "
         "Mailboxes starting with '.' are often not really mailboxes.");

  defvar("sendmethod", "SMTP", "Outgoing mail:Method", TYPE_STRING_LIST,
         "Method to use when sending mail.",
         ({ "SMTP", "sendmail" }));

  defvar("sendmail", "/usr/lib/sendmail", "Outgoing mail:Sendmail", 
	 TYPE_STRING,
	 "Location of sendmail.", 0, lambda () { return (QUERY (sendmethod) != "sendmail"); } );

  defvar("smtpserver", "localhost", "Outgoing mail:SMTP server address", 
	 TYPE_STRING,
	 "The address of the SMTP-server.");

  defvar("smtpport", 25,  "Outgoing mail:SMTP server-port", TYPE_INT,
	 "The portnumber of the SMTP-server.");

  defvar("smtpmaildomain", "your.domain.com", "Outgoing mail:SMTP mail domain", TYPE_STRING,
         "The outgoing mail domain.");

  defvar("includeipnumber", 0, "Outgoing mail:Show IP-address", TYPE_FLAG,
         "Include the header field 'X-Originating-IP' in outgoing mail "
	 "to indicate from where the mail was sent.");

  defvar("addmaildomain", 1, "Outgoing mail:Complete unqualified addresses", TYPE_FLAG,
         "If this is enabled, the default mail domain will be added to addresses "
	 "without a \"@\" before sending them to the SMTP server.");

  defvar("ssl3redir", 0, "URL:Redirect to SSL3", TYPE_FLAG,
         "Tries to find a SSL3 port to redirect the initial requests to. Useful if your virtual "
	 "server has more than one listen port and you prefer SSL3 for CAMAS.");

  defvar("urloverride", "", "URL:Server URL override", TYPE_STRING,
	 "Specify your server URL here if CAMAS fails to guess the correct URL."
	 " Example: 'https://your.server.com'.");

  defvar("uploaddir", "", "Files:File upload directory", TYPE_STRING,
         "Directory to hold uploaded attachment files. Leave empty to disable persistent attachment support. ");

  defvar("uploadquota", 0, "Files:File upload user quota", TYPE_INT,
         "Amount of Kb each user is allowed to upload. 0 gives unlimited quota.");

  defvar("uploadsoftquota", 0, "Files:File upload soft user quota", TYPE_INT,
         "Amount of Kb by wich the user can overrun the quota for the last uploaded file.");

  defvar("maxmailsize", 50000,  "User interface:Max mail size", TYPE_INT,
	 "If the mail data exeeds this size (in bytes), "
	 "it will only be shown as a download link.");

  defvar("ispell", file_stat("/usr/bin/ispell")?"/usr/bin/ispell":"",
	 "Spelling:Speller location", TYPE_STRING,
         "Location of ispell binary. Empty string disables spell-checking.");

  defvar("ispelldict", "american", "Spelling:Speller dictionaries",TYPE_STRING,
         "Commaseparated list of speller dictionaries, default first."
	 "(-d option). Display name can be specified after a colon (swedish:Svenska).");

  defvar("speller", "ispell", "Spelling:Spellchecker", TYPE_STRING_LIST,
         "CAMAS supports ispell and aspell. ",
         ({ "ispell", "aspell" }));

  defvar("addressbook", 1, "Features:Addressbook", TYPE_FLAG,
	 "Enables the addressbook. Addressbook is disabled if User setup is.");

  defvar("attachments", 1, "Features:Attachments", TYPE_FLAG,
	 "Enables sending of attachments.");

  defvar("usermailpath", 0, "Features:User mail path", TYPE_FLAG,
	 "Determines if users should be allowed to set the mail folder. Is disabled if User setup is.");

  defvar("mailboxes", 1, "Features:Mailboxes", TYPE_FLAG,
	 "If disabled, only INBOX can be accessed.");

  defvar("filterbook", 1, "Features:Mail Filter", TYPE_FLAG,
         "Enables the mailfilter. Mail filter is disabled if User setup is.");

  defvar("usersetup", 1, "Features:User setup", TYPE_FLAG,
	 "If enabled, each user has a setup (including addressbook, signature etc.) "
	 "saved either in a mailbox or in a file. "
	 "Is disabled if mailboxes is disabled and User preferences directory is empty.");

  defvar("usereditsetup", 1, "Features:User can change setup", TYPE_FLAG,
	 "Determines if the users should be able to change their own setup. "
	 "This option is ignored if \"User setup\" is disabled.");

  defvar("inboxes", 1, "Features:User specified inboxes", TYPE_FLAG,
	 "Lets the user specify a list of inboxes which may be presented "
	 "as a list in the user interface. ");

  defvar("useraddress", 0, "Features:User specified address", TYPE_FLAG,
	 "Enables a user to change his/her email address in the user setup.");

  //oliv3: change the doc: insert <imho_notify> in the layout
  defvar("mailnotify", 1, "Features:Mail notification window", TYPE_FLAG,
	 "Enables a user to open a mail notification window which is reloaded "
	 "periodically (every five minutes). This feature is is built on javascript "
	 "in the layout file, so make sure you use an updated layout file.");

  defvar("showhtml", 1, "Features:Show HTML messages", TYPE_FLAG,
	 "Enables a user to have messages in HTML format shown instead of "
	 "of the corresponding text version. HTML messages are parsed and "
	 "reformatted so that no unsecure HTML tags are displayed. For "
	 "example, links to mailparts are handled but all other links are "
	 "removed. Javascript is not allowed. "
	 "The parsing may be rather CPU-intensive, and the rendering on the "
	 "client side depends on the browser used. Heavily loaded HTML pages "
	 "may then cause unpredicted behaviour.");

  defvar("userheaders", 0, "Features:User specified mail headers", TYPE_FLAG,
	 "Enables a user to add extra headers to outgoing mail.");

  defvar("prefsdir", "", "User preferences:User preferences directory", TYPE_STRING,
         "Directory to hold user preferences. If this one is an empty string, user preferences will be saved in a mail folder. ");

  defvar("prefsbox", ".imhoprefs", "User preferences:User default preferences folder", TYPE_STRING,
         "Mail folder to hold user preferences. Relative to the default mail folder path. This option can be overridden by a different setting in the IMAP server list.");

  defvar("showprefsbox", 0, "User preferences:Show user preferences folder", TYPE_FLAG,
	 "If \"no\", the preferences folder is not displayed.");

  defvar("deletemethod", "move to trash", "Features:Delete method", TYPE_STRING_LIST,
	 "Choose whether mail is deleted immediately or moved to a trash folder first, or "
         "whether both methods should be available.",
	 ({"both","move to trash","delete immediately" }));

  defvar("usercanchangename", 1, "Features:User can change name", TYPE_FLAG,
         "Enables users to change their name in the configuration interface.<br /><i>Yes, this <b>is</b>"
	 " stupid, but some strange people wants such features...</i>.");

  defvar("userlanguage", 1, "Features:User can choose language", TYPE_FLAG,
	 "Enable a user to choose a different language than the one setup in the configuration interface."
	 /*,0, hide_feature_language*/);

  defvar("usertrashfolder", 1, "Features:User can specify trash folder", TYPE_FLAG,
         "Enable a user to specify the name of the trash folder other than <i>\"Trash\"</i>.");

  defvar("usersentfolder", 1, "Features:User can specify sentmail folder", TYPE_FLAG,
         "Enable a user to specify the name of the sent-mail folder other than <i>\"sent-mail\"</i>.");
 
  defvar("userdraftsfolder", 0, "Features:User can specify drafts folder", TYPE_FLAG,
         "Enable a user to specify the name of the drafts folder other than <i>\"Drafts\"</i>.");
  
  defvar("userbcccopy", 1, "Features:User can specify a default Bcc", TYPE_FLAG,
         "When enabled, user can specify a BCC address to be added to all outgoing mail.");
  
  defvar("showhiddenheaders", 1, "Features:User configurable 'show full headers'-button", TYPE_FLAG,
         "When reading a mail, users can show or hide the hidden headers of the mail. If "
	 "this option is enabled, the user can choose whether to show or hide this button.");

  defvar("usersentsaveattachments", 1, "Features:User can choose to save attachments", TYPE_FLAG,
         "Enable a user to secify to save the attachments in the sentmail folder.<br /><b>Note:</b>"
	 " if the sent mail folder cannot be specified, then this feature is automaticly disabled"
	 " and the attachement <b>will</b> be saved into sentmail folder.");

  defvar("showtext", 1, "Features:Show text messages", TYPE_FLAG,
	 "Display text attachments.");

  defvar("chgreplymsgprefix", 1, "Features:User can change reply prefix", TYPE_FLAG,
	 "If set, then user can change the default prefix added to lines of message they reply to.");
  
  defvar("debug", 0, "Debug", TYPE_FLAG,
	 "When on, debug messages will be logged in Caudium's debug logfile. "
	 "This information is very useful to the developers when fixing bugs.");

#if constant(Protocols.LDAP)
  defvar("ldap", 0, "LDAP Global Addressbook:Use ldap", TYPE_FLAG,
	 "Enables a user to search for names in a ldap directory");
  
  defvar("ldapserver","ldap://localhost","LDAP Global Addressbook:LDAP server",
         TYPE_STRING,
	 "Specifies the ldap server to use.");
  
  defvar("ldapsearchroot", "", "LDAP Global Addressbook:Search root", 
         TYPE_STRING,
	 "The searchroot for ldap searches. "
	 "For example: o=Your company, c=your country. "
	 "Could be empty");
  
  defvar("ldapuser", "", "LDAP Global Addressbook:User", TYPE_STRING,
	 "The user you should use connect. NOTE! This is ldap user format. "
         "Could be empty");
  
  defvar("ldappass", "", "LDAP Global Addressbook:Password", TYPE_STRING,
	 "The password you should use to connect. Could be empty.");
  
  defvar("ldapshowou", 1, "LDAP Global Addressbook:Show ou", TYPE_FLAG,
	 "If \"yes\" include last level ou in result");

  defvar("ldapversion", 2, "LDAP Global Addressbook:Version", TYPE_INT_LIST,
	 "The LDAP protocol version to use.", ({ 2, 3 }));
#endif

  defvar("dsn", 0, "Outgoing mail:DSN support", TYPE_FLAG,
 	 "Add DSN (Delivery Status Notification) support when sending mail. This requires "
	 "the use of SMTP for outgoing mail (instead of sendmail). You also have to make "
	 "sure your SMTP server supports DSN.");

  defvar("wrapcolumn", 70, "Outgoing mail:Wrap line column", TYPE_INT,
	 "Wrap outgoing message lines at this column. If set to zero, no wrapping will take place.");

  defvar("allowipchange", 1, "Allow changing user IP-address", TYPE_FLAG,
	 "Should CAMAS allow one session to use different IP addresses? "
	 "If some users are using dynamic addresses which change even during a session "
	 "(e.g. while composing mail) then this option should be \"yes\". A reason to set "
	 "it to \"no\" is that a stolen session id (from cookie- or log file) cannot be "
	 "used from a remote machine.");

  defvar("adminlogin", "", "Runtime admin interface:Login Name", TYPE_STRING, 
	 "The CAMAS Runtime Administration Interface can be reached by logging "
	 "in using this login name and the specified password.");

  defvar("adminpasswd", "", "Runtime admin interface:Password", TYPE_PASSWORD, 
	 "The CAMAS Runtime Administration Interface can be reached by logging "
	 "in using this password and the specified login name.");

  defvar("logfile", "", "Logging:Log file name", TYPE_STRING, 
	 "The file name of the log file in the real file system."
	 " If this is empty, logging is disabled.");

  defvar("loglogin", 1, "Logging:Log login and logout", TYPE_FLAG,
	 "Log information on users logging in and out.");

  defvar("logsendmail", 1, "Logging:Log sending of mail", TYPE_FLAG,
	 "Log information on all messages sent by users.");

  defvar("logerrors", 1, "Logging:Log errors", TYPE_FLAG,
	 "Log information on all errors that occur regarding IMAP and SMTP.");

  defvar("plugin", "", "General purpose plugin", TYPE_STRING|VAR_EXPERT,
         "Experimental plugin.");

  defvar("morecharsets", "", "Outgoing mail:Character sets",TYPE_STRING,
         "Commaseparated list of charctersets wich CAMAS tries to use "
	 "for outgoing mail before the last resort, UTF-8. CAMAS first tries "
	 "to fit the mail into the same characterset used by the mail the user "
	 "replied to. Then it tries iso-8859-1, the charcterset of the "
	 "active language file, this list of charactersets and UTF-8. "
	 "Specify those charactersets your users usually use to avoid their "
	 "mail being sent as UTF-8.");

  defvar("indexdatelongformat", DEFAULT_INDEXDATELONGFORMAT ,
	 "User interface dialogs:Mail index - Date format", 
	 TYPE_TEXT_FIELD,
	 "Set a format for the date.<br />"
	 "You can use the following replacements :<ul>"
	 "<li><b>$DD</b>: Day</li>"
	 "<li><b>$MM</b>: Month</li>"
	 "<li><b>$YY</b>: Short year</li>"
	 "<li><b>$YYYY</b>: Long year</li>"
	 "<li><b>$HOURS</b>: Hours</li>"
	 "<li><b>$MINUTES</b>: Minutes</li>"
	 "<li><b>$SECONDS</b>: Seconds</li>"
	 "<li><b>$TZ</b>: Timezone</li>"
	 "</ul>"
	 );

  defvar("indexdateshortformat", DEFAULT_INDEXDATESHORTFORMAT ,
	 "User interface dialogs:Mail index - Use short date",
	 TYPE_TEXT_FIELD,
	 "Set a format for the date if it is within the last week.<br />"
	 "If nothing set, use the <i>&quot;normal&quot;</i> full output, "
	 "specified in <i>&quot;Mail index - Date format&quot;</i> above.<br />"
	 "You can use the following replacements :<ul>"
	 "<li><b>$WDAY</b>: Short format weekday-name (eg. Fri for friday)</li>"
	 "<li><b>$WEEKDAY</b>: Full weekday-name (like Thursday)</li>"
	 "<li><b>$HOURS</b>: Hours</li>"
	 "<li><b>$MINUTES</b>: Minutes</li>"
	 "<li><b>$SECONDS</b>: Seconds</li>"
	 "<li><b>$TZ</b>: Timezone</li>"
	 "</ul>"
	 );

  defvar("signaturecols","55","User interface dialogs: Pages - Setup page, size of signature", TYPE_STRING,
         "Change the max line length for signature box in the preferences page");

  defvar("replyseparator", DEFAULT_REPLYSEPARATOR ,
	 "Compose mail:Reply Separator",
	 TYPE_STRING,
	 "Text separator for reply / reply to all messages.");

  defvar("forwardseparator", DEFAULT_FORWARDSEPARATOR ,
	 "Compose mail:Forward Separator",
	 TYPE_STRING,
	 "Text separator for forward messages.");

  defvar("composesavemail", 1, "Outgoing mail:By default save composed mail into sentmail folder", TYPE_FLAG,
	 "If enabled, mail will by default be saved when sending, but a checkbox with "
	 "'Don't save this mail' is shown. If disabled, mail will not be sent by default.");

  defvar("fwdsubjpf","Fwd:","Compose mail:Forward subject prefix",TYPE_STRING, 
	 "This is the prefix to insert before the subject when forwarding a mail.");

  defvar("fwdfrom","From:","Compose mail:Forward from prefix",TYPE_STRING,
	 "This is the prefix to insert before the sender's mail address.");
 
  defvar("fwdto","To:","Compose mail:Forward to prefix",TYPE_STRING,
	 "This is the prefix to insert before the To's mail address.");

  defvar("fwdreplyto","Reply-To:","Compose mail:Forward reply-to prefix",TYPE_STRING,
	 "This is the prefix to insert before the Reply-To mail address.");
 
  defvar("fwdcc","Cc:","Compose mail:Forward cc prefix",TYPE_STRING,
	 "This is the prefix to insert before the Carbon Copy mail address.");
 
  defvar("fwddate","Date:","Compose mail:Forward date prefix",TYPE_STRING,
	 "This is the prefix to insert before the Date of the mail.");

  defvar("organization", 0, "Features:Organization", TYPE_FLAG,
	 "Allow the user to specify the organization he belongs to.");

  defvar("displayinlineimages", "gif\njpeg", "Features:Display inline images", 
	 TYPE_TEXT_FIELD, "List of MIME-subtypes to display inline.<br />"
	 "Enter one subtype per line, examples are :<br />"
	 "gif, "
	 "ief, "
	 "jpeg, "
	 "png, "
	 "tiff, "
	 "x-cmu-raster, "
	 "x-ms-bmp, "
	 "x-portable-anymap, "
	 "x-portable-bitmap, "
	 "x-portable-graymap, "
	 "x-portable-pixmap, "
	 "x-rgb, "
	 "x-xbitmap, "
	 "x-xpixmap, "
	 "x-xwindowdump");

  defvar("notifysent", 1, "Outgoing mail:Sent message notification", TYPE_FLAG,
	  "Display a \"sent message\" window to the user.");

  defvar("stayinattachments", 1, "User interface dialogs:Pages - Compose page, Stay in attachments screen", TYPE_FLAG,
	  "If <b>yes</b>, the user will <b>not</b> automatically go back to the compose "
	  "screen after adding an attachment.",
	  0, hide_stayinattachments);

  defvar("mdntype", "Both", "Outgoing mail:MDN support", TYPE_STRING_LIST,
	  "MUA Mail Delivery Notification type.",
	  ({ "Both", "Netscape", "Microsoft" }));

  defvar("rxmlparsercalls", 3, "Debug:Maximum calls to parse_rxml", TYPE_INT|VAR_EXPERT,
	 "Maximum calls to parse_rxml");

  defvar("foldersinfo", 1, "User interface dialogs:Display folders information", TYPE_FLAG,
	 "Show some folders information: number of (total, recent, unseen) messages, folder size, ...");

  defvar("buggyimap", 0, "Incoming mail:IMAP server is not RFC-compliant", TYPE_FLAG | VAR_EXPERT,
	 "Some IMAP servers (e.g. Courier 1.3) are not RFC-compliant.<br />"
	 "For example, they will not return \"[TRYCREATE]\" when appending a message "
	 "in a non-existent folder.<br /> This will enable various workarounds to make "
	 "CAMAS work as needed. <br />"
	 "<hr />RFC2060, section 6.3.11 (APPEND Command)<br />"
	 "<pre>If the destination mailbox does not exist, a server <b>MUST</b> return an\n"
	 "error, and <b>MUST NOT</b> automatically create the mailbox.  Unless it is\n"
	 "certain that the destination mailbox can not be created, the server\n"
	 "<b>MUST</b> send the response code \"[TRYCREATE]\" as the prefix of the text\n"
	 "of the tagged NO response.  This gives a hint to the client that it\n"
	 "can attempt a CREATE command and retry the APPEND if the CREATE is\n"
	 "successful.</pre>");

  defvar("upgrading", 1, "Compatibility:Upgrading from IMHO to CAMAS", TYPE_FLAG,
	 "CAMAS will read 'IMHO Preferences' if 'CAMAS Preferences' are not found.");

  defvar("defaultlayout", "Default", "Compatibility:Layout to use", TYPE_STRING,
	 "The layout to use if none found in the old preferences files.", 0, lambda () { return !QUERY (upgrading); });

  defvar("frommaxlength", 0, "User interface dialogs:Mail index - Max length for the 'sender' field",
	 TYPE_INT | VAR_MORE,
	 "Set the max length for the 'sender' field. If set to 0, the whole string will be displayed.");

  defvar("subjectmaxlength", 0, "User interface dialogs:Mail index - Max length for the 'subject' field",
	 TYPE_INT | VAR_MORE,
	 "Set the max length for the 'subject' field. If set to 0, the whole string will be displayed.");

  defvar("foldernamemaxlength", 0, "User interface dialogs:Folders list - Max length allowed for a mailbox name",
	 TYPE_INT | VAR_MORE,
	 "Set the max length for a new mailbox name. If set to 0, there will be no check.");

  defvar("subjectlayer", 0, "User interface dialogs:Mail index - Show full subject in a floating table",
	 TYPE_FLAG | VAR_EXPERT,
	 "Show the full subject.");

  defvar("quota", 0, "User interface dialogs:Folders List - Mailbox quota",
	 TYPE_INT,
	 "Set an informational quota for the mailboxes (in kB). If set to 0, no quota will be displayed."
	 " Provided rxml tag: <b>&lt;imho_quota&gt;</b> (the quota itself),"
	 " and <b>&lt;imho_use&gt;</b> for the percentage of the quota used (rounded to an integer).");

  defvar ("mime_remap",
	  "text/rfc822-headers:text/plain\n"
	  "message/disposition-notification:text/plain",
	  "User interface dialogs:Readmail - Change MIME types",
	  TYPE_TEXT_FIELD,
	  "Change the MIME type of mail or attachments."
	  " Format is old_type/old_subtype:new_type/new_subtype");

  defvar ("jslogout", 0, "Features:Add JavaScript to close the session", TYPE_FLAG,
	  "If set, JavaScript code will be added to close the session if the user closes the window "
	  "instead of following a 'logout' link.");

  defvar ("gctimeout", 600, "Internals:Garbage collecting frequency",
	  TYPE_INT,
	  "Sets the delay (in seconds) between two garbage collections.");

  defvar ("sslfaulty", 0, "URL:No SSL if ssl_faulty", TYPE_FLAG,
	  "If set, SSL will be disabled if the client has <b>id->supports->ssl_faulty</b> set.");

  // Compatibility variables
  // FIXME: is this realy used for now ? -- Xavier
  defvar("imapserver", "localhost", "Incoming mail:IMAP server address.", 
         TYPE_STRING|VAR_EXPERT,
         "FOR COMPATIBILITY ONLY. DON'T USE! The address of the IMAP-server.");
  defvar("imapport", 143,  "Incoming mail:IMAP server-port", 
         TYPE_INT|VAR_EXPERT,
         "FOR COMPATIBILITY ONLY. DON'T USE! The portnumber of the IMAP-server.");
  defvar("maildomain", "your.domain.com", "Outgoing mail:Default mail domain", 
         TYPE_STRING|VAR_EXPERT,
         "FOR COMPATIBILITY ONLY. DON'T USE! The default mail domain, i.e the string after the '@' in the mail address in outgoing mail.");

  // Setting timezone there because camas or pike seem to see itself in GMT
  // even if the system timezone in not on GMT... Any ideas ?
  defvar("systz","GMT","System Timezone", TYPE_STRING_LIST,
         "Set there the timezone to the correct timezone of the machine "
         "running this CAMAS.", sort(indices(gmt_times)));

  defvar("attachmenttype", 1, "User interface dialogs:Readmail - Show the attachments type",
         TYPE_FLAG, "When reading a mail with attachments, add the type of the file.");

  imho_log("startup", 0);
}

void destroy ()
{

}

string status () {
  string status = "";
  object sess123 = my_configuration ()->get_provider ("123sessions");
  if (objectp (sess123)) {
    if(sess123->QUERY(storage) != "memory") {
      status += "<font color=red><font size=+3>WARNING !!!</font> "
                "123session module is not set to use <b>memory</b> storage "
		"method. CAMAS <b>CANNOT</b> work with other storage method "
		"than 'memory', this is not a bug from CAMAS or 123session "
		"</font><br />";
    }
    status += "Active sessions:\n" + sess123->status () + "<br />";
  } 
  status += "CAMAS Authentication :<br />";
  object auth_module = my_configuration ()->get_provider ("camas_auth");
  if (objectp (auth_module)) {
    status += auth_module->status ();
  }
  else {
    status += "<font color=red>Warning:</font> no authentication module installed. ";
    status += "CAMAS will still work, but you might want to install a suitable authentication module.";
  }
  status += "<br /><br />";

  status += my_configuration ()->get_provider ("camas_layout_manager")->status ();
  status += "<br />";

  if (my_configuration ()->get_provider ("rxml:core")->module_name[0..2] == "XML") {
    status += "XML Parser status: ";
    if (global_kludges)
      status += global_kludges + " kludges.";
    else {
      status += "fine, thanks.";
      if (global_ok)
	status += " (" + global_ok + " page" + ((global_ok == 1) ? "" : "s") + " correctly parsed).";
    }
  }

  return status;
}
  
void 
destroy_clients (object conf) {
  if (QUERY (debug))
    write ("destroy_clients...\n");

  object sess123 = conf->get_provider ("123sessions");
  if (objectp (sess123)) {
    mapping sessions = sess123->sessions ();
 
    foreach(indices(sessions),string s)
      if(sessions[s]->values->imapclient)
        destruct(sessions[s]->values->imapclient);
  }
}

// Function to draw some form imagebutton using Camas form button provider
string formdrawbutton(string img, string action, string value, void|string onClick)
{
  string workaround = MIME.encode_base64(Locale.Charset.encoder("UTF-8")->feed(value)->drain());
  string out = "<camas_formbutton utf8text=\""+workaround+"\" text=\""+value+"\" action=\"" + action +"\" ";
         out+= "file=\"" + img + "\" ";
  if (onClick) out+="onClick=\"" + onClick + "\" ";
  out += ">";
  return out;
}

string check_variable(string var, mixed value) {
  
  switch (var) {
    
  case "uploadquota":
  case "uploadsoftquota":
  case "frommaxlength":
  case "subjectmaxlength":
  case "foldernamemaxlength":
    if (value < 0)
      return "Must be positive integer or zero.";
    break;
    
  case "headersize":
  case "subjectsize":
  case "msgrows":
  case "msgcols":
  case "gctimeout":
    if (value <= 0)
      return "Must be positive integer.";
    break;

  case "prefsdir":
  case "uploaddir":
    if(sizeof(value)) {
      array stat=file_stat(value);
      if(!stat || stat[1]!=-2)
	return "Must be a directory or empty string.";
    }
    break;
    
  case "imapservers":
    if (sizeof(value/" ") < 2 && sizeof(value / "\t") < 2)
      return("Must contain at least one server in format 'domain imapserver'");
    break;
    
  case "ispell": {
    if(!sizeof(value))
      break;
    array stat=file_stat(value);
    if(!stat || !(stat[0] & 73))
      return "Must be empty or an executable file.";
  }
    break;
    
  case "ispelldict":
    if(!sizeof(value))
      return "Must not be empty.";
    break;
    
  case "sendmail":
    if (QUERY (sendmethod) == "SMTP")
      break;
    array stat = file_stat (value);
    if (!stat || !(stat[0] & 73))
      return "Must be an executable file.";
    break;
    
  case "plugin": {
    if(!sizeof(value))
      break;
    plugin_loaded=0;
    plugin=0;
    array stat=file_stat(value);
    if(!stat)
      return "Must be empty or existing file.";
  }
    break;
    
  }
  return 0;
}

// Import PINE address book
string import_pinebook (string abook)
{
  string ab = "";
  array (string) words = ({ "", "", "" });
  int i = 0;
  while(sscanf(abook,"(%s)%s",words[i],abook) ||
	sscanf(abook, "%[^\t\n]%*c%s", words[i], abook) > 1) {
    i++;
    if (sizeof(abook) && abook[0] == '\n')
      abook = abook[1..];
    if (i > 2) {
      i = 0;
      words[2] = replace(words[2], "\n", "");
      array names = Array.map (ab / "\n", lambda (string s){ return ((s / ":")[0]);});
      string name = words[0];
      int no = 0;
      while (search (names, name + (no ? " - " + (string)no : "")) != -1)
	no++;
      if (no)
	name += (" - " + (string)no);
      if (sizeof (ab))
	ab += "\n" + name + ":" + words[2];
      else
	ab = name + ":" + words[2];
    }
  }
  return (ab);
}

// Import LDIF (Ldap / Netscape) Address book
string import_ldifbook(string abook){
  string ab = "";
  array (string) lines = abook / "\n";
  string name = "", address = "", data, type;
  int i = 0;

  while(i < sizeof(lines)) {
    if (sscanf(lines[i], "%s: %s", type, data) == 2) {
      if (type == "dn") {
	if (sizeof (name) > 0) {
	  if (sizeof (ab) > 0)
	    ab += "\n"+name+":"+address;
	  else
	    ab = name+":"+address;
	}
	name = "";
	address = "";
	if (sscanf(data, "cn=%s,mail=%s", name, address) < 2)
	  sscanf(data, "cn=%s", name);
      }
      if (type == "member") {
	string na = "", ad = "";
	sscanf(data, "cn=%s,mail=%s", na, ad);
	if (sizeof(ad) > 0) {
	  if (sizeof(address) > 0)
	    address += ", "+ad;
	  else
	    address = ad;
	}
      }
    }
    i++;
  }
  if (sizeof(name) > 0) {
    array names=Array.map(ab/"\n",lambda(string s){return ((s/":")[0]);});
    int no=0;
    while(search(names,name+(no?" - "+(string)no:""))!=-1)
      no++;
    if(no)
      name+=(" - "+(string)no);
    if (sizeof(ab))
      ab += "\n"+name+":"+address;
    else
      ab = name+":"+address;
  }
  return(ab);
}

// Import CSV (M$ Outlook export format, Excel, etc..) addressbook format
string import_csvbook(array(string) abook)
{
  string ab="";
  string name="", address="";
  
  foreach(abook, string foo)
  {
    if(sscanf(foo,"%s;%s;%*s",name,address)>=2)
    {
      // This is an entry, check if this is an email
      if(search(address,"@")>=1) 
      {
        // there is at least one caracter before @ so we can
        // afford this address as a valid email address...
        if (sizeof (ab) > 0)
          ab += "\n"+name+":"+address;
        else
          ab = name+":"+address;
      }
    }
  }
  return ab;
}

// Import addressbook subsystem
string import_addressbook(string file) {
  string adds = replace(file, "\r", "");
  array (string) lines = adds / "\n";
  if (sizeof(lines) > 0) {
    string name="", address="";
    if (sscanf(lines[0], "%s\t%s", name, address) == 2)
      return import_pinebook(adds);
    if (sscanf(lines[0], "dn: %s", address) == 1)
      return import_ldifbook(adds);
    if (sscanf(lines[0], "%s;%s;%*s",name, address) >=2)
      return import_csvbook(lines - ({ "" }));
  }
  return "";	// Unknown format :)
}

void start (int num, object conf) {
  module_dependencies (conf, 
	({ "123session", "graphic_text", "killframe",
	   "camas_language", "camas_layout_manager", "camas_layout_default",
	   "camas_images", "camas_html", "camas_formbuttons",
	   "camas_imho" }));
  
  foreach (call_out_info (), array a)
    if ((sizeof (a) > 5) && (a[3] == "CAMAS_Survivor") && (sizeof (a) < 7 || a[6] == my_configuration ())) {
      remove_call_out (a[5]);
      break;
    }

  if (gc_call_out)
    remove_call_out (gc_call_out);
  gc_call_out = call_out (camas_gc, QUERY (gctimeout));

  array(string) layouts = conf->get_provider ("camas_layout_manager")->list_layouts ();
  prefproperties->layout = layouts[0];

  if (!plugin_loaded)
    load_plugin ();

  load_global_addressbook ();

  init_image_types ();
  global_kludges = global_ok = 0;
}

void stop () {
  mapping g = ([ ]);
  g["prefproperties"] = prefproperties;
#if constant(thread_create)
  g["global_lock"]=global_lock;
#endif
  g["plugin_loaded"]=plugin_loaded;
  g["plugin"]=plugin;
  g["global_addressbook"]=global_addressbook;

  if (gc_call_out)
    remove_call_out (gc_call_out);

  array a=({ 0 });
  // Caudium tries to kill us. Escape...
  // Destroy all sessions after 60 seconds if the new CAMAS hasn't been compiled by then.
  a[0]=call_out(lambda(mixed ... foo){foo[1]->destroy_clients(my_configuration());},60,"CAMAS_Survivor",new(object_program(this_object()),g,variables,0,0,0,0,0,0,0,0,0,0),a,my_configuration());
}


void setup_state (object id, mapping arg) 
  // Creates sessobj->nextpage and
  // id->misc->imho->target
  // Parses arguments
{
  if(id->misc->imho && id->misc->imho->sessobj) {
    mapping sessobj=id->misc->imho->sessobj;
    //string target = "",sidetarget="",uptarget="";
    // XB: try to handle direct login by post to url/?login=login&passwd=pass
    // XB FIXME: how to create the 123sessions variable without problem ?
    string target = "",sidetarget="",uptarget="",login="", passwd="";
    if (arg["target"])
      target = sidetarget = uptarget = arg["target"];
    if (arg["sidetarget"]) // Target to a frame in the same level
      sidetarget = arg["sidetarget"];
    if (arg["uptarget"]) // Target in a frame level up
      uptarget = arg["uptarget"];
    // XB: try to handle direct login (part 2)
    if(arg["login"])
      login = arg["login"];
    if(arg["passwd"])
      passwd = arg["passwd"];
    if(sizeof(login) > 0)
      id->misc->imho->login = login;
    if(sizeof(passwd) > 0)
      id->misc->imho->passwd = passwd;
    if (sizeof(target) == 0)
      target = uptarget;
    if (sizeof(target) == 0)
      target = sidetarget;
    if (sizeof(target) > 0)
      id->misc->imho->nexttarget = id->misc->imho->nexturl+target;
    else
      id->misc->imho->nexttarget = id->misc->imho->nexturl+id->misc->imho->frame;
    if (sizeof(sidetarget) > 0)
      id->misc->imho->nextsidetarget = id->misc->imho->nexturl+sidetarget;
    else
      id->misc->imho->nextsidetarget = id->misc->imho->nexturl+id->misc->imho->frame;
    if (sizeof(uptarget) > 0)
      id->misc->imho->nextuptarget = id->misc->imho->nexturl+uptarget;
    else
      id->misc->imho->nextuptarget = id->misc->imho->nexturl+id->misc->imho->frame;
    id->misc->imho->nextpage = id->misc->imho->nexturl+id->misc->imho->frame;
    id->misc->imho->target = target;
    id->misc->imho->sidetarget = sidetarget;
    id->misc->imho->uptarget = uptarget;
    if (arg->face)
    {
      id->misc->imho->face = arg->face;

      if (arg->size)
	id->misc->imho->size= arg->size;
      else
	id->misc->imho->size=0;
    }
    else
      id->misc->imho->face = 0;
    id->misc->imho->nogtext = 0;
    if (arg->nogtext)
      id->misc->imho->nogtext = 1;
    id->misc->imho->notopbuttons = 0;
    if (arg->notopbuttons)
      id->misc->imho->notopbuttons = 1;
    id->misc->imho->nobottombuttons = 0;
    if (arg->nobottombuttons)
      id->misc->imho->nobottombuttons = 1;
    if(arg->nobackbutton)
      id->misc->imho->nobackbutton=1;
    if(arg->columns)
      id->misc->imho->columns=arg->columns;
  }
}

string query_location() { return QUERY (location); }

// FIXME: It's here because imho_kludge_parse_rxml() is this function *realy*
//        usefull ???? -- Xavier
string cont_imho_text (string tag_name, mapping arg, string content, object id) {
  object camas_imho = id->conf->get_provider("camas_imho");
  if (objectp(camas_imho))
    return camas_imho->cont_imho_text(tag_name, arg, content, id);
  else
    return "";
}

// Used for <imho_totalsize>
// FIXME: Can we move that into camas_imho.pike ? - Xavier
int get_totalsize (object id) {
  return ((id->misc->imho && id->misc->imho->sessobj) ? (id->misc->imho->sessobj->allfolderssize >> SIZE_SHIFT): 0);
}

// FIXME: imho_kludge_parse_rxml() hell there... Need to be fixed...
string tag_imho_image (string tag_name, mapping args, object id, object file) {
  return my_configuration ()->get_provider ("camas_images")->tag_imho_image (tag_name, args, id, file);
}

void spellcheck (mapping sessobj) {
  sessobj["spelling"]= ({ });
  sessobj["misspelled"]= ({ });
  sessobj["checkword"] = 0;

  object file1=Stdio.File();
  object file2=file1->pipe();
  object file3=Stdio.File();
  object file4=file3->pipe();
  string res;

  Process.create_process(({QUERY (ispell),"-a","-d",sessobj["ispelldict"]})+ ((QUERY (speller)=="aspell") ? ({ "--ignore-repl" }):({ })),(["stdin":file2,"stdout":file4 ]) );
  //add a space before each line to prevent ispell from interpreting single-word lines as commands
  //FIXME: how should charsets be treated?
  string stripped_message=Locale.Charset.encoder("iso-8859-1","")->feed(sessobj->message)->drain();
  file1->write((Array.map((stripped_message)/"\n",lambda(string s){return " "+s;}))*"\n");
  file1->close();
  file2->close();
  file4->close();
  res=file3->read();
  file3->close();

  array ispell_data=res/"\n";
  array input_rows=(stripped_message)/"\n";
    
  if(sizeof(ispell_data)>1) {
    int i,row=0,pos=0,pos2;
    string word,suggestions;
    for(i=1;i<sizeof(ispell_data)-1 && row<sizeof(input_rows);i++) {
      if(!sizeof(ispell_data[i])){ // next row
	(sessobj->spelling)+=({ input_rows[row][pos..]+"\n" });
	row++;
	pos=0;
      }
      else {
	// pos2 will be a position ahead original data, due to the extra space.
        switch(ispell_data[i][0]) {
	 case '&': // misspelled, suggestions
          sscanf(ispell_data[i],"& %s %*d %d:%s",word,pos2,suggestions);
	  (sessobj->spelling)+=({ input_rows[row][pos..pos2-2] ,({ word , word })+Array.map(suggestions/",",String.trim_whites) });
          (sessobj->misspelled)+=({ sizeof(sessobj->spelling)-1 });
          pos=pos2-1+sizeof(word);
          break;
	 case '#': //misspelled
	  sscanf(ispell_data[i],"# %s %d",word,pos2);
	  (sessobj->spelling)+=({ input_rows[row][pos..pos2-2] ,({ word , word }) });
	  (sessobj->misspelled)+=({ sizeof(sessobj->spelling)-1 });
	  pos=pos2-1+sizeof(word);
	  break;
	  
	}
      }
    }
  }
}


string handle_admin_interface(object id, mapping sessobj){
  string ret = "";
  string action = "";
  object lock;
  
  foreach(indices(id->variables),string var) {
    string foo;
    if (var)
    if(sscanf(var,"action%s",foo))
      action=foo;
  }

  ret+="<html><head><title>CAMAS Runtime Admin</title></head>";
  ret+="<body leftmargin=0 topmargin=0 bottomargin=0 rightmargin=0 marginheight=0 marginwidth=0 bgcolor=\"#ffffff\" text=\"#000000\" link=\"#0000ff\" vlink=\"#0000ff\" alink=\"#ff0000\">";
  ret+="<table width=\"100%\" cellspacing=\"0\" cellpadding=\"0\" border=\"0\"><tr><td>";
  ret+="<imho_image border=\"0\" quant=\"250\" fg=\"white\" bg=\"#3030c0\" align=\"left\" width=\"500\" height=\"32\" image=\"bannerleft\" hspace=\"0\" text=\"CAMAS Runtime Admin\" />";
  ret+="<imho_image width=\"100%\" quant=\"250\" fg=\"&imho_thcolor;\" hspace=\"0\" bg=\"#3030c0\" border=\"0\" height=\"32\" image=\"bannerright\" />";
  ret+="</td></tr></table>";

  switch(action) {
   case "changesort":
    if (id->variables->col)
      sessobj->adminsortcolumn = (int) id->variables->col;
    if (id->variables->dir)
      sessobj->sortup = (int) id->variables->dir;
    sessobj->users = Array.sort_array(sessobj->users,lambda(array a1,array a2, int col, int dir) {
						       return( dir?(a1[col] > a2[col]):(a1[col] < a2[col]) ); }, sessobj->adminsortcolumn, sessobj->sortup);
    break;
   case "logout":
#if constant(thread_create)
    lock = global_lock->lock();
#endif
    destruct(sessobj->imapclient);
#if constant(thread_create)
    destruct(lock);
#endif
    ret+"<br /><br />";
    ret+=MSG(M_LOGOUTMSG);
    ret+="<end_session>";
    ret+="</body></html>";
    return ret;
    break;  
   case "index":
    sessobj->showfrom = (int) id->variables->showfrom;
    break;
   default:
    sessobj->users = ({ });

#if constant(thread_create)
    lock = global_lock->lock();
#endif

    object sess123 = id->conf->get_provider ("123sessions");
    if (objectp (sess123)) {
      mapping sessions = sess123->sessions ();

      if (!sessobj->users)
  	sessobj->users = ({ });

      foreach (indices (sessions), string s) {
        mapping sess = sessions[s];

        array user = ({ });
        if (sess->values->login) {
  	  user+=({ sess->values->address });
	  user+=({ sess->values->logintime });
	  user+=({ time() - sess->lastusage });
	  user+=({ sess->values->ip });
	  user+=({ (sess->values->mails ? sizeof (sess->values->mails) : 0) });
	  user+=({ sess->values->layout });
	  user+=({ sess->values->language });
  	  sessobj->users += ({ user });
        }
      }
    }
#if constant(thread_create)
    destruct(lock);
#endif

    sessobj->users = Array.sort_array(sessobj->users,lambda(array a1,array a2, int col, int dir) {
						       return( dir?(a1[col] < a2[col]):(a1[col] > a2[col]) ); }, sessobj->adminsortcolumn, sessobj->sortup);
    sessobj->showfrom = 0;
    break;
  }


  ret+="<a name=activeusers><gh3>"+sizeof(sessobj->users)+" Active Users</gh3></a><br />";
  ret+="<table width=\"100%\"><tr>";
  array columns = ({ "Login", "Login Time", "Idle Time", "IP-Address", "# Msgs in Box", "Layout", "Language" });
  for (int i = 0; i < sizeof (columns); i++) {
    ret += "<th bgcolor=\"#000070\" nowrap><font color=\"white\"><b><a href=\"";
    ret += id->misc->imho->nexturl + "?actionchangesort=1&col=" + i + "&dir=";
    ret += (sessobj->sortup == 0 && sessobj->adminsortcolumn == i ? "1" : "0");
    ret += "\"><imho_image bg=\"#000070\" border=\"0\" alt=\"" + MSG(M_CHANGESORTORDER);
    ret += "\" size=\"60\" scale=\"0.25\" image=\"";
    ret += ((sessobj->adminsortcolumn == i) ? (sessobj->sortup ? "arrowup" : "arrowdown") : "arrownone");
    ret += "\" /></a> " + columns[i] + "</b></font></th>";
  }
  ret += "</tr>\n";
  
  int i = 0;
  
  for (i = sessobj->showfrom; i < sizeof(sessobj->users) && i < sessobj->showfrom+20; i++) {
    array user = sessobj->users[i];
    ret+="<tr bgcolor=\""+((i/3)&1?"#fcfce0":"white")+"\">";
    ret+="<td>"+user[0]+"</td>";
    ret+="<td>"+ctime(user[1])+"</td>";
    ret+="<td>"+(user[2]/60)+" mins </td>";
    ret+="<td>"+user[3]+"</td>";
    ret+="<td>"+user[4]+"</td>";
    ret+="<td>"+user[5]+"</td>";
    ret+="<td>"+user[6]+"</td>";
    ret+="</tr>";
  }
  ret+="</table>\n";
  ret+=sprintf("&nbsp;Showing users %d-%d of %d", sessobj->showfrom+1, sessobj->showfrom+20<sizeof(sessobj->users)?sessobj->showfrom+20:sizeof(sessobj->users),sizeof(sessobj->users));
  if (sessobj->showfrom > 0) {
    ret += "<a href=\"" + id->misc->imho->nexturl + "?actionindex=1&showfrom=";
    ret += (sessobj->showfrom - 20 > 0 ? sessobj->showfrom - 20 : 0) + "\">[Previous]</a>";
  }
  if (sessobj->showfrom + 20 < sizeof (sessobj->users)) {
    ret += "<a href=\"" + id->misc->imho->nexturl + "?actionindex=1&showfrom=";
    ret += (sessobj->showfrom + 20) + "\">[Next]</a>";
  }
  ret+="<br />";

  ret+="<form action=\""+id->misc->imho->nexturl+"admin#activeusers\" method=\"get\" name=\"imhoupdateadmin\">";
  ret+="<imho_submit name=\"actionupdate\" value=\"Update\">";
  ret+="<imho_submit name=\"actionlogout\" value=\"Logout\">";
  ret+="</form><hr />\n";

  int uptime = time() - stats->starttime;
  ret+="<a name=stats><gh3>Statistics</gh3></a><br />";
  ret+="&nbsp;Statistics since last restart:";
  ret+="<table>\n";
  ret+="<tr><td><b>Uptime:</b></td><td>"+sprintf("%.1f hrs",((float) uptime)/3600.0)+"</td></tr>";
  ret+="<tr><td><b>Logins:</b></td><td>"+stats->nologins+"</td></tr>";
  ret+="<tr><td><b>Failed logins:</b></td><td>"+stats->nofailedlogins+"</td></tr>";
  ret+="<tr><td><b>Logouts:</b></td><td>"+stats->nologouts+"</td></tr>";
  ret+="<tr><td><b>Autologouts:</b></td><td>"+stats->noautologouts+"</td></tr>";
  ret+="<tr><td><b>Messages sent:</b></td><td>"+stats->nomailsent+"</td></tr>";
  ret+="<tr><td><b>Bytes sent:</b></td><td>"+stats->nobytessent+"</td></tr>";
  ret+="<tr><td><b>IMAP errors:</b></td><td>"+stats->noimapfailures+"</td></tr>";
  ret+="<tr><td><b>SMTP errors:</b></td><td>"+stats->nosmtpfailures+"</td></tr>";
  ret+="</table>";

  ret+="<form action=\""+id->misc->imho->nexturl+"admin#stats\" method=\"get\" name=\"imhoupdateadmin\">";
  ret+="<imho_submit name=\"actionupdate\" value=\"Update\">";
  ret+="<imho_submit name=\"actionlogout\" value=\"Logout\">";
  ret+="</form><hr />\n";

  ret+="</body></html>";
  return ret;
}


mapping process_request (string file, object id)
{
  mapping sessobj = id->misc->session_variables;
  string action = "";
  array fileparts = file / "/";

  if (sessobj->administrator)
    return create_output (id);

  if (id->variables->mailpart)
    return id->conf->get_provider ("camas_html")->mailpart (id);

  // FIXME: is'nt Image and Image.XFace allways included in Pike 7.x??
  // In theory that doesn't have to be the case, but in general I'd say yes.
  // However Image.GIF might not always be nowadays... :)
  // Besides, we still do support Pike 0.6 (until after the release - then
  // we can clean up 0.6 (and even 0.5 compat - oh the horror). -- david
#if constant(Image.XFace.decode)
  if (id->variables->xface) {
    object img;
    if (catch (img = Image.XFace.decode (MIME.decode_base64 (id->variables->xface))))
      return 0;
#if 0 // constant(Image.PNG.encode)
    if (id->supports->png)
      return http_string_answer (Image.PNG.encode (img), "image/png");
#else
    return http_string_answer (Image.GIF.encode (img), "image/gif");
#endif
  }
#endif

  if(id->variables->download){
    if((int)id->variables->download < sizeof(sessobj->files)){
      mapping file=sessobj->files[(int)id->variables->download];
      object downfile=open(QUERY (uploaddir)+"/"+(sessobj->login)+"_"+encode_fname(file->fname),"r");
      if (downfile)
	return http_file_answer(downfile,file->type,file->size);
      else
	return 0;
    }
    else
      return 0;
  }

  if (id->variables->layout && (id->variables->layout != "__user"))
    sessobj["overridelayout"] = id->variables->layout;

  foreach (indices (id->variables), string var) {
    string foo;
    if (sscanf (var, "action%s", foo))
      action = foo;
  }

  // if type=image in submit
  int foo;
  foo=sizeof(action);
  if(foo>2 && (action[foo-2..]==".x" || action[foo-2..]==".y") )
    action=action[..foo-3];

  if (sessobj->loadfiles) {
    sessobj["loadfiles"]=0;
    get_files(sessobj);
  }

  if (action != "") {
    sessobj->selected = ({ });
    sessobj->selectedfiles = ({ });
    sessobj->selectedattachments = ({ });

    foreach (indices (id->variables), string var) {
      if (id->variables[var] == "1") {
	int n, nr, tonr = -1;
	if (n = sscanf (var, "msg%d-%d", nr, tonr)) {
	  switch (n) {
	  case 1: sessobj->selected += ({ nr }); break;
	  case 2: sessobj->selected += ({ nr + ":" + tonr }); break;
	  default: break;
	  }
	}
	else {
	  if (sscanf (var, "file%d", nr))
	    sessobj->selectedfiles += ({ nr });
	}
      }

      if (var == "attachments") {
	foreach (id->variables[var] / "\0", string s)
	  if ((int)s >= 0)
	    sessobj->selectedattachments += ({ (int) s });
      }
    }

    if ((sessobj->status < 0) && action != "login")
      action = "";

    switch (action) {
      
     case "login":
      if (sessobj->status < 0) {
	sessobj->logintime = time ();
	sessobj->passwd = FROM_UTF8 (id->variables->passwd || ""); // FIX For Stupid 8bit passwords !
	sessobj->login = id->variables->login || "";

        if (sessobj->login != QUERY (adminlogin)) {
	  if (sessobj->debug)
	    write ("user login: ");
	  sessobj->address = sessobj->defaultaddress = 0;

	  object auth_module = id->conf->get_provider ("camas_auth");
	  if (objectp (auth_module)) {
	    if (sessobj->debug) {
	      write (sprintf ("checking for auth_error() ... %s found.\n",
			      (auth_module->auth_error) ? "" : "not"));
	      write ("sessobj->login= " + sessobj->login + "\n");
	    }

	    string|int auth_login = auth_module->getlogin (sessobj->login);
	    
	    if (intp (auth_login)) { // error :(
	      if (auth_module->auth_error) {
		if (sessobj->debug)
		  write ("failed: [" + auth_module->auth_error (auth_login) + "] (" + auth_login + ")\n");
		sessobj->loginerror = auth_module->auth_error (auth_login);
	      }
	      sessobj->login = "";
	      sessobj->status = LOGINFAILED;
	      break;
	    }
	    else { // ok :)
	      array|int auth_result = auth_module->getfullnames (auth_login);
	      if (sessobj->debug) {
		write ("ok: " + auth_login + "\n");
		write (sprintf ("auth_result is: %O\n", auth_result));
	      }
	      sessobj->login = id->variables->login = auth_login;
	      if (!intp(auth_result)) {
		sessobj->name = String.capitalize (auth_result[0]) + " " + String.capitalize (auth_result[1]);
		sessobj->address = sessobj->defaultaddress = auth_result[2];
		if(sessobj->debug)
		  write("address = " + sessobj->address + ", default= " + sessobj->defaultaddress + "\n");
		sessobj->quota = (auth_module->getquota) ? auth_module->getquota (sessobj->login) : 0;
	      }
	    }
	  }
	  else {
	    if (sessobj->debug)
	      write ("no authentication module found.\n");
	    sessobj->address = sessobj->defaultaddress = sessobj->login + "@" + QUERY (smtpmaildomain);
	    sessobj->quota = (int)QUERY (quota);
	  }
	}

	// Figure out imap-server and domainname
	// If user entered user@domain, use @domain instead of the chosen value in the dialog
	array loginsplit = sessobj->login / "@";
	sessobj["imapserver"] = "";
	
	// If not user@domain and use LDAP for searching user's imapserver
	if (sizeof(loginsplit)!=2) {
	  object auth_module = id->conf->get_provider ("camas_auth");
          int version;
          if(objectp(auth_module)) { 
           if(sessobj->debug) {
	      version = auth_module->version();
              write(sprintf("Auth module version : %d \n",version));
	      if(version >= 2) {
                write("Version is >= 2, getimapserver() function should be here... ");
                write(sprintf("%s\n",(auth_module->getimapserver) ? "Yes!":"No"));
                if(! auth_module->getimapserver) version = 1;
              }
            }
            if(version >= 2) {
               // Now trying to find the imapserver
               string|int imapsrv = auth_module->getimapserver(loginsplit[0]);
               if(stringp(imapsrv)) {
                 sessobj["imapserver"] = imapsrv;
		 sessobj["imapport"] = 143;
		 sessobj["mailpath"] = sessobj["defaultmailpath"] = QUERY (defaultmailpath);
		 sessobj["prefsbox"] = sessobj["defaultmailpath"] + QUERY (prefsbox);
               }
            }
	  }
	  if (QUERY (debug))
	    perror("CAMAS: LDAP get imapserver "+sessobj["imapserver"]+"\n");
	}
	int i=1;
	foreach(imap_servers(), array arr) {
	  array servs = arr[1]/":";
	  string server = servs[0];
	  int port = 143;
	  string mailpath = QUERY (defaultmailpath);
	  if (arr[2]->mailpath)
	    mailpath = arr[2]->mailpath;
	  string prefsbox = QUERY (prefsbox);
	  if (arr[2]->prefsbox)
	    prefsbox = arr[2]->prefsbox;
	  if (sizeof(servs) > 1)
	    port = (int) servs[1];
	  sessobj["folderstyle"] = arr[2]->folderstyle||"";
	  if (sizeof(loginsplit) == 2) {
	    if(loginsplit[1] == arr[0]) {
	      // Exact match, we're done
	      sessobj["imapserver"] = server;
	      sessobj["imapport"] = port;
	      sessobj["mailpath"] = sessobj["defaultmailpath"] = mailpath;
	      sessobj["prefsbox"] = sessobj["defaultmailpath"] + prefsbox;
	      sessobj["lcfgmap"] = arr[2];
	      //sessobj["address"] = sessobj["defaultaddress"] = sessobj->login+"@"+arr[0];
	      if(!sessobj->address)
		sessobj["address"] = sessobj["defaultaddress"] = loginsplit[0]+"@"+arr[0];
	      sessobj["maildomain"] = arr[0];
	      if (QUERY (debug))
		perror("CAMAS: imap-server exact match on domain "+arr[0]+"\n");
	      break;
	    }
	    else
	      if (loginsplit[1] == arr[1]) {
		// Exact match, we're done
		sessobj["imapserver"] = server;
		sessobj["imapport"] = port;
		sessobj["mailpath"] = sessobj["defaultmailpath"] = mailpath;
		sessobj["prefsbox"] = sessobj["defaultmailpath"]+prefsbox;
		sessobj["lcfgmap"] = arr[2];
		//sessobj["address"] = sessobj["defaultaddress"] = sessobj->login+"@"+arr[0];
		if(!sessobj->address)
		  sessobj["address"] = sessobj["defaultaddress"] = loginsplit[0]+"@"+arr[1];
		sessobj["maildomain"] = arr[0];
		if (QUERY (debug))
		  perror("CAMAS: imap-server exact match on server "+arr[1]+"\n");
		break;
	      }
	      else
		if (search(arr[0], loginsplit[1]) >= 0) {
		  // Best-effort match, continue through the
		  // list in case we find a better match
		  sessobj["imapserver"] = server;
		  sessobj["imapport"] = port;
		  sessobj["mailpath"] = sessobj["defaultmailpath"] = mailpath;
		  sessobj["prefsbox"] = sessobj["defaultmailpath"]+prefsbox;
		  sessobj["lcfgmap"] = arr[2];
		  if (!sessobj->address)
		    sessobj["address"] = sessobj["defaultaddress"] = sessobj->login+"@"+arr[0];
		  sessobj["maildomain"] = arr[0];
		  if (QUERY (debug))
		    perror("CAMAS: imap-server partial match on "+arr[0]+"\n");
		}
	  }
	  else if (sessobj["imapserver"] == server) {
	    // server found from LDAP match one of the listed servers
	    sessobj["imapserver"] = server;
	    sessobj["imapport"] = port;
	    sessobj["mailpath"] = sessobj["defaultmailpath"] = mailpath;
	    sessobj["prefsbox"] = sessobj["defaultmailpath"]+prefsbox;
	    sessobj["lcfgmap"] = arr[2];
	    if (!sessobj->address)
	      sessobj["address"] = sessobj["defaultaddress"] = sessobj->login+"@"+arr[0];
	    sessobj["maildomain"] = arr[0];
	    if (QUERY (debug))
	      perror("CAMAS: imap-server (from LDAP) match on "+server+"\n");
	    break;
	  }
	  else if (sessobj["imapserver"]=="" &&
                   (!id->variables->imapserver ||
		    i == (int)id->variables["imapserver"])) {
	    sessobj["imapserver"] = server;
	    sessobj["imapport"] = port;
	    sessobj["mailpath"] = sessobj["defaultmailpath"] = mailpath;
	    sessobj["prefsbox"] = sessobj["defaultmailpath"]+prefsbox;
	    sessobj["lcfgmap"] = arr[2];
	    if (!sessobj->address)
	      sessobj["address"] = sessobj["defaultaddress"] = sessobj->login+"@"+arr[0];
	    sessobj["maildomain"] = arr[0];
	    break;
	  }
	  i++;
	}
	sessobj->status = LOGINFAILED;
	if ((sessobj->login == QUERY (adminlogin)) && crypt (sessobj->passwd, QUERY (adminpasswd))) {
	  sessobj->address="ADMINISTRATOR";
	  sessobj->administrator = 1;
	  sessobj->status = ADMINMAIN;
	  return create_output (id);
	}
	else {
	  array lcoms = ({ imap_cmd("load_user_prefs","abortfail",0) });
	  if (feature (FEAT_MAILBOXES)) {
	    if (sizeof(QUERY (newbiecreatefolder))) {
	      foreach (QUERY (newbiecreatefolder)/",", string folder) {
		lcoms += ({
		  imap_cmd("create",
			   "newmailbox", sessobj->mailpath + strip (folder))
		});
	      }
	    }
	    foreach(({"sent", "trash", "drafts"}), string bar) {
	      if (query("newbiecreate"+bar+"folder")) {
		//write ("oliv3: create " + sessobj->mailpath + sessobj[bar + "folder"] + "\n");
		lcoms += ({
		  imap_cmd("create",
			   "newmailbox", sessobj->mailpath + sessobj[bar + "folder"])
	  });
	}
	    }
	    lcoms += ({
	      imap_cmd("low_list", "path", sessobj->mailpath), 
	      imap_cmd("check_mailboxes"),
	      imap_cmd("status","mailbox",sessobj->mailbox[MB_FOLDERNAME_IDX]),
	      imap_cmd("get_headers",
		       "output",imap_cmd_var("mails"),
		       "updatelistpos",1,
		       "setsortcolumn",imap_cmd_var("mailssortcolumn")),
	      imap_cmd("start"),
	      imap_cmd("apply_mail_filters", "updatembox", imap_cmd_var ("mails"))
	    });
	    sessobj->imapclient->imap_command(IMHO, id,sessobj, lcoms);
	return http_pipe_in_progress();
      }
	}
      }
      break;

     case "logout":
      sessobj["status"] = LOGOUT;
      if (sessobj->imapclient) {
	imho_log("logout", ([ "login":sessobj->address ]));
	sessobj->imapclient->imap_command(IMHO, id,sessobj,({ imap_cmd("low_logout", "noselect", 1) }));
	return http_pipe_in_progress();
      }
      break;

     case "compose":
      sessobj["replytoidx"] = -1;
      if (id->variables->to)
        sessobj["to"] = TO_UNICODE(id->variables->to);
      else
        sessobj["to"] = "";
      sessobj["cc"] = "";
      sessobj["bcc"] = sessobj->autobcc;
      sessobj["subject"] = "";
      sessobj["attachments"] = ({ });
      sessobj["message"] = "\n\n" + sessobj["signature"];
      sessobj["status"] = COMPOSE;
      break;

     case "gotoaddattachment":
      sessobj["to"]=TO_UNICODE(id->variables->to);
      sessobj["cc"]=TO_UNICODE(id->variables->cc);
      sessobj["bcc"]=TO_UNICODE(id->variables->bcc);
      sessobj["subject"]=TO_UNICODE(id->variables->subject);
      sessobj["message"]=TO_UNICODE(id->variables->message);
      sessobj["status"] = ATTACHMENTS;
      break;

     case "removeattachment":
      sessobj["status"] = COMPOSE;
     case "removeattachment2":
      array deletes = ({ });
      foreach(sessobj->selectedattachments, int i)
	deletes += ({ sessobj->attachments[i] });
      sessobj->attachments -= deletes;
      break;

     case "continuecompose":
      sessobj["status"] = COMPOSE;
      break;

      /*
     case "continueeditaddress":
      sessobj["status"] = EDITADDRESS;
      break;
      */

     case "continueeditfilter":
      sessobj["status"] = EDITADDRESSFILTER;
      break;

     case "addfileattachment":
      foreach(sessobj->selectedattachments, int i)
	sessobj->attachments += ({ sessobj->files[i] });
      sessobj["status"] = COMPOSE;
      break;

     case "uploadattachment":
      if(feature (FEAT_ATTACHMENTS) && sizeof(id->variables->file)) {
	string fname=(id->variables->fixedfilename && sizeof(id->variables->fixedfilename))?id->variables->fixedfilename:id->variables["file.filename"];
	fname=((((fname)/"/")[-1])/"\\")[-1];
	sessobj["attachments"]+= ({ ([ "fname" : TO_UNICODE(fname), "size" : sizeof(id->variables->file), "type" : filetype(fname), "data": id->variables->file ]) });
      }
      if (!feature (FEAT_STAYINATTACHMENTS))
        sessobj["status"] = COMPOSE;
      break;

     case "addressbookto":
     case "addressbookcc":
     case "addressbookbcc":
      sessobj["to"]=TO_UNICODE(id->variables->to);
      sessobj["cc"]=TO_UNICODE(id->variables->cc);
      sessobj["bcc"]=TO_UNICODE(id->variables->bcc);
      sessobj["subject"]=TO_UNICODE(id->variables->subject);
      sessobj["message"]=TO_UNICODE(id->variables->message);
      sessobj->recipientfield = 0;
      sscanf(action, "addressbook%s", sessobj->recipientfield);
      sessobj["status"] = ADDRESSBOOK;
      break;

     case "addressbooktoccbcc":
      sessobj["to"]=TO_UNICODE(id->variables->to);
      sessobj["cc"]=TO_UNICODE(id->variables->cc);
      sessobj["bcc"]=TO_UNICODE(id->variables->bcc);
      sessobj["subject"]=TO_UNICODE(id->variables->subject);
      sessobj["message"]=TO_UNICODE(id->variables->message);
      sessobj["status"] = ADDRESSBOOK2;
      break;

#if constant(Protocols.LDAP)
     case "ldapto":
     case "ldapcc":
     case "ldapbcc":
     case "ldaptoccbcc":
      sessobj["to"]=TO_UNICODE(id->variables->to);
      sessobj["cc"]=TO_UNICODE(id->variables->cc);
      sessobj["bcc"]=TO_UNICODE(id->variables->bcc);
      sessobj["subject"]=TO_UNICODE(id->variables->subject);
      sessobj["message"]=TO_UNICODE(id->variables->message);
      sessobj->recipientfield = 0;
      sscanf(action, "ldap%s", sessobj->recipientfield);
      sessobj["status"] = LDAPSEARCH;
      break;

     case "searchldap":
      sessobj->ldapadd = getldapaddr(this_object (), TO_UNICODE(id->variables->namecont));
      sessobj["status"] = LDAPRESULT;
      break;    	

     case "searchldaptoccbcc":
      sessobj->ldapadd = getldapaddr(this_object (), TO_UNICODE(id->variables->namecont));
      sessobj["status"] = LDAPRESULT2;
      break;    	
#endif

     case "addressbook":
      sessobj->recipientfield = 0;
      sessobj["status"] = ADDRESSBOOK;
      break;

     case "addressbook2":
      sessobj["status"] = ADDRESSBOOK2;
      break;

     case "filterbook":
      sessobj->recipientfield = 0;
      sessobj["status"] = MAILFILTER;
      break;

     case "gotoimportaddress":
      sessobj["status"] = IMPORTADDRESS;
      break;

     case "importaddress":
      //FIXME: coding of files?
      if(sizeof(id->variables->file)) {
	string newbook = import_addressbook(id->variables->file);
	if (sizeof(newbook) > 0) {
	  array (string) addresses = sessobj->addressbook/"\n";
	  addresses += newbook / "\n";

	  //addresses[0] = ' '; //added to index addressbook correctly.

	  sessobj->addressbook = sort (Array.uniq (addresses)) * "\n";
	  sessobj["status"] = ADDRESSBOOK;
	  sessobj->imapclient->imap_command(IMHO, id,sessobj, ({ imap_cmd("save_user_prefs") }));
	  return http_pipe_in_progress();
	} else {
	  sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
	  sessobj["dialogactions"] = ({ "actionaddressbook" });
	  sessobj["dialogtext"] = MSG(M_CANNOTIMPORT);
	  sessobj["dialogtarget"] = id->misc->imho->frame; 
	  sessobj["status"] = DIALOGBOX;
	}
      }
      sessobj["status"] = ADDRESSBOOK;
      break;

     case "editaddress":
      if (id->variables->address) {
	sessobj->editaddress = ((int) id->variables->address);
	sessobj["status"] = EDITADDRESS;
      }
      sessobj->editaddressmode = "old";
      break;

     case "editaddressfilter":
      if (id->variables->namefield) {
        sessobj->editaddressfilter = ((int) id->variables->namefield);
        sessobj["status"] = EDITADDRESSFILTER;
      }
      sessobj->editaddressfiltermode = "old";
      break;

     case "newaddress":
      string name="", address="";
      if (id->variables->address)
	address = fix_coding(id->variables->address);
      if (id->variables->name)
	name = id->variables->name;
      if (id->variables->take)
	sessobj->editaddressmode = "take";
      else
	sessobj->editaddressmode = "new";
      if (sizeof(sessobj->addressbook) > 0)
	sessobj->addressbook += "\n"+replace(name,":","")+":"+address;
      else
	sessobj->addressbook = replace(name,":","")+":"+address;
      sessobj["status"] = EDITADDRESS;
      sessobj->editaddress = sizeof(sessobj->addressbook/"\n")-1;
      break;

     case "newaddressfilter":
      string namefield="", filterexpression="", filterfolder="";
      if (id->variables->namefield)
	namefield = fix_coding(id->variables->namefield);
      if (id->variables->filterfolder)
	filterfolder = fix_coding(id->variables->filterfolder);
      if (id->variables->filterexpression)
	filterexpression = id->variables->filterexpression;
      if (id->variables->take)
	sessobj->editaddressfiltermode = "take"; //oliv3: never used ?
      else
	sessobj->editaddressfiltermode = "new";
      sessobj->newaddressfilter = replace (filterexpression, ":", "") + ":" + filterexpression;
      sessobj["status"] = EDITADDRESSFILTER;
      break;

     case "editaddressdone":
      if (id->variables->address && id->variables->name) {
	array (string) addresses = sessobj->addressbook/"\n";
	addresses[sessobj->editaddress] = replace(TO_UNICODE(id->variables->name),":", "")+":"+TO_UNICODE(id->variables->address); 
	  sessobj->addressbook = sort(addresses)*"\n";
	  if (sessobj->editaddressmode == "take")
	    sessobj["status"] = READMAIL;
	  else
	    sessobj["status"] = ADDRESSBOOK;
	  if (feature (FEAT_EXTENDEDABOOK) && sessobj["extendedabook"]) {
	    string lval;
	    array(string) fn = sessobj["extendedabook"]->get_fields();
	    foreach(indices(fn), int i) {
	      lval = id->variables[sprintf("ebook%d",i)];
	    if (lval) {
	      sessobj["extendedabook"]->set_entry(id->variables->name, fn[i],
						    TO_UNICODE(lval));
	    }

	      if (sessobj->debug)
		perror(sprintf("Handling ebook field %s: %s\n", fn[i], lval||"NONE"));
	    }
	  }
      }
      sessobj->imapclient->imap_command(IMHO, id,sessobj, ({ imap_cmd("save_user_prefs") }));
      return http_pipe_in_progress();
      break;

     case "editaddressfilterdone":
      if (id->variables->namefield && id->variables->filterfolder && id->variables->filterexpression) {
	array (string) filters = (sizeof(sessobj->filterbook)>0)?(sessobj->filterbook/"\n"):({});
	string namefield = replace(TO_UNICODE(id->variables->namefield),":", "");
	string filterexpression = replace(TO_UNICODE(id->variables->filterexpression),":", "");
	string filterfolder = TO_UNICODE(id->variables->filterfolder);
	while (sizeof(namefield)>0 && namefield[0]==' ') namefield = namefield[1..];
	while (sizeof(filterexpression)>0 && filterexpression[0]==' ') filterexpression = filterexpression[1..];
	while (sizeof(filterfolder)>0 && filterfolder[0]==' ') filterfolder = filterfolder[1..];
	if ((sizeof(namefield)>0) && (sizeof(filterexpression)>0) && sizeof(filterfolder)>0) {
	  if (sessobj->editaddressfiltermode=="old")
	    filters[sessobj->editaddressfilter] = namefield+":"+filterexpression+":"+filterfolder;
	  else
	    if (sizeof(filters) > 0)
	      filters = ({ namefield+":"+filterexpression+":"+filterfolder }) + filters;
	    else
	      filters = ({ namefield+":"+filterexpression+":"+filterfolder });
	  sessobj->filterbook = sort(filters)*"\n";
	  if (sessobj->editaddressfiltermode == "take")
	    sessobj["status"] = READMAIL;
	  else
	    sessobj["status"] = MAILFILTER;
	  sessobj->filters_have_changed = 1;
	  sessobj->imapclient->imap_command(IMHO, id,sessobj, ({ imap_cmd("save_user_prefs") }),"");
	  return http_pipe_in_progress();
	} else {
          sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
          sessobj["dialogactions"] = ({ "actioncontinueeditfilter" });
          sessobj["dialogtext"] = MSG(M_EMPTYADDRESSFILTER);
          sessobj["status"] = DIALOGBOX;
	}
      }
      break;

     case "deleteaddress": 
      array (string) addresses = sessobj->addressbook/"\n";
      if (feature (FEAT_EXTENDEDABOOK) && QUERY (deleteinebook)) {
	string badguy = (addresses[sessobj->editaddress] / ":")[0];
	sessobj->extendedabook->del_entry(badguy);
      }
      addresses -= ({ addresses[sessobj->editaddress] });
      sessobj->addressbook = addresses * "\n";
      sessobj->status = ADDRESSBOOK;

      sessobj->imapclient->imap_command(IMHO, id,sessobj, ({ imap_cmd("save_user_prefs") }));
      return http_pipe_in_progress();
      break;

     case "canceleditaddress":
      if (sessobj->editaddressmode != "old") {
	array (string) addresses = sessobj->addressbook/"\n";
	addresses -= ({ addresses[-1] });
	sessobj->addressbook = addresses*"\n";
      }
      if (sessobj->editaddressmode == "take")
	sessobj["status"] = READMAIL;
      else
	sessobj["status"] = ADDRESSBOOK;
      break;

    case "searchmail":
      if (id->variables->text1 && id->variables->searchfield1) {
	if (sessobj->searchstring && sizeof(sessobj->searchstring) > 0)
	  sessobj->searchstring+= " "+MSG(M_SEARCHAND)+" ";
	else
	  sessobj->searchstring = "";
	sessobj->searchstring += "\""+TO_UNICODE(id->variables->text1)+"\" ";
	switch(id->variables->searchfield1) {
	case "text": sessobj->searchstring += MSG(M_SEARCHANYWHERE);
	  break;
	case "to": sessobj->searchstring += MSG(M_SEARCHTOFIELD);
	  break;
	case "from": sessobj->searchstring += MSG(M_SEARCHFROMFIELD);
	  break;
	case "subject": sessobj->searchstring += MSG(M_SEARCHSUBJECT);    
	  break;
	case "body": sessobj->searchstring += MSG(M_SEARCHBODY);
	  break;
	}
	sessobj["status"] = MAILINDEX;
	sessobj->imapclient->imap_command(IMHO, id,sessobj, ({ imap_cmd("search", "searchtext", TO_UNICODE(id->variables->text1), "searchfield", id->variables->searchfield1, "headers", imap_cmd_var("mails")) }));
	return http_pipe_in_progress();
      } else
	sessobj["status"] = SEARCHMAIL;
      break;

     case "addrecipient":
      if (id->variables->address) {
	if (sessobj->recipientfield == "to" ||
	    sessobj->recipientfield == "cc" ||
	    sessobj->recipientfield == "bcc") {
	  if (sizeof(sessobj[sessobj->recipientfield]) > 0)
	    sessobj[sessobj->recipientfield] += ", "+TO_UNICODE(id->variables->address);
	  else
	    sessobj[sessobj->recipientfield] = TO_UNICODE(id->variables->address);
	}
      }
      sessobj["status"] = COMPOSE;      
      break;

     case "addrecipients":
      foreach (indices (id->variables), string k) {
	int no = 0;
	if (sscanf (k, "recipient_to_%d", no) == 1) { // found a "to" recipient
	  string key2 = "address_" + no; // the address key
	  if (sizeof (sessobj["to"]) > 0)
	    sessobj["to"] += ", "+TO_UNICODE(id->variables[key2]);
	  else
	    sessobj["to"] = TO_UNICODE(id->variables[key2]);
	}
	if (sscanf (k, "recipient_cc_%d", no) == 1) { // found a "cc" recipient
	  string key2 = "address_" + no; // the address key
	  if (sizeof (sessobj["cc"]) > 0)
	    sessobj["cc"] += ", "+TO_UNICODE(id->variables[key2]);
	  else
	    sessobj["cc"] = TO_UNICODE(id->variables[key2]);
	}
	if (sscanf (k, "recipient_bcc_%d", no) == 1) { // found a "bcc" recipient
	  string key2 = "address_" + no; // the address key
	  if (sizeof (sessobj["bcc"]) > 0)
	    sessobj["bcc"] += ", "+TO_UNICODE(id->variables[key2]);
	  else
	    sessobj["bcc"] = TO_UNICODE(id->variables[key2]);
	}
      }
      sessobj["status"] = COMPOSE;      
      break;

     case "deleteaddressfilter":
      array (string) filters = sessobj->filterbook/"\n";
      filters -= ({ filters[sessobj->editaddressfilter] });
      sessobj->filterbook = filters*"\n";
      if (sessobj->editaddressfiltermode == "take")
        sessobj["status"] = READMAIL;
      else
        sessobj["status"] = MAILFILTER;
      sessobj->imapclient->imap_command(IMHO, id,sessobj, ({ imap_cmd("save_user_prefs") }),"");
      return http_pipe_in_progress();
      break;

     case "canceleditaddressfilter":
      if (sessobj->editaddressfiltermode != "old") {
        sessobj->newaddressfilter = "";
      }
      if (sessobj->editaddressfiltermode == "take")
        sessobj["status"] = READMAIL;
      else
        sessobj["status"] = MAILFILTER;
      break;

     case "createfolder":
      sessobj["status"] = FOLDERLIST;
      if (id->variables->foldername && (sizeof (id->variables->foldername) > 0)
	  && (QUERY (foldernamemaxlength) ? (strlen (id->variables->foldername) <= QUERY (foldernamemaxlength)) : 1)
	  && ((id->variables->foldername = TO_UNICODE (id->variables->foldername)) != MSG(M_NEWMBOXNAME))) {

	string path=id->variables->path || "";
	int mbox_idx=-1;
	string newmbox="";
	int ok=0;
	if(path!="")
	  mbox_idx=Array.search_array(sessobj->mailboxes,
				      lambda(array a,string path) 
				      {
					if(a[MB_FOLDERNAME_IDX] == path)
					  return 1;
					return 0;
				      },
				      path); 
	if(path=="") {
	  newmbox=sessobj->mailpath+id->variables->foldername;
	  ok=1;
	}
	else
	  if(mbox_idx!=-1) {
	    newmbox=sessobj->mailpath+sessobj->mailboxes[mbox_idx][MB_DISPLAYNAME_IDX]+sessobj->mailboxes[mbox_idx][MB_SEPARATOR_IDX]+id->variables->foldername;
	    ok=1;
	  }
	if(ok) {
	  if(sessobj->mboxencode)
	    newmbox=encode_mboxname(newmbox);
	  array imap_commands = ({ imap_cmd("create", "newmailbox", newmbox, "error", MSG(M_CREATEMBOXERROR)),
				   imap_cmd("low_list", "path", sessobj->mailpath) });
	  sessobj->imapclient->imap_command(IMHO, id, sessobj, imap_commands);
	  return http_pipe_in_progress();
	}
      } else {
	sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
	sessobj["dialogactions"] = ({ "actionfolderlist" });
	sessobj["dialogtext"] = MSG(M_NEWMBOXNONAME);
	sessobj["dialogtarget"] = id->misc->imho->frame; 
	sessobj["status"] = DIALOGBOX;
      }
      break;

     case "deletefolder":
      array (string) mboxes = ({});
      string mboxtext = "";
      foreach(indices(id->variables),string var) {
	if(id->variables[var] == "1" && sscanf(var,"delf_%s", string mbox)) {
	  mboxes += ({ mbox });
	  mboxtext += mb_displayname_from_name(mbox,sessobj)+"\n";
	}
      }
      if (sizeof(mboxes) > 0) {
	sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK), 
				      MSG(M_DIALOGCANCEL) });
	sessobj->deletemboxes = mboxes;
	sessobj["dialogactions"] = ({ "actiondeletefoldersure", 
				      "actionfolderlist" });
	sessobj["dialogtext"] = MSG(M_MBOXREMOVEP)+"\n"+mboxtext;
      } else {
	sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
	sessobj["dialogactions"] = ({ "actionfolderlist" });
	sessobj["dialogtext"] = MSG(M_MBOXMARKONE);
      }
      sessobj["dialogtarget"] = id->misc->imho->frame; 
      sessobj["status"] = DIALOGBOX;
      break;

     case "deletefoldersure":
      sessobj["status"] = FOLDERLIST;
      array imap_commands = ({ imap_cmd("delete_folders", "folders", imap_cmd_var("deletemboxes")),imap_cmd("low_list", "path", sessobj->mailpath) });
      sessobj->imapclient->imap_command(IMHO, id, sessobj, imap_commands);
      return http_pipe_in_progress();
      break;

     case "folderlist":
      sessobj["status"] = FOLDERLIST;
      imap_commands = ({ imap_cmd("low_list", "path", sessobj->mailpath) });
      if (QUERY (foldersinfo))
	imap_commands += ({ imap_cmd("scan_folders") });
      sessobj->imapclient->imap_command (IMHO, id, sessobj, imap_commands);
      return http_pipe_in_progress();
      break;

     case "folderlistshow":
      sessobj["status"] = FOLDERLIST;
      break;

     case "cancel":
      if(sessobj["replytoidx"] == -1)
	sessobj["status"] = MAILINDEX;
      else
	sessobj["status"] = READMAIL;
      sessobj->attachments = ({ });
      break;

     case "checkword":
      sessobj["checkword"] = (int)id->variables->word;
      break;

     case "spellprev":
      sessobj["checkword"]--;
      break;

     case "spellnext":
      sessobj["checkword"]++;
      break;

     case "spellreplace":
      (sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][1]=TO_UNICODE(id->variables->newword);
      sessobj["checkword"]++;
      break;

     case "spellselect":
      if( (((int)id->variables->selectedword) >= 0) && (((int)id->variables->selectedword) < sizeof((sessobj->spelling[(sessobj->misspelled)[sessobj->checkword]]))-2)) {
	(sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][1]=(sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][(int)id->variables->selectedword + 2];
      }
      sessobj["checkword"]++;
      break;

     case "spelldone":
      sessobj->message="";
      foreach(sessobj->spelling,mixed foo) {
	if(arrayp(foo))
	  sessobj->message+=foo[1];
	else
	  sessobj->message+=foo;
      }
      sessobj["status"] = COMPOSE;
      break;

     case "spellcheck2":
      id->variables->ispelldict=id->variables->ispelldict2;
     case "spellcheck":
      if(sizeof(QUERY (ispell))) {
	sessobj["to"]=TO_UNICODE(id->variables->to);
	sessobj["cc"]=TO_UNICODE(id->variables->cc);
	sessobj["bcc"]=TO_UNICODE(id->variables->bcc);
	sessobj["subject"]=TO_UNICODE(id->variables->subject);
	sessobj["message"]=TO_UNICODE(id->variables->message);
	sessobj["ispelldict"]=id->variables->ispelldict;
	spellcheck(sessobj);
	if(sizeof(sessobj->spelling))
	  sessobj["status"] = SPELLCHECK;
      }
      break;

     case "send":
      array fixedaddr;
      string brokenaddr = "";
      if (id->variables->to) {
	fixedaddr = fixaddresses(TO_UNICODE(id->variables->to));
	sessobj["to"]=fixedaddr[0];
	if (sizeof(brokenaddr) == 0) 
	  brokenaddr = fixedaddr[1];
      }
      if (id->variables->cc) {
	fixedaddr = fixaddresses(TO_UNICODE(id->variables->cc));
	sessobj["cc"]=fixedaddr[0];
	if (sizeof(brokenaddr) == 0) 
	  brokenaddr = fixedaddr[1];
      }
      if (id->variables->bcc) {
	fixedaddr = fixaddresses(TO_UNICODE(id->variables->bcc));
	sessobj["bcc"]=fixedaddr[0];
	if (sizeof(brokenaddr) == 0) 
	  brokenaddr = fixedaddr[1];
      }
      if (id->variables->subject)
      sessobj["subject"]=TO_UNICODE(id->variables->subject);
      if (id->variables->message)
      sessobj["message"]=TO_UNICODE(id->variables->message);
      if (sizeof(QUERY (sitesig))) {
	sessobj["message"] += "\n" + QUERY (sitesig);
      }
      if (sizeof(sessobj->to) >= 1 && sizeof(brokenaddr) == 0) {
	if (id->variables->dsnsuccess)
	  sessobj->dsnsuccess=((int) id->variables->dsnsuccess);
	else
	  sessobj->dsnsuccess=0;
	if (id->variables->dsndelay)
	  sessobj->dsndelay=((int) id->variables->dsndelay);
	else
	  sessobj->dsndelay=0;
	if (id->variables->mdn)
	  sessobj->mdn=((int) id->variables->mdn);
	else
	  sessobj->mdn=0;

	sessobj->selected = ({ });
	sendmail(sessobj,1);
	imho_log("sendmail", ([ "from":sessobj->address,
				"to":sessobj->to, "cc":sessobj->cc,
				"bcc":sessobj->bcc, 
				"size":sizeof(sessobj->sentmaildata) ]));
	array imap_commands=({ });
	if (QUERY (sendmethod) == "SMTP")
	  {
	    void|string error = smtp_send (IMHO, sessobj);

	    if (error)
	      {
		sessobj->status = DIALOGBOX;
		sessobj->dialogtext = MSG(M_SMTPERROR);
		sessobj->dialogactions = ({ "actioncontinuecompose" });
		sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
		if (sessobj->displaysmtperrors) 
		  sessobj->dialogtext += "\n" + error;
		imho_log ("smtpfail", ([ "login": sessobj->address, 
					 "command": "smtp_send",
					 "line": error ]));
	      }
	  }
	else
	  sessobj->attachments = ({ }); // Remove all attachments from mem if already sent

	if (sessobj->status != DIALOGBOX) {
	  if (feature (FEAT_MAILBOXES) && 
	      ((feature (FEAT_SAVEMAILCOPY) && id->variables->nosavemail == 0) ||
	       (!feature (FEAT_SAVEMAILCOPY) && !(id->variables->dosavemail==0))) &&
	      sizeof(sessobj->sentfolder)) {
	    string appendbox=sessobj["mailpath"]+sessobj->sentfolder;
	    if(sessobj->mboxencode)
	      appendbox=encode_mboxname(appendbox);
	    
	    if (QUERY (buggyimap)) {
	      write ("buggyimap => creating: " + appendbox + "\n");
	      imap_commands += ({ imap_cmd ("create", "newmailbox", appendbox) });
	    }
	    imap_commands+= ({imap_cmd("low_append","tobox",appendbox,"data",sessobj->sendmail,"error",MSG(M_SENDSAVEFAILED)) });
	    //m_delete(sessobj,"sendmail");
	  }
	  
	  if(sessobj["replytoidx"]!=-1) {
	    sessobj->mails[sessobj["replytoidx"]]->imap->FLAGS+=({ "\\Answered" });
	    imap_commands+=({ imap_cmd("add_flag","uid",sessobj["mails"][sessobj->replytoidx]->imap->UID,"flag","\\Answered" )});
	  }
	  
	  if(sizeof(imap_commands))
	    sessobj->imapclient->imap_command(IMHO, id,sessobj,imap_commands);
	  
	  if (QUERY (notifysent)) {
	    if(sessobj["replytoidx"] == -1)
	      sessobj["dialogactions"] = ({ "actionmailindex" });
	    else {
	      sessobj["dialogactions"] = ({ "actionread" });
	      sessobj["dialogoptions"] = ({ 
		"msguid=" + (string)sessobj->cmailuid,
		"mbox=" + http_encode_url(sessobj->mailbox[MB_FOLDERNAME_IDX])
	      });
	    }
	    sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
	    //oliv3: [enhancement] use sessobj->recipients ?
	    // yes ... if this give us faster parsing - Xavier
	    string recipients = sessobj->to;
	    if (sessobj->cc != "")
	      recipients += ", " + sessobj->cc;
	    if (sessobj->bcc != "")
	      recipients += ", " + sessobj->bcc;
	    sessobj["dialogtext"] = MSGA (M_SMTPOK, ({ sessobj->subject, recipients }));
	    sessobj["dialogtarget"] = id->misc->imho->frame;
	    sessobj["status"] = DIALOGBOX;
	  }
	  else {
	    if (sessobj->replytoidx == -1)
	      sessobj->status = MAILINDEX;
	    else
	      sessobj->status = READMAIL;
	  }

	  if(sizeof(imap_commands))
	    return http_pipe_in_progress();
	}
      }
      else {
	sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
	sessobj["dialogactions"] = ({ "actioncontinuecompose" });
	if (sizeof(sessobj->to) == 0) 
	  sessobj["dialogtext"] = MSG(M_SENDNORECV);
	else
	  sessobj["dialogtext"] = MSGA(M_SENDBROKENADDR, ({ fix_header(brokenaddr) }));
	sessobj["status"] = DIALOGBOX;
      }
      break;

     case "savedraft":
      if (id->variables->to)
	sessobj["to"]=TO_UNICODE(id->variables->to);
      if (id->variables->cc)
	sessobj["cc"]=TO_UNICODE(id->variables->cc);
      if (id->variables->bcc)
	sessobj["bcc"]=TO_UNICODE(id->variables->bcc);
      if (id->variables->subject)
	sessobj["subject"]=TO_UNICODE(id->variables->subject);
      if (id->variables->message)
	sessobj["message"]=TO_UNICODE(id->variables->message);

      sessobj["status"] = MAILINDEX;

      sessobj->selected = ({ });
      sendmail(sessobj,1);
      sessobj->attachments = ({ }); // Remove all attachments from mem
      imap_commands=({ });
      sessobj->sentmaildata = 0;
      imap_commands += ({ 
	imap_cmd ("low_append",
		  "tobox", (QUERY (defaultmailpath) + sessobj->draftsfolder),
		  "data", sessobj->sendmail),
      });
      m_delete (sessobj, "sendmail");

      if ((sessobj->mailbox[MB_FOLDERNAME_IDX] == (QUERY (defaultmailpath)+sessobj->draftsfolder)) && sessobj->draftuid) {
	//write ("oliv3: delete the old draft..."+sessobj->draftuid+"\n");
	imap_commands += ({
	  imap_cmd ("delete",
		    "mailbox", (QUERY (defaultmailpath) + sessobj->draftsfolder),
		    "uids", ({ sessobj->draftuid })),
	  imap_cmd ("status","mailbox",sessobj->mailbox[MB_FOLDERNAME_IDX]),
	  imap_cmd ("get_headers",
		    "output", imap_cmd_var("mails"),
		    "updatelistpos",1,
		    "setsortcolumn",imap_cmd_var("mailssortcolumn"))
	});
      }

      if(sizeof(imap_commands)) {
	sessobj->imapclient->imap_command(IMHO, id,sessobj,imap_commands);
	return http_pipe_in_progress();
      }
      break;

     case "deleteall":
      array uids = Array.map (sessobj->mails, lambda (mapping m) { return m->imap->UID; });
      if (sizeof (uids)) {
	array cmds = ({ });
	for (int i = 0; i < sizeof (uids); ) {
	  array uids2 = ({ });
	  for (int k = 0; (k < 100) && (i < sizeof (uids)); k++)
	    uids2 += ({ uids[i++] });
	  cmds += ({ imap_cmd ("delete", "uids", uids2 + ({ }), "updatembox", imap_cmd_var ("mails")) });
	}
	cmds += ({ imap_cmd ("get_headers", "output", imap_cmd_var("mails"),
			     "updatelistpos", 1,
			     "setsortcolumn", imap_cmd_var ("mailssortcolumn")),
	imap_cmd ("low_close"),	   imap_cmd ("status", "mailbox", "INBOX") /*imap_cmd ("low_close")*/ });
	sessobj->imapclient->imap_command(IMHO, id, sessobj, cmds);
	return http_pipe_in_progress();
      }
      break;

     case "delete":
       if (sizeof (sessobj->selected))
	 {
	   sessobj->imapclient->imap_command(IMHO, id,sessobj,({ imap_cmd("delete","uids",sessobj->selected,"updatembox",imap_cmd_var("mails")) }));
	   return http_pipe_in_progress();
	 }
      break;

     case "deleteask":
      if (sizeof (sessobj->selected))
	{
	  sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK), MSG(M_DIALOGCANCEL) });
	  sessobj->deletemails = sessobj->selected;
	  sessobj["dialogactions"] = ({ "actiondeletesure", "actionindex" });
	  sessobj["dialogtext"] = MSGA(M_DELETEMARKEDP, ({ sizeof(sessobj->selected) }));
	}
      else
	{
	  sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
	  sessobj["dialogactions"] = ({ "actionindex" });
	  sessobj["dialogtext"] = MSG(M_DELETEMARKEDNONE);
	}
      sessobj["dialogtarget"] = id->misc->imho->frame; 
      sessobj["status"] = DIALOGBOX;
      break;

     case "deletesure":
      //FIXME: move from sessobj to url
      sessobj["status"] = MAILINDEX;
      sessobj->imapclient->imap_command(IMHO, id,sessobj,({ imap_cmd("delete", "uids",sessobj->deletemails+({ }),"updatembox",imap_cmd_var("mails")) }));
      return http_pipe_in_progress();
      break;

     case "trash":
      if (sizeof (sessobj->selected) > 0) {
	sessobj->copytobox = sessobj->mailpath + sessobj->trashfolder;
	if (sessobj->mboxencode)
	  sessobj->copytobox = encode_mboxname (sessobj->copytobox);
	sessobj->imapclient->imap_command (IMHO, id, sessobj,
					   ({ imap_cmd("move","uids",sessobj->selected+({ }),"updatembox",imap_cmd_var("mails")) }));
	return http_pipe_in_progress();
      }
      sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
      sessobj["dialogactions"] = ({ "actionindex" });
      sessobj["dialogtarget"] = id->misc->imho->frame; 
      sessobj["dialogtext"] = MSG(M_DELETEMARKEDNONE);
      sessobj["status"] = DIALOGBOX;
      break;


      //oliv3 FIXME what is this move/move2 thing ?
      // and mbox/mbox2 ?
      // idem for ispell2...
      // stenad ANSWER: see id->misc->imho->nobottombuttons
      //  old config-flag?!

      string mboxtomoveto=0;
      /*
     case "move2":
      if(id->variables->mbox2 && id->variables->mbox2!="imhonomailbox")
	mboxtomoveto=id->variables->mbox2;
      */
     case "move":
      if(!mboxtomoveto && id->variables->mbox && id->variables->mbox!="imhonomailbox")
	mboxtomoveto=id->variables->mbox;
      if(mboxtomoveto && sizeof(sessobj->selected)) {
	sessobj["copytobox"]=mboxtomoveto;
	sessobj->imapclient->imap_command(IMHO, id,sessobj,({ imap_cmd("move","uids",sessobj->selected+({ }),"updatembox",imap_cmd_var("mails") ) }));
	return http_pipe_in_progress();
      }
      break;

     case "reload":
      //FIXME: use close?
      sessobj->searchstring = "";
      sessobj->imapclient->imap_command(IMHO, id,sessobj, ({
	imap_cmd("status","mailbox",sessobj->mailbox[MB_FOLDERNAME_IDX]),
	imap_cmd("get_headers","output",imap_cmd_var("mails"),"updatelistpos",1,"setsortcolumn",imap_cmd_var("mailssortcolumn")) }));
      return http_pipe_in_progress();
      break;

    case "checkactivemailboxes": // Just check, don't reload
      sessobj->imapclient->imap_command (IMHO, id, sessobj, 
					 ({ imap_cmd ("check_mailboxes"/*, "noselect", 1*/) }));
      return http_pipe_in_progress();
      break;

     case "reloadactivemailboxes":
       sessobj->searchstring = "";
       sessobj->imapclient->imap_command(IMHO, id,sessobj, ({
	 imap_cmd ("check_mailboxes"),
	 imap_cmd ("status", "mailbox", sessobj->mailbox[MB_FOLDERNAME_IDX]),
	 imap_cmd ("get_headers", "output", imap_cmd_var ("mails"),
		   "updatelistpos", 1, "setsortcolumn", imap_cmd_var ("mailssortcolumn")),
	 imap_cmd ("apply_mail_filters", "updatembox", imap_cmd_var ("mails")) }));
       return http_pipe_in_progress();
       break;

     case "togglesortorder":
      if (!sessobj->sessionsortorder)
	sessobj->sessionsortorder = sessobj->sortorder;
      if (sessobj->sessionsortorder == "forward")
	sessobj->sessionsortorder = "backward";
      else
	sessobj->sessionsortorder = "forward";
      break;

    case "changesort":
      if (id->variables->col) {
 	sessobj->sortcolumn = id->variables->col;
      }
      break;

     case "trashthis":
      sessobj["copytobox"]=sessobj->mailpath+sessobj->trashfolder;
      if(sessobj->mboxencode)
	sessobj->copytobox=encode_mboxname(sessobj->copytobox);
      int prevuid=(int)id->variables->prevuid || -1;
      int nextuid=(int)id->variables->nextuid || -1;
      int uid=(nextuid==-1?prevuid:nextuid);
      sessobj->cmailidx = find_mail(sessobj->mails, uid);
      sessobj->cmailuid = uid;
      if (uid ==-1) {
	sessobj->status = MAILINDEX;
	sessobj->imapclient->imap_command(IMHO, id,sessobj,({ imap_cmd("move","uids",sessobj->selected+({ }),"updatembox",imap_cmd_var("mails") ) }));
      } else {
	sessobj->imapclient->imap_command(IMHO, id,sessobj,({ imap_cmd("move","uids",sessobj->selected+({ }),"updatembox",imap_cmd_var("mails") ), imap_cmd("get_mail","uid",uid,"output",imap_cmd_var("cmail")), imap_cmd ("check")  }));
      }
      return http_pipe_in_progress();
      break;

     case "deletethis":
      prevuid=(int)id->variables->prevuid || -1;
      nextuid=(int)id->variables->nextuid || -1;
      uid=(nextuid==-1?prevuid:nextuid);
      sessobj->cmailidx = find_mail(sessobj->mails, uid);
      sessobj->cmailuid = uid;
      if (uid == -1) {
	sessobj->status = MAILINDEX;
	sessobj->imapclient->imap_command(IMHO, id,sessobj,({ imap_cmd("delete","uids",sessobj->selected+({ }),"updatembox",imap_cmd_var("mails") ) }));
      } else {
	sessobj->imapclient->imap_command(IMHO, id,sessobj,({ imap_cmd("delete","uids",sessobj->selected+({ }),"updatembox",imap_cmd_var("mails") ), imap_cmd("get_mail","uid",uid,"output",imap_cmd_var("cmail")), imap_cmd ("check")  }));
      }
      return http_pipe_in_progress();
      break;

     case "readprev":
     case "readnext":
     case "read":
      sessobj["status"] = READMAIL;

      switch(action) {
       case "readprev":
	uid=(int)id->variables->prevuid;
	break;
       case "readnext":
	uid=(int)id->variables->nextuid;
	break;
       default:
	uid=(int)id->variables->msguid;
      }
      //write ("find_mail (" + uid + ")\n");
      sessobj->cmailidx = find_mail(sessobj->mails, uid);
      sessobj->cmailuid = uid;
      //write ("cmailidx= " + sessobj->cmailidx + "\n");
      //write ("cmailuid= " + sessobj->cmailuid + "\n");
      if (sessobj->cmailidx != -1) {
	sessobj->imapclient->imap_command(IMHO, id,sessobj, ({ 
	  imap_cmd("get_mail","mailbox",id->variables->mbox||0,"uid",uid,"output",imap_cmd_var("cmail")), imap_cmd ("check")  }));
	return http_pipe_in_progress();
      }
      else
	sessobj->status = MAILINDEX;
      break;

     case "showheaders":
      sessobj->showheaders = 1;
      break;

     case "hideheaders":
      sessobj->showheaders = 0;
      break;

     case "reply":
     case "replytoall":
      object mail=sessobj["cmail"];

      if(mail->headers["reply-to"] && sizeof(mail->headers["reply-to"]))
	sessobj["to"] = fixaddresses(fix_coding(mail->headers["reply-to"]))[0];
      else
	sessobj["to"] = fixaddresses(fix_coding(mail->headers->from))[0];

      sessobj["subject"] = fix_coding(mail->headers->subject);

      if (lower_case(sessobj["subject"][0..2]) != "re:")
	sessobj["subject"] = "Re: "+sessobj["subject"];
      if (mail->headers["message-id"])
	sessobj["message-id"] = mail->headers["message-id"];

      sessobj["message"] = "";
      if (sessobj["replyincludemsg"] == "1") {
	array(object) msgs = ({ mail });
	while(sizeof(msgs) > 0) {
	  object mess = msgs[0];
	  msgs = msgs[1..];
	  if (mess->body_parts)
	    msgs = mess->body_parts + msgs;
	  else
	    if(mess->type == "text" && mess->subtype != "html") {
	      mixed err=0;
	      string msg;
	      err=catch {msg=Locale.Charset.decoder(mess->charset)->feed(mess->getdata()||"")->drain();};
	      if(err)
		msg=mess->getdata()||"";
	      sessobj["message"] = sessobj["replymsgprefix"]+replace(msg,"\n","\n"+sessobj["replymsgprefix"]);
	      sessobj["replytocharset"]=err?"iso-8859-1":mess->charset;
	      msgs = ({ });
	    }
	}
	sessobj["message"] = QUERY (replyseparator) + "\n" + sessobj["message"]; 
      }

      sessobj["message"] += "\n" + sessobj["signature"];
      sessobj->cc = "";

      if(action=="replytoall") {
	array (string) ccs = address_split(mail->headers->to) + address_split((mail->headers->cc||"")) - ({ "" }); // Add all in "To:" and "CC:" as "CC:" 
	// recipients, except ourselves
	foreach (indices(ccs), int i) {
	  sscanf(ccs[i],"%*[ ]%s",ccs[i]);
	  string ad = ccs[i];
	  sscanf(ccs[i], "%*s<%s>", ad);
	  if (lower_case(ad) != lower_case(sessobj->address)) {
	    if (sizeof(sessobj->cc))
	      sessobj->cc += ", ";
	    sessobj->cc += ccs[i];
	  }
	}
      }
      sessobj["cc"] = fix_coding(sessobj->cc);
      sessobj["bcc"] = sessobj->autobcc;
      sessobj["replytoidx"] = sessobj["cmailidx"];
      sessobj["status"] = COMPOSE;
      sessobj["attachments"] = ({ });
      break;

     case "composedraft":
      mail = sessobj["cmail"];
      sessobj["draftuid"] = sessobj["cmailuid"];
      // FIXME: add this in debug log if it is set to on
      //write ("Retaking draft uid= " + sessobj->draftuid + "\n");
      sessobj->to = mail->headers->to?fix_coding(mail->headers->to):"";
      sessobj->cc = mail->headers->cc?fix_coding(mail->headers->cc):"";
      sessobj->bcc = mail->headers->bcc?fix_coding(mail->headers->bcc):"";
      sessobj->replytoidx = -1;
      sessobj->status = COMPOSE;
      sessobj->attachments = ({ });
      sessobj->subject = fix_coding(mail->headers->subject);
      sessobj->message = "";
      
      if(mail->body_parts){
	int messageadded=0;
	foreach(mail->body_parts, object mess) {
	  if(!messageadded && 
	     mess->type == "text" && 
	     mess->subtype != "html") {
	    sessobj["message"] += mess->getdata()||"";
	    messageadded=1;
	  } else {
	    string name=mess->disp_params["filename"] || mess->params["name"] || "unknown";
            name = MIME.decode_words_text_remapped(name);
	    mapping file=([ ]);
	    file["fname"]=name;
	    file["data"]=mess->getdata()||"";
	    file["size"]=sizeof(file->data);
	    file["type"]=filetype(TO_UTF8(name));
	    sessobj->attachments += ({ file });
	  }
	}
      }
      else 
	sessobj["message"] += mail->getdata()||"";
      break;

     case "forward":
      mail = sessobj["cmail"];
      sessobj["message"]= sessobj["signature"];
      sessobj["message"] += "\n" + QUERY (forwardseparator) + "\n";
      sessobj["to"] = "";
      sessobj["cc"] = "";
      sessobj["bcc"] = sessobj->autobcc;
      sessobj["replytoidx"] = -1;
      sessobj["status"] = COMPOSE;
      sessobj["attachments"] = ({ });
      sessobj["subject"] = fix_coding(mail->headers->subject);
      if (sessobj["subject"][0..3] != "Fwd:")
	sessobj["subject"] = QUERY (fwdsubjpf)+" "+sessobj["subject"];
      sessobj["message"]+=QUERY (fwdfrom)+" "+fix_coding(mail->headers->from)+"\n";
      if(mail->headers["reply-to"])
	sessobj["message"]+=QUERY (fwdreplyto)+" "+ fix_coding(mail->headers["reply-to"]);
      sessobj["message"]+=QUERY (fwdto)+" "+fix_coding(mail->headers->to)+"\n";
      if(mail->headers->cc)
	sessobj["message"]+=QUERY (fwdcc)+" "+ fix_coding(mail->headers->cc )+"\n";
      sessobj["message"]+=QUERY (fwddate)+" "+ fix_coding(mail->headers->date || "");
      sessobj["message"]+="\n\n";

      
      if(mail->body_parts){
	int messageadded=0;
	foreach(mail->body_parts, object mess) {
	  if(!messageadded && 
	     mess->type == "text" && 
	     mess->subtype != "html") {
	    sessobj["message"] += mess->getdata()||"";
	    messageadded=1;
	  } else if(mess->type == "multipart" 
	  		&& mess->subtype == "alternative") {
		foreach( mess->body_parts, object altmess ) {
			if( altmess->type == "text" &&
				altmess->subtype == "plain" ) {
				sessobj["message"] += altmess->getdata()||"";
			} else {
				mapping file=([]);
				file["fname"]="BODY-alternative";
				file["data"]=altmess->getdata()||"";
				file["size"]=sizeof(file->data);
				file["type"]=( altmess->type && altmess->subtype
                                	?  altmess->type+"/"+altmess->subtype
                                	:  "application/octet-stream" );
				sessobj->attachments += ({ file });
                        }
                } /* foreach */

	  } else {
	    string name=mess->disp_params["filename"] || mess->params["name"] || "unknown";
	    name=MIME.decode_words_text_remapped(name);
	    mapping file=([ ]);
	    file["fname"]=name;
	    file["data"]=mess->getdata()||"";
	    file["size"]=sizeof(file->data);
	    file["type"]=filetype(TO_UTF8(name));
	    sessobj->attachments += ({ file });
	  }
	}
      }
      else 
	if( mail->type == "text" ) {
		sessobj["message"] += mail->getdata()||"";
	} else {
		string name=mail->get_filename() || mail->params["name"] || "unknown";
		name=MIME.decode_words_text_remapped(name);
		mapping file=([ ]);
		file["fname"]=name;
		file["data"]=mail->getdata()||"";
		file["size"]=sizeof(file->data);
		file["type"]=filetype(string_to_utf8(name));
		sessobj->attachments += ({ file });
	}
      break;


     case "index":
     case "mailindex":
      sessobj["status"] = MAILINDEX;

      int mbox_idx;
      if (id->variables->mbox && 
	  (mbox_idx=Array.search_array(sessobj->mailboxes,
				       lambda(array a,string mbox) 
				       {
					 if(a[MB_FOLDERNAME_IDX] == mbox)
					   return 1;
					 return 0;
				       },
				       id->variables->mbox)) != -1 ) {
	if (sessobj->mailbox[MB_FOLDERNAME_IDX] != id->variables->mbox) {
	  sessobj->mailbox = sessobj->mailboxes[mbox_idx];
	  if (sessobj->debug)
	    write ("\nchecking headers (" + sessobj->mailbox[MB_FOLDERNAME_IDX] + ")...\n");
	  sessobj->searchstring = "";
	  array imap_commands = ({ 
	      imap_cmd("status","mailbox",sessobj->mailbox[MB_FOLDERNAME_IDX]),
	      imap_cmd("get_headers","output",imap_cmd_var("mails"),"updatelistpos",1,"setsortcolumn",imap_cmd_var("mailssortcolumn")) });
	  if (sessobj->mailbox[MB_FOLDERNAME_IDX] == "INBOX")
	    imap_commands += ({ imap_cmd("apply_mail_filters", "updatembox", imap_cmd_var ("mails")) });
	  sessobj->imapclient->imap_command (IMHO, id, sessobj, imap_commands);
	  return http_pipe_in_progress();
	}
      }
      break;

     case "gotoblock":
      int from;
      if (id->variables->msg && (sscanf (id->variables->msg, "%d", from) == 1))
	sessobj->firstvisiblemail = from;
      sessobj->status = MAILINDEX;
      break;

     case "nextblock":
      sscanf(sessobj->visiblemail, "%d", sessobj->visible);
      sessobj->firstvisiblemail += sessobj->visible; /* MI handles ovfl */
      sessobj->status = MAILINDEX;
      break;

     case "prevblock":
      sscanf(sessobj->visiblemail, "%d", sessobj->visible);
      sessobj->firstvisiblemail -= sessobj->visible; /* MI handles underfl */
      sessobj->status = MAILINDEX;
      break;

     case "files":
      if(sizeof(QUERY (uploaddir)))
	sessobj["status"] = FILES;
      else
	sessobj->status = MAILINDEX;
      break;

     case "upload":
      if(id->variables->file && sizeof(id->variables->file)) {
	int totsize=0;
	int allowupload=1;
	foreach(sessobj["files"],mapping file){
	  totsize+=file->size;
	}
	if((int)QUERY (uploadquota)) {
	  if ((((((int)QUERY (uploadquota)+(int)QUERY (uploadsoftquota))*1024)-totsize-sizeof(id->variables->file))) <= 0)
	    allowupload=0;
	  if ((((((int)QUERY (uploadquota))*1024)-totsize)) <= 0)
	    allowupload=0;
	}

	if(allowupload) {
	  string fname=(id->variables->fixedfilename && sizeof(id->variables->fixedfilename))?id->variables->fixedfilename:id->variables["file.filename"];
	  fname=((((fname)/"/")[-1])/"\\")[-1];
	  string encfname=QUERY (uploaddir)+"/"+lower_case(sessobj["login"])+"_"+encode_fname(fname);
	  object file=open(encfname,"wct");
	  if(file) {
	    file->write(id->variables->file);
	    file->close();
	  
	    sessobj["files"]+= ({ ([ "fname" : TO_UNICODE(fname), "size" : sizeof(id->variables->file), "type" : filetype(fname) ]) });
	  }
	  else
	    report_warning("CAMAS: Could not write uploaded file: "+encfname+"\n");
	}
      }
      break;

     case "deletefiles":
      array (mapping) files_to_delete = ({ });
      foreach(sessobj->selectedfiles,int i)
	if(i < sizeof(sessobj->files))
	  files_to_delete += ({ sessobj->files[i] });
      foreach(files_to_delete,mapping file)
	rm(QUERY (uploaddir)+"/"+sessobj->login+"_"+encode_fname(file->fname));
      sessobj->files -= files_to_delete;
      break;
	

     case "setup":
      if(feature (FEAT_USEREDITSETUP))
	sessobj->status = SETUP;
      break;

     case "savesetup":
      foreach(indices(prefproperties), string prop)
	if (id->variables[prop] && sessobj["usersetup"+prop])
	  sessobj[prop] = strip(TO_UNICODE(id->variables[prop]));

      //oliv3 what is this used for ?
      // I think it's used for saving setup elsewhere than in imap mail -Xavier
      /*
      if (sessobj["mailpath"]){
	string mp = sessobj["mailpath"];
	sessobj["mailpath"] = mp;
      }
      */

      if(sessobj->language!="english" && !sessobj->lang_module->lang_progs[sessobj->language]) {
	// fuzzy select instead
	string lang=Array.filter(sessobj->language/"",lambda(string s) { return (s[0]>='a' && s[0]<='z');} )*"";
	string lang2=0;
	array langs=indices(sessobj->lang_module->lang_progs);
	foreach(langs, string l) {
	  if(Array.filter(l/"",lambda(string s) { return (s[0]>='a' && s[0]<='z');} )*"" == lang) {
	    lang2=l;
	    break;
	  }
	}
	if(lang2)
	  sessobj->language=lang2;
      }

      if(!feature (FEAT_USERMAILPATH))
	sessobj["mailpath"]=sessobj["defaultmailpath"];
      if(!feature (FEAT_USERADDRESS))
	sessobj["address"]=sessobj->defaultaddress;
      if (sessobj["signature"]) 
	sessobj["signature"] = sessobj["signature"];
      if (sessobj->sortorder)
	sessobj->sessionsortorder = sessobj->sortorder;
      if (sessobj->trashfolder)
	sessobj->trashfolder = strip_path(sessobj->trashfolder);
      if (sessobj->sentfolder)
	sessobj->sentfolder = strip_path(sessobj->sentfolder);
      if (sessobj->draftsfolder)
	sessobj->draftsfolder = strip_path(sessobj->draftsfolder);
      sessobj["status"] = MAILINDEX;
      sessobj->imapclient->imap_command (IMHO, id, sessobj, ({ imap_cmd ("save_user_prefs") }));
      return http_pipe_in_progress();
      break;

     case "sendmdn":
      mail = sessobj["cmail"];
      sessobj->to = fix_coding (mail->headers->from);
      sessobj->subject = MSG (M_MDNHEADER) + fix_coding(mail->headers->subject);
      if (mail->headers["message-id"])
	sessobj["message-id"] = mail->headers["message-id"];
      sessobj->message = MSGA (M_MDNMESSAGE, ({ fix_coding (mail->headers->to) }));
      sessobj->cc = "";
      sessobj->bcc = "";
      sessobj->attachments = ({ });
      sessobj->dsnsuccess = 0;
      sessobj->dsndelay = 0;
      sessobj->mdn = 0;
      sendmail(sessobj,1);

      void|string error = smtp_send (IMHO, sessobj);

      if (error)
	{
	  sessobj->status = DIALOGBOX;
	  sessobj->dialogtext = MSG(M_SMTPERROR);
	  sessobj->dialogactions = ({ "actioncontinuecompose" });
	  sessobj->dialogstrings = ({ MSG(M_DIALOGOK) });
	  if (sessobj->displaysmtperrors) 
	    sessobj->dialogtext += "\n" + error;
	  imho_log ("smtpfail", ([ "login": sessobj->address, 
				   "command": "smtp_send",
				   "line": error ]));
	}
      sessobj->status = READMAIL;
      break;

     case "renamefolder":
      sessobj["status"] = FOLDERLIST;
      if (id->variables->newfoldername && sizeof(id->variables->newfoldername)>0 &&
	  id->variables->oldfoldername && sizeof(id->variables->oldfoldername)>0 &&
	  ((id->variables->newfoldername=TO_UNICODE(id->variables->newfoldername)) != MSG(M_NEWMBOXNAME)) &&
          (id->variables->oldfoldername != "imhonomailbox")) {

	string path=id->variables->path || "";
	int mbox_idx=-1;
	int ok=0;
	if(path!="")
	  mbox_idx=Array.search_array(sessobj->mailboxes,
				      lambda(array a,string path) 
				      {
					if(a[MB_DISPLAYNAME_IDX] == path)
					  return 1;
					return 0;
				      },
				      path); 
	if(path=="") {
	  sessobj["newfoldername"]=sessobj->mailpath+id->variables->newfoldername;
	  ok=1;
	}
	else
	  if(mbox_idx!=-1) {
	    sessobj["newfoldername"]=sessobj->mailboxes[mbox_idx][MB_FOLDERNAME_IDX]+sessobj->mailboxes[mbox_idx][MB_SEPARATOR_IDX]+id->variables->newfoldername;
	    ok=1;
	  }

	if (ok) {
	  sessobj->oldfoldername=id->variables->oldfoldername;
	  if(sessobj->mboxencode)
	    sessobj->newfoldername=encode_mboxname(sessobj->newfoldername);
	  array imap_commands = ({ imap_cmd("rename_folder"),
				   imap_cmd("low_list", "path", sessobj->mailpath) });
	  sessobj->imapclient->imap_command(IMHO, id, sessobj, imap_commands, MSG(M_RENAMEMBOXERROR));
	  return http_pipe_in_progress();
	}
      }
      else {
	sessobj["dialogstrings"] = ({ MSG(M_DIALOGOK) });
	sessobj["dialogactions"] = ({ "actionfolderlist" });
	sessobj["dialogtext"] = MSG(M_NOMBOXERROR);
	sessobj["dialogtarget"] = id->misc->imho->frame; 
	sessobj["status"] = DIALOGBOX;
      }
      break;

    }
  }
  
  return create_output (id);
}

mixed create_output (object id) {
  string ret = "";
  mapping sessobj = id->misc->session_variables;
  //oliv3 FIXME Kludgy
  id->misc->imho->sessobj = sessobj;
  id->misc->imho->nexturl = server_url(id) + QUERY (location);

  string screen = "unknown";
  if (screennames[sessobj->status])
    screen = screennames[sessobj->status][0];
  ret+="<define name=\"imhoscreen\">"+screen+"</define>";
  if (screen == "compose") {
    // Checking if we reply or forward a mail instead of just composing it
    string action = "";
    foreach (indices(id->variables), string var )
    {
      string foo;
      if (sscanf(var, "action%s", foo))
        action = foo;
    }
    // if type=image in submit action
    if(sizeof(action)>2 && (action[sizeof(action)-2..]==".x" || action[sizeof(action)-2..]==".y"))
      action=action[..sizeof(action)-3];
    switch (action) 
    {
      case "reply":
      case "replytoall":
            action = "reply";
            break;
      case "forward":
            action = "forward";
            break;
      default: action = "compose";
    }
    ret += "<define name=\"composescreen\">"+action+"</define>";
    perror("Action : "+action+"\n");
    action = 0;
  }
  string state = "in";
  if (sessobj->status < 0)
    state = "login";
  if (sessobj->status == LOGOUT)
    state = "out";

  ret+="<define name=\"imhostate\">"+state+"</define>";
  ret+="<define name=\"imhouser\">"+sessobj->login+"</define>";
  ret+="<define name=\"imhoframe\">"+id->misc->imho->frame+"</define>";
  ret+="<define name=\"imhomailnotify\">"+(feature (FEAT_MAILNOTIFY)?"yes":"no")+"</define>";

  string newmail = "0";
  if (sessobj->cache && sizeof (sessobj->cache))
    foreach (indices (sessobj->cache), string mbox)
      if (sessobj->cache[mbox] && sizeof (sessobj->cache[mbox]->new_mails))
	newmail = "1";
  ret += "<define name=\"imhonewmail\">" + newmail + "</define>";

  mapping answer;
  if (sessobj->administrator)
    ret += handle_admin_interface (id, sessobj);
  else {
    object layout_manager = id->conf->get_provider ("camas_layout_manager");
    if (sessobj->overridelayout) {
      // FIXME: add this in debug log - Xavier
      //write ("overiding layout: " + sessobj->overridelayout + "\n");
       if (layout_manager->has_frameset (sessobj->overridelayout) && !(< "login", "out" >)[sessobj->status]  && (id->misc->imho->frame == "_top"))
        ret += layout_manager->get_frameset (sessobj->overridelayout);
       else
        ret += layout_manager->get_layout (sessobj->overridelayout, sessobj->status);
    } else {
      // FIXME: this to ... in debug log - Xavier
      //write ("layout: " + sessobj->layout + "\n");
      if (layout_manager->has_frameset (sessobj->layout) && !(< "login", "out" >)[sessobj->status] && (id->misc->imho->frame == "_top"))
       ret += layout_manager->get_frameset (sessobj->layout);
      else
       ret += layout_manager->get_layout (sessobj->layout, sessobj->status);
    }
  }
  ret+="\n<!-- Page generated by CAMAS - http://camas.caudium.net/ -->";

  int kludges = 0;
  string val;

  //        we should probably try to detect wich parse_rxml() we
  //        use since the XML-one uses Parser.HTML wich is
  //        widestring-safe (i think).
  //  Ok. This is fixed the best it can be. -- david
  
  //  Use kludge parsing when the old parser is used -- david
  if (id->misc->_xml_parser)
    val = Locale.Charset.encoder (sessobj->charset,
				  sessobj->lang_module->lang_replacement_strings[sessobj->language] || "?"
				  )->feed (parse_rxml (ret, id))->drain ();
  else
    val = Locale.Charset.encoder (sessobj->charset,
				  sessobj->lang_module->lang_replacement_strings[sessobj->language] || "?"
				  )->feed (imho_kludge_parse_rxml (ret, id))->drain ();

  array btags = ({ "<end_", "<imho_", "<camas_", "<define", "<comment", "<insert" });
  array bsearch = map (btags, lambda (string tag) { return search (val, tag); } );

  int broken = -1;
  foreach (bsearch, int pos)
    if (pos != -1) {
      if (broken == -1)
	broken = pos;
      else if (pos < broken)
	broken = pos;
    }
  
  if (broken == -1)
    ++global_ok;
  else
  while ((broken != -1) && (kludges < QUERY (rxmlparsercalls))) {
    ++kludges;
    ++global_kludges;
    if (QUERY (debug)) {
      write ("found an unparsed tag. kludges= " + kludges + "\n");
      write ("data= " + val[broken..broken+30] + " (...)\n");
    }
    string ok = val[0..broken-1];
    string new_answer;
    
    //  Use kludge parsing when the old parser is used -- david
    if(id->misc->_xml_parser)
      new_answer = parse_rxml (val[broken..sizeof(val)-1], id);
    else
      new_answer = imho_kludge_parse_rxml (val[broken..sizeof(val)-1], id);
    val = ok + new_answer;
    
    bsearch = map (btags, lambda (string tag) { return search (val, tag); } );
    broken = -1;
    foreach (bsearch, int pos)
      if (pos != -1) {
	if (broken == -1)
	  broken = pos;
	else if (pos < broken)
	  broken = pos;
      }
  }

  answer = http_low_answer (200, val);
  // Make sure all pages are reloaded and set Content-type
  answer += ([ "expires": 0, "type": "text/html; charset=" + sessobj->charset ]);
  // Make sure that users' mails are _never_ stored on disk.
  answer += ([ "extra_heads":
	       ([
		 "Pragma": "no-cache",
		 "Cache-Control": "no-store"
	       ])
            ]);
  answer += ([ "is_dynamic": 1 ]);
  return answer;
}

// Now called only when the old parser is used.
string imho_kludge_parse_rxml (string data,object id) {
  // FIXME: Can we use in parse_rxml something like
  //        object camas_image = id->conf->get_provider("camas_images");
  //        and in parse_rxml (data,id), id, 
  //                   ({ ({ "imho_image", 0, camas_image->tag_imho_image, 0 }),
  //                      ...
  //        instead ? -- Xavier
  // And is this really usefull ????? -- Xavier
  return imho_parse_rxml (parse_rxml (data, id), id,
			  ({ ({ "imho_image", 0, tag_imho_image, 0 }),
			     ({ "imho_text", 1, cont_imho_text, 1 }) }));
}

string imho_parse_rxml(string data,object id,array tags){

  foreach(tags,array tag) {
    int pos;
  
    while((pos=search(data,"<"+tag[0]))!=-1) {
      int pos2=search(data,">",pos);
      int pos3;
      int pos4=pos2;
      if(tag[1]) {
	pos3=search(data,"</"+tag[0],pos2);
	pos4=search(data,">",pos3);
      }
      string tag_all=data[pos..pos4];
      string tag_args=data[pos+sizeof(tag[0])+1..pos2-1];
      
      array i=({ 0 });
      array args_a=((Array.map(tag_args/"\"",
			       lambda(string s,array i){return (i[0]=i[0]?0:1)?replace(s," ","\0"):s;},
			       i)*"\"")/"\0")-({ "" });
      mapping args=([  ]);
      string s1,s2;
      foreach(args_a,string s)
	if(sscanf(s,"%s=%s",s1,s2)==2){
	  args[s1]=replace(s2,"\"","");
	}

      string replacement;
      if(tag[1])
	replacement=tag[2](tag[0],args,data[pos2+1..pos3-1],id,0);
      else
	replacement=tag[2](tag[0],args,id,0);
      if(tag[3] && search(replacement,"gtext")!=-1)
	//strip down and run parse_rxml if gtext present
	data=replace(data,tag_all,parse_rxml(Locale.Charset.encoder("iso-8859-1","")->feed(replacement)->drain(),id));
      else
	data=replace(data,tag_all,replacement);
    }
  }
  return data;
}

array imap_servers(void|string _imapservers) {
  string imapservers = _imapservers || QUERY (imapservers);

  // Upgrade compatibility for IMHO < 0.98
  if(imapservers==DEFAULT_IMAPSERVERS)
    return ({ ({ QUERY (maildomain),QUERY (imapserver)+":"+QUERY (imapport),([ ]) }) });

  array list = (imapservers-"\r")/"\n";
  array servers=({ });
  
  int i=0;
  foreach(list, string line) {
    if (strlen(line)>0) {
      string ldomain = "", lserver = "", lcfg = "";
      sscanf(line, "%*[ ]%s%*[\t ]%s%*[\t ]%s", ldomain, lserver, lcfg);
      if (strlen(lserver)==0) {
	sscanf(line, "%*[ ]%s%*[\t ]%s", ldomain, lserver);
      }
      if (sizeof(ldomain) > 0) {
	mapping lcfgmap=([]);
	foreach ((lcfg / ",") , string param) {
	  array parsplt = param / ":";
	  if(sizeof(parsplt[0]-" "))
	    if (sizeof(parsplt)>1) {
	      lcfgmap[parsplt[0]] = parsplt[1];
	    } else {
	      lcfgmap[param] = 1;
	    }
	}
	servers += ({ ({ldomain, lserver, lcfgmap}) });
      }
    }
  }
  // Only the non-empty elements returned
  return servers;
}

string encode_imap_servers(array servers) {
  servers=Array.map(servers,lambda(array server) {
			      string cfg;
			      if(sizeof(server[2])) {
				array _cfg = ({ });
				foreach(indices(server[2]),string s)
				  _cfg += ({ s+":"+server[2][s] });
				cfg = _cfg*",";
			      }
			      return server[0]+"\t"+server[1]+
				(cfg?"\t"+cfg:"");
			    });
  return servers*"\n";
}

string imap_servers_selector() {
  string lst = "";
  array servers = imap_servers();
  if (sizeof(servers) > 1) {
    lst = "<select name=\"imapserver\">\n";
    int i=1;
    foreach(servers, array arr) {
      lst +="<option value=\""+((string) i++);
      lst +="\">@"+arr[0]+"</option>\n";
    }
    lst += "</select>\n";
  }
  return lst;
}

// ------------------ Screens definitions

// ------------------ Login screen

string camas_screen_login (object id, int screen) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "login";
  if (screen == LOGINFAILED) {
    page += MSGA(M_NOLOGIN, ({ (sessobj->loginerror) ? sessobj->loginerror : MSG(M_LOGINERROR) }));
    page += "<br />\n<br />\n";
  }
  else
    if (screen == LOGINFAILEDIMAP)
      page += MSG(M_IMAPERROR) + "<br />\n<br />\n";

  string preset_login = 0;
  if (plugin && plugin->imho_preset_login)
    preset_login = plugin->imho_preset_login (id);

  page += "<form target=\"" + id->misc->imho->target + "\" action=\"" + id->misc->imho->nexttarget;
  page += "\" method=\"post\" name=\"imhologinform\"><input type=hidden name=actionlogin value=\"1\">";

  array imho_from = ({ "$MSGLOGIN", "$MSGPASSWORD", "$INPUTLOGIN", "$INPUTPASSWORD", "$INPUTOK" });
  array imho_to = ({ MSG(M_LOGIN), MSG(M_PASSWORD) });

  string bofh = "<input name=\"login\"";
  if (id->supports->javascript)
    bofh += "<if defined=\"jscript is on\"> onChange=\"this.form.passwd.focus()\" </if>";
  if (preset_login)
    bofh += " value=" + html_encode_tag_value(preset_login);
  bofh += ">";
  
  object auth_module = id->conf->get_provider ("camas_auth");
  if (objectp (auth_module) && auth_module->getemaillogin ()) {
    string|int domain = auth_module->getdnsname ();
    if (stringp (domain))
      bofh += "<imho_text nfont='Arial Rounded' scale=0.3>@" + domain + "</imho_text>";
  }
  
  imho_to += ({ bofh });
  
  array (string) layouts = id->conf->get_provider ("camas_layout_manager")->list_layouts ();
  
  bofh = "<input type=\"password\" name=\"passwd\"";
  if ((sizeof (layouts) == 1) && id->supports->javascript)
    bofh += "<if defined=\"jscript\" is on> onChange=\"this.form.submit()\" </if>";
  bofh += ">";
  imho_to += ({ bofh });
  imho_to += ({ formdrawbutton ("m_login_ok", "OK", MSG(M_LOGIN_OK)) });
  page += replace (QUERY (laylogin1), imho_from, imho_to);
  
  if (sizeof (layouts) > 1)
    {
      imho_from = ({ "$MSGUSRINFACE",
		     "$INPUTSELECTINFACE" });
      imho_to = ({ MSG(M_USERINTERFACE) });
      bofh = "<select name=\"layout\">";
      bofh+= "<option value=\"__user\" selected>" + MSG(M_SAVEDUSERINTERFACE) + "</option>\n";
      foreach (layouts, string lay) {
	bofh += "<option value=\"" + lay + "\">";
	bofh += lay + "</option>\n";
      }
      bofh+="</select>";
      imho_to += ({ bofh });
      page += replace (QUERY (laylogin2), imho_from, imho_to);
    }
  
  if (id->supports->javascript) {
    page += "<if defined=\"jscript is on\"> ";
    page += "<script language=\"JavaScript\" type=\"text/javascript\">\n<!--\n";
    if (preset_login)
      page += "document.imhologinform.passwd.focus();\n";
    else
      page += "document.imhologinform.login.focus();\n";
    page += "// -->\n</script></define>\n";
  }
  
  page += "</form>";
  
  return page;
}

// ------------------ Logout screen

string camas_screen_logout (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;
  
  id->misc->imho->screen = "logout";
  id->misc->imho->title = MSG(M_LOGGEDOUT);
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  destruct (sessobj->imapclient);
#if constant(thread_create)
  destruct (lock);
#endif
  page += "<end_session>";
  page += replace (QUERY (laylogout), "$LOGOUTMSG", MSG(M_LOGOUTMSG));

  return page;
}

// ------------------ Compose screen

string camas_screen_compose (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "compose";
  id->misc->imho->title = MSG(M_COMPOSEMAIL);

  // FIXME: What the hell is this ? - Xavier
  int feature_addressbook = feature (FEAT_ADDRESSBOOK);
  int feature_ldap = feature (FEAT_LDAP);

  array imho_from = ({ "$INPUTSEND",
		       "$INPUTCANCEL",
		       "$INPUTSPELLCHECK",
		       "$INPUTSPELLCHOOSE",
		       "$INPUTSAVEDRAFT" });

  array imho_to = ({ formdrawbutton ("m_send", "actionsend", MSG(M_SEND)),
		     formdrawbutton ("m_cancelsend", "actionindex", MSG(M_CANCELSEND)) });

  string buttons = "", bofh = "";

  if (sizeof (QUERY (ispell))) {
    imho_to += ({ formdrawbutton ("m_spellcheck", "actionspellcheck", MSG(M_SPELLCHECK)) });
    bofh = "<select name=\"ispelldict\">";
    foreach (QUERY (ispelldict) / ",", string dict) 
      bofh += "<option value=\"" + dict + "\">" + html_encode_string (dict) + "</option>";
    bofh += "</select>";
    imho_to += ({ bofh });
  }
  else
    imho_to += ({ "", "" });

  imho_to += ({ formdrawbutton ("m_savedraft", "actionsavedraft", MSG(M_SAVEDRAFT)) });

  buttons = replace (QUERY (laycompose1), imho_from, imho_to);

  page += "<form name=\"imhocomposeform\" target=\"" + id->misc->imho->target;
  page += "\" action=\"" + id->misc->imho->nexttarget + "\" method=\"post\">";

  if (!id->misc->imho->notopbuttons)
    page += buttons;

  imho_from = ({ "$MSGFROM",
		 "$MSGEMAILFROM",
		 "$MSGORGANIZATION",
		 "$MSGTO",
		 "$INPUTTO",
		 "$INPUTADDRBOOKTO",
		 "$INPUTADDRBOOK",
		 "$INPUTLDAPTO",
		 "$INPUTLDAP",
		 "$MSGCC",
		 "$INPUTCC",
		 "$INPUTADDRBOOKCC",
		 "$INPUTLDAPCC",
		 "$MSGBCC",
		 "$INPUTBCC",
		 "$INPUTADDRBOOKBCC",
		 "$INPUTLDAPBCC",
		 "$MSGSUBJECT",
		 "$INPUTSUBJECT" });

  string organization = (sizeof (QUERY (siteorg))) ? ("(" + QUERY (siteorg) + ")")
    : (sizeof (sessobj->organization) ? ("(" + html_encode_string (sessobj["organization"]) + ")") : "");

  imho_to = ({ "<imho_text nfont='Arial Rounded' scale=0.4>" + MSG(M_FROM) + ":</imho_text>",
	       html_encode_string (sessobj->name) + " &lt;" + html_encode_string (sessobj->address) + "&gt;",
	       organization,
	       "<imho_text nfont='Arial Rounded' scale=0.4>" + MSG(M_TO) + ":</imho_text>",
	       "<input name=\"to\" size=\"" + QUERY (headersize) + "\" value=" + html_encode_tag_value (sessobj->to) + ">" });
		 
  if (feature_addressbook)
    {
      imho_to +=({ formdrawbutton ("m_addressbook", "actionaddressbookto", MSG(M_ADDRESSBOOK)) });
      imho_to +=({ formdrawbutton ("m_addressbook", "actionaddressbooktoccbcc", MSG(M_ADDRESSBOOK)) });
    }
  else
    imho_to += ({ "", "" });

  if (feature_ldap)
    {
      imho_to += ({ formdrawbutton ("m_ldap", "actionldapto", MSG(M_LDAP)) });
      imho_to += ({ formdrawbutton ("m_ldap", "actionldaptoccbcc", MSG(M_LDAP)) });
    }
  else
    imho_to += ({ "", "" });

  imho_to += ({ "<imho_text nfont='Arial Rounded' scale=0.4>" + MSG(M_CC) + ":</imho_text>",
		"<input name=\"cc\" size=\"" + QUERY (headersize)
		+ "\" value=" + html_encode_tag_value (sessobj->cc) + ">" });
		  
  if (feature_addressbook)
    imho_to += ({ formdrawbutton ("m_addressbook", "actionaddressbookcc", MSG(M_ADDRESSBOOK)) });
  else
    imho_to += ({ "" });

  if (feature_ldap)
    imho_to += ({ formdrawbutton ("m_ldap", "actionldapcc", MSG(M_LDAP)) });
  else
    imho_to += ({ "" });

  imho_to += ({ "<imho_text nfont='Arial Rounded' scale=0.4>" + MSG(M_BCC) + ":</imho_text>",
		"<input name=\"bcc\" size=\"" + QUERY (headersize)
		+ "\" value=" + html_encode_tag_value (sessobj->bcc) + ">" });

  if (feature_addressbook)
    imho_to += ({ formdrawbutton ("m_addressbook", "actionaddressbookbcc", MSG(M_ADDRESSBOOK)) });
  else
    imho_to += ({ "" });

  if (feature_ldap)
    imho_to += ({ formdrawbutton ("m_ldap", "actionldapbcc", MSG(M_LDAP)) });
  else
    imho_to += ({ "" });
    
  imho_to += ({ "<imho_text nfont='Arial Rounded' scale=0.4>" + MSG(M_SUBJECT) + ":</imho_text>",
		"<input name=\"subject\" size=\"" + QUERY (subjectsize)
		+ "\" value=" + html_encode_tag_value(sessobj->subject) + ">" });

  page += replace (QUERY (laycompose2), imho_from, imho_to);

  if (feature (FEAT_ATTACHMENTS)) {
    int no_att = sizeof (sessobj->attachments);
    int s = (no_att > 4) ? 4 : ((no_att < 1) ? 1 : no_att);
    imho_from = ({ "$MSGATTACHMENTS",
		   "$INPUTFILEATT",
		   "$INPUTADD",
		   "$INPUTDEL" });
    imho_to = ({ "<imho_text nfont='Arial Rounded' scale=0.4>" + MSG(M_ATTACHMENTS) + ":</imho_text>" });
      
    bofh = "<select name=\"attachments\" multiple size=\"" + (string)s + "\">";
    int i = 0;
    if (!sizeof (sessobj->attachments)) 
      bofh += "<option value=\"-1\">(" + MSG(M_NOATTACHMENTS) + ")</option>";
    foreach (sessobj->attachments, mapping file) {
      bofh += "<option value=\"" + (string)i + "\">";
      bofh += html_encode_string (file->fname + " (" + file->size + " bytes)");
      bofh += "</option>";
      i++;
    }
    bofh += "</select>";
    imho_to += ({ bofh,
		  formdrawbutton ("m_addattachment", "actiongotoaddattachment", MSG(M_ADDATTACHMENT)),
		  formdrawbutton ("m_removemarkedattachments", "actionremoveattachment", MSG(M_REMOVEMARKEDATTACHMENTS)) });
    page += replace (QUERY (laycompose3), imho_from, imho_to);
  }
    
  imho_from = ({ "$INPUTMSGAREA",
		 "$INPUTSAVECOPY",
		 "$INPUTDSNDELAY",
		 "$INPUTDSNSUCCESS",
		 "$INPUTMDN" });
  imho_to = ({ "<textarea rows=\"" + QUERY (msgrows) + "\" cols=\"" + QUERY (msgcols) +
	       "\" name=\"message\" wrap=\"physical\">" + html_encode_string (sessobj->message) + "</textarea>" });
    
  if (feature (FEAT_MAILBOXES) && sizeof (sessobj->sentfolder))
    {
      if(feature (FEAT_SAVEMAILCOPY))
	imho_to += ({ "<input type=\"checkbox\" name=\"nosavemail\" value=\"1\"> " + MSG(M_DONTSAVEMAIL) }); 
      else
	imho_to += ({ "<input type=\"checkbox\" name=\"dosavemail\" value=\"1\"> " + MSG(M_SAVEMAILCOPY) });
    }
  else
    imho_to += ({ "" });

  if (feature (FEAT_DSN) && (QUERY (sendmethod) == "SMTP"))
    imho_to += ({ "<input type=\"checkbox\" name=\"dsndelay\" value=\"1\"> " + MSG(M_DSN_DELAY),
		  "<input type=\"checkbox\" name=\"dsnsuccess\" value=\"1\"> " + MSG(M_DSN_SUCCESS) });
  else
    imho_to += ({ "", "" });

  imho_to += ({ "<input type=\"checkbox\" name=\"mdn\" value=\"1\"> " + MSG(M_MDN) });

  page += replace (QUERY (laycompose4), imho_from, imho_to);

  page += replace (buttons, ({ "ispelldict", "actionspellcheck" }), ({ "ispelldict2", "actionspellcheck2" }) );

  //page+="</td></tr>\n</form></table>";
  page += "</form>";
  if (id->supports->javascript) {
    page += "<script language=\"JavaScript\" type=\"text/javascript\">\n<!--\n";
    page += "document.imhocomposeform.to.focus();\n";
    page += "// -->\n</script>\n";
  }

  return page;
}

// ------------------ Attachments screen

string camas_screen_attachments (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "attachments";
  id->misc->imho->title = MSG(M_ATTACHMENTSHEADER);

  page += "<center><br />\n<table ";
  if (!sizeof (QUERY (uploaddir)))
    page += "width=\"50%\""; 
  page += "><tr>";

  if (sizeof (QUERY (uploaddir))) {
    page += "<th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
    page += MSG(M_ADDFILEASATTACHMENT) + "</font></th>";
  }

  page += "<th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_UPLOADATTACHMENT) + "</font></th></tr>\n<tr>";

  if (sizeof (QUERY (uploaddir))) {
    page += "<td width=\"50%\" bgcolor=\"&imho_tdbgcolor;\"><form target=\"" + id->misc->imho->target;
    page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget + "\">";
    page += "<select name=\"attachments\" multiple size=\"3\">";

    if (!sizeof (sessobj->files)) 
      page += "<option value=\"-1\">(" + MSG(M_NOATTACHFILES) + ")</option>";
    else {
      int i = 0;
      foreach (sessobj->files, mapping file) {
	page += "<option value=\"" + (string)i + "\">" + html_encode_string (file->fname);
	page += "</option>";
	i++;
      }
    }

    page += "</select><br />\n";
    page += formdrawbutton ("m_addmarkedtoattachments", "actionaddfileattachment", MSG(M_ADDMARKEDTOATTACHMENTS));
    page += "</form></td>";
  }

  page += "<td width=\"50%\" bgcolor=\"&imho_tdbgcolor;\"><form name=\"imhouploadform\" target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" enctype=\"multipart/form-data\" action=\"" + id->misc->imho->nexttarget + "\"><br />\n";
  page += "<input name=\"file\" type=\"file\">";

  if (id->supports->javascript)
    page += "<input type=\"hidden\" name=\"fixedfilename\" value=\"\" /><br />\n";
  else
    page += "<br />\n" + MSG(M_WINDOWSBUG);

  page += "<br />\n<input type=\"hidden\" name=\"actionuploadattachment\" value=\"1\" />";

  if (id->supports->javascript)
    page += formdrawbutton ("m_uploadtoattachments", "1", MSG(M_UPLOADTOATTACHMENTS),
			    "document.forms.imhouploadform.fixedfilename.value=document.forms.imhouploadform."
			    "file.value.replace(/\\\\/g,'\\\\\\\\')");
  else
    page += formdrawbutton ("m_uploadtoattachments", "1", MSG(M_UPLOADTOATTACHMENTS));

  page += "</form><br />\n";
  page += "</td></tr>\n</table>";

  if (sizeof (sessobj->attachments)) {
    int no_att = sizeof (sessobj->attachments);
    int s = (no_att > 4) ? 4 : ((no_att < 1) ? 1 : no_att);

    page += "<center><form target=\"" + id->misc->imho->target;
    page += "\" action=\"" + id->misc->imho->nexttarget + "\" method=\"post\">";

    string bofh = "<select name=\"attachments\" multiple size=\"" + (string)s + "\">";
    int i = 0;

    foreach (sessobj->attachments, mapping file) {
      bofh += "<option value=\"" + (string)i + "\">";
      bofh += html_encode_string (file->fname + " (" + file->size + " bytes)");
      bofh += "</option>";
      i++;
    }
    bofh += "</select>";

    page += bofh;
    page += "<br />" + formdrawbutton ("m_removemarkedattachments", "actionremoveattachment2", MSG(M_REMOVEMARKEDATTACHMENTS));
    page += "</form></center>";
  }

  page += "<table><tr><td><form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget + "\">";
  page += formdrawbutton ("m_backtocompose", "actioncontinuecompose", MSG(M_BACKTOCOMPOSE));
  page += "</form></td></tr>\n</table></center>";

  return page;
}

// ------------------ Addressbook screen

string camas_screen_addressbook (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "addressbook";
  id->misc->imho->title = MSG(M_ADDRESSBOOKTITLE);

  page += "<form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget + "\">";
  page += "<table width=\"100%\"><tr>";
  page += "<th class=\"name\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += (sessobj->recipientfield ? MSG(M_INDEXNAMERECIPIENT) : MSG(M_INDEXNAMEEDIT)) + "</font></th>";
  page += "<th class=\"address\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_ADDRESS) + "</font></th></tr>";

  int i = 0;
  if (!sizeof (sessobj->addressbook))
    page += "<tr><td>" + MSG(M_NOADDRESSES) + "</td></tr>";
  else {
    foreach (sessobj->addressbook / "\n", string line) {
      string name = "", address = ""; 
      if (sscanf (line, "%s:%s", name, address) == 2) {
	page += "<tr><td class=\"name\" bgcolor=\"&imho_tdbgcolor;\">";
	if (sessobj->recipientfield) {
	  page += "<a  target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
	  page += "?actionaddrecipient=1&address=" + http_encode_url (address) + "\">";
	  page += fix_header (name) + "</a>";
	}
	else {
	  if (feature (FEAT_EDITADDRESSBOOK)) {
	    page += "<a  target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
	    page += "?actioneditaddress=1&address=" + (string)i + "\">";
	    page += fix_header (name) + "</a>";
	  }
	  else
	    page += fix_header (name);
	}
	page += "</td><td class=\"address\" bgcolor=\"&imho_tdbgcolor;\">";
	page += replace (fix_header (address), ",", "<br />") + "</td></tr>";
      }
      i++;
    }
  }

  if (sizeof (global_addressbook) > 0) {
    page += "<tr><td class=\"name\" bgcolor=\"&imho_tdbgcolor;\">";
    page += "<b>" + MSG(M_GLOBALADDRESSES) + "</b></td></tr>";
    foreach (global_addressbook / "\n", string line) {
      string name = "", address = ""; 
      if (sscanf (line, "%s:%s", name, address) == 2) {
	page += "<tr><td class=\"name\" bgcolor=\"&imho_tdbgcolor;\">";
	if (sessobj->recipientfield) {
	  page +="<a  target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
	  page += "?actionaddrecipient=1&address=" + http_encode_url (address) + "\">";
	  page += fix_header (name) + "</a>";
	}
	else
	  page += fix_header (name);

	page += "</td><td class=\"address\" bgcolor=\"&imho_tdbgcolor;\">";
	page += replace (fix_header (address), ",", "<br />") + "</td></tr>";
	i++;
      }
    }
  }

  page += "</table>";
  if (feature (FEAT_EDITADDRESSBOOK))
    page += formdrawbutton ("m_newaddress", "actionnewaddress", MSG(M_NEWADDRESS));
  page += formdrawbutton ("m_importaddressbook", "actiongotoimportaddress", MSG(M_IMPORTADDRESSBOOK));
  page += "</form><br />";

  return page;
}

// ------------------ Addressbook screen (multiple selection)

string camas_screen_addressbook2 (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "addressbook";
  id->misc->imho->title = MSG(M_ADDRESSBOOKTITLE);

  page += "<form method=\"get\" action=\"" + id->misc->imho->nextpage + "\">";
  page += "<table width=\"100%\"><tr>";
  page += "<th class=\"to\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_TO) + "</font></th>";
  page += "<th class=\"cc\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_CC) + "</font></th>";
  page += "<th class=\"bcc\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_BCC) + "</font></th>";
  page += "<th class=\"name\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_INDEXNAME);
  page += "</font></th>";
  page += "<th class=\"address\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_ADDRESS);
  page += "</font></th></tr>";

  int i = 0;
  if (!sizeof (sessobj->addressbook))
    page += "<tr><td>" + MSG(M_NOADDRESSES) + "</td></tr>";
  else {
    foreach (sessobj->addressbook / "\n", string line) {
      string name = "", address = ""; 
      if (sscanf (line, "%s:%s", name, address) == 2) {
	page += "<tr>";
	page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_to_" + i + "\"></td>";
	page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_cc_" + i + "\"></td>";
	page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_bcc_" + i + "\"></td>";
	page += "<td class=\"name\" bgcolor=\"&imho_tdbgcolor;\">";
	page += fix_header (name);
	page += "</td><td class=\"address\" bgcolor=\"&imho_tdbgcolor;\">";
	page += "<input type=\"hidden\" name=\"address_" + i + "\" value=\"" + address + "\">";
	page += replace (fix_header (address), ",", "<br />") + "</td></tr>";
	i++;
      }
    }

    if (sizeof (global_addressbook) > 0) {
      page += "<tr><td colspan=\"5\" class=\"name\" bgcolor=\"&imho_tdbgcolor;\">";
      page += "<b>" + MSG(M_GLOBALADDRESSES) + "</b></td></tr>";
      foreach (global_addressbook / "\n", string line) {
	string name = "", address = ""; 
	if (sscanf (line, "%s:%s", name, address) == 2) {
	  page += "<tr>";
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_to_" + i + "\"></td>";
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_cc_" + i + "\"></td>";
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_bcc_" + i + "\"></td>";
	  page += "<td class=\"name\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += fix_header (name);
	  page += "</td><td class=\"address\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += "<input type=\"hidden\" name=\"address_" + i + "\" value=\"" + address + "\">";
	  page += replace (fix_header (address), ",", "<br />") + "</td></tr>";
	  i++;
	}
      }
    }
  }

  page += "</table>";
  page += formdrawbutton ("m_addrecipients", "actionaddrecipients", MSG(M_ADDRECIPIENTS));
  page += formdrawbutton ("m_backtocompose", "actioncontinuecompose", MSG(M_BACKTOCOMPOSE));
  page += "</form><br />";

  return page;
}

// ------------------ LDAP search results screen

string camas_screen_ldapresult (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "ldapresult";
  id->misc->imho->title = MSG(M_LDAPTITLE);

  page += "<table width=\"100%\"><tr>";
  page += "<th class=\"name\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += (sessobj->recipientfield ? MSG(M_INDEXNAMERECIPIENT) : MSG(M_INDEXNAMEEDIT)) + "</font></th>";
  page += "<th class=\"address\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_ADDRESS) + "</font></th>";
  if (QUERY (ldapshowou))
    page += "<th class=\"ou\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_OU) + "</font></th>";
  page += "</tr>";

  if (!sizeof (sessobj->ldapadd))
    page += "<tr><td>" + MSG(M_NOADDRESSES) + "</td></tr>";
  else {
    foreach (sessobj->ldapadd / "\n", string line) {
      string name = "", address = "", ou="";
      if (QUERY (ldapshowou)) {
	if (sscanf (line, "%s:%s:%s", name, address, ou) == 3) {
	  page += "<tr><td class=\"name\" bgcolor=\"&imho_tdbgcolor;\"><a target=\"";
	  page += id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;				
	  page += "?actionaddrecipient=1&address=" + http_encode_url (address) + "\">";
	  page += fix_header (name) + "</a></td><td class=\"address\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += replace (fix_header (address), ",", "<br />") + "</td>";
	  page += "<td class=\"ou\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += replace (fix_header (ou), ",", "<br />") + "</td>";
	  page += "</tr>";
	}
      }
      else {
	if (sscanf (line, "%s:%s", name, address) == 2) {
	  page += "<tr><td class=\"name\" bgcolor=\"&imho_tdbgcolor;\"><a target=\"";
	  page += id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
	  page += "?actionaddrecipient=1&address=" + http_encode_url (address) + "\">";
	  page += fix_header (name) + "</a></td><td class=\"address\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += replace (fix_header (address), ",", "<br />") + "</td></tr>";
	}
      }
    }
  }
  page += "</table><br />";
  page += "<form>" + formdrawbutton ("m_backtocompose", "actioncontinuecompose", MSG(M_BACKTOCOMPOSE)) + "</form>";

  return page;
}

// ------------------ LDAP search results screen (multiple selection)

string camas_screen_ldapresult2 (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "ldapresult";
  id->misc->imho->title = MSG(M_LDAPTITLE);

  page += "<form method=\"post\" action=\"" + id->misc->imho->nextpage + "\">";

  page += "<table width=\"100%\"><tr>";
  page += "<th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_TO) + "</font></th>";
  page += "<th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_CC) + "</font></th>";
  page += "<th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_BCC) + "</font></th>";
  page += "<th class=\"name\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_INDEXNAME) + "</font></th>";
  page += "<th class=\"address\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_ADDRESS) + "</font></th>";
  if (QUERY (ldapshowou))
    page += "<th class=\"ou\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">" + MSG(M_OU) + "</font></th>";
  page += "</tr>";

  int i = 0;
  if (!sizeof (sessobj->ldapadd)) {
    page += "<tr><td>" + MSG(M_NOADDRESSES) + "</td></tr>";
    page += "<tr><td>";
    page += formdrawbutton ("m_backtocompose", "actioncontinuecompose", MSG(M_BACKTOCOMPOSE));
    page += "</td></tr></table></form>";
  }
  else {
    foreach (sessobj->ldapadd / "\n", string line) {
      string name = "", address = "", ou="";
      if (QUERY (ldapshowou)) {
	if (sscanf (line, "%s:%s:%s", name, address, ou) == 3) {
	  page += "<tr>"; 
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_to_" + i + "\"></td>";
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_cc_" + i + "\"></td>";
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_bcc_" + i + "\"></td>";
	  page += "<td class=\"name\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += "<input type=\"hidden\" name=\"address_" + i + "\" value=\"" + address + "\">";
	  page += fix_header (name) + "</td><td class=\"address\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += replace (fix_header (address), ",", "<br />") + "</td>";
	  page += "<td class=\"ou\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += replace (fix_header (ou), ",", "<br />") + "</td>";
	  page += "</tr>";
	}
      }
      else {
	if (sscanf (line, "%s:%s", name, address) == 2) {
	  page += "<tr>";
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_to_" + i + "\"></td>";
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_cc_" + i + "\"></td>";
	  page += "<td align=\"center\"><input type=\"checkbox\" name=\"recipient_bcc_" + i + "\"></td>";
	  page += "<td class=\"name\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += "<input type=\"hidden\" name=\"address_" + i + "\" value=\"" + address + "\">";
	  page += fix_header (name) + "</td><td class=\"address\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += replace (fix_header (address), ",", "<br />") + "</td></tr>";
	}
      }
      i++;
    }
    page += "</table>";
    page += formdrawbutton ("m_addrecipients", "actionaddrecipients", MSG(M_ADDRECIPIENTS));
    page += formdrawbutton ("m_backtocompose", "actioncontinuecompose", MSG(M_BACKTOCOMPOSE));
    page += "</form>";
  }

  return page;
}

// ------------------ Mail filters screen

string camas_screen_mailfilter (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "filterbook";
  id->misc->imho->title = MSG(M_FILTERBOOKTITLE);

  page += "<form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget + "\">";
  page += "<table width=\"100%\"><tr>";
  page += "<th class=\"namefield\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += (sessobj->recipientfield ? MSG(M_INDEXNAMEFIELDFILTER) : MSG(M_INDEXNAMEEDITFIELDFILTER));
  page += "</font></th>";
  page += "<th class=\"filterexpression\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += (sessobj->recipientfield ? MSG(M_INDEXNAMERECIPIENTFILTER) : MSG(M_INDEXNAMEEDITFILTER));
  page += "</font></th>";
  page += "<th class=\"filterfolder\" bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_ADDRESSFILTER) + "</font></th></tr>";

  if (!sizeof (sessobj->filterbook))
    page += "<tr><td>" + MSG(M_NOADDRESSESFILTER) + "</td></tr>";
  else {
    int i = 0;
    foreach (sessobj->filterbook / "\n", string line) {
      string namefield = "", filterexpression = "", filterfolder = ""; 
      if (sscanf (line, "%s:%s:%s", namefield, filterexpression, filterfolder) == 3) {
	page += "<tr><td class=\"namefield\" bgcolor=\"&imho_tdbgcolor;\">";
	page += "<a target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
	page += "?actioneditaddressfilter=1&namefield=" + (string)i + "\">";
	page += fix_header (namefield) + "</a>";
	
	page += "</td><td class=\"filterexpression\" bgcolor=\"&imho_tdbgcolor;\">";
	page += "<a target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
	page += "?actioneditaddressfilter=1&namefield=" + (string)i + "\">";
	page += fix_header (filterexpression) + "</a>";

	page += "</td><td class=\"filterfolder\" bgcolor=\"&imho_tdbgcolor;\">";
	page += "<a target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
	page += "?actioneditaddressfilter=1&namefield=" + (string)i + "\">";
	page += fix_header (filterfolder)+"</a>";
	page += "</td></tr>";

	i++;
      }
    }
  }

  page += "</table>";
  if (feature (FEAT_EDITFILTERBOOK))
    page += formdrawbutton ("m_newfilter", "actionnewaddressfilter", MSG(M_NEWADDRESSFILTER));
  page += "</form><br />";

  return page;
}

// ------------------ LDAP search screen

string camas_screen_ldapsearch (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "ldapsearch";
  id->misc->imho->title = MSG(M_LDAPTITLE);

  page += "<form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget + "\">";
  page += "<center><table><th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_NAMECONT) +"</font></th>";
  page += "<tr><td bgcolor=\"&imho_tdbgcolor;\">";
  page += "<input name=\"namecont\" type=\"text\"> </td></tr>";
  page += "<tr><td bgcolor=\"&imho_tdbgcolor;\">";
  if (sessobj->recipientfield == "toccbcc")
    page += "<input name=\"actionsearchldaptoccbcc\" type=\"hidden\" value=\"1\">";
  else
    page += "<input name=\"actionsearchldap\" type=\"hidden\" value=\"1\">";
  page += formdrawbutton ("m_searchldap", "", MSG(M_SEARCHLDAP));
  page += "</td></tr></table></center></form>";

  return page;
}

// ------------------ Import adress screen

string camas_screen_importaddress (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "importaddress";
  id->misc->imho->title = MSG(M_ADDRESSBOOKTITLE);

  page += "<form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" enctype=\"multipart/form-data\" action=\"" + id->misc->imho->nexttarget + "\">";
  page += "<center><table><th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_IMPORTADDRESSBOOKTITLE) + "</font></th>";
  page += "<tr><td bgcolor=\"&imho_tdbgcolor;\">";
  page += MSG(M_UPLOADADDRESSBOOK)+":<br />";
  page += "<input name=\"file\" type=\"file\"><br />" + MSG(M_PINELOCATION) + "</td></tr>";
  page += "<tr><td><input type=\"hidden\" name=\"actionimportaddress\" value=\"1\">";
  page += formdrawbutton ("m_uploadandimport", "\"\"", MSG(M_UPLOADANDIMPORT));
  page += "</td></tr></table></center></form>";

  return page;
}

// ------------------ Edit address screen

string camas_screen_editaddress (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "editaddress";
  id->misc->imho->title = MSG(M_ADDRESSBOOKTITLE);

  page += "<form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget + "\">";
  page += "<center><table><tr><th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += ((sessobj->editaddressmode == "old") ? MSG(M_EDITADDRESS) : MSG(M_ADDADDRESS)) + "</font></th></tr>";
  page += "<tr><td bgcolor=\"&imho_tdbgcolor;\">";
  string name = "", address = "";
  sscanf ((sessobj->addressbook / "\n")[((int)sessobj->editaddress)], "%s:%s", name, address);
  string baz = QUERY (layeditaddress);

  if (sizeof (baz)) {
    array imho_from = ({ "$NAME", "$ADDRESS", "$BUTTONSAVE", "$BUTTONDELETE", "$BUTTONCANCEL" });
    array imho_to = ({
      html_encode_string (name),
      html_encode_string (address),
      formdrawbutton ("m_saveaddress", "actioneditaddressdone", MSG(M_SAVEADDRESS)),
      formdrawbutton ("m_deleteaddress", "actiondeleteaddress", MSG(M_DELETEADDRESS)),
      formdrawbutton ("m_canceladdress", "actioncanceleditaddress", MSG(M_CANCELADDRESS))
    });
    baz = replace (baz, imho_from, imho_to);

    if (feature (FEAT_EXTENDEDABOOK)) {
      array(string) fn = sessobj["extendedabook"]->get_fields ();
      foreach (indices (fn), int i) {
	baz = replace (baz, "$" + fn[i] + "$", html_encode_string
		       (sessobj["extendedabook"]->get_entry (name, fn[i])));
	baz = replace (baz, "%" + fn[i] + "%", html_encode_string
		       (sprintf ("ebook%d", i)));
      }
    }
    page += baz;
  }
  else {
    page += MSG(M_INDEXNAME) + ":<br /><input name=\"name\" size=\"60\" value=";
    page += html_encode_tag_value (name) + "><br />";
    page += MSG(M_ADDRESSFORMAT) + ":<br /><input name=\"address\" size=\"60\" value=";
    page += html_encode_tag_value (address) + "><br /></td></tr>";
    if (feature (FEAT_EXTENDEDABOOK)) {
      array(string) fn = sessobj["extendedabook"]->get_fields ();
      foreach (indices (fn), int i) {
	page += "<tr><td>";
	page += html_encode_string(fn[i]) + ":<br /><input name=";
	page += html_encode_tag_value(sprintf("ebook%d",i)) + " size=60 value=";
	page += html_encode_tag_value(sessobj["extendedabook"]->get_entry (name, fn[i]));
	page += "><br />";
	page += "</td></tr>";
      }
    }
    page += "<tr><td>";
    page += formdrawbutton ("m_saveaddress", "actioneditaddressdone", MSG(M_SAVEADDRESS));
    if (sessobj->editaddressmode == "old")
      page += formdrawbutton ("m_deleteaddress", "actiondeleteaddress", MSG(M_DELETEADDRESS));
    page += formdrawbutton ("m_canceladdress", "actioncanceleditaddress", MSG(M_CANCELADDRESS));
    page += "</td></tr></table></center>";
  }
  page += "</form>";

  return page;
}

// ------------------ Edit mail filter screen

string camas_screen_editaddressfilter (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "editaddressfilter";
  id->misc->imho->title = MSG(M_FILTERBOOKTITLE);

  page += "<form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget + "\">";

  string namefield = "", filterexpression = "", filterfolder = "", varfolder = "", varfield = "";
  array fieldsfilters = ({ "From", "Subject", "Date", "To", "Cc" });

  if (sessobj->editaddressfiltermode == "old")
    sscanf ((sessobj->filterbook / "\n")[((int)sessobj->editaddressfilter)], "%s:%s:%s",
	    namefield, filterexpression, filterfolder);
  else
    sscanf (sessobj->newaddressfilter, "%s:%s:%s", namefield, filterexpression, filterfolder);

  if (sizeof (QUERY (layeditaddressfilter))) {
    string baz = QUERY (layeditaddressfilter);
    array imho_from = ({
      "$INPUTNAMEFIELD",
      "$INPUTFIELD",
      "$INPUTNAMEFILTER",
      "$INPUTFILTER",
      "$INPUTFOLDER",
      "$BUTTONFILTERSAVE",
      "$BUTTONFILTERDELETE",
      "$BUTTONFILTERCANCEL"
    });
    varfolder += "<select name=\"filterfolder\"><option value=\"imhonomailbox\">" + MSG(M_SELECTMBOX) + "</option>\n";
    for (int i = 0; i < sizeof (sessobj->mailboxes); i++) {
      if (!(sessobj->mailboxes[i][MB_FLAGS_IDX] & MB_NOSELECT)) {
	string mbox = sessobj->mailboxes[i][MB_DISPLAYNAME_IDX];
	mbox = sessobj->imapclient->translate_frommbox (sessobj, mbox);
	if (sessobj->mailboxes[i][MB_FOLDERNAME_IDX] != sessobj->mailbox[MB_FOLDERNAME_IDX]) {
	  if (filterfolder == ("INBOX" + mbox)) {
	    varfolder += "<option value=" + html_encode_tag_value (sessobj->mailboxes[i][MB_FOLDERNAME_IDX]);
	    varfolder += " selected>" + html_encode_string (mbox) + "</option>\n";
	  }
	  else {
	    varfolder += "<option value=" + html_encode_tag_value (sessobj->mailboxes[i][MB_FOLDERNAME_IDX]) + ">";
	    varfolder += html_encode_string (mbox) + "</option>\n";
	  }
	}
      }
    }
    varfolder += "</select>";

    varfield += "<select name=\"namefield\"><option value=\"imhonofield\">" + MSG(M_SELECTFIELD) + "</option>\n";
    foreach (fieldsfilters , string elem)
      if (namefield == elem)
	varfield += "<option value=\"" + elem + "\" selected>" + elem + "</option>\n";
      else
	varfield += "<option value=\"" + elem + "\">" + elem + "</option>\n";
    varfield += "</select>";

    array imho_to = ({
      "",
      varfield,
      "<input name=\"nameexpression\" value=\"\">",
      "<input name=\"filterexpression\" value=" + html_encode_tag_value (filterexpression) + ">",
      varfolder,
      //"<input name=filterfolder value="+html_encode_tag_value(filterfolder)+">",
      formdrawbutton ("m_savefilter", "actioneditaddressfilterdone", MSG(M_SAVEADDRESSFILTER)),
      formdrawbutton ("m_deletefilter", "actiondeleteaddressfilter", MSG(M_DELETEADDRESSFILTER)),
      formdrawbutton ("m_cancelfilter", "actioncanceleditaddressfilter", MSG(M_CANCELADDRESSFILTER))
    });
    baz = replace (baz, imho_from, imho_to);
    page += baz;
  }
  else {
    page += "<center><table><th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
    page += ((sessobj->editaddressfiltermode == "old") ? MSG(M_EDITADDRESSFILTER) : MSG(M_ADDADDRESSFILTER));
    page += "</font></th>";
    page += "<tr><td bgcolor=\"&imho_tdbgcolor;\">";
    page += MSG(M_INDEXFIELDFILTER) + ":<br /><input name=\"namefield\" size=\"60\" value=";
    page += html_encode_tag_value (namefield) + "><br />";
    page += MSG(M_INDEXNAMEFILTER) + ":<br /><input name=\"filterexpression\" size=\"60\" value=";
    page += html_encode_tag_value (filterexpression) + "><br />";
    page += MSG(M_FILTERFOLDER) + ":<br /><input name=\"filterfolder\" size=\"60\" value=";
    page += html_encode_tag_value (filterfolder) + "><br /></td></tr>";
    page += "<tr><td>";
    page += formdrawbutton ("m_savefilter", "actioneditaddressfilterdone", MSG(M_SAVEADDRESSFILTER));
    // **LYF**
    if (sessobj->editaddressfiltermode == "old")
      // **LYF**
      page += formdrawbutton ("m_deletefilter", "actiondeleteaddressfilter", MSG(M_DELETEADDRESSFILTER));
    page += formdrawbutton ("m_cancelfilter", "actioncanceleditaddressfilter", MSG(M_CANCELADDRESSFILTER));
    page += "</td></tr></table></center>";
  }
  page += "</form>";

  return page;
}

// ------------------ Mail index screen

string camas_screen_mailindex (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  //oliv3 page+="IMAP server: " + sessobj->realimapserver + "<hr />";
  int in_drafts = ((sessobj->mailbox[MB_FOLDERNAME_IDX] == (QUERY (defaultmailpath) + sessobj->draftsfolder)));
  id->misc->imho->screen = "mailindex";
  int mails = sizeof (sessobj->mails);

  page += "<form method=\"post\" target=\"" + id->misc->imho->uptarget;
  page += "\" action=\"" + id->misc->imho->nextuptarget + "\">";

  if (sessobj->searchstring && strlen (sessobj->searchstring))
    page += MSG(M_SEARCHSHOWINGMAIL) + sessobj->searchstring + ":<br />";
    
  if (mails) {
    if (sessobj->mailssortcolumn && (sessobj->mailssortcolumn != sessobj->sortcolumn)) {
      sessobj->mailssortcolumn = sessobj->sortcolumn;
      mapping colvals = ([ ]);

      for (int i = 0; i < mails; i++) {
	object mail = sessobj->mails[i];
	switch (sessobj->sortcolumn) {

	case "date":
	  colvals[sortable_date (mail->imap->ENVELOPE[DATE_IDX], mail->imap->INTERNALDATE)
		  + " " + sprintf ("%08d", mail->number)] = i;
	  break;

	case "from":
	  string from_name, from_addr;
	  if (mail->imap->ENVELOPE[FROM_IDX]) {
	    from_name = mail->imap->ENVELOPE[FROM_IDX][0][0];
	    from_addr = mail->imap->ENVELOPE[FROM_IDX][0][2] + "@" + mail->imap->ENVELOPE[FROM_IDX][0][3];
	    from_addr = replace (from_addr, "@.MISSING-HOST-NAME.", "");
	  }
	  else
	    from_name = "-";
	  from_name = fix_header (from_name ? from_name : from_addr);
	  colvals[lower_case (from_name) + " " + (string)sprintf ("%08d", mail->number)] = i;
	  break;

	case "to":
	  string to_name, to_addr;
	  if (mail->imap->ENVELOPE[TO_IDX]) {
	    //write (sprintf ("to= %O\n", mail->imap->ENVELOPE[TO_IDX]));
	    to_name = mail->imap->ENVELOPE[TO_IDX][0][0];
	    to_addr = mail->imap->ENVELOPE[TO_IDX][0][2] + "@" + mail->imap->ENVELOPE[TO_IDX][0][3];
	    to_addr = replace (to_addr, "@.MISSING-HOST-NAME.", "");
	  }
	  else 
	    to_name = "-";
	  to_name = fix_header (to_name ? to_name : to_addr);

	  colvals[lower_case (to_name) + " " + (string)sprintf ("%08d", mail->number)] = i;
	  break;

	case "subject":
	  string nsub, subj = fix_header (mail->imap->ENVELOPE[SUBJECT_IDX]) || "?";
	  while (sscanf (lower_case (subj), "re:%*[ \t]%s", nsub) >= 1)
	    subj = nsub;
				 colvals[subj + " " + (string)sprintf ("%08d", mail->number)] = i;
				 break;
				  
	case "size":
	  colvals[sprintf ("%08d", ((int)mail->imap["RFC822.SIZE"]))
		  + " " + (string)sprintf ("%08d", mail->number)] = i;
	  break;

	default:
	  colvals[(string)sprintf ("%08d", mail->number)] = i;
	  break;
	}
      }
 
      array newmails = allocate (mails);
      array order = sort (indices (colvals));
      for (int t = 0; t < mails; t++)
	newmails[t] = sessobj->mails[colvals[order[t]]];
      sessobj->mails = newmails;
    }

    sscanf (sessobj->visiblemail, "%d", sessobj->visible);

    if (sessobj->visible + sessobj->firstvisiblemail > mails)
      sessobj->firstvisiblemail = mails - sessobj->visible;
    if (sessobj->firstvisiblemail < 0)
      sessobj->firstvisiblemail = 0;

    sessobj->lastvisiblemail = sessobj->firstvisiblemail + sessobj->visible - 1;
    if ((sessobj->lastvisiblemail + 1) > mails)
      sessobj->lastvisiblemail = mails - 1;
    if (!sessobj->sessionsortorder)
      sessobj->sessionsortorder = sessobj->sortorder;

    string buttons = "<table><tr><td>";

    if (feature (FEAT_MAILBOXES))
      {
	if (sessobj->mailbox[MB_DISPLAYNAME_IDX] == sessobj->trashfolder)
	  {
	    buttons += formdrawbutton ("m_deletemarked", "actiondelete", MSG(M_DELETEMARKED));
	    buttons += formdrawbutton ("m_deletetrash", "actiondeleteall", MSG(M_DELETEALLTRASH));
	  }
	else
	  {
	    if ((< "both", "move to trash" >)[QUERY (deletemethod)] && sizeof (sessobj->trashfolder))
	      buttons += formdrawbutton ("m_movetotrash", "actiontrash", MSG(M_MOVETOTRASH));
	    if (QUERY (deletemethod) != "move to trash" || !sizeof (sessobj->trashfolder))
	      buttons += formdrawbutton ("m_deletemarked", "actiondeleteask", MSG(M_DELETEMARKED));
	  }

	buttons += formdrawbutton ("m_movemarked", "actionmove", MSG(M_MOVEMARKED));
	buttons += "<select name=\"mbox\"><option value=\"imhonomailbox\">" + MSG (M_SELECTMBOX) + "</option>\n";
	for (int i = 0; i < sizeof (sessobj->mailboxes); i++)
	  {
	    if (!(sessobj->mailboxes[i][MB_FLAGS_IDX] & MB_NOSELECT))
	      {
		string mbox = sessobj->mailboxes[i][MB_DISPLAYNAME_IDX];
		if (sessobj->mailboxes[i][MB_FOLDERNAME_IDX] != sessobj->mailbox[MB_FOLDERNAME_IDX])
		  {
		    mbox = sessobj->imapclient->translate_frommbox (sessobj, mbox);
		    buttons += "<option value=";
		    buttons += html_encode_tag_value (sessobj->mailboxes[i][MB_FOLDERNAME_IDX]);
		    buttons += ">" + html_encode_string (mbox) + "</option>\n";
		  }
	      }
	  }
	buttons += "</select>\n";
      }
    else
      buttons += formdrawbutton ("m_deletemarked", "actiondeleteask", MSG(M_DELETEMARKED));

    buttons += formdrawbutton ("m_searchmail", "actionsearchmail", MSG(M_SEARCHMAIL));
    buttons += formdrawbutton ("m_checknewmail", "actionreload", MSG(M_CHECKNEWMAIL));
      
    buttons += "</td></tr></table>"; 

    string mailnumbers = "";
    if (mails > sessobj->visible) {
      mailnumbers += "<table><tr><td>";

      string mnrs = sprintf ("<b>[%d-%d]</b>", sessobj->firstvisiblemail + 1, sessobj->lastvisiblemail + 1);
      int i = sessobj->firstvisiblemail;
      while (i > 0) {
	i -= sessobj->visible;
	if (i < 0)
	  i = 0;
	int t = (((i + sessobj->visible) >= mails) ? (mails - 1) : (i + sessobj->visible - 1));
	mnrs = "<a href=\"" + id->misc->imho->nextpage + "?actiongotoblock=1&msg="
	  + i + "\">" + sprintf ("[%d-%d]</a> ", (i + 1), (t + 1)) + mnrs;
      }
      i = sessobj->firstvisiblemail + sessobj->visible;

      while (i < mails) {
	int t = (((i + sessobj->visible) >= mails) ? (mails - 1) : (i + sessobj->visible - 1));
	mnrs += " <a href=\"" + id->misc->imho->nextpage + "?actiongotoblock=1&msg=";
	mnrs += ((string)i) + "\">" + sprintf ("[%d-%d]</a>", (i + 1), (t + 1));
	i += sessobj->visible;
      }

      mailnumbers += mnrs + " ";
      mnrs = "";
      if (sessobj->firstvisiblemail > 0) {
	mailnumbers += " <a href=\"" + id->misc->imho->nextpage + "?actionprevblock=1\">";
	mailnumbers += html_encode_string (MSGA (M_BACKN, ({ sessobj->visible }))) + "</a>";
      }
      if ((sessobj->lastvisiblemail + 1) < mails) {
	mailnumbers += " <a href=\"" + id->misc->imho->nextpage + "?actionnextblock=1\">";
	mailnumbers += html_encode_string (MSGA (M_FORWARDN, ({ sessobj->visible }))) + "</a>";
      }
      mailnumbers += "</td></tr>\n</table>";
    }
      
    if (!id->misc->imho->notopbuttons)
      page += mailnumbers + buttons;
      
    // the index itself
    page += "<table width=\"100%\"><tr>";

    array columns;
    if (!id->misc->imho->columns)
      columns = ({ "num", "mark", "date", "fromto", "subject" });
    else {
      columns = Array.map (id->misc->imho->columns / ",", lower_case);
      if (!has_value (columns, "mark"))
	columns = ({ "mark" }) + columns;
      if (!has_value (columns, "subject"))
	columns += ({ "subject" });
    }

    columns = replace (columns,
		       "fromto",
		       ((sessobj->mailbox[MB_DISPLAYNAME_IDX] == sessobj->sentfolder) || in_drafts) ? "to" : "from");

    string sortarrow = "<a href=\"" + id->misc->imho->nextpage + "?actiontogglesortorder=1\">";
    sortarrow += "<imho_image fg=\"&imho_thcolor;\" bg=\"&imho_thbgcolor;\" border=\"0\" src=";
    if (sessobj->sessionsortorder == "forward")
      sortarrow += "\"&imhoarrowdownsrc;\" image=\"arrowdown\"";
    else 
      sortarrow += "\"&imhoarrowupsrc;\" image=\"arrowup\"";
    sortarrow += " alt=\"" + MSG(M_CHANGESORTORDER) + "\" size=\"&imhoarrowsize;\" scale=\"0.25\" /></a>";
 
    foreach (columns, string column) {
      switch (column) {

      case "num":
	page += "<th class=\"num\" bgcolor=\"&imho_thbgcolor;\" valign=\"middle\">";
	if (sessobj->sortcolumn == column)
	  page += sortarrow;
	else
	  page +=
	    "<a href=\"" + id->misc->imho->nextpage + "?actionchangesort=1" +
	    "&col=" + column + "\"><imho_image fg=\"&imho_thcolor;\"" +
	    " bg=&imho_thbgcolor; border=0" +
	    " image=\"arrownone\" alt=\"" + MSG(M_CHANGESORTORDER) +
	    "\" size=\"&imhoarrowsize;\" scale=\"0.25\" /></a>";
	page += "<font class=\"num\" color=\"&imho_thcolor;\">" + MSG(M_NUMBER) + "</font></th>";
	break;

      case "mark":
	page += "<th class=\"mark\" bgcolor=\"&imho_thbgcolor;\" valign=\"middle\">";
	page += "<font class=\"mark\" color=\"&imho_thcolor;\">" + MSG(M_MARKFLAG) + "</font></th>";
	break;

      case "new":
	page += "<th class=\"new\" bgcolor=\"&imho_thbgcolor;\" valign=\"middle\">";
	page += "<font class=\"new\" color=\"&imho_thcolor;\">" + MSG(M_NEWFLAG) + "</font></th>";
	break;

      case "answered":
	page += "<th class=\"answered\" bgcolor=\"&imho_thbgcolor;\" valign=\"middle\">";
	page += "<font class=\"answered\" color=\"&imho_thcolor;\">" + MSG(M_ANSWEREDFLAG) + "</font></th>";
	break;

      case "date":
	page += "<th class=\"date\" bgcolor=\"&imho_thbgcolor;\" valign=\"middle\">";
	if (sessobj->sortcolumn == column)
	  page += sortarrow;
	else
	  page +=
	    "<a href=\"" + id->misc->imho->nextpage + "?actionchangesort=1" +
	    "&col=" + column + "\"><imho_image fg=\"&imho_thcolor;\"" +
	    " bg=\"&imho_thbgcolor;\" border=\"0\"" +
	    " image=\"arrownone\" alt=\"" + MSG(M_CHANGESORTORDER) +
	    "\" size=\"&imhoarrowsize;\" scale=\"0.25\" /></a>";
	page += "<font class=\"date\" color=\"&imho_thcolor;\">" + MSG(M_DATE);
	if(QUERY(featuredelay))
	  page += "&nbsp;(" + MSG(M_DELAY) + ")";
	page += "</font></th>";
	break;
      case "from":
	page += "<th class=\"from\" bgcolor=\"&imho_thbgcolor;\" width=\"20%\" valign=\"middle\">";
	if (sessobj->sortcolumn == column)
	  page += sortarrow;
	else
	  page +=
	    "<a href=\"" + id->misc->imho->nextpage + "?actionchangesort=1" +
	    "&col=" + column + "\"><imho_image fg=\"&imho_thcolor;\"" +
	    " bg=\"&imho_thbgcolor;\" border=\"0\"" +
	    " image=\"arrownone\" alt=\"" + MSG(M_CHANGESORTORDER) +
	    "\" size=\"&imhoarrowsize;\" scale=\"0.25\" /></a>";
	page += "<font class=\"from\" color=\"&imho_thcolor;\">" + MSG(M_FROM) + "</font></th>";
	break;

      case "to":
	page += "<th class=\"to\" bgcolor=\"&imho_thbgcolor;\" width=\"20%\" valign=\"middle\">";
	if (sessobj->sortcolumn == column)
	  page += sortarrow;
	else
	  page +=
	    "<a href=\"" + id->misc->imho->nextpage + "?actionchangesort=1" +
	    "&col=" + column + "\"><imho_image fg=\"&imho_thcolor;\"" +
	    " bg=\"&imho_thbgcolor;\" border=\"0\"" +
	    " image=\"arrownone\" alt=\"" + MSG(M_CHANGESORTORDER) +
	    "\" size=\"&imhoarrowsize;\" scale=\"0.25\" /></a>";
	page += "<font class=\"to\" color=\"&imho_thcolor;\">" + MSG(M_TO) + "</font></th>";
	break;

      case "subject":
	page += "<th class=\"subject\" bgcolor=\"&imho_thbgcolor;\" width=\"60%\" valign=\"middle\">";
	if (sessobj->sortcolumn == column)
	  page += sortarrow;
	else
	  page +=
	    "<a href=\"" + id->misc->imho->nextpage + "?actionchangesort=1" +
	    "&col=" + column + "\"><imho_image fg=\"&imho_thcolor;\"" +
	    " bg=\"&imho_thbgcolor;\" border=\"0\"" +
	    " image=\"arrownone\" alt=\"" + MSG(M_CHANGESORTORDER) +
	    "\" size=\"&imhoarrowsize;\" scale=\"0.25\" /></a>";
	page += "<font class=\"subject\" color=\"&imho_thcolor;\">" + MSG(M_SUBJECT) + "</font></th>";
	break;

      case "size":
	page += "<th class=\"size\" bgcolor=\"&imho_thbgcolor;\" valign=\"middle\">";
	if (sessobj->sortcolumn == column)
	  page += sortarrow;
	else
	  page +=
	    "<a href=\"" + id->misc->imho->nextpage + "?actionchangesort=1" +
	    "&col=" + column + "\"><imho_image fg=\"&imho_thcolor;\"" +
	    " bg=\"&imho_thbgcolor;\" border=\"0\"" +
	    " image=\"arrownone\" alt=\"" + MSG(M_CHANGESORTORDER) +
	    "\" size=\"&imhoarrowsize;\" scale=\"0.25\" /></a>";
	page += "<font class=\"size\" color=\"&imho_thcolor;\">" + MSG(M_SIZE) + "</font></th>";
	break;
      }
      page += "\n";
    }
    page += "</tr>\n";

    if (sessobj->sortcolumn == "num") { // range select only works in mailbox order

      if ((sessobj->firstvisiblemail > 0) && (sessobj->sessionsortorder == "forward")) {
	page += "<tr>";
	foreach (columns, string column) {
	  switch (column) {

	  case "num":
	    page += "<td class=\"num\" bgcolor=\"&imho_tdbgcolor;\">1-";
	    page += (string)((int)sessobj->firstvisiblemail) + "</td>";
	    break;

	  case "mark":
	    page += "<td class=\"mark\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += "<input type=\"checkbox\" name=\"msg";
	    page += (string)sessobj->mails[0]->imap->UID + "-";
	    page += sessobj->mails[(int)sessobj->firstvisiblemail - 1]->imap->UID;
	    page += "\" value=\"1\" /></td>";
	    break;
	    
	  case "subject":
	    page += "<td class=\"subject\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += MSGA (M_MAILSHOWN, ({ 1, sessobj->firstvisiblemail, mails })) + "</td>";
	    break;
	     
	  default:
	    page += "<td class=\"" + column + "\">&nbsp;</td>";
	    break;
	  }
	  page += "\n";
	}
	page += "</tr>\n";
      }
	
      if (((sessobj->lastvisiblemail + 1) < mails) && (sessobj->sessionsortorder != "forward")) {
	page += "<tr>";
	foreach (columns, string column) {
	  switch (column) {

	  case "num":
	    page += "<td class=\"num\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += (string)((int)sessobj->lastvisiblemail + 2) + "-";
	    page += (string)((int)mails) + "</td>";
	    break;

	  case "mark":
	    page += "<td class=\"mark\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += "<input type=\"checkbox\" name=\"msg";
	    page += sessobj->mails[((int)sessobj->lastvisiblemail + 1)]->imap->UID + "-";
	    page += sessobj->mails[((int)mails - 1)]->imap->UID + "\" value=\"1\" /></td>";
	    break;
	      
	  case "subject":
	    page += "<td class=\"subject\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += MSGA (M_MAILSHOWN, ({ sessobj->lastvisiblemail + 2, mails, mails })) + "</td>";
	    break;

	  default:
	    page += "<td class=\"" + column + "\">&nbsp;</td>";
	    break;
	  }
	  page += "\n";
	}
	page += "</tr>\n";
      }
    }

    int frommail, tomail, deltamail;
    if (sessobj->sessionsortorder == "forward") {
      frommail = sessobj->firstvisiblemail;
      tomail = sessobj->lastvisiblemail + 1;
      deltamail = 1;
    }
    else {
      tomail = sessobj->firstvisiblemail - 1;
      frommail = sessobj->lastvisiblemail;
      deltamail = -1;
    }

    for (int i = frommail; i != tomail; i+=deltamail) {
      object mail = sessobj["mails"][i];
      page += "<tr>";
      // New mail?
      int unseen = !has_value (mail->imap->FLAGS, "\\Seen");
      foreach (columns, string column) {
	switch (column) {
	case "num": 
	  page += "<td class=\"" + (unseen ? "h" : "") + "num\" bgcolor=\"&imho_tdbgcolor;\"><b>";
	  page += (string)(sessobj->mails[i]->number + 1) + "</b></td>";
	  break;
	      
	case "mark":
	  page += "<td class=\"" + (unseen ? "h" : "") + "mark\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += "<input type=\"checkbox\" name=\"msg";
	  page += (string)sessobj->mails[i]->imap->UID + "\" value=\"1\" />";
	  page += "</td>";
	  break;

	case "new":
	  page += "<td class=\"" + (unseen ? "h" : "") + "new\" bgcolor=\"&imho_tdbgcolor;\" nowrap>";
	  page += (unseen ? 
		   "<imho_image src=\"&imhochecksrc;\" fg=&imhocheckfg; bg=&imho_tdbgcolor; "
		   "size=&imhochecksize; scale=\"0.25\" image=\"ncheck\" alt=\"X\" />" : " ") + "</td>";
	  break;

	case "answered":
	  // Answered?
	  page += "<td class=\"" + (unseen ? "h" : "") + "answered\" bgcolor=\"&imho_tdbgcolor;\" nowrap>";
	  page += (has_value (mail->imap->FLAGS, "\\Answered") ?  
		   "<imho_image src=\"&imhochecksrc;\" fg=&imhocheckfg; bg=&imho_tdbgcolor; "
		   "size=&imhochecksize; scale=\"0.25\" image=\"rcheck\" alt=\"X\" />" : " ") + "</td>";
	  break;

	case "date":
	  page += "<td class=\"" + (unseen ? "h" : "") + "date\" bgcolor=\"&imho_tdbgcolor;\" nowrap>";
	  page += (unseen ? "<b>" : "");
	  page += mailindex_date (sessobj, mail->imap->ENVELOPE[DATE_IDX], mail->imap->INTERNALDATE);
	  if(QUERY(featuredelay))
	    page += mailindex_delay (sessobj, mail->imap->ENVELOPE[DATE_IDX], mail->imap->INTERNALDATE);	  
          page += (unseen ? "</b>" : "") + "</td>";
	  break;

	case "from":
	  // From
	  string from_name, from_addr;
	  if (mail->imap->ENVELOPE[FROM_IDX]) {
	    from_name = mail->imap->ENVELOPE[FROM_IDX][0][0];
	    from_addr = mail->imap->ENVELOPE[FROM_IDX][0][2] + "@" + mail->imap->ENVELOPE[FROM_IDX][0][3];
	    from_addr = replace (from_addr, "@.MISSING-HOST-NAME.", "");
	  }
	  else 
	    from_name = "-";

	  from_name = fix_header (from_name ? from_name : from_addr);
	  if (QUERY (frommaxlength))
	    from_name = CAMAS.Tools.scut(from_name,QUERY(frommaxlength));

	  page += "<td class=\"" + (unseen ? "h" : "") + "from\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += (unseen ? "<b>" : "") + from_name + (unseen ? "</b>" : "");
	  page += "</td>";
	  break;
	      
	case "to":
	  // To
	  string to_name, to_addr;
	  if (mail->imap->ENVELOPE[TO_IDX]) {
	    to_name = mail->imap->ENVELOPE[TO_IDX][0][0];
	    to_addr = mail->imap->ENVELOPE[TO_IDX][0][2] + "@" + mail->imap->ENVELOPE[TO_IDX][0][3];
	    to_addr = replace (to_addr,"@.MISSING-HOST-NAME.","");	      
	  }
	  else 
	    to_name = "-";

	  to_name = fix_header (to_name ? to_name : to_addr);
	  if (QUERY (frommaxlength))
	    to_name = CAMAS.Tools.scut(to_name,QUERY (frommaxlength));

	  page += "<td class=\"" + (unseen ? "h" : "");
	  page += "to\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += (unseen ? "<b>" : "") + to_name + (unseen ? "</b>" : "") + "</td>";
	  break;

	case "subject":
	  // Subject
	  page += "<td class=\"" + (unseen ? "h" : "");
	  page += "subject\" bgcolor=\"&imho_tdbgcolor;\">" + (unseen ? "<b>" : "");
	  page += "<a target=\"" + id->misc->imho->uptarget + "\" href=\"" + id->misc->imho->nextuptarget;
	  page += "?actionread=1&msguid=" + (string)sessobj->mails[i]->imap->UID;
	  page += "&mbox=" + http_encode_url (sessobj->mailbox[MB_FOLDERNAME_IDX]) + "\"";

	  string subject = fix_header (mail->imap->ENVELOPE[SUBJECT_IDX]);
	  if (!strlen (subject))
	    subject = "?";

	  if (QUERY (subjectmaxlength))
	    subject = CAMAS.Tools.scut(subject,QUERY(subjectmaxlength));

	  if (QUERY (subjectlayer) && id->supports->javascript) {
	    page += " onmouseover=\"window.status='';";
	    if (id->supports->msie) {
	      page += " document.all.laymsg" + (string)sessobj->mails[i]->imap->UID + ".style.visibility = 'visible';";
	      page += " document.all.laymsg" + (string)sessobj->mails[i]->imap->UID + ".style.left = event.x - 150;";
	      page += " document.all.laymsg" + (string)sessobj->mails[i]->imap->UID + ".style.top = event.y - 100;\"";
	      page += " onmouseout=\"document.all.laymsg" + (string)sessobj->mails[i]->imap->UID;
	      page += ".style.visibility = 'hidden';\"";
	    }
	    else {
	      page += " document.layers['laymsg" + (string)sessobj->mails[i]->imap->UID + "'].visibility = 'visible';";
	      page += " document.layers['laymsg" + (string)sessobj->mails[i]->imap->UID + "'].left = Xpos - 150;";
	      page += " document.layers['laymsg" + (string)sessobj->mails[i]->imap->UID + "'].top = Ypos - 100;\"";
	      page += " onmouseout=\"document.layers['laymsg" + (string)sessobj->mails[i]->imap->UID;
	      page += "'].visibility = 'hidden';\"";
	    }
	  }
	  page += ">";

	  page += subject;
	  page += "</a>" + (unseen ? "</b>" : "") + "</td>";
	  break;

	case "size":
	  // Size
	  page += "<td class=\"" + (unseen ? "h" : "") + "size\" bgcolor=\"&imho_tdbgcolor;\">";
	  page += (unseen ? "<b>" : "") + ((string)(((int)mail->imap["RFC822.SIZE"]) / 1024)) + " K";
	  page += (unseen ? "</b>" : "") + "</td>";
	  break;
	}
	page += "\n";
      }
      page += "</tr>\n";
    }
      
    if (sessobj->sortcolumn == "num") { // range select only works in mailbox order
      if ((sessobj->firstvisiblemail > 0) && (sessobj->sessionsortorder != "forward")) {
	page += "<tr>";
	foreach (columns, string column) {
	  switch (column) {

	  case "num":
	    page += "<td class=\"num\" bgcolor=\"&imho_tdbgcolor;\">1-";
	    page += (string)((int)sessobj->firstvisiblemail) + "</td>";
	    break;

	  case "mark":
	    page += "<td class=\"mark\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += "<input type=\"checkbox\" name=\"msg" + (string)sessobj->mails[0]->imap->UID + "-";
	    page += (string)sessobj->mails[(int)sessobj->firstvisiblemail - 1]->imap->UID + "\" value=\"1\" /></td>";
	    break;

	  case "subject":
	    page += "<td class=\"subject\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += MSGA (M_MAILSHOWN, ({ 1, sessobj->firstvisiblemail, mails })) + "</td>";
	    break;

	  default:
	    page += "<td class=\"" + column + "\">&nbsp;</td>";
	    break;
	  }
	  page += "\n";
	}
	page += "</tr>\n";
      }

      if (((sessobj->lastvisiblemail + 1) < mails) && (sessobj->sessionsortorder == "forward")) {
	page += "<tr>";
	foreach (columns, string column) {
	  switch (column) {

	  case "num":
	    page += "<td class=\"num\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += (string)((int) sessobj->lastvisiblemail + 2) + "-" + (string)((int)mails) + "</td>";
	    break;
	    
	  case "mark":
	    page += "<td class=\"mark\" bgcolor=\"&imho_tdbgcolor;\"><input type=\"checkbox\" name=\"msg";
	    page += (string)sessobj->mails[(int) sessobj->lastvisiblemail + 1]->imap->UID + "-";
	    page += (string)sessobj->mails[(int)mails - 1]->imap->UID + "\" value=\"1\" /></td>";
	    break;
	    
	  case "subject":
	    page += "<td class=\"subject\" bgcolor=\"&imho_tdbgcolor;\">";
	    page += MSGA (M_MAILSHOWN, ({ sessobj->lastvisiblemail + 2, mails, mails })) + "</td>";
	    break;

	  default:
	    page += "<td class=\"" + column + "\">&nbsp;</td>";
	    break;
	  }
	  page += "\n";
	}
	page += "</tr>\n";
      }
    }

    page += "</table>";

    if (id->supports->javascript) {
      page += "<input type=\"checkbox\" name=\"all\" onClick=\"";
      page += " for(var i=0;i<this.form.elements.length;i++)";
      page += " {var inpt=this.form.elements[i];if(inpt.name.substr(0,3)=='msg')";
      page += " inpt.checked=this.form.all.checked}\" />" + MSG (M_MARKALL);
    }
    page += "<br />\n";

    if (!id->misc->imho->nobottombuttons) {
      page += mailnumbers;
      page += buttons;
      /*page += replace (buttons, 
	({ "name=\"mbox\"", "name=\"actionmove\""  }), 
	({ "name=\"mbox2\"","name=\"actionmove2\"" }));*/
    }

    page += "</form>";

    if (QUERY (subjectlayer) && id->supports->javascript) {
      //oliv3 experimental layers support
      if (!id->supports->msie) {
	page += "<script language=\"JavaScript\" type=\"text/javascript\">\n<!--\n";
	page += "document.captureEvents(Event.MOUSEMOVE);\n"
	  "document.onMouseMove = MoveHandler;\n"
	  "function MoveHandler(e) {\n"
	  "Xpos = e.pageX;\n"
	  "Ypos = e.pageY;"
	  "return true;\n"
	  "}\n";
	page += "// -->\n</script>\n";
      }

      for (int i = frommail; i != tomail; i+=deltamail) {
	object mail = sessobj->mails[i];

	string subject = fix_header (mail->imap->ENVELOPE[SUBJECT_IDX]);
	if (!strlen (subject))
	  subject = "?";

	page += "<div id=\"laymsg";
	page += (string)mail->imap->UID + "\"";
	page += " style=\"position: absolute; z-index: 10; visibility: hidden;";
	page += " top: 0px; left: 0px;\">";
	page += "<table bgcolor=black><tr><td bgcolor=\"yellow\">" + subject + "</td></tr></table>";
	page += "</div>\n";
      }
    }
  }
  else {
    page += "<br />\n" + MSG (M_NOMAILS) + "<br />";
    page += formdrawbutton ("m_getnewmail", "actionreload", MSG(M_CHECKNEWMAIL));
    page += "</form>";
  }

  //page += "<hr /><div align=left><imho_dumpid></div>";
  return page;
}

// ------------------ Search mail screen

string camas_screen_searchmail (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "searchmail";
  id->misc->imho->title = MSG(M_SEARCHMAILTITLE);
      
  page += "<form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget + "\">";
  page += "<center><table><th bgcolor=\"&imho_thbgcolor;\"><font color=\"&imho_thcolor;\">";
  page += MSG(M_SEARCHMAILTITLE) + "</font></th>";
  page += "<tr><td bgcolor=\"&imho_tdbgcolor;\">";
  page += MSGA(M_SEARCHMAILBOX, ({ sessobj->mailbox[MB_DISPLAYNAME_IDX] }));
  page += "</td></tr>";
  if (sessobj->searchstring && sizeof(sessobj->searchstring) > 0) {
    page += "<tr><td bgcolor=\"&imho_tdbgcolor;\">";
    page += sessobj->searchstring;
    page += "</td></tr>";
  }
  //page +="<br />";
  page += "<tr><td bgcolor=\"&imho_tdbgcolor;\">";
  page += "<input name=\"text1\" size=\"20\" />";
  page += "<select name=\"searchfield1\">";
  page += "<option value=\"text\" selected>" + MSG(M_SEARCHANYWHERE) + "</option>";
  page += "<option value=\"to\">" + MSG(M_SEARCHTOFIELD) + "</option>";
  page += "<option value=\"from\">" + MSG(M_SEARCHFROMFIELD) + "</option>";
  page += "<option value=\"subject\">" + MSG(M_SEARCHSUBJECT) + "</option>";    
  page += "<option value=\"body\">" + MSG(M_SEARCHBODY) + "</option>";
  page += "</select><br /><br />";
  page += formdrawbutton ("m_searchmail", "actionsearchmail", MSG(M_SEARCHMAIL));
  page += "</td></tr></table></form>";

  return page;
}

// ------------------ Folder list screen

string camas_screen_folderlist (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  string fl_layfolderstop = (sizeof (QUERY (layfolderstop)) ? QUERY (layfolderstop) : DEFAULT_LAYFOLDERSTOP);
  string fl_layfoldersrow = (sizeof (QUERY (layfoldersrow)) ? QUERY ( layfoldersrow) : DEFAULT_LAYFOLDERSROW);
  string fl_layfoldersbottom = (sizeof (QUERY (layfoldersbottom)) ? QUERY (layfoldersbottom) : DEFAULT_LAYFOLDERSBOTTOM);

  id->misc->imho->screen = "folderlist";
  id->misc->imho->title = MSG(M_MAILBOXES);

  string path = id->variables->path || sessobj->lastpath;
  //write ("path= " + path + "\n");
  string pathname;
  int deletable_folders = 0;
  int can_create_folders = 1;
  int mbox_idx = Array.search_array (sessobj->mailboxes,
				     lambda (array a, string path) {
				       if (a[MB_FOLDERNAME_IDX] == path)
					 return 1;
				       return 0;
				     },
				     path); 
  array mbox_path = ({ });

  if (mbox_idx == -1) {
    path = "";
    pathname = "";
  }
  else {
    pathname = sessobj->mailboxes[mbox_idx][MB_DISPLAYNAME_IDX];
    mbox_path = sessobj->mailboxes[mbox_idx][MB_HIERARCHY_IDX];
    if (sessobj->mailboxes[mbox_idx][MB_FLAGS_IDX] & MB_IMPLICIT_FOLDER)
      can_create_folders = 0;
  }
  sessobj->lastpath = path;

  //write (sprintf ("sessobj->mailboxes= %O\n", sessobj->mailboxes));

  array mboxes_to_show = Array.filter (sessobj->mailboxes,
				       lambda (array a, array mbox_path) {
					 if (sizeof (a[MB_HIERARCHY_IDX]) != (sizeof (mbox_path) + 1)) {
					   //write ("test 1 failed: " + sizeof (a[MB_HIERARCHY_IDX])
					   //+ " != " + (sizeof (mbox_path) + 1) + "\n");
					   return 0;
					 }
					 for (int i = 0; i < sizeof (mbox_path); i++)
					   if (mbox_path[i] != a[MB_HIERARCHY_IDX][i]) {
					     //write ("test 2 failed: " + mbox_path[i]
					     //     + " != " + a[MB_HIERARCHY_IDX][i] + "\n");
					     return 0;
					   }
					 return 1;
				       },
				       mbox_path); 

  if (path != "") {
    mboxes_to_show = 
      ({ ({ "", (sessobj->mailboxes[mbox_idx][MB_HIERARCHY_IDX][0..sizeof(sessobj->mailboxes[mbox_idx][MB_HIERARCHY_IDX])-2]) * sessobj->mailboxes[mbox_idx][MB_SEPARATOR_IDX], MB_NOSELECT | MB_PREVIOUS_FOLDER, "", ({ MSG(M_PREVIOUS_LEVEL) }) }) }) + mboxes_to_show;
  }

  //write (sprintf ("mboxes_to_show= %O\n", mboxes_to_show));

  // stene FIXME: merge mailboxes if "folderstyle" is "mailandfolders"

  page += "<form method=\"post\" action=\""+id->misc->imho->nextpage+"\">";
  page += replace (fl_layfolderstop, ({ "$PATH" }), ({ html_encode_string (path) }));

  array imho_from = ({ "$MARK", "$NAME", "$MESSAGES", "$RECENT", "$UNSEEN", "$SIZE" });

  int i=0;
  int i2=-1;
  foreach (mboxes_to_show, array mbox_a) {
    string row_mark="", row_name="", row_messages="", row_recent="", row_unseen="", row_size="";

    i2++;
    int deletable=!((mbox_a[MB_FLAGS_IDX] & MB_PREVIOUS_FOLDER) ||
		    mbox_a[MB_FOLDERNAME_IDX] == "INBOX" ||
		    mbox_a[MB_FOLDERNAME_IDX] == (sessobj->mailpath + "sent-mail") ||
		    mbox_a[MB_FOLDERNAME_IDX] == (sessobj->mailpath + "Trash") ||
		    mbox_a[MB_FOLDERNAME_IDX] == (sessobj->mailpath + "Drafts") ||
		    mbox_a[MB_FOLDERNAME_IDX] == sessobj->prefsbox ||
		    (mbox_a[MB_FLAGS_IDX] & MB_IMPLICIT_FOLDER) ||
		    // folder has children?
		    (Array.search_array (sessobj->mailboxes,
					lambda(array a,array mbox_path) {
					  if (sizeof(a[MB_HIERARCHY_IDX]) <= sizeof (mbox_path))
					    return 0;
					  int i;
					  for(i=0;i<sizeof(mbox_path);i++)
					    if(mbox_path[i]!=a[MB_HIERARCHY_IDX][i])
					      return 0;
					  return 1;
					},
					mbox_a[MB_HIERARCHY_IDX]) 
		     != -1 )
			 
		    );
      
    deletable_folders |= deletable;

    if(! (mbox_a[MB_FLAGS_IDX] & (MB_NOSELECT | MB_IMPLICIT_FOLDER))) {

      string mbox_name=mbox_a[MB_HIERARCHY_IDX][-1];
      mbox_name = sessobj->imapclient->translate_frommbox(sessobj,mbox_name);

      if(deletable)
	row_mark = "<input type=\"checkbox\" name="+html_encode_tag_value("delf_"+mbox_a[MB_FOLDERNAME_IDX])+" value=\"1\">";
      else
	row_mark = "&nbsp;";

      row_name = "<a class=\"mailbox\" target=\"" + id->misc->imho->target + "\" href=\"";
      row_name += id->misc->imho->nexttarget + "?actionmailindex=1&mbox=";
      row_name += http_encode_url (mboxes_to_show[i2][MB_FOLDERNAME_IDX]) + "\">";
      row_name += html_encode_string (mbox_name) + "</a>";

      string mbname = mbox_a[MB_FOLDERNAME_IDX];
      if (QUERY (foldersinfo)) {
	row_messages = (string)sessobj->foldersinfo[mbname][MB_MESSAGES_IDX];
	row_recent = (string)sessobj->foldersinfo[mbname][MB_RECENT_IDX];
	row_unseen = (string)sessobj->foldersinfo[mbname][MB_UNSEEN_IDX];
	row_size = (string)(sessobj->foldersinfo[mbname][MB_SIZE_IDX] >> SIZE_SHIFT);
      }

      array imho_to = ({ row_mark, row_name, row_messages, row_recent, row_unseen, row_size });

      page += replace (fl_layfoldersrow, imho_from, imho_to);

      i++;
    }

    if(! (mbox_a[MB_FLAGS_IDX] & (MB_NOINFERIORS))) {

      string mbox_name=mbox_a[MB_HIERARCHY_IDX][-1];
      if (mbox_a[MB_FOLDERNAME_IDX] == "INBOX")
	mbox_name = MSG(M_INBOX);
	
      if(deletable)
	row_mark = "<input type=\"checkbox\" name=\""+html_encode_tag_value("delf_"+mbox_a[MB_FOLDERNAME_IDX])+"\" value=\"1\">";
      else
	row_mark = "&nbsp;";

      string mbname = mbox_a[MB_FOLDERNAME_IDX];
      if (mbname != "") {
	row_name = "<a class=\"mailbox\" target=\"" + id->misc->imho->target + "\" href=\"";
	row_name += id->misc->imho->nexttarget + "?actionfolderlistshow=1&path=";
	row_name += http_encode_url (mbox_a[MB_FOLDERNAME_IDX]) + "\">";
	row_name += "<img border=\"0\" src=\"&imho_folder;\" alt=\"\"> ";
	row_name += html_encode_string (mbname) + "</a>";

	if (QUERY (foldersinfo)) {
	  row_messages = (string)sessobj->foldersinfo[mbname][MB_MESSAGES_IDX];
	  row_recent = (string)sessobj->foldersinfo[mbname][MB_RECENT_IDX];
	  row_unseen = (string)sessobj->foldersinfo[mbname][MB_UNSEEN_IDX];
	  row_size = (string)(sessobj->foldersinfo[mbname][MB_SIZE_IDX] >> SIZE_SHIFT);
	}
      }
      else {
	row_name = "<a class=\"mailbox\" target=\"" + id->misc->imho->target + "\" href=\"";
	row_name += id->misc->imho->nexttarget + "?actionfolderlistshow=1&path=";
	//oliv3: was ok
	//row_name += http_encode_url (mbox_a[MB_DISPLAYNAME_IDX]) + "\">";
	if (stringp (mbox_a[MB_DISPLAYNAME_IDX]))
	  row_name += http_encode_url (mbox_a[MB_DISPLAYNAME_IDX]);
	row_name += "\">";


	row_name += "<img border=\"0\" src=\"&imho_folder;\" alt=\"\"> " + mbox_name + "</a>";
      }

      array imho_to = ({ row_mark, row_name, row_messages, row_recent, row_unseen, row_size });

      page += replace (fl_layfoldersrow, imho_from, imho_to);

      i++;
    }
  }

  string delete_button = "";
  if (deletable_folders)
    delete_button = formdrawbutton ("m_deletemarkedmbox", "actiondeletefolder", MSG(M_DELETEMARKEDMBOX));

  string create_newname = "", create_button = "";
  if (can_create_folders) {
    create_button = formdrawbutton ("m_creatembox", "actioncreatefolder", MSG(M_CREATEMBOX));
    create_newname = "<input name=\"foldername\" value=\"" + MSG(M_NEWMBOXNAME) + "\"";
    if (id->supports->javascript)
      create_newname += " onChange=\"this.form.actioncreatefolder.click()\"";
    create_newname += ">";
  }
    
  string rename_select = "", rename_button = "", rename_newname = "";
  rename_select = "<select name=\"oldfoldername\"><option value=\"imhonomailbox\">";
  rename_select += MSG(M_SELECTMBOX) + "</option>\n";
  for (int i = 0; i < sizeof (sessobj->mailboxes); i++) {
    if (sessobj->mailbox[MB_FOLDERNAME_IDX] != sessobj->mailboxes[i][MB_FOLDERNAME_IDX]) 
      if (!(sessobj->mailboxes[i][MB_FLAGS_IDX] & MB_NOSELECT)) {
	int ok = 1;
	string mbox = sessobj->mailboxes[i][MB_DISPLAYNAME_IDX];
	if (sessobj->mailboxes[i][MB_FOLDERNAME_IDX] == "INBOX") ok = 0;
	else if (sessobj->mailboxes[i][MB_FOLDERNAME_IDX] == (sessobj->mailpath + "sent-mail")) ok = 0;
	else if (sessobj->mailboxes[i][MB_FOLDERNAME_IDX] == (sessobj->mailpath + "Trash")) ok = 0;
	else if (sessobj->mailboxes[i][MB_FOLDERNAME_IDX] == (sessobj->mailpath + "Drafts")) ok = 0;
	else if (sessobj->mailboxes[i][MB_FOLDERNAME_IDX] == sessobj->prefsbox) ok = 0;
	if (ok) {
	  rename_select += "<option value=" + html_encode_tag_value(sessobj->mailboxes[i][MB_FOLDERNAME_IDX]);
	  rename_select += ">" + html_encode_string (mbox) + "</option>\n";
	}
      }
  }
  rename_select += "</select>";
  rename_button = formdrawbutton ("m_renamembox", "actionrenamefolder", MSG(M_RENAMEMBOX));

  rename_newname = "<input name=\"newfoldername\" value=\"" + MSG(M_NEWMBOXNAME) + "\"";
  if (id->supports->javascript)
    rename_newname += " onChange=\"this.form.actionrenamefolder.click()\"";
  rename_newname += ">";

  imho_from = ({ 
    "$NEWFOLDERNAME", "$CREATE", // create folder
    "$OLDFOLDERNAME", "$RENAMEFOLDERNAME", "$RENAME", // rename folder
    "$DELETE" // delete folder
  });

  array imho_to = ({ 
    create_newname, create_button, // create folder
    rename_select, rename_newname, rename_button, // rename folder
    delete_button // delete folder
  });
    
  string bottom = fl_layfoldersbottom;
  page += replace (bottom, imho_from, imho_to);

  page += "<input type=\"hidden\" name=\"path\" value=\"" + path + "\">";
  page += "</form>";

  //--oliv3
  //page += "<hr /><imho_dumpid>";
  return page;
}

// ------------------ Mail Delivery Notification screen

string camas_screen_mdndialog (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  object mail = sessobj->cmail;
  sessobj->imapclient->imap_command (IMHO, id, sessobj, ({ imap_cmd ("add_flag", "uid", sessobj->mails[sessobj->cmailidx]->imap->UID, "flag", IMAP_MDN_FLAG, "stop", 1) }));
  sessobj->mails[sessobj->cmailidx]->imap->FLAGS |= ({ IMAP_MDN_FLAG });
  string notify_to = mail->headers["disposition-notification-to"] || mail->headers["return-receipt-to"];

  //write ("AR: this mail needs a reply to: " + notify_to + "\n");
  page += "<br /><form method=\"post\" action=\"" + id->misc->imho->nextpage + "\">";
  page += "<div align=\"center\">";
  page += MSGA (M_NOTIFYTO, ({ fix_header (notify_to) }));
  page += "<br />";
  page += formdrawbutton ("m_sendmdn", "actionsendmdn", MSG(M_YES));
  page += formdrawbutton ("m_read", "actionreadmail", MSG(M_NO));
  page += "</form></div>";

  return page;
}

// ------------------ Read Mail screen

string camas_screen_readmail (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  int in_drafts = (sessobj->mailbox[MB_FOLDERNAME_IDX] == (QUERY (defaultmailpath)+sessobj->draftsfolder));
  if (!in_drafts)
    sessobj->draftuid = 0;

  id->misc->imho->screen = "readmail";
  id->misc->imho->title = MSG(M_MAIL);
  
  sessobj->cmailidx = find_mail (sessobj->mails, sessobj->cmailuid);
  object mail = sessobj->cmail;
  
  if (!mail || (sessobj->cmailidx < 0)) {
    page += "<div align=\"center\"><br />";
    page += MSG (M_MAILMISSING);
    if (id->misc->imho->nobackbutton) {
      //page+="<br />\n<form method=\"post\" target=\"_top\" action=\""+server_url(id)+QUERY (location)+"_top\">"
      page += "<br />\n<form method=\"post\" target=\"" + id->misc->imho->target;
      page += "\" action=\""+id->misc->imho->nexttarget+"\">";
      page += "<input type=\"hidden\" name=\"actionindex\" value=\"1\">";
      page += "<input type=\"hidden\" name=\"mbox\" value=\"" + sessobj->mailbox[MB_FOLDERNAME_IDX] + "\">";
      page += formdrawbutton ("m_mailmissingback", "\"\"", MSG (M_MAILMISSINGBACK));
      page += "</form></div>";
    }
    return page;
  }
  
  if (!in_drafts && (mail->headers["disposition-notification-to"] || mail->headers["return-receipt-to"])
      && (!has_value (sessobj->mails[sessobj->cmailidx]->imap->FLAGS, IMAP_MDN_FLAG)))
    return create_main (id, MDNDIALOG);
  
  if (sizeof(sessobj->mails) > sessobj->cmailidx && !has_value ((sessobj->mails[sessobj->cmailidx]->imap->FLAGS), "\\Seen"))
    sessobj->mails[sessobj->cmailidx]->imap->FLAGS += ({ "\\Seen" });

  if (!mail->headers->to)
    mail->headers->to = "?"; 
  if (!mail->headers->from)
    mail->headers->from = "?"; 
  
  int nextuid = -1;
  int prevuid = -1;
  if (sizeof(sessobj->mails)>sessobj->cmailidx + 1)
    nextuid=((int)sessobj->mails[(sessobj->cmailidx) + 1]->imap->UID);
  if (sessobj->cmailidx>0)
    prevuid=((int)sessobj->mails[(sessobj->cmailidx) - 1]->imap->UID);
  
  string buttons = "\n<table border=\"0\" cellpadding=\"0\" cellspacing=\"0\"><tr>";
  buttons += "<td><form target=\""+id->misc->imho->uptarget+"\" method=\"post\" action=\""+id->misc->imho->nextuptarget+"\">";

  if (in_drafts)
    // Mail from draft
    buttons += formdrawbutton ("m_compose", "actioncomposedraft", MSG(M_CONTINUECOMPOSE));
  else {
    // Reply
    buttons += formdrawbutton ("m_reply", "actionreply", MSG(M_REPLY));
    // Reply to all
    buttons += formdrawbutton ("m_replytoall", "actionreplytoall", MSG(M_REPLYTOALL));
    // Forward
    buttons += formdrawbutton ("m_forward", "actionforward", MSG(M_FORWARD));
  }
  buttons += "<input type=\"hidden\" name=\"mbox\" value=";
  buttons += html_encode_tag_value (sessobj->mailbox[MB_FOLDERNAME_IDX]) + " />";
  buttons += "<input type=\"hidden\" name=\"msg" + sessobj->cmailuid + "\" value=\"1\" />";
  buttons += "</form></td><td>";
  // buttons+="<form method=\"post\" action=\"" + id->misc->imho->nextpage + "\">";
  buttons += "<form target=\"" + id->misc->imho->uptarget;
  buttons += "\" method=\"post\" action=\"" + id->misc->imho->nextuptarget + "\">";
  buttons += "<input type=\"hidden\" name=\"mbox\" value=";
  buttons += html_encode_tag_value (sessobj->mailbox[MB_FOLDERNAME_IDX]) + " />";
  buttons += "<input type=\"hidden\" name=\"msg" + sessobj->cmailuid + "\" value=\"1\" />";
  
  buttons += "<input type=\"hidden\" name=\"prevuid\" value=\"" + prevuid + "\">";
  buttons += "<input type=\"hidden\" name=\"nextuid\" value=\"" + nextuid + "\">";

  if (sessobj->mailbox[MB_DISPLAYNAME_IDX] != sessobj->trashfolder) {
    if (feature (FEAT_MAILBOXES) && sizeof (sessobj->trashfolder)
	&& (<"both","move to trash">)[QUERY (deletemethod)] )
      buttons += formdrawbutton ("m_movethistotrash", "actiontrashthis", MSG(M_MOVETHISTOTRASH));
    if (!feature (FEAT_MAILBOXES) || (QUERY (deletemethod) != "move to trash")
	|| !sizeof(sessobj->trashfolder))
      buttons += formdrawbutton ("m_delete", "actiondeletethis", MSG(M_DELETE));
  }
  buttons += "</form></td><td>";
  buttons += "<form method=\"post\" action=\"" + id->misc->imho->nextpage + "\">";
  //buttons+="<form method=\"post\" target=\""+id->misc->imho->uptarget+"\" action=\""+id->misc->imho->nextuptarget+"\">";
  buttons += "<input type=\"hidden\" name=\"prevuid\" value=\"" + prevuid + "\">";
  buttons += "<input type=\"hidden\" name=\"nextuid\" value=\"" + nextuid + "\">";

  if (QUERY (showhiddenheaders) && (sessobj->showhiddenheaders == "1")) {
    // Show/hide headers
    if (sessobj->showheaders)
      buttons += formdrawbutton ("m_hidefullheaders", "actionhideheaders", MSG(M_HIDEFULLHEADERS));
    else
      buttons += formdrawbutton ("m_showfullheaders", "actionshowheaders", MSG(M_SHOWFULLHEADERS));
  }
  
  // Read previous/next
  if (prevuid != -1)
    buttons += formdrawbutton ("m_readprev", "actionreadprev", MSG(M_READPREV));
  if (nextuid != -1)
    buttons += formdrawbutton ("m_readnext", "actionreadnext", MSG(M_READNEXT));

  buttons += "</form></td></tr></table>\n";

  int feature_addressbook = feature (FEAT_EDITADDRESSBOOK);
    
  if (!id->misc->imho->notopbuttons)
    page += buttons;

  // Formating headers :-)
  page += "<table border=\"0\">";
  page += "<tr><td><font class=\"header1\"><b>";
  if (feature_addressbook && QUERY (multipleselect))
    {
      page += "<a target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
      page += "?actionnewaddress=1&address=" + http_encode_url (mail->headers->from) + "&take=1\">";
    }
  page += MSG(M_FROM) + ":";
  if (feature_addressbook && QUERY (multipleselect))
    page += "</a>";
  page += "</b></font></TD><TD><font class=\"header2\"><b>";
  
  int address_links = (feature_addressbook && has_value (lower_case (QUERY (selectaddresses)), "from"));
  if (address_links)
    {
      string from = String.trim_whites (mail->headers->from);
      page += "<a target=\""+id->misc->imho->target+"\" href=\""+id->misc->imho->nexttarget;
      page += "?actionnewaddress=1&address="+http_encode_url(from)+"&take=1\">";
      page += fix_header (from);
      //write ("fix_header (" + from + ") -> " + fix_header (from) + "\n");
      page += "</a>";
    }
  else
    page += fix_header (mail->headers->from);
  
  // FIXME: isn't Image and Image.XFace allways included in Pike 7.x??
  // Not necessarily, but generally yes -- david
#if constant(Image.XFace.decode)
  if (mail->headers["x-face"])
    {
      page += "<img src=";
      page += html_encode_tag_value (id->misc->imho->nextpage + "?xface=" + http_encode_url (MIME.encode_base64 (mail->headers["x-face"])));
      page += " alt=\":-)\">";
    }
#endif
  page += "</b></font></TD></TR><TR><TD><font class=\"header1\"><b>";
  
  if (feature_addressbook && QUERY (multipleselect))
    {
      page += "<a target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
      page += "?actionnewaddress=1&address="+http_encode_url(mail->headers->to)+"&take=1\">";
    }
  page += MSG(M_TO) + ":";
  if (feature_addressbook && QUERY (multipleselect))
    page += "</a>";
  page += "</b></font></TD><TD><font class=\"header2\"><b>";
  
  address_links = (feature_addressbook && has_value (lower_case (QUERY (selectaddresses)), "to"));
  if (address_links) {
    array (string) tos = mail->headers->to / ",";
    int add = sizeof (tos);
    foreach (tos, string to) {
      to = String.trim_whites (to);
      page += "<a target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
      page += "?actionnewaddress=1&address=" + http_encode_url (to) + "&take=1\">";
      page += fix_header (to);
      page += "</a>";
      if (--add)
	page += ", ";
    }
  }
  else
    page += fix_header (mail->headers->to);
  
  if (mail->headers->cc) {
    page += "</b></font></td></tr><tr><td><font class=\"header1\"><b>";
    if (feature_addressbook && QUERY (multipleselect)) {
      page += "<a target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
      page += "?actionnewaddress=1&address=" + http_encode_url (mail->headers->cc) + "&take=1\">";
    }
    page += MSG(M_CC) + ":";
    if (feature_addressbook && QUERY (multipleselect))
      page += "</a>";
    page += "</b></font></td><td><font class=\"header2\"><b>";
    
    address_links = (feature_addressbook && has_value (lower_case (QUERY (selectaddresses)), "cc"));
    if (address_links) {
      array (string) ccs = mail->headers->cc / ",";
      int add = sizeof (ccs);
      foreach (ccs, string cc) {
	cc = String.trim_whites (cc);
	page += "<a target=\"" + id->misc->imho->target + "\" href=\"" + id->misc->imho->nexttarget;
	page += "?actionnewaddress=1&address=" + http_encode_url (cc) + "&take=1\">";
	page += fix_header (cc);
	page += "</a>";
	if (--add)
	  page += ", ";
      }
    }
    else
      page += fix_header (mail->headers->cc);
  }
  
  if (sizeof (QUERY (rfc822dateformat))) {
    page += "</b></font></td></tr><tr><td><font class=\"header1\"><b>";
    page += MSG(M_TIME)+" :</b></font></td><td nowrap><font class=\"header2\">";
    
    void|array pdate = parse_date (mail->headers->date);
    if (pdate) {
      array imho_from = ({ "$DD", "$MM", "$YY", "$TZ", "$HOURS", "$MINUTES", "$SECONDS", "$YYYY" });
      array imho_to   = ({ 
	pdate[DD_INDEX], pdate[MM_INDEX], pdate[YY_INDEX], pdate[TZ_INDEX],
	pdate[HOURS_INDEX], pdate[MINUTES_INDEX], pdate[SECONDS_INDEX], pdate[YYYY_INDEX]
      });
      page += replace (QUERY (rfc822dateformat), imho_from, imho_to);
    }
    else
      page += "?";
  }
  else
    if (QUERY (rfc822readdate)) {
      page += "</b></font></td></tr><tr><td><font class=\"header1\"><b>";
      page += MSG(M_TIME) + " :</font></b></td><td><font class=\"header2\">";
      page += fix_header (mail->headers->date);
    }
    else {
      page += "</b></font></td></tr><tr><td><font class=\"header1\"><b>";
      page += MSG(M_TIME) + " :</b></font></td><td nowrap><font class=\"header2\">";
      page += mailindex_date (sessobj, fix_header (mail->headers->date), 
			      fix_header(mail->headers->date));
    }
  
  page += "</font></td></tr><tr><td><font class=\"header1\"><b>";
  page += MSG(M_SUBJECT) + " :</b></font></td><td><font class=\"header2\"> ";
  page += fix_header (mail->headers->subject);
  page += "</font></td></tr>";

  if (sessobj->showheaders)
    foreach (indices (mail->headers), string h)
      if (!(< "to", "from", "cc", "date", "subject" >)[h])
	foreach (mail->headers[h] / "\0", string h2)
	  page += "<br /><b>" + h + ":</b> " + fix_header(h2);
  
  page += "</table>";
  
  array(object) msgs = ({ });
  if(mail->body_parts) 
    msgs += mail->body_parts;
  else
    msgs += ({ mail });
  
  int p = 0, msgshown = 0, part = 0;
  mapping refparts = ([ ]);
  array(string) partnos = ({ });
  array parenttypes = ({ });
  mapping parenttype = 0;
  string partno;
  object mess;
  
  if (sizeof (msgs)) {
    
    foreach (msgs, mess) {
      partnos += ({ ((string) p++) });
      parenttypes += ({ ([ "type": mail->type, "subtype": mail->subtype]) });
    }
    
    for (mess = msgs[0]; mess != 0; ) {
      partno = partnos[0];
      parenttype = parenttypes[0];
      msgs = msgs[1..];
      partnos = partnos[1..];
      parenttypes = parenttypes[1..];

      if (mess->body_parts) {
	msgs = mess->body_parts + msgs;
	for (p = sizeof (mess->body_parts) - 1; p >= 0; p--) {
	  partnos =  ({ partno + "," + ((string)p) }) + partnos;
	  parenttypes = ({ ([ "type": mess->type, "subtype": mess->subtype]) }) + parenttypes;
	}
      }
      else {
	string name = fix_coding (mess->disp_params["filename"] || mess->params["name"] || "");
	page+="<table width=\"100%\" bgcolor=\"&imho_tdbgcolor;\"><tr><td>";
        
	int dotext = (!msgshown || (feature (FEAT_SHOWTEXT) && (sessobj->showtext == "1")));
	int linkshown = 0;

	if (QUERY (mime_remap) && strlen (QUERY (mime_remap))) {
	  array remap = QUERY (mime_remap) / "\n";
	  foreach (remap, string rem) {
	    array fromto = rem / ":";
	    string tmp = mess->type + "/" + mess->subtype;
	    if (tmp == fromto[0]) {
	      mess->type = (fromto[1] / "/")[0];
	      mess->subtype = (fromto[1] / "/")[1];
	      if (QUERY (debug))
		write ("remapped: " + tmp + " => " + mess->type + "/" + mess->subtype + "\n");
	      break;
	    }
	  }
	}
	
	if ((!mess->headers["content-id"] || 
	     !refparts[mess->headers["content-id"]]) &&
	    !(!msgshown && mess->type == "text" &&
	      (parenttype->type == "multipart" && 
	       parenttype->subtype == "alternative")?((feature (FEAT_SHOWHTML)&&(sessobj->showhtml == "1"))?0:(mess->subtype=="plain" && dotext)):((mess->subtype == "plain") && dotext))) {
	  linkshown = 1;
	  page += "<a";
	  if (QUERY (attachwindow))
	    page += " target=\"_blank\"";
	  page += " href=\""+id->misc->imho->nextpage+"/";
	  page += (sizeof (name) ? http_encode_url (name) : "unknown");
	  page += "?mailpart=" + partno + "\"><img src=\"";
	  page += image_from_type (((mess->type) / "/")[0]) + "\" alt=\"\" border=\"0\"> ";
          if (QUERY (attachmenttype))
  	    page += replace (MSGA(M_ATTACHMENTLINK, ({ sizeof (name) ? html_encode_string (name) + ", " : "", "\0" })),
		  	     "\0",
			     mess->type + "/" + mess->subtype);
	  else
            if (sizeof (name))
    	      page += html_encode_string (name);
          page += "</a>";
	}
	// Big if-clause to find out whether this part should be shown...
	/*
	  write(sprintf ("refparts= %O\n", refparts));
	  write("content-id= " + mess->headers["content-id"] + "...\n");
	  write("refparts[content-id]= " + refparts[mess->headers["content-id"]] + "...\n");
	*/
	if ((mess->type == "text") && 
	    (!mess->headers["content-id"] || 
	     !refparts[mess->headers["content-id"]]) &&
	    (parenttype->type == "multipart" && 
	     parenttype->subtype == "alternative")?((feature (FEAT_SHOWHTML) &&(sessobj->showhtml == "1"))?(mess->subtype == "html" && !msgshown):((mess->subtype == "plain") && dotext)):((mess->subtype == "plain" && dotext) || (feature (FEAT_SHOWHTML)&&(sessobj->showhtml == "1") && (mess->subtype == "html") && !msgshown))) {
	  string data = mess->getdata () || "";
	  //msgshown = 1;
	  
	  if (linkshown)
	    page += "<hr />";
	  
	  if (sizeof (data) > ((int)QUERY (maxmailsize))) {
	    page += "<a target=\"_blank\" href=\"" + id->misc->imho->nextpage + "/";
	    page += (sizeof (name) ? http_encode_url (name) : "mail.txt");
	    page += "?mailpart=" + partno + "\"><img src=\"";
	    page += image_from_type (((mess->type) / "/")[0]);
	    page += "\" alt=\"\" border=\"0\">" + MSG(M_MAILTOOBIG) + "</a>";
	  }
	  else {
	    mixed err = 0;
	    string _mess;
	    err = catch {
	      _mess = Locale.Charset->decoder (mess->charset)->feed (data)->drain ();
	    };
	    
	    if (err)
	      page += " " + replace (MSGA(M_CHARSETWARNING, ({ "\0" })), "\0", html_encode_string (mess->charset)) + "<hr />";
	    
	    page += "<noparse>";
	    
	    if (mess->subtype == "html") {
	      //write("safe_html... ");
	      page += my_configuration ()->get_provider ("camas_html")->safe_html ((err ? data : _mess), id, refparts, mess->headers["content-base"]);
	      //write ("done.\n");
	    }
	    else {
	      page += "<font face=\"Courier\">";
	      page += replace (make_links (id, fixstring (err ? data : _mess)), "  ", "&nbsp;&nbsp;");
	      page += "</font>";
	    }
	    
	    page += "</noparse>";
	  }
	}
	else
	  if ((mess->type == "image") && image_types[lower_case (mess->subtype)] && 
	      (!mess->headers["content-id"] || !refparts[mess->headers["content-id"]])) {
	    /*write (sprintf ("type= %O, image_types= %O, mess->headers[\"content-id\"]=%O, "
	      "refparts[mess->headers[\"content-id\"]]= %O\n",
	      mess->type,
	      image_types[lower_case (mess->subtype)],
	      mess->headers["content-id"],
	      refparts[mess->headers["content-id"]]));*/
	    page += "<hr /><img src=\"" + id->misc->imho->nextpage + "/";
	    page += http_encode_url (name) + "?mailpart=" + partno + "\" alt=\"\">";
	  }
        
	page += "</td></tr>\n</table><br />\n";
	part++;
      }
      if (sizeof(msgs) > 0)
	mess = msgs[0];
      else
	mess = 0;
    }
  }
  
  if (!id->misc->imho->nobottombuttons)
    page += buttons;
  //oliv3
  //	page+="<hr /><imho_dumpid>";
  
  return page;
}

// ------------------ Files screen

string camas_screen_files (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "files";
  id->misc->imho->title = MSG(M_FILES);
  int totsize = 0;

  if (sizeof (sessobj->files)) {
    page += "<form method=\"post\" action=\"" + id->misc->imho->nextpage + "\">";
    page += "<table width=\"100%\"><tr><th class=\"mark\" bgcolor=\"&imho_thbgcolor;\">";
    page += "<font color=\"&imho_thcolor;\">" + MSG(M_MARKFLAG) + "</font></th>";
    page += "<th class=\"filename\" bgcolor=\"&imho_thbgcolor;\" width=\"100%\">";
    page += "<font color=\"&imho_thcolor;\">" + MSG(M_FILENAME) + "</font></th>";
    page += "<th class=\"size\" bgcolor=\"&imho_thbgcolor;\">";
    page += "<font color=\"&imho_thcolor;\">" + MSG(M_SIZE) + "</font></th>";
    page += "<th class=\"type\" bgcolor=\"&imho_thbgcolor;\">";
    page += "<font color=\"&imho_thcolor;\">" + MSG(M_MIMETYPE) + "</font></th></tr>\n";

    int i = 0;
    foreach (sessobj["files"], mapping file) {
      page += "<tr><td bgcolor=&imho_tdbgcolor;><input type=\"checkbox\" name=\"file";
      page += (string)i+ "\" value=\"1\"></td><td bgcolor=&imho_tdbgcolor;>";
      page += "<a href=\"" + id->misc->imho->nextpage + file->fname + "?download=";
      page += (string)i+ "\">" + file->fname + "</td><td bgcolor=&imho_tdbgcolor;>";
      page += file->size + "</td><td bgcolor=&imho_tdbgcolor;>" + file->type + "</td></tr>";
      i++;
      totsize += file->size;
    }
    page += "</table><table><tr><td>";
    page += formdrawbutton ("m_deletemarkedfiles", "actiondeletefiles", MSG(M_DELETEMARKEDFILES));
    page += "</td></tr>\n</table></form>";
  }
  else
    page += MSG(M_NOFILES);

  int allow_upload = 1;
  if ((int)QUERY (uploadquota)) {
    int space = (((((int)QUERY (uploadquota))*1024) - totsize) / 1024);
    page += MSGA(M_AVAILSPACE, ({ space })) + "<br />\n";
    if (space <= 0)
      allow_upload = 0;
  }

  if (allow_upload) {
    page += "<form name=\"imhouploadform\" method=\"post\"";
    page += " enctype=\"multipart/form-data\" action=\"" + id->misc->imho->nextpage + "\">";
    page += "<table><tr><td>";
    page += "<input name=\"file\" type=\"file\">";
    page += "<input type=\"hidden\" name=\"actionupload\" value=\"1\">";
    if (id->supports->javascript)
      {
	page += formdrawbutton ("m_upload", "\"\"", MSG(M_UPLOAD),
				"document.forms.imhouploadform.fixedfilename.value=document.forms.imhouploadform.file.value.replace(/\\\\/g,'\\\\\\\\')");
	page += "<input type=\"hidden\" name=\"fixedfilename\" value=\"\">";
      }
    page += "</td></tr>\n</table></form>";
    if (!id->supports->javascript)
      page += MSG(M_WINDOWSBUG);
  }
  
  return page;
}

// ------------------ Preferences screen

string camas_screen_setup (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  array(string) replacements = ({ "" }) * sizeof (SETUP_FIELDS);
  array(string) m_replacements = ({ "" }) * sizeof (SETUP_MFIELDS);

  id->misc->imho->screen = "setup";
  id->misc->imho->title = MSG(M_PREFS);

  if (!sessobj->prefsloaded) {
    sessobj->prefsloaded = 1;
    page += QUERY (newusermsg);
  }

  page += "<table><form target=\"_top\" method=\"post\" action=\"" + id->misc->imho->nexturl + "_top\">";
  
  if (feature (FEAT_USERCANCHANGENAME)) {
    m_replacements[SF_UN] = MSG(M_PNAME);
    replacements[SF_UN] = "<input name=\"name\" value=\"" + fixstring (sessobj->name) + "\">";
  }
  
  if (feature (FEAT_USERADDRESS)) {
    m_replacements[SF_UA] = MSG(M_PMAILADDRESS);
    replacements[SF_UA] = "<input name=\"address\" value=\"" + fixstring (sessobj->address) + "\">";
  }

  if (feature (FEAT_USERMAILPATH)) {
    m_replacements[SF_MP] = MSG(M_PMAILPATH);
    replacements[SF_MP] = "<input name=\"mailpath\" value=\"" + fixstring (sessobj->mailpath) + "\">";
  }
  
  if ((sizeof (sessobj->lang_module->lang_progs) > 1) && feature (FEAT_USERLANGUAGE)) {
    m_replacements[SF_LA] = MSG(M_PLANGUAGE);
    replacements[SF_LA] = "<select name=\"language\">";
    foreach (sort (indices (sessobj->lang_module->lang_progs)), string opt) {
      replacements[SF_LA] += "<option value=\"" + opt + "\"";
      if (sessobj->language == opt)
	replacements[SF_LA] += " selected";
      replacements[SF_LA] += ">" + upper_case (opt[0..0]) + opt[1..] + "</option>";
    } 
    replacements[SF_LA] += "</select>";
  }
    
  array (string) layouts = id->conf->get_provider ("camas_layout_manager")->list_layouts ();
  if (sizeof (layouts) > 1) {
    m_replacements[SF_LY] = MSG(M_PUSERINTERFACE);
    replacements[SF_LY] = "<select name=\"layout\">";
    foreach (layouts, string lay) {
      replacements[SF_LY] += "<option value=\"" + lay + "\"";
      if (sessobj->layout == lay)
	replacements[SF_LY] += " selected";
      replacements[SF_LY] += ">" + lay + "</option>";
    } 
    replacements[SF_LY] += "</select>";
  }
  
  m_replacements[SF_RI] = MSG(M_PINCLUDEMAIL);

  /*  if (QUERY (chgreplymsgprefix))
      page += "<table><tr><td>";*/

  replacements[SF_RI] = "<select name=\"replyincludemsg\">";
  replacements[SF_RI] += "<option value=\"1\"";
  if (sessobj->replyincludemsg == "1")
    replacements[SF_RI] += " selected";
  replacements[SF_RI] += ">" + MSG(M_YES) + "</option>";
  replacements[SF_RI] += "<option value=\"0\"";
  if (sessobj->replyincludemsg == "0")
    replacements[SF_RI] += " selected";
  replacements[SF_RI] += ">" + MSG(M_NO) + "</option>";
  replacements[SF_RI] += "</select>";

  if (QUERY (chgreplymsgprefix))
    {
      m_replacements[SF_RP] = MSG(M_PQUOTEPREFIX);
      replacements[SF_RP] = "<input name=\"replymsgprefix\" size=\"5\" value=\"";
      replacements[SF_RP] += fixstring (sessobj["replymsgprefix"])+"\">";
    }

  m_replacements[SF_SI] = MSG(M_PSIGNATURE);
  replacements[SF_SI] = "<textarea rows=\"4\" cols=\"" + QUERY (signaturecols) + "\" name=\"signature\">";
  replacements[SF_SI] += html_encode_string (sessobj->signature) + "</textarea>";
  
  if (feature (FEAT_USERHEADERS)) {
    m_replacements[SF_UH] = MSG(M_PHEADER);
    replacements[SF_UH] = "<textarea rows=\"2\" cols=\"70\" name=\"extraheader\">";
    replacements[SF_UH] += html_encode_string (sessobj->extraheader)+"</textarea><br />";
    replacements[SF_UH] += MSG(M_PHEADERDESC);
  }
  
  m_replacements[SF_LO] = MSG(M_PINACTIVELOGOUT);
  replacements[SF_LO] = "<select name=\"autologout\">";
  foreach (({ /* "1", */ "5", "10", "20", "60", "120" }), string opt) {
    replacements[SF_LO] += "<option value=\"" + opt + "\"";
    if (sessobj->autologout == opt)
      replacements[SF_LO] += " selected";
    replacements[SF_LO] += ">" + opt + "</option>";
  } 
  replacements[SF_LO] += "</select>";

  m_replacements[SF_VM] = MSG(M_PVISIBLEMESSAGES);
  replacements[SF_VM] = "<select name=\"visiblemail\">";
  foreach (({ "10", "15", "20", "30", "40", "60", "100" }), string opt) {
    replacements[SF_VM] += "<option value=\"" + opt + "\"";
    if (sessobj->visiblemail == opt)
      replacements[SF_VM] +=" selected";
    replacements[SF_VM] += ">" + opt + "</option>";
  } 
  replacements[SF_VM] += "</select>";
    
  m_replacements[SF_SO]= MSG(M_PSORTORDER);
  replacements[SF_SO] = "<select name=\"sortorder\">";
  foreach (({ "forward", "backward" }), string opt) {
    replacements[SF_SO] += "<option value=\"" + opt + "\"";
    if (sessobj->sortorder == opt)
      replacements[SF_SO] += " selected";
      replacements[SF_SO] += ">";
      string sortord = MSGA(M_PSORTORDERS, ({ opt }) );
      replacements[SF_SO] += upper_case (sortord[0..0]) + sortord[1..];
      replacements[SF_SO] += "</option>";
  } 
  replacements[SF_SO] += "</select>";

  m_replacements[SF_SC] = MSG(M_PSORTCOLUMN);
  replacements[SF_SC] = "<select name=\"sortcolumn\">";
  array columns = ({ "num", "date", "from", "to", "subject", "size" });

  foreach (columns, string col) {
    replacements[SF_SC] += "<option value=\"" + (string)col + "\"";
    if (sessobj->sortcolumn == (string)col)
      replacements[SF_SC] += " selected";
    replacements[SF_SC] += ">";
    
    switch (col) {
      case "num":
	replacements[SF_SC] += MSG(M_NUMBER);
	break;
      case "date":
	replacements[SF_SC] += MSG(M_DATE);
	break;
      case "from":
	replacements[SF_SC] += MSG(M_FROM);
	break;
      case "to":
	replacements[SF_SC] += MSG(M_TO);
	break;
      case "subject":
	replacements[SF_SC] += MSG(M_SUBJECT);
	break;
      case "size":
	replacements[SF_SC] += MSG(M_SIZE);
	break;
      }
      replacements[SF_SC] += "</option>";
  }
  replacements[SF_SC] += "</select>";

  if (feature (FEAT_SHOWHTML)) {
    m_replacements[SF_SH] = MSG(M_PSHOWHTML);
    replacements[SF_SH] = "<select name=\"showhtml\">";
    replacements[SF_SH] += "<option value=\"1\"";
    if (sessobj->showhtml == "1")
      replacements[SF_SH] += " selected";
    replacements[SF_SH] += ">" + MSG(M_YES) + "</option>";
    replacements[SF_SH] += "<option value=\"0\"";
    if (sessobj->showhtml == "0")
      replacements[SF_SH] += " selected";
    replacements[SF_SH] += ">" + MSG(M_NO) + "</option>";
    replacements[SF_SH] += "</select>";
  }
  
  if (feature (FEAT_SHOWTEXT)) {
    m_replacements[SF_ST] = MSG(M_PSHOWTEXT);
    replacements[SF_ST] = "<select name=\"showtext\">";
    replacements[SF_ST] += "<option value=\"1\"";
    if (sessobj->showtext == "1")
      replacements[SF_ST] += " selected";
    replacements[SF_ST] += ">" + MSG(M_YES) + "</option>";
    replacements[SF_ST] += "<option value=\"0\"";
    if (sessobj->showtext == "0")
      replacements[SF_ST] +=" selected";
    replacements[SF_ST] += ">" + MSG(M_NO) + "</option>";
    replacements[SF_ST] += "</select>";
  }
  
  if (feature (FEAT_MAILBOXES)) {
    if ((<"both", "move to trash">)[QUERY (deletemethod)])
      if (feature (FEAT_USERTRASHFOLDER)) {
	m_replacements[SF_FT] = MSG(M_PTRASHFOLDER);
	replacements[SF_FT] = "<input name=\"trashfolder\" value=\"" + fixstring (sessobj->trashfolder)+"\">";
      }
    
    if (feature (FEAT_USERDRAFTSFOLDER)) {
      m_replacements[SF_FD] = MSG(M_PDRAFTSFOLDER);
      replacements[SF_FD] = "<input name=\"draftsfolder\" value=\"" + fixstring (sessobj->draftsfolder) + "\">";
    }

    if (feature (FEAT_USERSENTFOLDER)) {
      m_replacements[SF_FS] = MSG(M_PSENTFOLDER);
      replacements[SF_FS] = "<input name=\"sentfolder\" value=\"" + fixstring (sessobj->sentfolder) + "\">";
      
      if (feature (FEAT_USERSENTSAVEATTACHMENTS)) { 
	m_replacements[SF_SA] = MSG(M_PSAVEATTACHMENTS);
	replacements[SF_SA] += "<select name=\"saveattachments\">";
	replacements[SF_SA] += "<option value=\"1\"";
	if (sessobj->saveattachments == "1")
	  replacements[SF_SA] += " selected";
	replacements[SF_SA] += ">" + MSG(M_YES) + "</option>";
	replacements[SF_SA] += "<option value=\"0\"";
	if (sessobj->saveattachments == "0")
	  replacements[SF_SA] += " selected";
	replacements[SF_SA] += ">" + MSG(M_NO) + "</option>";
	replacements[SF_SA] += "</select>";
      }
    }
  }
  
  if (feature (FEAT_USERBCCCOPY)) {
    m_replacements[SF_BC] = MSG(M_PDEFAULTBCC);
    replacements[SF_BC] = "<input name=\"autobcc\" value=\"" + fixstring (sessobj->autobcc) + "\">";
  }

  if (feature (FEAT_USERSETUPSHOWHIDDENHEADERS)) {
    m_replacements[SF_DH] = MSG(M_PSHOWHIDDENHEADERS);
    replacements[SF_DH] = "<select name=\"showhiddenheaders\">";
    replacements[SF_DH] += "<option value=\"1\"";
    if (sessobj->showhiddenheaders == "1")
      replacements[SF_DH] += " selected";
    replacements[SF_DH] += ">" + MSG(M_YES) + "</option>";
    replacements[SF_DH] += "<option value=\"0\"";
    if (sessobj->showhiddenheaders == "0")
      replacements[SF_DH] += " selected";
    replacements[SF_DH] += ">" + MSG(M_NO) + "</option>";
    replacements[SF_DH] += "</select>";
  }
  
  if (feature (FEAT_ORGANIZATION)) { 
    m_replacements[SF_OR] = MSG(M_PORGANIZATION);
    replacements[SF_OR] = "<input name=\"organization\" value=\"" + fixstring (sessobj->organization) + "\">";
  }

  page += QUERY(laysetup);
  page = replace (page, SETUP_MFIELDS, m_replacements);
  page = replace (page, SETUP_FIELDS, replacements);

  page += "<tr><td>&nbsp;</td><td>";
  page += formdrawbutton ("m_psaveanduse", "actionsavesetup", MSG(M_PSAVEANDUSE));
  page += "</td></tr>";
  page += "</form></table>";

  return page;
}

// ------------------ Dialog Box screen

string camas_screen_dialog (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "dialog";
  if (sizeof (sessobj->dialogstrings) == 1)
    id->misc->imho->title = MSG(M_INFO);
  else
    id->misc->imho->title = MSG(M_QUESTION);

  page += "<br />\n<br />\n<center><table border=2><tr><td><table>";
  page += "<tr><td bgcolor=&imho_thbgcolor;><br />\n</td><tr><tr><td>";
  page += "<table cellspacing=10 bgcolor=&imho_tdbgcolor;><tr><td>";
  page += fixstring (sessobj->dialogtext) + "</td></tr>\n</table></td></tr>\n";
  page += "<tr><td><center><form method=\"post\" target=\"";

  if (sessobj->dialogtarget) {
    page += sessobj->dialogtarget + "\" action=\"";
    page += id->misc->imho->nexturl+sessobj->dialogtarget + "\">";
  }
  else {
    page += id->misc->imho->target + "\" action=\"";
    page += id->misc->imho->nexttarget + "\">";
  }

  foreach (indices (sessobj->dialogstrings), int i)
      page += formdrawbutton("dialog_"+sessobj->dialogactions[i],
			     sessobj->dialogactions[i],
                             sessobj->dialogstrings[i]);

  //oliv3: optional arguments
  if (sessobj->dialogoptions) {
    foreach (sessobj->dialogoptions, string option) {
      array opts = option / "=";
      page += "<input type=\"hidden\" name=\"" + opts[0] + "\" value=\"" + opts[1] + "\">";
    }
    sessobj->dialogoptions = 0;
  }
  page += "</form></center></td></tr>\n</table></td></tr>\n</table></center>";
  
  return page;
}

// ------------------ Spell Checker screen

string camas_screen_spellcheck (object id) {
  string page = "";
  mapping sessobj = id->misc->session_variables;

  id->misc->imho->screen = "spellcheck";
  id->misc->imho->title = MSG(M_SPELLCHECK);

  if ((sessobj->checkword < 0) || (sessobj->checkword >= sizeof (sessobj->misspelled)))
    sessobj->checkword = 0;

  string spellret = "";
  int i = 0;
  int j = -1;
  foreach (sessobj->spelling, mixed foo) {
    j++;
    if (arrayp (foo)) {
      if ((sessobj->misspelled)[sessobj->checkword] == j) {
	spellret += "<b>" + html_encode_string (foo[1]) + "</b>";
	i++;
      }
      else {
	spellret += "<a href=" + id->misc->imho->nextpage + "?actioncheckword=1&word=";
	spellret += (i++) + ">" + html_encode_string (foo[1]) + "</a>";
      }
    }
    else
      spellret += html_encode_string (foo);
  }
  
  page += "<table width=\"100%\" bgcolor=\"&imho_tdbgcolor;\"><tr><td><font face=\"Courier\">";
  page += replace (spellret, "\n", "<br />\n");
  page += "</font></td></tr>\n</table>";

  if (sizeof (sessobj->misspelled)) {
    page += "<form method=\"post\" action=\"" + id->misc->imho->nextpage + "\">";
    if (sessobj->checkword > 0)
      page += formdrawbutton ("m_spellprev", "actionspellprev", MSG(M_SPELLPREV));
    if (sessobj->checkword < (sizeof(sessobj->misspelled) - 1))
      page += formdrawbutton ("m_spellnext", "actionspellnext", MSG(M_SPELLNEXT));
    page += "</form>";
  
    // original word : (sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][0]
  
    page += "<br />\n<form method=\"post\" action=\"" + id->misc->imho->nextpage + "\">";
    page += formdrawbutton ("m_spellchangeto", "\"\"", MSG(M_SPELLCHANGETO) + ":");
    page += "<input type=\"hidden\" name=\"actionspellreplace\" value=\"1\">";
    page += "<input name=\"newword\" value=";
    page += html_encode_tag_value ((sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][1]);
    page += "></form>";
    
    if (sizeof ((sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]]) > 2) {
      page += MSG(M_SPELLSUGGESTIONS) + ":<br />\n";
      page += "<form method=\"post\" action=\"" + id->misc->imho->nextpage + "\">";
      page += formdrawbutton ("m_spellchangeto", "\"\"", MSG(M_SPELLCHANGETO) + ":");
      page += "<input type=\"hidden\" name=\"actionspellselect\" value=\"1\">";
      page += "<select name=\"selectedword\">";
      int k = 0;
      foreach ((sessobj->spelling)[(sessobj->misspelled)[sessobj->checkword]][2..], string word) {
	page += "<option value=\"" + k + "\">" + html_encode_string (word) + "</option>";
	k++;
      }
      page += "</select></form>";
    }
  }
  
  page += "<form target=\"" + id->misc->imho->target;
  page += "\" method=\"post\" action=\"" + id->misc->imho->nexttarget;
  page += "\"><input type=\"hidden\" name=\"actionspelldone\" value=\"1\">";
  page += formdrawbutton ("m_spelldone", "\"\"", MSG(M_SPELLDONE));
  page += "</form>";

  return page;
}

// ------------------ Main screen

string create_main (object id, int screen) {
  switch (screen) {

  case LOGINFAILEDIMAP:
  case LOGINFAILED:
  case LOGINPROMPT:
    return camas_screen_login (id, screen);
    break;
    
  case LOGOUT:
    return camas_screen_logout (id);
    break;

  case COMPOSE:
    return camas_screen_compose (id);
    break;
    
  case ATTACHMENTS:
    return camas_screen_attachments (id);
    break;

  case ADDRESSBOOK:
    return camas_screen_addressbook (id);
    break;

  case ADDRESSBOOK2:
    return camas_screen_addressbook2 (id);
    break;

  case LDAPRESULT:
    return camas_screen_ldapresult (id);
    break;

  case LDAPRESULT2:
    return camas_screen_ldapresult2 (id);
    break;

  case MAILFILTER:
    return camas_screen_mailfilter (id);
    break;

  case LDAPSEARCH:
    return camas_screen_ldapsearch (id);
    break;

  case IMPORTADDRESS:
    return camas_screen_importaddress (id);
    break;

  case EDITADDRESS:
    return camas_screen_editaddress (id);
    break;

  case EDITADDRESSFILTER:
    // why was it not called 'editfilter' ? -oliv3
    return camas_screen_editaddressfilter (id);
    break;

  case MAILINDEX:
    return camas_screen_mailindex (id);
    break;

  case SEARCHMAIL:
    return camas_screen_searchmail (id);
    break;
    
  case FOLDERLIST:
    return camas_screen_folderlist (id);
    break;
    
  case MDNDIALOG:
    return camas_screen_mdndialog (id);
    break;

  case READMAIL:
    return camas_screen_readmail (id);
    break;

  case FILES:
    return camas_screen_files (id);
    break;

  case SETUP:
    return camas_screen_setup (id);
    break;

   case DIALOGBOX:
    return camas_screen_dialog (id);
    break;

   case SPELLCHECK:
    return camas_screen_spellcheck (id);
    break;

  default:
    return "<big><red>Error: no such screen= " + screen + "</red></big>";
  }
}

void camas_gc () {
  if (QUERY (debug))
    write ("garbage collecting: ");

  if (gc_call_out)
    remove_call_out (gc_call_out);

  int removed = 0;
  int bar = time ();
  object sess123 = my_configuration ()->get_provider ("123sessions");
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  mapping sessions = sess123->sessions ();

  if (sessions)
    foreach (indices (sessions), int foo) {
      int tmax;
      if (!sessions[foo]["values"]["autologout"] 
	  || !sscanf(sessions[foo]["values"]["autologout"], "%d", tmax))
	tmax = 20;
      if (sessions[foo]->values->status == COMPOSE)
	tmax = 180; // Do not timeout when composing mail...
      if (sessions[foo]->lastusage && (bar - (sessions[foo]->lastusage) > (tmax * 60))) {
	imho_log ("autologout", ([ "login": sessions[foo]->values->login ]));
        if (objectp (sessions[foo]->values->imapclient))
          destruct (sessions[foo]->values->imapclient);
	sess123->kill_session (foo);
	removed++;
      }
    }
#if constant(thread_create)
  destruct(lock);
#endif

  if (QUERY (debug)) {
    if (removed)
      write (sprintf ("removed %d session%s.\n", removed, ((removed == 1) ? "" : "s")));
    else
      write ("done.\n");
  }

  gc_call_out = call_out (camas_gc, QUERY (gctimeout));
}


      
mapping find_file (string file, object id) {
  //write ("find_file: " + file + "\n");
  //write (sprintf ("supports= %O\n", id->supports));
  array (string) fileparts = file / "/";

  string session = "0";
  string frame = "";

  // write ("id->misc->cacheable= " + id->misc->cacheable + "\n");
  id->misc->cacheable = 0;
  id->misc->is_dynamic = 1;
  //  write ("id->misc->cacheable= " + id->misc->cacheable + "\n");
  if (id->supports->msie && /*id->prestate &&*/ !id->prestate->nocache)
    id->prestate += (< "nocache" >);

  object sess123 = id->conf->get_provider ("123sessions");
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  if ((sizeof (fileparts) > 1) && ((fileparts[1] == "about") || (fileparts[1] == "jump"))) {
    sess123->delete_session (id, id->misc->session_id, 1);
    if (fileparts[1] == "about") {
#if constant(thread_create)
      destruct(lock);
#endif
      return http_redirect ("http://camas.caudium.net/");
    }
    else
      if (fileparts[1] == "jump") {
	int pos = search (file, "jump");
	string url = file[pos+5..sizeof (file)-1];
	if (has_value (url, "http:/") && (!has_value (url, "http://")))
	  url = replace (url, "http:/", "http://");
#if constant(thread_create)
	destruct (lock);
#endif
	return http_redirect (url);
      }
  }

  if (sizeof (fileparts) >= 1) {
    frame = fileparts[0];
    if (frame == "newbie")
      frame = fileparts[1];
  }

  sscanf (frame, "%s-%*s", frame);
  if (sizeof (frame) < 1)
    frame = "_top";
  
  if (session == "0" && id->misc->session_id 
      && (file != "" || (file == "" && id->variables->actionchangesort)) && (fileparts[0] != "newbie")) {
    session = id->misc->session_id;
    if (!id->misc->session_variables->ok) {
      sess123->delete_session (id, id->misc->session_id, 1);
#if constant(thread_create)
      destruct (lock);
#endif
      return redirect (id, "_top");
    }
  }

  if (session == "0" || (!QUERY (allowipchange) && (id->misc->session_variables->ip != id->remoteaddr))) {
    string s = id->misc->session_id;

    if ((QUERY (ssl3redir) || sizeof (QUERY (urloverride)) > 0)
	&& (fileparts[0] != "newbie") && (!QUERY (sslfaulty) || !id->supports->ssl_faulty)) {
      // To make sure the session has been
      //redirected (to a safe port) without sending the session in clear
      //during the redirect.
#if constant(thread_create)
      destruct (lock);
#endif
      if (id->misc->session_id)
        sess123->delete_session (id, id->misc->session_id, 1);
      return redirect (id, "_top");
    }

    id->misc->session_variables = ([ ]);
    default_session (id->misc->session_variables, id);
    id->misc->session_variables->status = LOGINPROMPT;
    // For example automatic reloads should not reset idle counter
    id->misc->session_variables->last = time ();
    id->misc->session_variables->ip = id->remoteaddr;
    id->misc->session_variables->ok = 1;
    frame = "_top";
    session = s;
  }

#if constant(thread_create)
  destruct (lock);
#endif

  mapping sessobj = id->misc->session_variables;
  if (!sessobj->lang_module)
    sessobj->lang_module = id->conf->get_provider ("camas_lang");

  string foo;
  if (sscanf (frame, "idle%s", foo) == 0) 
    sessobj["last"] = time ();
  if (!id->misc->imho)
    id->misc["imho"] = ([ ]);
  id->misc->imho->frame = frame;

  return process_request (file, id);
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: location
//! Mailreader server location.
//!  type: TYPE_LOCATION
//!  name: Mountpoint
//
//! defvar: newbiesetup
//! Choose whether new users (to CAMAS) will enter preferences screen when logged in for the first time.
//!  type: TYPE_FLAG
//!  name: New users:Enter setup
//
//! defvar: newbiecreatefolder
//! Comma separated list of foldernames which will be created when logging in for the first time.
//!  type: TYPE_STRING
//!  name: New users:Create folders
//
//! defvar: newbiecreatesentfolder
//! Auto create the folder where the outgoing mail is copied.
//!  type: TYPE_FLAG
//!  name: New users:Create sent folder
//
//! defvar: newbiecreatetrashfolder
//! Auto create the folder where are the deleted mail is saved.
//!  type: TYPE_FLAG
//!  name: New users:Create trash folder
//
//! defvar: newbiecreatedraftsfolder
//! Auto create the folder where are the draft mail is stored.
//!  type: TYPE_FLAG
//!  name: New users:Create drafts folder
//
//! defvar: sitesig
//! Piece of text added to every outgoing message.
//!  type: TYPE_TEXT_FIELD
//!  name: Outgoing mail:Site-wide signature
//
//! defvar: siteorg
//! Site-wide organization.
//!  type: TYPE_STRING
//!  name: Outgoing mail:Site-wide organization
//
//! defvar: featuredelay
//! Add a column delay to mailindex
//!  type: TYPE_FLAG
//!  name: User interface dialogs:Mail index - Delay
//
//! defvar: indexdelayformat
//! Set a format for the delay.<br />You can use the following replacements :<ul><li><b>$DDAYS</b>: Day</li><li><b>$DHOURS</b>: Hour</li><li><b>$DMINUTES</b>: Minute</li><li><b>$DSECONDS</b>: Second</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Mail index - Delay format
//
//! defvar: layeditaddress
//! Layout to edit an addressbook entry.
//!Leave empty to use the default layout (autogenerated).
//!Otherwise, you must use in your <tt>input</tt> tags :
//!<lu>
//!<li><b>$NAME</b> for the name value (primary key)</li>
//!<li><b>$ADDRESS</b> for the address value</li>
//!<li><b>$BUTTONSAVE</b> for the validation button</li>
//!<li><b>$BUTTONDELETE</b> for the deletion button</li>
//!<li><b>$BUTTONCANCEL</b> for the cancel button</li>
//!</lu>
//!You can also use, according to your setting of extended addressbook fields :
//!<lu>
//!<li><b>%<i>field</i>%</b> as the name of the <tt>input</tt> for the extended addressbook field named <i>field</i> (case sensitive)</li>
//!<li><b>$<i>field</i>$</b> as it's value</li>
//!</lu><br />
//!e.g.:<br />
//!Let's suppose we have two extended address book fields named <it>Phone</it> and <it>Postal</it>, you could write here :<p><tt>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Edit address
//
//! defvar: attachwindow
//! Enables multi-window view of attachments.
//!  type: TYPE_FLAG
//!  name: Features:Open attachments in new window
//
//! defvar: linkwindow
//! Enables opening of links in a new window.
//!  type: TYPE_FLAG
//!  name: Features:Open links in new window
//
//! defvar: extendedabook
//! Enables extended addressbook.
//!  type: TYPE_FLAG
//!  name: Features:Extended addressbook
//
//! defvar: extendedabookfields
//! Comma-separated list of field names.
//!  type: TYPE_STRING
//!  name: Features:Extended addressbook fields.
//
//! defvar: deleteinebook
//! When deleting an entry in the addresbook, also delete a matching one in the extended addressbook. You probably want to say yes unless you plan your users to use either plain or extended addressbook.
//!  type: TYPE_FLAG
//!  name: Features:Also delete in extended addressbook
//
//! defvar: globaladdressbook
//! List of addresses which will appear in all users' address books. Format: "&lt;name&gt;:&lt;address&gt;", e.g "CAMAS:camas@caudium.net".
//!
//!  type: TYPE_TEXT_FIELD
//!  name: Global address book: Addresses
//
//! defvar: globaladdressbookfile
//! A file containing global addressbook addresses. Can be in LDIF or Pine format. (File in real file system or empty.)
//!  type: TYPE_STRING
//!  name: Global address book: File
//
//! defvar: displayimaperrors
//! If this is set to 'yes', the IMAP error string from the server will be shown to the user in the error dialog. The text comes from the server and cannot be altered nor translated.
//!  type: TYPE_FLAG
//!  name: User interface:Display IMAP errors to user
//
//! defvar: displaysmtperrors
//! If this is set to 'yes', the SMTP error string from the server will be shown to the user in the error dialog. The text comes from the server and cannot be altered nor translated.
//!  type: TYPE_FLAG
//!  name: User interface:Display SMTP errors to user
//
//! defvar: useutf-8
//! UTF-8 is an encoding for 16-bit characters, which is used in CAMAS internally. If this option is enabled, CAMAS will output UTF-8 data to the browser if the browser supports it. If it is disabled, the character set will be taken from the current language file.
//!  type: TYPE_FLAG
//!  name: User interface:Output UTF-8 (if possible)
//
//! defvar: newusermsg
//! New users are first presented with the preferences screen. This message is shown then.
//!  type: TYPE_STRING
//!  name: User interface:New user message
//
//! defvar: imapservers
//! List of servers to use. <i><b>Format:</b> Domain Server[:port]	 [options]</i><br />Example:
//!<pre>acc.umu.se	imap.acc.umu.se<br />some.where	mail.some.where</pre>Option 'mailpath:path' to override the default IMAP mail folder path<br />Option 'prefsbox:name' to override the default Prefs foldername<br />Option 'folderstyle:mailandfolders' indicates that folders can contain both folders and mail<br />Example:<pre>acc.umu.se	imap.acc.umu.se	prefsbox:imhoprefs,mailpath:INBOX</pre><br />
//!  type: TYPE_TEXT_FIELD
//!  name: Incoming mail:IMAP servers
//
//! defvar: layeditaddressfilter
//! Layout to edit an filterbook entry.
//!Leave empty to use the default layout (autogenerated).
//!Otherwise, you can use :
//!<lu>
//!<li><b>$INPUTNAMEFIELD</b> for the name select option</li>
//!<li><b>$INPUTFIELD</b> for the field select option</li>
//!<li><b>$INPUTNAMEFILTER</b> for the name textbox (primary key)</li>
//!<li><b>$INPUTFILTER</b> for the filter textbox</li>
//!<li><b>$INPUTFOLDER</b> for the select option associated to extended filterbook field named (case sensitive)</li>
//!<li><b>$BUTTONFILTERSAVE</b> for the validation button</li>
//!<li><b>$BUTTONFILTERDELETE</b> for the deletion button</li>
//!<li><b>$BUTTONFILTERCANCEL</b> for the cancel button</li>
//!</lu>
//!
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Edit filter
//
//! defvar: laylogout
//! Replace the content of the logout page by some RXML contents.<br />Use <b>$LOGOUTMSG</b> to use imho standard language file messages.
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Logout page
//
//! defvar: laylogin1
//! Content of the login page with some RXML contents.<br />Use the following to replace with internal imho controls:<ul><li><b>$MSGLOGIN</b>: Login message from imho languages files</li><li><b>$MSGPASSWORD</b>: Password message from imho language files</li><li><b>$INPUTLOGIN</b>: Input box for login</li><li><b>$INPUTPASSWORD</b>: Input box for password</li><li><b>$INPUTOK</b>: The Ok button from imho languages files.</li></ul>You can set the define "jscript" to "on" if you wishe to use some "magic" javascript to help users to login...
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Login page, part 1
//
//! defvar: laylogin2
//! Content of the login page <i>(selection of layouts if the is more than one layout)</i>.<br />As usual you can use the following to replace with controls/messages from imho:<ul><li><b>$MSGUSRINFACE</b>:The message from imho language file to choose the user interface</li><li><b>$INPUTSELECTINFACE</b>:Popup with the differents users interfaces in ;-)</li></ul>.
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Login page, part2
//
//! defvar: laycompose1
//! Action button layout e.g. Send, Cancel and Spell check buttons.<br /> As usual you can use<ul><li><b>$INPUTSEND</b>: for the Send button</li><li><b>$INPUTCANCEL</b>: for the cancel button</li><li><b>$INPUTSPELLCHECK</b>: for the spell button</li><li><b>$INPUTSPELLCHOOSE</b>: for the popup list of languages for the spelling</li><li><b>$INPUTSAVEDRAFT</b>: allow the user to save a draft</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Compose page, Actions buttons
//
//! defvar: laycompose2
//! Header actions e.g. the input boxes, buttons to fill the mail headers.<br />As usual, some replacements fields :<ul><li><b>$MSGFROM</b>: Imho language 'from' text</li><li><b>$MSGEMAILFROM</b>: The email entered in the configuration dialogs</li><li><b>$MSGORGANIZATION</b>: The organization entered in the configuration dialogs</li><li><b>$MSGTO</b>: Imho language 'To' text</li><li><b>$MSGCC</b>: Imho language 'CC' text</li><li><b>$MSGBCC</b>: Imho language 'Bcc' text</li><li><b>$MSGSUBJECT</b>: Imho language 'Subject' text</li><li><b>$INPUTTO</b>: The input box for 'To'</li><li><b>$INPUTCC</b>: The input box for 'CC'</li><li><b>$INPUTBCC</b>: The input box for 'Bcc'</li><li><b>$INPUTSUBJECT</b>: The input box for the 'subject'</li><li><b>$INPUTADDRBOOKTO</b>: Button to add address from addressbook for 'To'</li><li><b>$INPUTADDRBOOKCC</b>: Button to add address from addressbook for 'Cc'</li><li><b>$INPUTADDRBOOKBCC</b>: Button to add address from addressbook for 'Bcc'</li><li><b>$INPUTADDRBOOK</b>: Button to add addresses from addressbook ('To', 'Cc', and 'Bcc')</li><li><b>$INPUTLDAPTO</b>: Button to add address from LDAP addressbook for 'To'</li><li><b>$INPUTLDAPCC</b>: Button to add address from LDAP addressbook for 'Cc'</li><li><b>$INPUTLDAPBCC</b>: Button to add address from LDAP addressbook for 'Bcc'</li><li><b>$INPUTLDAP</b>: Button to add addresses from LDAP addressbook ('To', 'Cc', and 'Bcc')</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Compose page, Headers actions
//
//! defvar: headersize
//! Width of the (to, cc, bcc) input fields.
//!  type: TYPE_INT
//!  name: User interface dialogs:Pages - Compose page, Headers size
//
//! defvar: subjectsize
//! Width of the subject field.
//!  type: TYPE_INT
//!  name: User interface dialogs:Pages - Compose page, Subject size
//
//! defvar: laycompose3
//! Attachments actions for composing mail.<br />As usual there is options/replacements :<ul><li><b>$MSGATTACHMENTS</b>: Imho language for adding attachements files</li><li><b>$INPUTFILEATT</b>: List of attached files</li><li><b>$INPUTADD</b>: Button to add new files</li><li><b>$INPUTDEL</b>: Button to del selected files</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Compose page, Attachments actions
//
//! defvar: laycompose4
//! Compose screen.<br />As usual some options/replacements :<ul><li><b>$INPUTMSGAREA</b>: The input box for message writting</li><li><b>$INPUTSAVECOPY</b>: Check box to save a copy into sent mail</li><li><b>$INPUTDSNDELAY</b> & <b>$INPUTDSNSUCCESS</b>: DSN options.</li><li><b>$INPUTMDN</b>: Request a MDN (Mail Delivery Notification)</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Compose page, Compose screen
//
//! defvar: msgrows
//! Rows in the input box.
//!  type: TYPE_INT
//!  name: User interface dialogs:Pages - Compose page, Input box rows
//
//! defvar: msgcols
//! Columns in the input box.
//!  type: TYPE_INT
//!  name: User interface dialogs:Pages - Compose page, Input box columns
//
//! defvar: layfoldersrow
//! The layout for the folders list.<br />As usual some options/replacements :<ul><li><b>$MARK</b>: Button to select the folder for deletion</li><li><b>$NAME</b>: The name of the folder</li><li><b>$MESSAGES</b>: The number of messages</li><li><b>$RECENT</b>: The number of recent messages</li><li><b>$UNSEEN</b>: The number of unseen messages</li><li><b>$SIZE</b>: The size of the folder (in kb).</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Folders List - Table row
//
//! defvar: layfolderstop
//! The layout for the top of the table. Provided rxml tag: <b>&lt;imho_totalsize&gt;</b> for the size of all the folders.<br />(also in &amp;session.allfolderssize;).
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Folders List - Table header
//
//! defvar: layfoldersbottom
//! The layout for the bottom of the table.<br />As usual some options/replacements :<ul><li><ul>Creating folders :<li><b>$NEWFOLDERNAME</b>: Text area for the name of the new folder</li><li><b>$CREATE</b>: Button to create the folder</li></ul></li><li><ul>Renaming folders :<li><b>$OLDFOLDERNAME</b>: Selects the folder to rename</li><li><b>$RENAMEFOLDERNAME</b>: Text area for the name of the new folder</li><li><b>$RENAME</b>: Button to rename the folder </li></ul></li><li><ul>Deleting folders :<li><b>$DELETE</b>: Button to delete marked folders</li></ul></li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Folders List - Table footer
//
//! defvar: laysetup
//! The layout for the setup screen.<br />As usual some replacements :<ul><li><b>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Pages - Setup screen
//
//! defvar: rfc822dateformat
//! Set a format for the date.<br />You can use the following replacements :<ul><li><b>$DD</b>: Day</li><li><b>$MM</b>: Month</li><li><b>$YY</b>: Short year</li><li><b>$YYYY</b>: Long year</li><li><b>$HOURS</b>: Hours</li><li><b>$MINUTES</b>: Minutes</li><li><b>$SECONDS</b>: Seconds</li><li><b>$TZ</b>: Timezone</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Readmail - Date format
//
//! defvar: selectaddresses
//! Allow the user to add individual addresses from the mail headers.<br />Allowed values are: From, To, Cc
//!  type: TYPE_STRING
//!  name: User interface dialogs:Readmail - Select addresses
//
//! defvar: multipleselect
//! If set, allow the user to add all addresses from the mail headers.
//!  type: TYPE_FLAG
//!  name: User interface dialogs:Readmail - Select multiple addresses
//
//! defvar: rfc822readdate
//! When set, standard RFC822 date is used instead of parsed date as in the folder index.
//!  type: TYPE_FLAG
//!  name: User interface dialogs:Readmail - Use RFC822 standard date
//
//! defvar: imapidlelogout
//! Close an open IMAP connection after this many seconds. The user session may still be active, and will not be disturbed. The IMAP connection will be reopened when the user issues an IMAP command.
//!  type: TYPE_INT
//!  name: Incoming mail:Idle IMAP connection timeout
//
//! defvar: mboxencode
//! Choose whether CAMAS should use IMAPs modified UTF-7 coding when refering to mailboxes or if it should send mailbox names unmodified. If you're having trouble accessing mailboxes with non-USASCII characters, try changing this option.
//!  type: TYPE_STRING_LIST
//!  name: Incoming mail:Mailbox addressing
//
//! defvar: defaultmailpath
//! The default mail folder path. Is usually "INBOX" (WU-IMAP) or "INBOX." (Courier-IMAP). Can be changed by the user, although the user's preferences is always saved here.
//!  type: TYPE_STRING
//!  name: Incoming mail:Default IMAP mail folder path
//
//! defvar: hidemboxes
//! Optionally hide mailboxes starting with the specified string. Mailboxes starting with '.' are often not really mailboxes.
//!  type: TYPE_STRING
//!  name: Incoming mail:Hide mailboxes starting with
//
//! defvar: sendmethod
//! Method to use when sending mail.
//!  type: TYPE_STRING_LIST
//!  name: Outgoing mail:Method
//
//! defvar: sendmail
//! Location of sendmail.
//!  type: TYPE_STRING
//!  name: Outgoing mail:Sendmail
//
//! defvar: smtpserver
//! The address of the SMTP-server.
//!  type: TYPE_STRING
//!  name: Outgoing mail:SMTP server address
//
//! defvar: smtpport
//! The portnumber of the SMTP-server.
//!  type: TYPE_INT
//!  name: Outgoing mail:SMTP server-port
//
//! defvar: smtpmaildomain
//! The outgoing mail domain.
//!  type: TYPE_STRING
//!  name: Outgoing mail:SMTP mail domain
//
//! defvar: includeipnumber
//! Include the header field 'X-Originating-IP' in outgoing mail to indicate from where the mail was sent.
//!  type: TYPE_FLAG
//!  name: Outgoing mail:Show IP-address
//
//! defvar: addmaildomain
//! If this is enabled, the default mail domain will be added to addresses without a "@" before sending them to the SMTP server.
//!  type: TYPE_FLAG
//!  name: Outgoing mail:Complete unqualified addresses
//
//! defvar: ssl3redir
//! Tries to find a SSL3 port to redirect the initial requests to. Useful if your virtual server has more than one listen port and you prefer SSL3 for CAMAS.
//!  type: TYPE_FLAG
//!  name: URL:Redirect to SSL3
//
//! defvar: urloverride
//! Specify your server URL here if CAMAS fails to guess the correct URL. Example: 'https://your.server.com'.
//!  type: TYPE_STRING
//!  name: URL:Server URL override
//
//! defvar: uploaddir
//! Directory to hold uploaded attachment files. Leave empty to disable persistent attachment support. 
//!  type: TYPE_STRING
//!  name: Files:File upload directory
//
//! defvar: uploadquota
//! Amount of Kb each user is allowed to upload. 0 gives unlimited quota.
//!  type: TYPE_INT
//!  name: Files:File upload user quota
//
//! defvar: uploadsoftquota
//! Amount of Kb by wich the user can overrun the quota for the last uploaded file.
//!  type: TYPE_INT
//!  name: Files:File upload soft user quota
//
//! defvar: maxmailsize
//! If the mail data exeeds this size (in bytes), it will only be shown as a download link.
//!  type: TYPE_INT
//!  name: User interface:Max mail size
//
//! defvar: ispell
//! Spelling:Speller location
//!  type: ""
//!  name: /usr/bin/ispell
//
//! defvar: ispelldict
//! Commaseparated list of speller dictionaries, default first.(-d option). Display name can be specified after a colon (swedish:Svenska).
//!  type: TYPE_STRING
//!  name: Spelling:Speller dictionaries
//
//! defvar: speller
//! CAMAS supports ispell and aspell. 
//!  type: TYPE_STRING_LIST
//!  name: Spelling:Spellchecker
//
//! defvar: addressbook
//! Enables the addressbook. Addressbook is disabled if User setup is.
//!  type: TYPE_FLAG
//!  name: Features:Addressbook
//
//! defvar: attachments
//! Enables sending of attachments.
//!  type: TYPE_FLAG
//!  name: Features:Attachments
//
//! defvar: usermailpath
//! Determines if users should be allowed to set the mail folder. Is disabled if User setup is.
//!  type: TYPE_FLAG
//!  name: Features:User mail path
//
//! defvar: mailboxes
//! If disabled, only INBOX can be accessed.
//!  type: TYPE_FLAG
//!  name: Features:Mailboxes
//
//! defvar: filterbook
//! Enables the mailfilter. Mail filter is disabled if User setup is.
//!  type: TYPE_FLAG
//!  name: Features:Mail Filter
//
//! defvar: usersetup
//! If enabled, each user has a setup (including addressbook, signature etc.) saved either in a mailbox or in a file. Is disabled if mailboxes is disabled and User preferences directory is empty.
//!  type: TYPE_FLAG
//!  name: Features:User setup
//
//! defvar: usereditsetup
//! Determines if the users should be able to change their own setup. This option is ignored if "User setup" is disabled.
//!  type: TYPE_FLAG
//!  name: Features:User can change setup
//
//! defvar: inboxes
//! Lets the user specify a list of inboxes which may be presented as a list in the user interface. 
//!  type: TYPE_FLAG
//!  name: Features:User specified inboxes
//
//! defvar: useraddress
//! Enables a user to change his/her email address in the user setup.
//!  type: TYPE_FLAG
//!  name: Features:User specified address
//
//! defvar: mailnotify
//! Enables a user to open a mail notification window which is reloaded periodically (every five minutes). This feature is is built on javascript in the layout file, so make sure you use an updated layout file.
//!  type: TYPE_FLAG
//!  name: Features:Mail notification window
//
//! defvar: showhtml
//! Enables a user to have messages in HTML format shown instead of of the corresponding text version. HTML messages are parsed and reformatted so that no unsecure HTML tags are displayed. For example, links to mailparts are handled but all other links are removed. Javascript is not allowed. The parsing may be rather CPU-intensive, and the rendering on the client side depends on the browser used. Heavily loaded HTML pages may then cause unpredicted behaviour.
//!  type: TYPE_FLAG
//!  name: Features:Show HTML messages
//
//! defvar: userheaders
//! Enables a user to add extra headers to outgoing mail.
//!  type: TYPE_FLAG
//!  name: Features:User specified mail headers
//
//! defvar: prefsdir
//! Directory to hold user preferences. If this one is an empty string, user preferences will be saved in a mail folder. 
//!  type: TYPE_STRING
//!  name: User preferences:User preferences directory
//
//! defvar: prefsbox
//! Mail folder to hold user preferences. Relative to the default mail folder path. This option can be overridden by a different setting in the IMAP server list.
//!  type: TYPE_STRING
//!  name: User preferences:User default preferences folder
//
//! defvar: showprefsbox
//! If "no", the preferences folder is not displayed.
//!  type: TYPE_FLAG
//!  name: User preferences:Show user preferences folder
//
//! defvar: deletemethod
//! Choose whether mail is deleted immediately or moved to a trash folder first, or whether both methods should be available.
//!  type: TYPE_STRING_LIST
//!  name: Features:Delete method
//
//! defvar: usercanchangename
//! Enables users to change their name in the configuration interface.<br /><i>Yes, this <b>is</b> stupid, but some strange people wants such features...</i>.
//!  type: TYPE_FLAG
//!  name: Features:User can change name
//
//! defvar: userlanguage
//! Enable a user to choose a different language than the one setup in the configuration interface.
//!  type: TYPE_FLAG
//!  name: Features:User can choose language
//
//! defvar: usertrashfolder
//! Enable a user to specify the name of the trash folder other than <i>"Trash"</i>.
//!  type: TYPE_FLAG
//!  name: Features:User can specify trash folder
//
//! defvar: usersentfolder
//! Enable a user to specify the name of the sent-mail folder other than <i>"sent-mail"</i>.
//!  type: TYPE_FLAG
//!  name: Features:User can specify sentmail folder
//
//! defvar: userdraftsfolder
//! Enable a user to specify the name of the drafts folder other than <i>"Drafts"</i>.
//!  type: TYPE_FLAG
//!  name: Features:User can specify drafts folder
//
//! defvar: userbcccopy
//! When enabled, user can specify a BCC address to be added to all outgoing mail.
//!  type: TYPE_FLAG
//!  name: Features:User can specify a default Bcc
//
//! defvar: showhiddenheaders
//! When reading a mail, users can show or hide the hidden headers of the mail. If this option is enabled, the user can choose whether to show or hide this button.
//!  type: TYPE_FLAG
//!  name: Features:User configurable 'show full headers'-button
//
//! defvar: usersentsaveattachments
//! Enable a user to secify to save the attachments in the sentmail folder.<br /><b>Note:</b> if the sent mail folder cannot be specified, then this feature is automaticly disabled and the attachement <b>will</b> be saved into sentmail folder.
//!  type: TYPE_FLAG
//!  name: Features:User can choose to save attachments
//
//! defvar: showtext
//! Display text attachments.
//!  type: TYPE_FLAG
//!  name: Features:Show text messages
//
//! defvar: chgreplymsgprefix
//! If set, then user can change the default prefix added to lines of message they reply to.
//!  type: TYPE_FLAG
//!  name: Features:User can change reply prefix
//
//! defvar: debug
//! When on, debug messages will be logged in Caudium's debug logfile. This information is very useful to the developers when fixing bugs.
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: ldap
//! Enables a user to search for names in a ldap directory
//!  type: TYPE_FLAG
//!  name: LDAP Global Addressbook:Use ldap
//
//! defvar: ldapserver
//! Specifies the ldap server to use.
//!  type: TYPE_STRING
//!  name: LDAP Global Addressbook:LDAP server
//
//! defvar: ldapsearchroot
//! The searchroot for ldap searches. For example: o=Your company, c=your country. Could be empty
//!  type: TYPE_STRING
//!  name: LDAP Global Addressbook:Search root
//
//! defvar: ldapuser
//! The user you should use connect. NOTE! This is ldap user format. Could be empty
//!  type: TYPE_STRING
//!  name: LDAP Global Addressbook:User
//
//! defvar: ldappass
//! The password you should use to connect. Could be empty.
//!  type: TYPE_STRING
//!  name: LDAP Global Addressbook:Password
//
//! defvar: ldapshowou
//! If "yes" include last level ou in result
//!  type: TYPE_FLAG
//!  name: LDAP Global Addressbook:Show ou
//
//! defvar: ldapversion
//! The LDAP protocol version to use.
//!  type: TYPE_INT_LIST
//!  name: LDAP Global Addressbook:Version
//
//! defvar: dsn
//! Add DSN (Delivery Status Notification) support when sending mail. This requires the use of SMTP for outgoing mail (instead of sendmail). You also have to make sure your SMTP server supports DSN.
//!  type: TYPE_FLAG
//!  name: Outgoing mail:DSN support
//
//! defvar: wrapcolumn
//! Wrap outgoing message lines at this column. If set to zero, no wrapping will take place.
//!  type: TYPE_INT
//!  name: Outgoing mail:Wrap line column
//
//! defvar: allowipchange
//! Should CAMAS allow one session to use different IP addresses? If some users are using dynamic addresses which change even during a session (e.g. while composing mail) then this option should be "yes". A reason to set it to "no" is that a stolen session id (from cookie- or log file) cannot be used from a remote machine.
//!  type: TYPE_FLAG
//!  name: Allow changing user IP-address
//
//! defvar: adminlogin
//! The CAMAS Runtime Administration Interface can be reached by logging in using this login name and the specified password.
//!  type: TYPE_STRING
//!  name: Runtime admin interface:Login Name
//
//! defvar: adminpasswd
//! The CAMAS Runtime Administration Interface can be reached by logging in using this password and the specified login name.
//!  type: TYPE_PASSWORD
//!  name: Runtime admin interface:Password
//
//! defvar: logfile
//! The file name of the log file in the real file system. If this is empty, logging is disabled.
//!  type: TYPE_STRING
//!  name: Logging:Log file name
//
//! defvar: loglogin
//! Log information on users logging in and out.
//!  type: TYPE_FLAG
//!  name: Logging:Log login and logout
//
//! defvar: logsendmail
//! Log information on all messages sent by users.
//!  type: TYPE_FLAG
//!  name: Logging:Log sending of mail
//
//! defvar: logerrors
//! Log information on all errors that occur regarding IMAP and SMTP.
//!  type: TYPE_FLAG
//!  name: Logging:Log errors
//
//! defvar: plugin
//! Experimental plugin.
//!  type: TYPE_STRING|VAR_EXPERT
//!  name: General purpose plugin
//
//! defvar: morecharsets
//! Commaseparated list of charctersets wich CAMAS tries to use for outgoing mail before the last resort, UTF-8. CAMAS first tries to fit the mail into the same characterset used by the mail the user replied to. Then it tries iso-8859-1, the charcterset of the active language file, this list of charactersets and UTF-8. Specify those charactersets your users usually use to avoid their mail being sent as UTF-8.
//!  type: TYPE_STRING
//!  name: Outgoing mail:Character sets
//
//! defvar: indexdatelongformat
//! Set a format for the date.<br />You can use the following replacements :<ul><li><b>$DD</b>: Day</li><li><b>$MM</b>: Month</li><li><b>$YY</b>: Short year</li><li><b>$YYYY</b>: Long year</li><li><b>$HOURS</b>: Hours</li><li><b>$MINUTES</b>: Minutes</li><li><b>$SECONDS</b>: Seconds</li><li><b>$TZ</b>: Timezone</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Mail index - Date format
//
//! defvar: indexdateshortformat
//! Set a format for the date if it is within the last week.<br />If nothing set, use the <i>&quot;normal&quot;</i> full output, specified in <i>&quot;Mail index - Date format&quot;</i> above.<br />You can use the following replacements :<ul><li><b>$WDAY</b>: Short format weekday-name (eg. Fri for friday)</li><li><b>$WEEKDAY</b>: Full weekday-name (like Thursday)</li><li><b>$HOURS</b>: Hours</li><li><b>$MINUTES</b>: Minutes</li><li><b>$SECONDS</b>: Seconds</li><li><b>$TZ</b>: Timezone</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Mail index - Use short date
//
//! defvar: signaturecols
//! Change the max line length for signature box in the preferences page
//!  type: TYPE_STRING
//!  name: User interface dialogs: Pages - Setup page, size of signature
//
//! defvar: replyseparator
//! Text separator for reply / reply to all messages.
//!  type: TYPE_STRING
//!  name: Compose mail:Reply Separator
//
//! defvar: forwardseparator
//! Text separator for forward messages.
//!  type: TYPE_STRING
//!  name: Compose mail:Forward Separator
//
//! defvar: composesavemail
//! If enabled, mail will by default be saved when sending, but a checkbox with 'Don't save this mail' is shown. If disabled, mail will not be sent by default.
//!  type: TYPE_FLAG
//!  name: Outgoing mail:By default save composed mail into sentmail folder
//
//! defvar: fwdsubjpf
//! This is the prefix to insert before the subject when forwarding a mail.
//!  type: TYPE_STRING
//!  name: Compose mail:Forward subject prefix
//
//! defvar: fwdfrom
//! This is the prefix to insert before the sender's mail address.
//!  type: TYPE_STRING
//!  name: Compose mail:Forward from prefix
//
//! defvar: fwdto
//! This is the prefix to insert before the To's mail address.
//!  type: TYPE_STRING
//!  name: Compose mail:Forward to prefix
//
//! defvar: fwdreplyto
//! This is the prefix to insert before the Reply-To mail address.
//!  type: TYPE_STRING
//!  name: Compose mail:Forward reply-to prefix
//
//! defvar: fwdcc
//! This is the prefix to insert before the Carbon Copy mail address.
//!  type: TYPE_STRING
//!  name: Compose mail:Forward cc prefix
//
//! defvar: fwddate
//! This is the prefix to insert before the Date of the mail.
//!  type: TYPE_STRING
//!  name: Compose mail:Forward date prefix
//
//! defvar: organization
//! Allow the user to specify the organization he belongs to.
//!  type: TYPE_FLAG
//!  name: Features:Organization
//
//! defvar: displayinlineimages
//! List of MIME-subtypes to display inline.<br />Enter one subtype per line, examples are :<br />gif, ief, jpeg, png, tiff, x-cmu-raster, x-ms-bmp, x-portable-anymap, x-portable-bitmap, x-portable-graymap, x-portable-pixmap, x-rgb, x-xbitmap, x-xpixmap, x-xwindowdump
//!  type: TYPE_TEXT_FIELD
//!  name: Features:Display inline images
//
//! defvar: notifysent
//! Display a "sent message" window to the user.
//!  type: TYPE_FLAG
//!  name: Outgoing mail:Sent message notification
//
//! defvar: stayinattachments
//! If <b>yes</b>, the user will <b>not</b> automatically go back to the compose screen after adding an attachment.
//!  type: TYPE_FLAG
//!  name: User interface dialogs:Pages - Compose page, Stay in attachments screen
//
//! defvar: mdntype
//! MUA Mail Delivery Notification type.
//!  type: TYPE_STRING_LIST
//!  name: Outgoing mail:MDN support
//
//! defvar: rxmlparsercalls
//! Maximum calls to parse_rxml
//!  type: TYPE_INT|VAR_EXPERT
//!  name: Debug:Maximum calls to parse_rxml
//
//! defvar: foldersinfo
//! Show some folders information: number of (total, recent, unseen) messages, folder size, ...
//!  type: TYPE_FLAG
//!  name: User interface dialogs:Display folders information
//
//! defvar: buggyimap
//! Some IMAP servers (e.g. Courier 1.3) are not RFC-compliant.<br />For example, they will not return "[TRYCREATE]" when appending a message in a non-existent folder.<br /> This will enable various workarounds to make CAMAS work as needed. <br /><hr />RFC2060, section 6.3.11 (APPEND Command)<br /><pre>If the destination mailbox does not exist, a server <b>MUST</b> return an
//!error, and <b>MUST NOT</b> automatically create the mailbox.  Unless it is
//!certain that the destination mailbox can not be created, the server
//!<b>MUST</b> send the response code "[TRYCREATE]" as the prefix of the text
//!of the tagged NO response.  This gives a hint to the client that it
//!can attempt a CREATE command and retry the APPEND if the CREATE is
//!successful.</pre>
//!  type: TYPE_FLAG|VAR_EXPERT
//!  name: Incoming mail:IMAP server is not RFC-compliant
//
//! defvar: upgrading
//! CAMAS will read 'IMHO Preferences' if 'CAMAS Preferences' are not found.
//!  type: TYPE_FLAG
//!  name: Compatibility:Upgrading from IMHO to CAMAS
//
//! defvar: defaultlayout
//! The layout to use if none found in the old preferences files.
//!  type: TYPE_STRING
//!  name: Compatibility:Layout to use
//
//! defvar: frommaxlength
//! Set the max length for the 'sender' field. If set to 0, the whole string will be displayed.
//!  type: TYPE_INT|VAR_MORE
//!  name: User interface dialogs:Mail index - Max length for the 'sender' field
//
//! defvar: subjectmaxlength
//! Set the max length for the 'subject' field. If set to 0, the whole string will be displayed.
//!  type: TYPE_INT|VAR_MORE
//!  name: User interface dialogs:Mail index - Max length for the 'subject' field
//
//! defvar: foldernamemaxlength
//! Set the max length for a new mailbox name. If set to 0, there will be no check.
//!  type: TYPE_INT|VAR_MORE
//!  name: User interface dialogs:Folders list - Max length allowed for a mailbox name
//
//! defvar: subjectlayer
//! Show the full subject.
//!  type: TYPE_FLAG|VAR_EXPERT
//!  name: User interface dialogs:Mail index - Show full subject in a floating table
//
//! defvar: quota
//! Set an informational quota for the mailboxes (in kB). If set to 0, no quota will be displayed. Provided rxml tag: <b>&lt;imho_quota&gt;</b> (the quota itself), and <b>&lt;imho_use&gt;</b> for the percentage of the quota used (rounded to an integer).
//!  type: TYPE_INT
//!  name: User interface dialogs:Folders List - Mailbox quota
//
//! defvar: mime_remap
//! Change the MIME type of mail or attachments. Format is old_type/old_subtype:new_type/new_subtype
//!  type: TYPE_TEXT_FIELD
//!  name: User interface dialogs:Readmail - Change MIME types
//
//! defvar: jslogout
//! If set, JavaScript code will be added to close the session if the user closes the window instead of following a 'logout' link.
//!  type: TYPE_FLAG
//!  name: Features:Add JavaScript to close the session
//
//! defvar: gctimeout
//! Sets the delay (in seconds) between two garbage collections.
//!  type: TYPE_INT
//!  name: Internals:Garbage collecting frequency
//
//! defvar: sslfaulty
//! If set, SSL will be disabled if the client has <b>id->supports->ssl_faulty</b> set.
//!  type: TYPE_FLAG
//!  name: URL:No SSL if ssl_faulty
//
//! defvar: imapserver
//! FOR COMPATIBILITY ONLY. DON'T USE! The address of the IMAP-server.
//!  type: TYPE_STRING|VAR_EXPERT
//!  name: Incoming mail:IMAP server address.
//
//! defvar: imapport
//! FOR COMPATIBILITY ONLY. DON'T USE! The portnumber of the IMAP-server.
//!  type: TYPE_INT|VAR_EXPERT
//!  name: Incoming mail:IMAP server-port
//
//! defvar: maildomain
//! FOR COMPATIBILITY ONLY. DON'T USE! The default mail domain, i.e the string after the '@' in the mail address in outgoing mail.
//!  type: TYPE_STRING|VAR_EXPERT
//!  name: Outgoing mail:Default mail domain
//
//! defvar: systz
//! Set there the timezone to the correct timezone of the machine running this CAMAS.
//!  type: TYPE_STRING_LIST
//!  name: System Timezone
//
//! defvar: attachmenttype
//! When reading a mail with attachments, add the type of the file.
//!  type: TYPE_FLAG
//!  name: User interface dialogs:Readmail - Show the attachments type
//
