/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2001 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_formbuttons.pike,v 1.11.2.6.2.7 2001/09/24 11:10:57 kiwi Exp $
 */
//
//! module: CAMAS: Form Button provider
//!  This module gives to CAMAS the correct buttons needed for the template.
//!  It choose for you the imagebutton, form button or even gbutton graphics
//!  button.<br />
//!  <b>This module is automatically selected if you select "CAMAS: Main
//!  module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PARSER
//! cvs_version: $Id: camas_formbuttons.pike,v 1.11.2.6.2.7 2001/09/24 11:10:57 kiwi Exp $
//
#include <module.h>
inherit "module";
inherit "caudiumlib";

constant cvs_version   = "$Id: camas_formbuttons.pike,v 1.11.2.6.2.7 2001/09/24 11:10:57 kiwi Exp $";
constant module_type   = MODULE_PARSER;
constant module_name   = "CAMAS: Form Button provider";
constant module_doc    = "This module gives to CAMAS the correct buttons "
                         "needed for the template. It choose for you the "
		         "imagebutton, form button or even gbutton "
			 "graphics buttons.<br /><b>This module is automatically "
                         "selected if you select \"CAMAS: Main module\"."
                         "</b>";
constant module_unique = 1;
constant thread_safe = 1;

// For debug
//#define CAMAS_DEBUG
#ifdef CAMAS_DEBUG
# define DEBUG(X) if(QUERY(debug)) perror("CAMAS BUTTONS: "+X+"\n");
#else
# define DEBUG(X)
#endif

// For default values in defvars
#define DEFAULTGBUTTON "<gbutton-url font='lucida sans unicode' bgcolor='lightblue'>$TEXT$</gbutton-url>"

// Definition to give us an easy way to make the code and understand it :)
#define CAMAS_MODULE  id->conf->get_provider("camas_main")

// Mapping used for camas new methods...
mapping buttonlist;

// Functions used to hide the defvar when not used
int hide_buttonpathcaudium () {
  if (QUERY(mode) == "pre-camas" )
    return !sizeof (QUERY (buttonpath));
  return 1;
}

int hide_buttonnocache() {
  if ((QUERY(mode) == "pre-camas" ) && (sizeof(QUERY(buttonpath))>0) && QUERY(buttonpathcaudium))
   return 0;
  return 1;
}

int hide_oldbuttonsystem() {
  if (QUERY(mode) == "pre-camas") return 0;
  return 1;
}

int hide_gbutton() {
  if (QUERY(mode) == "gbutton") return 0;
  return 1;
}

int hide_camasbasic() {
  if (QUERY(mode) == "camas-mode-basic") return 0;
  return 1;
}

// Module creation & configuration
void create()
{
#ifdef CAMAS_DEBUG
  // Debug
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the calls / errors into Caudium "
         "error log ?");
#endif
  // Mode setup
  defvar("mode","form-button","Formbutton mode",TYPE_STRING_LIST,
         "The mode used for providing image button.<br />"
	 "<ul>"
	 " <li><b>pre-camas</b> mode is the mode used when Image button was in the main module.</li>"
	 " <li><b>form-button</b> mode is the standart mode for those awfull forms buttons.</li>"
	 " <li><b>camas-mode-basic</b> mode is the new mode with flat file with images to load "
	 "     for every actions</li>"
	 " <li><b>gbutton</b> mode is allmost same as <i>form-button</i> but it use instead "
	 "     gbutton module to make images (work only when Caudium 1.1.x is used)</li>"
	 "</ul>" 
	 "<b>Warning</b>: from <i>camas-mode-basic</i> don't forget"
	 "to reload the module for file been loaded by this module.",
	 ({ "pre-camas", "form-button", "gbutton", "camas-mode-basic" }));
  // Old CAMAS premode usage
  defvar("buttonpath","","Image button http prefix", TYPE_STRING,
         "RFC 2616 (section 3.2.2) url prefix to find the form button images. If this prefix is "
	 "empty then standard form button will be used.<br />You can use either <i>http://some."
	 "where.is.this.images.com/img/</i> or caudium VFS directory instead",1,
         hide_oldbuttonsystem);
  defvar("buttonpathcaudium",0,"Image button are stored localy", TYPE_FLAG,
         "If you don't use RFC 2616 url, and the images are stored on the local webserver only. "
	 "Then RXML will be added to check if the image is present realy on the filesystem and "
	 "display form button instead if there is no image present. <br />"
         "<b>***Warning***</b> this option is <b>DANGEROUS</b> and can "
         "break the RXML parser and give some unattended effects. <b>DO NOT</b> use this "
         "option in a production enviroment. <i>You've been warned</i>.",0,hide_buttonpathcaudium);
  defvar("buttonnocache",1,"Do no cache accesses to image button", TYPE_FLAG,
         "When set, this option allow <b>developpers</b> to ask this module to not "
         "cache file test. <b>Warning</b>: this option should decrease performance in production "
         "enviroment", 0, hide_buttonnocache); 
  defvar("buttonpathextension",".gif","Extension to append to image files",TYPE_STRING,
         "File extension to use for the From button.",0,hide_buttonpathcaudium);
  // Gbutton part
  //  Gbutton is simple, it does only a single type of button
  defvar("gsyntax",DEFAULTGBUTTON,"GButton syntax", TYPE_STRING,
         "GButton syntax used for autocreate buttons for all form actions. "
         "Use <b>$TEXT$</b> to set the place where button's text will be "
         "added."
         "Please see &lt;gbutton-url&gt; manual for more informations.",
         0,hide_gbutton);
  defvar("gbuttonutf8",1,"Text is UTF8", TYPE_FLAG,
         "When set, connect directly to gbutton using module provider method, "
         "so UTF8 text and encoding are not lost by the RXML parser.<br />"
         "Note: You need Caudium 1.1.x and Gbutton v1.16 or news to enable this.",
         0, hide_gbutton);
  // CAMAS basic mode
  defvar("cbfile","modules/camas/formbuttons/camasbasic.txt","Camas Basic Mode file", TYPE_FILE,
         "The file where is defined the RXML code according the CAMAS's button "
	 "definitions. (add there some description about this format).<br/>"
	 "Path is relative to Caudium server.",
	 0,hide_camasbasic);
}

// Checking the variables
string check_variable(string var, mixed value)
{
 switch(var)
 {
   case "cbfile":
     if (!Stdio.is_file(value) && (QUERY(mode) == "camas-mode-basic")) 
       return "File doesn't exist or is not a plain file. Please check again.";
     break;
 }
 return 0;
}

// Functions depending how the module is setup

// === FORM BUTTONS ===
string form_buttons(string action, string value, void|string onClick)
{
 string out;	// Out string
 out = "<input type=\"submit\" name=\"" + action + "\" value=\"" + value +"\"";
 if (onClick)	// Some javascript magic stuff to add there
   out += " onClick=\"" + onClick + "\"";
 out += " />";	// End of tag (and XML compliant !)

 return out;	// Return the stuff
}

// === OLD PRE-CAMAS ===
string oldimgform_buttons(string img, string action, string value, void|string onClick)
{
 string out;	// out string
 out = "<input type=\"image\" name=\"" + action + "\" border=\"0\" value=\"" + value +"\"";
 out+= " src=\""+QUERY(buttonpath)+"/"+img+QUERY(buttonpathextension)+"\"";
 if (onClick)
  out+= " onClick=\"" + onClick + "\"";
 out += " />";
 return out;
}

string oldprecamas_buttons(string img, string action, string value, object id, void|string onClick)
{
 string out;	// out String
 
 if(QUERY(buttonpathcaudium))	// some magic used to find if the file exist or no
 { 
   string foo;
   if (QUERY(buttonnocache))
     foo=id->conf->try_get_file(fix_relative(QUERY(buttonpath)+"/"+img+QUERY(buttonpathextension),id),id,1,1);
   else
     foo=id->conf->try_get_file(fix_relative(QUERY(buttonpath)+"/"+img+QUERY(buttonpathextension),id),id,1);
   if (foo)
   {
     // File exist so we can use image formbuttons :)
     out = oldimgform_buttons(img, action, value, onClick);
   }
   else
   {
     // File doesn't exist only forms will be used
     out = form_buttons(action, value, onClick);
   }
 }
 else
 {
   // File is not local then consider that all files are there
   // if not... this is not out problem
   out = oldimgform_buttons(img, action, value, onClick);
 }
 return out;
}

// === GBUTTON MODE ===
string gbutton_buttons(mapping arg, object id)
{
 string out, temp;	// out string
 object gbp = id->conf->get_provider("gbutton");

 if (objectp(gbp) && gbp->tag_gbutton && QUERY(gbuttonutf8)) {
  string gb=gbp->tag_gbutton("gbutton-url",
                             (["font":"arial",	// Redax, why arial ?
                               "bg":"#c0c0c0",
                               "encoding":"utf-8"]),
                             arg->utf8text ? MIME.decode_base64(arg->utf8text):arg->text,
                             id);
  out  = "<input type=\"image\" name=\"" + arg->action + "\" border=\"0\" value=\"";
  out += arg->text + "\" src=\""+gb+"\"";
 } else {
  out  = "<input type=\"image\" name=\"" + arg->action + "\" border=\"0\" value=\"";
  out += arg->text +"\" src=\""+replace(QUERY(gsyntax),"$TEXT$",arg->text)+"\"";
 }
 if (arg->onClick)
  out+= " onClick=\"" + arg->onClick + "\"";
 out += " />";
 return out;
}

// === CAMAS BASIC ===
#define LINE_COMMENT	0x00
#define LINE_DIRECTIVE	0x01

class CamasBasicFile
{
   string datafile;	// file to be used...
   /*
    * PRIVATE FUNCTIONS
    */
   static int qualify_line(string line)
   {
     if (!line || line == "")
        return LINE_COMMENT;
     if (sizeof(line) >=1 && line [0..0] == "#")
        return LINE_COMMENT;
     if (sizeof(line) >=1 && line [0..0] == " ")
        return LINE_COMMENT;
     if (sizeof(line/"\t") == 2)
        return LINE_DIRECTIVE;
     return -1;
   }

   /*
    * Public functions
    */
   void create(string path)
   {
     datafile = Stdio.read_bytes(path);
     if (datafile == 0) throw("Unable to read "+path+" file !");
   }
   
   mapping doindex()
   {
     mapping out = ([ ]);
     foreach((datafile-"\r")/"\n",string foo)
     {
       if(qualify_line(foo) == LINE_DIRECTIVE)
       {
         array l= foo / "\t";

	 out += ([ l[0] : l[1] ]);
       }
     }
     return out;
   }
}

string camasbasic_buttons(string img, string action, string value, void|string onClick)
{
  string out, temp;

  temp = buttonlist[lower_case(img)];

  if (temp == 0) out = form_buttons(action, value, onClick);
  else out = replace ( temp, ({ "$TEXT$","$ACTION$" }), ({ value, action }) );
  return out;
}

// Tags now

//
//! tag: camas_formbutton
//!  Gives the good html string to camas to sent the right button/image
//!  or gbutton etc... (WARNING: This tag will be <b>REMOVED</b> in 1.2 so do
//!  not use it in your templates).
//! attribute: text
//!  The plain text message to write if there is no options
//! attribute: file
//!  The name of the file searched for a image on VFS or on another server
//! attribute: action
//!  The CAMAS's action to be done with this function
//! attribute: onClick
//!  Javascript stuff when this button is clicked (optional)
//
string tag_camas_formbutton(string tag_name, mapping arg, object id, object file)
{
  string out;	// out string...
  
  DEBUG("tag <camas_formbutton> called");

  // Some sanity checks : text, file and action are mandatory, return an error
  // if they are not present 
  if (!arg->text && !arg->file && !arg->action) 
   return "<!-- bad syntax, arguement text, file and action are all needed. -->";
 
  DEBUG("tag <camas_formbutton> args : text="+arg->text+", file="+arg->file+", action="+arg->action);  

  DEBUG("called in "+QUERY(mode));
  DEBUG(sprintf("Current layout : %O", id->misc->session_variables->layout));
  DEBUG(sprintf("Current language : %O", id->misc->session_variables->language));

  // now we can do the right work
  out = "";
  switch (QUERY(mode))
  {
    case "pre-camas":
          if (sizeof(QUERY(buttonpath)) == 0)
            out = form_buttons(arg->action, arg->text, arg->onClick);
          else
          { 
            out = oldprecamas_buttons(arg->file, arg->action, arg->text, id, arg->onClick);
          }
          break;
    case "form-button":
          out = form_buttons(arg->action, arg->text, arg->onClick);
          break;
    case "gbutton":
          out = gbutton_buttons(arg,id);
          break;
    case "camas-mode-basic":
          out = camasbasic_buttons(arg->file, arg->action, arg->text, arg->onClick);
	  break;
    case "camas-mode-advanced":
             out = "<!-- Not implemented yet -->"; break;
    default:
             out = "<!-- What ???? -->";
  }
  DEBUG("Output sent to the parser : "+out);
  return out;
}

// Tags that this module provides
mapping query_tag_callers()
{
  return([
          "camas_formbutton": tag_camas_formbutton
         ]);
}

// Containers now

// Containers that this module provides
mapping query_container_callers() { 
  return ([ ]);
}

// Start of the module
void start(int cnt, object conf)
{
 module_dependencies(conf, ({ "gbutton" }));
 if (QUERY(mode) == "camas-mode-basic") buttonlist = CamasBasicFile(QUERY(cbfile))->doindex();
 else buttonlist = 0;	// To help gc...
}


/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the calls / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: mode
//! The mode used for providing image button.<br /><ul> <li><b>pre-camas</b> mode is the mode used when Image button was in the main module.</li> <li><b>form-button</b> mode is the standart mode for those awfull forms buttons.</li> <li><b>camas-mode-basic</b> mode is the new mode with flat file with images to load      for every actions</li> <li><b>gbutton</b> mode is allmost same as <i>form-button</i> but it use instead      gbutton module to make images (work only when Caudium 1.1.x is used)</li></ul><b>Warning</b>: from <i>camas-mode-basic</i> don't forgetto reload the module for file been loaded by this module.
//!  type: TYPE_STRING_LIST
//!  name: Formbutton mode
//
//! defvar: buttonpath
//! RFC 2616 (section 3.2.2) url prefix to find the form button images. If this prefix is empty then standard form button will be used.<br />You can use either <i>http://some.where.is.this.images.com/img/</i> or caudium VFS directory instead
//!  type: TYPE_STRING
//!  name: Image button http prefix
//
//! defvar: buttonpathcaudium
//! If you don't use RFC 2616 url, and the images are stored on the local webserver only. Then RXML will be added to check if the image is present realy on the filesystem and display form button instead if there is no image present. <br /><b>***Warning***</b> this option is <b>DANGEROUS</b> and can break the RXML parser and give some unattended effects. <b>DO NOT</b> use this option in a production enviroment. <i>You've been warned</i>.
//!  type: TYPE_FLAG
//!  name: Image button are stored localy
//
//! defvar: buttonnocache
//! When set, this option allow <b>developpers</b> to ask this module to not cache file test. <b>Warning</b>: this option should decrease performance in production enviroment
//!  type: TYPE_FLAG
//!  name: Do no cache accesses to image button
//
//! defvar: buttonpathextension
//! File extension to use for the From button.
//!  type: TYPE_STRING
//!  name: Extension to append to image files
//
//! defvar: gsyntax
//! GButton syntax used for autocreate buttons for all form actions. Use <b>$TEXT$</b> to set the place where button's text will be added.Please see &lt;gbutton-url&gt; manual for more informations.
//!  type: TYPE_STRING
//!  name: GButton syntax
//
//! defvar: gbuttonutf8
//! When set, connect directly to gbutton using module provider method, so UTF8 text and encoding are not lost by the RXML parser.<br />Note: You need Caudium 1.1.x and Gbutton v1.16 or news to enable this.
//!  type: TYPE_FLAG
//!  name: Text is UTF8
//
//! defvar: cbfile
//! The file where is defined the RXML code according the CAMAS's button definitions. (add there some description about this format).<br/>Path is relative to Caudium server.
//!  type: TYPE_FILE
//!  name: Camas Basic Mode file
//
