/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: DateTools.pmod,v 1.12.2.2 2004/01/21 11:57:31 vida Exp $
 */

//! file: DateTools.pmod
//!  Date tools for CAMAS
//! cvs_version: $Id: DateTools.pmod,v 1.12.2.2 2004/01/21 11:57:31 vida Exp $

#include <module.h>
#include <camas/msg.h>          // Language macros
#include <camas/language.h>     // Language sym names M_*
#include <camas/globals.h>

/*
  Date-formats parsed

  Non-RFC822 conformant:
  26 Jul 2000 21:51:41 -0000,  2000-Jul-26 21:51:41 -0000
  2000-07-26 21:51:41 -0000,  Tue, 27 Jul 2000 15:31:30
  Thu, 27 Jul 2000 17:31:30 +0200 (CEST)

  RFC822 conformant:
  Thu, 27 Jul 2000 17:31:30 +0200,  Thu, 27 Jul 2000 17:31 +0200
  Tue, 30 May 2000 01:13:42 PDT,  Tue, 30 May 2000 01:13 PDT

  Output:
  Localized time as an array, see DD_INDEX, MM_INDEX, etc..
  => ({ "31", "08", "00", "14", "47", "53", "2000" , "+0100" , "0" });
  TODO: to be deprecated when removing IMHO screens
*/

#define ZERO2D(x) sprintf ("%02d", x)

void|array parse_date (string s) {
  object o = cache_lookup ("camas_calendar", s);
  mixed err;
  if(!o)
  {
    err = catch {
                o = Calendar.dwim_time (s)->set_timezone ("localtime");
              };
    cache_set("camas_calendar", s, o, CACHE_TIMEOUT);
  }
  if (err) {
    DATE_DEBUG ("Calendar.dwim_time ('" + s + "') failed\n");
    return;
  }
  else {
    array res = ({
                   (string)ZERO2D(o->month_day ()),
                   (string)ZERO2D(o->month_no ()),
                   ((string)o->year_no ())[2..3],
                   (string)ZERO2D(o->hour_no ()),
                   (string)ZERO2D(o->minute_no ()),
                   (string)ZERO2D(o->second_no ()),
                   (string)o->year_no (),
                   (string)o->tzname (),
                   (string)o->week_day (),
                   (string)o->year_day ()
                 });
    DATE_DEBUG(sprintf("parse_date(%s)=%O\n", s, res));
    return res;
  }
}

//! method: void|int unix_date(string s)
//!  Returns the unix date (seconds since 19700101) of the mail
//!  or 0 if it can't parse the date
//! returns:
//!  Standart time() date (since the epoch) in seconds. Returns a
//!  int when the date is parsable in the mail or void if there is
//!  an error.
//! arg: string s
//!  The date to convert into second since the epoch
void|int unix_date(string s)
{
  object o = cache_lookup ("camas_calendar", s);
  int unixtime = 0;
  mixed err;
  
  if(!o)
  {
    err = catch {
                o = Calendar.dwim_time (s)->set_timezone ("localtime");
              };
    cache_set("camas_calendar", s, o, CACHE_TIMEOUT);
  }

  if (err) {
    DATE_DEBUG ("Calendar.dwim_time ('" + s + "') failed\n");
    return;
  }
  else {
    unixtime = o->unix_time();
  }

  return unixtime;
}

string sortable_date(void|string rfc822date, void|string imapdate)
{
  string res = cache_lookup("camas_sortable_date", rfc822date + imapdate);
  if(!res)
  {
    res = _sortable_date(rfc822date, imapdate);
    cache_set("camas_sortable_date", rfc822date + imapdate, res, CACHE_TIMEOUT);
  }
  return res;
}

private string _sortable_date(void|string rfc822date, void|string imapdate) {
  string d = ctime(time(1)); // fall-back
  if (rfc822date && rfc822date != "")
    d = rfc822date;
  else
    if (imapdate && imapdate != "")
      d = imapdate;

  void|array p = parse_date(d);
  if (p)
    return sprintf("%s%s%s%s%s%s%s",
                   p[YYYY_INDEX], p[MM_INDEX], p[DD_INDEX],
                   p[HOURS_INDEX], p[MINUTES_INDEX], p[SECONDS_INDEX],
                   p[TZ_INDEX]);
  else {
      DATE_DEBUG("sortable_date: unparseable date(s): \"" + (rfc822date||"") +
             "\", \"" + (imapdate||"") + "\"\n");
    return "???????????????????";
  }
}

//! method: string mailindex_date(object id, void|string rfc822date, void|string imapdate, string indexdateshortformat, string indexdatelongformat)
//!  Returns a localized human readable date.
//!  The rfc822date is used for computing the date. The imapdate is used if the RFC822 date is not available.
//!  If the date is less than one week ago, the indexdateshortformat is used, otherwise, that's the indexlongdateformat.
//!  These results are cached for more performance.
//! arg: object id
//!  The Caudium request id object
//! arg: void|string rfc822date
//!  The RFC822 date to use
//! arg: void|string imapdate
//!  The IMAP date to use
//! arg: string indexdateshortformat
//!  The short date format to use
//! arg: string indexdatelongformat
//!  The long date format to use
//! returns:
//!  The date to show in the MAILINDEX screen given the rfc822date/imapdate and the indexdateshortformat/indexdatelongformat
string mailindex_date(object id, void|string rfc822date, void|string imapdate, 
    string indexdateshortformat, string indexdatelongformat)
{
  string res = cache_lookup("camas_mailindex_date", rfc822date + imapdate
      + indexdateshortformat + indexdatelongformat + CSESSION->language);
  if(!res)
  {
    res = _mailindex_date(id, rfc822date, imapdate
        , indexdateshortformat, indexdatelongformat);
    cache_set("camas_mailindex_date", rfc822date + imapdate
        + indexdateshortformat + indexdatelongformat + CSESSION->language, res, CACHE_TIMEOUT);
  }
  return res;
}

private string _mailindex_date(object id, void|string rfc822date, void|string imapdate,
  string indexdateshortformat, string indexdatelongformat)
{
  string d = ctime(time(1)); // fall-back
  array from = ({ });
  array to   = ({ });
  mapping lt;
  int z;

  if (rfc822date && rfc822date != "")
    d = rfc822date;
  else
    if (imapdate && imapdate != "")
      d = imapdate;

  void|array pdate = parse_date(d);
  if (!pdate) {
      DATE_DEBUG("mailindex_date: unparseable date(s): \"" + (rfc822date||"") +
             "\", \"" + (imapdate||"") + "\"\n");
    pdate = ({ "??", "??", "??", "??", "??", "??", "????", "+????", "?", "?" });
  }

  int current_time = time();
  int mail_time    = unix_date(d);
  
  // The short format is used if the mail came during last week (604800 seconds)
  if (sizeof(indexdateshortformat) && mail_time && (current_time - mail_time) < 604800)
  {
    DATE_DEBUG(sprintf("current_time is %d, mail day is %d -> displaying short format\n",
        current_time, mail_time));
    from = ({
              "$WDAY", "$WEEKDAY", "$TZ",
              "$HOURS", "$MINUTES", "$SECONDS"
            });

    // default value
    string wdayname = "?????";
    
    int today_yday = Calendar.now()->year_day();  // the year day of today
    int theday_yday = (int)pdate[YDAY_INDEX];    // the year day of the mail
    if (theday_yday == today_yday)
      wdayname = MSG(M_TODAY);
    else
    {
      // find the localized week day name
      switch (((int)pdate[WDAY_INDEX]))
      {
        case 1: wdayname = MSG(M_MONDAY); break;
        case 2: wdayname = MSG(M_TUESDAY); break;
        case 3: wdayname = MSG(M_WEDNESDAY); break;
        case 4: wdayname = MSG(M_THURSDAY); break;
        case 5: wdayname = MSG(M_FRIDAY); break;
        case 6: wdayname = MSG(M_SATURDAY); break;
        case 7: wdayname = MSG(M_SUNDAY); break;
        default: break;
      }
    }

    to = ({
            (wdayname[0..2]), wdayname, pdate[TZ_INDEX],
            pdate[HOURS_INDEX], pdate[MINUTES_INDEX], pdate[SECONDS_INDEX]
          });
    return replace (indexdateshortformat, from, to);
  }

  // if we are here, it's the longer format
  // TODO: would be cool to have a localized full name month
  from = ({
            "$DD", "$MM", "$YY", "$TZ",
            "$HOURS", "$MINUTES", "$SECONDS", "$YYYY"
          });
  to   = ({
            pdate[DD_INDEX], pdate[MM_INDEX], pdate[YY_INDEX], pdate[TZ_INDEX],
            pdate[HOURS_INDEX], pdate[MINUTES_INDEX], pdate[SECONDS_INDEX], pdate[YYYY_INDEX]
          });
  return replace (indexdatelongformat, from, to);
}

void|mapping parse_delay(string s) {
  object o = cache_lookup ("camas_calendar", s);
  mixed err;
  if(!o)
  {
    err = catch {
                o = Calendar.dwim_time (s)->set_timezone ("localtime");
              };
    cache_set("camas_calendar", s, o, CACHE_TIMEOUT);
  }
  if (err) {
    DATE_DEBUG ("Calendar.dwim_time ('" + s + "') failed\n");
    return;
  }
  else {
    int delay = time () - o->unix_time ();

    mapping pdelay = ([ "day": 0, "hour": 0, "min": 0, "sec": 0 ]);

    if(delay>0)
    {
      float day, hour, min, sec;

      day  = (float)delay / 86400;
      pdelay->day     = (int)floor(day);

      hour = (day - pdelay->day) * 24;
      pdelay->hour    = (int)floor(hour);

      min  = (hour - pdelay->hour) * 60;
      pdelay->min     = (int)floor(min);

      sec  = (min - pdelay->min) * 60;
      pdelay->sec = (int)floor(sec);
    }

    return pdelay;
  }
}

// TODO: to be deprecated
// keeped because of old imho templates
string mailindex_delay(string|string rfc822date, void|string imapdate,
  int delaystepnumber, string indexdelayformat) {
  string d = ctime(time(1));                         // fall-back
  string delay = "&nbsp;";                           // Delay to write on the page
  string days = "" , hours = "", min = "", sec = "";
  string delaystring = "delay";
  array from = ({ });
  array to   = ({ });

  if (rfc822date && rfc822date != "")
    d = rfc822date;
  else
    if (imapdate && imapdate != "")
      d = imapdate;

  void|mapping pdelay = parse_delay(d);

  if (!pdelay) {
      DATE_DEBUG("CAMAS_MAIN: mailindex_delay: unparseable date(s): \"" + (rfc822date||"") +
             "\", \"" + (imapdate||"") + "\"\n");
      pdelay = ([ "day":"??", "hour":"??", "min":"??", "sec":"??" ]);
  }

  // - the image format -> no more needed with new tags
  // - the step duration -> user prefs
  // - the step incrementation mode: linear, dichotomic... -> user prefs
  // - set a per user or per mailbox/directories step duration

  if((int)pdelay["day"]<=(delaystepnumber)){
    delaystring += (string)pdelay["day"];
  }
  else{
    delaystring += (string)delaystepnumber;
  }

  days = (string)pdelay["day"];
  if((int)pdelay["hour"] < 10)
    hours = "0" + (string)pdelay["hour"];
  else
    hours = (string)pdelay["hour"];
  if((int)pdelay["min"] < 10)
    min = "0" + (string)pdelay["min"];
  else
    min = (string)pdelay["min"];
  if((int)pdelay["sec"] < 10)
    sec = "0" + (string)pdelay["sec"];
  else
    sec = (string)pdelay["sec"];
  from = ({ "$DDAYS", "$DHOURS", "$DMINUTES", "$DSECONDS" });
  to = ({ days, hours, min, sec });
  delay += replace (indexdelayformat, from, to);
  return delay;
}

//! function : compute_delaystep
//!  Returns the delay step of the current mail
//! input : unix time of the mail
//! output : current delay step
int compute_delaystep(string delaycalc, int delaytime, int unixtime)
{
  int delaystep = 0;

  // Computing the delay in seconds
  int delay = time() - unixtime;

  // We don't want negative delays
  if(delay<0)
    delay=0;

  // Delay is now in minutes
  delay = delay/60;

  switch(delaycalc)
  {
  case "dichotomic":
    // TODO: computation for dichotomic delay
  case "linear":
  default:
    if(delaytime==0)
      delaytime=1440;
    delaystep = delay/delaytime;
  }

  delaystep = (int)floor((float)delaystep);

  return delaystep;
}

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */
