/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_addressbook_ldap.pike,v 1.28 2004/01/08 20:53:17 vida Exp $
 */

#if constant(Protocols.LDAP) && constant(Protocols.LDAP.client)
#define LDAP_OPERATION_ADD 0
#define LDAP_OPERATION_DELETE 1
#define LDAP_OPERATION_REPLACE 2

#include <module.h>
#include <camas/globals.h>
#include <camas/addressbook.h>
inherit "module";

constant cvs_version="$Id: camas_addressbook_ldap.pike,v 1.28 2004/01/08 20:53:17 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Addressbook LDAP";
constant module_doc  = "LDAP addressbook for CAMAS. This module provides LDAP specific methods for "
                       "managing addressbook. You can have several copies of this module if you have "
                       "several LDAP servers or base DN where you want Camas to search.<br/>"
                       "This addressbook uses the schema from objectClass <i>inetOrgPerson</i>, "
                       "<i>organizationalPerson</i>, <i>person</i>, <i>country</i> and <i>top</i> "
                       "indexed by email address.<br/>"
                       "See <a href=\"http://community.roxen.com/developers/idocs/rfc/rfc2798.html\">RFC 2798"
                       "</a> for more informations.";
constant module_unique = 0;
constant thread_safe = 1;

// first maps HTML input name to attributes.
// Change the name of the query in create() after
// you modify this mapping.
constant default_inputs2attributes = ([ 
  AD_NAME: ({ "cn", "the name : first + last" }),
  AD_SURNAME: ({ "sn", "the last name" }),
  AD_NICKNAME: ({ "givenName", "the nickname" }),
  AD_HOMEPHONE: ({ "homePhone", "the home phone number" }),
  AD_WORKPHONE: ({ "telephoneNumber", "the work phone number" }),
  AD_TITLE: ({ "title", "the title in the company" }),
  AD_COMPANY: ({ "o", "the company name" }),
  AD_DEPARTMENT: ({ "ou", "the department in the company" }),
  AD_HOMEPOSTALADDRESS: ({ "homePostalAddress", "the home postal address" }),
  AD_WORKPOSTALADDRESS: ({ "postalAddress", "work postal address" }),
  AD_WORKCITY: ({ "l", "" }),
  AD_WORKSTATE: ({ "st", "" }),
  AD_WORKCOUNTRY: ({ "c", "" }),
  AD_PAGER: ({ "pager", "" }),
  AD_MOBILE: ({ "mobile", "" }),
  AD_URI: ({ "labeledURI", "Uniform Resource Identifier with optional label" }),
  AD_DESCRIPTION: ({ "description", "" }),
  AD_POSTALCODE: ({ "postalCode", "" }),
  AD_MAIL: ({ "mail", "" }),
]);

mapping inputs2attributes;

void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
#endif
  // increase the number x of attrx_ if you modify default_inputs2attributes
  foreach ( indices (default_inputs2attributes), int attr ) 
      defvar("attr2_" + (string)attr, default_inputs2attributes[attr][0],
             "Default properties:" + default_inputs2attributes[attr][0],
             TYPE_STRING, default_inputs2attributes[attr][1]);

  defvar("ldapserver","ldap://localhost","LDAP server",
         TYPE_STRING,
         "Specifies the ldap server to use.");

  defvar("ldapsearchroot", "", "Search root (Base DN)",
         TYPE_STRING,
         "The root DN for every LDAP operations."
         "<p>For example: <i>uid=$LOGIN, o=Your company, c=your country</i>. If you allowed write access to this "
         "addressbook, this DN will also be used to insert contacts at this location.<br/>"
         "For instance if you set <i>uid=$LOGIN, o=Your company, c=your country</i>, DN "
         "<i>mail=myfriend@foobar.com, uid=toto, o=Your company, c=your country</i> will be added "
         "when user toto login and add the email address <i>myfriend@foobar.com</i> in his address book.</p>"
         "<p>Note that only entries under this DN will be returned. For example if you set "
         "<i>o=Your company, c=your country</i>, the content of the object "
         "<i>o=Your company, c=your country</i> "
         "will not be displayed.<br/>"
         "Same example, if you set <i>uid=$LOGIN, o=Your company, c=your country</i> and that that "
         "the user's login is foobar then the entry <i>uid=foobar, o=Your company, c=your country</i> "
         "will not be displayed but "
         "<i>mail=myfriend@mycompany.com, uid=foobar, o=Your company, c=your country</i> will.</p>"
         "The following replacements are "
         "available: <ul><li><b>$ORGANIZATION</b>: The organization sets in user's preferences</li>"
         "<li><b>$LOGIN</b>: The login of the user</li>"
         "<li><b>$NAME</b>: The user name</li>"
         "<li><b>$EMAIL</b>: The email address of the user</li>"
         "<li><b>$LANGUAGE</b>: The user's language</li>"
#if constant (CAMAS.Local.LDAPFilter)
         "<li><b>$MANUAL</b>: Will fetch the filter from the get_root() function in Pike module "
         "CAMAS.Local.LDAPFilter (see file etc/modules/CAMAS.pmod/Local.pmod/LDAPFilter.pmod)</li>"
#endif
         "</ul>");
  
  defvar("ldapfilter", "", "Search filter", TYPE_STRING,
         "The filter to apply for each user search in this address book. Could be empty. "
         "For example you might want to filter results limited to the user's company entries "
         "by adding a filter like ou=$ORGANIZATION.<br/>The following replacements are "
         "available: <ul><li><b>$ORGANIZATION</b>: The organization sets in user's preferences</li>"
         "<li><b>$LOGIN</b>: The login of the user</li>"
         "<li><b>$NAME</b>: The user name</li>"
         "<li><b>$EMAIL</b>: The email address of the user</li>"
         "<li><b>$LANGUAGE</b>: The user's language</li>"
#if constant (CAMAS.Local.LDAPFilter)
         "<li><b>$MANUAL</b>: Will fetch the filter from the get_filter() function in Pike module "
         "CAMAS.Local.LDAPFilter (see file etc/modules/CAMAS.pmod/Local.pmod/LDAPFilter.pmod)</li>"
#endif
         "</ul>");
  

  defvar("ldapuser", "", "User (Bind DN)", TYPE_STRING,
         "The user you should use connect. NOTE! This is ldap user format. "
         "Could be empty");

  defvar("ldappass", "", "Password", TYPE_STRING,
         "The password you should use to connect. Could be empty.");

  defvar("ldapversion", 2, "Version", TYPE_INT_LIST,
         "The LDAP protocol version to use.", ({ 2, 3 }));

  defvar("myname", "Global addressbook", "Name of this addressbook",
         TYPE_STRING, "The name of this addressbook that will appear to the user");

  defvar("allowwrite", 0, "Allow write access", TYPE_FLAG,
         "If set to yes the user will be able to write in this addressbook if "
         "the LDAP server allows this. Usually you don't want to enable this option "
         "for LDAP based addressbook.");

  defvar("allowlist", 0, "Allow listing entire LDAP server content", TYPE_FLAG,
         "If set to yes the user will have a list of every contacts on the "
         "server. This is usually not good unless you have few contacts on the "
         "server");

  defvar("isdefault", 0, "Is default", TYPE_FLAG,
         "If set to yes, users will have this addressbook selected when they "
         "enter the address book screen the first time during their session.");
  
  defvar("autocreate", 0, "Auto create root", TYPE_FLAG,
         "Do we create directory entry in LDAP  for the one specified in Search root "
         "if it doesn't exist ?<br/>"
         "<p>For example, if you set <i>uid=$LOGIN, o=Your company, c=your country</i> "
         "in search root (Base DN) option and that user with $LOGIN <i>foobar</i> "
         "add <i>myfriend@foobar.com</i> in the address book "
         "do we create the entry uid=foobar, o=Your company, c=your country</i> "
         "automatically ?</p>");

  defvar("attrs2create", "objectClass: top\nobjectClass: person\nobjectClass: "
         "organizationalPerson\nobjectClass: inetOrgPerson\nobjectClass: country\n"
         "cn: $NAME\nsn: $NAME\nmail: $EMAIL\nc: Your country", 
         "The attributes of the entry to create if the entry doesn't exist",
          TYPE_TEXT,
         "<p>If a root entry is created as described in the Auto create root option, these "
         "attributes will be added automatically. Usually you don't want to change that.</p>"
         "Put each attribute on one line."
         "<br/>The following replacements are "
         "available: <ul><li><b>$ORGANIZATION</b>: The organization sets in user's preferences</li>"
         "<li><b>$LOGIN</b>: The login of the user</li>"
         "<li><b>$NAME</b>: The user name</li>"
         "<li><b>$EMAIL</b>: The email address of the user</li>"
         "<li><b>$LANGUAGE</b>: The user's language</li>",
         0, hide_defvar_objectClass);
         
}

int hide_defvar_objectClass()
{
  return !QUERY(autocreate);
}

private object connect(object ldap_con)
{
  ldap_con = Protocols.LDAP.client(QUERY(ldapserver));
  ldap_con->bind(QUERY(ldapuser), QUERY(ldappass), QUERY(ldapversion));
  CDEBUG(sprintf("Connecting to %O with user %O and pass %O ldapversion %d\n",
        QUERY(ldapserver), QUERY(ldapuser), QUERY(ldappass), QUERY(ldapversion)));
  return ldap_con;
}

void start(int cnt, object conf)
{
  inputs2attributes = ([ ]);
  foreach(indices(default_inputs2attributes), int attr)
  {
    inputs2attributes += ([ attr: query("attr2_" + (string) attr) ]);
  }
  CDEBUG(sprintf("inputs2attributes=%O", inputs2attributes));
}

string query_provides()
{
  return("camas_addressbook");
}

void stop()
{
}

// build the common filter applied on every searches
private string add_common_filter(string user_filter, object id)
{
  string common_filter = QUERY(ldapfilter);
  string filter;
  if(sizeof(common_filter))
  {
    array from = ({ "$ORGANIZATION", "$LOGIN", "$NAME", "$EMAIL", "$LANGUAGE" });
    array to = ({ "", "", "", "", "" });
    // failsafe, just in case
    if(CSESSION)
      to = ({ CSESSION->organization, CSESSION->login, CSESSION->name, CSESSION->address,
          CSESSION->language });
#if constant (CAMAS.Local.LDAPFilter)
    from += ({ "$MANUAL" });
    to += ({ CAMAS.Local.LDAPFilter.get_filter(id, get_name()) });
#endif
    common_filter = replace(common_filter, from, to);
    filter = "(&"+user_filter+"("+common_filter+"))";
  }
  else
    filter = user_filter;
  CDEBUG("filter="+filter);
  return filter;
}

private string get_rootdn(string dn, object id)
{
   array from = ({ "$ORGANIZATION", "$LOGIN", "$NAME", "$EMAIL", "$LANGUAGE" });
   array to = ({ "", "", "", "", "" });
   if(CSESSION)
     to = ({ CSESSION->organization, CSESSION->login, CSESSION->name, CSESSION->address,
     CSESSION->language });
#if constant (CAMAS.Local.LDAPFilter)
   from += ({ "$MANUAL" });
   to += ({ CAMAS.Local.LDAPFilter.get_root(id, get_name()) });
#endif
   // replace doesn't like when the destination array is not string
   to = map(to, lambda(string|int value) { return (string) value; });
   dn = replace(dn, from, to);
   CDEBUG("dn="+dn);
   return dn;
}

private mapping(int:mixed) ldapattributes2internal (mapping(string:mixed) ldap_attrs)
{
  // first remove the attribute we don't need in Camas
  m_delete(ldap_attrs, "dn");
  m_delete(ldap_attrs, "objectClass");
  array ind_entry = indices(ldap_attrs);
  array ind_inputs2attributes = indices(inputs2attributes);
  array val_inputs2attributes = values(inputs2attributes);
  // convert attributes name to our own internal names
  ind_entry = map(ind_entry, lambda(string s) {
      for(int i = 0; i < sizeof(inputs2attributes); i++)
        if(s == val_inputs2attributes[i])
          return ind_inputs2attributes[i];
  });
  // convert utf8 strings from LDAP to internal Pike string
  array val_entry = ({ });
  foreach(values(ldap_attrs), mixed value)
  {
    if(stringp(value))
    {
      // old versions of Pike 7.2 already decode 
      // the utf8 string from LDAP server to internal
      // Pike string. If we start utf8_to_string a second
      // time on it, we get a backtrace
      // newer don't decode utf8.
      string tmp;
      mixed err = catch {
        tmp = utf8_to_string(value);
      };
      if(err)
        tmp = value;
      val_entry += ({ tmp });
    }
    else if(arrayp(value))
    {
      value = map(value, lambda(string s) {
          string tmp;
          mixed err = catch {
            tmp = utf8_to_string(s);
          };
          if(err)
            tmp = s;
          return tmp;
        });
      val_entry += ({ value });
    }
    else
      val_entry += ({ value });
  }
  return mkmapping(ind_entry, val_entry);
}

private mapping(string:mixed) internalattributes2ldap (mapping(int:mixed) internal_attrs)
{
  array ind_entry = indices(internal_attrs);
  array val_entry = values(internal_attrs);
  array ind_inputs2attributes = indices(inputs2attributes);
  array val_inputs2attributes = values(inputs2attributes);
  // convert attributes name to our own internal names
  ind_entry = map(ind_entry, lambda(int s) {
      for(int i = 0; i < sizeof(inputs2attributes); i++)
      {
        if(s == ind_inputs2attributes[i])
          return val_inputs2attributes[i];
      }
      
    });
  // entries in a LDAP directory must be UTF8 encoded
  val_entry = map(val_entry, lambda(mixed element) {
      if(arrayp(element))
        element = map(element, string_to_utf8);
      if(stringp(element))
        element = string_to_utf8(element);
      return element;
      });
  return mkmapping(ind_entry, val_entry);
}

private array(mapping(int:array(string))) ldap_search(string filter, string dn, object id)
{
  array res = ({ });
  int i = 0;
  object cur_result;
  // the LDAP connection
  object ldap_con;
  // trim whites between "," in the dn
  dn = map(dn / ",", String.trim_whites) * ",";
  ldap_con = connect(ldap_con);
  ldap_con->set_basedn(get_rootdn(dn, id));
  ldap_con->set_scope(2);
  cur_result = ldap_con->search(add_common_filter(filter, id));
  if(!cur_result)
    set_lastprotocol_error((cur_result != 0), ldap_con->error_string(), "search", id);
  else
  {
    mapping|int entry;
    do
    {
      entry = cur_result->fetch();
      // do not return the current object, only sub objects, it's not a bug, it's a feature (tm) :)
      if(entry)
      {
        string currentdn = entry["dn"][0];
        if(currentdn != dn)
          res += ({ ldapattributes2internal(entry) });
        else
          CDEBUG(sprintf("currentdn is the same as root dn '%s', will not display\n",
                dn));
      }
      i++;
      // prevent a DOS in case there are more than 10K entries
      if(i > 10000)
        break;
    }
    while(cur_result->next());
    CDEBUG(sprintf("There is(are) %d entry(ies) matching filter %s at dn %s\n", sizeof(res), filter, dn));
  }
  return res;
}

// The API change: the returning code was changed in Pike 7.3+ to follow his logic better.
private void set_lastprotocol_error(int error_code, string error_string, string function_name, object id)
{
  string msg = sprintf("Problem at %s: %s", function_name, error_string);
  if(__MAJOR__ > 7 && __MINOR__ >= 3)
  {
    if(!error_code)
    {
      CDEBUG(msg);
      id->misc->camas->ldaperror = msg;
    }
  }
  else
    if(error_code)
    {
      CDEBUG(msg);
      id->misc->camas->ldaperror = msg;
    }
}

// some sanity checks mainly because the RFC requires some field to be 
// feeded
private mapping(int:string) checkattrs(mapping(int:string) variables2insert)
{
// remove empty attributes coz openldap doesn't like them
  //foreach(indices(variables2insert), int attribute)
  //  if(sizeof(variables2insert[attribute]) == 0)
  //    m_delete(variables2insert, attribute);
  if(!variables2insert[AD_NAME] || !sizeof(variables2insert[AD_NAME]))
    variables2insert += ([ AD_NAME: "?" ]);
  if(!variables2insert[AD_SURNAME] || !sizeof(variables2insert[AD_SURNAME]))
    variables2insert += ([ AD_SURNAME: "?" ]);
  if(!variables2insert[AD_WORKCOUNTRY] || !sizeof(variables2insert[AD_WORKCOUNTRY]))
    variables2insert += ([ AD_WORKCOUNTRY: "?" ]);
  return variables2insert;
}

private void createroot(object ldap_con, string dn, object id)
{
  dn = get_rootdn(dn, id);
  array(string) arrdn = map(dn / ",", String.trim_whites);

  CDEBUG(sprintf("Creating dn %s if it does not exist...\n", arrdn[1..]* ","));
  ldap_con->set_scope(0);
  ldap_con->set_basedn(arrdn[1..] * ",");
  object result = ldap_con->search("objectClass=*");
  if(!result || !result->fetch())
  {
    // entry does not exist: create it
    CDEBUG("it does not exist, creating...\n");
    // extract and parse value set by the administrator in the CIF 
    // for default attributes to add
    array(mixed) defaultattrs2add = replace(QUERY(attrs2create), "\r", "") / "\n" - ({ "" });
    defaultattrs2add = map(defaultattrs2add, 
         lambda(string value) 
         {
           return value / ":";
         });
    array(string) key = arrdn[1] / "=";
    // entry must always contain the least significant entry of a dn
    mapping attrs = ([ key[0]: ({ key[1] }) ]);
    // for the replacement of CIF 'variables'
    array from = ({ "$ORGANIZATION", "$LOGIN", "$NAME", "$EMAIL", "$LANGUAGE" });
    array to = ({ "", "", "", "", "" });
    // failsafe, just in case
    if(CSESSION)
      to = ({ CSESSION->organization, CSESSION->login, CSESSION->name, CSESSION->address,
          CSESSION->language });
    foreach(defaultattrs2add, array attr)
    {
      attr[1] = replace(String.trim_whites(attr[1]), from, to);
      if(!attrs[attr[0]])
        attrs += ([ attr[0]: ({ attr[1] }) ]);
      else
        attrs[attr[0]] += ({ attr[1] });
    }
    int error_code = ldap_con->add(arrdn[1..] * ",", attrs);
    set_lastprotocol_error(error_code, ldap_con->error_string(), "add", id);
  }
}

/*
 * What we provide here
 */

//
//! method: int version(void)
//!  Give the CAMAS_ADDRESSBOOK api version
//!  supported by the module
//! returns:
//!  the version of the API
//
int version()
{
  return 1;
}

//! method: string get_name(void)
//!  get the name of this addressbook
string get_name()
{
  return QUERY(myname);
}

//! method: int writeable(void)
//!  Is this addressbook writeable ?
int writeable()
{
  return QUERY(allowwrite);
}

//! method: int isdefault(void)
//!  Is this addressbook the default one for users ?
int isdefault()
{
  return QUERY(isdefault);
}

//! method: array(int) get_all_attributes()
//!  Returns every attributes supported by this addressbook
array(int) get_all_attributes()
{
  return indices(inputs2attributes);
}

//! method: int|string get_lastprotocol_error()
//!  Returns the last protocol error if any or 0 otherwise
int|string get_lastprotocol_error(object id)
{
  return id->misc->camas->ldaperror;
}

//
//! method: array(mapping(int:string)) get_all(object id)
//!  Returns every entries on this addressbook, maybe disable by administrator/protocol
//!  Returns an empty string if there are no entries
array(mapping(int:array(string))) get_all(object id)
{
  if(QUERY(allowlist))
  {
    CDEBUG("listing allowed, try to display every entries\n");
    return ldap_search("cn=*", get_rootdn(QUERY(ldapsearchroot), id), id);
  }
  else
  {
    CDEBUG("listing not allowed, nothing will be searched here\n");
    return ({ });
  }
}

//
//! method: array(mapping(int:array(string))) get(mapping(int:string) variables2search, object id, int or_search)
//!   get every entries matching the given values in the variables2search mapping
//! arg: mapping(int:string) variables2search
//!  A mapping indexed by internal attributes value (internals means name will be replaced by cn later) and 
//!  containing for each key the value it must have. If several attributes are provided we simply make an AND
//! arg: object id
//!  Caudium id object
//! arg: void|int or_search
//!  Is this search and'ed or or'ed ?. By default attributes are and'ed
//! arg: void|int wildcardsearch 
//!  For a given attribute, do we search words matching a part of the search filter (wildcard type search) 
//!  or only the words matching ?
//!  the exact search filter
//! returns:
//!  Every entries present on this addressbook that match the given criterias
//!  An empty string if none match
array(mapping(int:array(string))) get(mapping(int:string) variables2search, object id, void|int or_search, 
    void|int wildcardsearch)
{
  mapping(string:string) variables2search_ldap = internalattributes2ldap(variables2search);
  int donotsearch = 1;
  string filter;
  if(or_search)
    filter = "(|";
  else
    filter = "(&";
  foreach(indices(variables2search_ldap), string index)
  {
    if(wildcardsearch)
      filter += "(" + index + "=*" + variables2search_ldap[index] + "*)";
    else
      filter += "(" + index + "=" + variables2search_ldap[index] + ")";
    if(sizeof(variables2search_ldap[index]))
      donotsearch = 0;
  }
  filter += ")";
  CDEBUG("user_filter="+filter);
  if(donotsearch)
    return ({ });
  return ldap_search(filter, get_rootdn(QUERY(ldapsearchroot), id), id);
}

//
//! method: int|void add(mapping(int:string) variables2insert, object id)
//!  Insert variables into this addressbook starting at the DN configured in the configuration interface
//! arg: mapping(string:string) variables2insert
//!  The variables to insert
//! arg: object id
//!  The Caudium id object 
//! returns:
//!  Missing field in case of errors
int|void add(mapping(int:string) variables2insert, object id)
{
  // don't reject write access denied if the user is not logged
  if(CSESSION && !CSESSION->administrator_addressbook && !writeable())
    throw(({ "Write access denied by administrator", backtrace() }));
  string dn = get_rootdn(QUERY(ldapsearchroot), id);
  object ldap_con;
  if(!variables2insert[AD_MAIL])
    return AD_MAIL;
  else
    // trim any leading or trailing white space since we
    // can't search on leading or trailing spaces after
    dn = "mail="+String.trim_whites(variables2insert[AD_MAIL])+","+dn;
  // if the attribute is here but doesn't contain anything,
  // simply ignore it
  if(variables2insert[AD_MAIL] == "")
    return;
  // remove empty attributes coz openldap doesn't like them
  // when adding
  foreach(indices(variables2insert), int attribute)
    if(sizeof(variables2insert[attribute]) == 0)
      m_delete(variables2insert, attribute);
  variables2insert = checkattrs(variables2insert);
  mapping attrs = internalattributes2ldap(variables2insert);
  foreach(indices(attrs), int index)
  {
    attrs[index] = ({ String.trim_whites(attrs[index]) });
  }
  attrs += ([ "objectClass": 
                              ({ 
                                "country",
                                "organizationalPerson",
                                "inetOrgPerson",
                                "person",
                                "top"
                              })
            ]);
  ldap_con = connect(ldap_con);
  if(QUERY(autocreate))
  {
    createroot(ldap_con, dn, id);
  }
  CDEBUG(sprintf("adding attrs to LDAP at dn %s: %O\n", dn, attrs));
  int error_code = ldap_con->add(dn, attrs);
  set_lastprotocol_error(error_code, ldap_con->error_string(), "add", id);
}

//
//! method int|void modify(mapping(int:string) variables2insert, string oldemail, object id)
//!  Modify entry into this addressbook starting at the DN configured in the configuration interface
//! arg: mapping(string:string) variables2modify
//!  The variables to modify
//! arg: string oldemail
//!  The old email of the user
//! arg: object id
//!  The Caudium id object 
//! returns:
//!  Missing field in case of errors
int|void modify(mapping(int:string) variables2modify, string oldemail, object id)
{
  if(CSESSION && !CSESSION->administrator_addressbook && !writeable())
    throw(({ "Write access denied by administrator", backtrace() }));
  string rootdn = get_rootdn(QUERY(ldapsearchroot), id);
  string dn;
  string mail;
  object ldap_con;
  mapping(int:array(string)) oldentry = get(([ AD_MAIL: oldemail ]), id)[0];
  ldap_con = connect(ldap_con);
  if(!variables2modify[AD_MAIL])
    return AD_MAIL;
  else
  {
    mail = String.trim_whites(variables2modify[AD_MAIL]);
    dn = "mail="+mail+","+rootdn;
  }
  variables2modify = checkattrs(variables2modify);
  // if the mail changed we must changed the DN of this entry (:
  if(mail != oldemail)
  {
    string olddn = "mail="+oldemail+","+rootdn;
    ldap_con->modifydn(olddn, "mail="+mail, 1);
  }
  mapping attrs = internalattributes2ldap(variables2modify);
  mapping oldattrs = internalattributes2ldap(oldentry);
  mapping attrs2modify = ([ ]);
  foreach(indices(attrs), string index)
  {
    // if the user sets an non empty value and the attribute doesn't exist
    // in the previous entry we assume he wants to create it
    if(!oldattrs[index] && sizeof(attrs[index]) > 0)
    {
      attrs2modify += ([ index: ({ LDAP_OPERATION_ADD, String.trim_whites(attrs[index]) }) ]);
      continue;
    }
    // if the user sets an empty value while it does exist on the previous
    // entry, we assume he wants to delete it
    if(sizeof(attrs[index]) == 0 && oldattrs[index])
    {
      attrs2modify += ([ index: ({ LDAP_OPERATION_DELETE }) ]);
      continue;
    }
    if(oldattrs[index] && oldattrs[index] != attrs[index])
      attrs2modify += ([ index: ({ LDAP_OPERATION_REPLACE, String.trim_whites(attrs[index]) }) ]);
  }
  int error_code = ldap_con->modify(dn, attrs2modify);
  set_lastprotocol_error(error_code, ldap_con->error_string(), "modify", id);
}

//
//! method: int|void delete(string email, object id)
//!  Delete variables into this addressbook starting at the DN configured in the configuration interface
//! arg: mapping(string:string) variables2insert
//!  The variables to insert
//! arg: object id
//!  The Caudium id object 
//! returns:
//!  Missing field in case of errors
int|void delete(string email, object id)
{
  if(CSESSION && !CSESSION->administrator_addressbook && !writeable())
    throw(({ "Write access denied by administrator", backtrace() }));
  string dn = get_rootdn(QUERY(ldapsearchroot), id);
  object ldap_con;
  if(!email)
    return AD_MAIL;
  else
    dn = "mail="+email+","+dn;
  CDEBUG("Deleting dn="+dn+"\n");
  ldap_con = connect(ldap_con);
  int error_code = ldap_con->delete(dn);
  set_lastprotocol_error(error_code, ldap_con->error_string(), "delete", id);
}

#endif // Protocols.LDAP

/*
 * If you visit a file that doesn't contain these lines at its end, please     
 * cut and paste everything from here to that file.                            
 */                                                                            

/*
 * Local Variables:                                                            
 * c-basic-offset: 2                                                           
 * End:                                                                        
 *                                                                             
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

