/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_html.pike,v 1.43.2.2 2004/03/22 13:22:15 vida Exp $
 */

#include <module.h>
inherit "module";
inherit "caudiumlib";
#include <camas/globals.h>

//
//! module: CAMAS: HTML Module
//!  Module to render HTML mails into CAMAS.<br />
//!  Used when option "show HTML mails" is enabled. It is used for <i>safe
//!  html rendering</i> the HTML mails received before displaying them to end
//!  CAMAS users. You can specify there what HTML code is safe or not.<br />
//!  <b>This module is automatically selected if you select "CAMAS: Main
//!  module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_html.pike,v 1.43.2.2 2004/03/22 13:22:15 vida Exp $
//

constant cvs_version = "$Id: camas_html.pike,v 1.43.2.2 2004/03/22 13:22:15 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: HTML Module";
constant module_doc  = "Module to render HTML mails into CAMAS.<br />"
                       "Used when option \"show HTML mails\" is enabled. It is "
                       "used for <i>safe html rendering</i> the HTML mails "
                       "received before displaying then to end CAMAS users. "
                       "You can specify there what HTML code is safe or not. "
                       "<br /><b>This module is automatically selected if you "
                       "select \"CAMAS: Main module\".</b>";
constant module_unique = 1;
constant thread_safe = 1;		// I think this module should be :)

// --------------------------------------------------------------------------------------

constant html4_containers = ({
                               "a", "abbr", "acronym", "address", "applet", "area", "b", "base", "bdo",
                               "big", "blockquote", "button", "caption", "center", "cite", "code",
                               "colgroup", "dd", "del", "dfn", "dir", "dl", "dt", "em", "fieldset",
                               "form", "font", "frameset", "h1", "h2", "h3", "h4", "h5", "h6",
                               "i", "iframe", "ins", "isindex", "kbd", "label",
                               "legend", "li", "map", "menu", "noframes", "noscript", "object",
                               "ol", "optgroup", "option", "p", "pre", "q", "s", "samp", "script", "select",
                               "small", "strike", "strong", "sub", "sup", "table", "tbody", "td",
                               "textarea", "tfoot", "th", "thead", "tr", "tt", "u", "ul", "var"
                             });

constant html4_tags = ({
                         "area", "base", "basefont", "br", "col", "hr", "frame", "img", "input",
                         "meta", "param"
                       });

constant html4_css = ({
                        "div", "span", "style", /* not HTML 4.01: */ "layer"
                      });

constant tag_references = ({
                             "action", "background", "cite", "classid", "codebase", "data",
                             "href", "longdesc", "rel", "src", "usemap"
                           });

// --------------------------------------------------------------------------------------

mapping (string:mixed) containers = ([ ]), tags = ([ ]);
mapping (string:mixed) containers_inline = ([ ]), tags_inline = ([ ]);
mapping (string:mixed) containers_mailpart = ([ ]), tags_mailpart = ([ ]);

string describe_allowed () {
  return "";
}

string describe_form_allowed (array var, mixed path) {
  array misc = var[VAR_MISC][3];
  string res = "<select name=\"" + path + "\" multiple size=\"" + sizeof (misc) + "\">";
  for (int i = 0; i < sizeof (misc); i++) {
    if (var[VAR_VALUE] && has_value (var[VAR_VALUE], misc[i]))
      res += "<option selected>" + misc[i] + "</option>";
    else
      res += "<option>" + misc[i] + "</option>";
  }
  res += "</select>&nbsp;<input type=\"submit\" value=\"Ok\">";
  return res;
}

array set_from_form_allowed (string val, int type, object o, mixed ... rest) {
  return (val / "\0");
}

void create () {
#ifdef CAMAS_DEBUG
  defvar ("debug", 0, "Debug:General debug", TYPE_FLAG,
          "When on, <b>general</b> debug messages will be logged in Caudium's debug logfile. "
          "This information is very useful to the developers when fixing bugs.");
#endif

  defvar ("showheaders", 1, "Headers", TYPE_FLAG,
          "When on, show headers of the mails when opened in a new window."
          "Note that you will need Camas: Tags module to have this feature "
          "working. The header part will be available in the mailpart screen "
          "so that you can put entities in it.");
  
  defvar ("wraplines", -1, "Wrap lines", TYPE_INT,
          "If positive the message in plain text will be wrapped to the value "
          "you specify here");
  
  defvar ("nojs", "both", "Disallow JavaScript", TYPE_STRING_LIST,
          "Set this if you do not want JavaScript to pass through. <br />"
          "It will remove all 'on...' attributes in HTML elements, and the &lt;script&gt; container.",
          ({ "inline", "mailpart", "both", "no" }));

  defvar ("nocss", "both", "Disallow Cascading Style Sheets", TYPE_STRING_LIST,
          "Set this if you do not want style sheets to pass through. <br />"
          "Removed containers:<ul>"
          "<li>&lt;span&gt;</li>"
          "<li>&lt;style&gt;</li>"
          "<li>&lt;div&gt;</li>"
          "</ul>"
          "Removed tags:<ul>"
          "<li>&lt;link&gt;</li>"
          "</ul>",
          ({ "inline", "mailpart", "both", "no" }));

  defvar ("allowed_containers", ({ }), "Allowed HTML containers", TYPE_CUSTOM,
          "The containers allowed to pass through.",
          // function callbacks for the configuration interface
          ({ describe_allowed, describe_form_allowed, set_from_form_allowed,
             // extra argument
             (html4_containers - ({ "applet", "object", "var" })) }) );

  defvar ("allowed_tags", ({ }), "Allowed HTML tags", TYPE_CUSTOM,
          "The tags allowed to pass through.",
          // function callbacks for the configuration interface
          ({ describe_allowed, describe_form_allowed, set_from_form_allowed,
             // extra argument
             (html4_tags - ({ "meta", "param" })) }) );
}

int do_js (int mailpart) {
  int test = (mailpart) ? (!(< "mailpart", "both" >)[QUERY (nojs)]) : (!(< "inline", "both" >)[QUERY (nojs)]);
  CDEBUG ("do_js => " + test + "\n");
  return test;
}

int do_css (int mailpart) {
  int test = (mailpart) ? (!(< "mailpart", "both" >)[QUERY (nocss)]) : (!(< "inline", "both" >)[QUERY (nocss)]);
  CDEBUG ("do_css => " + test + "\n");
  return test;
}

mapping _containers = ([
                         "title": safe_ignore_container
                       ]);

void start () {
  containers_inline   = _containers + ([
      "body": safe_body,
      "head": safe_ignore_container, 
      "html": safe_ignore_container,
    ]);
  containers_mailpart = _containers;
  
  tags_inline = ([
                 "!doctype": safe_ignore_tag,
                 ]);

  if (QUERY (allowed_containers)) {
    foreach (QUERY (allowed_containers), string t)
      containers += ([ t: safe_container ]);

    foreach (html4_containers - QUERY (allowed_containers), string t)
      containers += ([ t: safe_ignore_container ]);
  }

  if ((QUERY (nojs) == "inline") || (QUERY (nojs) == "both"))
    containers_inline += ([ "script": safe_ignore_container ]);
  else
    containers_inline += ([ "script": safe_container ]);
  
  if ((QUERY (nojs) == "mailpart") || (QUERY (nojs) == "both"))
    containers_mailpart += ([ "script": safe_ignore_container ]);
  else
    containers_mailpart += ([ "script": safe_container ]);
  
  if ((QUERY (nocss) == "inline") || (QUERY (nocss) == "both")) {
    tags_inline += ([ "link": safe_ignore_tag ]);
    foreach (html4_css, string t)
      containers_inline += ([ t: safe_ignore_container ]);
    //containers_inline += ([ t: dump_container ]);
  }
  else {
    tags_inline += ([ "link": dump_tag ]);
    foreach (html4_css, string t)
      //containers_inline += ([ t: dump_container ]);
      containers_inline += ([ t: safe_container ]);
  }
  
  if ((QUERY (nocss) == "mailpart") || (QUERY (nocss) == "both")) {
    tags_mailpart += ([ "link": safe_ignore_tag ]);
    foreach (html4_css, string t)
      containers_mailpart += ([ t: safe_ignore_container ]);
    //containers_mailpart += ([ t: dump_container ]);
  }
  else {
    tags_mailpart += ([ "link": dump_tag ]);
    foreach (html4_css, string t)
      //containers_mailpart += ([ t: dump_container ]);
      containers_mailpart += ([ t: safe_container ]);
  }
  
  if (QUERY (allowed_tags)) {
    foreach (QUERY (allowed_tags), string t)
      tags += ([ t: safe_tag ]);
    
    foreach (html4_tags - QUERY (allowed_tags), string t)
      tags += ([ t: safe_ignore_tag ]);
  }
  
  CDEBUG (sprintf("tags=%O\n", tags));
  CDEBUG (sprintf("tags_inline= %O\n", tags_inline));
  CDEBUG (sprintf("tags_mailpart= %O\n", tags_mailpart));
  CDEBUG (sprintf("containers=%O\n", containers));
  CDEBUG (sprintf("containers_inline= %O\n", containers_inline));
  CDEBUG (sprintf("containers_mailpart= %O\n", containers_mailpart));
}

string status () {
  return "CAMAS HTML Module.";
}

string query_provides () {
  return "camas_html";
}

string add_base (mapping m, string ref, string content_base_abs, string content_base_rel) {
  if (!m[ref] || !strlen (m[ref]))
    return "";

  if (m[ref][0] == '/') {
    CDEBUG ("changing link (abs): " + ref + "= " + m[ref] + " => " + content_base_abs + m[ref] + "\n");
    return content_base_abs + m[ref];
  }
  else {
    CDEBUG ("changing link (rel): " + ref + "= " + m[ref] + " => " + content_base_rel + m[ref] + "\n");
    return content_base_rel + m[ref];
  }
}

mapping disable_js (mapping m) {
  foreach (indices (m), string att)
  if ((sizeof (m[att]) > 2) && (lower_case (att[0..1]) == "on")) {
    CDEBUG ("disabling attribute " + att + ": " + m[att] + "\n");
    m_delete (m, att);
  }

  foreach (indices (m), string att)
  if ((sizeof (m[att]) > 10) && (lower_case (m[att][0..10]) == "javascript:")) {
    CDEBUG ("disabling attribute " + att + ": " + m[att] + "\n");
    m_delete (m, att);
  }

  return m;
}

int is_relative_url (string url) {
  return (sscanf (url, "%*[+-.a-zA-Z0-9]:%*s") != 2);
}

// safe_container and safe_tag used by safe_html to sort out safe tags
string safe_container (string tag, mapping m,
                       string contents, object id, mapping refparts,
                       void|string content_base_abs, void|string content_base_rel, void|int mailpart) {
  foreach (tag_references, string ref) {
    if (m[ref] && (sizeof (m[ref]) > 4) && (lower_case (m[ref][0..3]) == "cid:")) {
      refparts["<" + m[ref][4..] + ">"] = 1;
      m[ref] =CSESSION->nexturl + "/image?mailpart=" + HTTP_ENCODE_URL (m[ref]);
    }
    else {
      if (content_base_abs && m[ref] && is_relative_url (m[ref]))
        m[ref] = add_base (m, ref, content_base_abs, content_base_rel);
    }
  }

  if (!do_js (mailpart))
    m = disable_js (m);

  string q = CAMAS.Tools.make_tag_attributes (m);
  return "{2" + tag + (strlen(q) ? " " + q : "") + "}2" + contents + "{2/" + tag + "}2";
}

string safe_tag (string tag, mapping m, object id, mapping refparts,
                 void|string content_base_abs, void|string content_base_rel, void|int mailpart) {
  foreach (tag_references, string ref) {
    if (m[ref] && (sizeof (m[ref]) > 4) && (lower_case (m[ref][0..3]) == "cid:")) {
      refparts["<" + m[ref][4..] + ">"] = 1;
      m[ref] = CSESSION->nexturl + "/image?mailpart=" + HTTP_ENCODE_URL (m[ref]);
    }

    if (content_base_abs && m[ref] && is_relative_url (m[ref]))
      m[ref] = add_base (m, ref, content_base_abs, content_base_rel);
  }

  if (!do_js (mailpart))
    m = disable_js (m);

  string q = CAMAS.Tools.make_tag_attributes (m);

  return "{2" + tag + (strlen(q) ? " " + q : "") + " /}2";
}

string dump_tag (string tag, mapping m, object id, mapping refparts,
                 void|string content_base_abs, void|string content_base_rel, void|int mailpart) {
  string q = CAMAS.Tools.make_tag_attributes (m);

  return "{2" + tag + (strlen(q) ? " " + q : "") + " /}2";
}

string dump_container (string tag, mapping m,
                       string contents, object id, mapping refparts,
                       void|string content_base_abs, void|string content_base_rel, void|int mailpart) {
  string q = CAMAS.Tools.make_tag_attributes (m);

  return "{2" + tag + (strlen(q) ? " " + q : "") + "}2" + contents + "{2/" + tag + "}2";
}

string safe_ignore_container (string tag, mapping m,
                              string contents, object id, mapping refparts,
                              void|string content_base_abs, void|string content_base_rel, void|int mailpart) {
  CDEBUG ("safe_ignore_container (" + tag + ")\n");
  return contents;
}

string safe_ignore_tag (string tag, mapping m, object id, mapping refparts,
                        void|string content_base_abs, void|string content_base_rel, void|int mailpart) {
  return "";
}

string safe_body (string tag, mapping m,
                  string contents, object id, mapping refparts,
                  void|string content_base_abs, void|string content_base_rel, void|int mailpart) {
  string ret = "";

  foreach (tag_references, string ref) {
    if (m[ref] && (sizeof(m[ref]) > 4) && (m[ref][0..3] == "cid:")) {
      refparts["<" + m[ref][4..] + ">"] = 1;
      m[ref] = CSESSION->nexturl + "/image?mailpart=" + HTTP_ENCODE_URL (m[ref]);
    }
  }

  if (!do_js (mailpart))
    m = disable_js (m);

  if (mailpart) {

    string q = CAMAS.Tools.make_tag_attributes (m);
    ret += "{2body" + (strlen(q) ? " " + q : "") + "}2" + contents + "{2/body}2";
  }
  else {
    ret += "{2table cellpadding=0 border=0 cellspacing=0";

    if (do_css (mailpart)) {
      if (m->bgcolor)
        ret += " bgcolor=\"" + m->bgcolor + "\"";
      if (m->text)
        ret += " color=\"" + m->text + "\"";
      if (m->background && content_base_abs) {
        if (m->background[0] == '/')
          ret += " style=\"background-image: url('" + content_base_abs + m->background + "');\"";
        else
          ret += " style=\"background-image: url('" + content_base_rel + m->background + "');\"";
      }
    }
    ret += "}2";

    ret += "{2tr}2{2td}2" + contents + "{2/td}2{2/tr}2";
    ret += "{2/table}2";
  }

  return ret;
}

// let only allowed tags through and make sure all containers are closed

string safe_html (string in, object id, mapping refparts, void|string content_base, void|int mailpart) {
  string content_base_abs = 0;
  string content_base_rel = 0;

  CDEBUG("safe_html, input=" + in + "\n");

  if (content_base) {
    CDEBUG ("content_base= " + content_base);

    sscanf (content_base, "%*[\"]%s%*[\"]", content_base);
    CDEBUG ("\n => " + content_base);

    //oliv3: an URL linking to a path/filename is not ok in Content-Base, but Netscape sets one...
    string scheme = "", server = "", scheme_specific_part = "";
    sscanf (content_base, "%[+-.a-zA-Z0-9]:%*[/]%s%*[/]%s", scheme, server, scheme_specific_part);

    array split = explode_path (scheme_specific_part);
    if (sscanf (split[-1], "%*s.%*s") == 2) {
      scheme_specific_part = (split[0..sizeof (split)-2] * "/");
      CDEBUG ("\n => " + scheme_specific_part);
    }

    CDEBUG ("\nscheme= " + scheme + "\nserver= " + server + "\nscheme_specific_part= " + scheme_specific_part + "\n");

    content_base_abs = scheme + "://" + server;
    content_base_rel = scheme + "://" + server + "/" + scheme_specific_part;

    CDEBUG ("\n => " + content_base);

    content_base = (content_base / " ") * "";

    CDEBUG ("\n abs= " + content_base_abs + "\n");
    CDEBUG ("rel= " + content_base_rel + "\n");
  }

  in = replace (in, ({ "{", "}" }), ({ "{1", "}1" }));

  mapping(string:mixed) tags_cb = (mailpart) ? tags + tags_mailpart : tags + tags_inline;
  mapping(string:mixed) containers_cb = (mailpart) ? containers + containers_mailpart : containers + containers_inline;

  if (id->misc->_tags)
    foreach (indices (id->misc->_tags), string t)
      tags_cb += ([ t: safe_ignore_tag ]);

  if (id->misc->_containers)
    foreach (indices (id->misc->_containers), string c)
      containers_cb += ([ c: safe_ignore_container ]);

  string parsed = CAMAS.Parse.parse_html (in, tags_cb, containers_cb,
                              id, refparts, content_base_abs, content_base_rel, mailpart);

  parsed = replace (parsed, ({ "{1", "}1", "{2", "}2" }), ({ "{", "}", "<", ">" }));

  CDEBUG("safe_html: parsed input=" + parsed + "\n");
  
  return parsed;
}

string add_base_href (string mail, void|string content_base) {
  CDEBUG ("[HTML] content_base= " + content_base + "\n");

  if (!content_base)
    return mail;

  mapping cont = ([ "head": lambda (object p, mapping m, string c) {
                      string res = "<head><base href=" + content_base + ">";
                      res += c + "</head>";
                      return ({ res });
                    } ]);

  object p = Parser.HTML ();
  p->case_insensitive_tag (1);
  p->ignore_unknown (1);
  mail = p->add_containers (cont)->finish (mail)->read ();
  destruct (p);

  return mail;
}

array(string) low_mailpart(object mail, object id)
{
  mapping refparts = ([ ]);
  string data = mail->getdata () || "";
  string type = mail->type + "/" + mail->subtype;
  mixed err;
   
  err = catch {
    if (type == "text/html") {
      data = add_base_href (data, mail->headers["content-base"]);
      data = safe_html (data, id, refparts, 0, 1);
    }
  };
  if ((type == "text/plain" && QUERY(wraplines) > 0) || err) {
    if(err)
      report_warning("Camas HTML: fail to safe parse mail, reverting to text.\n%s\n", err[0]);
   	data = CAMAS.Tools.wrap_lines(data, QUERY(wraplines));
  }
  return ({ data, type });
}

string|int get_textparts(object lmail, object id, int i)
{
  if(i > 20)
    return "Error: Too deep recursion while trying to extract text parts of the mail";
  if(lmail->type == "text" && lmail->subtype == "plain")
    return low_mailpart(lmail, id)[0] || "";
  if(lmail->type == "text" && lmail->subtype == "html")
    return lmail->getdata() || "";
  foreach(lmail->body_parts, object _lmail)
  {
    string|int part = get_textparts(_lmail, id, ++i);
    if(part && stringp(part))
      return part;
  }
  return 0;
}

array gettext_from_mail(object lmail, object id)
{
  int i = 0;
  string|int mail;
  mail = get_textparts(lmail, id, 0);
  // try to find the text if the message is multipart
  // there are no text/plain part in this message, convert manually
  return ({ CAMAS.Tools.html2text(mail || ""), "text/plain" });
}

mapping mailpart (object id)
{
  mapping refparts = ([ ]);
  object mail = CSESSION->cmail;

  if (!mail)
    return HTTP_LOW_ANSWER (404, "Mailpart not found.");

  int i = 0;
  string mp = id->variables->mailpart;

  // frameset window code is returned only if:
  //  - new window has been requested in the variables 
  //  - showing headers is allowed in the CIF
  //  - Camas tags module is here
  if(id->variables->newwindow && QUERY(showheaders))
  {
    string frameset = "";
    frameset += "<html><head></head>";
    frameset += "<frameset rows=\"100,*\">";
    frameset += "  <frame name=\"headers\" src=\""+ CAMAS.Tools.make_get_url(id, 
        ([ "screen": "mailpart" ])) + "\">";
		frameset += make_tag("frame", ([ "name":"contents", "src":CAMAS.Tools.make_get_url(id, ([ ]), ([ "mailpart":mp ]), id->variables->filename ) ]));
    frameset += "</frameset>";
    frameset += "</html>";
    return HTTP_STRING_ANSWER(frameset,"text/html");
  }
 
  if (sscanf (mp, "cid:%s", mp) == 1) {
    mp = "<" + mp + ">";
    array mails = ({ mail });
    while (sizeof (mails) && (mp != mail->headers["content-id"])) {
      mail = mails[0];
      mails = mails[1..];
      if (mail->body_parts)
        mails += mail->body_parts;
    }
    if (mail->headers["content-id"] == mp) {
      array(string) res;
      mixed err = catch { res = low_mailpart(mail, id); };
      if(err)
      {
        report_warning("Camas HTML: fail to safe parse mail, reverting to text.\n%s\n", describe_backtrace(err));
        res = gettext_from_mail(CSESSION->cmail, id);
      }
      return HTTP_STRING_ANSWER (res[0], res[1]);
    }
    return HTTP_LOW_ANSWER (404, "Mailpart not found.");
  }
  else {
    sscanf (mp, "%d,%s", i, mp);
    while ((i >= 0) && mail && mail->body_parts && (sizeof (mail->body_parts) > i)) {
      mail = mail->body_parts[i];
      i = -1;
      sscanf (mp, "%d,%s", i, mp);
    }
    array(string) res;
    mixed err = catch {
      res = low_mailpart(mail, id);
    };
    if(err) 
    { 
      report_warning("Camas HTML: fail to safe parse mail, reverting to text.\n%s\n", describe_backtrace(err));
      res = gettext_from_mail(CSESSION->cmail, id);
    }
    return HTTP_STRING_ANSWER (res[0], res[1]);
  }
  return HTTP_LOW_ANSWER (404, "Mailpart not found.");
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! When on, <b>general</b> debug messages will be logged in Caudium's debug logfile. This information is very useful to the developers when fixing bugs.
//!  type: TYPE_FLAG
//!  name: Debug:General debug
//
//! defvar: showheaders
//! When on, show headers of the mails when opened in a new window.Note that you will need Camas: Tags module to have this feature working. The header part will be available in the mailpart screen so that you can put entities in it.
//!  type: TYPE_FLAG
//!  name: Headers
//
//! defvar: wraplines
//! If positive the message in plain text will be wrapped to the value you specify here
//!  type: TYPE_INT
//!  name: Wrap lines
//
//! defvar: nojs
//! Set this if you do not want JavaScript to pass through. <br />It will remove all 'on...' attributes in HTML elements, and the &lt;script&gt; container.
//!  type: TYPE_STRING_LIST
//!  name: Disallow JavaScript
//
//! defvar: nocss
//! Set this if you do not want style sheets to pass through. <br />Removed containers:<ul><li>&lt;span&gt;</li><li>&lt;style&gt;</li><li>&lt;div&gt;</li></ul>Removed tags:<ul><li>&lt;link&gt;</li></ul>
//!  type: TYPE_STRING_LIST
//!  name: Disallow Cascading Style Sheets
//
//! defvar: allowed_containers
//! The containers allowed to pass through.
//!  type: TYPE_CUSTOM
//!  name: Allowed HTML containers
//
//! defvar: allowed_tags
//! The tags allowed to pass through.
//!  type: TYPE_CUSTOM
//!  name: Allowed HTML tags
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

