(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Franois Pessaux, projet Cristal, INRIA Rocquencourt     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*            Jun Furuse, projet Cristal, INRIA Rocquencourt           *)
(*                                                                     *)
(*  Copyright 1999,2000,2001,2002,2001,2002                            *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)
(* The image data structure definition. *)

(**************************************************************** Exceptions *)

exception Out_of_image
  (* exception for illegal point access *)
exception Wrong_image_type
  (* exception for illegal internal image type *)
exception Wrong_file_type
  (* exception for unsupported image FILE format *)

(************************************************************* Generic image *)

type t =
   | Index8 of Index8.t
   | Rgb24 of Rgb24.t
   | Index16 of Index16.t
   | Rgba32 of Rgba32.t
   | Cmyk32 of Cmyk32.t
  (* Generic image type *)
  
(******************************************************************** Colors *)

(* Colors: the copies of color.mli *)
type rgb = Color.rgb = { mutable r: int; mutable g: int; mutable b: int }

type rgba = Color.rgba = { color: rgb; mutable alpha: int }

type cmyk = Color.cmyk = 
    { mutable c : int; mutable m : int; mutable y : int; mutable k : int }

type map = Color.map = {
    mutable max: int;  
      (* maximum number allowed in the color map (-1 = unlimited) *)
    mutable map: rgb array
  }

(********************************************************* Image file format *)

(* Image formats *)
type format = 
   | Gif
   | Bmp
   | Jpeg
   | Tiff  
   | Png
   | Xpm
   | Ppm
   | Ps
;;

(************************************************ Image file name extensions *)

(* functions for filename extensions *)
val extension : format -> string
  (* returns the corresponding extension "gif", "bmp" etc. for given format *)
val guess_format : string -> format
  (* returns the image format guessed from the file extension of 
     a given file name *)

(* lower interface *)
val get_extension : string -> string * string
val guess_extension : string -> format

(******************************************** Image file header informations *)

type colormodel = Info.colormodel =
   | Gray | RGB | Index | GrayA | RGBA
;;

(* Infos attached to bitmaps *)
type info = Info.info =
   | Info_DPI of float (* dot per inch *)
   | Info_BigEndian | Info_LittleEndian (* endianness of image file *)
   | Info_ColorModel of colormodel (* color model of image file *)
   | Info_Depth of int (* Image bit depth *)
   | Info_Corrupted (* For corrupted PNG files *)
;;

(* info query *)
val dpi : info list -> float option 

(* Image file header *)
type header = {
     header_width : int;
     header_height : int;
     header_infos : info list
  }
;;

val file_format : string -> format * header
  (* [file_format filename] reads the header of image file [filename]
     and returns its format and some useful information found in the
     header (ex. width, height). [file_format] does not read image 
     contents, but just quickly returns file header information.
    
     [file_format] does not depend on any external libraries *)

(**************************************************** Image file I/O options *)

(* Load options *)
type load_option =
   | Load_Progress of (float -> unit) (* For progress meters *)
   | Load_Resolution of float * float (* Pixel/Inch for rasterization of PS *)
;;

(* Save options *)
type save_option =
   | Save_Quality of int (* Save quality for Jpeg compression *)
   | Save_Progress of (float -> unit) (* For progress meters *)
   | Save_Interlace (* Interlaced Gif *)
;;

(* Option queries *)
val load_progress : load_option list -> (float -> unit) option
val load_resolution : load_option list -> (float * float) option
val save_progress : save_option list -> (float -> unit) option
val save_interlace : save_option list -> bool
val save_quality : save_option list -> int option

(******************************** The type for methods of image file formats *)

type format_methods = {
    check_header: (string -> header);
    load: (string -> load_option list -> t) option;
    save: (string -> save_option list -> t -> unit) option;
  } 

(************************************************ Generic image manupilation *)

val add_methods : format -> format_methods -> unit
  (* If you write new drivers for some image format, use this function
     to register their loading/saving functions into the libaray *)

val load : string -> load_option list -> t
  (* [load filename options] read the header of an image file [filename],
     loads the image by calling corresponding loading method, and 
     returns it. If the file format is not supported by the library,
     a Wrong_file_type exception will be raised. You can specify loading 
     options in [options] such as progressive meter function. *) 

val save : string -> format option -> save_option list -> t -> unit
  (* [save filename formatopt options image] saves [image] into a file 
     [filename]. The image format can be specified by [formatopt]. 
     If [formatopt] is [Some format], then [format] is used. If it is 
     [None], then the image format is guessed from [filename].
     You can specify some saving parameters [options]. Some options are 
     specific to some image formats and do not work with the others. *) 

val size : t -> int * int
  (* Returns size (width and height) of image *)

val destroy : t -> unit
  (* Free the image. If you turn on image swapping (see bitmap.mli), 
     you need to call this function explicitly to tell the library
     that this image is no longer used. *)

val blit : t -> int -> int -> t -> int -> int -> int -> int -> unit
(* [blit src sx sy dst dx dy width height] copies the rectangle
   region of [src] at (sx,sy)-(sx+width-1,sy+height-1) to [dst], at
   (dx,dy)-(dx+width-1,dy+height-1) *)

(*********************************************** Image representation module *)

(* Each image representation module must contain these types and functions, 
   at least *)
module type T = sig
  type t
  type elt

  val create : int -> int -> t  
    (* [create width height] creates an image of the size (width x height) *)

  val destroy : t -> unit
    (* [destroy image] frees the memory and swap files of [image] 
       You have to call this function explicitly if you turn on
       image swapping (See bitmap.mli). *)

  val get : t -> int -> int -> elt
  val set : t -> int -> int -> elt -> unit
    (* [get image x y] and [set image x y v] reads/writes the pixel 
       information at (x,y) of [image]. If (x,y) is out of the image,
       they raise Out_of_image exception. *)

  val unsafe_get : t -> int -> int -> elt
  val unsafe_set : t -> int -> int -> elt -> unit
    (* [unsafe_get] and [unsafe_set] are the same functions as [get] 
       and [set], but they lack the image region checks. So it is fast.
       But you have to use them with being sure that the specified point 
       is in the image. Otherwise the result is unknown, and sometimes 
       a runtime error occurs. *)

  val sub : t -> int -> int -> int -> int -> t
  (* [sub dst x y width height] returns sub-bitmap of [dst],
     at (x,y)-(x+width-1,y+height-1). *)

  val blit : t -> int -> int -> t -> int -> int -> int -> int -> unit
  (* [blit src sx sy dst dx dy width height] copies the rectangle
     region of [src] at (sx,sy)-(sx+width-1,sy+height-1) to [dst], at
     (dx,dy)-(dx+width-1,dy+height-1) *)
end
