/**
  \class CVideoDeviceInput
  
  Every video device has one or more inputs (called channels in the
  Video4Linux API). Each input refers to one of the physical inputs
  on the card/chip, like Tuner, Composite, S-video, etc. This class represents
  such an input.
  
  In addition, each input can have 0 or more tuners attached to it (to keep
  matters simple...). A tuner does the actual frequency setting, and is 
  represented by a \ref CVideoDeviceTuner class.
  
  There are three basic TV systems on this planet: PAL, NTSC and SECAM.
  There are also small variations in these systems, called <i>norms</i>.
  There are norms like PAL-B, -D, -G, -H, -I, M, -N, -MC; NTSC has two
  versions, `plain' NTSC (as used in the USA) and NTSC-Japan. SECAM seems
  to have only one norm. In practice, these norms are all the same when
  viewing a broadcast; as far as I know the main difference lies in the
  assignment of the hidden line numbers to services like TeleText, Closed
  Captioning, etc.

  The Video4Linux API unfortunately has two different places where you set
  norm and system. You set the norm with the channel, and the system within
  the tuner. Within the BTTV driver, these are mixed up: norm = system. 
  The VIDIOCSCHAN call allows you to set NTSC-JAPAN, for example, yet
  VIDIOCSTUNER limits it to PAL, NTSC and SECAM only. For this reason
  the norm selection has been incorporated into this class.
*/

#include <stdio.h>
#include <string.h>

#include "VideoDeviceInput.h"
#include "VideoDevice.h"

/** 
  \brief Constructor
  
  In the constructor the channel information is queried and stored. It
  will also create any Tuner objects that may be needed.
*/

CVideoDeviceInput::CVideoDeviceInput(CVideoDevice *video, int channel)
{
   struct video_channel vchan;
   char buf[33];

   pVideo = video;
   Channel = channel;
   
   Type = Unknown;
   flags = 0;
   Norm = 0;
   CurrentTuner = -1;

   Tuners.setAutoDelete(TRUE);
   // Query data 
   vchan.channel = Channel;
   if (ioctl(pVideo->CamFD, VIDIOCGCHAN, &vchan) == 0) {
     strncpy(buf, vchan.name, 32);
     buf[32] = '\0';
     Name = buf;

     if (vchan.flags & VIDEO_VC_TUNER) {
       Tuners.resize(vchan.tuners);
       for (int i = 0; i < vchan.tuners; i++)
          Tuners.insert(i, new CVideoDeviceTuner(pVideo, i));
     }
     else
       Tuners.resize(0);
     switch(vchan.type) {
       case VIDEO_TYPE_TV:     Type = TV;     break;
       case VIDEO_TYPE_CAMERA: Type = Camera; break;
     }
     flags = vchan.flags;
     Norm = vchan.norm;
   }
   else
     printf("CVIdeoDeviceInput: Warning: no channel info available.\n");
}  

/**
  \brief Return channel number.
*/
int CVideoDeviceInput::GetNumber() const
{
   return Channel;
}

/**
  \brief Return symbolic name for input.
*/
QString CVideoDeviceInput::GetName() const
{
   return Name;
}

/**
  \brief Returns whether this input has an audio channel associated with it.
*/
bool CVideoDeviceInput::HasAudio() const
{
   if (flags & VIDEO_VC_AUDIO)
     return TRUE;
   return FALSE;
}

/**
  \brief Return type for this input.
  
  Returns a value from the \ref InputTypes enum, either TV or Camera
*/  
int CVideoDeviceInput::GetType() const
{
   return Type;
}

/**
  \brief Return number of tuners.

  Most inputs don't have a tuner, or at most 1. Multiple tuners could
  be used for multi-norm cards (each norm having a separate tuner), but
  this hasn't happend sofar.
*/
int CVideoDeviceInput::GetTuners() const
{
   return Tuners.count();
}

/**
*/
CVideoDeviceTuner *CVideoDeviceInput::GetTuner(int number) const
{
   return Tuners.at((uint) number);
}

/**
  \brief Return current tuner number
  \return Tuner number, or -1 if the current tuner is unknown
  
  Since there is no way to query the current selected tuner from the device,
  this function returns -1 until a tuner has been selected.
*/
int CVideoDeviceInput::GetCurrentTuner() const
{
   return CurrentTuner;
}


bool CVideoDeviceInput::SelectTuner(int number)
{
   struct video_tuner v;
   
   v.tuner = number;
   v.mode = Norm;
   if (ioctl(pVideo->CamFD, VIDIOCSTUNER, &v) == 0)
     return TRUE;
   return FALSE;
}


void CVideoDeviceInput::SetNorm(int norm)
{
   if (norm < MAX)
     Norm = norm;
}


/**
  \brief Make this input the current one.
  
*/
bool CVideoDeviceInput::Select()
{
   if (pVideo)
     return pVideo->SelectInput(Channel);
   return FALSE;
}
