/**
  \class CCodecImage

  This class is a 16 bit representation of an image. It can be used to
  do various operations on images, like addition, subtraction.

*/

#include "CodecImage.h"

CCodecImage::CCodecImage()
{
   m_pData = new ImageData;
}

/**
  \brief Copy constructor
*/
CCodecImage::CCodecImage(const CCodecImage &src)
{
   m_pData = src.m_pData;
   m_pData->ref();
}

CCodecImage::CCodecImage(int width, int height)
{
   m_pData = new ImageData;
   Resize(width, height);
}

CCodecImage::~CCodecImage()
{
   if (m_pData && m_pData->deref())
   {
     delete m_pData;
     m_pData = 0;
   }
}

// private

// protected

/**
  \brief Perform sanity check
  \param one The first image
  \param two The second image
  \return true When the dimension match and data pointers are valid

  This function performs a simple sanity check between the two images
  \b one and \b two. It returns true when the dimensions
  match, both data pointers are valid, and not the same.

*/
bool CCodecImage::SanityCheck(const CCodecImage &one, const CCodecImage &two)
{
   if (one.m_pData == 0 || two.m_pData == 0 || one.m_pData == two.m_pData)
     return false;
   if (one.m_pData->pImage == 0 || two.m_pData->pImage == 0)
     return false;
   if (one.m_pData->Width != two.m_pData->Width || one.m_pData->Height != two.m_pData->Height)
     return false;
   return true;
}

void CCodecImage::detach()
{
   if (m_pData != 0 && m_pData->Count != 1)
   {
     *this = copy();
   }
}

/**
  \brief Create deep copy
*/
CCodecImage CCodecImage::copy() const
{
   if (m_pData == 0 || m_pData->pImage == 0)
     return CCodecImage();
   else
   {
     CCodecImage image(m_pData->Width, m_pData->Height);
     if (image.m_pData != 0)
     {
       memcpy(image.m_pData->pImage, m_pData->pImage, m_pData->Pixels * sizeof(int16_t));
     }
     return image;
   }
}


// public

int CCodecImage::Width() const
{
   if (m_pData == 0)
     return 0;
   return m_pData->Width;
}

int CCodecImage::Height() const
{
   if (m_pData == 0)
     return 0;
   return m_pData->Height;
}

void CCodecImage::Resize(int w, int h)
{
   if (m_pData == 0)
     return;
   detach();
   m_pData->Malloc(w, h);
}

void CCodecImage::Load(int width, int height, const unsigned char *src)
{
   int i;
   int16_t *dst;

   if (m_pData == 0 || src == 0)
     return;
   Resize(width, height);
   if (m_pData->pImage == 0)
     return;

   dst = m_pData->pImage;
   for (i = m_pData->Pixels; i > 0; i--)
   {
      *dst = *src - 128;
      src++;
      dst++;
   }
}

void CCodecImage::Store(int width, int height, unsigned char *dst)
{
   int i;
   int16_t *src, val;

   if (m_pData == 0 || m_pData->pImage == 0 || dst == 0)
     return;
   if (m_pData->Width != width || m_pData->Height != height)
     return;

   src = m_pData->pImage;
   for (i = m_pData->Pixels; i > 0; i--)
   {
      val = *src;
      val += 128;
      if (val < 0)
      {
        *dst = 0;
      }
      else if (val > 255)
      {
        *dst = 255;
      }
      else
      {
        *dst = val & 0xff;
      }
      src++;
      dst++;
   }
}


CCodecImage &CCodecImage::operator =(const CCodecImage &src)
{
   src.m_pData->ref();
   if (m_pData->deref())
   {
     delete m_pData;
     m_pData = 0;
   }
   m_pData = src.m_pData;
   return *this;
}

/**
  \brief Subtract image from this image
  \param sub The image to be subtracted
  \return The subtraced CCodecImage

  This function will subtract the pixel values from the \b sub
  image from this image, modifying it.
*/
CCodecImage &CCodecImage::operator -=(const CCodecImage &sub)
{
   int i;
   const int16_t *src;
   int16_t *dst;

   /* Perform sanity checking; dimensions, pointers */
   if (SanityCheck(*this, sub))
   {
     dst = m_pData->pImage;
     src = sub.m_pData->pImage;
     for (i = m_pData->Pixels; i > 0; i--)
     {
        *dst -= *src;
        src++;
        dst++;
     }
   }
   return *this;
}

/**
  \brief Subtract two images from each other
  \param sub The image that will be subtract from this
  \return The subtraced CCodecImage
  \remarks The size of the two images must be the same

  This function will subtract the pixel values from the \b sub
  image from this image; neither of the two images are modified.
  The pixels are subtracted one by one; the result is not clipped.
*/
CCodecImage CCodecImage::operator -(const CCodecImage &sub) const
{
   CCodecImage result;
   int i;
   const int16_t *src1, *src2;
   int16_t *dst = 0;

   if (SanityCheck(*this, sub) && result.m_pData != 0)
   {
     result.Resize(m_pData->Width, m_pData->Height);
     dst = result.m_pData->pImage;
     if (dst != 0)
     {
       src1 = m_pData->pImage;
       src2 = sub.m_pData->pImage;
       for (i = m_pData->Pixels; i > 0; i--)
       {
          *dst = *src1 - *src2;
	  src1++;
	  src2++;
	  dst++;
       }
     }
   }
   return result;
}
