#ifndef FTPCLIENT_HPP
#define FTPCLIENT_HPP

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <qdns.h>
#include <qfile.h>
#include <qhostaddress.h>
#include <qsocket.h>
#include <qsocketnotifier.h>
#include <qstring.h>

#include "FTPListen.h"

class CFTPClient: public QObject
{
   Q_OBJECT
public:
   enum Commands {
     cmdNop,
     cmdLogin,
     cmdLogout,
     cmdSetType,
     cmdChangeDir,
     cmdListDir,
     cmdUpload,
     cmdDownload,
     cmdRename,
     // end of real commands

     cmdPassive,
     cmdDeleteLocalFile,       ///< For destructive uploads
     cmdDestroy,
   };

   enum States {
     stNOC,		///< Not connected
     stDNSBusy,		///< Looking up hostname...
     stConnecting,	///< Busy connecting to server
     stConnected,	///< Server reached; awaiting response from FTP daemon
     stLogin,		///< Sending username
     stAuthenticate,	///< Sending password

     stIdle,		///< Connected OK, but not doing anything

     stSendingPort,	///< Sending PORT command to remote end
     stWaitData,	///< Waiting for data connection
     stTransfer,	///< Transfering data
     stClosingData,	///< Waiting for 226

     stFailed,		///< Failed status; can happen at every command
     stUnknown,		///< Unkown or unexpected result code
   };

   enum TransferMethods {
     tmUnknown,
     tmAscii,
     tmBinary,
  };

private:
   QString m_UserName, m_Password;
   QString m_ServerName;
   int m_ServerPort;

   QSocket *m_pCtrlFD;
   int m_DataFD;
   QSocketNotifier *m_pDataConnectNotifier;
   QSocketNotifier *m_pDataNotifier;
   CFTPListen *m_pListenFD;
   QFile m_LocalFile;
   QHostAddress m_MyAddress;

   char *inputbuffer, *linebuffer;
   QString m_ResponseBuffer; // complete response, including multiline
   char LastChar;
   int LineLen;          // Characters in linebuffer
   int Response;         // FTP response code
   QString RemoteFileName;

   bool m_Passive;	 // Are we in passive mode?
   QHostAddress m_PassiveAddress;
   Q_UINT16 m_PassivePort;

   QString outputbuffer;

   char *m_TransferBuffer;
   int m_TransferBufferSize, m_TransferBufferHead, m_TransferBufferUsed;
   int TotalTransfered;

   TransferMethods m_TransferMethod; // ASCII vs. BINARY
   States CurrentState;
   Commands CurrentCommand;

   bool m_Log;

   void InitBuffers();
   void CloseAllFD();
   void SetupControl();
   void CloseControl();
   bool SetupListen();
   void CloseListen();
   bool SetupLocal(const QString &filename, bool write, bool truncate = true);
   void CloseLocal();
   bool SetupDataActive();
   bool SetupDataPassive();
   void CloseData();
   void HookData();

   void SetState(States new_st, int response = 0);

   void InterpretLine();
   void InterpretResponse();

   void Send();
   void SendUser();
   void SendPass();
   void SendList();
   void SendPort(const QHostAddress &, Q_UINT16 port);
   void SendStore(const QString &filename);

   void StartSending();
   void StartReceiving();

private slots:
   void ControlRead();
   void ControlError(int);
   void ControlResolved();
   void ControlConnected();
   void ControlClosed();

   void ListenConnect();

   void DataConnect(int socket);
   void DataRead();
   void DataWrite();
   void DataClose();

public:
   static const char *CommandStr[];
   static const char *StateStr[];

   CFTPClient(QObject *parent = 0, const char *name = 0);
   ~CFTPClient();

   void SetLogging(bool log);

   int GetCommand() const;
   int GetState() const;
   void SetPassive();
   void SetActive();

   void Connect(const QString &user, const QString &pass, const QString &server, int port = 21);

   void Upload(const QString &local_file, const QString &remote_file = QString::null);
   void SetTypeAscii();
   void SetTypeBinary();
   void ChangeDir(const QString &new_dir);
   void ListDir();
   void Rename(const QString &from, const QString &to);

   void Logout();

   QString GetErrorString(int result) const;

signals:
   /** This signal is emitted whenever the internal state machine changes.
     The result is the numeric FTP result code, if known. There are however a
     few extra 'result' codes defined, for situations that do not apply to
     FTP servers:

     800 Local file not found (upload)
     810 Server name not resolved
     811 Failed to connect to server


    */
   void StateChange(int command, int new_state, int result, const QString &server_msg);

   /// Emitted when login was succesfull
   void LoggedIn();
   /// Emitted when login failed
   void LoginFailed();
   /// Emitted when the control connection was closed (either by us or the remote party)
   void ControlPortClosed();
   void TimeOut();

   void ListDirEntry(const QString &filename);

   /// Emitted every now and then to report on progress. Reports total amount of bytes transfered so far
   void Progress(int offset);
};

#endif
