/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


//
// $Id: AmrDerive.cpp,v 1.25 2002/04/26 22:40:00 car Exp $
//

#include <winstd.H>
#include <new>
#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <unistd.h>

#include "REAL.H"
#include "Box.H"
#include "FArrayBox.H"
#include "ParmParse.H"
#include "ParallelDescriptor.H"
#include "DataServices.H"
#include "Utility.H"
#include "VisMF.H"
#include "Derived.H"

//
// This MUST be defined if don't have pubsetbuf() in I/O Streams Library.
//
#ifdef BL_USE_SETBUF
#define pubsetbuf setbuf
#endif

static
void
PrintUsage (const char* progName)
{
    std::cout << '\n';
    std::cout << "Usage:" << '\n';
    std::cout << progName << '\n';
    std::cout << "    infile  = inputFileName" << '\n';
    std::cout << "    outfile = outputFileName" << '\n';
    std::cout << "    type    = derivedTypeName" << '\n';
    std::cout << "   [-help]" << '\n';
    std::cout << "   [-verbose]" << '\n';
    std::cout << "   [fab.format = FormatType]" << '\n';
    std::cout << "      where FormatType=IEEE32(default), ASCII, 8BIT, or NATIVE" << '\n';
    std::cout << '\n';

    std::cout << "Available types:" << '\n';
    for (int nder = 0; nder < NumDeriveFunc; ++nder)
    {
        std::cout << "  " << DerivedQuants[nder].name << '\n';
    }
    std::cout << '\n';
    exit(1);
}

int
main (int   argc,
      char* argv[])
{
    BoxLib::Initialize(argc,argv);

    if (argc == 1)
        PrintUsage(argv[0]);
    //
    // Make sure to catch new failures.
    //
    //set_new_handler(Utility::OutOfMemory);

    //ParallelDescriptor::StartParallel(&argc, &argv);

    ParmParse pp;

    if (pp.contains("help"))
        PrintUsage(argv[0]);

    FArrayBox::setFormat(FABio::FAB_IEEE_32);
    //
    // Set default info.
    //
    Real gam(1.667);
    FORT_SETGAMMA(&gam);


    Real alp(2.37);
    FORT_SETALPHA(&alp);

    //
    // Scan the arguments.
    //
    std::string iFileDir, iFile, oFile, oFileDir;

    bool verbose = false;
    if (pp.contains("verbose"))
    {
        verbose = true;
        AmrData::SetVerbose(true);
    }
    pp.query("infile", iFile);
    if (iFile.empty())
        BoxLib::Abort("You must specify `infile'");

    pp.query("outfile", oFile);
    if (oFile.empty())
        BoxLib::Abort("You must specify `outfile'");

    int ntype = pp.countname("type");
    if (ntype == 0)
        BoxLib::Abort("You must specify at least one `type'");

    Array<std::string> derives(ntype);

    int i;
    for (i = 0; i < ntype; ++i)
    {
        pp.getkth("type",i,derives[i]);
        if (verbose && ParallelDescriptor::IOProcessor())
            std::cout << "Derived quantities = " << derives[i] << '\n';
    }
    DataServices::SetBatchMode();
    FileType fileType(NEWPLT);
    
    DataServices dataServices(iFile, fileType);

    if (!dataServices.AmrDataOk()) {
        //
        // don't call DataServices::ExitRequest with DataServices::SetBatchMode()
        //
        //DataServices::Dispatch(DataServices::ExitRequest, NULL);
	ParallelDescriptor::EndParallel();
    }

    AmrData& amrData = dataServices.AmrDataRef();

    BL_ASSERT(amrData.NComp() == NSTATECOMPS);

    int finestLevel = amrData.FinestLevel();
    //
    // Make the result MultiFab's.
    //
    Array<MultiFab*> mfout(finestLevel+1);

    FArrayBox srcFab;

    for (int iLevel = 0; iLevel <= finestLevel; ++iLevel)
    {
        const BoxArray& ba = amrData.boxArray(iLevel);
        mfout[iLevel]      = new MultiFab(ba, ntype, 0);
        MultiFab& mfo      = *mfout[iLevel];

        for (int nt = 0; nt < ntype; ++nt)
        {
            int whichDerived  = -1;
            bool foundDerived = false;

            for (int nder = 0; nder < NumDeriveFunc; ++nder)
            {
                if (derives[nt] == DerivedQuants[nder].name)
                {
                    foundDerived = true;
                    whichDerived = nder;
                    break;
                }
            }
            if (!foundDerived)
            {
                std::cerr << "Invalid derived quantity:  " << derives[nt] << std::endl;
                BoxLib::Abort();
            }
            Array<MultiFab*> mfin(amrData.NComp());

            int validComp = -1;

            for (int iComp = 0; iComp < NSTATECOMPS; ++iComp)
            {
                if (DerivedQuants[whichDerived].derMap[iComp] == 1)
                {
                    mfin[iComp] = &amrData.GetGrids(iLevel, iComp);
                    validComp   = iComp;
                }
            }
            BL_ASSERT(validComp >= 0);

            for (MFIter ntmfi(*mfin[validComp]); ntmfi.isValid(); ++ntmfi)
            {
                int ng = ntmfi.index();

                srcFab.resize (ntmfi.validbox(), NSTATECOMPS);

                for (int iComp = 0; iComp < NSTATECOMPS; ++iComp)
                {
                    if (DerivedQuants[whichDerived].derMap[iComp] == 1)
                    {
                        srcFab.copy((*mfin[iComp])[ng], 0, iComp);
                    }
                }
                const Real* srcptr = srcFab.dataPtr();
                const int*  losrc  = srcFab.loVect();
                const int*  hisrc  = srcFab.hiVect();
                FArrayBox& destFab = mfo[ng];
                Real* destptr      = destFab.dataPtr(nt);
                const int* lodest  = destFab.loVect();
                const int* hidest  = destFab.hiVect();

                DerivedQuants[whichDerived].func(srcptr,
                                                 losrc[0],losrc[1],losrc[2],
                                                 hisrc[0],hisrc[1],hisrc[2],
                                                 amrData.NComp(),
                                                 amrData.GridLocLo()[iLevel][ng].dataPtr(),
                                                 amrData.GridLocHi()[iLevel][ng].dataPtr(),
                                                 destptr,
                                                 lodest[0],lodest[1],lodest[2],
                                                 hidest[0],hidest[1],hidest[2],
                                                 lodest,hidest);
            }
        }
    }

    if (ParallelDescriptor::IOProcessor())
        if (!BoxLib::UtilCreateDirectory(oFile,0755))
            BoxLib::CreateDirectoryFailed(oFile);
    //
    // Force other processors to wait till directory is built.
    //
    ParallelDescriptor::Barrier();

    std::string oFileHeader(oFile);
    oFileHeader += "/Header";
  
    VisMF::IO_Buffer io_buffer(VisMF::IO_Buffer_Size);

    std::ofstream os;
  
    os.rdbuf()->pubsetbuf(io_buffer.dataPtr(), io_buffer.size());
  
    if (verbose && ParallelDescriptor::IOProcessor())
        std::cout << "Opening file = " << oFileHeader << '\n';

    os.open(oFileHeader.c_str(), std::ios::out|std::ios::binary);

    if (os.fail())
        BoxLib::FileOpenFailed(oFileHeader);
    //
    // Start writing plotfile.
    //
    os << amrData.PlotFileVersion() << '\n';
    int n_var = ntype;
    os << n_var << '\n';
    for (int n = 0; n < ntype; n++) os << derives[n] << '\n';
    os << BL_SPACEDIM << '\n';
    os << amrData.Time() << '\n';
    os << finestLevel << '\n';
    for (i = 0; i < BL_SPACEDIM; i++) os << amrData.ProbLo()[i] << ' ';
    os << '\n';
    for (i = 0; i < BL_SPACEDIM; i++) os << amrData.ProbHi()[i] << ' ';
    os << '\n';
    for (i = 0; i < finestLevel; i++) os << amrData.RefRatio()[i] << ' ';
    os << '\n';
    for (i = 0; i <= finestLevel; i++) os << amrData.ProbDomain()[i] << ' ';
    os << '\n';
    for (i = 0; i <= finestLevel; i++) os << 0 << ' ';
    os << '\n';
    for (i = 0; i <= finestLevel; i++)
    {
        for (int k = 0; k < BL_SPACEDIM; k++)
            os << amrData.DxLevel()[i][k] << ' ';
        os << '\n';
    }
    os << amrData.CoordSys() << '\n';
    os << "0\n"; // The bndry data width.
    //
    // Write out level by level.
    //
    for (int iLevel = 0; iLevel <= finestLevel; ++iLevel)
    {
        //
        // Write state data.
        //
        int nGrids = amrData.boxArray(iLevel).size();
        char buf[64];
        sprintf(buf, "Level_%d", iLevel);
    
        if (ParallelDescriptor::IOProcessor())
        {
            os << iLevel << ' ' << nGrids << ' ' << amrData.Time() << '\n';
            os << 0 << '\n';
    
            for (i = 0; i < nGrids; ++i)
            {
                for (int n = 0; n < BL_SPACEDIM; n++)
                {
                    os << amrData.GridLocLo()[iLevel][i][n]
                       << ' '
                       << amrData.GridLocHi()[iLevel][i][n]
                       << '\n';
                }
            }
            //
            // Build the directory to hold the MultiFabs at this level.
            //
            std::string Level(oFile);
            Level += '/';
            Level += buf;
    
            if (!BoxLib::UtilCreateDirectory(Level, 0755))
                BoxLib::CreateDirectoryFailed(Level);
        }
        //
        // Force other processors to wait till directory is built.
        //
        ParallelDescriptor::Barrier();
        //
        // Now build the full relative pathname of the MultiFab.
        //
        static const std::string MultiFabBaseName("/MultiFab");
    
        std::string PathName(oFile);
        PathName += '/';
        PathName += buf;
        PathName += MultiFabBaseName;
    
        if (ParallelDescriptor::IOProcessor())
        {
            //
            // The full name relative to the Header file.
            //
            std::string RelativePathName(buf);
            RelativePathName += '/';
            RelativePathName += MultiFabBaseName;
            os << RelativePathName << '\n';
        }
        VisMF::Write(*mfout[iLevel], PathName, VisMF::OneFilePerCPU);
        delete mfout[iLevel];
    }

    os.close();
    //
    // don't call DataServices::ExitRequest with DataServices::SetBatchMode()
    //
    //DataServices::Dispatch(DataServices::ExitRequest, NULL);
    //ParallelDescriptor::EndParallel();
    BoxLib::Finalize();
    return 0;
}
