/***************************************************************************
                          CdboDialogActionFactory.cpp  -  description
                             -------------------
    begin                : Wed Jul 18 2001
    copyright            : (C) 2001 by Alex Podolsky
    email                : alexpod@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <iostream.h>

#include <qobject.h>
#include <qdir.h>
#include <qstringlist.h>

#include <kapp.h>
#include <kconfig.h>
#include <kprocess.h>
#include <kmessagebox.h>

#include "CdboDialogBase.h"
#include "CdboDialogActionFactory.h"
#include "SessionTypeDialog.h"

#include "dialogWidgets/IsoOptPnl.h"
#include "dialogWidgets/CdboOpt.h"

CdboDialogActionFactory::CdboDialogActionFactory( CdboDialogBase* d, const char* name) :
	QObject(0, name){
	//////////
	user = d;
	config = kapp->config();
	DEBUG_COMMANDS = true;
	checkingDrive = false;
	burning = false;
	erasing = false;
	readingData = false;
	readingAudio = false;
	creatingISO = false;
	creatingCdOnTheFly = false;
	fixating = false;
	unfixated = false;
	resetSessionType();
}
CdboDialogActionFactory::~CdboDialogActionFactory(){

}
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////// PUBLIC INTERFACE    /////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::checkDrive(){
	checkingDrive = true;
	initDriveCheckProcess();
	lunchProcess(cdboProc);
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::fixate(){
	fixating = true;
	initFixateProcess();
	lunchProcess(cdboShellProc);
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::erase(){
	if(user->getType() != CdboDialogBase::EraseDialogType){
		emit processDone(false);
		return;
	}
	erasing = true;
	initEraseProcess();
	lunchProcess(cdboProc);
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::burn(QString image, bool audio ){
	burning = true;
	initBurnProcess(image, audio);
	lunchProcess(cdboShellProc);
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::readData(QString in, QString *out){
	readingData = true;
	initReadDataProcess(in,out);
	lunchProcess(cdboProc);
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::createISO(QStringList *flist, QString imageName){

	creatingISO = true;
	isoImageName = imageName;
	this->flist = flist;
	isoFileMappings = 0;

	if(initCreateISOProcess())
		lunchProcess(cdboShellProc);

}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::createISO(QString *mappings, QString imageName){

	creatingISO = true;
	isoImageName = imageName;
	this->flist = 0;
	isoFileMappings = mappings;

	if(initCreateISOProcess())
		lunchProcess(cdboShellProc);

}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::createCdOnTheFly(QStringList *flist){
	creatingCdOnTheFly = true;
	isoImageName = "|";
	this->flist = flist;
	isoFileMappings = 0;

	if(!initCreateISOProcess())
		return;

	initBurnProcess("-", false);
	lunchProcess(cdboShellProc);
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::createCdOnTheFly(QString *mappings){
	creatingCdOnTheFly = true;
	isoImageName = "|";
	this->flist = 0;
	isoFileMappings = mappings;

	if(!initCreateISOProcess())
		return;

	initBurnProcess("-", false);
	lunchProcess(cdboShellProc);
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::convertMp3(QStringList *flist, QString imageName){
	//creatingISO = true;
	initConvertMp3Process(flist,imageName);
	lunchProcess(cdboShellProc);
}
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////// PROCESS INIT    /////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initDriveCheckProcess(){
	cdboProc = new KProcess();
	*cdboProc << "cdrecord";
	///
	*cdboProc  << "-v";
	
	*cdboProc  << "-checkdrive";
	*cdboProc  << "driveropts=help";
	*cdboProc  << "dev=" + user->getScsiDevice();
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initEraseProcess(){
	
	cdboProc = new KProcess();
	user->status("Erasing...");
	*cdboProc << "cdrecord";
	config->setGroup("Default Settings");	
	switch(config->readNumEntry("Level Of Output", CdboOpt::StandartOutput)){
		case CdboOpt::StandartOutput:{
			*cdboProc  << "-v"; break;
		}
		case CdboOpt::CustomOutput:{
			switch(config->readNumEntry("Custom Level Of Output", 1)){
				case 0:	 *cdboProc  << "-s"; break;
				case 1:	 *cdboProc  << "-v"; break;
				case 2:	 *cdboProc  << "-V"; break;
				case 3: *cdboProc  << "-VV"; break;
			}
			if(config->readBoolEntry("Show Debug Info", false))
				*cdboProc  << "-d";
			break;
		}
		default:{
			*cdboProc  << "-s"; break;
		}
	}
	config->setGroup("Erase options");
     	if(config->readBoolEntry("Dummy Erase", false)){
		user->status("Erasing (dummy)...");
		*cdboProc  << "-dummy";
	}
	
     if(config->readBoolEntry("Eject After Erase", true))
		*cdboProc  << "-eject";
	
	if(config->readBoolEntry("Force Blank", true))
		*cdboProc  << "-force";
				
	*cdboProc  << "blank=" + user->getActionType();
	*cdboProc  << "speed=" + user->getSpeed();
	*cdboProc  << "dev=" + user->getScsiDevice();
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initFixateProcess(){
	
	user->status("Fixating...");
	////////////////////////////////////////////////////////////
	
	cdboShellProc = new KShellProcess();	
	
	*cdboShellProc << "cdrecord";
	config->setGroup("Default Settings");	
	switch(config->readNumEntry("Level Of Output", CdboOpt::StandartOutput)){
		case CdboOpt::StandartOutput:{
			*cdboShellProc  << "-v"; break;
		}
		case CdboOpt::CustomOutput:{
			switch(config->readNumEntry("Custom Level Of Output", 1)){
				case 0:	 *cdboShellProc  << "-s"; break;
				case 1:	 *cdboShellProc  << "-v"; break;
				case 2:	 *cdboShellProc  << "-V"; break;
				case 3: *cdboShellProc  << "-VV"; break;
			}
			if(config->readBoolEntry("Show Debug Info", false))
				*cdboShellProc  << "-d";
			break;
		}
		default:{
			*cdboShellProc  << "-s"; break;
		}
	}
	config->setGroup("Recording Options");
	///// eject or not
	if(config->readBoolEntry("Eject Media At End", true))
		*cdboShellProc  << "-eject";
		
	//////// speed
	*cdboShellProc  << "speed="  + user->getSpeed();
	
	//////// dummy
	if(user->getActionType() == "dummy"){
		user->status("Fixating (dummy)...");
		*cdboShellProc  << "-dummy";
	}
	
	//////////// fixate
	*cdboShellProc  << "-fix";
	
	
	if(config->readBoolEntry("Burn Proof", false))
		*cdboShellProc  << "driveropts=burnproof";
	
	
	if(config->readBoolEntry("Use Custom Options", false))
		addCustomBurnOpt();
	
	//////////// device
	*cdboShellProc  << "dev=" + user->getScsiDevice();
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initReadDataProcess(QString in, QString *out){
	user->status("Reading CD-ROM...");
	*out += ".iso";
	cdboProc = new KProcess();	
	*cdboProc << "dd";
	*cdboProc << "if=" + in;
	*cdboProc << "of=" + *out + "";
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initBurnProcess(QString image, bool audio ){
	user->status("Writing...");
	////////////////////////////////////////////////////////////
	
	if(image != "-")
		cdboShellProc = new KShellProcess();	
	
	*cdboShellProc << "cdrecord";
	config->setGroup("Default Settings");	
	switch(config->readNumEntry("Level Of Output", CdboOpt::StandartOutput)){
		case CdboOpt::StandartOutput:{
			*cdboShellProc  << "-v"; break;
		}
		case CdboOpt::CustomOutput:{
			switch(config->readNumEntry("Custom Level Of Output", 1)){
				case 0:	 *cdboShellProc  << "-s"; break;
				case 1:	 *cdboShellProc  << "-v"; break;
				case 2:	 *cdboShellProc  << "-V"; break;
				case 3: *cdboShellProc  << "-VV"; break;
			}
			if(config->readBoolEntry("Show Debug Info", false))
				*cdboShellProc  << "-d";
			break;
		}
		default:{
			*cdboShellProc  << "-s"; break;
		}
	}
	config->setGroup("Recording Options");
	///// eject or not
	if(config->readBoolEntry("Eject Media At End", true))
		*cdboShellProc  << "-eject";
		
	//////// speed
	*cdboShellProc  << "speed="  + user->getSpeed();
	
	//////// dummy
	if(user->getActionType() == "dummy"){
		user->status("Writing (dummy)...");
		*cdboShellProc  << "-dummy";
	}
	
	/////////////////////////////////////////////////////////////////+++++++++++
	config->setGroup("ISO Settings");	
	
	bool ms = config->readEntry("Session Type", "Single") ==  "Multi-Session";
	if(ms){
		if(user->getType() == CdboDialogBase::NewCdDialogType &&
				sessionType != SessionTypeDialog::finalSession &&
				sessionType != SessionTypeDialog::invalidSession){
			*cdboShellProc  << "-multi";
		}
		if(creatingCdOnTheFly)
			*cdboShellProc  << "-waiti";
	}
	/////////////////////////////////////////////////////////////////+++++++++++
	config->setGroup("Recording Options");
	if(user->getActionType().find("append") != -1){
		ms = true;
		if(sessionType == SessionTypeDialog::firstSession || sessionType == SessionTypeDialog::middleSession)
			*cdboShellProc  << "-multi";
		else if(sessionType == SessionTypeDialog::invalidSession){
			int result = KMessageBox::questionYesNo(0,
					"Will this be the final session on this CD?",
        				"Final session?",
        				QString("&Yes"),
        				QString("&No"));
			switch(result) {
    				case KMessageBox::No:{
					sessionType = SessionTypeDialog::middleSession;
					*cdboShellProc  << "-multi";
				}
				case KMessageBox::Yes:{
					sessionType = SessionTypeDialog::finalSession;
				}
			}
		}
		if(user->getActionType() == "dummy append"){
			user->status("Appending (dummy)...");
			*cdboShellProc  << "-dummy";
		}
		else
			user->status("Appending...");
	}
		
	if(config->readBoolEntry("Disk At Once", false) && !ms)
		*cdboShellProc  << "-dao";
	
	if(!config->readBoolEntry("Fixate CD", true)){
		unfixated = true;
		*cdboShellProc  << "-nofix";
	}
	
	if(config->readBoolEntry("Burn Proof", false))
		*cdboShellProc  << "driveropts=burnproof";
	
	QString strBuff = config->readEntry("Media Catalog Number", "");
	if(strBuff != "")
		*cdboShellProc  << "mcn=" + strBuff;
	
	if(config->readBoolEntry("Use Custom Options", false))
		addCustomBurnOpt();
	
	//////////// device
	*cdboShellProc  << "dev=" + user->getScsiDevice();
	
	if(audio)
		*cdboShellProc  << "-audio" ;
	else
		*cdboShellProc  << "-data" ;
	
	//////////// image
	/*
	strBuff = imageAddr.right(imageAddr.length() - imageAddr.findRev("/") - 1);
	//cerr << "Filter: " << filter << endl;
	QDir tempDir(tmpLbl->text(), strBuff);
	
	QStringList flist = tempDir.entryList();
	strBuff = flist.join("' '" + tmpLbl->text());
	strBuff.prepend("'" + tmpLbl->text());
	strBuff.append("'");
	*cdboShellProc << strBuff;*/
	
	if(image == "-")
		*cdboShellProc  << " - ";
	else
		*cdboShellProc  << "\"" + image + "\"";
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::addCustomBurnOpt(){
	////////////////////////////////////////////////////////////
	config->setGroup("Recording Options");
	
	QString strBuff = config->readEntry("FIFO Size", "");
	if(strBuff != "")
		*cdboShellProc  << "fs=" + strBuff;
		
	if(config->readBoolEntry("Use Inf Files", false))
		*cdboShellProc  << "-useinfo";
	
	int iBuff = config->readNumEntry("Default Pregap", 0);
	*cdboShellProc  << "defpregap=" + QString::number(iBuff);

	iBuff = config->readNumEntry("SCSI Time Out", 0);
	*cdboShellProc  << "timeout=" + QString::number(iBuff);

	strBuff = config->readEntry("Driver Name", "");
	if(strBuff != "")
		*cdboShellProc  << "driver=" + strBuff;
	strBuff = config->readEntry("Driver Options", "");
	if(strBuff != "")
		*cdboShellProc  << "driveropts=" + strBuff;
}

//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::readAudio(QString in, QString *out){
	imageAddr = out;
	drName = in;
	readingAudio = true;
	user->blockBtns(true);
	///////////////////////////////////////////////////////
	config->setGroup("Audio Read Options");
	if(config->readBoolEntry("Extract All", true)){
		countAudioTracks();
		return;
	}
	
	fromTrack = config->readEntry("From Track", "1");
	toTrack = config->readEntry("To Track", "1");
	
	if(config->readBoolEntry("Use cdda2wav", true))
		initCdda2wavProcess();
	else
		initCdparanoiaProcess();
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initCdda2wavProcess(){
	readingAudio = true;
	
	cdboShellProc = new KShellProcess();	
	*cdboShellProc << "cdda2wav" << "-g";
	////////////////////////////
    	config->setGroup("Default Settings");	
	switch(config->readNumEntry("Level Of Output", CdboOpt::StandartOutput)){
		case CdboOpt::StandartOutput:{
			break;
		}
		case CdboOpt::CustomOutput:{
			switch(config->readNumEntry("Custom Level Of Output", 1)){
				case 0:	 *cdboShellProc  << "--quiet Q"; break;
				case 1:	 break;
				case 2:	 *cdboShellProc  << "-v 2"; break;
				case 3: *cdboShellProc  << "-v 64"; break;
			}
			if(config->readBoolEntry("Show Debug Info", false))
				*cdboShellProc  << "-V";
			break;
		}
		default:{
			*cdboShellProc  << "--quiet -Q"; break;
		}
	}
	
	///////////////////////////////////////////////////////
	config->setGroup("Audio Read Options");
	
	*cdboShellProc << "-D" << drName;
	
	int iBuff = config->readNumEntry("CD-Rom Speed", 0);
	if(iBuff > 0)
		*cdboShellProc << "-S " + QString::number(iBuff);
		
	if(config->readBoolEntry("Wait For Signal", false))
		*cdboShellProc << "-w";
	
	switch(config->readNumEntry("CD Type", 0)){
		case 0: *cdboShellProc << "-x"; break;
		case 1: *cdboShellProc << "-c 2"; break;
		case 2: *cdboShellProc << "-c 1"; break;
		case 3: *cdboShellProc << "-c s"; break;
	}		
	if(config->readBoolEntry("Echo", false))
		*cdboShellProc << "-e";
	
	*cdboShellProc << "-t " + fromTrack + "+" + toTrack;
	
	
	if(config->readBoolEntry("Split Output", false))
		*cdboShellProc << "-B";

	switch(config->readNumEntry("Output Type", 0)){
		case 0:{
			*cdboShellProc << "-Owav" ;
			*imageAddr += ".wav";
			break;
		}
		case 1:{
			*cdboShellProc << "-Oaiff";
			*imageAddr += ".aiff";
			break;
		}
		case 2:{
			*cdboShellProc << "-Oaifc";
			*imageAddr += ".aifc";
			break;
		}
		case 4:{
			*cdboShellProc << "-Oau";
			*imageAddr += ".au";
			break;
		}
		case 3:{
			*cdboShellProc << "-Ocdr";
			*imageAddr += ".cdr";
			break;
		}
		default:{
			*cdboShellProc << "-Owav" ;
			*imageAddr += ".wav";
			break;
		}
	}
	switch(config->readNumEntry("CD-Rom Indianess", 0)){
		case 1: *cdboShellProc << "-C big"; break;
		case 2: *cdboShellProc << "-C little"; break;
	}
	switch(config->readNumEntry("Output Indianess", 0)){
		case 1: *cdboShellProc << "-E big"; break;
		case 2: *cdboShellProc << "-E little"; break;
	}
	QString i = " \"" + *imageAddr + "\" ";
	*cdboShellProc << i;
	
	if(config->readBoolEntry("Split Output", false))
		imageAddr->insert(imageAddr->findRev("."), "*");
	lunchProcess(cdboShellProc);
}	
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initCdparanoiaProcess(){
	readingAudio = true;
	
	cdboShellProc = new KShellProcess();	
	*cdboShellProc << "cdparanoia";
	
	////////////////////////////
    	config->setGroup("Default Settings");	
	switch(config->readNumEntry("Level Of Output", CdboOpt::StandartOutput)){
		case CdboOpt::StandartOutput:{
			break;
		}
		case CdboOpt::CustomOutput:{
			switch(config->readNumEntry("Custom Level Of Output", 1)){
				case 0:	 *cdboShellProc  << "--quiet"; break;
				case 1:	 break;
				case 2:	 *cdboShellProc  << "-v"; break;
				case 3: *cdboShellProc  << "-v"; break;
			}
			break;
		}
		default:{
			*cdboShellProc  << "--quiet"; break;
		}
	}
	
	///////////////////////////////////////////////////////
	config->setGroup("Audio Read Options");
	*cdboShellProc << "-d" << drName;
	
	int iBuff = config->readNumEntry("CD-Rom Speed", 0);
	if(iBuff > 0)
		*cdboShellProc << "-S " + QString::number(iBuff);
	
	if(config->readBoolEntry("Abort On Skip", false))
	 	*cdboShellProc << "-X";
	
	if(config->readBoolEntry("Split Output", false))
		*cdboShellProc << "-B";
	
	switch(config->readNumEntry("CD-Rom Indianess", 0)){
		case 1: *cdboShellProc << "-C"; break;
		case 2: *cdboShellProc << "-c"; break;
	}

	switch(config->readNumEntry("Output Type", 0)){
		case 0:{
			*cdboShellProc << "-w" ;
			*imageAddr += ".wav";
			break;
		}
		case 1:{
			*cdboShellProc << "-f";
			*imageAddr += ".aiff";
			break;
		}
		case 2:{
			*cdboShellProc << "-a";
			*imageAddr += ".aifc";
			break;
		}
		case 3:{
			if(config->readNumEntry("Output Indianess", 0) == 2){
				*cdboShellProc << "-r";
				*imageAddr += ".cdr";
			}
			else{
				*cdboShellProc << "-R";
				*imageAddr += ".cdr";
			}
			break;
		}
		default:{
			*cdboShellProc << "-w" ;
			*imageAddr += ".wav";
			break;
		}
	}
	
	*cdboShellProc << fromTrack + "-" + toTrack;
	
	QString i = "\""+ *imageAddr + "\"";
	*cdboShellProc << i;
	
	if(config->readBoolEntry("Split Output", false))
		imageAddr->insert(imageAddr->findRev("/") + 1, "*");
	
	lunchProcess(cdboShellProc);
}
//---------------------------------------------------------------------------------------------
bool CdboDialogActionFactory::initCreateISOProcess(){

	cdboShellProc = new KShellProcess();

	*cdboShellProc << "mkisofs";
	////////////////////////////
    	config->setGroup("Default Settings");
	switch(config->readNumEntry("Level Of Output", CdboOpt::StandartOutput)){
		case CdboOpt::StandartOutput:{
			*cdboShellProc  << "-v"; break;
		}
		case CdboOpt::CustomOutput:{
			switch(config->readNumEntry("Custom Level Of Output", 1)){
				case 0:	 *cdboShellProc  << "-quiet"; break;
				case 1:	 *cdboShellProc  << "-v"; break;
				case 2:	 *cdboShellProc  << "-vv"; break;
				case 3: *cdboShellProc  << "-vv"; break;
			}
			break;
		}
		default:{
			*cdboShellProc  << "-quiet"; break;
		}
	}
	*cdboShellProc  << "-gui";

	/////////////////////////////////////////////////////////////////+++++++++++??????????????????????????????????????
	config->setGroup("ISO Settings");
	if(config->readEntry("Session Type", "Single") == "Multi-Session"  &&
		sessionType == SessionTypeDialog::invalidSession){

		SessionTypeDialog *std = new SessionTypeDialog(&sessionType, dynamic_cast<QWidget *> (user), "std", true);
		std->exec();
		delete std;
		switch(sessionType){
			case SessionTypeDialog::firstSession: {
				initStdISOProcess();
				return true;
			}
			case SessionTypeDialog::middleSession:{
				if(config->readBoolEntry("Check old sessions", false))
					checkLastSessions();
				else
					fetchLastSession();
				return false;
			}
			case SessionTypeDialog::finalSession:	{
				if(config->readBoolEntry("Check old sessions", false))
					checkLastSessions();
				else
					fetchLastSession();
				return false;
			}
			default:{
				delete cdboShellProc;
				resetSessionType();
				creatingISO = false;
				creatingCdOnTheFly = false;
				user->output(CdboDialogBase::ErrorMessage,
				 	"Process canceled by user.");
				user->blockBtns(false);
				currentProc = 0;
				user->status("Ready.");
				emit processCancelled();
				return false;
			}
		}
	}
	else
		initStdISOProcess();
	return true;
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initStdISOProcess(){
	//////////////////////////////////////////////
    	config->setGroup("ISO Settings");

	int type = config->readNumEntry("ISO Type", IsoOptPnl::LinWinISO);

	switch(type){
		case IsoOptPnl::LinuxISO:{
			*cdboShellProc << "-r -L -l -allow-multidot -allow-lowercase";
			//cerr << "IsoOptPnl::LinuxISO type\n";
			break;
		}
		case IsoOptPnl::LinWinISO:{
			*cdboShellProc << "-r -J -D";
			//cerr << "IsoOptPnl::LinWinISO type\n";
			break;
		}
		/*case IsoOptPnl::AudioISO:{
			user->blockBtns(false);
			delete cdboShellProc;
			currentProc = 0;
			user->status("Ready.");
			emit failedToLunchProcess();
			return;
		}*/
		default:{
			initCustomISOProcess();
			//cerr << "Custom type\n";
			break;
		}
	}

	if(config->readBoolEntry("Bootable", false))
		addBootOptions();

	if(config->readBoolEntry("Hybrid", false))
		addHybridOptions();

	if(config->readBoolEntry("Multivolumed Set", false)){
		*cdboShellProc << "-volset-size " + config->readEntry("Volume Set Size", "0");
		*cdboShellProc << "-volset-seqno " + config->readEntry("Volume Set Sequence", "0");
	}


	if(config->readBoolEntry("Follow Sym Links", false))
		*cdboShellProc << "-f";
	if(config->readBoolEntry("Generate Tbl", false))
		*cdboShellProc << " -T";
	if(config->readBoolEntry("Omit Version", false))
		*cdboShellProc << "-N";
	if(config->readBoolEntry("Hide RR", false))
		*cdboShellProc << "-hide-rr-moved";

	/////////////// labels
	QString strBuff  = config->readEntry("Application ID");
	if(strBuff != "")
		*cdboShellProc << "-A '" + strBuff + "'";

	strBuff  = config->readEntry("Copyright");
	if(strBuff != "")
		*cdboShellProc << "-copyright '" + strBuff + "'";

	strBuff  = config->readEntry("Volume ID");
	if(strBuff != "")
		*cdboShellProc << "-V '" + strBuff + "'";

	strBuff  = config->readEntry("Volume Set Name");
	if(strBuff != "")
		*cdboShellProc << "-V '" + strBuff + "'";

	strBuff  = config->readEntry("Publisher ID");
	if(strBuff != "")
		*cdboShellProc << "-P '" + strBuff + "'";

	strBuff  = config->readEntry("Preparer ID");
	if(strBuff != "")
		*cdboShellProc << "-p '" + strBuff + "'";


	strBuff  = config->readEntry("Custom ISO Options");
	if(strBuff != "")
		*cdboShellProc << strBuff;
/////////////////////////////////////////////////////////////////////////
	if(isoImageName == "|"){
		if(!flist){
			*cdboShellProc << "-graft-points";
			*cdboShellProc << *isoFileMappings;
		}
		else
			*cdboShellProc << createFileListString(flist);
		*cdboShellProc << " | ";
	}
	else{
		*cdboShellProc << "-o \"" + isoImageName + "\"";
		if(!flist){
			*cdboShellProc << "-graft-points";
			*cdboShellProc << *isoFileMappings;
		}
		else
			*cdboShellProc << createFileListString(flist);
	}

}


//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initCustomISOProcess(){

	if(config->readBoolEntry("Rock Ridge", false))
		*cdboShellProc  << "-r";
	if(config->readBoolEntry("Joliet", false))
		*cdboShellProc  << "-J";
	// filenames
	if(config->readBoolEntry("Untrans File Names", false)){
		*cdboShellProc  << "-U";
	}	
	else if(config->readBoolEntry("Relaxed File Names", false)){
		*cdboShellProc  << "-L -l -allow-multidot -allow-lowercase";
		if(!config->readBoolEntry("Do Not Omit Trailing Period", false))
			*cdboShellProc  << "-d";
	}
	else{
        	if(config->readBoolEntry("Begin Period File", false))
        		*cdboShellProc  << "-L";
        	if(config->readBoolEntry("Long File Names", false))
        		*cdboShellProc  << "-l";
        	if(config->readBoolEntry("Multidot File Names", false))
        		*cdboShellProc  << "-allow-multidot";
        	if(config->readBoolEntry("Lower Case File Names", false))	
        		*cdboShellProc  << "-allow-lowercase";
        	if(!config->readBoolEntry("Do Not Omit Trailing Period", false))
			*cdboShellProc  << "-d";
	}
	
	if(config->readBoolEntry("Deep Dir Relocation", false))
		*cdboShellProc  << "-D";
}
//==========================================
QString CdboDialogActionFactory::createFileListString(QStringList *fileList){
	QStringList flist = *fileList;
	QString fnames = "-graft-points";
	QFileInfo *d;
	for(int i = 0; i < (int)flist.count(); i++){
		
		d = new QFileInfo( flist[i] );
		if ( !d->exists()){
			delete d;
 			continue;
 		}
 		if(d->isDir())
 			fnames += " '/" + d->fileName() + "/'='" +  flist[i] + "/' ";
 		else
			fnames += " '/" + d->fileName() + "'='" +  flist[i] + "' ";
	}
	return fnames;
	/*
	QString namesToRemove;
	QString fnames = " ";
	QDir dir;
	QFileInfo *d;
	
	for(int i = 0; i < (int)flist.count(); i++){
		
		d = new QFileInfo( flist[i] );
		if ( !d->exists()){
			delete d;
 			continue;
 		}
 		if(namesToRemove.find(" -x \"" + flist[i] + "\" ") != -1)
   			namesToRemove.replace(QRegExp(" -x \"" + flist[i]  + "\" "), "");
   		else	
   			//cerr << "' -x \"" + flist[i] + "\" '" << " not present in '" << namesToRemove << "'\n";
 		
 		if(d->isFile() || flist[i] == "/"){
 			if(!fnames.contains(" \"" + flist[i] + "\" "))
 				fnames = fnames + " \"" + flist[i] + "\" ";
 			delete d;
 			continue;
 		}	
		else{
			
          		QString buff = d->fileName();
         		
          		dir = d->dir();
          		bool root = dir.isRoot();
          		dir.setFilter(QDir::All|QDir::Hidden);
          		QStringList list = dir.entryList();	
          		list.remove(".");
          		list.remove("..");
          		list.remove(buff);
           		
          		buff = d->dirPath(true);
          		
          		if(!root)
          			buff = buff + "/";
          		
          		for(int y = 0; y < (int)list.count(); y++){
          			if(namesToRemove.find(" -x \"" +buff + list[y] + "\" ") != -1){
          				//cerr << "' -x \"" + buff + list[y] + " '" << "\" present in '" << namesToRemove << "'\n";
          				//namesToRemove.replace(QRegExp(" -x " + buff + list[y]  + " "), "");
          				break;
          			}
          			else{
          				//cerr << "' -x \"" + buff + list[y] + "\" '" << " not present in '" << namesToRemove << "'\n";
          				namesToRemove = namesToRemove + " -x \"" + buff + list[y] + "\" ";
          			}
          		}
          		if(!fnames.contains(buff))
          			fnames = fnames + " \"" + buff + "\" ";
           	
           	delete d;
           	
   		}
	}
	return namesToRemove + fnames;*/
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::addBootOptions(){
	QString bootCatalog = config->readEntry("Boot Catalog", "");
	QString bootImage = config->readEntry("Boot Image", "");
	if(bootCatalog != "" && bootImage != ""){
		*cdboShellProc  << "-c " + bootCatalog;
		*cdboShellProc  << "-b " + bootImage;		
	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::addHybridOptions(){
	if(config->readBoolEntry("Apple", false))
		*cdboShellProc  << "-apple";
	
	if(config->readBoolEntry("HFS", false) && config->readBoolEntry("Probe", false))
		*cdboShellProc  << "-hfs -probe";
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::initConvertMp3Process(QStringList *, QString imageName){
	
	cdboShellProc = new KShellProcess();	
		
	*cdboShellProc << "mpg123 -s ";
	////////////////////////////
    	config->setGroup("Default Settings");	
	switch(config->readNumEntry("Level Of Output", CdboOpt::StandartOutput)){
		case CdboOpt::StandartOutput:{
			*cdboShellProc  << "-v"; break;
		}
		case CdboOpt::CustomOutput:{
			switch(config->readNumEntry("Custom Level Of Output", 1)){
				case 0:	 *cdboShellProc  << "-quiet"; break;
				case 1:	 break;
				case 2:	 *cdboShellProc  << "-v"; break;
				case 3: *cdboShellProc  << "-v"; break;
			}
			break;
		}
		default:{
			*cdboShellProc  << "-quiet"; break;
		}
	}
	//*cdboShellProc << "mpg123 -s ";
}
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////// PROCESS LUNCH    ////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::lunchProcess(KProcess *proc){
	connect(proc, SIGNAL(processExited(KProcess *)),
		this, SLOT(processDoneSlot(KProcess *)));
	
	connect(proc, SIGNAL(receivedStdout (KProcess *, char *, int)),
		this, SLOT(receivedMessageSlot(KProcess *, char *, int)));
	
	connect(proc, SIGNAL(receivedStderr (KProcess *, char *, int)),
		this, SLOT(receivedErrMessageSlot(KProcess *, char *, int )));
	
	///////////////////////////////////////////////////////
	if(DEBUG_COMMANDS){
		cerr << "\n\n\n";
		QStrList *list = proc->args();
		QString sssss;
		for(sssss=list->first();sssss != 0; sssss=list->next())
			cerr << sssss << " ";
		cerr << "\n\n\n";
	}
	//return;//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	
	//////////////////////////////////////////////////////////////////
		
	if(proc->start(KProcess::NotifyOnExit, KProcess::AllOutput)){
		currentProc = proc;
		user->blockBtns(true);
		if(checkingDrive){
			user->output(CdboDialogBase::StatusOkMessage, "Checking recording device!");
			user->status("Checking device...");
		}
		if(erasing){
			user->output(CdboDialogBase::StatusOkMessage, "Starting erase process!");
			//user->status("Erase in process....");
		}
		if(fixating){
			user->output(CdboDialogBase::StatusOkMessage, "Starting fixate CD process!");
			user->status("Fixating CD....");
		}
		if(burning){
			user->output(CdboDialogBase::StatusOkMessage, "Starting burn process!");
			
		}
		if(readingData){
			user->output(CdboDialogBase::StatusOkMessage, "Starting read process!");
			user->status("Reading data CD....");
		}
		if(readingAudio){
			user->output(CdboDialogBase::StatusOkMessage, "Starting read process!");
			user->status("Reading audio CD....");
		}
		if(creatingISO){
			user->output(CdboDialogBase::StatusOkMessage, "Starting create ISO process!");
			user->status("Creating ISO Image...");
		}
		if(creatingCdOnTheFly){
			user->output(CdboDialogBase::StatusOkMessage, "Starting create CD 'On-the-fly' process!");
			user->status("Creating CD 'On-the-fly'...");
		}		
	}
	else{
		if(checkingDrive){
			checkingDrive = false;
			user->output(CdboDialogBase::ErrorMessage,
				"Failed to check device! Make sure you have enough privileges.");
		}
		if(erasing){
			erasing = false;
			user->output(CdboDialogBase::ErrorMessage,
				"Failed to start erase process! Make sure you have enough privileges.");
		}
		if(fixating){
			fixating = false;
			user->output(CdboDialogBase::ErrorMessage,
				"Failed to start fixate CD process! Make sure you have enough privileges.");
		}
		if(burning){
			burning = false;
			unfixated = false;
			user->output(CdboDialogBase::ErrorMessage,
				"Failed to start burn process! Make sure you have enough privileges.");
		}
		if(readingData){
			readingData = false;
			user->output(CdboDialogBase::ErrorMessage,
				"Failed to start read process! Make sure you have enough privileges.");
		}
		if(readingAudio){
			readingAudio = false;
			user->output(CdboDialogBase::ErrorMessage,
				"Failed to start read process! Make sure you have enough privileges.");
		}
		if(creatingISO){
			creatingISO = false;
			user->output(CdboDialogBase::ErrorMessage,
				"Failed to start create ISO process! Make sure you have enough privileges.");
		}
		if(creatingCdOnTheFly){
			creatingCdOnTheFly = false;
			user->output(CdboDialogBase::ErrorMessage,
				 "Failed to start create CD 'On-the-fly' process! Make sure you have enough privileges.");
		}
		user->blockBtns(false);
		delete proc;
		currentProc = 0;
		resetSessionType();
		user->status("Ready.");
		emit failedToLunchProcess();
	}	
}	
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::processDoneSlot(KProcess *proc){
	if(proc->normalExit() && proc->exitStatus() == 0){
		if(checkingDrive){
			checkingDrive = false;
			user->output(CdboDialogBase::StatusOkMessage, "Driver for the current drive is present and valid!");
		}
		if(erasing){
			erasing = false;
			user->output(CdboDialogBase::StatusOkMessage, "Erase completed successfully!");
			user->status("Ready.");
			user->blockBtns(false);
		}
		if(fixating){
			fixating = false;
			user->output(CdboDialogBase::StatusOkMessage, "CD fixated successfully!");
				
		}
		if(burning){
			burning = false;
			user->output(CdboDialogBase::StatusOkMessage, "Burn completed successfully!");
			if(unfixated){
				unfixated = false;
				user->output(CdboDialogBase::InfoMessage, "This CD was not fxated.");
			}
		}
		if(readingData){
			readingData = false;
			user->output(CdboDialogBase::StatusOkMessage, "Read completed successfully!");
		}
		if(readingAudio){
			readingAudio = false;
			user->output(CdboDialogBase::StatusOkMessage,"Read completed successfully!");
		}
		if(creatingISO){
			creatingISO = false;
			if(!creatingCdOnTheFly)
				user->output(CdboDialogBase::StatusOkMessage,"ISO image created successfully!");
		}
		if(creatingCdOnTheFly){
			creatingCdOnTheFly = false;
			user->output(CdboDialogBase::StatusOkMessage, "Create CD 'On-the-fly' process completed successfully!");
			if(unfixated){
				unfixated = false;
				user->output(CdboDialogBase::InfoMessage, "This CD was not fxated.");
			}
		}
		delete proc;
		currentProc = 0;
		emit processDone(true);
	}
	else{
		if(checkingDrive){
			checkingDrive = false;
			user->output(CdboDialogBase::StatusNotOkMessage, "Not a valid device!");
		}
		if(erasing){
			erasing = false;
			user->output(CdboDialogBase::StatusNotOkMessage, "Erase failed!");
		}
		if(fixating){
			fixating = false;
			user->output(CdboDialogBase::StatusNotOkMessage, "Failed to fixate this CD!");
				
		}
		if(burning){
			burning = false;
			unfixated = false;
			user->output(CdboDialogBase::StatusNotOkMessage, "Burn failed!");
		}
		if(readingData){
			readingData = false;
			user->output(CdboDialogBase::StatusNotOkMessage, "Read of data CD failed");
		}
		if(readingAudio){
			readingAudio = false;
			user->output(CdboDialogBase::StatusNotOkMessage,"Read of audio CD failed!");
		}
		if(creatingISO){
			creatingISO = false;
			user->output(CdboDialogBase::StatusNotOkMessage,"Failed to create ISO image!");
		}
		if(creatingCdOnTheFly){
			creatingCdOnTheFly = false;
			user->output(CdboDialogBase::StatusNotOkMessage, "Failed to create CD 'On-the-fly'!");
		}
		delete proc;
		currentProc = 0;
		resetSessionType();
		user->status("Ready.");
		qApp->processEvents();
		user->blockBtns(false);
		emit processDone(false);
	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::receivedMessageSlot(KProcess *, char *buffer, int buflen){	
	if(buflen > 0){
		
		QString buff(buffer);
		buff = buff.left(buflen);
		
		//cerr << "\n" << buff;
		user->output(CdboDialogBase::InfoMessage, buff);
 	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::receivedErrMessageSlot(KProcess *, char *buffer, int buflen){
	if(buflen > 0){
		QString buff(buffer);
		buff = buff.left(buflen);
		//cerr << "\n" << buff;
		user->output(CdboDialogBase::ProcessMessage, buff);
 	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::killCurrentTask(){
	if(currentProc){
		if(currentProc->isRunning()){
			currentProc->kill();
			delete currentProc;
		}
	}
	checkingDrive = false;
	burning = false;
	erasing = false;
	readingData = false;
	readingAudio = false;
}
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////// REMOVE     /////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::removeTmpImage(QString image, QString dirName){
	if(!image) return;
	
	QString filter = image.left(image.findRev(".")) + "*";
	filter = filter.right(filter.length() - filter.findRev("/") - 1);
	//cerr << "Filter: " << filter << endl;
	QDir tempDir(dirName, filter);
	
	QStringList flist = tempDir.entryList();
		
	for(int i = 0; i < (int)flist.count(); i++){
	//cerr << "Removing: " + tmpLbl->text() + flist[i] << endl;
		if(!QFile::remove(dirName + "/" + flist[i])){
			user->output(CdboDialogBase::ErrorMessage, "Failed to remove " + (dirName +"/" + flist[i]) + "!");
		}
	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::removeOneFile(QString fileName){
	if(!fileName) return;
	
	if(!QFile::remove(fileName))
		user->output(CdboDialogBase::ErrorMessage, "Failed to remove " + fileName + "!");
}
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////// UTILITIES    /////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//---------------------------------------------------------------------------------------------
bool CdboDialogActionFactory::checkTmp(QString tempDir, int minSize){
	minTmpSize = minSize;
	tempDirName = tempDir;
	QDir dir(tempDirName);
	if(!dir.exists()) return false;
	
	user->status("Checking temp size...");
	
	cdboProc = new KProcess();
	
	*cdboProc << "df" << "-m" << tempDir;/////////////////////////////////////////////////////////////////////////////////////

	QApplication::connect(cdboProc, SIGNAL(processExited(KProcess *)),
      			this, SLOT(sizeCheckProcessDoneSlot(KProcess *)));
 	QApplication::connect(cdboProc, SIGNAL(receivedStdout (KProcess *, char *, int)),
		this, SLOT(receivedSizeMessageSlot(KProcess *, char *, int)));
	
	
	if(!cdboProc->start(KProcess::NotifyOnExit, KProcess::AllOutput)){
		delete cdboProc;
		user->output(CdboDialogBase::ErrorMessage,
				"Failed to check temprary directory! Make sure you have enough privileges.");
		user->status("Ready.");
	}
	return true;
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::receivedSizeMessageSlot(KProcess *, char *buffer, int buflen){	
	if(buflen > 0){
		QString buff(buffer);
		buff = buff.left(buflen);
		
		QStringList comOutput(QStringList::split("\n",buff));
		comOutput = QStringList::split(" ", comOutput[1]);
		buff = comOutput[3].simplifyWhiteSpace();
		
		if(buff.toInt() > minTmpSize)
			hasSpase = true;
		else
			hasSpase = false;
		
		config->setGroup("Default Settings");	
		config->writeEntry("tmp Size", buff.toInt() / 1024);
		user->output(CdboDialogBase::InfoMessage,
			"Directory '"  + tempDirName + "' has " +  buff + "MB available.");		
 	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::sizeCheckProcessDoneSlot(KProcess *proc){
	delete proc;
	user->status("Ready.");
	if(!hasSpase)
		emit tmpCheckProcessDone(false);
	else
		emit tmpCheckProcessDone(true);
}
//===================================================
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::countAudioTracks(){

	user->status("Counting audio tracks...");
	user->output(CdboDialogBase::InfoMessage, "Counting audio tracks...");
	cdboProc = new KProcess();
	*cdboProc << "cdda2wav" << "-J" << "-g" << "-D" << drName;

	QApplication::connect(cdboProc, SIGNAL(processExited(KProcess *)),
      			this, SLOT(countAudioProcessDoneSlot(KProcess *)));
 	QApplication::connect(cdboProc, SIGNAL(receivedStdout (KProcess *, char *, int)),
		this, SLOT(receivedCountAudioMessageSlot(KProcess *, char *, int)));
	QApplication::connect(cdboProc, SIGNAL(receivedStderr (KProcess *, char *, int)),
		this, SLOT(receivedCountAudioMessageSlot(KProcess *, char *, int)));
		
	if(!cdboProc->start(KProcess::NotifyOnExit, KProcess::AllOutput)){
		user->output(CdboDialogBase::ErrorMessage, "Failed to count number of tracks! "\
				"Make sure you have enough privileges.");
		delete cdboProc;
		user->blockBtns(false);
		user->status("Ready.");
		emit failedToLunchProcess();
	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::receivedCountAudioMessageSlot(KProcess *, char *buffer, int buflen){	
	//cerr << "Counting audio tracks...\n";
	if(buflen > 0){
		QString buff(buffer);
		//cerr << buff;
		//cerr << "\n";
		if(buff.contains("Permission denied")) {
			user->output(CdboDialogBase::ErrorMessage, "Failed to count number of tracks! "\
				"Make sure you have enough privileges.");
			user->blockBtns(false);
			user->status("Ready");
			emit processDone(false);
			killCurrentTask();
			return;
		}
		if(buff.contains("Inquiry command failed") || buff.contains("error") || buff.contains("load cdrom please")) {
			user->output(CdboDialogBase::ErrorMessage, "Failed to count number of tracks! "\
				"This device does not seem to contain a valid audio CD.");
			user->blockBtns(false);
			user->status("Ready");
			emit processDone(false);
			killCurrentTask();
			return;
		}
		
		buff = buff.left(buflen);
		buff = buff.replace(QRegExp("\n")," ");
		int pos = buff.find("Tracks:");
		if(pos != -1){
			pos += 7;
			buff = buff.right(buff.length() - pos);
			buff = buff.left(buff.find("."));
			
			toTrack = buff.left(buff.find(" "));

			QString totTime = buff.right(buff.length() - buff.find(" "));
			
			user->output(CdboDialogBase::InfoMessage, "There are: " + toTrack + " tracks,  total time: " + totTime);		
			
			fromTrack = "1";
			killCurrentTask();
			
			if(config->readBoolEntry("Use cdda2wav", true))
				initCdda2wavProcess();
			else
				initCdparanoiaProcess();			
		}	
 	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::countAudioProcessDoneSlot(KProcess *proc){
	delete proc;
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::fetchLastSession(){

	user->status("Fetching session info...");
	//if(cdboProc != 0)
	//	delete cdboProc;
	cdboProc = new KShellProcess();
	
	*cdboProc << "cdrecord";
	*cdboProc << "-msinfo";
	*cdboProc <<  "dev=" + user->getScsiDevice();
	
	if(DEBUG_COMMANDS){
		cerr << "\n\n\n";
		QStrList *list = cdboProc->args();
		QString sssss;
		for(sssss=list->first();sssss != 0; sssss=list->next())
			cerr << sssss << " ";
		cerr << "\n\n\n";
	}
	
	
	QApplication::connect(cdboProc, SIGNAL(processExited(KProcess *)),
      			this, SLOT(fetchLastSessionDoneSlot(KProcess *)));
 	QApplication::connect(cdboProc, SIGNAL(receivedStdout (KProcess *, char *, int)),
		this, SLOT(receivedFetchLastSessionMessageSlot(KProcess *, char *, int)));
	QApplication::connect(cdboProc, SIGNAL(receivedStderr (KProcess *, char *, int)),
		this, SLOT(receivedFetchLastSessionMessageSlot(KProcess *, char *, int)));
		
	if(!cdboProc->start(KProcess::NotifyOnExit, KProcess::AllOutput)){
		user->output(CdboDialogBase::ErrorMessage, "Failed to start 'fetch last session info' process! "\
				"Make sure you have enough privileges.");
		delete cdboProc;
		if(cdboShellProc != 0)
			delete cdboShellProc;
			
		user->blockBtns(false);
		user->status("Ready.");
		emit failedToLunchProcess();
	}
	else
		user->output(CdboDialogBase::InfoMessage, "Fetching last session info, please wait.");
}

//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::receivedFetchLastSessionMessageSlot(KProcess *proc, char *buffer, int buflen){
	if(buflen > 0){

		QString buff(buffer);
		buff = buff.left(buflen);
		//cerr << "^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;
		//cerr << "*buffer = " << buff << endl;
		//cerr << "^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;

		if(buff.contains("cdrecord: Drive needs to reload the media to return to proper status.")){
			user->output(CdboDialogBase::ProcessMessage, "Drive needs to reload the media to return to proper status.");
			return;
		}
		if(buff.contains("Permission denied")) {
			proc->kill();
			user->output(CdboDialogBase::ErrorMessage, "Failed to fetch last session info! "\
				"Make sure you have enough privileges.");
			user->blockBtns(false);
			user->status("Ready");
			emit failedToLunchProcess();
			resetSessionType();
			creatingISO = false;
			creatingCdOnTheFly = false;
			delete cdboShellProc;
			return;
		}
		if(buff.contains("Usage:")) {
			proc->kill();
			user->output(CdboDialogBase::ErrorMessage, "Ooops! "\
				"I did not understand the combination of options you've given me :-/");
			user->output(CdboDialogBase::ErrorMessage,
				"Please submit this as a bug to the email located in about dialog.");
			user->blockBtns(false);
			user->status("Ready");
			emit failedToLunchProcess();
			resetSessionType();
			creatingISO = false;
			creatingCdOnTheFly = false;
			delete cdboShellProc;
			return;
		}

		if(buff.contains("error")) {
			//cerr << "---------------------------------------------------------------------------------" << endl;
			//cerr << "buff = " << buff << endl;
			proc->kill();
			user->output(CdboDialogBase::ErrorMessage, "This CD does not seem to contain a valid session!");
			user->output(CdboDialogBase::ProcessMessage, buff);
			user->output(CdboDialogBase::ErrorMessage, "Failed to fetch last session info.");
			user->blockBtns(false);
			user->status("Ready");
			emit failedToLunchProcess();
			resetSessionType();
			creatingISO = false;
			creatingCdOnTheFly = false;
			delete cdboShellProc;
			return;
		}

		
		buff = buff.replace(QRegExp("\n"), "");

		//cerr << "============= " << buff << "===========" << endl;


		*cdboShellProc << "-C " + buff;
		*cdboShellProc << "-M " + user->getScsiDevice();

		config->setGroup("ISO Settings");
		if(config->readBoolEntry("Check old sessions names", false))
			*cdboShellProc << "-check-oldnames ";
     	if(config->readBoolEntry("Force Rock Ridge", false))
			*cdboShellProc << "-force-rr -no-rr";


		if(creatingISO == true){
			initStdISOProcess();
			lunchProcess(cdboShellProc);
		}
		else{
			initStdISOProcess();
			initBurnProcess("-", false);
			lunchProcess(cdboShellProc);
		}
		//isoImageName = "";
 	}
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::fetchLastSessionDoneSlot(KProcess *proc){
	delete proc;
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::checkLastSessions(){
	gotAnswer = false;
	user->status("Checking previous sessions...");
	user->blockBtns(true);
	cdboProc = new KShellProcess();
	*cdboProc << "mkisofs";
	*cdboProc << "-check-session " + user->getScsiDevice();
	
	
	if(DEBUG_COMMANDS){
		cerr << "\n\n\n";
		QStrList *list = cdboProc->args();
		QString sssss;
		for(sssss=list->first();sssss != 0; sssss=list->next())
			cerr << sssss << " ";
		cerr << "\n\n\n";
	}
	
	
	QApplication::connect(cdboProc, SIGNAL(processExited(KProcess *)),
      			this, SLOT(fetchLastSessionDoneSlot(KProcess *)));
 	QApplication::connect(cdboProc, SIGNAL(receivedStdout (KProcess *, char *, int)),
		this, SLOT(receivedCheckLastSessionsMessageSlot(KProcess *, char *, int)));
	QApplication::connect(cdboProc, SIGNAL(receivedStderr (KProcess *, char *, int)),
		this, SLOT(receivedCheckLastSessionsMessageSlot(KProcess *, char *, int)));
		
	if(!cdboProc->start(KProcess::NotifyOnExit, KProcess::AllOutput)){
		user->output(CdboDialogBase::ErrorMessage, "Failed to start 'check previous sessions' process! "\
				"Make sure you have enough privileges.");
		delete cdboProc;
		if(cdboShellProc != 0)
			delete cdboShellProc;
			
		user->blockBtns(false);
		user->status("Ready.");
		emit failedToLunchProcess();
	}
	else
		user->output(CdboDialogBase::InfoMessage, "Checking previous sessions, please wait.");
}
//---------------------------------------------------------------------------------------------
void CdboDialogActionFactory::receivedCheckLastSessionsMessageSlot(KProcess *, char *buffer, int buflen){
	if(gotAnswer)
		return;
	if(buflen > 0){

		QString buff(buffer);
		//cerr << "^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;
		//cerr << "*buffer = " << buff << endl;
		//cerr << "^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^" << endl;


		if(buff.contains("Permission denied")) {
			gotAnswer = true;
			user->output(CdboDialogBase::ErrorMessage, "Failed to fetch last session info! "\
				"Make sure you have enough privileges.");
			user->blockBtns(false);
			user->status("Ready");
			emit failedToLunchProcess();
			resetSessionType();
			creatingISO = false;
			creatingCdOnTheFly = false;
			delete cdboShellProc;
			return;
		}
		if(buff.contains("Usage:")) {
			gotAnswer = true;
			user->output(CdboDialogBase::ErrorMessage, "Ooops! "\
				"I did not understand the combination of options you've given me :-/");
			user->output(CdboDialogBase::ErrorMessage,
				"Please submit this as a bug to the email located in about dialog.");
			user->blockBtns(false);
			user->status("Ready");
			emit failedToLunchProcess();
			resetSessionType();
			creatingISO = false;
			creatingCdOnTheFly = false;
			delete cdboShellProc;
			return;
		}
		if(buff.contains("error")) {
			gotAnswer = true;
			user->output(CdboDialogBase::ErrorMessage, "This CD does not seem to contain a valid session.");
			user->blockBtns(false);
			user->status("Ready");
			emit failedToLunchProcess();
			resetSessionType();
			creatingISO = false;
			creatingCdOnTheFly = false;
			delete cdboShellProc;
			return;
		}
		if(buff.contains("NO Rock Ridge present")) {
			gotAnswer = true;
			user->output(CdboDialogBase::StatusNotOkMessage,
						"Neither Rock Ridge nor TRANS.TBL name translations were found on previous sessions!");
			int result = KMessageBox::warningYesNo(0,
					"Neither Rock Ridge nor TRANS.TBL name translations were found on previous sessions!\n"\
					"Whould you like to continue anyway?",
        				"No Rock Ridge present",
        				QString("&Yes"),
        				QString("&Cancel"));
        				
			switch(result){
				case KMessageBox::Yes:{
					fetchLastSession();
					return;
				}
       			case KMessageBox::No:{
       				user->output(CdboDialogBase::ErrorMessage, "Process canceled by user.");
					user->blockBtns(false);
					user->status("Ready");
					emit failedToLunchProcess();
					resetSessionType();
					creatingISO = false;
					creatingCdOnTheFly = false;
					delete cdboShellProc;
					return;
				}
			}	
		}
		else{// if(buff.contains("Rock Ridge signatures found")) {
		
			gotAnswer = true;
			user->output(CdboDialogBase::StatusOkMessage, "Previous sessions seem to be valid.");
			qApp->processEvents();
			fetchLastSession();
		}
 	}
}