/*
cdcd - Command Driven CD player
Copyright (C) 1998-99 Tony Arcieri
Copyright (C) 2001 Fabrice BAUZAC

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>

#include "cmd_edit.h"
#include "str.h"
#include "global.h"
#include "interface.h"

Execfunc cmd_edit_title, cmd_edit_artist, cmd_edit_genre, cmd_edit_track,
  cmd_edit_quit, cmd_edit_help, cmd_edit_cdcd, cmd_edit_submit;

struct Command edit_cmds[] = {
  ALIAS("?", "help"),
  { "help", 0, "You're dumb.", cmd_edit_help, 0, 0 },
  { "..", "", "Go back to the cdcd prompt.", cmd_edit_cdcd, 0, 0 },
  { "quit", "", "Quit cdcd.", cmd_edit_quit, 0, 0 },
  ALIAS("exit", "quit"),
  { "name", "[NAME]", "Get or set the name of the current CD.",
    cmd_edit_title, 0, 1 },
  { "artist", "[ARTIST]", "Get or set the artist of the current CD.",
    cmd_edit_artist, 0, 1 },
  {
    "submit", "[email address]",
    "Submit the CDDB entry for the current CD to the CDDB.  "
    "If you don't specify your e-mail address, "
    "you will be prompted to enter it.", cmd_edit_submit, NULL, 1
  },
  { "genre", "[GENRE]", "Get or set the genre of the current CD.",
    cmd_edit_genre, 0, 1 },
  { "track", "TRACKNUMBER [TRACKNAME]",
    "Get or set the name of a particular track.",
    cmd_edit_track, 0, 1 },
  { 0, 0, 0, 0, 0, 0 }
};

int
cmd_edit_submit (char **argv)
{
  struct disc_info disc;

  if (!argv[1] || argv[2])
    {
      puts("wrong number of arguments");
      return 0;
    }
  else if (cdcd_cd_stat (cd_desc, &disc) < 0)
    {
      puts ("Error: couldn't stat CD");
      return 0;
    }
	   
  cddb_submit (cd_desc, argv[1]);
  return 0;
}

int
cmd_edit_track(char **argv)
{
  struct disc_data data;
  struct disc_info disc;
  int track;
  char *name = NULL;
  int track_prov = 0, name_prov = 0;

  lookup_now (cd_desc, &data);

  if (cdcd_cd_stat (cd_desc, &disc) < 0)
    {
      puts ("Couldn't stat CD.\n");
      return 0;
    }

  for(++argv; *argv; ++argv) {
    if(!track_prov && !read_int(*argv, &track)) {
      track--;
      track_prov = 1;
    } else if(!name_prov) {
      name = *argv;
      name_prov = 1;
    } else {
      puts("Error during argument parsing");
      return 0;
    }
  }
  if(!track_prov) {
    puts("Not enough parameters");
    return 0;
  }
  if(track < 0 || track > disc.disc_total_tracks-1) {
    printf("Invalid track %d\n", track+1);
    return 0;
  }
#define TN data.data_track[track].track_name
  if(!name_prov) {
    printf("Track %d is %s.\n", track+1, TN);
    return 0;
  }
  strncpy(TN, name, 127);
  TN[127] = 0;
#undef TN
  cddb_write_disc_data(cd_desc, data);
  return 0;
}

char *
str_genres()
{
  char *p;
  size_t s;
  int len=0;
  int i;
  p = (char*) xmalloc(s=1);
  *p = 0;
  for(i=1; i<=11; ++i) {
    if(i>1) {
      char *sep = i==11? ", and ": ", ";
      int prfx = strlen(sep);
      if(len + prfx + 1 > s)
	p = (char*) xrealloc(p, s=2*s+prfx+1);
      strcpy(p+len, sep);
      len += prfx;
    }
    {
      int g_len;
      g_len = strlen(cddb_genre(i));
      if(len + g_len + 1 > s)
	p = (char*) xrealloc(p, s=2*s+g_len+1);
      strcpy(p+len, cddb_genre(i));
      len += g_len;
    }
  }
  return p;
}

int
cmd_edit_genre(char **argv)
{
  int i;
  struct disc_data data;
  lookup_now(cd_desc, &data);
  if(!argv[1]) {
    printf("Genre: %s\n", cddb_genre(data.data_genre));
    return 0;
  }
  if(argv[2]) {
    puts("Too many arguments");
    return 0;
  }
  for(i=1; i<=11; ++i)
    if(!strcasecmp(argv[1], cddb_genre(i))) {
      cddb_erase_entry(data); /* Could anyone explain me what this
                                 function does? */
      data.data_genre = i;
      cddb_write_disc_data(cd_desc, data);
      return 0;
    }
  {
    char *p;
    p = str_genres();
    pprintf(get_width()-1,
	    "%s: no such genre.  Valid genres are: %s.", argv[1], p);
    putchar('\n');
    free(p);
  }
  return 0;
}

#define CMDEDIT(symb, str)                           \
int                                                  \
cmd_edit_ ## symb(char **argv)                       \
{                                                    \
  struct disc_data data;                             \
  lookup_now(cd_desc, &data);                        \
                                                     \
  if(!argv[1]) {                                     \
    printf(str ": %s\n", data.data_ ## symb);        \
    return 0;                                        \
  }                                                  \
  if(argv[2]) {                                      \
    puts("Too many arguments");                      \
    return 0;                                        \
  }                                                  \
  strncpy(data.data_ ## symb, argv[1], 127);         \
  data.data_ ## symb[127] = 0;                       \
                                                     \
  cddb_write_disc_data(cd_desc, data);               \
  return 0;                                          \
}

CMDEDIT(artist, "Artist");
CMDEDIT(title, "Name");

#undef CMDEDIT

int
cmd_edit_help(char **argv)
{
  cmdhelp(edit_cmds, argv, 0);
  return 0;
}

int
cmd_edit_cdcd(char **argv)
{
  NOARG;
  return XRET_BACK;
}

int
cmd_edit_quit(char **argv)
{
  NOARG;
  return XRET_QUIT;
}

char *
edit_command_matcher(const char *text, int state)
{
  return command_matcher(edit_cmds, text, state);
}

int
cmd_edit_mainloop()
{
  return cmd_mainloop(edit_command_matcher, "cdcd/edit> ",
		      edit_cmds);
}

void
init_cmd_edit()
{
  sort_commands(edit_cmds);
}
