#
# Copyright 2000, 2001, 2003, 2004 Tivano Software GmbH
#
# This software is distributed under the terms of the GNU General
# Public License. See the file COPYING for additional information.
#
# This file contains helper functions for the cdrw-taper.
#
# Thanks to Tim Writer <tim@starnix.com> for implementing support for
# "include" directives in amanda.conf.

package Amanda::Taper::MediaInfo;

# Create a new Amanda::Taper::MediaInfo object containing information about
# the media in the "current" slot.
# Parameters:
#  $taper - a reference to the current Amanda::Taper object
sub new {
    my $class = shift;
    my $taper = shift;

    my $self = {};
    bless $self, $class;

    my ($slot, $dev) = $taper->loadSlot("current");
    if ($dev !~ /^(.*?):/) {
	return "Cannot extract cdrecord device specification from $dev!";
    }
    my $mountDev = $1; # for growisofs
    $dev = $';

    my $info = 0;
    if (defined($taper->{MEDIAINFO})) {
	$info = `$taper->{MEDIAINFO} $mountDev 2>&1`;
	if ($info =~ /Mounted Media:\s+\S+ (DVD[-+]RW?).*Track Size:\s+(\d+)\*2KB/si) {
	    $self->{type} = $1;
	    $self->{erasable} = ($1 eq "DVD+RW" || $1 eq "DVD-RW");
	    $self->{capacity} = $2 * 2; # in kilobytes!
	    $self->{need_format} = ($self->{erasable} && $info =~ /Disc status:\s+blank/si);
	    return $self;
	}
    }

    my $atip = `$taper->{CDRECORD} -atip dev=$dev 2>&1`;
    if ($atip =~ /ATIP info from disk:(.*ATIP start of lead out:\s+(\d+) .*)/si) {
	$atip = $1;
	$self->{capacity} = $2 * 2; # in kilobytes!
	$self->{erasable} = ($atip =~ /Is erasable/);
	$self->{type} = ($self->{erasable} ? "CDRW" : "CDR");
    } else {
	my $message = "";
	if (defined($taper->{MEDIAINFO})) {
	    $message = "dvd+rw-mediainfo returned '$info', ";
	}
	$message .= "cdrecord -atip returned '$atip'";
	$message =~ s/[\r\n]+/_/g;
	$taper->log_add("ERROR", "Can't read media info, using defaults: $message");
	$self->{type} = "Unknown";
	$self->{erasable} = 0;
	$self->{capacity} = $Amanda::Taper::MEDIABLOCKS * $Amanda::Taper::BLOCKSIZE / 1024; # in kilobytes!
	$self->{need_format} = 0;
    }

    return $self;
}

# Returns the media type,
# one of "CDR", "CDRW", "DVD+R", "DVD+RW", "DVD-R", "DVD-RW", "Unknown"
sub getType {
my $self = shift;

    return $self->{type};
}

# Returns the media capacity in kilobytes
sub getCapacity {
my $self = shift;

    return $self->{capacity};
}

# Returns true if the media can be erased + overwritten
sub isErasable {
my $self = shift;

    return $self->{erasable};
}

# Returns true if the media must be formatted before use
sub needFormat {
my $self = shift;

    return $self->{need_format};
}

package Amanda::Taper;

use IO::File;
use Fcntl ':flock';

##
## Configurable settings. Change as needed to adapt to your system.
##

# Amanda config directory
$CONFIG_ROOT = "/etc/amanda";

# Directory where intermediate directories will be created
$DUMP_DIR = "/var/tmp/amanda-dumps";

# Uncomment if intermedia dirs should be deleted after burning
$DELETE_DIRS = 1;

# Uncomment if you want to allow writing on CD-R media (non-erasable)
$WRITE_NON_ERASABLE = 0;

# Regexp for CD-RW devices
$CD_DEVICE_RE = "/dev/s(cd|g)\\d+:?.*";

# Original taper binary
$TAPER_ORIG = "/usr/lib/amanda/taper.orig";

##
## No user editable settings below here
##

# Blocksize of output media in bytes (2k for CD-RW)
$BLOCKSIZE = 2048;

# Blocks per media (CDRW: 650 MB = 650 * 1024k / 2k = 332800)
$MEDIABLOCKS = 332800;
# Blocks per media (DVD[-+]RW: 4700 MB = 4700 * 1000 * 1000 / 2k = 2294921)
# $MEDIABLOCKS = 2294921;

# "Export" config variables
our ($CONFIG_ROOT, $DUMP_DIR, $DELETE_DIRS, $WRITE_NON_ERASABLE, $CD_DEVICE_RE,
     $TAPER_ORIG, $BLOCKSIZE);

$VERSION = "0.4";

# Create a new Amanda::Taper object and configure it
# Parameters:
#  $cfg - the name of the amanda configuration to use (a dir below $CONFIG_ROOT)
#  $logfn - a reference to a logging function that will be called back with
#           a loglevel and a message
sub new {
    my $class = shift;
    my $cfg = shift;
    my $logfn = shift;

    my $self = {};
    bless $self, $class;
    $self->{logfn} = $logfn;
    $self->{VERSION} = $VERSION;

    if (!exists($ENV{PATH}) || $ENV{PATH} eq "") {
	$ENV{PATH} = "/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/bin:/usr/local/sbin";
    }
    # Fool growisofs...
    delete $ENV{SUDO_COMMAND};

    if ($self->getPaths()) {
        $self->readAmandaConfig($cfg);
        return $self;
    }

    return 0;
}

# Check which required / optional commands are on our $PATH
sub getPaths {
    my $self = shift;

    # Required: binaries for mount, umount, sendmail, mkisofs and cdrecord
    #$MOUNT    = "mount";
    #$UMOUNT   = "umount";
    #$CDRECORD = "cdrecord";
    #$MKISOFS  = "mkisofs";
    #$SENDMAIL = "/usr/lib/sendmail";
    $self->{MOUNT} = $self->getPath("mount", 1);
    $self->{UMOUNT} = $self->getPath("umount", 1);
    $self->{MKISOFS} = $self->getPath("mkisofs", 1);
    $self->{CDRECORD} = $self->getPath("cdrecord", 1);
    $self->{SENDMAIL} = $self->getPath("sendmail", 1);

    # Optional: growisofs, dvd+rw-format and dvd+rw-mediainfo
    $self->{GROWISOFS} = $self->getPath("growisofs", 0);
    $self->{MEDIAINFO} = $self->getPath("dvd+rw-mediainfo", 0);
    $self->{DVDRWFORMAT} = $self->getPath("dvd+rw-format", 0);

    return $self->{MOUNT} && $self->{UMOUNT} && $self->{MKISOFS}
	   && $self->{CDRECORD} && $self->{SENDMAIL};
}

# Try to find the path of the executable given as arg #1. If it can't be found
# and arg #2 is true, log this.
sub getPath {
    my $self = shift;
    my $cmd = shift;
    my $logit = shift;

    $cmd =~ s/\"/\\\"/g;
    my $path = `which "$cmd" 2>/dev/null`;
    $path =~ s/[\r\n]+//g;
    if ($? || $path eq "") {
	if ($logit) {
#	    $self->log_add("FATAL",
#		 	   "Can't find system command '$cmd' on search path!");
	    print STDERR "FATAL: Can't find system command '$cmd' on search path ('".$ENV{PATH}."')!\n";
	}
	return 0;
    }

    return $path;
}

# Read Amanda's config file and chdir to the config directory
sub readAmandaConfig {
    my $self = shift;
    my $cfg = shift;
    $self->{config} = $cfg;

    $self->{AMANDA_CONF} = {};

    # Set some defaults for Amanda configuration variables
    $self->{AMANDA_CONF}->{runtapes} = 1;
    $self->{AMANDA_CONF}->{tapelist} = "tapelist";

    my @fh = ();
    my $fh = new IO::File("$CONFIG_ROOT/$cfg/amanda.conf")
	or die "taper: Cannot open $CONFIG_ROOT/$cfg/amanda.conf";
    while (1) {
	my $line = <$fh>;
	if (!$line) {
	    $fh->close();
	    last unless @fh;
	    $fh = pop @fh;
	    next;
	}
	next if $line =~ /^\s*\#/;
	if ($line =~ /^includefile\s+(\"([^\"]*)\"|\S+)/i) {
	    my $value = $1;
	    $value =~ s/(^"|"$)//g;
	    $value =~ s/[\\\"]/\\$&/g;
	    push @fh, $fh;
	    $fh = new IO::File $value
	        or die "taper: Cannot open $value";
	} elsif ($line =~ /^(tapedev|rawtapedev|tapelist|tapecycle|labelstr|logdir|runtapes|tapetype|tpchanger)\s+(\"([^\"]*)\"|\S+)/i) {
	    my $var = $1;
	    my $value = $2;
	    $value =~ s/(^"|"$)//g;
	    $self->{AMANDA_CONF}->{$var} = $value;
	} elsif ($line =~ /^(mailto)\s+\"([^\"]*)\"/i) {
	    my $var = $1;
	    my $value = $2;
	    $value =~ s/(^"|"$)//g;
	    my @mailto = split(/ +/, $value);
	    $self->{AMANDA_CONF}->{mailto} = \@mailto;
#	} elsif ($line =~ /^define tapetype\s+\"([^\"]*)\"/i) {
	}
    }
 
    # Make logdir path absolute if it isn't already
    if ($self->{AMANDA_CONF}->{logdir} !~ /^\//) {
	$self->{AMANDA_CONF}->{logdir} = "$CONFIG_ROOT/$cfg/"
					 .$self->{AMANDA_CONF}->{logdir};
    }

    # Make tapelist filename absolute if it isn't already
    if ($self->{AMANDA_CONF}->{tapelist} !~ /^\//) {
	$self->{AMANDA_CONF}->{tapelist} = "$CONFIG_ROOT/$cfg/"
					   .$self->{AMANDA_CONF}->{tapelist};
    }

    # CD to the config directory
    chdir "$CONFIG_ROOT/$cfg"
	or die "taper: cannot chdir to $CONFIG_ROOT/$cfg: $!";
 
    # read the "tape"-list (better: media list :-) The media list is
    # sorted in reverse datestamp order (newest entry first).
    my @medialist = ();
    open(ML, "<".$self->{AMANDA_CONF}->{tapelist})
	or die "taper: Cannot open "
	       .$self->{AMANDA_CONF}->{tapelist}.": $!";
    while (<ML>) {
	s/[\r\n]//g;
	push @medialist, $_;
    }
    close ML;
    @medialist = sort by_datestamp @medialist;
    $self->{medialist} = \@medialist;

    # If a tape changer is specified without a complete path, we need to
    # append our own directory to $PATH. Then, we ask it for info.
    if (defined($self->{AMANDA_CONF}->{tpchanger})) {
	# THIS IS A HACK
	$ENV{"PATH"} .= ":".$INC[$#INC];
	my $changerinfo = `$self->{AMANDA_CONF}->{tpchanger} -info`;
	if (($? & 0xff) || ($? >> 8)) {
	    die "taper: tape changer failed with exit status $?";
	}
	if ($changerinfo !~ /^(.+)\s+(\d+)\s+([01])\s+$/) {
	    die "taper: can't parse "
		.$self->{AMANDA_CONF}->{tpchanger}
		." -info output: $changerinfo";
	} else {
	    $self->{changer_slots} = $2;
	    $self->{changer_gravity} = $3;
	    $self->{changer_media} = {};
	}
    }

    $self->{nonErasable} = [];
    $self->{usedMedia} = {};
    my @availableMedia = $self->check_disk();
    $self->{availableMedia} = \@availableMedia;
    my @usableMedia = $self->findUsableMedia();
    $self->{usableMedia} = \@usableMedia;
}

# This returns a list of the labels of all disks currently in the available
# drives. If the tape changer cannot go backwards, this steps through the
# media until it finds a usable one.
sub check_disk {
    my $self = shift;
    my @labels = ();
    $self->{mediaInfos} = {};

    my ($slot, $dev) = $self->loadSlot("current");
    if (!defined($slot)) { return @labels; }
    my $label = $self->getLabel($dev);
    my $mediaInfo = new Amanda::Taper::MediaInfo($self);
    if ($label) {
	push @labels, $label;
	$self->{changer_media}->{$label} = $slot;
    } elsif ($WRITE_NON_ERASABLE && ! $mediaInfo->isErasable()) {
	push @{$self->{nonErasable}}, "current";
    }
    $self->{mediaInfos}->{$slot} = $mediaInfo;

    if (defined($self->{AMANDA_CONF}->{tpchanger})) {
	if (!$self->{changer_gravity}) {
	    # Cycle through all slots and take note of the media labels
	    for (my $i = 1; $i < $self->{changer_slots}; $i++) {
		($slot, $dev) = $self->loadSlot("next");
		if (!defined($slot)) { $#labels = -1; return @labels; }
		$label = $self->getLabel($dev);
		$mediaInfo = new Amanda::Taper::MediaInfo($self);
		if ($label) {
		    $self->{changer_media}->{$label} = $slot;
		    push @labels, $label;
	        } elsif ($WRITE_NON_ERASABLE && ! $mediaInfo->isErasable()) {
		    push @{$self->{nonErasable}}, "slot";
		}
		$self->{mediaInfos}->{$slot} = $mediaInfo;
	    }
	} else { # Gravity changer - can't go backwards
	    # Cycle through slots until we find a media label that is_usable
	    my $i = 0;
	    while ($label && !$self->is_usable($label)
		   && $i++ < $self->{changer_slots}) {
		($slot, $dev) = $self->loadSlot("next");
		if (!defined($slot)) { $#labels = -1; return @labels; }
		$label = $self->getLabel($dev);
		$mediaInfo = new Amanda::Taper::MediaInfo($self);
		if ($label) {
		    $self->{changer_media}->{$label} = $slot;
		    push @labels, $label;
	        } elsif ($WRITE_NON_ERASABLE && ! $mediaInfo->isErasable()) {
		    push @{$self->{nonErasable}}, $slot;
		}
		$self->{mediaInfos}->{$slot} = $mediaInfo;
	    }
	}
    }
    return @labels;
}

# Executes a "tpchanger -slot " command and parses the response.
# Returns (slotname, device) on success, or (undef, undef) on error
sub loadSlot {
    my $self = shift;
    my $slot = shift;
    my $dev;

    if (!defined($self->{AMANDA_CONF}->{tpchanger}) && $slot eq "current") {
	return ($slot, $self->{AMANDA_CONF}->{rawtapedev});
    }

    $res = `$self->{AMANDA_CONF}->{tpchanger} -slot $slot`;
    if (($? & 0xff) || ($? >> 8)) {
	$self->log_add("ERROR", "changer failed to load slot $slot with code $? and response $res");
    } else {
	if ($res =~ /^(\S+)\s+(\S+)\s*$/) {
	    $slot = $1;
	    $dev = $2;
	} else {
	    $self->log_add("ERROR", "failed to parse changer response $res");
	    $slot = undef;
	}
    }

    return ($slot, $dev);
}

# Load the media with the given label on some writer device. Returns the
# device name on success, false otherwise.
# If the specified tape changer is a gravity changer, the media in the
# "current" slot *must* be the one to load.
sub loadMedia {
    my $self = shift;
    my $label = shift;
    my $dev = 0;
 
    if (defined($self->{AMANDA_CONF}->{tpchanger})) {
	my $slot;
	if (!$changer_gravity) {
	    if (!exists($self->{changer_media}->{$label})) {
		return 0;
	    }
	    $slot = $self->{changer_media}->{$label};
	} else {
	    $slot = "current";
	}
	($slot, $dev) = $self->loadSlot($slot);
    } else { # no changer
	$dev = $self->{AMANDA_CONF}->{rawtapedev};
    }

    # Now we know the device. Check if the media has the correct label.
    if ($dev && $self->getLabel($dev) eq $label) {
	return $dev;
    } else {
	$self->log_add("ERROR", "label on media in $dev is not $label");
    }

    return 0;
}

# Find an intermediate storage directory that can take a file of the given
# size. Returns undef if nothing can be found.
sub findMediaDir {
    my $self = shift;
    my $wantedBlocks = shift;
    my $best = 0;

    foreach my $usedMedia (keys %{$self->{usedMedia}}) {
	if ($self->{usedMedia}->{$usedMedia}->{freeBlocks} >= $wantedBlocks &&
		(!$best || $self->{usedMedia}->{$best}->{freeBlocks} > $self->{usedMedia}->{$usedMedia}->{freeBlocks})) {
	    $best = $usedMedia;
	}
    }
    if ($best) { return $best; }

    # Nothing found. Any more disks available?
    if ($#{$self->{usableMedia}} < 0) { return undef; }
    my $next = shift @{$self->{usableMedia}};
    if (-d "$DUMP_DIR/$next") {
	system("rm -r $DUMP_DIR/$next/*");
    } else {
	mkdir("$DUMP_DIR/$next", 0700);
    }

    # Write amanda label file
    if (!open(LABEL, ">$DUMP_DIR/$next/AMANDA_LABEL")) {
	return undef;
    }
    print LABEL "$next\n";
    close LABEL;

    $self->{usedMedia}->{$next} = {"freeBlocks" => $MEDIABLOCKS};
    # Try to be better than that: if $next is in one of our changer slots,
    # we have the corresponding media size:
    if (exists($self->{changer_media}->{$next})) {
	$self->{usedMedia}->{$next}->{freeBlocks} =
		$self->{mediaInfos}->{$self->{changer_media}->{$next}}->getCapacity() / ($BLOCKSIZE/1024);
    } elsif ($#{$self->{nonErasable}} >= 0) {
	# Not quite correct if more than one nonErasable media is present
	$self->{usedMedia}->{$next}->{freeBlocks} =
		$self->{mediaInfos}->{$self->{nonErasable}->[0]}->getCapacity() / ($BLOCKSIZE/1024);
    }

    return $next;
}

# Copy a file from the holding disk into the dump disk.  Since we're
# writing to a medium with a file system, we don't really need the 32k
# header that amanda puts in front of each dump. Instead, we write the
# relevant information from that header to a file called
# $basename.info and the dump itself to a file called
# $basename.(dump|tar)[.gz] (depending on dump method and
# compression). $basename is the name of the file in the holding disk.
sub dump_file {
    my $self = shift;
    my ($input_name, $targetDir) = @_;

    my ($basename) = ($input_name =~ /.*\/(.*)/);
    if (!open(IN, "$input_name")) {
	 return "cannot read $input_name: $!";
    }

    # Read the 32k amanda header.
    my ($buffer, $info);
    $error = "cannot read amanda header from $input_name";
    if (read(IN, $info, 32*1024) == 32*1024) {
	# Remove trailing 0-bytes
	$info =~ s/\0*$//s;
	my ($compext, $dumpext) =
	    ($info =~ /AMANDA: FILE \d+ [-\w.]+ \S+ lev \d+ comp (\S+) program (\S+)/);
	if ($compext && $dumpext) {
	    # Don't append a compression extension if the dump is uncompressed
	    $compext = "" if $compext eq "N";
	    # Use '.tar' as dump extension if the dump program is "tar" or "gtar"
	    # Otherwise, use the program's base name.
	    $dumpext =~ s%.*/%%;
	    $dumpext = "tar" if $dumpext =~ /^g?tar$/;
	    $dumpext = ".$dumpext";

	    # Write the information to $basename.info
	    $error = "cannot write info file $targetDir/$basename.info";
	    if (open(OUT, ">$targetDir/$basename.info")) {
		print OUT "$info\n";
		close OUT;
	    }

	    # Now dump the actual file
	    # TODO: concatenate multiple chunks of the file
	    $error = "cannot write dump file $targetDir/$basename$dumpext$compext";
	    if (open(OUT, ">$targetDir/$basename$dumpext$compext")) {
		$error = 0;
		while (read(IN, $buffer, 32*1024) && !$error) {
		    if (!(print OUT $buffer)) {
			$error = "error while writing $targetDir/$basename$dumpext$compext";
		    }
		}
		close OUT;
	    }
	}
    }
    close IN;
    return $error ? "$error: $!" : 0;
}

# Returns the size of the output file in BLOCKs - on CDRW, blocks are 2k bytes
sub fileSize {
    my $self = shift;
    my $filename = shift;
    return int(((-s $filename) + $BLOCKSIZE - 1) / $BLOCKSIZE);
}

# Compare two media list entries for sorting in reverse datestamp
# (newest entry first) order
sub by_datestamp {
  my ($ds_a) = ($a =~ /(\d+)/);
  my ($ds_b) = ($b =~ /(\d+)/);
  return $ds_b <=> $ds_a;
}

# Check if the disk with a given label may be overwritten
sub is_usable {
    my $self = shift;
  my ($check_label) = @_;
  my $count = 0;
  my ($datestamp, $label, $reuse_flag);
  foreach my $entry (@{$self->{medialist}}) {
    ($datestamp, $label, $reuse_flag) = split /\s+/, $entry;
    $count++ if $reuse_flag eq "reuse";
    if ($label eq $check_label) {
	# Tape is in the tapelist -> may be reused if $reuse allows it and
	# the tape is old enough, i.e. $count >= $tapecycle or $datestamp == 0
        if ($reuse_flag ne "reuse") {
	    return 0;
	}
	return $datestamp == 0 || $count >= $self->{AMANDA_CONF}->{tapecycle};
    }
  }
  # Not in the tapelist? -> New tape, may be used
  return 1;
}

# This generates a list of media labels to be used in this run
sub findUsableMedia {
    my $self = shift;
    my @media = ();
    my %available = ();
    my $label;
    my %tmp;

    foreach $label (@{$self->{availableMedia}}) {
	$tmp{$label} = $available{$label} = $self->is_usable($label);
    }

    # Find those available media that are not in the medialist
    # (i. e. new media), and use these
    foreach $label (@{$self->{medialist}}) {
	($datestamp, $lbl, $reuse_flag) = split /\s+/, $label;
	delete $tmp{$lbl};
    }
    foreach $label (keys %tmp) {
	if ($#media < $self->{AMANDA_CONF}->{runtapes} - 1 && $tmp{$label}) {
	    push @media, $label;
	    delete $available{$label};
	}
    }

    # Append available to @media in order of appearance in reverse medialist
    # (i. e. oldest first)
    for (my $i = $#{$self->{medialist}}; $i >= 0 && $#media < $self->{AMANDA_CONF}->{runtapes} - 1; $i--) {
	($datestamp, $label, $reuse_flag) = split /\s+/, $self->{medialist}->[$i];
	if ($available{$label}) { push @media, $label; }
    }

    # Append remaining usable labels to @media in order of appearance in
    # reverse medialist (i. e. oldest first)
    for (my $i = $#{$self->{medialist}}; $i >= 0 && $#media < $self->{AMANDA_CONF}->{runtapes} - 1; $i--) {
	($datestamp, $label, $reuse_flag) = split /\s+/, $self->{medialist}->[$i];
	if (!exists($available{$label}) && $self->is_usable($label)) {
	    push @media, $label;
	}
    }

    return @media;
}

# Actually burn an intermediate dir to the media with the given label.
# If $label is 0 we'll try to write on non-erasable media in some device.
# Returns false on success, otherwise an error message.
# For gravity changers, the media with the given label must be in the
# "current" slot. After burning, the changer will advance to the next slot.
sub burnDir {
    my $self = shift;
    my ($dir, $label) = @_;
    my $dev;
    my $blank = " blank=fast";

    if ($label) {
        $dev = $self->loadMedia($label);
    } elsif ($#{$self->{nonErasable}} >= 0) {
	my $slot = shift @{$self->{nonErasable}};
	($slot, $dev) = $self->loadSlot($slot);
	$blank = "";
    } else {
	return "No non-erasable media left!";
    }

    if (!$dev) {
	return "Cannot load media with label $label!";
    }

    if ($dev !~ /^(.*?):/) {
	return "Cannot extract cdrecord device specification from $dev!";
    }
    my $mountDev = $1; # for growisofs
    $dev = $';

    my $command;
    my $mediaInfo = new Amanda::Taper::MediaInfo($self);
    if ($mediaInfo->getType() =~ /^CD/) {
	$command = "$self->{MKISOFS} -J -R -pad -quiet $dir | ".
			"$self->{CDRECORD} dev=$dev -data$blank -";
    } else {
	$command = "$self->{GROWISOFS} -use-the-force-luke -Z $mountDev -J -R -pad -quiet $dir";
    }

    # Include the full mkisofs/cdrecord/growisofs output in error messages,
    # ignore the (quite verbose:-) output if there were no errors
    my $result = `($command) 2>&1`;
    my $status = $? >> 8;
    
    if (defined($self->{AMANDA_CONF}->{tpchanger}) && $self->{changer_gravity}) {
	$self->loadSlot("next");
    }

    return $status ? "'$command' finished with exit status $status\n\nLog:\n$result" : 0;
}

# Append an entry to the amanda logfile. If the logfile doesn't exist,
# log to STDERR
sub log_add {
    my $self = shift;
  my ($type, $message) = @_;
 
    if (! -r "$self->{AMANDA_CONF}->{logdir}/log") {
	print STDERR "$type taper $message\n";
	return;
    }

  # Open and lock the log file
  open(LOG, ">>$self->{AMANDA_CONF}->{logdir}/log")
    or die "taper: cannot open log file $self->{AMANDA_CONF}->{logdir}/log: $!";
  flock(LOG, LOCK_EX)
    or die "taper: cannot lock the log file $self->{AMANDA_CONF}->{logdir}/log: $!";
  seek(LOG, 0, 2);

  # write the entry
  print LOG "$type taper $message\n";
    
  # unlock and close the file
  flock(LOG, LOCK_UN)
    or die "taper: cannot unlock the log file $self->{AMANDA_CONF}->{logdir}/log: $!";
  close(LOG)
    or die "taper: cannot close the log file $self->{AMANDA_CONF}->{logdir}/log: $!";
}

# Read the amanda label from the given device.
# Returns 0 on failure, the label otherwise.
sub getLabel {
    my $self = shift;
    my $dev = shift;
    my $label = 0;
    my $writeDev = $dev;

    # Strip the "0,0,0" from the device name
    if ($dev =~ /:/) {
	$dev = $`;
	$writeDev = $';
    } else {
	$self->log_add("ERROR", "device specification $dev doesn't include a mount device");
	return 0;
    }

    # Mount the disk. Requires a 'mount' program that allows mounting
    # of $dev on $CDRW_MOUNT_DIR as non-priviledged user by specifying
    # either the device or the mount point (might be Linux specific).
    my $error = `$self->{MOUNT} -r $dev 2>&1`;
    $error =~ s/\s+/ /gs;
    if ($error) {
	$self->log_add("ERROR", "could not mount disk: $error");
    } else {
	# read and check the media label
	if (!open(LABEL, "<$self->{AMANDA_CONF}->{tapedev}/AMANDA_LABEL")) {
	    $self->log_add("ERROR", "not an amanda disk");
	} else {
	    $label = <LABEL>;
	    close LABEL;
	    $label =~ s/[\r\n]+//sg;
	    if ($label !~ $self->{AMANDA_CONF}->{labelstr}) {
		$self->log_add("ERROR", "label $label doesn't match '$self->{AMANDA_CONF}->{labelstr}'");
		$label = 0;
	    }
	}

	# Umount the disk.
	$error = `$self->{UMOUNT} $self->{AMANDA_CONF}->{tapedev} 2>&1`;
	$error =~ s/\s+/ /gs;
	if ($error) {
	    $self->log_add("ERROR", "could not umount disk: $error");
	}
    }

    return $label;
}

1;

