/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * A few small modifications (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _GNU_SOURCE
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <ncursesw/ncurses.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <libintl.h>
#include <locale.h>
#include <sys/types.h>
#include <sys/wait.h>


#include "options.h"
#include "log.h"
#include "main.h"
#include "config.h"
#include "color.h"
#include "gettext.h"
#include "utils.h"
#include "cdw_widgets.h"
#include "cdrom_ctrl.h"



extern struct conf config;


void after_event(char *message, int adddic)
{
	if (strcmp(config.showlog, "1") == 0)
		log_window(message, config.logfile, 0);
#ifdef HAVE_LIBMYSQLCLIENT
	if ((strcmp(config.autodic, "1") == 0) && (adddic == 1))
		add_to_dic();
#endif
#ifdef HAVE_LIBSQLITE
	if ((strcmp(config.autodic, "1") == 0) && (adddic == 1))
		add_to_dic();
#endif
	if (strcmp(config.eject, "1") == 0) {
		eject_tray(config.cdrwdevice);
	}
}




/*
 * Filter function for scandir()
 *
 * Filter function for scandir() - returns zero value for
 * current directory ('.') and parent directory ('..') entries.
 * Entries, for which filter returns zero value, are ommited by scandir()
 *
 * \param const struct dirent *unused
 *
 * \returns 0 if entry is '.' or '..'. 1 otherwise
 */
int one(const struct dirent *unused)
{
	if (strcmp(unused->d_name, ".") && strcmp(unused->d_name, "..")) {
		return 1;
	} else
		return 0;
}




/*
 * Get size of all files and dirs that are in table with paths
 *
 * Get size of all files and dirs (recursively) that are in table with paths.
 *
 * \param char **pathtable - pointer to table of *char strings with full (non-relative) paths
 * \param int paths - size of table
 *
 * \returns size of all dirs/files from table or -1 if failure
 */
long long int dirsize(char **pathtable, int paths)
{
	long long int d_size = 0;

	int i = 0;
	for (i = 0; i < paths; i++) {
		long long int rv = _dirsize(*(pathtable + i));
		if (rv == -1 ) {
			return -1; /* error */
		} else {
			d_size += rv;
		}
	}
	return d_size;
}




/*
 * Get size of given dir (recursively) or file
 *
 * Get size of given dir (recursively) or file. Do not follow
 * symlinks (that's easy to change).
 *
 * \param char *path - non-relative path to dir or file.
 *
 * \returns -1 on failure, value >=0 on success
 */
long long int _dirsize(char *path)
{
	struct stat *finfo;
	int ok;
	long long int _d_size = 0;

	finfo = (struct stat *) malloc(sizeof(struct stat));
	if ((ok = stat(path, finfo)) != 0) { /* lstat() - don't follow links */
		return -1;
	}

	if ( S_ISREG(finfo->st_mode) ) { /* regular file */
		_d_size += finfo->st_size;
	} else if ( S_ISLNK(finfo->st_mode) ) { /* symbolic link */
		_d_size += finfo->st_size; /* st_size is lenght of path without terminating null */
	} else if ( S_ISDIR(finfo->st_mode) ) { /* directory */
		_d_size += finfo->st_blocks * 512;
		/* get directory listing */
		struct dirent **eps;
		int n = scandir(path, &eps, one, &alphasort);
		if (n > 0) {
			int i;
			for (i = 0; i < n; i++) {
				/* for every dir entry create new full path to this entry and scan it */
				char *newpath = (char *) malloc(strlen(eps[i]->d_name) + strlen(path) + 4);
				sprintf(newpath, "%s/%s", path, eps[i]->d_name);
				long long int dd = _dirsize(newpath);
				if (dd > 0) {
					_d_size += dd;
				}

				free(eps[i]);
				eps[i] = NULL;

				free(newpath);
				newpath = NULL;
			}
		}

		free(eps); /* FIXME - this should be in 'if' or outside of 'if'? */
		eps = NULL;
	}

	free(finfo);
	finfo = NULL;

	return _d_size;
}




sig_atomic_t child_exit_status;
/*
 * Remove zombie process created by run_command()
 */
void clean_up_child_process(int signal_number)
{
	/* clean child process */
	int status;
	wait(&status);
	child_exit_status = status;

	return;
}




/*
 * Process main(argc, argv[]) arguments
 */
void process_commandline_args(int argc, char *argv[])
{

/*      printf(_("Usage: %s\n"),argv[0]); */
#ifdef HAVE_LIBMYSQLCLIENT
	if ((strcmp((char *) argv[1], "--catalog") == 0) || (strcmp((char *) argv[1], "-c") == 0)) {
		initscr();
		start_color();
		cbreak();
		noecho();
		keypad(stdscr, TRUE);
		// colors
		init_curses_colors();
		lines = LINES - 2;
		if ((LINES < 24) || (COLS < 79)) {
			clean_before_cdw_exit();
			/* 2TRANS: error message displayed when terminal requirements are not met */
			printf(_("Needed min 80x25 terminal!"));
			exit(-1);
		}
		cddb_window();
		clean_before_cdw_exit();
		exit(0);
	}
#endif

	if ((strcmp((char *) argv[1], "--version") == 0) || (strcmp((char *) argv[1], "-v") == 0)) {
		clean_before_cdw_exit();
		printf("%s-%s (C) 2002-2003 Balazs Varkonyi\n\n", PACKAGE, VERSION);
		printf("%s-%s (C) 2007 Kamil Ignacak\n\n", PACKAGE, VERSION);
		exit(0);
	}
	if ((strcmp((char *) argv[1], "--help") == 0) || (strcmp((char *) argv[1], "-h") == 0)) {
		endwin();
		/* 2TRANS: %s is program name */
		printf(_("Usage: %s [options]\n\n"), argv[0]);
		printf(_("Options:\n\n"));
#ifdef HAVE_LIBMYSQLCLIENT
		printf(_("  --catalog | -c            : start in Disk Catalog mode\n"));
#endif
		printf(_("  --help    | -h            : show this screen\n"));
		printf(_("  --version | -v            : show version\n\n"));
		clean_before_cdw_exit();
		exit(0);
	}
	/* 2TRANS: first %s is program name, second - argument given to the program */
	printf(_("%s: invalid option -- %s\n"), argv[0], argv[1]);
	/* 2TRANS: %s is program name */
	printf(_("Try `%s --help`\n"), argv[0]);
	clean_before_cdw_exit();
	exit(0);
}




/*
 * Simple wrapper for locale setting functions
 */
int cdw_locale_init(void)
{
	setlocale(LC_ALL, "");
	textdomain(PACKAGE);
	bindtextdomain(PACKAGE, LOCALEDIR);

	return 0;
}




/**
 * Check existence of given file, store lstat() info in one of arguments
 * check if given file (regular file or dir) exists, has expected permissions
 * is of given type. Store collected information in struct stat variable.
 *
 * \param char *fullpath - full path to given file
 * \param int permissions - file permissions: S_IRUSR, S_IWUSR, S_IXUSR or logical sum
 * \param int filetype - CDW_FILE or CDW_DIR
 * \param struct stat *finfo - allocated variable that will be filled with file information; caller can set it to NULL if he is not interested in file properties
 *
 * \returns CDW_FILE_INVALID, CDW_PERM_ERROR, CDW_SYS_ERROR, CDW_OK;
 */
cdw_rv_t is_valid_path(char *fullpath, mode_t permissions, int filetype, struct stat *finfo)
{
	bool internal_finfo = false;
	if (finfo == NULL) {
		internal_finfo = true;
		finfo = (struct stat *) malloc(sizeof(struct stat));

	}

	if (lstat(fullpath, finfo) == -1) { /* lstat failure */
		/* fprintf(stderr, _("lstat failure\n")); */
		return CDW_SYS_ERROR;
	}

	/* check type of file */
	switch (filetype) {
		case CDW_FILE: /* regular file */
			if ( S_ISREG(finfo->st_mode) ) {
				break;
			} else {
				return CDW_FILE_INVALID; /* not a normal file */
			}
		case CDW_DIR: /* directory */
			if ( S_ISDIR(finfo->st_mode) ) {
				break;
			} else {
				return CDW_FILE_INVALID; /* not a dir */
			}
		default:
			return CDW_FILE_INVALID; /* other problem related to filetype */
	} /* switch */

	/* check permissions */
	if (permissions == -1) { /* don't ask for permissions */
		if (internal_finfo) {
			free(finfo);
			finfo = NULL;
		}
		return CDW_OK;
	} else {
		uid_t uid = geteuid();
		bool user_perm = ((finfo->st_uid == uid) && ((finfo->st_mode & S_IRWXU) == permissions));

		if (user_perm) {
			if (internal_finfo) {
				free(finfo);
				finfo = NULL;
			}
			/* fprintf(stderr, "Correct permissions for selected file %s\n", fullpath); */
			return CDW_OK;
		} else {
			if (internal_finfo) {
				free(finfo);
				finfo = NULL;
			}
			/* fprintf(stderr, "Wrong permissions for selected file %s\n", fullpath); */
			return CDW_PERM_ERROR; /* wrong permissions */
		}
	}
}



/**
 * Wrapper for is_valid_path() that asks user for valid path n times
 *
 * \param path - initial path, allocated string that an be modified by function
 * \param filetype - type of file: CDW_DIR or CDW_FILE
 * \param permissions - file permissions: S_IRUSR, S_IWUSR, S_IXUSR or logical sum; set to -1 if you don't want to check permissions
 * \param n - how many times user can enter invalid path?
 * \param title - title of dialog window
 * \param promt - message displayed in dialog window
 *
 * \returns CDW_OK if user entered valid path,
 *          CDW_CANCEL if user canceled entering the path,
 *                     or the path was invalid in n tries
 */
cdw_rv_t get_valid_path(char *path, mode_t permissions, int filetype, int n, char *title, char *prompt)
{
	int i = 0;
	for (i = 0; i < n; i++) {
		cdw_rv_t rv = input_box(title, prompt, path, 0);

		if (rv == CDW_OK) {
			rv = is_valid_path(path, permissions, filetype, NULL);

			if (rv == CDW_OK) {
				return CDW_OK;
			} else if (rv == CDW_PERM_ERROR) {
				/* 2TRANS: this is dialog window title */
				dialogbox(_("Message"),
					  /* 2TRANS: this is error message in dialog window */
					  _("Selected file has wrong permissions"), DIALOG_OK);
				/* retry */
			} else if (rv == CDW_FILE_INVALID) {
				/* 2TRANS: this is dialog window title */
				dialogbox(_("Message"),
					  /* 2TRANS: this is error message in dialog window */
					  _("Selected file is invalid"), DIALOG_OK);
				/* retry */
			} else {
				/* 2TRANS: this is dialog window title */
				dialogbox(_("Message"),
					  /* 2TRANS: this is error message in dialog window - unknown error */
					  _("Some error occured. Please retry."), DIALOG_OK);
				/* retry */
			}
		} else { /* user abandoned action */
			return CDW_CANCEL;
		}
	}

	return CDW_CANCEL;
}




/*
 * Setup code dealing with zombie processes created during run_command() call
 *
 * \returns 0
 */
int zombie_handler_init(void)
{
	/* handling terminated child process - data structure*/
	struct sigaction sigchild_action;

	/* handling terminated child process - setup */
	memset(&sigchild_action, 0, sizeof(sigchild_action));
	sigchild_action.sa_handler = &clean_up_child_process;
	sigaction(SIGCHLD, &sigchild_action, NULL);

	return 0;
}




/**
 * Get home directory
 *
 * Function puts into it's argument user home directory (with ending '\0'),
 * or returns error. Error may occur when there is no HOME env. variable,
 * when HOME dir does not exists or HOME dir does not have 'r' and 'w'
 * permissions set.
 *
 * \param char *path - empty buffer for path to user home directory - this pointer will be initialized by function; value is set to NULL if error
 *
 * \returns path length (without trailing '\0') if no error,
 *          CDW_SYS_ERROR if no such env. variable,
 *          CDW_FILE_INVALID when dir does not exist,
 *          CDW_MEM_ERROR when memory allocation error occured,
 *          CDW_PERM_ERROR when r/w permissions are wrong,
 *          CDW_GEN_ERROR - other error
 */
int get_home_dir(char **path)
{
	int len = 0;

	/* paranoid mode = on */

	struct stat *finfo = (struct stat *) malloc(sizeof(struct stat));
	if (finfo == NULL) {
		return CDW_MEM_ERROR;
	}

	if (getenv("HOME") == NULL) { /* getenv error - no such variable */
		free(finfo);
		finfo = NULL;
		return CDW_SYS_ERROR;
	} else {
		int rv = 0;
		len = strlen(getenv("HOME"));
		*path = (char *) malloc(len + 1);

		if (*path == NULL) {
			free(finfo);
			finfo = NULL;
			return CDW_MEM_ERROR;
		}

		strncpy(*path, getenv("HOME"), len);
		strncpy(*path + len, "\0", 1);

		rv = is_valid_path(*path, S_IWUSR|S_IRUSR|S_IXUSR, CDW_DIR, finfo);

		if (rv == 0) { /* lstat OK */
			free(finfo);
			finfo = NULL;
			return len; /* everything is OK */
		} else {
			free(*path);
			*path = NULL;
			free(finfo);
			finfo = NULL;
			return rv; /* wrong permissions */
		}
	}
}




/**
 * Get temporary directory
 *
 * Function puts into it's argument some directory (with ending '\0'),
 * or returns error. Error may occur when there is no path selected by user
 * or dir does not have 'rwx' permissions set - it means that this function
 * checks for validity of folder. Function returns with -1 if
 * user hits ESCAPE key.
 *
 * \param char *path - empty buffer for path to directory - this pointer will be initialized by function; value is set to NULL if error
 *
 * \returns path length (without trailing '\0') if no error,
 *          CDW_FILE_INVALID when dir does not exist,
 *          CDW_PERM_ERROR when r/w permissions are wrong,
 *          CDW_MEM_ERROR when memory allocation error occured,
 *          CDW_GEN_ERROR - other error
 */
int get_tmp_dir(char **path)
{
	int len = 0;
	int rv = 0;

	struct stat *finfo = (struct stat *) malloc(sizeof(struct stat));
	if (finfo == NULL) {
		return CDW_MEM_ERROR;
	}

	*path = (char *) malloc(200); /* FIXME - magic number */
	if (*path == NULL) {
		free(finfo);
		finfo = NULL;
		return CDW_MEM_ERROR;
	}

	/* some sane initial value */
	strcpy(*path, "/tmp");

	/* 2TRANS: this is title of dialog window */
	rv = input_box(_("Select directory"),
		       /* 2TRANS: this is message in dialog window. Please keep '\n'. */
		       _("Please select temporary directory.\n   Press ESC to skip."), *path, 0);
	if (rv == CDW_ESCAPE) {
		return -1;
	}

	/* remove spaces and slashes form end of path string */
	rtrim(*path);
	len = strlen(*path);
	if ((*(*path + len - 1) == '\\') || (*(*path + len - 1	) == '/')) {
		len--;
	}

	strncpy(*path + len, "\0", 1);

	rv = is_valid_path(*path, S_IWUSR|S_IRUSR|S_IXUSR, CDW_DIR, finfo);

	if (rv == 0) { /* lstat OK */
		free(finfo);
		finfo = NULL;
		return len; /* everything is OK */
	} else {
		free(*path);
		*path = NULL;
		free(finfo);
		finfo = NULL;
		return CDW_FILE_INVALID; /* wrong permissions? */
	}
}




char *ltrim(char *str)
{
	return str+strspn(str, " \t");
}




char *rtrim(char *str)
{
	char *p;

	for (p = str; *p; ++p);
	while ( (*--p == ' ') || (*p == '\t') );
	*++p = '\0';
	return str;
}



/* shamelesly stolen from libc documentation - LGPL code */
char *concat(const char *str, ...)
{
	va_list ap;
	char *s;
	size_t allocated = 100;
	char *result = (char *) malloc(allocated);

        if (result != NULL) {
		char *newp;
		char *wp;

		va_start(ap, str);

		wp = result;
		for (s = str; s != NULL; s = va_arg(ap, const char *)) {
			size_t len = strlen(s);

			/* Resize the allocated memory if necessary.  */
			if (wp + len + 1 > result + allocated) {
				allocated = (allocated + len) * 2;
				newp = (char *) realloc(result, allocated);
				if (newp == NULL) {
					free(result);
					return NULL;
				}
				wp = newp + (wp - result);
				result = newp;
			}

			wp = mempcpy(wp, s, len);
		}

		/* Terminate the result string.  */
		*wp++ = '\0';

		/* Resize memory to the optimal size.  */
		newp = realloc(result, wp - result);
		if (newp != NULL)
			result = newp;

		va_end(ap);
	}

	return result;
}




void my_itoa(int i, char *string)
{
	int r;
	char *s = string + 10;

	*(s) = '\0';
	s--;

	do {
		r = i % 10;
		i = i/10;
		*s = (char) (r + '0');
		s--;
	} while (i >= 10);
	*(s) = (char)(i + '0');


	return;
}




/* unused code below */
#if 0 // this function probably will be no longer used, since we use wide characters ncurses libraries
/**
 * Remove non-printable characters from path
 *
 * Replace non-printable characters (like '»') from file's path.
 * Such characters may be visible in files selector window, but
 * path containig such characters is not displayed by 'Selected files'
 * list, because it is implemented using ncurses MENU (and I don't know
 * how to fix it yet). Non-printable characters are replaced by '?'.
 *
 * It may be possible to replace thiese characters only in strings
 * displayed in 'Selected files' window and print them unaltered to
 * cdrecord-source-files-paths tmp file.
 *
 * \param char **string - pointer to path - this parameter may be modified by function
 *
 */
void make_ncurses_printable_string(char **string)
{

	return;

	int len = strlen(*string);

	int i = 0;
	for (i = 0; i < len; i++) {
		if (! isprint(*((*string) + i))) {
			(*((*string) + i)) = '?';
			if (! isprint(*((*string) + i + 1))) {
				i++;
}
}
}

	return;
}








/* this is part of cdw_rv_t is_valid_path(char *fullpath, mode_t permissions, int filetype, struct stat *finfo),
it should have checked for permission to access a file based on permissions of user OR group, but it does not work somehow for directories:
user can't write to dir if it is in dir's group and permissions are d---rwx--- */
if (permissions == -1) { /* don't ask for permissions */
		if (internal_finfo) {
			free(finfo);
			finfo = NULL;
}
		return CDW_OK;
} else {
		uid_t uid = geteuid();
		uid_t gid = getegid();
		bool user_perm = ((finfo->st_uid == uid) && ((finfo->st_mode & S_IRWXU) == permissions));
		/* shifting by one converts user permissions into group
		 * permissions, additinal mask with S_IRWXG ensures that
		* we check only group byte */
		bool group_perm = ((finfo->st_gid == gid) && ((finfo->st_mode & S_IRWXG) == ((permissions >> 3) & S_IRWXG)));
		fprintf(stderr, "\n\n");
						if (group_perm) {
						fprintf(stderr, "group perm = true\n");
}
						if (user_perm) {
						fprintf(stderr, "user perm = true\n");
}
						fprintf(stderr, "stat group perm: %d, user perm: %d\n", (finfo->st_mode & S_IRWXG) >> 3, (finfo->st_mode & S_IRWXU) >> 6);
						fprintf(stderr, "permissions variable, group: %d, user: %d\n", ((permissions >> 3) & S_IRWXG) >> 3, permissions >> 6);

						if (user_perm || group_perm) {
						if (internal_finfo) {
						free(finfo);
						finfo = NULL;
}
						fprintf(stderr, "Correct permissions for selected file %s\n", fullpath);
						return CDW_OK;
} else {
						if (internal_finfo) {
						free(finfo);
						finfo = NULL;
}
						fprintf(stderr, "Wrong permissions for selected file %s\n", fullpath);
						return CDW_PERM_ERROR; /* wrong permissions */
}
}














#endif // if 0



