/* BringToFrontGraphicsChange.m
 *
 * Copyright (C) 1993-2003 by vhf interservice GmbH
 * Authors:  Georg Fleischmann
 *
 * created:  1993 based on the Draw example files
 * modified: 2003-06-26
 *
 * The ChangeDetail.h and ChangeDetail.m files contain
 * the @interfaces and @implementations for the 11 
 * subclasses of ChangeDetail, as well as for ChangeDetail
 * itself. We grouped all the classes into one pair of 
 * files because the classes are so tiny and their behavior
 * is so similar.
 *
 * ChangeDetail
 *     ArrowChangeDetail
 *     DimensionsChangeDetail
 *     FillColorChangeDetail
 *     FillModeChangeDetail
 *     LineCapChangeDetail
 *     LineColorChangeDetail
 *     LineJoinChangeDetail
 *     LineWidthChangeDetail
 *     MoveChangeDetail
 *     OrderChangeDetail
 *     StepWidthChangeDetail
 *     RadialCenterChangeDetail
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include "undo.h"

@interface ChangeDetail(PrivateMethods)
- (BOOL)personalChangeExpected;
@end

@implementation ChangeDetail

- initGraphic:aGraphic change:aChange
{
    //NSMutableArray *subGraphics;
    //int count, i;
    //id changeDetail;

    graphic = aGraphic;
    change = aChange;
#if 0
    if ([graphic isKindOfClass:[VGroup class]] && [self useNestedDetails])
    {
        detailsDetails = [[NSArray alloc] init];
	subGraphics = [(VGroup *)graphic subGraphics];
	count = [subGraphics count];
	changeExpected = NO;
	for (i = 0; i < count; i++) {
	    changeDetail = [[[aChange changeDetailClass] alloc] initGraphic:[subGraphics objectAtIndex:i] change:aChange];
	    changeExpected = changeExpected || [changeDetail changeExpected];
	    [detailsDetails addObject:changeDetail];
	}
    }
    else
#endif
    {
        detailsDetails = nil;
	changeExpected = [self personalChangeExpected];
    }
    return self;
}

- (void)dealloc
{
    [detailsDetails removeAllObjects];
    [detailsDetails release];
    [super dealloc];
}

- (void)setLayer:(int)lay
{
    layer = lay;
}
- (int)layer
{
    return layer;
}

- (VGraphic *)graphic
{
    return graphic;
}

- (BOOL)useNestedDetails
{
    return YES;
}

- (BOOL)changeExpected
{
    return changeExpected;
}

- (void)recordDetail
{
    if (detailsDetails)
        [detailsDetails makeObjectsPerformSelector:@selector(recordDetail)];
    else
      [self recordIt]; 
}

- (void)undoDetail
{
    if (detailsDetails)
        [detailsDetails makeObjectsPerformSelector:@selector(undoDetail)];
    else
      [self undoIt]; 
}

- (void)redoDetail
{
    if (detailsDetails)
        [detailsDetails makeObjectsPerformSelector:@selector(redoDetail)];
    else
      [self redoIt]; 
}

- (void)recordIt
{
    /* Implemented by subclasses */
     
}

- (void)undoIt
{
    /* Implemented by subclasses */
     
}

- (void)redoIt
{
    /* Implemented by subclasses */
     
}

- (BOOL)personalChangeExpected
{
    return YES;
}

@end


@implementation DimensionsChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)recordDetail
{
    oldBounds.size = [graphic size];
}
- (void)undoDetail
{
    newBounds.size = [graphic size];
    [graphic setSize:oldBounds.size];
}
- (void)redoDetail
{
    [graphic setSize:newBounds.size];
}
@end

@implementation RadiusChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)recordDetail
{
    oldRadius = [graphic radius];
}
- (void)undoDetail
{
    newRadius = [graphic radius];
    [graphic setRadius:oldRadius];
}
- (void)redoDetail
{
    [graphic setRadius:newRadius];
}
@end

@implementation LockChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)recordDetail
{
    isLocked = [graphic isLocked];
}
- (void)undoDetail
{
    [graphic setLocked:isLocked];
}
- (void)redoDetail
{
    [graphic setLocked:(isLocked)?NO:YES];
}
@end

@implementation FillChangeDetail
- (void)recordIt
{
    oldFillMode = [graphic filled];
}
- (void)undoIt
{
    newFillMode = [graphic filled];
    [graphic setFilled:oldFillMode];
}
- (void)redoIt
{
    [graphic setFilled:newFillMode];
}
- (BOOL)personalChangeExpected
{
    return ([graphic filled] != [(FillGraphicsChange*)change fill]);
}
@end

@implementation ColorChangeDetail
- (void)recordIt
{
    [oldColor release];
    switch ( [change colorNum] )
    {
        default: oldColor = [[graphic color] copy]; break;
        case 1:  oldColor = [[(VPolyLine*)graphic fillColor] copy]; break;
        case 2:  oldColor = [[(VPolyLine*)graphic endColor] copy];
    }
}
- (void)undoIt
{
    switch ( [change colorNum] )
    {
        default: [graphic setColor:oldColor]; break;
        case 1:  [(VPolyLine*)graphic setFillColor:oldColor]; break;
        case 2:  [(VPolyLine*)graphic setEndColor:oldColor];
    }
}
- (void)redoIt
{    NSColor * color = [change color];

    switch ( [change colorNum] )
    {
        default: [graphic setColor:color]; break;
        case 1:  [(VPolyLine*)graphic setFillColor:color]; break;
        case 2:  [(VPolyLine*)graphic setEndColor:color];
    }
}
- (BOOL)personalChangeExpected
{
    switch ( [change colorNum] )
    {
        default: return (![[graphic color] isEqual:[change color]]);
        case 1:  return (![[(VPolyLine*)graphic fillColor] isEqual:[change color]]);
        case 2:  return (![[(VPolyLine*)graphic endColor] isEqual:[change color]]);
    }
}
@end

@implementation NameChangeDetail
- (void)recordIt
{
    oldName = [[(VMark*)graphic name] copy];
}
- (void)undoIt
{
    [(VMark*)graphic setName:oldName];
}
- (void)redoIt
{
    [(VMark*)graphic setName:[(NameGraphicsChange*)change name]];
}
- (void)dealloc
{
    [oldName release];
    [super dealloc];
}
@end

@implementation WidthChangeDetail
- (void)recordIt
{
    oldLineWidth = [graphic width]; 
}
- (void)undoIt
{
    [graphic setWidth:oldLineWidth]; 
}
- (void)redoIt
{
    float lineWidth = [change lineWidth];
    [graphic setWidth:lineWidth]; 
}
- (BOOL)personalChangeExpected
{
    return ([graphic width] != [change lineWidth]);
}
@end

@implementation LengthChangeDetail
- (void)recordIt
{
    oldLength = [graphic length]; 
}
- (void)undoIt
{
    [graphic setLength:oldLength]; 
}
- (void)redoIt
{
    float length = [(LengthGraphicsChange*)change length];
    [graphic setLength:length]; 
}
- (BOOL)personalChangeExpected
{
    return ([graphic length] != [(LengthGraphicsChange*)change length]);
}
@end

@implementation MoveChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)undoDetail
{
    NSPoint offset = [change undoVector];
    [graphic moveBy:offset];
}
- (void)redoDetail
{
    NSPoint offset = [change redoVector];
    [graphic moveBy:offset];
}
@end

@implementation MovePointChangeDetail
- (void)recordIt
{
    if ( [change ptNum]<0 )
        ptNum = ([change moveAll]) ? 0 : [graphic selectedKnobIndex];
    else
        ptNum = [change ptNum];

    control = [(App*)NSApp control];
    oldPoint = [graphic pointWithNum:ptNum];
}
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)undoIt
{
    newPoint = [graphic pointWithNum:ptNum];
    if ( [change moveAll] )
        [graphic moveTo:oldPoint];
    else
    {   if (control && [graphic isKindOfClass:[VArc class]])
            [(VArc*)graphic movePoint:ptNum to:oldPoint control:control];
        else if (control && [graphic isKindOfClass:[VPath class]])
            [(VPath*)graphic movePoint:ptNum to:oldPoint control:control];
        else if (control && [graphic isKindOfClass:[VGroup class]])
            [(VGroup*)graphic movePoint:ptNum to:oldPoint control:control];
        else
            [graphic movePoint:ptNum to:oldPoint];
    }

}
- (void)redoIt
{
    if ( [change moveAll] )
        [graphic moveTo:newPoint];
    else
    {   if (control && ([graphic isKindOfClass:[VArc class]]))
            [(VArc*)graphic movePoint:ptNum to:newPoint control:control];
        else if (control && [graphic isKindOfClass:[VPath class]])
            [(VPath*)graphic movePoint:ptNum to:newPoint control:control];
        else if (control && [graphic isKindOfClass:[VGroup class]])
            [(VGroup*)graphic movePoint:ptNum to:newPoint control:control];
        else
            [graphic movePoint:ptNum to:newPoint];
    }

}
@end

@implementation RotateChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)undoDetail
{
    if ( [change useGraphicOrigin] )
        [graphic setAngle:[change undoAngle] withCenter:[graphic pointWithNum:0]];
    else
        [graphic setAngle:[change undoAngle] withCenter:[(RotateGraphicsChange*)change center]];
}
- (void)redoDetail
{
    if ( [change useGraphicOrigin] )
        [graphic setAngle:[change redoAngle] withCenter:[graphic pointWithNum:0]];
    else
        [graphic setAngle:[change redoAngle] withCenter:[(RotateGraphicsChange*)change center]];
}
@end

@implementation MirrorChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)undoDetail
{
    [graphic mirrorAround:[(MirrorGraphicsChange*)change center]];
}
- (void)redoDetail
{
    [graphic mirrorAround:[(MirrorGraphicsChange*)change center]];
}
@end

@implementation ScaleChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)undoDetail
{
    [graphic scale:1.0/[change xScale] :1.0/[change yScale] withCenter:[(ScaleGraphicsChange*)change center]];
}
- (void)redoDetail
{
    [graphic scale:[change xScale] :[change yScale] withCenter:[(ScaleGraphicsChange*)change center]];
}
@end

@implementation AngleChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)recordDetail
{
    switch ( [change angleNum] )
    {
        default: undoAngle = [(VArc*)graphic begAngle]; break;
        case 1:  undoAngle = [(VArc*)graphic angle]; break;
        case 2:  undoAngle = [(VPolyLine*)graphic graduateAngle];
    }
}
- (void)undoDetail
{
    switch ( [change angleNum] )
    {
        default: [(VArc*)graphic setBegAngle:undoAngle]; break;
        case 1:  [(VArc*)graphic setAngle:undoAngle]; break;
        case 2:  [(VPolyLine*)graphic setGraduateAngle:undoAngle];
    }
}
- (void)redoDetail
{
    switch ( [change angleNum] )
    {
        default: [(VArc*)graphic setBegAngle:[change redoAngle]]; break;
        case 1:  [(VArc*)graphic setAngle:[change redoAngle]]; break;
        case 2:  [(VPolyLine*)graphic setGraduateAngle:[change redoAngle]];
    }
}
@end

@implementation OrderChangeDetail
- (BOOL)useNestedDetails
{
    return NO;
}
- (void)recordGraphicPositionIn:(NSArray*)layList
{   int	l, cnt;

    for ( l=0, cnt=[layList count]; l<cnt; l++ )
    {   NSArray	*list = [[layList objectAtIndex:l] list];
        int	ix;

        if ( (ix = [list indexOfObject:graphic]) != NSNotFound )
        {   graphicPosition = ix;
            layer = l;
            break;
        }
    }
}
- (unsigned)graphicPosition
{
    return graphicPosition;
}
@end

@implementation StepWidthChangeDetail
- (void)recordIt
{
    oldStepWidth = [(VArc*)graphic stepWidth]; 
}
- (void)undoIt
{
    [(VArc*)graphic setStepWidth:oldStepWidth]; 
}
- (void)redoIt
{
    float stepWidth = [change stepWidth];
    [(VArc*)graphic setStepWidth:stepWidth]; 
}
- (BOOL)personalChangeExpected
{
    return ([(VArc*)graphic stepWidth] != [change stepWidth]);
}
@end

@implementation RadialCenterChangeDetail
- (void)recordIt
{
    oldRadialCenter = [(VArc*)graphic radialCenter]; 
}
- (void)undoIt
{
    [(VArc*)graphic setRadialCenter:oldRadialCenter]; 
}
- (void)redoIt
{
    NSPoint rCenter = [(RadialCenterGraphicsChange*)change radialCenter];
    [(VArc*)graphic setRadialCenter:rCenter]; 
}
- (BOOL)personalChangeExpected
{   NSPoint	gRC = [(VArc*)graphic radialCenter];
    NSPoint	cRC = [(RadialCenterGraphicsChange*)change radialCenter];
    return (gRC.x != cRC.x || gRC.y != cRC.y);
}
@end
