/* help.c
 * 
 * This program is distributed under the GNU General Public License.
 * Copyright (C) 2000 Eugene Osintsev <gene@linuxave.net>
 */

#include <curses.h>
#include <setjmp.h>
#include "scr.h"

/* Pointer to help window */
static WINDOW *win = 0;

/* Show arrows on help window border with valid scrolling directions */
static void show_scroll_arrows(int x, int y, int maxx, int maxy);

static sigjmp_buf jmp;

/*------------------------------------------------------------------*/

int show_help()
{
    int ch, x, y;
    int maxx = ((80 < COLS) ? 0 : (80 - COLS));
    int maxy = ((25 < LINES) ? 0 : (25 - LINES));
    int sig;
    
    win = newpad(25, 80);
    keypad(win, TRUE);
    wbkgdset(win, C_PANEL);
    wclear(win);
    
    leaveok(stdscr, TRUE);

    if ((sig = sigsetjmp(jmp, 1)) != 0) {
        delwin(win);
        win = 0;
        leaveok(stdscr, FALSE);
        return sig;
    }

    wattrset(win, C_INFO);
    wmove(win, 1, 0);
    waddstr(win,
           "  Home (C-a)   Go to first glyph   C-n   Go to next glyph on both panels\n"
           "  End  (C-e)   Go to last glyph    C-p   Go to previous glyph on both panels\n"
           "  PgDn (C-f)   Go to next glyph\n"
           "  PgUp (C-b)   Go to previous glyph      q   Quit cfe\n"
	   "\n"
           "  d   Go to next difference between glyphs\n"
           "  g   Go to glyph by number (e.g., \"123g\" -- go to glyph #123)\n"
           "  m   Go to next modified glyph\n"
           "  M   Go to next unmodified glyph\n"
	   "\n"
           "  c   Copy glyph to buffer        i   Show miscellaneous info\n"
           "  p   Paste glyph from buffer     x   Show Unicode numbers for current glyph\n"
           "  s   Save file                   u   Undo last changes\n"
	   "\n"
           "  l   Shift glyph left            h   Flip glyph horizontally\n"
           "  r   Shift glyph right           v   Flip glyph vertically\n"
           "  t   Shift glyph top\n"
           "  b   Shift glyph bottom          Space   Set bit on/off\n"
	   "\n"
           "  Tab   Change panel              ?    Show help (F1 helps too)\n"
           "  C-l   Redraw screen             Esc  Close this help"
	   );

    /* Highlight hot keys */
    wattrset(win, C_BITS);
    
    mvwaddstr(win, 1, 2, "Home (C-a)");   mvwaddstr(win, 1, 35, "C-n");
    mvwaddstr(win, 2, 2, "End  (C-e)");   mvwaddstr(win, 2, 35, "C-p");
    mvwaddstr(win, 3, 2, "PgDn (C-f)");
    mvwaddstr(win, 4, 2, "PgUp (C-b)");   mvwaddstr(win, 4, 41, "q");

    mvwaddch(win, 6, 2, 'd');
    mvwaddch(win, 7, 2, 'g');
    mvwaddch(win, 8, 2, 'm');
    mvwaddch(win, 9, 2, 'M');

    mvwaddch(win, 11, 2, 'c');  mvwaddch(win, 11, 34, 'i');
    mvwaddch(win, 12, 2, 'p');  mvwaddch(win, 12, 34, 'x');
    mvwaddch(win, 13, 2, 's');  mvwaddch(win, 13, 34, 'u');
    
    mvwaddch(win, 15, 2, 'l');  mvwaddch(win, 15, 34, 'h');
    mvwaddch(win, 16, 2, 'r');  mvwaddch(win, 16, 34, 'v');
    mvwaddch(win, 17, 2, 't');
    mvwaddch(win, 18, 2, 'b');  mvwaddstr(win, 18, 34, "Space");
    
    mvwaddstr(win, 20, 2, "Tab");  mvwaddch(win, 20, 34, '?');
    mvwaddstr(win, 21, 2, "C-l");  mvwaddstr(win, 21, 34, "Esc");

    mvwaddstr(win, 20, 50, "F1");

    x = y = 0;
    maxx = ((80 < COLS) ? 0 : (80 - COLS));
    maxy = ((25 < LINES) ? 0 : (25 - LINES));
    
    /* Border */    
    attrset(C_PANEL);
    border(VLINE, VLINE, HLINE, HLINE, ULCORNER, URCORNER, LLCORNER, LRCORNER);
    mvaddstr(0, (COLS-6)/2, " Help ");
    show_scroll_arrows(x, y, maxx, maxy);

    refresh();
    prefresh(win, y, x, 1, 1, LINES-2, COLS-2);

    while ((ch = wgetch(win)) != 27) {  /* Quit help on <Esc> */
        switch (ch) {
        case KEY_UP:
            if (y > 0)
                y--;
            break;
        case KEY_DOWN:
            if (y < maxy)
                y++;
            break;
        case KEY_LEFT:
            if (x > 0)
                x--;
            break;
        case KEY_RIGHT:
            if (x < maxx)
                x++;
            break;
        case KEY_HOME:
            y = 0;
            break;
        case KEY_END:
            y = maxy;
            break;
        }
        prefresh(win, y, x, 1, 1, LINES-2, COLS-2);
        show_scroll_arrows(x, y, maxx, maxy);
        refresh();
    }
    
    delwin(win);
    win = 0;
    leaveok(stdscr, FALSE);
    return 0;
}

/*------------------------------------------------------------------*/

void close_help(int sig)
{
    if (win) {
        siglongjmp(jmp, sig);
    }
}

/*------------------------------------------------------------------*/

void show_scroll_arrows(int x, int y, int maxx, int maxy)
{
    if (x > 0) {
        attrset(C_MSG);
        mvaddch(LINES-1, 1, LARROW);
    } else {
        attrset(C_PANEL);
        mvaddch(LINES-1, 1, HLINE);
    }

    if (x < maxx) {
        attrset(C_MSG);
        mvaddch(LINES-1, COLS-2, RARROW);
    } else {
        attrset(C_PANEL);
        mvaddch(LINES-1, COLS-2, HLINE);
    }
    
    if (y > 0) {
        attrset(C_MSG);
        mvaddch(1, COLS-1, UARROW);
    } else {
        attrset(C_PANEL);
        mvaddch(1, COLS-1, VLINE);
    }

    if (y < maxy) {
        attrset(C_MSG);
        mvaddch(LINES-2, COLS-1, DARROW);
    } else {
        attrset(C_PANEL);
        mvaddch(LINES-2, COLS-1, VLINE);
    }
}


/* EOF */
