/* scr.c
 * 
 * This program is distributed under the GNU General Public License.
 * Copyright (C) 2000 Eugene Osintsev <gene@linuxave.net>
 */

#include <config.h>
#include <curses.h>
#include <string.h>
#include "font.h"
#include "scr.h"
#include "undo.h"

extern struct CFE_FONT  font[2];
extern struct CFE_GLYPH glyph[2];

/* The height of font glyphs */
static int height;

/* String representation of font types */
static char font_type[2][4] = {"RAW", "RAW"};

/* Message string and its attribute (C_MSG or C_ERR) */
static char msg_str[81];
static int  msg_attr;

/*------------------------------------------------------------------*/

void cfe_init_scr(int is_ascii, int is_bw)
{
    height = font[0].height;
    
    if (font[0].type == F_PSF)
        strcpy(font_type[0], "PSF");
    
    if (font[1].type == F_PSF)
        strcpy(font_type[1], "PSF");
    
    initscr();
    keypad(stdscr, TRUE);
    cbreak();
    noecho();

    if (is_ascii) {
        ULCORNER = '+';
        URCORNER = '+';
        LLCORNER = '+';
        LRCORNER = '+';
        HLINE = '-';
        VLINE = '|';
        LTEE = '+';
        RTEE = '+';
        LARROW = '<';
        RARROW = '>';
        DARROW = 'v';
        UARROW = '^';
    } else {
        ULCORNER = ACS_ULCORNER;
        URCORNER = ACS_URCORNER;
        LLCORNER = ACS_LLCORNER;
        LRCORNER = ACS_LRCORNER;
        HLINE = ACS_HLINE;
        VLINE = ACS_VLINE;
        LTEE = ACS_LTEE;
        RTEE = ACS_RTEE;
        LARROW = ACS_LARROW;
        RARROW = ACS_RARROW;
        DARROW = ACS_DARROW;
        UARROW = ACS_UARROW;
    }

    if (has_colors() && !is_bw) {
        start_color();
#ifdef XTERM_COLORS            
        if (strcmp("xterm", termname())) {        
#endif  /* XTERM_COLORS */        
            init_pair(1, COLOR_WHITE, COLOR_BLACK);
            init_pair(2, COLOR_WHITE, COLOR_BLACK);
            init_pair(3, COLOR_CYAN, COLOR_BLACK);
            init_pair(4, COLOR_CYAN, COLOR_BLACK);
            init_pair(5, COLOR_RED, COLOR_BLACK);

            C_BITS = COLOR_PAIR(1) | A_BOLD;
            C_INFO = COLOR_PAIR(2) | A_NORMAL;
            C_PANEL = COLOR_PAIR(3) | A_NORMAL;
            C_MSG = COLOR_PAIR(4) | A_BOLD;
            C_ERR = COLOR_PAIR(5) | A_BOLD;
#ifdef XTERM_COLORS            
        } else {
            init_pair(1, COLOR_BLACK, COLOR_WHITE);
            init_pair(2, COLOR_BLACK, COLOR_WHITE);
            init_pair(3, COLOR_BLUE, COLOR_WHITE);
            init_pair(4, COLOR_MAGENTA, COLOR_WHITE);
            init_pair(5, COLOR_RED, COLOR_WHITE);

            C_BITS = COLOR_PAIR(1) | A_BOLD;
            C_INFO = COLOR_PAIR(2) | A_NORMAL;
            C_PANEL = COLOR_PAIR(3) | A_NORMAL;
            C_MSG = COLOR_PAIR(4) | A_BOLD;
            C_ERR = COLOR_PAIR(5) | A_BOLD;
        }
#endif  /* XTERM_COLORS */        
        bkgdset(COLOR_PAIR(1));
    } else {
        C_BITS = A_BOLD;
        C_INFO = A_NORMAL;
        C_PANEL = A_NORMAL;
        C_MSG = A_UNDERLINE;
        C_ERR = A_STANDOUT;
    }
}

/*------------------------------------------------------------------*/

void alert(const char *str)
{
    strncpy(msg_str, str, 81);
    msg_str[80] = '\0';
    msg_attr = C_ERR;
}

/*------------------------------------------------------------------*/

void message(const char *str)
{
    strncpy(msg_str, str, 81);
    msg_str[80] = '\0';
    msg_attr = C_MSG;
}

/*------------------------------------------------------------------*/

void resize_message(int cols, int lines)
{
    char str[23];
    
    attrset(C_ERR);
    move(0, 0);
    clrtoeol();
#ifdef HAVE_SNPRINTF
    snprintf(str, 23, "Resize window to %dx%d", cols, lines);
#else
    sprintf(str, "Resize window to %dx%d", cols, lines);
#endif  /* HAVE_SNPRINTF */
    addnstr(str, COLS);
}

/*------------------------------------------------------------------*/

void draw_frame()
{
    char str[34];
    int i, len, add[2] = {0, 38};

    clear();
    attrset(C_PANEL);
    strcpy(str, "Console font editor ");
    strcat(str, VERSION);
    len = strlen(str);
    if (font[1].name)
        move(0, (75 - len) / 2);
    else
        move(0, (37 - len) / 2);
    addstr(str);

    for (i = 0; i <= 1 && font[i].name; i++) {
        attrset(C_INFO);

        len = strlen(font[i].name);
        if (len <= 33) {
            mvaddstr(3, 2 + add[i], font[i].name);
        } else {
            strncpy(str, font[i].name, 16);
            str[16] = '\0';
            strcat(str, "~");
            strcat(str, font[i].name + len - 16);
            mvaddstr(3, 2 + add[i], str);
        }

        attrset(C_PANEL);

        mvaddstr(5, 25 + add[i], "Char:  000");
        mvaddstr(7, 26 + add[i], "Col:  1");
        mvaddstr(8, 26 + add[i], "Row:  1");
        
        mvaddstr(4 + height, 32 + add[i], font_type[i]);

        mvaddch(2, add[i], ULCORNER);
        hline(HLINE, 35);
        mvaddch(2, 36 + add[i], URCORNER);

        mvaddch(3, add[i], VLINE);
        mvaddch(3, 36 + add[i], VLINE);

        mvaddch(4, add[i], LTEE);
        hline(HLINE, 35);
        mvaddch(4, 36 + add[i], RTEE);

        mvvline(5, add[i], VLINE, height);
        mvvline(5, 18 + add[i], VLINE, height);
        mvvline(5, 23 + add[i], VLINE, height);
        mvvline(5, 36 + add[i], VLINE, height);

        mvaddch(5 + height, add[i], LLCORNER);
        hline(HLINE, 35);
        mvaddch(5 + height, 36 + add[i], LRCORNER);
    }

    /* Font size & How to get help */

    attrset(C_PANEL);
    mvaddstr(LINES - 1, 0, "Font size: ");
    attrset(C_INFO);
    sprintf(str, "8x%d", height);
    mvaddstr(LINES - 1, 11, str);

    attrset(C_PANEL);
    mvaddstr(LINES - 1, COLS - 22, "Press ? for help info");
    attrset(C_INFO);
    mvaddch(LINES - 1, COLS - 16, '?');

    /* Glyph drawing */
    draw_glyph();
}

/*------------------------------------------------------------------*/

void draw_glyph()
{
    char str[4];
    int i, j, k, add[2] = {0, 38};

    for (i = 0; i <= 1 && font[i].name; i++) {
        /* Touch the glyph data because glyph[i].num may be changed
         * by pressing a navigation key.
         */
        glyph[i].data = font[i].data + font[i].height * glyph[i].num;
        
        /* Bits */
        for (j = 0; j < height; j++) {
            attrset(C_BITS);
            for (k = 0; k < 8; k++) {
#ifdef DOUBLED_8BIT                
                if (k == 7 &&
                    ((glyph[i].num >= 0xC0 && glyph[i].num <= 0xDF) ||
                     (glyph[i].num >= 0x1C0 && glyph[i].num <= 0x1DF))) {

                    attrset(C_ERR);
                } else {
                    attrset(C_BITS);
                }
#endif  /* DOUBLED_8BIT */                
                if (glyph[i].data[j] & (0200 >> k))
                    mvaddch(5 + j, 2 + add[i] + k * 2, 'O');
                else
                    mvaddch(5 + j, 2 + add[i] + k * 2, '.');
            }
            attrset(C_PANEL);
            sprintf(str, "%02X", glyph[i].data[j]);
            mvaddstr(5 + j, 20 + add[i], str);
        }

        attrset(C_INFO);

        /* Char */
        sprintf(str, "%03d", glyph[i].num);
        mvaddstr(5, 32 + add[i], str);

        /* Coords */
        sprintf(str, "%d", glyph[i].x + 1);
        mvaddstr(7, 32 + add[i], str);
        sprintf(str, "%d ", glyph[i].y + 1);
        mvaddstr(8, 32 + add[i], str);
    }

    if (glyph_modified(glyph[0].num))
        mvaddstr(4 + height, 25, "MOD");
    else
        mvaddstr(4 + height, 25, "   ");
    
    attrset(C_PANEL);

    if (file_modified())
        mvaddch(2, 34, '*');
    else
        mvaddch(2, 34, HLINE);
}

/*------------------------------------------------------------------*/

void draw_message()
{
    if (msg_attr == C_MSG)
        attrset(C_MSG);
    else if (msg_attr == C_ERR)
        attrset(C_ERR);

    move(1, 0);
    clrtoeol();
    addnstr(msg_str, (COLS < 80 ? COLS : 80));
}

/*------------------------------------------------------------------*/

int chk_win_size(int chk)
{
    static int is_valid_size = 0;
    int cols = (font[1].name ? 75 : 37), lines = font[0].height + 7;
    
    if (chk) {
        if (COLS < cols || LINES < lines) {
            clear();
            resize_message(cols, lines);
            move(LINES-1, COLS-1);
            is_valid_size = 0;
        } else {
            is_valid_size = 1;
        }
    }

    return is_valid_size;
}


/* EOF */
