;;;; describe.scm - Code to print description of data objects
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare (uses extras srfi-4))


(define ##describe#bytevector-data
  `((u8vector "vector of unsigned bytes" ,u8vector-length ,u8vector-ref)
    (s8vector "vector of signed bytes" ,s8vector-length ,s8vector-ref)
    (u16vector "vector of unsigned 16-bit words" ,u16vector-length ,u16vector-ref)
    (s16vector "vector of signed 16-bit words" ,s16vector-length ,s16vector-ref)
    (u32vector "vector of unsigned 32-bit words" ,u32vector-length ,u32vector-ref)
    (s32vector "vector of signed 32-bit words" ,s32vector-length ,s32vector-ref)
    (f32vector "vector of 32-bit floats" ,f32vector-length ,f32vector-ref)
    (f64vector "vector of 64-bit floats" ,f64vector-length ,f64vector-ref) ) )


(cond-expand
 [(not tinyclos) (set! describe-object (lambda (x y) #f))]
 [else] )

(define describe
  (let ([sprintf sprintf]
	[printf printf] 
	[length length]
	[list-ref list-ref]
	[string-ref string-ref]
	[describe-object describe-object]
	[hash-table-for-each hash-table-for-each] )
    (lambda (x . port)
      (let ([out (:optional port ##sys#standard-output)])

	(define (descseq name plen pref start)
	  (let ([len (fx- (plen x) start)])
	    (when name (fprintf out "~A of length ~S~%" name len))
	    (do ([i 0 (fx+ i 1)])
		((fx>= i len))
	      (fprintf out " ~S: ~S~%" i (pref x (fx+ start i))) ) ) )

	(when (##sys#permanent? x)
	  (fprintf out "statically allocated (~X) " (##sys#block-address x)) )
	(cond [(char? x)
	       (let ([code (char->integer x)])
		 (fprintf out "character ~S, code: ~S, #x~X, #o~O~%" x code code code) ) ]
	      [(eq? x #t) (fprintf out "boolean true~%")]
	      [(eq? x #f) (fprintf out "boolean false~%")]
	      [(null? x) (fprintf out "empty list~%")]
	      [(eof-object? x) (fprintf out "end-of-file~%")]
	      [(eq? (void) x) (fprintf out "unspecified object~%")]
	      [(fixnum? x)
	       (fprintf out "exact integer ~S, #x~X, #o~O" x x x)
	       (when (fx< x 256) (fprintf out ", character ~S" (integer->char x)))
	       (newline) ]
	      [(eq? x (##sys#slot '##sys#arbitrary-unbound-symbol 0))
	       (fprintf out "unbound value~%") ]
	      [(number? x) (fprintf out "inexact number ~S~%" x)]
	      [(string? x) (descseq "string" ##sys#size string-ref 0)]
	      [(vector? x) (descseq "vector" ##sys#size ##sys#slot 0)]
	      [(symbol? x)
	       (unless (##sys#symbol-has-toplevel-binding? x) (display "unbound " out))
	       (fprintf out "symbol with name ~S~%" (##sys#symbol->string x)) ]
	      [(list? x) (descseq "list" length list-ref 0)]
	      [(pair? x) (fprintf out "pair with car ~S and cdr ~S~%" (car x) (cdr x))]
	      [(procedure? x)
	       (let ([len (##sys#size x)])
		 (> len 3)
		 (if (eq? ##tinyclos#entity-tag (##sys#slot x (- len 1)))
		     (describe-object x out)
		     (fprintf out "procedure with code pointer ~X~%" (##sys#peek-unsigned-integer x 0)) ) ) ]
	      [(port? x)
	       (fprintf out
		"~a port with name ~S and file pointer ~X~%"
		(if (##sys#slot x 1) "output" "input")
		(##sys#slot x 3)
		(##sys#peek-unsigned-integer x 0) ) ]
	      [(cond-expand [tinyclos (instance? x)] [else #f]) (describe-object x out)]
	      [(##sys#pointer? x) (fprintf out "machine pointer ~X~%" (##sys#peek-unsigned-integer x 0))]
	      [(##sys#bytevector? x)
	       (let ([len (##sys#size x)])
		 (fprintf out "byte vector of size ~S:~%" len)
		 (##describe#hexdump x len ##sys#byte out) ) ]
	      [(##sys#structure? x 'hash-table)
	       (fprintf out "hash-table with ~S elements:~%" (##sys#slot x 2))
	       (hash-table-for-each
		(lambda (k v) (fprintf out " ~S\t-> ~S~%" k v))
		x) ]
	      [(##sys#structure? x 'environment)
	       (fprintf out "an evaluation environment~%") ]
	      [(##sys#generic-structure? x)
	       (let* ([st (##sys#slot x 0)]
		      [data (assq st ##describe#bytevector-data)] )
		 (if data
		     (apply descseq (append (cdr data) (list 0)))
		     (begin
		       (fprintf out "structure of type ~S:~%" (##sys#slot x 0))
		       (descseq #f ##sys#size ##sys#slot 1) ) ) ) ]
	      [else (fprintf out "unknown object~%")] )
	(void) ) ) ) )


;;; Display hexdump:

(define dump
  (let ([error error])
    (lambda (x . out)
      (let ([out (:optional out ##sys#standard-output)])
	(cond [(##sys#bytevector? x) (##describe#hexdump x (##sys#size x) ##sys#byte out)]
	      [(string? x) (##describe#hexdump x (##sys#size x) ##sys#byte out)]
	      [(and (##sys#generic-structure? x) (assq (##sys#slot x 0) ##describe#bytevector-data))
	       (let ([bv (##sys#slot x 1)])
		 (##describe#hexdump bv (##sys#size bv) ##sys#byte out) ) ]
	      [else (error "can not dump object" x)] ) ) ) ) )

(define ##describe#hexdump
  (let ([display display]
	[string-append string-append]
	[make-string make-string]
	[write-char write-char] )
    (lambda (bv len ref out)

      (define (justify n m base lead)
	(let* ([s (number->string n base)]
	       [len (##sys#size s)] )
	  (if (fx< len m)
	      (string-append (make-string (fx- m len) lead) s)
	      s) ) )

      (do ([a 0 (fx+ a 16)])
	  ((fx>= a len))
	(display (justify a 4 10 #\space) out)
	(write-char #\: out)
	(do ([j 0 (fx+ j 1)]
	     [a a (fx+ a 1)] )
	    ((or (fx>= j 16) (fx>= a len))
	     (and-let* ([(fx>= a len)]
			[o (fxmod len 16)]
			[(not (fx= o 0))] )
	       (do ([k (fx- 16 o) (fx- k 1)])
		   ((fx= k 0))
		 (display "   " out) ) ) )
	  (write-char #\space out)
	  (display (justify (ref bv a) 2 16 #\0) out) )
	(write-char #\space out)
	(do ([j 0 (fx+ j 1)]
	     [a a (fx+ a 1)] )
	    ((or (fx>= j 16) (fx>= a len)))
	  (let ([c (ref bv a)])
	    (if (and (fx>= c 32) (fx< c 128))
		(write-char (integer->char c) out)
		(write-char #\. out) ) ) ) 
	(write-char #\newline out) ) ) ) )


(cond-expand [(not use-modules)])

(define-module describe
  (unqualified)
  (export describe dump) )
