/*--------------------------------------------------------------
  
  File        : pde_heatflow2D.cpp

  Description : A simple Heat flow on 2D images

  Copyright  : David Tschumperle - http://www.greyc.ensicaen.fr/~dtschump/
   
  This software is governed by the CeCILL  license under French law and
  abiding by the rules of distribution of free software.  You can  use, 
  modify and/ or redistribute the software under the terms of the CeCILL
  license as circulated by CEA, CNRS and INRIA at the following URL
  "http://www.cecill.info". 
  
  As a counterpart to the access to the source code and  rights to copy,
  modify and redistribute granted by the license, users are provided only
  with a limited warranty  and the software's author,  the holder of the
  economic rights,  and the successive licensors  have only  limited
  liability. 
  
  In this respect, the user's attention is drawn to the risks associated
  with loading,  using,  modifying and/or developing or reproducing the
  software by the user in light of its specific status of free software,
  that may mean  that it is complicated to manipulate,  and  that  also
  therefore means  that it is reserved for developers  and  experienced
  professionals having in-depth computer knowledge. Users are therefore
  encouraged to load and test the software's suitability as regards their
  requirements in conditions enabling the security of their systems and/or 
  data to be ensured and,  more generally, to use and operate it in the 
  same conditions as regards security. 
  
  The fact that you are presently reading this means that you have had
  knowledge of the CeCILL license and that you accept its terms.
  
   -------------------------------------------------------------*/

#include "../CImg.h"

int main(int argc,char **argv) {
  cimg_usage("Perform a simple Heat Flow on 2D images");
  cimg_library::CImg<> img(cimg_option("-i","img/milla.ppm","Input image")), veloc(img,false);
  const double dt = cimg_option("-dt",1.0,"Adapting time step");
  img.
    noise(cimg_option("-ng",0.0,"Add gaussian noise"),0).
    noise(cimg_option("-nu",0.0,"Add uniform noise"),1).
    noise(cimg_option("-ns",0.0,"Add Salt&Pepper noise"),2);
  cimg_library::CImgDisplay profile(400,300,"Intensity Profile"), disp(img,"Heat flow 2D",0);
  disp.move((cimg_library::CImgDisplay::screen_dimx()-disp.width-profile.width)/2,
	    (cimg_library::CImgDisplay::screen_dimy()-disp.height)/2);

  profile.move(disp.window_x + 8 + disp.window_width, disp.window_y);
  CImg_3x3(I,float);
  float white[3]={255,255,255};

  for (int iter=0; !disp.closed && disp.key!=cimg_library::cimg::keyQ && disp.key!=cimg_library::cimg::keyESC; iter++) {
    cimg_mapV(img,k) cimg_map3x3(img,x,y,0,k,I) veloc(x,y,k) = Inc + Ipc + Icn + Icp - 4*Icc;
    const cimg_library::CImgStats stats(veloc,false);
    const double xdt = dt/(stats.max-stats.min);
    img += veloc*xdt;
    cimg_library::CImg<>(img).draw_text(2,2,white,NULL,1,"iter = %d",iter).display(disp).wait(25);
    if (disp.mouse_x>=0) {
      const int
        mx = disp.mouse_x, my = disp.mouse_y,
        mnx = mx*profile.dimx()/disp.dimx();
      const unsigned char red[3]={255,0,0}, green[3]={0,255,0}, blue[3]={0,0,255}, white[3]={255,255,255};
      cimg_library::CImg<unsigned char>(profile.dimx(),profile.dimy(),1,3,0).
        draw_graph(img.ref_line(my,0,0),red,0,255,0).
        draw_graph(img.ref_line(my,0,1),green,0,255,0).
        draw_graph(img.ref_line(my,0,2),blue,0,255,0).
        draw_line(mnx,0,mnx,profile.dimy()-1,white,cimg_library::cimg::rol(0xFF00FF00,iter%32),0.5f).
        draw_text(2,2,white,NULL,1,"(x,y)=(%d,%d)",mx,my).        
        display(profile);
    }
    profile.resize();
    disp.resize(disp);
  }

  return 0;
}
