#include "cp_types.h"
#include "cp_proto.h"

/* routines for random walk computations, tangency packings only. */

/* ========= euclidean functions =========== */

double G(double x,double y,double z)
     /* angle at x */
{
  double a;

  a=x*(x+y+z);
  return ( acos((a-y*z)/(a+y*z)) );
} /* G */

double G1(double x,double y,double z)
     /* angle ftn partial. First, convert arguments to s-radii. */
{
  double s;

  s=x+y+z;
  return ((-1.0)*(sqrt(x*y*z)*sqrt(s))*(1.0/s+1.0/x)/((x+y)*(x+z)) );
} /* G1 */

double G2(double x,double y,double z)
     /* angle ftn partial. First, convert arguments to s-radii. */
{
  return ( (sqrt(x*z)/(sqrt(y)*((x+y)*sqrt(x+y+z)))) );
} /* G2 */

double G3(double x,double y,double z)
     /* angle ftn partial. First, convert arguments to s-radii. */
{
  return ( (sqrt(x*y)/(sqrt(z)*((x+z)*sqrt(x+y+z)))) );
} /* G3 */

/* ========= hyperbolic functions ========== */

double g(double x,double y,double z)
     /* angle function. First, convert arguments to s-radii. */
{
  x=1.0/(x*x); y=1.0/(y*y); z=1.0/(z*z);
  return (acos( ((x*y+1.0)*(x*z+1.0)-2.0*x*(y*z+1.0)) / 	
		((x*y-1.0)*(x*z-1.0)) ));
} /* g */

double g1(double x,double y,double z)
     /* angle ftn partial. First, convert arguments to s-radii. */
{
  x=1.0/(x*x); y=1.0/(y*y); z=1.0/(z*z);
  return ( (-1.0)*(x*x*y*z-1.0)*sqrt((y-1.0)*(z-1.0)) / 
	   ((x*y-1.0)*(x*z-1.0)*sqrt(x*(x-1.0)*(x*y*z-1.0))) );
} /* g1 */


double g2(double x,double y,double z)
     /* angle ftn partial. First, convert arguments to s-radii. */
{
  x=1.0/(x*x); y=1.0/(y*y); z=1.0/(z*z);
  return (sqrt(x*(x-1.0)*(z-1.0)) / 
	  ((x*y-1.0)*sqrt((y-1.0)*(x*y*z-1.0))) );
} /* g2 */

double g3(double x,double y,double z)
     /* angle ftn partial. First, convert arguments to s-radii. */
{
  x=1.0/(x*x); y=1.0/(y*y); z=1.0/(z*z);
  return ( sqrt(x*(x-1.0)*(y-1.0)) / 
	   ((x*z-1.0)*sqrt((z-1.0)*(x*y*z-1.0))) );
} /* g3 */

double area_div(struct p_data *p,int v)
     /* deriv of hyp area. */
{
  int k;
  double accum=0.0;
  struct R_data *pR_ptr;
  struct K_data *pK_ptr;

  pR_ptr=p->packR_ptr;pK_ptr=p->packK_ptr;
  if (p->hes>=0 || v<1 || v>p->nodecount) return 0.0; 
  for (k=0;k<pK_ptr[v].num;k++)
    accum += Av(pR_ptr[v].rad,pR_ptr[pK_ptr[v].flower[k]].rad,
		pR_ptr[pK_ptr[v].flower[k+1]].rad);
  return accum;
} /* area_div */

double Av(double v,double u,double w)
     /* rate of change of area of triangle.*/
{
  return ((-1.0)*(g1(v,u,w)+g2(u,v,w)+g3(w,u,v)));
} /* Av */

double dtheta_dlog(struct p_data *p,int v)
     /* compute the derivative of the angle_sum theta at vertex v with 
respect to the log of the actual hyperbolic radius of v. */
{
  double vv,dtdv,h;

  vv=p->packR_ptr[v].rad;
  if (vv<=0) return 0.0;
  vv=1.0/(vv*vv);
  h=radius(p,v);
  dtdv=node_conductance(p,v)/((1.0-vv)*sqrt(vv));
  return (dtdv*2.0*h*exp(2.0*h));
} /* dtheta_dlog */

/* ============= computations ========== */

double twin_conductance(struct p_data *p,int v)
     /* conductance to ground in hyp case */
{
  double coef,vv;

  if (p->hes == 0 || p->hes >0) return 0; 
  if ((vv=p->packR_ptr[v].rad)<=okerr) return 0.0;
  vv=1.0/(vv*vv);
  coef=(vv-1.0)*sqrt(vv);
  return (area_div(p,v)*coef);
} /* twin_conductance */

double Cvw(double v,double u,double w,double a,int flag,int hes)
     /* conductance of edge from v to w. flag: 0=2 nbh, 1=only left 
nbh (a), 2=only right nbh (u). */
{
  double vv,coef;

  if (hes>0) return 0; /* spherical not yet done */
  if (hes<0)
    {
      if (v<=0 || w<=0) return 0;
      vv=1.0/(v*v); /* convert to s-radius */
      coef=(vv-1.0)*sqrt(vv);
      if (flag==1) return (g3(w,a,v)*coef);
      if (flag==2) return (g3(w,u,v)*coef);
      return ( (g3(w,u,v)+g3(w,a,v))*coef);
    }
  else 
    {
      coef=v;
      if (flag==1) return (G3(w,a,v)*coef);
      if (flag==2) return (G3(w,u,v)*coef);
      return ( (G3(w,u,v)+G3(w,a,v))*coef );
    }
} /* Cvw */

double node_conductance(struct p_data *p,int v)
     /* add up edge (and in hyp, twin) conductances */
{
  int k;
  double accum,coef,vv;
  struct R_data *pR_ptr;
  struct K_data *pK_ptr;

  pR_ptr=p->packR_ptr;pK_ptr=p->packK_ptr;
  if (p->hes>0) return 0.0;
  if (p->hes==0)
    {
      accum=0.0;
      for (k=1;k<pK_ptr[v].num;k++) /* middle faces */
	accum += Cvw(pR_ptr[v].rad,pR_ptr[pK_ptr[v].flower[k-1]].rad,
		     pR_ptr[pK_ptr[v].flower[k]].rad,
		     pR_ptr[pK_ptr[v].flower[k+1]].rad,0,p->hes);
      accum += Cvw(pR_ptr[v].rad,0.0,pR_ptr[pK_ptr[v].flower[0]].rad,
		   pR_ptr[pK_ptr[v].flower[1]].rad,1,p->hes); /* first face */
      accum += Cvw(pR_ptr[v].rad,
		   pR_ptr[pK_ptr[v].flower[pK_ptr[v].num-1]].rad,
		   pR_ptr[pK_ptr[v].flower[pK_ptr[v].num]].rad,
		   0.0,2,p->hes); /* last face */
      return accum;
    }
  if (p->hes<0)
    {
      vv=pR_ptr[v].rad;
      if (vv<okerr) return 0.0;
      vv=1.0/(vv*vv);
      coef=(vv-1.0)*sqrt(vv);
      accum=area_div(p,v)*coef;
      for (k=1;k<pK_ptr[v].num;k++) /* middle faces */
	accum += Cvw(pR_ptr[v].rad,pR_ptr[pK_ptr[v].flower[k-1]].rad,
		     pR_ptr[pK_ptr[v].flower[k]].rad,
		     pR_ptr[pK_ptr[v].flower[k+1]].rad,0,p->hes);
      accum += Cvw(pR_ptr[v].rad,0.0,pR_ptr[pK_ptr[v].flower[0]].rad,
		   pR_ptr[pK_ptr[v].flower[1]].rad,1,p->hes); /* first face */
      accum += Cvw(pR_ptr[v].rad,
		   pR_ptr[pK_ptr[v].flower[pK_ptr[v].num-1]].rad,
		   pR_ptr[pK_ptr[v].flower[pK_ptr[v].num]].rad,
		   0.0,2,p->hes); /* last face */
      return accum;
    }
  return 0.0;
} /* node_conductance */

double edge_conduct(struct p_data *p,int v,int indx)
     /* from v to indx neighbor */
{
  int w,left_index;
  double dum;
  struct R_data *pR_ptr;
  struct K_data *pK_ptr;

  pR_ptr=p->packR_ptr;pK_ptr=p->packK_ptr;
  w=pK_ptr[v].flower[indx];
  if (pK_ptr[v].bdry_flag && indx==0) /* bdry and upstream ngb */
    {
      dum=Cvw(pR_ptr[v].rad,0.0,
	      pR_ptr[w].rad,
	      pR_ptr[pK_ptr[v].flower[1]].rad,1,p->hes);
      return (dum);
    }
  if (pK_ptr[v].bdry_flag && indx==pK_ptr[v].num) /* bdry, down ngb */
    {
      dum=Cvw(pR_ptr[v].rad,
	      pR_ptr[pK_ptr[v].flower[pK_ptr[v].num-1]].rad,
	      pR_ptr[w].rad,0.0,2,p->hes);
      return (dum);
    }
  if (indx==0) left_index=pK_ptr[v].num-1;
  else left_index=indx-1;
  dum=Cvw(pR_ptr[v].rad,
	  pR_ptr[pK_ptr[v].flower[left_index]].rad,
	  pR_ptr[w].rad,
	  pR_ptr[pK_ptr[v].flower[indx+1]].rad,0,p->hes);
  return (dum);
} /* edge_conduct */

