;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: property-sheet.lisp,v 1.25 2002/04/04 22:01:42 apharris Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

(defun gradate-color (color)
  (flet ((extract-hex (offset)
           (parse-integer (subseq color offset (+ offset 2)) :radix 16))
         (insert-hex (rgb)
           (destructuring-bind (r g b)
               rgb
             (format nil "#~2x~2x~2x" r g b)))
         (scale-triple (pct r g b)
           (list (floor (* (/ pct 100.0) r))
                 (floor (* (/ pct 100.0) g))
                 (floor (* (/ pct 100.0) b)))))
    (let ((r (extract-hex 1))
          (g (extract-hex 3))
          (b (extract-hex 5)))
      (list color
            (insert-hex (scale-triple 80 r g b))
            (insert-hex (scale-triple 50 r g b))))))


(defun hairline-row (stream columns &optional (color +dark-gray+))
  (declare (optimize (speed 3)))
  (html-stream
   stream
   ((:tr :bgcolor color)
    ((:td :colspan columns)
     (write-spacer stream)))))

;; ------------------------------------------------------------
;; property-sheet
;;
;; for display of linked sets of editable properties

(defclass property-sheet (repeater)
  ((title        :initarg :title
                 :initform nil)
   (color        :initarg :color
                 :initform "#ffffff")
   (border       :initarg :border
                 :initform 0)
   (result       :initarg :save-page
                 :initform nil)
   (property-set :initarg :property-set
                 :initform nil)
   (display-set  :initarg :display-set
                 :initform :standard)
   (control-set  :initarg :control-set
                 :initform '(:editing))
   (edit-labels  :initarg :labels
                 :initform t)
   (edit-state   :initarg :edit-state
                 :initform nil))
  (:documentation
   "manages display and edit of a property set"))

(defmethod render-html ((self property-sheet) stream)
  (with-slots (title color border control-set property-set
                           display-set edit-labels edit-state)
    self
    (let* ((current-fields (object-property-sheet-set (element-value self) property-set display-set))
           (all-fields     (object-property-sheet-set (element-value self) property-set :all))
           (chopped-fields (break-list current-fields :br))
           (column-count-1 (length chopped-fields))
           (column-count (* 3 column-count-1)))
      (html-stream
       stream
       ((:table :width "100%" :bgcolor +mid-gray+ :border 0 :cellpadding 0 :cellspacing 0)
        (when (or control-set title)
          (hairline-row stream column-count)
          (html-stream
           stream
           (:tr
            ((:td :align :left :valign :top :colspan column-count)
             ((:table :width "100%" :bgcolor +header-bg+ :border 0 :cellspacing 0 :cellpadding 0)
              (:tr
               (:td
                ((:table :width "100%" :border 0 :cellspacing 0 :cellpadding 2)
                 (:tr
                  ((:td :nowrap t :align :left :valign :middle :class "list_header")
                   "&nbsp;&nbsp;&nbsp;&nbsp;"
                   ((:font :color "#000000")
                    (:princ (or title "&nbsp;"))))
                  ((:td :nowrap t :align :right :valign :middle :class "list_header")
                   "&nbsp;"
                   (when control-set
                     (when (< (length current-fields) (length all-fields))
                       (if (eql display-set :all)
                           (with-action (stream self all-fields "STANDARD")
                             (write-string "Show&nbsp;standard&nbsp;fields..." stream))
                           (with-action (stream self all-fields "ALL")
                             (write-string "Show&nbsp;all&nbsp;fields..." stream)))
                       (write-string "&nbsp;&nbsp;|&nbsp;&nbsp;" stream))
                     (with-action (stream self start-editing "ALL")
                       (write-string "Edit&nbsp;all&nbsp;fields..." stream)))))))))))))
        (hairline-row stream column-count)
        (dotimes (y (apply #'max (mapcar #'length chopped-fields)))
          (html-stream
           stream
           (:tr
            (dotimes (x (length chopped-fields))
              (let ((property (nth y (nth x chopped-fields)))) 
                (destructuring-bind (property caption value type editable)
                    (or property '(nil "&nbsp;" nil nil nil))
                  (html-stream
                   stream
                   ((:td :nowrap t :valign :top :align :right)
                    "&nbsp;&nbsp;"
                    (:princ (gettext caption))
                    "&nbsp;&nbsp;")
                   ((:td :valign :middle :bgcolor :white)
                    "&nbsp;&nbsp;"
                    (cond ((and editable (editing? self property))
                           (property-sheet-edit type self property value stream))
                          (value
                           (property-sheet-display type self value stream))
                          ((null type)
                           (write-string "&nbsp;" stream))
                          (t
                           (property-sheet-display-null type self stream))))
                   ((:td :width 1 :valign :bottom :align :right)
                    "&nbsp;"
                    (if (and editable edit-labels (not (editing? self property)))
                        (with-action (stream self start-editing (string property))
                          (write-string "Edit..." stream))
                        (write-string "&nbsp;" stream))
                    "&nbsp;"))))))))
        (when (and edit-state
                   (child-element self :save-button :if-does-not-exist nil))
          (hairline-row stream column-count)
          (html-stream
           stream
           (:tr
            ((:td :colspan column-count)
             ((:table :width "100%" :cellpadding 4 :cellspacing 0 :border 0)
              (:tr
               ((:td :valign :middle :align :left)
                (render-child self :cancel-button stream)
                "&nbsp;"
                (render-child self :save-button stream)))))))))))))


(defmethod editing? ((self property-sheet) property)
  (with-slots (edit-state)
    self
    (or (eql edit-state t)
        (member property edit-state))))

;; WMs

(define-wm start-editing ((self property-sheet) (field keyword))
  (with-slots (edit-state)
    self
    (case field
      (:all
       (setf edit-state t))
      (:none
       (setf edit-state nil))
      (t
       (when (eql edit-state t)
         (setf edit-state nil))
       (if (member field edit-state)
           (setf edit-state (remove field edit-state))
           (pushnew field edit-state)))))
  nil)

(define-wm all-fields ((self property-sheet) (display-set keyword))
  (setf (slot-value self 'display-set) display-set)
  nil)

(defmethod edited-property-value ((self property-sheet) property)
  (if-bind (element (gethash property (slot-value self 'imho::children)))
      (values (element-value element) t)
      (values nil nil)))
             
;; default implementations

(defmethod property-sheet-edit ((self t) element property property-value stream)
  (if-bind (editor-class (property-sheet-editor self))
      (let ((value (edited-property-value element property)))
        (imho::repeat-element-2 element stream property value editor-class
                                property-value))
      (format stream "[EDIT ~s / ~s]" self property-value)))

(defmethod property-sheet-editor ((self t))
  nil)

(defmethod property-sheet-display ((self t) element value stream)
  (declare (ignore element))
  (format stream "[~s / ~s]" self value))

(defmethod property-sheet-display-null ((self t) element stream)
  (declare (ignore element))
  (format stream "[~s / NULL]" self))

;; string

(defmethod property-sheet-editor ((self (eql 'string)))
  'text-field)

(defmethod property-sheet-display ((self (eql 'string)) element value stream)
  (declare (ignore element))
  (if (string-empty-p value)
      (write-space stream)
    (write-string value stream)))

(defmethod property-sheet-display-null ((self (eql 'string)) element stream)
  (declare (ignore element))
  (write-string "<i>None</i>" stream))

;; boolean

(defmethod property-sheet-editor ((self (eql 'boolean)))
  'checkbox)

(defmethod property-sheet-display ((self (eql 'boolean)) element value stream)
  (declare (ignore element value))
  (write-string "Yes" stream))

(defmethod property-sheet-display-null ((self (eql 'boolean)) element stream)
  (declare (ignore element))
  (write-string "No" stream))

;; integer

(defmethod property-sheet-display ((self (eql 'integer)) element value stream)
  (declare (ignore element))
  (format stream "~A" value))

(defmethod property-sheet-display-null ((self (eql 'integer)) element stream)
  (declare (ignore element))
  (write-string "<i>None</i>" stream))


