;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: request.lisp,v 1.48 2002/03/29 20:36:05 craig Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

;; Read form encoded content

;; Turns "foo=bar&baz=qux" into (("foo" "bar")("baz" "qux"))

(defun url-unescape (string)
  (let ((dest (make-string (length string))))
    (do ((i 0 (incf i))
         (j 0 (incf j)))
        ((= i (length string)) (subseq dest 0 j))
      (cond ((equal (aref string i) #\%)
             (incf i)
             (setf (aref dest j)
                   (code-char
                    (parse-integer string :start i :end (+ i 2) :radix 16)))
             (incf i))
            ((equal (aref string i) #\+)
             (setf (aref dest j)
                   #\Space))
            (t
             (setf (aref dest j)
                   (aref string i)))))))

(defun read-form-encoded (string)
  ;; (print string)
  (mapcar (lambda (pair)
            (let ((pair (split pair #\=)))
              (list (car pair)
                    (url-unescape (cadr pair)))))
          (split string #\&)))

(defun get-header-value (request key)
  (cdr (assoc key (request-headers-in request))))

(defun get-cookie (key)
  (labels ((parse-cookies ()
             (when-bind (header (get-header-value *active-request* :cookie))
                (setf (request-cookies-in *active-request*) (split-cookies header))))
           (split-cookies (string)
             ;; NAME1=OPAQUE_STRING1; NAME2=OPAQUE_STRING2
             (mapcar (lambda (pair)
                       (let ((pair (split (string-trim " " pair) #\=)))
                         (cons (car pair) (cadr pair))))
                     (split string #\;))))
    (let ((cookies (request-cookies-in *active-request*)))
      (cond
        (cookies
         (cdr (assoc key cookies :test #'string=)))
        (t
         (when (parse-cookies)
           (get-cookie key)))))))

(defun parse-content (request)
  (when-bind (content (request-client-content request))
    (let ((type (get-header-value request :content-type)))
      (ecase (car type)
        (:multipart/form-data
         (let* ((boundary (concatenate 'string "--"
                                       (cdr (assoc :boundary (cdr type)))))
                (content  (read-multipart boundary content)))
	   
           (setf (request-client-content request) nil)
           (dolist (section content)
             (destructuring-bind (headers . content)
                 section
               (let* ((dis (cddr (assoc :content-disposition headers)))
                      (type (cadr (assoc :content-type headers)))
                      (name (unquote (cdr (assoc :name dis))))
                      (filename (cdr (assoc :filename dis))))
		 (cond
                  ((and type
                        (or (string-equal type "image" :end1 5) ; Added
                            (string-equal type "application" :end1 11)))
                   ;; Normally the request-client-content only contains the
                   ;; element-external-name and the data, but I have added
                   ;; two a-list fields as well: :FILENAME and :TYPE
                   ;; so that these can be retrieved by the form web-method
                   (push (nconc (list name content)
                                (when filename
                                  (list (list :filename (unquote filename))))
                                (when type
                                  (list (list :type type))))
                         (request-client-content request)))
                  (t
                   ;; Same here
                   (push (nconc (list name (array-to-string content))
                                (when filename
                                  (list (list :filename (unquote filename))))
                                (when type
                                  (list (list :type type))))
                         (request-client-content request)))))))))))))

(defun unquote (string)
  (subseq string 1 (1- (length string))))

(defun parse-k-v-pair (string)
  (setq string (string-trim " 	" string))
  (split-up-trim string #\=))
  
(defun parse-content-type-header (string)
  (destructuring-bind (type &rest rest)
      (split string #\;)
    (setq type (make-keyword type))
    (setq rest (remove-if (lambda (x) (= 0 (length x))) rest))
    (setq rest (mapcar #'parse-k-v-pair rest))
    (cons type rest)))

(defun chop-array (seq1 seq2)
  (let* ((length (length seq1))
         (results nil)
         (secs nil)
         (offset 0))
    (loop
     (let ((i (search seq1 seq2 :start2 offset)))
       (if i
           (push i results)
           (return))
       (setf offset (1+ i))))
    (do ((res (reverse results) (cdr res))
          (prev nil (car res)))
         ((null res)
          (push (subseq seq2 (if prev (+ length prev) 0)) secs))
      (push (subseq seq2 (if prev (+ length prev) 0) (car res))
            secs))
    (reverse secs)))
  
(defun read-multipart (boundary content &aux results)
  (let ((boundary (string-to-array boundary)))
    (setq results (chop-array boundary content))
    (mapcar #'read-section (subseq results 1 (- (length results) 1)))))

;;; Altered to allow the insertion of multiple characters in between
;;; each element of the list

(defun %string-join (chars list &aux value)
  (multiple-value-bind
        (chs len)
      (typecase chars
        (base-char (values (make-array 1 :initial-element (char-code chars)) 1))
        (vector (values chars (length chars))))
    (case (length list)
      (1
       (setf value (car list)))
      (t
       (setf value #())
       (dolist (cont list)
         (setf value
               (concatenate 'vector value cont chs)))
       (setf value (subseq value
                           0 (- (length value)
                                len))))))
  value)

(defun read-section (section)
  (let ((smegs (chop-array #(13 10) section)))
    (if (= 0 (length (car smegs)))
        (setq smegs (cdr smegs)))
    (if (= 0 (length (car (last smegs))))
        (setq smegs (subseq smegs 0 (1- (length smegs)))))
    (let* ((end (position-if (lambda (x) (= 0 (length x))) smegs))
           (headers (mapcar #'read-section-header
                            (mapcar #'array-to-string
                                    (subseq smegs 0 end))))
           (contents (subseq smegs (1+ end))))
      ;; If we remove #(13 10) we'd better put it back as #(13 10),
      ;; unless we can demonstrate that the contents are text.
      ;; Probably doable, but this works for now.
      (cons headers (%string-join #(13 10) contents)))))

(defun split-up-trim (header char)
  (let ((pos (position char header)))
    (let ((key (intern (string-upcase (subseq header 0 pos)) :keyword)))
      (cons key (string-trim " " (subseq header (1+ pos)))))))

(defun split-up-header (header)
  (let ((pos (position #\: header)))
    (let ((key (intern (string-upcase (subseq header 0 pos)) :keyword)))
      (cons key (subseq header (1+ pos))))
    pos))

(defun read-section-header (section-header)
  (destructuring-bind (key . value)
      (split-up-trim section-header #\:)
    (case key
      (:content-disposition
       (cons key
             (imho::parse-content-type-header value)))
      (t
       (list key value)))))

(defun find-application (name)
  (gethash name *imho-active-apps*))

(defun client-ip ()
  (when-bind (req *active-request*)
    (if-bind (remote-ip (or
                           (cdr (assoc :remote-ip-addr (request-headers-in req)))
                           (cdr (assoc :remotehost (request-headers-in req)))))
      (when-bind (hostent (ext:lookup-host-entry remote-ip))
        (car (ext:host-entry-addr-list hostent)))
      nil)))

