;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: iso-8601.lisp,v 1.3 2003/03/24 21:56:16 adam Exp $
;;;
;;; Copyright (c) 2000 - 2003 onShore Development, Inc.

(in-package :odcl)

(define-condition iso-8601-syntax-error (error)
  ((bad-component;; year, month whatever
    :initarg :bad-component
    :reader bad-component)))

(defun parse-timestring (timestring &key (start 0) end junk-allowed)
  "parse a timestring and return the corresponding wall-time.  If the
timestring starts with P, read a duration; otherwise read an ISO 8601
formatted date string."
  (declare (ignore junk-allowed))  ;; FIXME
  (let ((string (subseq timestring start end)))
    (if (char= (aref string 0) #\P)
        (parse-iso-8601-duration string)
        (parse-iso-8601-time string))))

(defvar *iso-8601-duration-delimiters*
  '((#\D . :days)
    (#\H . :hours)
    (#\M . :minutes)
    (#\S . :seconds)))

(defun iso-8601-delimiter (elt)
  (cdr (assoc elt *iso-8601-duration-delimiters*)))

(defun iso-8601-duration-subseq (string start)
  (let* ((pos (position-if #'iso-8601-delimiter string :start start))
	 (number (when pos (parse-integer (subseq string start pos) :junk-allowed t))))
    (when number
      (values number
	      (1+ pos)
	      (iso-8601-delimiter (aref string pos))))))

(defun parse-iso-8601-duration (string)
  "return a wall-time from a duration string"
  (block parse
    (let ((days 0) (secs 0) (hours 0) (minutes 0) (index 1))
      (loop
       (multiple-value-bind (duration next-index duration-type)
           (iso-8601-duration-subseq string index)
         (case duration-type
           (:hours
            (incf hours duration))
           (:minutes
            (incf minutes duration))
           (:seconds
            (incf secs duration))
           (:days
            (incf days duration))
           (t
            (return-from parse (make-duration :day days :hour hours :minute minutes :second secs))))
         (setq index next-index))))))

(defun syntax-parse-iso-8601 (string)
  (let (year month day hour minute second gmt-sec-offset)
    (handler-case
        (progn
          (setf year   (parse-integer (subseq string 0 4))
                month  (parse-integer (subseq string 5 7))
                day    (parse-integer (subseq string 8 10))
                hour   (if (<= 13 (length string))
                           (parse-integer (subseq string 11 13))
                           0)
                minute (if (<= 16 (length string))
                           (parse-integer (subseq string 14 16))
                           0)
                second (if (<= 19 (length string))
                           (parse-integer (subseq string 17 19))
                           0)
                gmt-sec-offset (if (<= 22 (length string))
                                   (* 60 60 (parse-integer (subseq string 19 22)))
                                   0))
          (unless (< 0 year)
            (error 'iso-8601-syntax-error
                   :bad-component '(year . 0)))
          (unless (< 0 month)
            (error 'iso-8601-syntax-error
                   :bad-component '(month . 0)))
          (unless (< 0 day)
            (error 'iso-8601-syntax-error
                   :bad-component '(month . 0)))
          (values year month day hour minute second gmt-sec-offset))
      (simple-error ()
        (error 'iso-8601-syntax-error
               :bad-component
               (car (find-if (lambda (pair) (null (cdr pair)))
                             `((year . ,year) (month . ,month)
                               (day . ,day) (hour ,hour)
                               (minute ,minute) (second ,second)
                               (timezone ,gmt-sec-offset)))))))))

;; e.g. 2000-11-11 00:00:00-06

(defun parse-iso-8601-time (string)
  "return the wall-time corresponding to the given ISO 8601 datestring"
  (multiple-value-bind (year month day hour minute second offset)
      (syntax-parse-iso-8601 string)
    (make-time :year year
               :month month
               :day day
               :hour hour
               :minute minute
               :second second
               :offset offset)))
