;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;; $Id: threaded-tree.lisp,v 1.3 2003/03/24 21:51:31 adam Exp $
;;
;;; Copyright (c) 2002 - 2003 onShore Development, Inc.

;; Rudiments of a threaded AVL tree
;;
;; This is very horked - do not use (unless you can fix it yessef)
;;
;; Knuth ACP 6.2.3, Algorithm A

(in-package :odcl)

(defstruct (tavl-node (:conc-name tavl/)
                              (:constructor tavl/create))
  left                                  ; subtree L
  right                                 ; subtree R
  (left-tag t)                          ; tag L - thread me
  (right-tag t)                         ; tag R - thread me
  (balance 0 :type fixnum)              ; balance factor
  data)                                 ; data

(defclass tavl-tree (avl-tree)
  ()
  (:default-initargs :head (tavl/create)))

(defmethod tree-root ((self tavl-tree))
  (tavl/right (tree-head self)))

(defun tavl/link (dir node)
  (ecase dir
    (-1 (unless (tavl/left-tag node)
          (tavl/left node)))
    (1  (unless (tavl/right-tag node)
          (tavl/right node)))))

(defsetf tavl/link (dir node) (snode)
  `(ecase ,dir
    (-1 (setf (tavl/left ,node) ,snode
         (tavl/left-tag ,node) nil))
    (1  (setf (tavl/right ,node) ,snode
         (tavl/right-tag ,node) nil))))

(defun tavl/mapc (map-function node &aux stack (go-left t))
  "stack based iteration until threads are complete"
  (while node
    (if (and go-left (tavl/left node))
        (progn
          (push node stack)
          (setq node (tavl/left node)))
        (progn
          (funcall map-function node)
          (if (tavl/right node)
              (setq node (tavl/right node)
                    go-left t)
              (setq node (pop stack)
                    go-left nil))))))

(defun tavl/node-copy (root)
  (when root
    (tavl/create :left (tavl/node-copy (tavl/left root))
                :right (tavl/node-copy (tavl/right root))
                :data (tavl/data root)
                :balance (tavl/balance root))))

;; public

(defmethod tree-member ((self avl-tree) data &aux found)
  (let ((node (tree-root self))
        (test (tree-test self)))
    (while (and node (not found))
      (cond ((funcall test data (tavl/data node))
             (setq node (tavl/left node)))
            ((funcall test (tavl/data node) data)
             (setq node (tavl/right node)))
            (t 
             (setq found t))))
    (when node
      (tavl/data node))))

(defmethod tree-map ((fn function) (self avl-tree))
  (tavl/mapc (lambda (node)
              (setf (tavl/data node) (funcall fn (tavl/data node))))
            (tree-root self)))

(defmethod tree-first ((self tavl-tree))
  (when-bind (node (tree-root self))
    (while (tavl/left node)
      (setq node (tavl/left node)))
    (tavl/data node)))

(defmethod tree-last ((self tavl-tree))
  (when-bind (node (tree-root self))
    (while (tavl/right node)
      (setq node (tavl/right node)))
    (tavl/data node)))

(defmethod tree->list ((self tavl-tree) &aux list)
  (tavl/mapc (lambda (node)
               (push (tavl/data node) list))
             (tree-root self))
  (nreverse list))

(defmethod clear ((self tavl-tree))
  (setf (tavl/right (tree-root self)) nil))

;; interface

(defmethod c-add ((tree tavl-tree) item &aux result-data)
  (assert (not (null item)))
  (flet ((merge (data &optional existing-node)
           (unless existing-node
             (setf existing-node (tavl/create)))
           (let ((data (funcall (tree-replace tree)
                                (tavl/data existing-node)
                                data)))
             (setf (tavl/data existing-node) data
                   result-data data)
             existing-node)))
    ;; A1 [Initialize.]
    (let ((test (tree-test tree))
          (_t (tree-head tree))
          (s (tavl/link 1 (tree-head tree)))
          (p (tavl/link 1 (tree-head tree)))
          (q nil)
          (r nil))
      (unless p
        (setf (tavl/link 1 (tree-head tree)) (merge item))
        (return-from c-add result-data))
      (loop
       ;; A2 [Compare.]
       (cond ((funcall test item (tavl/data p))
              ;; A3 [Move left.]
              (setq q (tavl/link -1 p))
              (when (null q)
                (setf q (merge item)
                      (tavl/left p) q)
                (return))
              (unless (= 0 (tavl/balance q))
                (setf _t p
                      s q))
              (setf p q))
             ((funcall test (tavl/data p) item)
              ;; A4 [Move right.]
              (setq q (tavl/link 1 p))
              (when (null q)
                (setf q (merge item)
                      (tavl/link 1 p) q)
                (return))
              (unless (= 0 (tavl/balance q))
                (setf _t p
                      s q))
              (setf p q))
             (t
              (merge item p)
              (return-from c-add result-data))))
      ;; A5 [Insert.] (initialization done above)
      ;; A6 [Adjust balance factors.]
      (let ((a (if (funcall test item (tavl/data s)) -1 1)))
        (setf p (tavl/link a s)
              r p)
        (while (not (eql p q))
          (cond ((funcall test item (tavl/data p))
                 (setf (tavl/balance p) -1
                       p (tavl/left p)))
                ((funcall test (tavl/data p) item)
                 (setf (tavl/balance p) 1
                       p (tavl/right p)))
                (t
                 (error "logic error"))))
        ;; A7 [Balancing act.]
        (cond ((= (tavl/balance s) 0)
               ;; i)
               (setf (tavl/balance s) a)
               (incf (slot-value tree 'height))
               (return-from c-add result-data))
              ((= (tavl/balance s) (- a))
               ;; ii)
               (setf (tavl/balance s) 0)
               (return-from c-add result-data))
              ((= (tavl/balance s) a)
               (cond ((= (tavl/balance r) a)
                      ;; A8 [Single rotation.]
                      (setf p r
                            (tavl/link a s) (tavl/link (- a) r)
                            (tavl/link (- a) r) s
                            (tavl/balance s) 0
                            (tavl/balance r) 0))
                     ((= (tavl/balance r) (- a))
                      ;; A9 [Double rotation.]
                      (setf p (tavl/link (- a) r)
                            (tavl/link (- a) r) (tavl/link a p)
                            (tavl/link a p) r
                            (tavl/link a s) (tavl/link (- a) p)
                            (tavl/link (- a) p) s)
                      (cond ((= a (tavl/balance p))
                             (setf (tavl/balance s) (- a))
                             (setf (tavl/balance r) 0))
                            ((= 0 (tavl/balance p))
                             (setf (tavl/balance s) 0)
                             (setf (tavl/balance r) 0))
                            ((= (- a) (tavl/balance p))
                             (setf (tavl/balance s) 0)
                             (setf (tavl/balance r) a))))
                     (t
                      (error "logic error")))
               (setf (tavl/balance p) 0)
               (if (eql s (tavl/right _t))
                   (setf (tavl/right _t) p)
                   (setf (tavl/left _t) p)))
              (t
               (error "logic error")))))))

(defun tavl/del-l (node branch)
  (let ((br (tavl/link branch node)))
    (ecase (tavl/balance br)
      (1  (decf (tavl/balance br)) t)
      (0  (decf (tavl/balance br)) nil)
      (-1 (let ((br-l (tavl/left br)))
            (ecase (tavl/balance br-l)
              (-1 (setf (tavl/left br) (tavl/right br-l)
                        (tavl/right br-l) br
                        (tavl/balance br) 0
                        (tavl/balance br-l) 0
                        (tavl/link branch node) br-l)
                  t)
              (0  (setf (tavl/left br) (tavl/right br-l)
                        (tavl/right br-l) br
                        (tavl/balance br) -1
                        (tavl/balance br-l) 1
                        (tavl/link branch node) br-l)
                  nil)
              (1  (let ((br-l-r (tavl/right br-l)))
                    (setf (tavl/right br-l) (tavl/left br-l-r)
                          (tavl/left br-l-r) br-l
                          (tavl/left br) (tavl/right br-l-r)
                          (tavl/right br-l-r) br)
                    (case (tavl/balance br-l-r)
                      (-1 (setf (tavl/balance br) 1
                                (tavl/balance br-l) 0))
                      (0  (setf (tavl/balance br) 0
                                (tavl/balance br-l) 0))
                      (1  (setf (tavl/balance br) 0
                                (tavl/balance br-l) -1)))
                    (setf (tavl/link branch node) br-l-r
                          (tavl/balance br-l-r) 0)
                    t))))))))

(defun tavl/del-r (node branch)
  (let ((br (tavl/link branch node)))
    (ecase (tavl/balance br)
      (-1 (incf (tavl/balance br)) t)
      (0  (incf (tavl/balance br)) nil)
      (1  (let ((br-r (tavl/right br)))
            (ecase (tavl/balance br-r)
              (1  (setf (tavl/right br) (tavl/left br-r)
                        (tavl/left br-r) br
                        (tavl/balance br) 0
                        (tavl/balance br-r) 0
                        (tavl/link branch node) br-r)
                  t)
              (0  (setf (tavl/right br) (tavl/left br-r)
                        (tavl/left br-r) br
                        (tavl/balance br) 1
                        (tavl/balance br-r) -1
                        (tavl/link branch node) br-r)
                  nil)
              (-1 (let ((br-r-l (tavl/left br-r)))
                    (setf (tavl/left br-r) (tavl/right br-r-l)
                          (tavl/right br-r-l) br-r
                          (tavl/right br) (tavl/left br-r-l)
                          (tavl/left br-r-l) br)
                    (case (tavl/balance br-r-l)
                      (-1 (setf (tavl/balance br) 0
                                (tavl/balance br-r) 1))
                      (0  (setf (tavl/balance br) 0
                                (tavl/balance br-r) 0))
                      (1  (setf (tavl/balance br) -1
                                (tavl/balance br-r) 0)))
                    (setf (tavl/link branch node) br-r-l
                          (tavl/balance br-r-l) 0)
                    t))))))))

(defmethod c-remove ((self tavl-tree) item)
  (with-slots (test)
    self
    (labels ((%%tree-delete (node branch q)
               (let ((br (tavl/link branch node)))
                 (if (tavl/right br)
                     (when (%%tree-delete br 1 q)
                       (tavl/del-l node branch))
                     (progn
                       (setf (tavl/data q) (tavl/data br)
                             (tavl/link branch node) (tavl/left br))
                       t))))
             (%tree-delete (node branch)
               (when-bind (br (tavl/link branch node))
                 (cond ((funcall test item (tavl/data br))
                        (when (%tree-delete br -1)
                          (tavl/del-r node branch)))
                       ((funcall test (tavl/data br) item)
                        (when (%tree-delete br 1)
                          (tavl/del-l node branch)))
                       (t
                        (cond ((null (tavl/right br))
                               (setf (tavl/link branch node) (tavl/left br))
                               t)
                              ((null (tavl/left br))
                               (setf (tavl/link branch node) (tavl/right br))
                               t)
                              (t
                               (when (%%tree-delete br -1 br)
                                 (tavl/del-r node branch)))))))))
      ;; delete to the right
      (%tree-delete (slot-value self 'head) 1))))

(defmethod c-copy ((self tavl-tree))
  (with-slots (head test)
    self
    (make-instance 'avl-tree
                   :test test
                   :head (tavl/node-copy head))))
