;;; -*- Mode: Lisp -*-
;;;; MaiSQL --- Common Lisp Interface Layer to SQL Databases
;;;; This is copyrighted software.  See documentation for terms.
;;;; 
;;;; recording.lisp --- SQL broadcast streams
;;;; 
;;;; Checkout Tag: $Name:  $
;;;; $Id: recording.lisp,v 1.10 2001/08/29 16:46:16 craig Exp $

(in-package :MAISQL-SYS)

(defun start-sql-recording (&key (type :command) (database *default-database*))
  "Begin recording SQL command or result traffic onto stream. By
default the broadcast stream is just *standard-output*.  TYPE
determines whether SQL command or result traffic is recorded, or both.
It must be either :command, :result or :all, and defaults to :command.
DATABASE defaults to *default-database*"
  (if (or (equal type :all) (equal type :command))
      (setf (command-recording-stream database)
	    (make-broadcast-stream
	     *standard-output*)))
  (if (or (equal type :all) (equal type :result))
      (setf (result-recording-stream database)
	    (make-broadcast-stream
	     *standard-output*))))

(defun stop-sql-recording (&key (type :command) (database *default-database*))
  "Stops recording of SQL command or result traffic.  TYPE determines
whether to stop SQL command or result traffic, or both.  It must be
either :command, :result or :all, defaulting to :command.  DATABASE
defaults to *default-database*"
  (if (or (equal type :all)
	  (equal type :command))
      (setf (command-recording-stream database) nil))
  (if (or (equal type :all)
	  (equal type :result))
      (setf (result-recording-stream database) nil)))


(defun sql-recording-p (&key (type :command) (database *default-database*))
  "Returns t if recording of TYPE of SQL interaction specified is
enabled.  TYPE must be either :command, :result, :all or :any.
DATABASE defaults to *default-database*."
  (if (or (and (equal type :command)
	       (command-recording-stream database))
	  (and (equal type :result)
	       (result-recording-stream database))
	  (and (equal type :all)
	       (result-recording-stream database)
	       (command-recording-stream database))
	  (and (equal type :any)
	       (or (result-recording-stream database)
		   (command-recording-stream database))))
      t
    nil))


(defun add-sql-stream (stream &key (type :command)
			      (database *default-database*))
  "Add the given STREAM as a component stream for the recording
broadcast stream for the given SQL interaction TYPE.  TYPE must be
either :command, :result, or :all, defaulting to :command.  DATABASE
defaults to *default-database*"
  (if (or (equal type :all) (equal type :command))
      (unless (member stream (list-sql-streams :type :command :database database))
        (and (setf (command-recording-stream database)
                   (apply #'make-broadcast-stream
                          (cons stream (list-sql-streams :type :command
                                                         :database database))))
             stream)))
  (if (or (equal type :all) (equal type :result))
      (unless (member stream (list-sql-streams :type :result :database database))
        (and (setf (result-recording-stream database)
              (apply #'make-broadcast-stream
                     (cons stream (list-sql-streams :type :result
                                                    :database database))))
             stream))))
			      

(defun delete-sql-stream (stream &key (type :command)
				 (database *default-database*))
  "Removes the given STREAM from the recording broadcast stream for the given TYPE of SQL interaction.  TYPE must be either :command, :result, or :all, defaulting to :command.  DATABASE defaults to *default-database*"
  (if (or (equal type :all) (equal type :command))
      (setf (command-recording-stream database)
	    (apply #'make-broadcast-stream
		   (remove stream (list-sql-streams :type :command
						    :database database)))))
  (if (or (equal type :all) (equal type :result))
      (setf (result-recording-stream database)
	    (apply #'make-broadcast-stream
		   (remove stream (list-sql-streams :type :result
						    :database database))))))


(defun list-sql-streams (&key (type :command) (database *default-database*))
  "Returns the set of streams which the recording broadcast stream
send SQL interactions of the given TYPE sends data."
  (cond
   ((equal type :command)
    (when (command-recording-stream database)
      (broadcast-stream-streams (command-recording-stream database))))
   ((equal type :result)
    (when (result-recording-stream database)
      (broadcast-stream-streams (result-recording-stream database))))
   (t
    (error "Unknown recording type. ~A" type))))

(defun record-sql-command (expr database)
  (if database
      (with-slots (command-recording-stream)
		  database
		  (if command-recording-stream 
		      (format command-recording-stream ";; ~A ~A => ~A~%"
                              #-cmu (princ-to-string (get-universal-time))
			      #+cmu
                              (ext:format-universal-time nil (get-universal-time)
							 :print-timezone nil
							 :print-weekday nil)
			      (database-name database)
			      expr)))))

(defun record-sql-result (res database)
  (if database
      (with-slots (result-recording-stream)
		  database
		  (if result-recording-stream 
		      (format result-recording-stream ";; ~A ~A <= ~A~%"
                              #-cmu (princ-to-string (get-universal-time))
			      #+cmu
                              (ext:format-universal-time nil (get-universal-time)
							 :print-timezone nil
							 :print-weekday nil)
			      (database-name database)
			      res)))))
  
  
(defmethod query ((query-expression string)
		  &key (database *default-database*))
  (record-sql-command query-expression database)
  (let ((res (database-query query-expression database)))
    (record-sql-result res database)
    res))

(defmethod execute-command ((sql-expression string)
			    &key (database *default-database*))
  "Execute the SQL command expression sql-expression on the given database.
Returns true on success or nil on failure."
  (record-sql-command sql-expression database)
  (let ((res (database-execute-command sql-expression database)))
    (record-sql-result res database)
    res))
  
