;; -*- Mode: Lisp -*-
;; $Id: syntax.lisp,v 1.13 2000/11/16 20:16:12 craig Exp $
;; Square bracket reader macro for precompilation of sql syntax

(in-package :maisql-sys)

(eval-when (:compile-toplevel :load-toplevel)
  
(defvar *original-reader-enter* nil)

(defvar *original-reader-exit* nil)

(defvar *sql-macro-open-char* #\[)

(defvar *sql-macro-close-char* #\])

(defvar *restore-sql-reader-syntax* nil)

;;
;; Exported function for disabling SQL syntax.
;;
(defmacro disable-sql-reader-syntax ()
  "Disable SQL '[' syntax."
  '(eval-when (:compile-toplevel :load-toplevel :execute)
     (setf *restore-sql-reader-syntax* nil)
     (%disable-sql-reader-syntax)))

(defmacro locally-disable-sql-reader-syntax ()
  '(eval-when (:compile-toplevel :load-toplevel :execute)
    (%disable-sql-reader-syntax)))

(defun %disable-sql-reader-syntax ()
  (set-macro-character *sql-macro-open-char* *original-reader-enter*)
  (setq *original-reader-enter* nil))

;;
;; Exported function for enabling SQL syntax.
;;
(defmacro enable-sql-reader-syntax ()
  "Enable SQL '[' syntax."
  '(eval-when (:compile-toplevel :load-toplevel :execute)
     (setf *restore-sql-reader-syntax* t)
     (%enable-sql-reader-syntax)))

(defmacro locally-enable-sql-reader-syntax ()
  '(eval-when (:compile-toplevel :load-toplevel :execute)
     (%enable-sql-reader-syntax)))

(defun %enable-sql-reader-syntax ()
  (setq *original-reader-enter* (get-macro-character *sql-macro-open-char*))
  (set-macro-character *sql-macro-open-char* #'sql-reader-open)
  (enable-sql-close-syntax))

(defmacro restore-sql-reader-syntax ()
  '(eval-when (:compile-toplevel :load-toplevel :execute)
     (if *restore-sql-reader-syntax*
	 (%enable-sql-reader-syntax)
       (%disable-sql-reader-syntax))))

;;
;; Internal function that disables the close syntax when leaving sql
;; context.
;;
(defun disable-sql-close-syntax ()
  (set-macro-character *sql-macro-close-char* *original-reader-exit*)
  (setq *original-reader-exit* nil))

;;
;; Internal function that enables close syntax when entering SQL
;; context.
;;
(defun enable-sql-close-syntax ()
  (setq *original-reader-exit* (get-macro-character *sql-macro-close-char*))
  (set-macro-character *sql-macro-close-char*
		       (get-macro-character #\))))

(defun sql-reader-open (stream char)
  (declare (ignore char))
  (let ((sqllist (read-delimited-list #\] stream t)))
    (if (sql-operator (car sqllist))
	(cons (sql-operator (car sqllist)) (cdr sqllist))
      (apply #'generate-sql-reference sqllist))))

(defun sql-expression (&key string table alias attribute type params)
    (cond
     (string
      (make-instance 'sql :string string))
     (attribute
      (make-instance 'sql-ident-attribute  :name attribute
		     :qualifier (or table alias)
		     :type type
		     :params params))
     ((and table (not attribute))
      (make-instance 'sql-ident-table :name table
		     :table-alias alias))))

(defun generate-sql-reference (&rest arglist)
  (cond ((= (length arglist) 1)	; string, table or attribute
	 (if (stringp (car arglist))
	     (sql-expression :string (car arglist))
	   (sql-expression :attribute (car arglist))))
	((<= 2 (length arglist))
	 (let ((sqltype (if (keywordp (caddr arglist))
			    (caddr arglist) nil))
	       (sqlparam (if (keywordp (caddr arglist))
			     (caddr arglist))))
	   (cond
	    ((stringp (cadr arglist))
	     (sql-expression :table (car arglist)
			     :alias (cadr arglist)
			     :type sqltype))
	    ((keywordp (cadr arglist))
	     (sql-expression :attribute (car arglist)
			     :type (cadr arglist)
			     :params sqlparam))
	    (t
	     (sql-expression :attribute (cadr arglist)
			     :table (car arglist)
			     :params sqlparam
			     :type sqltype)))))
	(t
	 (error 'maisql-sql-syntax-error :reason "bad expression syntax"))))

)