# ClamTk, copyright (C) 2004-2010 Dave M
#
# This file is part of ClamTk.
#
# ClamTk is free software; you can redistribute it and/or modify it
# under the terms of either:
#
# a) the GNU General Public License as published by the Free Software
# Foundation; either version 1, or (at your option) any later version, or
#
# b) the "Artistic License".
package ClamTk::Results;

use strict;
#use warnings;    # disabled upon release
$|++;

use File::Basename;
use File::Copy;
use Digest::MD5 qw/md5_hex/;

use encoding 'utf8';

use Gtk2;
use Gtk2::SimpleList;
use Glib qw/TRUE FALSE/;

use Locale::gettext;
use POSIX qw/locale_h/;
textdomain('clamtk');
setlocale( LC_MESSAGES, '' );
bind_textdomain_codeset( 'clamtk', 'UTF-8' );

my $popup;
my $slist;
my $hover_label;

sub display {
    shift;    # don't need package name
    my $cmd_scan = shift;
    my ($hash) = @_;

    $popup = Gtk2::Window->new();
    $popup->signal_connect( destroy => sub { $popup->destroy } );
    $popup->set_default_size( 600, 250 );
    $popup->set_title( gettext('Scanning Results') );

    my $pbox = Gtk2::VBox->new();
    $popup->add($pbox);
    $hover_label =
        ( $cmd_scan == 0 )
        ? Gtk2::Label->new( gettext('Possible infections have been found.') )
        : ( $cmd_scan == 1 )
        ? Gtk2::Label->new( gettext('The scan is complete.') )
        : '';
    $pbox->pack_start( $hover_label, FALSE, FALSE, 5 );

    # This scrolled window holds the slist
    my $scrolled_win = Gtk2::ScrolledWindow->new;
    $pbox->pack_start( $scrolled_win, TRUE, TRUE, 0 );
    $scrolled_win->set_shadow_type('etched_in');
    $scrolled_win->set_policy( 'never', 'automatic' );

    $slist = Gtk2::SimpleList->new(
        gettext('File')         => 'markup',
        gettext('Status')       => 'markup',
        gettext('Action Taken') => 'markup',
    );
    $scrolled_win->add($slist);
    $scrolled_win->grab_focus();

    #$slist->get_selection->set_mode('single');
    $slist->get_selection->set_mode('multiple');
    $slist->set_rules_hint(TRUE);
    $slist->set_headers_clickable(TRUE);

    $slist->set_reorderable(TRUE);
    map { $_->set_resizable(TRUE) } $slist->get_columns;

    #map { $_->set_fixed_width(250) } $slist->get_columns;
    #map { $_->set_sizing('fixed') } $slist->get_columns;

    $slist->set(
        hover_selection => TRUE,
        hover_expand    => TRUE
    );

    # Display the possible infections
    for my $keys ( sort keys %$hash ) {
        push @{ $slist->{data} },
            [
            $hash->{$keys}->{name}, $hash->{$keys}->{status},
            $hash->{$keys}->{action},
            ];
        Gtk2->main_iteration while ( Gtk2->events_pending );
    }

    my $tooltips = Gtk2::Tooltips->new;
    $tooltips->set_tip( $slist,
        gettext('Select a file and right-click for options...') );
    $tooltips->disable;

    # below: the right-click functionality. also uses 'sub confirm'.
    $slist->signal_connect(
        button_press_event => sub {
            my ( $widget, $event ) = @_;
            return FALSE unless $event->button == 3;
            my @sel   = $slist->get_selected_indices;
            my $deref = $sel[0];
            defined $deref or return;

            # We have to avoid parsing mail for now:
            # http://clamtk.sourceforge.net/faq.html#inbox
            my $maildirs = join( '|',
                '.thunderbird', '.mozilla-thunderbird',
                '.evolution',   'Mail',
                'kmail' );

            my $menu = Gtk2::Menu->new();

            # Right-click 'quarantine' option
            my $quar_pop =
                Gtk2::ImageMenuItem->new( gettext('Quarantine this file') );
            my $quar_image =
                Gtk2::Image->new_from_stock( 'gtk-refresh', 'menu' );
            $quar_pop->set_image($quar_image);
            $quar_pop->signal_connect(
                activate => sub {
                    for my $t (@sel) {
                        next if ( $hash->{$t}->{name} =~ /$maildirs/ );
                        next if ( -d $hash->{$t}->{name} );
                        main_confirm( $t, 'q' );
                    }
                }
            );
            $quar_pop->show();
            $menu->append($quar_pop)
                unless dirname( $hash->{$deref}->{name} )
                    =~ /^\/(proc|sys|dev)/;

            # Right-click 'delete' option
            my $delete_pop =
                Gtk2::ImageMenuItem->new( gettext('Delete this file') );
            my $del_image =
                Gtk2::Image->new_from_stock( 'gtk-delete', 'menu' );
            $delete_pop->set_image($del_image);
            $delete_pop->signal_connect(
                activate => sub {
                    for my $u (@sel) {
                        next if ( $hash->{$u}->{name} =~ /$maildirs/ );
                        next if ( -d $hash->{$u}->{name} );
                        main_confirm( $u, 'd' );
                    }
                }
            );
            $delete_pop->show();
            $menu->append($delete_pop)
                unless dirname( $hash->{$deref}->{name} )
                    =~ /^\/(proc|sys|dev)/;

            # Right-click 'save-as' option.
            # This is useful if the file is 'opened' (intercepted)
            # as a $web_browser download
            my $save_pop =
                Gtk2::ImageMenuItem->new_from_stock( 'gtk-save-as', undef );
            $save_pop->signal_connect(
                activate => sub {
                    return if ( -d $slist->{data}[$deref][0] );
                    my $save_dialog = Gtk2::FileChooserDialog->new(
                        gettext('Save As...'), undef, 'save',
                        'gtk-cancel' => 'cancel',
                        'gtk-ok'     => 'ok',
                    );
                    $save_dialog->set_do_overwrite_confirmation(TRUE);
                    $save_dialog->set_default_response('ok');

                    if ( "ok" eq $save_dialog->run ) {
                        my $tmp = $save_dialog->get_filename;
                        $save_dialog->destroy();
                        move( $slist->{data}[$deref][0], $tmp )
                            or do {
                            show_message_dialog( $popup, 'error', 'close',
                                gettext('Could not save that file.') );
                            return FALSE;
                            };
                        $slist->{data}[$deref][2] = gettext('Moved');
                        main_confirm( $deref, 's' );
                        show_message_dialog( $popup, 'info', 'close',
                            gettext('File saved.') );
                        return FALSE;
                    } else {
                        $save_dialog->destroy();
                    }
                }
            );
            $save_pop->show();
            $menu->append($save_pop);

            # Right-click 'cancel' option
            my $cancel_pop =
                Gtk2::ImageMenuItem->new_from_stock( 'gtk-cancel', undef );
            $cancel_pop->signal_connect( activate => sub { return; } );
            $cancel_pop->show();
            $menu->append($cancel_pop);
            $menu->popup( undef, undef, undef, undef, $event->button,
                $event->time );

            return TRUE;
        }
    );
    $popup->show_all();
}

sub main_confirm {
    # This sub accepts both quarantine and delete options.
    # The quarantine happens without prompting, but deleting
    # always prompts the user.
    # $number = row of selected file so we can reference it
    # $do_this = quarantine, delete
    my $number  = shift;
    my $do_this = shift;

    my $full_name = $slist->{data}[$number][0];
    my $status    = $slist->{data}[$number][1];
    my $action    = $slist->{data}[$number][2];

    # do_this = quarantine (q), delete (d), or save (s).
    # The #CCCCCC color gives a neat grayed-out look
    # once the file is quarantined, deleted, or saved
    if ( $do_this eq 'q' ) {
        if ( not -e $full_name ) {
            $hover_label->set_text(
                sprintf gettext('File has been moved or deleted already.') );
            main_slist_delete($number);
            return;
        }
        if ( move_to_quarantine($number) ) {
            $hover_label->set_text( gettext('File has been quarantined.') );
            $slist->{data}[$number][2] =
                sprintf gettext("<b>Quarantined</b>");
        } else {
            $hover_label->set_text(
                gettext('File could not be quarantined.') );
            return;
        }
    } elsif ( $do_this eq 'd' ) {
        if ( not -e $full_name ) {
            $hover_label->set_text(
                gettext('File has been moved or deleted already.') );
            main_slist_delete($number);
            return;
        }
        my $confirm_message = gettext('Really delete this file?');
        $confirm_message .= ' ' . '(' . basename($full_name) . ')';
        my $confirm =
            Gtk2::MessageDialog->new( $popup, [qw(modal destroy-with-parent)],
            'question', 'ok-cancel', $confirm_message );

        if ( 'cancel' eq $confirm->run ) {
            $confirm->destroy;
            return;
        } else {
            $confirm->destroy;
            if ( unlink($full_name) ) {
                $hover_label->set_text( gettext('File has been deleted.') );
                $slist->{data}[$number][2] =
                    sprintf gettext("<b>Deleted</b>");
            } else {
                $hover_label->set_text(
                    gettext('File could not be deleted.') );
                return;
            }
        }
    } elsif ( $do_this eq 's' ) {
        my $new_action = gettext('Moved');
        $slist->{data}[$number][0] =
            "<span foreground='#CCCCCC'>$full_name</span>";
        $slist->{data}[$number][1] =
            "<span foreground='#CCCCCC'>$status</span>";
        $slist->{data}[$number][2] =
            "<span foreground='#CCCCCC'><b>$new_action</b></span>";
        return;
    }
    main_slist_delete($number);
    return;
}

sub main_slist_delete {
    my $number    = shift;
    my $full_name = $slist->{data}[$number][0];
    my $status    = $slist->{data}[$number][1];
    my $action    = $slist->{data}[$number][2];
    $slist->{data}[$number][0] =
        "<span foreground='#CCCCCC'>$full_name</span>";
    $slist->{data}[$number][1] = "<span foreground='#CCCCCC'>$status</span>";
    $slist->{data}[$number][2] = "<span foreground='#CCCCCC'>$action</span>";
    $popup->queue_draw;
    return;
}

sub move_to_quarantine {
    my $number = shift;

    my $full_name = $slist->{data}[$number][0];
    my $basename  = basename($full_name);

    my $paths = ClamTk::App->get_path('viruses');

    # Ensure quarantine directory exists
    if ( not -e $paths or not -d $paths ) {
        show_message_dialog( $popup, 'info', 'close',
            gettext('Quarantine directory does not exist.') );
        return;
    }

    # Get permissions
    my $mode = ( stat($full_name) )[2];
    my $perm = sprintf( "%03o", $mode & oct(7777) );

    chmod oct(600), $full_name;
    system( 'mv', $full_name, "$paths/$basename" ) == 0
        or do {
        # When a 'mv' fails, it still probably did a 'cp'...
        # 'mv' copies the file first, then unlinks the source.
        # d'oh... so just to make sure, unlink the intended target
        # and THEN return.  No need to check for failure.
        unlink("$paths/$basename");
        return;
        };
    rename( "$paths/$basename", "$paths/$basename.VIRUS" );

    # Update restore file by adding file, path and md5
    my $ctx = Digest::MD5->new;
    open( my $F, '<', "$paths/$basename.VIRUS" ) or do {
        warn "Could not open file for md5-ing: $!\n";
        return;
    };
    $ctx->addfile(*$F);
    my $md5 = $ctx->hexdigest;
    close($F);

    ClamTk::Prefs->restore( $md5, 'add', $full_name, $perm );

    if ( not -e $full_name ) {
        return 1;
    } else {
        return -1;
    }
}

sub show_message_dialog {
    my ( $parent, $type, $button, $message ) = @_;

    my $dialog;
    $dialog =
        Gtk2::MessageDialog->new_with_markup( $parent,
        [qw(modal destroy-with-parent)],
        $type, $button, $message );

    $dialog->run;
    $dialog->destroy;
    return;
}

1;
