
#include "Core/precomp.h"

#ifdef USE_NETWORK

#include <API/Core/System/error.h>
#include <API/Core/System/mutex.h>
#include <API/Core/System/mutex.h>
#include <API/Core/Network/netcomputer.h>
#include <API/Core/Network/netgroup.h>
#include <API/Core/Network/netmessage.h>
#include <API/Core/System/thread.h>
#include <API/Core/System/cl_assert.h>
#include <API/Core/System/system.h>
#include <Core/Network/Generic/network_delivery_impl.h>
#include <Core/Network/Generic/netgame_server.h>
#include <Core/Network/Generic/netgame_generic.h>
#include <Core/Network/Generic/network_generic.h>
#include <Core/IOData/Generic/outputsource_memory_generic.h>
#include <Core/IOData/Generic/inputsource_memory_generic.h>

CL_NetGame_Server::CL_NetGame_Server(
	CL_Network_Generic *network,
	const char *game_id,
	int port)
: CL_NetGame_Generic(network)
{
	mutex = CL_Mutex::create();

	this->game_id = game_id;
	this->port = port;
	
	id_counter = 1; // server is zero, that's why we start at 1.
	
	udp_connection = network->provider->create_udp_connection(port);
	
	network->provider->start_accept_on_port(port);

	exit_thread = false;
	thread = CL_Thread::create(this);
	thread->start();
}

CL_NetGame_Server::~CL_NetGame_Server()
{
	exit_thread = true;
	thread->wait();
	delete thread;

	network->provider->stop_accept_on_port(port);
	
	// clean up connections to clients:
	{
		for (
			std::list<CL_NetComputer_Host*>::iterator it = prejoin.begin();
			it != prejoin.end();
			it++)
		{
			delete *it;
		}
	}
	{
		for (
			std::list<CL_NetComputer_Host*>::iterator it = computers.begin();
			it != computers.end();
			it++)
		{
			delete *it;
		}
	}

	while (leave_queue.empty() == false)
	{
		delete leave_queue.front();
		leave_queue.pop();
	}

	delete mutex;
}

const CL_NetComputer *CL_NetGame_Server::get_server() const
{
	// Should we return NULL if server? Or should we have a instance
	// representing ourself?

	return NULL;
}

const CL_NetGroup *CL_NetGame_Server::get_all() const
{
	return &all;
}

bool CL_NetGame_Server::peek(int channel) const
{
	CL_MutexSection mutex_section(mutex);

	CL_NetChannelQueue *queue = find_queue(channel);
	if (queue != NULL) return !queue->empty();
	
	return false;
}

CL_NetMessage CL_NetGame_Server::receive(int channel, int timeout)
{
	CL_MutexSection mutex_section(mutex);

	CL_NetChannelQueue *queue = find_queue(channel);
	if (queue != NULL)
	{
		while (queue->empty() && timeout > 0) 
		{
			int s = timeout >= 20 ? 20 : timeout;
			CL_System::sleep(s);
			timeout -= s;
		}
		if (queue->empty()) throw CL_Error("No message to receive.");
		CL_NetMessage msg = queue->front();
		queue->pop();
		return msg;
	}
	
	CL_NetMessage msg;
	return msg;
}

void CL_NetGame_Server::send(
	const int dest_channel,
	const CL_NetComputer *dest,
	const CL_NetMessage &message,
	bool reliable /*= true*/)
{
	CL_MutexSection mutex_section(mutex);

	static bool warning = true;
	if (warning && reliable == false)
	{
		cl_info(info_network, "cannot send data unreliable (udp): not implemented yet!");
		warning = false;
	}

	CL_NetComputer_Host *dest_host = (CL_NetComputer_Host *) dest;
	CL_NetChannelQueue *queue = find_queue(dest_channel);
	
	if (queue == NULL)
		return; // nobody has read or write access. Don't send message.

	if (has_read_access(queue, dest_host) == false)
		return; // dest_host is not allowed to read messages on channel.

	// permissions ok, send message:
	CL_OutputSource_MemoryGeneric output;
	output.write_int32(Packet_NetChannel_Message_ToClient);
	output.write_int32(dest_channel);
	output.write_int32(message.data.size());
	output.write(message.data.data(), message.data.size());
	
	CL_ConnectionPacket packet;
	packet.size = output.size();
	packet.data = output.get_data();

	dest_host->connection->send(packet);
}

void CL_NetGame_Server::send(
	const int dest_channel,
	const CL_NetGroup *dest,
	const CL_NetMessage &message,
	bool reliable /*= true*/)
{
	CL_MutexSection mutex_section(mutex);

	for (
		std::list<CL_NetComputer*>::const_iterator it = dest->computers.begin();
		it != dest->computers.end();
		it++)
	{
		send(
			dest_channel,
			*it,
			message,
			reliable);
	}
}

CL_NetComputer *CL_NetGame_Server::receive_computer_leave()
{
	CL_MutexSection mutex_section(mutex);

	if (leave_queue.empty()) return NULL;

	CL_NetComputer *ret = leave_queue.front();
	leave_queue.pop();

	// Remove all access rules for this object.
	for (
		std::map<int, CL_NetChannelQueue>::iterator it = netchannels.begin();
		it != netchannels.end();
		it++)
	{
		(*it).second.access.erase(((CL_NetComputer_Host *) ret)->id);
	}

	return ret;
}

const CL_NetComputer *CL_NetGame_Server::receive_computer_join()
{
	CL_MutexSection mutex_section(mutex);

	if (join_queue.empty()) return NULL;
	const CL_NetComputer *ret = join_queue.front();
	join_queue.pop();

	return ret;
}

const CL_NetComputer *CL_NetGame_Server::receive_computer_rejoin()
{
	CL_MutexSection mutex_section(mutex);

	if (rejoin_queue.empty()) return NULL;
	const CL_NetComputer *ret = rejoin_queue.front();
	rejoin_queue.pop();

	return ret;
}

bool CL_NetGame_Server::receive_game_closed()
{
	return false; // we are the server - only closed if deleted.
}

int CL_NetGame_Server::access_status(int /*channel*/) const
{
	// server has always full access.
	return ACCESS_CHANNEL_READ|ACCESS_CHANNEL_WRITE;
}

bool CL_NetGame_Server::is_writable(int /*channel*/) const
{
	return true;
}

bool CL_NetGame_Server::is_readable(int /*channel*/) const
{
	return true;
}

// Client side only:
int CL_NetGame_Server::receive_access_changed()
{
	return -1; // we are not a client!
}

// Server side only:
void CL_NetGame_Server::set_access(
	int channel,
	const CL_NetComputer *computer,
	int access_rights)
{
	CL_MutexSection mutex_section(mutex);

	CL_NetComputer_Host *computer_host = (CL_NetComputer_Host *) computer;
	CL_NetChannelQueue *queue = create_queue(channel);

	int cur_access = 0;
	std::map<int,int>::iterator access_it = queue->access.find(computer_host->id);
	if (access_it != queue->access.end())
	{
		cur_access = (*access_it).second;
		(*access_it).second = access_rights;
	}
	else
	{
		queue->access.insert(std::pair<int,int>(computer_host->id, access_rights));
	}

	if (cur_access != access_rights)
	{
		CL_OutputSource_MemoryGeneric output;
		output.write_int32(Packet_NetChannel_AccessChange);
		output.write_int32(channel);
		output.write_int32(access_rights);
	
		CL_ConnectionPacket packet;
		packet.size = output.size();
		packet.data = output.get_data();

		computer_host->connection->send(packet);
	}
}

void CL_NetGame_Server::set_access(
	int channel,
	const CL_NetGroup *group,
	int access_rights)
{
	CL_MutexSection mutex_section(mutex);

	for (
		std::list<CL_NetComputer *>::const_iterator it = group->computers.begin();
		it != group->computers.end();
		it++)
	{
		set_access(
			channel,
			*it,
			access_rights);
	}
}

void CL_NetGame_Server::keep_alive()
{
	CL_MutexSection mutex_section(mutex);
	
	if (exit_thread) return;

	// check connections for data:
	std::list<CL_NetComputer_Host*>::iterator it_computers = computers.begin();
	while (it_computers != computers.end())
	{
		while ((*it_computers)->connection->peek())
		{
			CL_ConnectionPacket msg = (*it_computers)->connection->receive();
			
			CL_InputSource_MemoryGeneric input(msg.data, msg.size, true);
			switch (input.read_int32())
			{
			case Packet_NetChannel_AccessChange:
				cl_info(info_network, "Network Protocol error!");
				cl_assert(false);
				break;
			
			case Packet_NetChannel_Message_ToServer:
				{
					static bool warning = true;
					if (warning)
					{
						cl_info(info_network, "permissions not checked: Not implemented yet.");
						warning = false;
					}
					CL_NetChannelQueue *queue = create_queue(input.read_int32());
					CL_NetMessage game_msg;
					int size = input.read_int32();
					char *data = new char[size];
					game_msg.from = *it_computers;
					input.read(data, size);
					game_msg.data.append(data, size);
					queue->push(game_msg);
				}
				break;

			default:
				cl_info(info_network, "Network Protocol error!");
				cl_assert(false);
			}
		}

		if ((*it_computers)->connection->connection_lost())
		{
			leave_queue.push(*it_computers);
			all.computers.remove(*it_computers);
			network->provider->remove_connection((*it_computers)->connection);
			it_computers = computers.erase(it_computers);
		}
		else
		{
			it_computers++;
		}
	}


	// check for "hello to you too" connections:
	std::list<CL_NetComputer_Host*>::iterator it_prejoin = prejoin.begin();
	while (it_prejoin != prejoin.end())
	{
		if ((*it_prejoin)->connection->connection_lost()) // how rude!
		{
			delete *it_prejoin;
			it_prejoin = prejoin.erase(it_prejoin);
			
			continue;
		}

		if ((*it_prejoin)->connection->peek())
		{
			CL_ConnectionPacket msg = (*it_prejoin)->connection->receive();
			
			CL_InputSource_MemoryGeneric input(msg.data, msg.size);
			if (msg.size >= (int) sizeof(int))
			{
				switch (input.read_int32())
				{
				case Packet_Hello_ToYouToo: // ahh - he wants to join.
					computers.push_back(*it_prejoin);
					all.computers.push_back(*it_prejoin);
					join_queue.push(*it_prejoin);
					it_prejoin = prejoin.erase(it_prejoin);
					continue;

				default:
					cl_info(info_network, "Network Protocol error!");
					cl_assert(false);
				}
			}
			
			delete[] ((char*) msg.data);
		}

		it_prejoin++;
	}
	
	// accept incoming clients:
	while (true)
	{
		CL_Connection *new_client = network->provider->accept();
		if (new_client == NULL) break;
		
		CL_OutputSource_MemoryGeneric msg;
		msg.write_int32(Packet_Hello);
		msg.write_int32(id_counter);
		msg.write_int32(game_id.length());
		msg.write(game_id.c_str(), game_id.length());
		
		CL_ConnectionPacket packet(msg.get_data(), msg.size());
		new_client->send(packet);
		
		prejoin.push_back(new CL_NetComputer_Host(new_client, id_counter++));
	}
	
	// check for UDP data:
	while (udp_connection->peek())
	{
		CL_UDPConnectionPacket netmsg = udp_connection->receive();
		
		try
		{
			CL_InputSource_MemoryGeneric message(netmsg.data, netmsg.size);
			
			int msgtype = message.read_int32();
			switch (msgtype)
			{
			case 0: // game ping message. Reply with game_id.
				{
					std::string req_game_id = message.read_string();
					if (req_game_id == game_id)
					{
						CL_OutputSource_MemoryGeneric msg;
						msg.write_int32(1); // game ping reply message.
						msg.write_string(game_id.c_str());
					
						CL_UDPConnectionPacket reply;
						reply.data = msg.get_data();
						reply.size = msg.size();
						reply.ip_addr = netmsg.ip_addr;
						reply.port = netmsg.port;
						udp_connection->send(reply);
					}
				}
				break;
				
			case 1: // game ping reply message.
				// ignore it. server shouldn't get this message!
				break;
				
			case 2: // normal udp game packet.
				// not implemented yet.
				break;
			}
		}
		catch (CL_Error err)
		{
			// protocol error. We need some logging stuff.
		}
		
		delete[] netmsg.data;
	}
}

void CL_NetGame_Server::run()
{
	while (exit_thread == false)
	{
		keep_alive();
		network->provider->wait_for_connection_data(mutex);
	}
}

CL_NetChannelQueue *CL_NetGame_Server::find_queue(int netchannel) const
{
	std::map<int, CL_NetChannelQueue>::const_iterator it = netchannels.find(netchannel);
	if (it == netchannels.end()) return NULL;
	
	return (CL_NetChannelQueue *) (&(*it).second);
}

CL_NetChannelQueue *CL_NetGame_Server::create_queue(int netchannel)
{
	CL_NetChannelQueue *found = find_queue(netchannel);
	if (found != NULL) return found;

	CL_NetChannelQueue c;
	netchannels.insert(std::pair<int, CL_NetChannelQueue>(netchannel, c));

	return &netchannels[netchannel];
}

bool CL_NetGame_Server::has_read_access(
	CL_NetChannelQueue *channel,
	CL_NetComputer_Host *host)
{
	std::map<int, int>::iterator it = channel->access.find(host->id);
	if (it == channel->access.end()) return false;
	
	return ((*it).second & ACCESS_CHANNEL_READ) == ACCESS_CHANNEL_READ;
}

// ---------

CL_NetComputer_Host::CL_NetComputer_Host(
	CL_Connection *connection,
	int id)
{
	this->connection = connection;
	this->id = id;
}

CL_NetComputer_Host::~CL_NetComputer_Host()
{
	delete connection;
}

#endif
