#include "Core/precomp.h"
#include "API/gl.h"
#include "API/GL/texture.h"

#define BSP_PARTITION_LIMIT 1

CL_SceneRenderer::CL_SceneRenderer()
{
	std::cout << "Face size: " << sizeof(SFace) << std::endl;
	std::cout << "Partition size: " << sizeof(SBSPartition) << std::endl;
	root = partition_alloc.get();
	root->front = NULL;
	root->back = NULL;
	partitions.push_back(root);
	cur_texture = NULL;
	cur_cols[0] = 1.0f;
	cur_cols[1] = 1.0f;
	cur_cols[2] = 1.0f;
	cur_cols[3] = 1.0f;
	cur_texcoords[0] = 0.0f;
	cur_texcoords[1] = 0.0f;
	cur_vertex = 0;
	cur_state = 0;
	cur_portal_vertex = 0;
	cur_cube = new SCube;
	cur_face = new SFace;
	cur_strip = new STStrip;
	showed_cubes = NULL;
	cubes = NULL;
	cubes_size = 0;
}

CL_SceneRenderer::~CL_SceneRenderer()
{
	clear();
}

void CL_SceneRenderer::clear()
{
	partitions.clear();
	partition_alloc.clear();
}

void CL_SceneRenderer::begin_cube(int cube_id)
{
	cur_cube_id = cube_id;
}

void CL_SceneRenderer::end_cube()
{
	if (cur_cube_id >= cubes_size)
	{
		int new_size = cubes_size*2;
		if (new_size <= cur_cube_id) new_size = cur_cube_id+1;
		SCube **new_cubes = new SCube*[new_size];
		memcpy(new_cubes, cubes, sizeof(SCube*)*cubes_size);
		for (int i=cubes_size;i<new_size;i++) new_cubes[i] = NULL;
		delete[] cubes;
		cubes_size = new_size;
		cubes = new_cubes;
	}
	cl_assert(cubes[cur_cube_id] == NULL);
	cubes[cur_cube_id] = cur_cube;
	root->cubes.push_back(cur_cube_id);
	cur_cube = new SCube;
}

void CL_SceneRenderer::begin_portal(int connect_id)
{
	cur_state = 2;
	for (int i=0;i<6;i++)
	{
		if (cur_cube->neighbors[i] == -1)
		{
			cur_cube->neighbors[i] = connect_id;
			cur_portal_vertex = i*4;
			break;
		}
	}
}

void CL_SceneRenderer::end_portal()
{
}

void CL_SceneRenderer::begin_facelist()
{
	cur_state = 0;
}

void CL_SceneRenderer::end_facelist()
{
}

void CL_SceneRenderer::begin_tstriplist()
{
	cur_state = 1;
}

void CL_SceneRenderer::end_tstriplist()
{
	cur_cube->strips.push_back(cur_strip);
	cur_strip = new STStrip;
}

void CL_SceneRenderer::add_vertex(float x, float y, float z)
{
	switch (cur_state)
	{
	case 0:
		{
			float *vdata = &cur_face->vdata[cur_vertex*9];
			vdata[0] = cur_cols[0];
			vdata[1] = cur_cols[1];
			vdata[2] = cur_cols[2];
			vdata[3] = cur_cols[3];
			vdata[4] = cur_texcoords[0];
			vdata[5] = cur_texcoords[1];
			vdata[6] = x;
			vdata[7] = y;
			vdata[8] = z;
			cur_face->texture = cur_texture;

			cur_vertex++;

			if (cur_vertex == 3)
			{
				cur_cube->faces.push_back(cur_face);
				cur_vertex = 0;
			}
		}
		break;
	case 1:
		{

			float *vdata = new float[(cur_strip->vcount+1)*9];
			memcpy(vdata, cur_strip->vdata, sizeof(float)*9*cur_strip->vcount);
			float *new_v = &vdata[cur_strip->vcount*9];
			new_v[0] = cur_cols[0];
			new_v[1] = cur_cols[1];
			new_v[2] = cur_cols[2];
			new_v[3] = cur_cols[3];
			new_v[4] = cur_texcoords[0];
			new_v[5] = cur_texcoords[1];
			new_v[6] = x;
			new_v[7] = y;
			new_v[8] = z;
			cur_strip->texture = cur_texture;
			cur_strip->vcount++;
			delete[] cur_strip->vdata;
			cur_strip->vdata = vdata;
		}
		break;
	case 2:
		{
			for (int i=0;i<6;i++) if (cur_cube->neighbors[i] == -1) break;
			i--;
			cl_assert(i>=0);
			cur_cube->portal_coords[cur_portal_vertex*3+0] = x;
			cur_cube->portal_coords[cur_portal_vertex*3+1] = y;
			cur_cube->portal_coords[cur_portal_vertex*3+2] = z;
			cur_portal_vertex++;
		}
		break;
	}
}

void CL_SceneRenderer::set_texture(CL_Texture *texture)
{
	cl_assert((void *) texture != (void *) 0xfdfdfdfd);
	cur_texture = texture;
}

void CL_SceneRenderer::set_color(float r, float g, float b, float a)
{
	cur_cols[0] = r;
	cur_cols[1] = g;
	cur_cols[2] = b;
	cur_cols[3] = a;
}

void CL_SceneRenderer::set_texcoord(float u, float v)
{
	cur_texcoords[0] = u;
	cur_texcoords[1] = v;
}

inline CL_Vector CL_SceneRenderer::find_plane_normal(float *vdata)
{
	CL_Vector v1(
		vdata[1*9+6]-vdata[6],
		vdata[1*9+7]-vdata[7],
		vdata[1*9+8]-vdata[8]);

	CL_Vector v2(
		vdata[2*9+6]-vdata[6],
		vdata[2*9+7]-vdata[7],
		vdata[2*9+8]-vdata[8]);

	return v1.cross(v2);
}

inline CL_Vector CL_SceneRenderer::find_portal_plane_normal(float *vdata)
{
	CL_Vector v1(
		vdata[1*3+0]-vdata[0],
		vdata[1*3+1]-vdata[1],
		vdata[1*3+2]-vdata[2]);

	CL_Vector v2(
		vdata[2*3+0]-vdata[0],
		vdata[2*3+1]-vdata[1],
		vdata[2*3+2]-vdata[2]);

	return v1.cross(v2);
}

inline float CL_SceneRenderer::get_plane_distance(
	float *pos,
	const CL_Vector &plane_normal,
	float plane_d)
{
	float dx = plane_normal.x*pos[0];
	float dy = plane_normal.y*pos[1];
	float dz = plane_normal.z*pos[2];
	float d = dx+dy+dz+plane_d;

	return d;
}

int CL_SceneRenderer::get_split_delta(
	const CL_Vector &n,
	float d,
	bool copy,
	SBSPartition *p,
	SBSPartition *front,
	SBSPartition *back)
{
	int fcount = 0;
	int bcount = 0;

	std::vector<int> ccopy;

	std::vector<int>::iterator c_it;
	for (c_it=p->cubes.begin();c_it!=p->cubes.end();c_it++)
	{
		for (std::vector<SFace *>::iterator f_it=cubes[*c_it]->faces.begin();
		f_it!=cubes[*c_it]->faces.end();
		f_it++)
		{
			float *vdata = (*f_it)->vdata;
			int fc = 0;
			int bc = 0;
			for (int i=0;i<3;i++)
			{
				float v = get_plane_distance(
					&vdata[i*9+6],
					n,
					d);
				if (v > 0) 
				{
					fc++;
				}
				else if (v < 0) 
				{
					bc++;
				}
			}
			if (bc == 0 && fc > 0)
			{
				fcount++;
			}
			else if (fc == 0 && bc > 0)
			{
				bcount++;
			}
		}
		for (std::vector<STStrip *>::iterator s_it=cubes[*c_it]->strips.begin();
		s_it!=cubes[*c_it]->strips.end();
		s_it++)
		{
			float *vdata = (*s_it)->vdata;
			int fc = 0;
			int bc = 0;
			for (int i=0;i<(*s_it)->vcount;i++)
			{
				float v = get_plane_distance(
					&vdata[i*9+6],
					n,
					d);
				if (v > 0) 
				{
					fc++;
				}
				else if (v < 0) 
				{
					bc++;
				}
			}
			if (bc == 0 && fc > 0)
			{
				fcount++;
			}
			else if (fc == 0 && bc > 0)
			{
				bcount++;
			}
		}
		if (copy)
		{
			if (fcount > 0 && bcount == 0)
			{
				front->cubes.push_back(*c_it);
			}
			else if (bcount > 0 && fcount == 0)
			{
				back->cubes.push_back(*c_it);
			}
			else
			{
				ccopy.push_back(*c_it);
			}
		}
	}

	if (copy)
	{
		p->cubes.clear();
		p->cubes.swap(ccopy);
	}

	return abs(fcount - bcount);
}

void CL_SceneRenderer::find_even_partition(
	SBSPartition *p,
	SBSPartition *front,
	SBSPartition *back)
{
	int best_cube = 0;
	int best_prim = 0;
	int best_delta = 1000000; 
	int pos = 0;
	int type = 0;
	bool copy_dest = false;

	for (std::vector<int>::iterator c_it=p->cubes.begin();c_it!=p->cubes.end();c_it++)
	{
		for (std::vector<SFace *>::iterator f_it=cubes[*c_it]->faces.begin();f_it!=cubes[*c_it]->faces.end();f_it++)
		{
			int fcount = 0;
			int bcount = 0;
			CL_Vector n = find_plane_normal((*f_it)->vdata);
			float plane_d = 
				-((*f_it)->vdata[6]*n.x+
				  (*f_it)->vdata[7]*n.y+
				  (*f_it)->vdata[8]*n.z);

			int delta = get_split_delta(n, plane_d, false, p, NULL, NULL);
			if (delta < best_delta)
			{
				best_cube = *c_it;
				best_prim = pos;
				best_delta = delta;
				type = 0;
			}
			pos++;
		}
		pos = 0;
		for (std::vector<STStrip *>::iterator s_it=cubes[*c_it]->strips.begin();s_it!=cubes[*c_it]->strips.end();s_it++)
		{
			int fcount = 0;
			int bcount = 0;
			CL_Vector n = find_plane_normal((*s_it)->vdata);
			n.normalize();
			float plane_d = 
				-((*s_it)->vdata[6]*n.x+
				  (*s_it)->vdata[7]*n.y+
				  (*s_it)->vdata[8]*n.z);

			int delta = get_split_delta(n, plane_d, false, p, NULL, NULL);
			if (delta < best_delta)
			{
				best_cube = *c_it;
				best_prim = pos;
				best_delta = delta;
				type = 1;
			}
			pos++;
		}
	}

	if (type == 0)
	{
		CL_Vector n = find_plane_normal((float *) cubes[best_cube]->faces[best_prim]->vdata);
		n.normalize();
		float plane_d = 
			-(cubes[best_cube]->faces[best_prim]->vdata[6]*n.x+
			  cubes[best_cube]->faces[best_prim]->vdata[7]*n.y+
			  cubes[best_cube]->faces[best_prim]->vdata[8]*n.z);
		get_split_delta(n, plane_d, true, p, front, back);
	}
	else
	{
		CL_Vector n = find_plane_normal((float *) cubes[best_cube]->strips[best_prim]->vdata);
		n.normalize();
		float plane_d = 
			-(cubes[best_cube]->strips[best_prim]->vdata[6]*n.x+
			  cubes[best_cube]->strips[best_prim]->vdata[7]*n.y+
			  cubes[best_cube]->strips[best_prim]->vdata[8]*n.z);
		get_split_delta(n, plane_d, true, p, front, back);
	}
}

bool CL_SceneRenderer::partition_rec(SBSPartition *cur)
{
	SBSPartition *front = partition_alloc.get();
	front->front = NULL;
	front->back = NULL;

	SBSPartition *back = partition_alloc.get();
	back->front = NULL;
	back->back = NULL;

	partitions.push_back(front);
	partitions.push_back(back);

	int org_cube_count = cur->cubes.size();

	find_even_partition(cur, front, back);

	cur->front = front;
	cur->back = back;

	bool ret = true;

	if (front->cubes.size() > BSP_PARTITION_LIMIT && front->cubes.size() < org_cube_count)
	{
		ret &= partition_rec(front);
	}
	if (back->cubes.size() > BSP_PARTITION_LIMIT && back->cubes.size() < org_cube_count)
	{
		ret &= partition_rec(back);
	}

	return ret;
}

bool CL_SceneRenderer::generate()
{
	for (int i=0;i<cubes_size;i++)
	{
		if (cubes[i] == NULL) continue;
	}
	std::cout << "Number of cubes in scene: " << root->cubes.size() << std::endl;
	if (root->cubes.size() > BSP_PARTITION_LIMIT)
	{
		bool result = partition_rec(root);
		showed_cubes = new bool[cubes_size];
		return result;
	}
	else
	{
		return true;
	}
}

inline void CL_SceneRenderer::show_facelist(const std::vector<SFace *> &faces)
{
	for (std::vector<SFace *>::const_iterator it=faces.begin();
	it!=faces.end();
	it++)
	{
		if ((*it)->texture != NULL) (*it)->texture->bind();

		glBegin(GL_TRIANGLES);

		float *vdata = (float *) (*it)->vdata;
		for (int i=0;i<3;i++)
		{
			glColor4fv(vdata);
			glTexCoord2fv(&vdata[4]);
			glVertex3fv(&vdata[6]);
			vdata += 9;
		}

		glEnd();
		face_count++;
	}
}

inline void CL_SceneRenderer::show_trianglestrip(const std::vector<STStrip *> &strips)
{
	for (std::vector<STStrip *>::const_iterator it=strips.begin();
	it!=strips.end();
	it++)
	{
		if ((*it)->texture != NULL) (*it)->texture->bind();

		glBegin(GL_TRIANGLE_STRIP);
//		glBegin(GL_LINE_LOOP);

		float *vdata = (float *) (*it)->vdata;
		for (int i=0;i<(*it)->vcount;i++)
		{
			glColor4fv(vdata);
			glColor3f(1,1,1);
			glTexCoord2fv(&vdata[4]);
			glVertex3fv(&vdata[6]);
			vdata += 9;
		}

		glEnd();
		face_count+=(*it)->vcount-2;
	}
}

void CL_SceneRenderer::show_rec(SBSPartition *partition)
{
	if (partition == NULL) return;

	if (partition->front != NULL || partition->back != NULL)
	{
		SBSPartition *primary;
		SBSPartition *secondary;

		float v = get_plane_distance(pos, partition->plane_normal, partition->plane_d);
		if (v > 0)
		{	
			primary = partition->front;
			secondary = partition->back;
		}
		else
		{
			primary = partition->back;
			secondary = partition->front;
		}

		show_rec(primary);

		for (std::vector<int>::iterator c_it=partition->cubes.begin();
		c_it!=partition->cubes.end();c_it++)
		{
			show_facelist(cubes[*c_it]->faces);
			show_trianglestrip(cubes[*c_it]->strips);
		}

		float t =
			(partition->plane_normal.x*(partition->pos[0]-pos[0])+
			 partition->plane_normal.y*(partition->pos[1]-pos[1])+
			 partition->plane_normal.z*(partition->pos[2]-pos[2]))/
			(partition->plane_normal.x*dir.x+
			 partition->plane_normal.y*dir.y+
			 partition->plane_normal.z*dir.z);

		if (t > 0)
		{
			show_rec(secondary);
		}
		else
		{
			int i = 5;
		}
	}
	else // partition->front == NULL && partition->back == NULL
	{
		for (std::vector<int>::iterator c_it=partition->cubes.begin();
		c_it!=partition->cubes.end();c_it++)
		{
			show_facelist(cubes[*c_it]->faces);
			show_trianglestrip(cubes[*c_it]->strips);
		}
	}
}

void CL_SceneRenderer::show_bsp(SBSPartition *cur, float col)
{
	if (cur == NULL || cur->pos == NULL) return;
	if (col < 0.4f) col = 0.4f;
	glColor3f(col, col, col);
	glVertex3f(cur->pos[0], cur->pos[1], cur->pos[2]);
	glVertex3f(
		cur->pos[0]+cur->plane_normal.x*20,
		cur->pos[1]+cur->plane_normal.y*20,
		cur->pos[2]+cur->plane_normal.z*20);

	show_bsp(cur->front, col-0.1f);
	show_bsp(cur->back, col-0.1f);
}

bool CL_SceneRenderer::inside_cube(int cube_id, const CL_Vector &pos)
{
	float pos_a[3];
	pos_a[0] = pos.x;
	pos_a[1] = pos.y;
	pos_a[2] = pos.z;
	std::vector<SFace *>::iterator f_it_end = cubes[cube_id]->faces.end();
	for (std::vector<SFace *>::iterator f_it=cubes[cube_id]->faces.begin();
		f_it!= f_it_end;f_it++)
	{
		CL_Vector n = find_plane_normal((*f_it)->vdata);
		float plane_d = 
			-((*f_it)->vdata[6]*n.x+
			  (*f_it)->vdata[7]*n.y+
			  (*f_it)->vdata[8]*n.z);

		float v = get_plane_distance(pos_a, n, plane_d);
		if (v > 0) return false;
	}
	std::vector<STStrip *>::iterator s_it_end = cubes[cube_id]->strips.end();
	for (std::vector<STStrip *>::iterator s_it=cubes[cube_id]->strips.begin();
		s_it!= s_it_end;s_it++)
	{
		CL_Vector n = find_plane_normal((*s_it)->vdata);
		float plane_d = 
			-((*s_it)->vdata[6]*n.x+
			  (*s_it)->vdata[7]*n.y+
			  (*s_it)->vdata[8]*n.z);

		float v = get_plane_distance(pos_a, n, plane_d);
		if (v > 0) return false;
	}
	for (int i=0;i<6;i++)
	{
		if (cubes[cube_id]->neighbors[i] != -1)
		{
			float *ndata = &cubes[cube_id]->portal_coords[i*4*3];
			CL_Vector n = find_portal_plane_normal(ndata);
			float plane_d = 
				-(ndata[0]*n.x+
				  ndata[1]*n.y+
				  ndata[2]*n.z);

			float v = get_plane_distance(pos_a, n, plane_d);
			if (v > 0) return false;
		}
	}
	return true;
}

int CL_SceneRenderer::find_cube_location(SBSPartition *partition, const CL_Vector &pos)
{
	int location = -1;
	if (partition->front != NULL && partition->back != NULL)
	{
		float v = get_plane_distance(this->pos, partition->plane_normal, partition->plane_d);
		if (v > 0)
		{	
			location = find_cube_location(partition->front, pos);
			if (location == -1) location = find_cube_location(partition->back, pos);
		}
		else
		{
			location = find_cube_location(partition->back, pos);
			if (location == -1) location = find_cube_location(partition->front, pos);
		}
	}
	if (location != -1) return location;

	for (std::vector<int>::iterator it = partition->cubes.begin();
	it!=partition->cubes.end();it++)
	{
		if (inside_cube(*it, pos)) return *it;
	}

	return -1;
}

void CL_SceneRenderer::show_cubes_rec(int cube_id, int connect_cube, float *coords, int rcount)
{
//	if (rcount >= 4) return;
	cl_assert(cube_id >= 0 && cube_id < cubes_size && cubes[cube_id]!=NULL);
	if (showed_cubes[cube_id]) return;
	showed_cubes[cube_id] = true;

	show_facelist(cubes[cube_id]->faces);
	show_trianglestrip(cubes[cube_id]->strips);

	if (connect_cube != -1)
	{
		float *org_coords = NULL;
		for (int n=0;n<6;n++)
		{
			if (cubes[cube_id]->neighbors[n] == connect_cube) 
			{
				org_coords = &cubes[cube_id]->portal_coords[n*4*3];
				break;
			}
		}
		cl_assert(org_coords != NULL);

		CL_Plane p1(
			CL_Vector(coords[0], coords[1], coords[2]),
			CL_Vector(coords[3]-coords[0], coords[4]-coords[1], coords[5]-coords[2]),
			CL_Vector(coords[6]-coords[0], coords[7]-coords[1], coords[8]-coords[2]));

		CL_Plane p2(
			CL_Vector(coords[9], coords[10], coords[11]),
			CL_Vector(coords[3]-coords[9], coords[4]-coords[10], coords[5]-coords[11]),
			CL_Vector(coords[6]-coords[9], coords[7]-coords[10], coords[8]-coords[11]));

		for (int i=0;i<6;i++)
		{
			if (cubes[cube_id]->neighbors[i] == -1) break;
			if (cubes[cube_id]->neighbors[i] == connect_cube) continue;

			float plan_coords[4*2];
			float plan_coords2[4*2];

			float *c_coords = &cubes[cube_id]->portal_coords[i*4*3];
			float *c_it = c_coords;
			cl_assert(coords != c_coords);
			for (int v=0;v<4;v++)
			{
				CL_Line l(
					CL_Vector(c_it[0], c_it[1], c_it[2]),
					CL_Vector(c_it[0]-pos[0], c_it[1]-pos[1], c_it[2]-pos[2]));

				CL_Vector isect = l.intersect_pos(p1);
				CL_Vector isect2 = l.intersect_pos(p2);
				CL_Vector t = p1.plan_coords(isect);
				CL_Vector t2 = p2.plan_coords(isect2);

				plan_coords[v*2+0] = t.x;
				plan_coords[v*2+1] = t.y;
				plan_coords2[v*2+0] = t2.x;
				plan_coords2[v*2+1] = t2.y;

				c_it += 3;
			}
			CL_Rectangle r1(plan_coords, &plan_coords[1], 4, 4, 2, 2);
			CL_Rectangle r2(plan_coords2, &plan_coords2[1], 4, 4, 2, 2);

			CL_Plane *sel_plane = NULL;
			CL_Rectangle clip;
			if (r1.get_intersection(CL_Rectangle(0,0,1,1), &clip))
			{
				sel_plane = &p1;
			}
			else if (r2.get_intersection(CL_Rectangle(0,0,1,1), &clip))
			{
				sel_plane = &p2;
			}
			if (sel_plane != NULL)
			{
				cl_assert(clip.x1 >= 0 && clip.y1 >= 0 && clip.x2 <=1 && clip.y2 <= 1 &&
					      clip.x1 < clip.x2 && clip.y1 < clip.y2);
				float clip_coords[4*3];
				clip_coords[0] = sel_plane->s.x+clip.x1*sel_plane->e0.x+clip.y1*sel_plane->e1.x;
				clip_coords[1] = sel_plane->s.y+clip.x1*sel_plane->e0.y+clip.y1*sel_plane->e1.y;
				clip_coords[2] = sel_plane->s.z+clip.x1*sel_plane->e0.z+clip.y1*sel_plane->e1.z;

				clip_coords[3] = sel_plane->s.x+clip.x2*sel_plane->e0.x+clip.y1*sel_plane->e1.x;
				clip_coords[4] = sel_plane->s.y+clip.x2*sel_plane->e0.y+clip.y1*sel_plane->e1.y;
				clip_coords[5] = sel_plane->s.z+clip.x2*sel_plane->e0.z+clip.y1*sel_plane->e1.z;

				clip_coords[6] = sel_plane->s.x+clip.x2*sel_plane->e0.x+clip.y2*sel_plane->e1.x;
				clip_coords[7] = sel_plane->s.y+clip.x2*sel_plane->e0.y+clip.y2*sel_plane->e1.y;
				clip_coords[8] = sel_plane->s.z+clip.x2*sel_plane->e0.z+clip.y2*sel_plane->e1.z;

				clip_coords[9] = sel_plane->s.x+clip.x1*sel_plane->e0.x+clip.y2*sel_plane->e1.x;
				clip_coords[10] = sel_plane->s.y+clip.x1*sel_plane->e0.y+clip.y2*sel_plane->e1.y;
				clip_coords[11] = sel_plane->s.z+clip.x1*sel_plane->e0.z+clip.y2*sel_plane->e1.z;

/*				CL_Plane c_p1(
					CL_Vector(c_coords[0], c_coords[1], c_coords[2]),
					CL_Vector(c_coords[3]-c_coords[0], c_coords[4]-c_coords[1], c_coords[5]-c_coods[2]),
					CL_Vector(c_coords[6]-c_coords[0], c_coords[7]-c_coords[1], c_coords[8]-c_coods[2]));

				CL_Plane c_p1(
					CL_Vector(c_coords[9], c_coords[10], c_coords[11]),
					CL_Vector(c_coords[3]-c_coords[9], c_coords[4]-c_coords[10], c_coords[5]-c_coods[11]),
					CL_Vector(c_coords[6]-c_coords[9], c_coords[7]-c_coords[10], c_coords[8]-c_coods[11]));

				float sec_plan_coords[8*2];
				for (int v=0;v<4;v++)
				{
					CL_Line l(
						CL_Vector(coords[v*3+0], coords[v*3+0], coords[v*3+2]),
						CL_Vector(coords[v*3+0]-pos[0], coords[v*3+1]-pos[1], coords[v*3+2]-pos[2]));

					CL_Vector isect = l.intersect_pos(c_p1);
					CL_Vector isect2 = l.intersect_pos(c_p2);
					CL_Vector t = c_p1.plan_coords(isect);
					CL_Vector t2 = c_p2.plan_coords(isect2);

					sec_plan_coords[v*4+0] = t.x;
					sec_plan_coords[v*4+1] = t.y;
					sec_plan_coords2[v*4+2] = t2.x;
					sec_plan_coords2[v*4+3] = t2.y;
				}
				CL_Rectangle cr(sec_plan_coords, &sec_plan_coords[1], 8, 8, 2, 2);
				CL_Rectangle fresult;
				if (cr.get_intersection(CL_Rectangle(0,0,1,1), &fresult))
				{
					float clip_coords_final[4*3];
					clip_coords_final[0] = sel_plane->s.x+clip.x1*sel_plane->e0.x+clip.y1*sel_plane->e1.x;
					clip_coords_final[1] = sel_plane->s.y+clip.x1*sel_plane->e0.y+clip.y1*sel_plane->e1.y;
					clip_coords_final[2] = sel_plane->s.z+clip.x1*sel_plane->e0.z+clip.y1*sel_plane->e1.z;

					clip_coords_final[3] = sel_plane->s.x+clip.x2*sel_plane->e0.x+clip.y1*sel_plane->e1.x;
					clip_coords_final[4] = sel_plane->s.y+clip.x2*sel_plane->e0.y+clip.y1*sel_plane->e1.y;
					clip_coords_final[5] = sel_plane->s.z+clip.x2*sel_plane->e0.z+clip.y1*sel_plane->e1.z;

					clip_coords_final[6] = sel_plane->s.x+clip.x2*sel_plane->e0.x+clip.y2*sel_plane->e1.x;
					clip_coords_final[7] = sel_plane->s.y+clip.x2*sel_plane->e0.y+clip.y2*sel_plane->e1.y;
					clip_coords_final[8] = sel_plane->s.z+clip.x2*sel_plane->e0.z+clip.y2*sel_plane->e1.z;

					clip_coords_final[9] = sel_plane->s.x+clip.x1*sel_plane->e0.x+clip.y2*sel_plane->e1.x;
					clip_coords_final[10] = sel_plane->s.y+clip.x1*sel_plane->e0.y+clip.y2*sel_plane->e1.y;
					clip_coords_final[11] = sel_plane->s.z+clip.x1*sel_plane->e0.z+clip.y2*sel_plane->e1.z;

					glDisable(GL_TEXTURE_2D);
	//				glDisable(GL_DEPTH_TEST);
					glBegin(GL_LINES);
					glColor3f(1,1,0);
					glVertex3fv(clip_coords_final);
					glVertex3fv(clip_coords_final+3);

					glColor3f(1,0,0);
					glVertex3fv(clip_coords_final+3);
					glVertex3fv(clip_coords_final+6);

					glColor3f(0,1,0);
					glVertex3fv(clip_coords_final+6);
					glVertex3fv(clip_coords_final+9);

					glColor3f(0,0,1);
					glVertex3fv(clip_coords_final+9);
					glVertex3fv(clip_coords_final);

					glEnd();
	//				glEnable(GL_DEPTH_TEST);

					show_cubes_rec(cubes[cube_id]->neighbors[i], cube_id, clip_coords_final, rcount+1);
				}*/
				show_cubes_rec(cubes[cube_id]->neighbors[i], cube_id, c_coords, rcount+1);
			}
		}
	}
	else
	{
		for (int i=0;i<6;i++)
		{
			if (cubes[cube_id]->neighbors[i] == -1) break;
			show_cubes_rec(cubes[cube_id]->neighbors[i], cube_id, &cubes[cube_id]->portal_coords[i*4*3], rcount+1);
		}
	}

}

void CL_SceneRenderer::show(const CL_Vector &pos, const CL_Vector &dir, const CL_Vector &up)
{
	float vp[3];
//	vp[0] = 0;
//	vp[1] = 0;
//	vp[2] = 0;
	vp[0] = pos.x;
	vp[1] = pos.y;
	vp[2] = pos.z;
	this->pos = vp;
	this->dir = dir;
	this->up = up;

	static bool show_level = true;

	face_count = 0;

	if (show_level)
	{
		int start_cube = find_cube_location(root, pos);
		if (start_cube < 0 || start_cube >= cubes_size)
		{
			return;
		}
		for (int i=0;i<cubes_size;i++)
		{
			showed_cubes[i] = false;
		}
		std::cout << "Inside cube: " << start_cube << std::endl;
		std::cout << pos.x << ", " << pos.y << ", " << pos.z << std::endl;
		show_cubes_rec(start_cube, -1, NULL, 0);
	}
	else
	{
		show_bsp(root, 1.0f);
	}
}

