/*

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------



*/

#include "Core/precomp.h"

#include <stdio.h>
#include "API/PNG/provider_png.h"

CL_Surface *CL_PNGProvider::create(CL_String file, CL_InputSourceProvider *provider, 
				     bool transparent,
				     bool ignore_alphachannel)
{
  return CL_Surface::create(new CL_PNGProvider(file, provider, 
					       transparent, ignore_alphachannel), true);
}

CL_PNGProvider::CL_PNGProvider(CL_String name, CL_InputSourceProvider *_provider,
				   bool _transparent,
				   bool _ignore_alphachannel)
{
  
	if (_provider == NULL)
	{
		provider = CL_InputSourceProvider::create_file_provider(".");
	}
	else
	{
		provider = _provider->clone();
	}

	ignore_alphachannel = _ignore_alphachannel;
	transparent = _transparent;
	use_alphapixels = transparent && !ignore_alphachannel;

	trans_redcol = 0;
	trans_greencol = 0;
	trans_bluecol = 0;

	trans_col = -1;
	m_uses_src_colorkey = false;

	locked = false;
	filename = name;
	image = color_map = NULL;

}

CL_PNGProvider::~CL_PNGProvider()
{
	perform_unlock();
	delete provider;
}

unsigned int CL_PNGProvider::get_red_mask() const
{
	return CL_Color::get_red_mask(RGBA8888);
}

unsigned int CL_PNGProvider::get_green_mask() const
{
	return CL_Color::get_green_mask(RGBA8888);
}

unsigned int CL_PNGProvider::get_blue_mask() const
{
	return CL_Color::get_blue_mask(RGBA8888);
}

unsigned int CL_PNGProvider::get_alpha_mask() const
{
	return CL_Color::get_alpha_mask(RGBA8888);
}

void CL_PNGProvider::read_header()
{
  //initial fileinfo
  png_read_info(png_ptr, info_ptr); 

  //reduce 16bit/channel to 8Bit/channel
  png_set_strip_16(png_ptr);  

  int bit_depth =  png_get_bit_depth(png_ptr,info_ptr) ;
  int color_type = png_get_color_type(png_ptr,info_ptr);
  
  /* possible color_types :
     --------------------------
     PNG_COLOR_TYPE_GRAY
     (bit depths 1, 2, 4, 8, 16)
     PNG_COLOR_TYPE_GRAY_ALPHA
     (bit depths 8, 16)
     PNG_COLOR_TYPE_PALETTE
     (bit depths 1, 2, 4, 8)
     PNG_COLOR_TYPE_RGB
     (bit_depths 8, 16)
     PNG_COLOR_TYPE_RGB_ALPHA
     (bit_depths 8, 16)
     
     PNG_COLOR_MASK_PALETTE
     PNG_COLOR_MASK_COLOR
     PNG_COLOR_MASK_ALPHA  
  */

  if((color_type != PNG_COLOR_TYPE_RGB_ALPHA) &&
     (color_type != PNG_COLOR_TYPE_RGB) &&
     (color_type != PNG_COLOR_TYPE_PALETTE))
    {
	  std::cerr << "PNGSurface: only the following formats are supported: (by now)";
      std::cerr << "   PNG_COLOR_TYPE_RGB_ALPHA" << std::endl;
      std::cerr << "   PNG_COLOR_TYPE_RGB" << std::endl;
      std::cerr << "   PNG_COLOR_TYPE_PALETTE" << std::endl;
      
      std::cerr << std::endl << "bailing out ......" << std::endl;
      cl_assert(false);
    }

  /* Expand paletted colors into true RGB triplets */
  if (color_type == PNG_COLOR_TYPE_PALETTE)
    {
      std::cerr << "[COLOR_TYPE_PALETTE]";
      png_set_expand(png_ptr);
    }

  if(color_type == PNG_COLOR_TYPE_RGB)
    {
      std::cerr << "[COLOR_TYPE_RGB]";
      //nothing to do here... ;-)

    }


  if(color_type == PNG_COLOR_TYPE_RGB_ALPHA)
    {
      std::cerr << "[COLOR_TYPE_RGB_ALPHA]";
      /* Set the background color to draw transparent and alpha images over.
       */
      png_color_16 my_background ;
      my_background.red = trans_redcol;
      my_background.green = trans_greencol;
      my_background.blue = trans_bluecol;
      
      png_color_16 *image_background;
      
      if (png_get_bKGD(png_ptr, info_ptr, &image_background))
	png_set_background(png_ptr, image_background,
			   PNG_BACKGROUND_GAMMA_FILE, 1, 1.0);
      else
	png_set_background(png_ptr, &my_background,
			   PNG_BACKGROUND_GAMMA_SCREEN, 0, 1.0);
    }
    
  
  if (transparent)
    {
  
      if(info_ptr->num_trans > 0)
	{
	  std::cerr << "PNGSurface : "<< info_ptr->num_trans 
	       << " transparency colors found." << std::endl;
	  int i;
	  for (i=0; i<info_ptr->num_trans; i++) 
	    {
	      std::cerr << "PNGSurface : trans_col[ " << i << "] = " 
		   << info_ptr->trans[i] << std::endl;         
	    }
	  //we have only support for one transparent color?!
	  trans_col = info_ptr->trans[0];

	}
      else
	{
	  //get transparent values from the image:
	  trans_redcol = info_ptr->trans_values.red;
	  trans_greencol = info_ptr->trans_values.green;
	  trans_bluecol = info_ptr->trans_values.blue;
	  
 	  //cerr << "PNGSurface :Transvalue.red: " <<  (int)trans_redcol << endl;
 	  //cerr << "PNGSurface :Transvalue.green: " <<  (int)trans_greencol << endl;
 	  //cerr << "PNGSurface :Transvalue.blue: " << (int)trans_bluecol << endl;
	  
	  trans_col = 
	    (((unsigned int) trans_redcol) << 24) +
	    (((unsigned int) trans_greencol)<<16) +
	    (((unsigned int) trans_bluecol) << 8);
	}
    	m_uses_src_colorkey = true;
    }
  else
    {
      std::cerr << "PNGSurface :Transparency information is ignored." << std::endl;
    } 
  


  //at this point  we have lost alpha-information or never posessed....
  //so fillup ... with 0xff (@pos 4)
  png_set_filler(png_ptr, 0xff, PNG_FILLER_AFTER);  
  
  //reread infostruct to reflect the made settings
  png_read_update_info(png_ptr, info_ptr);  

  no_sprs = 1;
  pitch  = png_get_rowbytes(png_ptr, info_ptr) / 4;   
  width  = png_get_image_width(png_ptr,info_ptr);
  height = png_get_image_height(png_ptr,info_ptr);
  bit_depth =  png_get_bit_depth(png_ptr,info_ptr) ;
  bpp = bit_depth *4;
  color_type = png_get_color_type(png_ptr,info_ptr);


  //cerr << "PNGSurface loaded: " <<width<<" x "<<height<<" with "<< bpp <<" bits/pixel" <<endl;

}
  
void CL_PNGProvider::read_data()
{
 
  image = new unsigned char[pitch * height * 4];
 

  //setup pointers to each row in our target image
  png_bytep *row_pointers=new png_bytep[info_ptr->height];
  for (unsigned int y=0; y<info_ptr->height; y++) {
    row_pointers[y]=image+pitch*y*4;
  }
  
  png_read_image(png_ptr, row_pointers);    


  
}





void *CL_PNGProvider::get_data() const
{
  return image;
}

/*
	Lock the surfaceprovider - which basically means open the file
	and read the image into a temporary memory buffer - until
	unlock() is called.
*/
void CL_PNGProvider::perform_lock()
{
	if (locked) return;

	//setting up PNGLIB stuff
	png_ptr = png_create_read_struct
	  (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (!png_ptr)
	  cl_assert(false);
	info_ptr = png_create_info_struct(png_ptr);
	if (!info_ptr)
	  {
	    png_destroy_read_struct(&png_ptr,
				    (png_infopp)NULL, (png_infopp)NULL);
	    cl_assert(false);
	  }
	end_info = png_create_info_struct(png_ptr);
	if (!end_info)
	  {
	    png_destroy_read_struct(&png_ptr, &info_ptr,
				    (png_infopp)NULL);
	    cl_assert(false);
	  }   
	if (setjmp(png_ptr->jmpbuf)) {
	  png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
	  cl_assert(false);
	}  

	
	cl_assert(provider != NULL);
	input_source = provider->open_source(filename);
	cl_assert(input_source!=NULL);

	no_sprs = 1;

	//tell libpng form whom it get the fileData
	png_set_read_fn(png_ptr, this, &CL_PNGProvider::pngread_file);

	//read INFO-Struct & setup everything
	read_header();
	//actually read data ...
	read_data();

	//remove our data_provider from libpng
	png_set_read_fn(png_ptr,NULL,NULL);

	//free memory ...
	png_destroy_read_struct(&png_ptr, &info_ptr,&end_info);   

	delete input_source;


	locked = true;
}

void CL_PNGProvider::perform_unlock()
{
	locked = false;
	delete[] color_map;
	delete[] image;

	image = color_map = NULL;
}

// Resource support:

class CL_PNG_ResourceSource : public CL_ResourceSource_Surface
{
public:
	virtual const char *get_name() { return "png"; }

	virtual bool can_create(
		std::string ext,
		CL_ResourceOptions *options)
	{
		if (ext == ".png") return true;
		if (options->exists("png")) return true;

		return false;
	}

	virtual CL_SurfaceProvider *create(
		std::string filename,
		CL_ResourceOptions *options,
		CL_ResourceManager *parent)
	{
		return new CL_PNGProvider(filename.c_str(), NULL);
	}

};

static CL_PNG_ResourceSource *res_source_png = NULL;

void CL_SetupPNG::init()
{
	res_source_png = new CL_PNG_ResourceSource;
}

void CL_SetupPNG::deinit()
{
	delete res_source_png;
	res_source_png = NULL;
}
