/*
  Copyright (C) 1999-2001 Ricardo Ueda Karpischek

  This is free software; you can redistribute it and/or modify
  it under the terms of the version 2 of the GNU General Public
  License as published by the Free Software Foundation.

  This software is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this software; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
  USA.
*/

/*

event.c: GUI initialization and event handler

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <fcntl.h>
#include <stdarg.h>
#include "common.h"
#include "gui.h"

/* Event masks */
#define ev1 ButtonPressMask|ButtonReleaseMask|KeyPressMask
#define ev2 ExposureMask|PointerMotionMask|PointerMotionMask
#define ev3 StructureNotifyMask|FocusChangeMask
/*   #define ev3 StructureNotifyMask */
#define evmasks (ev1|ev2|ev3)

/* #define evmasks ((1L<<25)-1) */

/* Indexes of the menus */
int CM_F;
int CM_E;
int CM_T;
int CM_V;
int CM_D;
int CM_O;
int CM_A;
int CM_L;
int CM_R;

/* items on File menu */
int CM_F_LOAD;
int CM_F_SAVE;
int CM_F_SZ;
int CM_F_REP;
int CM_F_QUIT;

/* items on Edit menu */
int CM_E_DOUBTS;
int CM_E_RESCAN;
int CM_E_FILL;
int CM_E_FILL_C;
int CM_E_PP;
int CM_E_PP_C;
int CM_E_SP;
int CM_E_SM;
int CM_E_ST;
int CM_E_SW;
int CM_E_SH;
int CM_E_SN;
int CM_E_DU;
int CM_E_CM;
int CM_E_SETPT;
#ifdef PATT_SKEL
int CM_E_RSKEL;
#endif

/* items on VIEW menu */
int CM_V_SKEL;
int CM_V_BORDER;
int CM_V_HS;
int CM_V_HB;
int CM_V_VHS;
int CM_V_WCLIP;
int CM_V_MAP;
int CM_V_CC;
int CM_V_MAT;
int CM_V_CMP;
int CM_V_MAT_K;
int CM_V_CMP_K;
int CM_V_CLOSURES;
int CM_V_SYMBOLS;
int CM_V_WORDS;
int CM_V_GLINES;
int CM_V_ST;
int CM_V_PRES;
int CM_V_BO;
int CM_V_IO;

/* items on Options menu */
int CM_O_CURR;
int CM_O_ALL;
int CM_O_SMALL;
int CM_O_MEDIUM;
int CM_O_LARGE;
int CM_O_HIDE;
int CM_O_OF;
int CM_O_RS;
int CM_O_BB;
int CM_O_DEF;
int CM_O_DKEYS;
int CM_O_AMENU;
int CM_O_SUM;

/* items on the Alphabet menu */
int CM_A_ARABIC;
int CM_A_CYRILLIC;
int CM_A_GREEK;
int CM_A_HEBREW;
int CM_A_LATIN;
int CM_A_KANA;
int CM_A_NUMBER;
int CM_A_IDEOGRAM;

/* items on the Languages menu */
int CM_L_BR;
int CM_L_DE;
int CM_L_EN;
int CM_L_FR;
int CM_L_GR;
int CM_L_IT;
int CM_L_PT;
int CM_L_RU;
int CM_L_UK;

/* items on Window options menu */
int CM_D_TS;
int CM_D_OS;
int CM_D_HERE;
int CM_D_ADD;
int CM_D_SLINK;
int CM_D_ALINK;
int CM_D_DIS;
int CM_D_SDIAG;
int CM_D_WDIAG;
int CM_D_GM;

/* items on OCR steps menu */
int CM_R_CONS;
int CM_R_OPT;
int CM_R_REV;
int CM_R_BLOCK;
int CM_R_CLASS;
int CM_R_GEOMERGE;
int CM_R_BUILD;
int CM_R_SPELL;
int CM_R_OUTP;
int CM_R_DOUBTS;

/*

Do we have the focus?

*/
int focused = 1;

/* (devel)

Redraw flags
------------

The redraw flags inform the function redraw about which portions
of the application window must be redraw. The precise meaning of
each flag depends on the implementation of the redraw function,
that can be analysed directly on the source code.

    redraw_button .. one specific button or all buttons
    redraw_bg     .. redraw background
    redraw_grid   .. the grid on fatbits windows
    redraw_stline .. the status line
    redraw_dw     .. all visible windows
    redraw_inp    .. all text input fields
    redraw_tab    .. tabs and their labels
    redraw_zone   .. rectangle that defines the zone
    redraw_menu   .. menu bar and currently open menu
    redraw_j1     .. redraw junction 1 (page tab)
    redraw_j2     .. redraw junction 2 (page tab)
    redraw_pbar   .. progress bar
    redraw_map    .. alphabet map

An individual button may be redraw to reflect a status change
(on/off). The junction 1 is the junction of the top and middle
windows on the page tab, and the junction 2 is the junction of
the middle and bottom window on the page tab. The correspondig
flags are used when resizing some window on the page tab.

If redraw_menu is 2, the menu is entirely redrawn. If redraw_menu is
1, then the draw_menu function will redraw only the last selected item
and the newly selected item, except if the menu is being drawn by the
first time.

The progress bar is displayed on the bottom of the window to
reflect the progress of some slow operation. By now, the
progress bar is unused.

*/
int redraw_button,
    redraw_bg,
    redraw_grid,
    redraw_stline,
    redraw_dw,
    redraw_inp,
    redraw_tab,
    redraw_zone,
    redraw_menu,
    redraw_j1,
    redraw_j2,
    redraw_pbar=0,
    redraw_map;

/*

cfont is the display buffer. It contains a matrix of pixels that
will be displayed on the windows PAGE_FATBITS and PATTERN.

*/
char cfont[FS*FS];

/* buffer of messages */
char mb[MMB+1],mba[MMB+1];

/* font used to display text messages, its height and width */
XFontStruct *dfont = NULL;
int DFH,DFW,DFA,DFD;

/* vertical separation between text lines */
int VS = 2;

/*

The windows.

doc  .. the scanned page, reduced mode
fb   .. the scanned page, fat bits mode
f    .. font editor
outp .. the OCR output
sb   .. the symbol browser
dl   .. the list of pages
sl   .. the list of patterns

*/
dwdesc dw[TOP_DW+1];

/*

Restricted clip window for the PAGE window.

*/
int PHR,PVR,PDM,PDT;

/* active window and active mode for each tab */
int CDW=-1;
int PAGE_CM=PAGE,PATTERN_CM=PATTERN,TUNE_CM=TUNE;

/* Xcolors */
XColor white,black,gray,darkgray,vdgray;

/*

BUTTONS

*/
char button[BUTT_PER_COL];
int dmmode=1;

/* the event */
XEvent xev;
XButtonEvent *bev;
int have_s_ev;

/* Context */
char *displayname = NULL;
Display *xd=NULL;
GC xgc;
char *cschema;
int mclip = 0;

/*

The window and its buffer.

TODO: use_xb initial value must be 1 (as buffered I/O is still
experimental, it's currently off by default).

*/
Window XW;
XID xw;
Pixmap pm;
int pmw,pmh;
int have_xw=0,have_pm=0,use_xb=0;

/* default name for each tab */
char *tabl[] = {"page","patterns","tune","history"};

/*

Currently active tab. Possible values are PAGE, PATTERN or
TUNE. These names are just the names of some windows from
these tabs (one for each) re-used to identify each tab.

*/
int tab;

/*

Zoomed pixel size. ZPS must be an odd integer, and may be changed
through the -z command-line flag (please check the currently
limits imposed on ZPS value at the getopt loop).

*/
int ZPS=1;

/* current color */
int COLOR=BLACK;

/* (devel)

The zone
--------

Clara allows to create a zone within the document. By default the
zone is the entire document. The zone limits are given by the
"limits" array. The top left is (limits[0],limits[1]) as the
figure show:

    +-------------+
    |(0,1)   (6,7)|
    |             |
    |(4,5)   (2,3)|
    +-------------+

Currently the zone is useful only to be saved as a new
document. The OCR operations always consider the entire document.

*/
int limits[LIMITS_SIZE];
int limits_top = 0;

/*

To measure the duration of a mouse click.

*/
struct timeval tclick;

/*

These variables are documented below (search the service
enter_wait).

*/
int waiting_key = 0,key_pressed;
char inp_str[MMB+1];
int inp_str_ps;
/*

Create a new menu.

*/
cmdesc *addmenu(int a,char *tt,int m)
{
    /* enlarge CM */
    if (++TOP_CM >= CM_SZ) {
        int a=0;

        /*
            The pointer to the currently active menu must
            be recomputed accordingly to the new CM address.
        */
        if (cmenu != NULL)
            a = cmenu-CM;
        CM = c_realloc(CM,sizeof(cmdesc)*(CM_SZ+=30),NULL);
        if (cmenu != NULL)
            cmenu = CM + a;
    }

    /* Create the menu */
    CM[TOP_CM].a = a;
    strncpy(CM[TOP_CM].tt,tt,MAX_MT);
    CM[TOP_CM].tt[MAX_MT] = 0;
    if (m <= 0)
        m = 5;
    CM[TOP_CM].m = m;
    CM[TOP_CM].n = 0;
    CM[TOP_CM].l = c_realloc(NULL,m*(MAX_MT+1)+m,NULL);
    CM[TOP_CM].t = CM[TOP_CM].l + m*(MAX_MT+1);
    CM[TOP_CM].c = -1;
    return(CM+TOP_CM);
}

/*

Add item to menu.

*/
int additem(cmdesc *m,char *t,int p,int g)
{
    int id;

    if (m->n >= m->m) {
        int o;

        o = m->m;
        m->l = c_realloc(m->l,(m->m+=5)*(MAX_MT+2),NULL);
        m->t = m->l + m->m*(MAX_MT+1);
        memcpy(m->t,m->l+o*(MAX_MT+1),o);
    }
    strncpy(m->l+(MAX_MT+1)*m->n,t,MAX_MT);
    m->l[MAX_MT] = 0;
    m->t[m->n] = p | (g ? CM_NG : 0);

    id = m->n + ((m-CM)<< CM_IT_WD);
    ++(m->n);
    return(id);
}

/* (devel)

The function show_hint
----------------------

Messages are put on the status line (on the bottom of the
application X window) using the show_hint service. The show_hint
service receives two parameters: an integer f and a string (the
message).

If f is 0, then the message is "discardable". It won't be
displayed if a permanent message is currently being displayed.

If f is 1, then the message is "fixed". It won't be erased by a
subsequent show_hint call informing as message the empty string
(in practical terms, the pointer motion won't clear the message).

If f is 2, then the message is "permanent" (the message will be
cleared only by other fixed or permanent message).

If f is 3, any permanent or fixed messages will be discarded.

*/
void show_hint(int f,char *s, ...)
{
    va_list args;
    static int fixed = 0;

    va_start(args,s);

    /*
    printf("request (priority %d) to draw the message %s\n",f,s);
    */

    if (f == 3) {
        redraw_stline = 1;
        mb[0] = 0;
        fixed = 0;
    }

    /* fixed or permanent entry: redraw unconditionally */
    else if (f) {
        if (vsnprintf(mb,MMB,s,args) < 0)
            mb[0] = 0;
        else if ((trace) && (mb[0]!=0))
            tr(mb);
        redraw_stline = 1;
        fixed = f;
    }

    else {

        /*
            If the last hint was fixed, then refresh only if the
            new hint is nonempty.
        */
        if (fixed == 1) {
            if (s[0] != 0) {
                if (vsnprintf(mb,MMB,s,args) < 0)
                    mb[0] = 0;
                redraw_stline = 1;
                fixed = 0;
            }
        }

        /*
            If the last hint was not fixed, then refresh only if
            the hint changed.
        */
        else if ((fixed == 0) && (strcmp(mb,s) != 0)) {
            if (vsnprintf(mb,MMB,s,args) < 0)
                mb[0] = 0;
            redraw_stline = 1;
        }
    }
}

/*

Enter one waiting state, to permit the OCR tasks to accept
asynchronously a user input from the keyboard.

The parameter m is the wait mode (the variable waiting_key stores
the current wait mode):

    Mode 1: wait any key.
    Mode 2: wait ENTER or ESC.
    Mode 3: wait y, Y, n or N.
    Mode 4: wait a string.

Note that all OCR tasks remain stopped while the variable
waiting_key is nonzero.

On cases 2 and 3, the return status will be informed through the
variable key_pressed as follows:

    ENTER, y, or Y .. 1
    ESC, n or N    .. 0

On case 4, the string will return on inp_str.

If f==1, appends the valid inputs to the message to be displayed
on the status line.

*/
void enter_wait(char *s,int f,int m)
{
    char b[MMB+1];

    /* already waiting */
    if (waiting_key)
        return;

    /* build the message to warn the user */
    b[MMB] = 0;
    if (s == NULL)
        b[0] = 0;
    else {
        strncpy(b,s,MMB);
        if (f == 1)
            strncat(b," ",MMB-strlen(b));
    }
    if (f == 1) {
        if (m == 1)
            strncat(b,"(PRESS ANY KEY)",MMB-strlen(mb));
        else if (m == 2)
            strncat(b,"(ENTER to confirm, ESC to ignore)",MMB-strlen(mb));
        else if (m == 3)
            strncat(b,"(y/n)",MMB-strlen(mb));
    }

    /* display the message and enter wait mode */
    show_hint(2,b);
    waiting_key = m;

    /*
        In mode 4, the informed message is preserved because the editing
        service will need to redisplay it.
    */
    strncpy(inp_str,s,MMB);
    inp_str[MMB] = 0;
    inp_str_ps = strlen(inp_str);
}

/*

Step right the current window.

*/
void step_right(void)
{
    if (CDW == PAGE) {
        X0 += FS;
        check_dlimits(0);
        redraw_dw = 1;
    }
    else if (CDW == PAGE_FATBITS) {
        ++X0;
        check_dlimits(0);
        redraw_dw = 1;
    }
    else if (HTML) {
        X0 += DFW;
        check_dlimits(0);
        redraw_dw = 1;
    }
}

/*

Step left the current window.

*/
void step_left(void)
{
    if (CDW == PAGE) {
        X0 -= FS;
        check_dlimits(0);
        redraw_dw = 1;
    }
    else if (CDW == PAGE_FATBITS) {
        --X0;
        check_dlimits(0);
        redraw_dw = 1;
    }
    else if (HTML) {
        if (X0 > 0) {
            X0 -= DFW;
            if (X0 < 0)
                X0 = 0;
            redraw_dw = 1;
        }
    }
}

/*

Step up the current window.

*/
void step_up(void)
{
    if (CDW == PAGE) {
        Y0 -= FS;
        check_dlimits(0);
        redraw_dw = 1;
    }
    else if (CDW == PAGE_FATBITS) {
        --Y0;
        check_dlimits(0);
        redraw_dw = 1;
    }
    else if (HTML) {
        if (Y0 >= DFW) {
            Y0 -= DFW;
            redraw_dw = 1;
        }
    }
}

/*

Step down the current window.

*/
void step_down(void)
{
    if (CDW == PAGE) {
        Y0 += FS;
        check_dlimits(0);
        redraw_dw = 1;
    }
    else if (CDW == PAGE_FATBITS) {
        ++Y0;
        check_dlimits(0);
        redraw_dw = 1;
    }
    else if (HTML) {
        Y0 += DFW;
        check_dlimits(0);
        redraw_dw = 1;
    }
}

/*

Context-dependent "right" action.

*/
void right(void)
{

    if ((dw[PAGE_SYMBOL].v) || (dw[PATTERN].v)) {
        form_auto_submit();
    }

    /* select next menu of the menu bar */
    if (cmenu != NULL) {
        if (cmenu->a == 1) {
            if ((cmenu-CM >= TOP_CM) || ((++cmenu)->a!=1))
                cmenu = CM;
            CY0_orig = -1;
            force_redraw();
            set_mclip(0);
        }
    }

    /* go to the next pattern */
    else if ((CDW == PATTERN) || (dw[TUNE_PATTERN].v)) {
        if (topp >= 0) {

            /* find the next untransliterated pattern */
            if (*cm_e_od != ' ') {
                while ((++cdfc <= topp) &&
                       (pattern[cdfc].tr != NULL));
                if (cdfc > topp)
                    cdfc = 0;
            }

            /* or just go to the next one */
            else if (++cdfc > topp)
                cdfc = 0;

            /* edit it.. */
            edit_pattern();

            /* ..and recompute skeleton */
            dw[TUNE_PATTERN].rg = 1;
            dw[TUNE_SKEL].rg = 1;
            redraw_dw = 1;
        }
    }

    /* move the graphic cursor to the next symbol */
    else if (CDW == PAGE) {
        int l;

        if (curr_mc >= 0) {
            if (mc[curr_mc].E >= 0)
                curr_mc = mc[curr_mc].E;
            else if ((l=rsymb(curr_mc,1)) >= 0)
                curr_mc = l;
            check_dlimits(1);
            redraw_dw = 1;
        }
    }

    /* go to the next pattern type */
    else if (CDW == PATTERN_TYPES) {

        if (cpt < 1)
            cpt = 1;
        else if (cpt < toppt)
            ++cpt;
        else
            cpt = 1;
        dw[PATTERN_TYPES].rg = 1;
        redraw_dw = 1;
    }

    /* scroll right PAGE_FATBITS window */
    else if (CDW == PAGE_FATBITS) {
        ++X0;
        check_dlimits(0);
        redraw_dw = 1;
    }

    /* next revision act */
    else if (CDW == TUNE_ACTS) {
        if (curr_act+1 <= topa) {
            ++curr_act;
            redraw_dw = 1;
            dw[TUNE_ACTS].rg = 1;
        }
    }

    /* scroll right HTML documents */
    else if (HTML) {
        step_right();
    }
}

/*

Context-dependent "left" action.

*/
void left(void)
{

    if ((dw[PAGE_SYMBOL].v) || (dw[PATTERN].v)) {
        form_auto_submit();
    }

    /* select previous menu of the menu bar */
    if (cmenu != NULL) {
        int l;

        if (cmenu->a == 1) {
            l = cmenu-CM;
            for (l=cmenu-CM; (l > 0) && ((--cmenu)->a!=1); --l);
            if (l == 0)
                for (cmenu=CM+(l=TOP_CM); (l > 0) && ((--cmenu)->a!=1); --l);
            CY0_orig = -1;
            force_redraw();
            set_mclip(0);
        }
    }

    /* go to the previous pattern */
    else if ((CDW == PATTERN) || (dw[TUNE_PATTERN].v)) {
        if (topp >= 0) {

            /* find the previous untransliterated pattern */
            if (*cm_e_od != ' ') {
                while ((--cdfc >= 0) &&
                       (pattern[cdfc].tr != NULL));
                if (cdfc < 0)
                    cdfc = topp;
            }

            /* or just go to the previous one */
            else if (--cdfc < 0)
                cdfc = topp;

            /* edit it.. */
            edit_pattern();

            /* ..and recompute skeleton */
            dw[TUNE_PATTERN].rg = 1;
            dw[TUNE_SKEL].rg = 1;
            redraw_dw = 1;
        }
    }

    /* move the current symbol to the previous one */
    else if (CDW == PAGE) {
        int l;

        if (curr_mc >= 0) {
            if (mc[curr_mc].W >= 0)
                curr_mc = mc[curr_mc].W;
            else if ((l=lsymb(curr_mc,1)) >= 0)
                curr_mc = l;
            check_dlimits(1);
            redraw_dw = 1;
        }
    }

    /* go to the previous pattern type */
    else if (CDW == PATTERN_TYPES) {

        if ((0 <= toppt) && (toppt < cpt))
            cpt = toppt;
        else if (cpt > 1)
            --cpt;
        else
            cpt = toppt;
        dw[PATTERN_TYPES].rg = 1;
        redraw_dw = 1;
    }

    /* scroll right PAGE_FATBITS window */
    else if (CDW == PAGE_FATBITS) {
        if (X0 > 0) {
            --X0;
            redraw_dw = 1;
        }
    }

    /* previous revision act */
    else if (CDW == TUNE_ACTS) {
        if (curr_act > 0) {
            --curr_act;
            redraw_dw = 1;
            dw[TUNE_ACTS].rg = 1;
        }
    }

    /* scroll left HTML documents */
    else if (HTML) {
        step_left();
    }
}

/*

Context-dependent "up" action.

*/
void up(void)
{

    /* select previous item of the active menu */
    if (cmenu != NULL) {
        if (cmenu->c > 0)
            --(cmenu->c);
        else
            cmenu->c = cmenu->n - 1;
        if (cmenu->c >= 0) {
            set_mclip(1);
            redraw_menu = 1;
        }
        else
            cmenu->c = -1;
    }

    /* move the graphic cursor to the symbol above the current one */
    else if (CDW == PAGE) {
        int l;

        if (curr_mc >= 0) {
            if (mc[curr_mc].N >= 0)
                curr_mc = mc[curr_mc].N;
            else if ((l=tsymb(curr_mc,1)) >= 0)
                curr_mc = l;
            check_dlimits(1);
            redraw_dw = 1;
        }
    }

    /* scroll down HTML documents by one line */
    else if (HTML) {
        if (Y0 > 0) {
            Y0 -= (DFH+VS);
            if (Y0 < 0)
                Y0 = 0;
            redraw_dw = 1;
        }
    }

    /* scroll up PAGE_FATBITS window */
    else if (CDW == PAGE_FATBITS) {
        step_up();
    }
}

/*

Context-dependent "down" action.

*/
void down(void)
{

    /* select next item of the active menu */
    if (cmenu != NULL) {
        if ((cmenu->c < 0) || (cmenu->c+1 >= cmenu->n))
            cmenu->c = 0;
        else
            ++(cmenu->c);
        if (cmenu->c < cmenu->n) {
            set_mclip(1);
            redraw_menu = 1;
        }
        else
            cmenu->c = -1;
    }

    /* move the graphic cursor to the symbol below the current one */
    else if (CDW == PAGE) {
        int l;

        if (curr_mc >= 0) {
            if (mc[curr_mc].S >= 0)
                curr_mc = mc[curr_mc].S;
            else if ((l=bsymb(curr_mc,1)) >= 0)
                curr_mc = l;
            check_dlimits(1);
            redraw_dw = 1;
        }
    }

    /* scroll down HTML documents by one line */
    else if (HTML) {
        Y0 += DFH+VS;
        check_dlimits(0);
        redraw_dw = 1;
    }

    /* scroll up PAGE_FATBITS window */
    else if (CDW == PAGE_FATBITS) {
        step_down();
    }
}

/*

Context-dependent "prior" (pgup) action.

*/
void prior(void)
{
    int d;

    if (CDW == PAGE) {
        Y0 -= VR;
        check_dlimits(0);
        redraw_dw = 1;
    }

    else if (HTML) {
        d = VR;
        d /= (DFH+VS);
        d *= (DFH+VS);
        Y0 -= d;
        check_dlimits(0);
        redraw_dw = 1;
    }

    else if (CDW == PAGE_FATBITS) {
        Y0 -= FS;
        check_dlimits(0);
        redraw_dw = 1;
    }
}

/*

Context-dependent "next" (pgdw) action.

*/
void next(void)
{
    int d;

    if (CDW == PAGE) {
        Y0 += VR;
        check_dlimits(0);
        redraw_dw = 1;
    }

    else if (HTML) {
        d = VR;
        d /= (DFH+VS);
        d *= (DFH+VS);
        Y0 += d;
        check_dlimits(0);
        redraw_dw = 1;
    }

    else if (CDW == PAGE_FATBITS) {
        Y0 += FS;
        check_dlimits(0);
        redraw_dw = 1;
    }
}

/*

Menu actions.

After catching the selection of one menu item (at mactions_b1),
the associated action is performed here.

*/
void cma(int it)
{
    int new_mclip = 1;
    int a = it & CM_IT_M;
    int cur_dw,dismiss=1;

    /* radio */
    if ((cmenu->t[a] & CM_TM) == CM_TR) {
        char *s,t;
        int i;

        /* status of current item */
        s = cmenu->l + a*(MAX_MT+1)+1;
        t = *s;

        /* search group start */
        for (i=a; (i>=0) && 
                  (((cmenu->t)[a] & CM_TM) == CM_TR) &&
                  (((cmenu->t)[i] & CM_NG) == 0); --i);
        if (i < 0)
            ++i;

        /* deactivate all within group */
        do {
            cmenu->l[i*(MAX_MT+1)+1] = ' ';
            ++i;
        } while ((i<cmenu->n) &&
                 (((cmenu->t)[a] & CM_TM) == CM_TR) &&
                 (((cmenu->t)[i] & CM_NG) == 0));

        /* toggle current */
        if (t == ' ') {
            *s = 'X';
        }
        else {
            *s = ' ';
        }
    }

    /* toggle binary item */
    else if ((cmenu->t[a] & CM_TM) == CM_TB) {
        char *s;

        /* toggle */
        s = cmenu->l + a*(MAX_MT+1)+1;
        *s = (*s == ' ') ? 'X' : ' ';
        redraw_menu = 1;
        dismiss = 0;
    }

    /* "File" menu */
    if (cmenu-CM == CM_F) {

        /* Load page */
        if (it == CM_F_LOAD) {
            setview(PAGE_LIST);
            new_mclip = 0;
        }

        /* save session */
        else if (it == CM_F_SAVE) {
            if (cpage >= 0)
                start_ocr(-2,OCR_SAVE,0);
        }

        /* save zone */
        else if (it == CM_F_SZ) {
            wrzone("zone.pbm",1);
            new_mclip = 0;
        }

        /* write report */
        else if (it == CM_F_REP) {
            int F;

            cur_dw = CDW;
            CDW = PAGE_LIST;
            ge2txt();
            F = open("report.txt",O_WRONLY|O_CREAT|O_TRUNC,0644);
            write(F,text,strlen(text));
            close(F);
            show_hint(2,"Report saved on file report.txt");
            CDW = cur_dw;
            new_mclip = 0;
        }

        /* QUIT the program */
        else if (it == CM_F_QUIT) {
            finish = 2;
            start_ocr(-2,OCR_SAVE,0);
            new_mclip = 0;
        }
    }

    /* "View" menu */
    else if (cmenu-CM == CM_E) {

        /* sort criteria */
        if ((it == CM_E_SM) ||
            (it == CM_E_SP) ||
            (it == CM_E_ST) ||
            (it == CM_E_SW) ||
            (it == CM_E_SH) ||
            (it == CM_E_SN)) {

            dw[PATTERN_LIST].rg = 1;
            new_mclip = 0;
        }

        else if (it == CM_E_DU) {
            rm_untrans();
            dw[PATTERN_LIST].rg = 1;
            new_mclip = 0;
        }

        /* clear matches */
        else if (it == CM_E_CM) {
            clear_cm();
            dw[PATTERN_LIST].rg = 1;
            new_mclip = 0;
        }

        else if (it == CM_E_SETPT) {
            ocr_other = 1;
            start_ocr(-2,OCR_OTHER,0);
            new_mclip = 0;
        }

#ifdef PATT_SKEL
        else if (it == CM_E_RSKEL) {
            reset_skel_parms();
            dw[TUNE_PATTERN].rg = 1;
            dw[TUNE_SKEL].rg = 1;
            new_mclip = 0;
        }
#endif
    }

    /* "View" menu */
    else if (cmenu-CM == CM_V) {

        /* set FAT BITS mode */
        /*
        if ((cpage >= 0) && (it == CM_V_FAT)) {
            if (CDW != PAGE_FATBITS) {
                if ((CX0_orig >= DM) && (CY0_orig >= DT)) {
                    X0 = (CX0_orig-DM) * RF;
                    Y0 = (CY0_orig-DT) * RF;
                }
                setview(PAGE_FATBITS);
                check_dlimits(1);
                new_mclip = 0;
            }
        }
        */

        /* Toggle view HTML source flag: must regenerate HTML documents */
        if (it == CM_V_VHS) {
            int i;

            for (i=0; i<=TOP_DW; ++i)
                if (dw[i].html)
                    dw[i].rg = 1;
            new_mclip = 0;
            dismiss = 1;
        }

        /* recompute form contents */
        else if (it == CM_V_WCLIP) {
            if ((dw[PAGE_SYMBOL].v) && (curr_mc >= 0)) {
                dw[PAGE_SYMBOL].rg = 1;
                redraw_dw = 1;
            }
            else if ((dw[PATTERN].v) && (0<=cdfc) && (cdfc<=topp)) {
                dw[PATTERN].rg = 1;
                redraw_dw = 1;
            }
            new_mclip = 0;
            dismiss = 1;
        }

        /* toggle alphabet mapping display */
        else if (it == CM_V_MAP) {

            /* recompute window size and the placement of the components */
            comp_wnd_size(-1,-1);

            /* recompute the placement of the windows */
            setview(CDW);

            new_mclip = 0;
            dismiss = 1;
        }

        /* change PAGE display mode */
        else if ((it == CM_V_CLOSURES) ||
                 (it == CM_V_SYMBOLS) ||
                 (it == CM_V_WORDS)) {

            new_mclip = 0;
            redraw_dw = 1;
            dismiss = 1;
        }

        /* change PAGE display mode */
        else if (it == CM_V_PRES) {

#ifdef HAVE_SIGNAL
            if (pmode == 0)
                cpresentation(1);
#else
            show_hint(1,"Presentation unsupported!");
#endif
            new_mclip = 0;
            dismiss = 1;
        }

        /*
            when changing these presentation flags, some
            windows must be regenerated.
        */
        else if ((it == CM_V_SKEL) || (it == CM_V_BORDER)) {

            /* must copy pattern to cfont again */
            p2cf();
        }

        /*
            Clear status line to display progress warnings.
        */
        else if ((it == CM_V_MAT) || (it == CM_V_CMP) ||
                 (it == CM_V_MAT_K) || (it == CM_V_CMP_K)) {

            show_hint(1,"");
        }
    }

    /* "Window options" menu */
    else if (cmenu-CM == CM_D) {
        int i,j,x,y,t;

        {
            int last_dw;

            last_dw = CDW;
            CDW = PAGE;
            i = (CX0_orig-DM) / GS;
            j = (CY0_orig-DT) / GS;
            x = X0 + i*RF;
            y = Y0 + j*RF;
            t = thisone(x,y);
            CDW = last_dw;
        }

        /* use as pattern */
        if ((t>=0) && (it == CM_D_TS)) {
            new_mclip = 0;
            this_pattern = mc[t].bm;
        }

        /* OCR this symbol */
        else if ((t>=0) && (it == CM_D_OS)) {
            if (ocring == 0) {
                new_mclip = 0;
                justone = 1;
                curr_mc = t;
                start_ocr(-2,OCR_CLASS,1);
            }
        }

        /* make (CX0_orig,CY0_orig) the bottom left */
        else if (it == CM_D_HERE) {
	    
            i = (CX0_orig-DM) / GS;
            j = (CY0_orig-DT) / GS;

            if ((i > 0) && (j > 0)) {
                X0 += i * RF;
                Y0 -= (VR - j - 1) * RF;
                check_dlimits(1);
            }
            new_mclip = 0;
        }

        /* merge fragment with the current symbol */
        else if (it == CM_D_ADD) {

            if ((curr_mc >= 0) && (t >= 0)) {
                start_ocr(-2,OCR_REV,0);
                to_arg = t;
                to_rev = REV_MERGE;
            }

            new_mclip = 0;
        }

        /* create symbol link */
        else if (it == CM_D_SLINK) {

            if ((t >= 0) && (mc[t].sw<0)) {
                start_ocr(-2,OCR_REV,0);
                to_arg = t;
                to_rev = REV_SLINK;
            }

            new_mclip = 0;
        }

        /* disassemble symbol */
        else if (it == CM_D_DIS) {

            if ((curr_mc >= 0) && (t >= 0)) {
                start_ocr(-2,OCR_REV,0);
                to_rev = REV_DIS;
            }

            new_mclip = 0;
        }

        /* create accent link */
        else if (it == CM_D_ALINK) {

            if ((t >= 0) && (mc[t].sw<0)) {
                start_ocr(-2,OCR_REV,0);
                to_arg = t;
                to_rev = REV_ALINK;
            }

            new_mclip = 0;
        }

        /* diagnose symbol pairing */
        else if (it == CM_D_SDIAG) {

            if ((curr_mc >= 0) && (t >= 0))
                diag_pairing(t);

            new_mclip = 0;
        }

        /* diagnose word pairing */
        else if (it == CM_D_WDIAG) {

            if ((curr_mc >= 0) && (t >= 0))
                diag_wpairing(t);

            new_mclip = 0;
        }

        /* diagnose merging */
        else if (it == CM_D_GM) {

            if ((curr_mc >= 0) && (t >= 0))
                diag_geomerge(t);

            new_mclip = 0;
        }
    }

    /* "Options" menu */
    else if (cmenu-CM == CM_O) {

        /* change font */
        if ((it == CM_O_SMALL) ||
            (it == CM_O_MEDIUM) ||
            (it == CM_O_LARGE) ||
            (it == CM_O_DEF)) {

            /* set new font */
            set_xfont();

            /* recompute window size and components placement */
            comp_wnd_size(-1,-1);
            set_mclip(0);

            /* recompute the placement of the windows */
            setview(CDW);

            /* must regenerate HTML documents */
            {
                int i;

                for (i=0; i<=TOP_DW; ++i)
                    if (dw[i].html)
                        dw[i].rg = 1;
                new_mclip = 0;
            }

            new_mclip = 0;
            dismiss = 1;
        }

        /*
            Recompute the geometry and redisplay the window to
            reflect the new value of the flag *cm_hide.
        */
        else if (it == CM_O_HIDE) {
            setview(CDW);
            new_mclip = 0;
            dismiss = 1;
        }

        /* generate again the list of patterns */
        else if (it == CM_O_OF) {
            dw[PATTERN_LIST].rg = 1;
            new_mclip = 0;
        }

        /* report scale */
        else if (it == CM_O_RS) {
            new_mclip = 0;
            redraw_tab = 1;
            dismiss = 1;
        }

        /* report scale */
        else if (it == CM_O_BB) {
            new_mclip = 0;
            redraw_tab = 1;
            if (dw[PAGE].v)
                redraw_dw = 1;
            dismiss = 1;
        }
    }

    /* "Pattern" menu */
    else if (cmenu-CM == CM_R) {
        int pages;

        pages = (*cm_o_all != ' ') ? -1 : -2;

        /* cannot start new operation */
        if (ocring) {
        }

        /* start consisting structures */
        else if (it == CM_R_CONS) {
            start_ocr(pages,OCR_CONS,0);
        }

        /* start preparing patterns */
        else if (it == CM_R_OPT) {
            start_ocr(pages,OCR_OPT,0);
        }

        /* start reading revision data */
        else if (it == CM_R_REV) {
            start_ocr(pages,OCR_REV,0);
        }

        /* start detection of blocks */
        else if (it == CM_R_BLOCK) {
            start_ocr(pages,OCR_BLOCK,0);
        }

        /* start classification */
        else if (it == CM_R_CLASS) {
            start_ocr(pages,OCR_CLASS,0);
        }

        /* start geometric merging */
        else if (it == CM_R_GEOMERGE) {
            start_ocr(pages,OCR_GEOMERGE,0);
        }

        /* start */
        else if (it == CM_R_BUILD) {
            start_ocr(pages,OCR_BUILD,0);
        }

        /* start spelling */
        else if (it == CM_R_SPELL) {
            start_ocr(pages,OCR_SPELL,0);
        }

        /* start building words and lines */
        else if (it == CM_R_OUTP) {
            start_ocr(pages,OCR_OUTP,0);
        }

        /* start generating doubts */
        else if (it == CM_R_DOUBTS) {
            start_ocr(pages,OCR_DOUBTS,0);
        }

        new_mclip = 0;
    }

    /* "Alphabets" menu */
    else if (cmenu-CM == CM_A) {

        set_bl_alpha();
        redraw_map = 1;
    }

    /* must dismiss the context menu */
    if (dismiss) {
        cmenu = NULL;
        CX0_orig = CY0_orig = -1;

        /* workaround for some PAGE redraw problems */
        if (dw[PAGE].v)
            new_mclip = 0;

        set_mclip(new_mclip);
        force_redraw();
    }
    else {
        redraw_menu = 1;
    }
}

/*

Press key (for presentations).

*/
int press(char c,int *delay)
{
    xev.type = KeyPress;
    switch (c) {
        case 'r': xev.xkey.keycode = 27; break;
        case 'h': xev.xkey.keycode = 43; break;
    }
    have_s_ev = 1;
    *delay = 500000;
    return(0);
}

/*

Click menu item (for presentations).

*/
int mselect(int item,int *delay)
{
    int m,x,y=MH/2,px,py;
    XButtonEvent *be;

    m = item >> CM_IT_WD;
    x = ((CM[m].p*DFW) + ((CM[m].p+strlen(CM[m].tt))*DFW)) / 2;

    /* get pointer coordinates */
    get_pointer(&px,&py);

    /* go to menu */
    if (((cmenu == NULL) || ((cmenu-CM) != m)) &&
         ((px != x) || (py != y))) {

        XWarpPointer(xd,None,XW,0,0,0,0,x,y);
        *delay = 1000000;
        return(1);
    }

    /* open menu */
    else if (cmenu == NULL) {

        xev.type = ButtonPress;
        xev.xbutton.x = px;
        xev.xbutton.y = py;
        be = (XButtonEvent *) &xev;
        be->button = Button1;
        have_s_ev = 1;
        *delay = 500000;
        return(1);
    }

    /* no item selected */
    else if ((cmenu->c < 0) ||
             ((cmenu->c + ((cmenu-CM) << CM_IT_WD)) != item)) {

        int i;

        i = item & CM_IT_M;
        x = CX0 + CW - 10;
        y = i * (DFH + VS) + 5 + CY0 + DFH;
        XWarpPointer(xd,None,XW,0,0,0,0,x,y);
        *delay = 2000000;
        return(1);
    }

    /* click it */
    else {

        xev.type = ButtonPress;
        xev.xbutton.x = px;
        xev.xbutton.y = py;
        be = (XButtonEvent *) &xev;
        be->button = Button1;
        have_s_ev = 1;
        *delay = 100000;
        return(0);
    }
}

/*

Click symbol (for presentations).

*/
int sselect(int k,int *delay)
{
    int x,y,px,py;
    XButtonEvent *be;

    /* get pointer coordinates */
    get_pointer(&px,&py);

    /* go to tab */
    if (!dw[PAGE].v) {
        y = PT + TH/2;
        x = PM + 10 + (PAGE) * (20+TW) + 10 + TW/2;

        /* move pointer */
        if ((abs(px-x) > (TW/4)) || (abs(py-y) > TH/4))  {
            XWarpPointer(xd,None,XW,0,0,0,0,x,y);
            *delay = 1000000;
            return(1);
        }

        /* click the tab */
        else {
            xev.type = ButtonPress;
            xev.xbutton.x = px;
            xev.xbutton.y = py;
            be = (XButtonEvent *) &xev;
            be->button = Button1;
            have_s_ev = 1;
            *delay = 500000;
            return(1);
        }
    }

    /* make sure that symbol k is visible */
    else {
        int w,h;

        curr_mc = k;
        CDW = PAGE;
        x = DM + (((mc[k].l + mc[k].r) / 2) - X0) / RF;
        y = DT + (((mc[k].t + mc[k].b) / 2) - Y0) / RF;
        w = (mc[k].r - mc[k].l + 1) / RF;
        h = (mc[k].b - mc[k].t + 1) / RF;

        /* make the symbol visible */
        if ((mc[curr_mc].l < X0) ||
            (mc[curr_mc].r >= X0 + HR) ||
            (mc[curr_mc].b < Y0) ||
            (mc[curr_mc].t >= Y0 + VR)) {

            check_dlimits(1);
            force_redraw();
            *delay = 1000000;
            return(1);
        }

        /* move to symbol */
        else if ((abs(px-x) > w/4) || (abs(py-y) > w/4)) {

            XWarpPointer(xd,None,XW,0,0,0,0,x,y);
            *delay = 1000000;
            return(1);
        }

        /* click it */
        else {
            xev.type = ButtonPress;
            xev.xbutton.x = px;
            xev.xbutton.y = py;
            be = (XButtonEvent *) &xev;
            be->button = Button1;
            have_s_ev = 1;
            *delay = 500000;
            return(0);
        }
    }
}

/*

Click button (for presentations).

*/
int bselect(int k,int *delay)
{
    int x,y,px,py;
    XButtonEvent *be;

    /* get pointer coordinates */
    get_pointer(&px,&py);

    {
        int w,h;

        curr_mc = k;
        CDW = PAGE;
        x = 10 + BW/2;
        y = PT + k * (BH+VS) + BH/2;
        w = BW;
        h = BH;

        /* warp to button */
        if ((abs(px-x) > w/4) || (abs(py-y) > w/4)) {

            XWarpPointer(xd,None,XW,0,0,0,0,x,y);
            *delay = 1000000;
            return(1);
        }

        /* click it */
        else {
            xev.type = ButtonPress;
            xev.xbutton.x = px;
            xev.xbutton.y = py;
            be = (XButtonEvent *) &xev;
            be->button = Button1;
            have_s_ev = 1;
            *delay = 30000000;
            return(0);
        }
    }
}

/*

Load file (for presentations).

*/
int fselect(char *p,int *delay)
{
    int x,y,px,py;
    XButtonEvent *be;

    /* get pointer coordinates */
    get_pointer(&px,&py);

    /* go to tab */
    if (!dw[PAGE_LIST].v) {
        y = PT + TH/2;
        x = PM + 10 + (PAGE) * (20+TW) + 10 + TW/2;

        /* move pointer */
        if ((abs(px-x) > (TW/4)) || (abs(py-y) > TH/4))  {
            XWarpPointer(xd,None,XW,0,0,0,0,x,y);
            *delay = 1000000;
            return(1);
        }

        /* click the tab */
        else {
            xev.type = ButtonPress;
            xev.xbutton.x = px;
            xev.xbutton.y = py;
            be = (XButtonEvent *) &xev;
            be->button = Button1;
            have_s_ev = 1;
            *delay = 500000;
            return(1);
        }
    }

    /* make sure that file imre.pbm is visible */
    else {
        int i,w,h;

        for (i=0; (i<=topge) &&
                  ((ge[i].type != GE_TEXT) ||
                   (strcmp(p,ge[i].txt) != 0)); ++i);

        if (i > topge) {

            printf("nao achei %s\n",p);
            return(0);
        }

        curr_hti = i;

        x = DM + (((ge[i].l + ge[i].r) / 2) - X0) / RF;
        y = DT + (((ge[i].t + ge[i].b) / 2) - Y0) / RF;
        w = (ge[i].r - ge[i].l + 1) / RF;
        h = (ge[i].b - ge[i].t + 1) / RF;

        /* make the link visible */
        if ((ge[curr_hti].l < X0) ||
            (ge[curr_hti].r >= X0 + HR) ||
            (ge[curr_hti].b < Y0) ||
            (ge[curr_hti].t >= Y0 + VR)) {

            check_dlimits(1);
            force_redraw();
            *delay = 1000000;
            return(1);
        }

        /* move to link */
        else if ((abs(px-x) > w/4) || (abs(py-y) > w/4)) {

            XWarpPointer(xd,None,XW,0,0,0,0,x,y);
            *delay = 1000000;
            return(1);
        }

        /* click it */
        else {
            xev.type = ButtonPress;
            xev.xbutton.x = px;
            xev.xbutton.y = py;
            be = (XButtonEvent *) &xev;
            be->button = Button1;
            have_s_ev = 1;
            *delay = 5000000;
            return(0);
        }
    }
}

/*

Process keyboard events.

*/
void kactions(void)
{
    /* statuses */
    int ctrl,haveif;
    static int cx=0;

    /* other */
    int ls;
    edesc *c;
    int n;

    /* Buffers required by the KeyPress event */
    unsigned char buffer[101];
    KeySym ks;
    XComposeStatus cps;

    /*
        We're waiting any keyboard event.
    */
    /* any keyboard event unblocks the OCR */
    if (waiting_key == 1) {
        waiting_key = 0;
        if (nopropag == 0)
            return;
    }

    n = XLookupString(&(xev.xkey),(char *)buffer,100,&ks,&cps);

    /*
        We're waiting ENTER or ESC keypresses. Discard anything else.
    */
    if (waiting_key == 2) {
        if ((ks == 0xff0d) || (ks == 0xff1b)) {
            waiting_key = 0;
            mb[0] = 0;
            redraw_stline = 1;
            key_pressed = (ks == 0xff0d);
        }
        return;
    }

    /*
        We're waiting y, Y, n or N keypresses. Discard anything else.
    */
    /* Discard events other than y, Y, n or N */
    if (waiting_key == 3) {
        if ((ks == 'y') || (ks == 'Y') || (ks == 'n') || (ks == 'N')) {
            waiting_key = 0;
            mb[0] = 0;
            redraw_stline = 1;
            key_pressed = ((ks == 'y') || (ks == 'Y'));
        }
        return;
    }

    /* the C-g prefix */
    ctrl = (xev.xkey.state & ControlMask);
    if (ctrl && (ks == 103)) {

        cx = 0;
        return;
    }

    /* the C-x prefix */
    ctrl = (xev.xkey.state & ControlMask);
    if (ctrl && (ks == 120)) {

        cx = 1;
        return;
    }

    /* C-x special actions */
    if (cx) {

        switch (ks) {

            /* dump current symbol */
            case 100:
                if ((curr_mc < 0) || (tops < curr_mc)) {
                    warn("invalid symbol");
                }
                else if (pixel_mlist(curr_mc) < 0) {
                    warn("symbol too large");
                }
                else {
                    dump_cb();
                }
                break;

            /*

                ** EXPERIMENTAL STUFF **
                ------------------------

            */

            /* Candido de Figueiredo block separation */
            case 115:
                cf_block();
                break;

            /* reset */
            case 114:
                reset();
                break;

        }
        cx = 0;
        return;
    }

    /* Is there an active text input field? */
    c = ge + curr_hti;
    if ((0 <= curr_hti) && (curr_hti <= topge) && (c->type == GE_INPUT)) {
        haveif = 1;

        /* ignore ESC prefix */
        if (nopropag) {
            nopropag = 0;
            mb[0] = 0;
            redraw_stline = 1;
        }
    }
    else if (waiting_key == 4)
        haveif = 2;
    else
        haveif = 0;
    
    /* Control-L */
    if ((n == 1) && (buffer[0] == 12))
        force_redraw();

    /* Backspace within text input field */
    else if ((n == 1) && (haveif) && (ks == 0xffff)) {

        if (haveif == 1) {
            if ((ls=strlen(c->txt)) > 0)
                (c->txt)[ls-1] = 0;

            redraw_inp = 1;
            form_changed = 1;
        }
        else {
            if ((ls=strlen(inp_str)) > inp_str_ps)
                (inp_str)[ls-1] = 0;

            redraw_inp_str();
        }
    }

    /* Visible char */
    else if ((n == 1) && (' ' <= buffer[0])) {
        static int l=-1;
        int v;

        /* apply composition */
        if (*cm_o_dkeys == 'X')
            v = compose(&l,buffer[0]);
        else {
            v = 1;
            l = buffer[0];
        }

        /* add to HTML input field */
        if (haveif == 1) {
            ls = strlen(c->txt) + 1;
            if (v) {
                if ((ls >= c->tsz) && (c->tsz < MFTL+1))
                    c->txt = c_realloc(c->txt,c->tsz=MFTL+1,NULL);
                if (ls < c->tsz) {
                    (c->txt)[ls-1] = l;
                    (c->txt)[ls] = 0;
                    l = -1;
                }
            }
            redraw_inp = 1;
            form_changed = 1;
        }

        /* add to HTML input field */
        else if (haveif == 2) {
            ls = strlen(inp_str) + 1;
            if (v) {
                if (ls < MMB) {
                    (inp_str)[ls-1] = l;
                    (inp_str)[ls] = 0;
                    l = -1;
                }
            }
            redraw_inp_str();
        }

        /* submit symbol transliteration from PAGE window */
        else if (v && dw[PAGE].v && (curr_mc >= 0)) {

            /*
                Obs. we initialize the transliteration buffer to_tr
                after the start_ocr call becase start_ocr clear the
                buffer to_tr.
            */
            start_ocr(-2,OCR_REV,0);
            to_tr[0] = l;
            to_tr[1] = 0;
            to_rev = REV_TR;
            l = -1;
        }
    
        /* submit symbol transliteration from PATTERN window */
        else if (v && dw[PATTERN].v && (cdfc >= 0)) {
            pdesc *s;

            /* pattern is not from the current page */
            s = pattern + cdfc;
            if (strcmp(s->d,pagename) != 0)
                return;

            /*
               select the symbol from which the bitmap
               came from.
            */
            curr_mc = s->e;

            /*
                Obs. we initialize the transliteration buffer to_tr
                after the start_ocr call becase start_ocr clear the
                buffer to_tr.
            */
            start_ocr(-2,OCR_REV,0);
            to_tr[0] = l;
            to_tr[1] = 0;
            to_rev = REV_TR;
            l = -1;
        }

        /*
            switch page (for tests)
            -----------------------

            Manually reads the first, second or third page. To use
            this feature, press "1", "2" or "3" on the PAGE (list)
            window (of course there must exist at least pages on
            the current directory). This is used only for debugging
            purposes.
        */
        /*
        else if (CDW == PAGE_LIST) {
            if (l == '1')
                start_ocr(0,OCR_LOAD,0);
            else if (l == '2')
                start_ocr(1,OCR_LOAD,0);
            else if (l == '3')
                start_ocr(2,OCR_LOAD,0);
            l = -1;
        }
        */
    
        /* show the GPL */
        else if (v && ((l=='l') || (l=='L')) && (CDW == WELCOME)) {
            setview(GPL);
            redraw_dw = 1;
            l = -1;
        }
    }

    /* Special keys */
    else switch (ks) {

        /* Left */
        case 0xff51:
            left();
            break;
    
        /* Up */
        case 0xff52:
            up();
            break;
    
        /* Right */
        case 0xff53:
            right();
            break;
    
        /* Down */
        case 0xff54:
            down();
            break;
    
        /* Prior (pgup) */
        case 0xff55:
            prior();
            break;
    
        /* Next (pgdw) */
        case 0xff56:
            next();
            break;
    
        /* END */
        case 0xff57:
            if ((XRES - HR) >= 0) {
                X0 = XRES - HR;
            }
            break;
    
        /* HOME */
        case 0xff50:
            X0 = 0;
            break;
    
        /* ENTER */
        case 0xff0d:

            /* select current menu item */
            if (cmenu != NULL) {
                cma(cmenu->c + ((cmenu-CM) << CM_IT_WD));
            }

            /*
                ENTER pressed with active input field
                provoke form submission.
            */
            else if (((CDW == PAGE_SYMBOL) ||
                      (CDW == TUNE) ||
                      (CDW == TUNE_SKEL) ||
                      (CDW == PATTERN)) &&

                     (haveif == 1)) {

                form_auto_submit();
            }

            /* finished edition of input string */
            else if (haveif == 2) {
                waiting_key = 0;
                redraw_inp_str();
                memmove(inp_str,inp_str+inp_str_ps,strlen(inp_str)-inp_str_ps+1);
            }

            break;

        /* ESC */
        case 0xff1b:

            /* dismiss current menu */
            if (cmenu != NULL)
                dismiss(1);

            /* toggle BAD property */
            else {

                button[bbad] ^= 1;
                redraw_button = bbad;
                form_changed = 1;
                form_auto_submit();

                /*
                nopropag = 1-nopropag;
                if (nopropag)
                    enter_wait("ESC",0,1);
                */
            }

            /* skip current doubt */
            /*
            else if (CDW == PAGE_SYMBOL) {
                ++curr_doubt;
                edit_curr_doubt();
            }
            */

            break;

        /* C-x prefix */
        /* case 0xff1b: */

        default:
            db("unattended keycode %ld",ks);
            break;
    }
}

/*

Select the window where the mouse pointer is.

*/
void select_dw(int x,int y)
{
    int a,last_dw;

    get_pointer(&x,&y);

    /* loop all active windows */
    last_dw = CDW;

    if (*cm_o_hide == ' ')
        a = 10+RW;
    else
        a = 0;

    for (CDW=0; CDW<=TOP_DW; ++CDW) {
        if ((dw[CDW].v) &&
            (DM-10 <= x) && (x < DM+DW+(DS?a:10)) &&
            (DT-10 <= y) && (y < DT+DH+(DS?a:10))) {

            if (CDW != last_dw) {
                redraw_tab = 1;
            }
            return;
        }
    }

    /* no one was selected */
    CDW = last_dw;
}

/*

Detect the item from the menu bar currently under the pointer.

*/
int mb_item(int x,int y)
{
    if ((0 < y) && (y < MH)) {
        int i;

        for (i=0; i<=TOP_CM; ++i) {
            if ((CM[i].a == 1) &&
                (CM[i].p * DFW <= x) &&
                (x <= (CM[i].p+strlen(CM[i].tt)) * DFW)) {
                    return(i);
            }
        }
        return(-2);
    }
    return(-1);
}

/*

Mouse actions activated by mouse button 1.

*/
int mactions_b1(int help,int x,int y)
{
    int i,j,c,sb;
    int pb,ob;

    /* coordinates depends on scrollbars hiding */
    if (*cm_o_hide == ' ')
        sb = 10 + RW;
    else
        sb = 0;

    /*

        Context menu
        ------------

    */
    if (cmenu != NULL) {
        int n;

        if (help) {

            /* reselect automagically */
            n = mb_item(x,y);
            if ((n >= 0) && (n != cmenu-CM)) {
                dismiss(1);
                CX0 = x;
                CY0 = y;
                cmenu = CM+n;
                set_mclip(0);
                return(1);
            }

            show_hint(0,"");
            return(1);
        }

        /* process current item */
        if (cmenu->c >= 0) {
            cma(cmenu->c + ((cmenu-CM) << CM_IT_WD));
            return(0);
        }

        /* dismiss context menu */
        else {
            dismiss(1);
        }
    }

    /* change to selected window */
    select_dw(x,y);

    /*

        Menu bar
        --------

    */
    if ((i=mb_item(x,y)) >= 0) {

        if (help) {

            /* select automagically */
            if ((*cm_o_amenu == 'X') && (cmenu == NULL)) {
                CX0 = x;
                CY0 = y;
                cmenu = CM+i;
                redraw_menu = 1;
                return(1);
            }

            show_hint(0,"");
            return(1);
        }

        CX0 = x;
        CY0 = y;
        if (cmenu == NULL)
            form_auto_submit();
        cmenu = CM+i;
        redraw_menu = 1;
        return(0);
    }

    /*

        Selection of tabs
        -----------------

    */
    if ((PM+10 <= x) &&
        (x < PM+10+(TABN-1)*(TW+20)+TW) &&
        (PT <= y) && (y <= PT+TH) &&
        ((x - PM - 10) % (20 + TW) <= TW)) {

        i = (x - PM - 10) / (20 + TW);

        /*
           cannot circulate modes during OCR because the
           data structures are not ready (in some cases
           would segfault).
        */
        if (ocring) {

            /* do nothing */
        }

        /* show document */
        else if (i == PAGE) {

            if (help) {
                if (tab == PAGE)
                    if (cpage >= 0)
                        show_hint(0,"click to circulate modes");
                    else
                        show_hint(0,"select a file to load");
                else
                    show_hint(0,"click to enter page modes");
                return(1);
            }

            if ((CDW == PAGE_FATBITS) || (cpage == -1))
                setview(PAGE_LIST);
	    
            else if (CDW == PAGE_LIST)
                setview(PAGE);
	    
            else if (dw[PAGE].v)
                setview(PAGE_FATBITS);
	    
            else
                setview(PAGE_CM);
        }

        /* font editor */
        else if (i == PATTERN) {

            if (help) {
                if (tab == PATTERN)
                    show_hint(0,"click to circulate modes");
                else
                    show_hint(0,"click to browse and edit the patterns");
                return(1);
            }

            if (tab != PATTERN) {

                /*
                    BUG: this call will renderize again the PROPS
                    form. So if the user edit the input buffer,
                    change the tab, and after go back to PATTERN,
                    the data on the input field will be lost.
                */
                /*
                edit_pattern();
                */

                setview(PATTERN_CM);
            }
            else if (dw[PATTERN].v)
                setview(PATTERN_LIST);

            else if (dw[PATTERN_LIST].v)
                setview(PATTERN_TYPES);

            else
                setview(PATTERN);
        }

        /* broser of acts */
        else if (i == TUNE) {

            if (help) {
                if (tab != TUNE)
                    show_hint(0,"click to tune the OCR");
                return(1);
            }

            if (tab != TUNE) {

                setview(TUNE_CM);
            }
            else if (dw[TUNE].v) {

                setview(TUNE_SKEL);
            }
            else if (dw[TUNE_SKEL].v) {

                setview(TUNE_ACTS);
            }
            else
                setview(TUNE);
        }
    }

    /*

        Alphabet map
        ------------

    */
    else if ((*cm_v_map != ' ') &&
             (alpha != NULL) &&
             (PM-50 <= x) && (x < PM-50+alpha_cols*12+alpha_cols-1) &&
             (PT+1 <= y) && (y < PT+1+alpha_sz*15+alpha_sz-1)) {

        /* map column */
        i = (x-PM+50) / 13;

        /* map row */
        j = (y-PT-1) / 16;

        if (help) {
            char lb[MMB+1];

            lb[0] = lb[MMB] = 0;
            if ((0 <= j) && (j < alpha_sz)) {

                /* digits */
                if ((i == 0) && (alpha == number))
                    snprintf(lb,MMB,"this is the digit \"%s\"",alpha[j].ln);

                /* first column is the capital case */
                else if ((i == 0) && (alpha_cols >= 2) && (alpha[j].cc > 0))
                    snprintf(lb,MMB,"this is the letter \"%s\", capital",alpha[j].ln);

                /* second column is the small case */
                else if ((i == 1) && (alpha_cols >= 2) && (alpha[j].sc > 0))
                    snprintf(lb,MMB,"this is the letter \"%s\", small",alpha[j].ln);

                /* first column for alphabets without small case */
                else if ((i == 0) && (alpha_cols < 2) && (alpha[j].cc > 0))
                    snprintf(lb,MMB,"this is the letter \"%s\"",alpha[j].ln);

                /* third column is the transliteration */
                else if ((i == 2) && (alpha_cols >= 3))
                    snprintf(lb,MMB,"this is the latin keyboard map for letter \"%s\"",alpha[j].ln);
            }
            show_hint(0,lb);
            return(1);
        }
    }

    /*

        window
        ------

    */
    else if ((DM <= x) && (x < DM+DW) &&
             (DT <= y) && (y < DT+DH)) {

        i = (x-DM) / GS;
        j = (y-DT) / GS;

#ifdef FUN_CODES
        /*
            Toggle fun code 1.
        */
        if (CDW == WELCOME) {

            if (help) {
                return(1);
            }

            if (fun_code == 0) {
                fun_code = 1;
                redraw_dw = 1;
                new_alrm(50000,1);
            }
            else if (fun_code == 1) {
                fun_code = 0;
                redraw_dw = 1;
            }

            return(0);
        }
#endif

        /*
            define OCR zone

               +-----+
               |1   4|
               |     |
               |3   2|
               +-----+
        */
        if ((dw[PAGE].v) && (button[bzone] != 0)) {
            int k,s;
            char lb[MMB+1];

            if (help) {
                show_hint(0,"");
                return(1);
            }

            if ((0 <= limits_top) && (limits_top < 2)) {
                limits[2*limits_top] = X0 + i * RF;
                limits[2*limits_top+1] = Y0 + j * RF;
                if (++limits_top > 1) {
                    limits[4] = limits[0];
                    limits[5] = limits[3];
                    limits[6] = limits[2];
                    limits[7] = limits[1];
                    button[bzone] = 0;
                    limits_top += 2;
                    redraw_zone = 1;
                }
                lb[MMB] = 0;
                s = snprintf(lb,MMB,"limits:");
                for (k=0; k<limits_top; ++k) {
                    s += snprintf(lb+s,MMB," (%d,%d)",
                                  limits[2*k],limits[2*k+1]);
                }
                show_hint(1,lb);
            }

            else {
                db("inconsistent value for limits_top");
                button[bzone] = 0;
            }

            redraw_button = -1;
        }

        /* paint pixel */
        else if ((*cm_e_pp != ' ') && (CDW == PATTERN)) {

            if (help) {
                show_hint(0,"paint pixel");
                return(1);
            }

            COLOR = BLACK;
            pp(i,j);
            redraw_dw = 1;
        }

        /* clear pixel */
        else if ((*cm_e_pp_c != ' ') && (CDW == PATTERN)) {

            if (help) {
                show_hint(0,"clear pixel");
                return(1);
            }

            COLOR = WHITE;
            pp(i,j);
            redraw_dw = 1;
        }

        /* fill region with current color */
        else if ((*cm_e_fill != ' ') && (CDW == PATTERN)) {

            if (help) {
                show_hint(0,"paint region");
                return(1);
            }

            COLOR = BLACK;
            c = cfont[i+j*FS];
            pp(i,j);
            if (c != cfont[i+j*FS]) {
                cfont[i+j*FS] = c;
                pr(i,j,c);
            }
            redraw_dw = 1;
        }

        /* clear region with current color */
        else if ((*cm_e_fill_c != ' ') && (CDW == PATTERN)) {

            if (help) {
                show_hint(0,"clear region");
                return(1);
            }

            COLOR = WHITE;
            c = cfont[i+j*FS];
            pp(i,j);
            if (c != cfont[i+j*FS]) {
                cfont[i+j*FS] = c;
                pr(i,j,c);
            }
            redraw_dw = 1;
        }

        /* go to hyperref */
        else if (HTML) {

            /* no help for this item */
            if (help) {

                /* show_hint(0,""); */
                return(1);
            }

            show_hint(1,"");
            html_action(x,y);
        }

        /* select symbol */
        else if (CDW == PAGE) {
            int k;

            /* no current symbol */
            if ((k = thisone(X0+i*RF,Y0+j*RF)) < 0)
                return(0);

            if (help) {
                sdesc *m;
                char *a;
                char lb[MMB+1];

                lb[0] = 0;
                mb[MMB] = 0;
                m = mc + k;
                a = ((m->tr) == NULL) ? "?" : ((m->tr)->t);
                if ((*cm_v_words != ' ') && (m->sw >= 0)) {
                    snprintf(lb,MMB,"word %d%s",m->sw,
                              (word[m->sw].f & F_ITALIC) ? " italic" : "");
                }
                else {

#ifdef CF
                    /*
                        Display absent symbols on type 0 to
                        help building the bookfont (experimental).
                    */
                    int i,j;
                    unsigned char c[256];
                    pdesc *p;

                    for (i=0; i<=255; ++i) {
                        c[i] = 0;
                    }

                    for (i=0; i<=topp; ++i) {
                        p = pattern + i;
                        if ((p->pt == 0) &&
                            (p->tr != NULL) &&
                            (strlen(p->tr) == 1)) {

                            ++(c[((unsigned char *)(p->tr))[0]]);
                        }
                    }

                    j = 0;

                    for (i='a'; (j<MMB) && (i<='z'); ++i) {
                        if (c[i] < 5)
                            lb[j++] = i;
                    }
                    for (i='A'; (j<MMB) && (i<='Z'); ++i) {
                        if (c[i] < 5)
                            lb[j++] = i;
                    }
                    for (i='0'; (j<MMB) && (i<='9'); ++i) {
                        if (c[i] < 5)
                            lb[j++] = i;
                    }

                    lb[j] = 0;
#else
                    snprintf(lb,MMB,"symbol %d: \"%s\" (word %d)",k,a,m->sw);
#endif
                }
                show_hint(0,lb);
                return(1);
            }

            show_hint(1,"");
            if ((curr_mc = k) >= 0)
                dw[PAGE_SYMBOL].rg = 1;
            redraw_dw = 1;
            symb2buttons(curr_mc);
        }
    }

    /* Application Buttons */
    else if ((10 <= x) && (x <= (10+BW)) &&
             (PT <= y) && (y <= PT + BUTT_PER_COL*(BH+VS))) {

        i = (y-PT) / (BH+VS);
        if ((y - PT - (i*(BH+VS))) <= BH) {

            /* (book)

                The Application Buttons
                -----------------------

                The application buttons are those displayed on the
                left portion of the Clara X window. They're
                labelled "zoom", "OCR", etc. Three types of
                buttons are available. There are on/off buttons
                (like "italic"), multi-state buttons (like the
                alphabet button), where the state is informed by
                the current label, and there are buttons that
                merely capture mouse clicks, like the "zoom"
                button. Some are sensible both to mouse button 1
                and to mouse button 2, others are sensible only to
                mouse button 1.

            */

            /* (book)

                zoom - enlarge or reduce bitmaps. The mouse buttom 1
                enlarge bitmaps, the mouse button 2 reduce bitmaps.
                The bitmaps to enlarge or reduce are determined
                by the current window. If the PAGE window is active,
                then the scanned document is enlarged or reduced.
                If the PAGE (fatbits) or the PATTERN window is active,
                then the grid is enlarged or reduced. If the PAGE
                (symbol) or the PATTERN (props) or the PATTERN (list)
                window is active, then the web clip is enlarged or
                reduced.

            */
            if (i == bzoom) {

                if (help) {
                    show_hint(0,"mouse button 1 enlarge bitmaps, button 2 reduce");
                    return(1);
                }

                if (CDW == PAGE) {
                    if (RF > 1) {
                        --RF;
                        setview(PAGE);
                        check_dlimits(1);
                    }
                }

                else if ((CDW == PAGE_FATBITS) || (CDW == TUNE_PATTERN)) {
                    if (ZPS < 7) {
                        ZPS += 2;
                        comp_wnd_size(-1,-1);
                        setview(CDW);
                    }
                }

                else if ((CDW == PATTERN) ||
                         (CDW == PATTERN_LIST) ||
                         (CDW == PATTERN_TYPES) ||
                         (CDW == PAGE_SYMBOL)) {

                    if (plist_rf > 1) {
                        --plist_rf;
                        dw[PATTERN].rg = 1;
                        dw[PATTERN_LIST].rg = 1;
                        dw[PATTERN_TYPES].rg = 1;
                        dw[PAGE_SYMBOL].rg = 1;
                        redraw_dw = 1;
                    }
                }
            }

            /* move to document start */
            /*
            else if (i == sarrow) {

                if (help) {
                    show_hint(0,"button 1 is HOME, 2 is END");
                    return(1);
                }
                X0 = 0;
                redraw_dw = 1;
            }
            */

            /* move to document top */
            /*
            else if (i == tarrow) {

                if (help) {
                    show_hint(0,"button 1 is TOP, 2 is BOTTOM");
                    return(1);
                }
                Y0 = 0;
                redraw_dw = 1;
            }
            */

            /* (book)

                OCR - start a full OCR run on the current page or on
                all pages, depending on the state of the "Work on
                current page only" item of the Options menu.

            */
            else if (i == bocr) {

                if (help) {
                    show_hint(0,"button 1 starts full OCR, button 2 open OCR steps menu");
                    return(1);
                }

                /* OCR current page */
                if (*cm_o_curr != ' ') {
                    start_ocr(-2,-1,0);
                }

                /* OCR all pages */
                else {
                    start_ocr(-1,-1,0);
                }
            }

            /* (book)

                stop - stop the current OCR run (if any). OCR
                does not stop immediately, but will stop
                as soon as possible.

            */
            else if (i == bstop) {

                if (help) {
                    show_hint(0,"stop OCR");
                    return(1);
                }

                /* stop */
                if (ocring) {
                    stop_ocr = 1;
                    button[bstop] = 1;
                    redraw_button = bstop;
                }
            }

            /* (book)

                zone - start definition of the OCR zone. Currently
                zoning in Clara OCR is useful only for saving the
                zone can as a PBM file, using the "save zone" item
                on the "File" menu. By now, only one zone can be
                defined and the OCR operations consider the
                entire document, ignoring the zone.

            */
            else if (i == bzone) {

                if (help) {
                    if (button[bzone] == 0)
                        show_hint(0,"define zone manually");
                    else
                        show_hint(0,"");
                    return(1);
                }

                if ((dw[PAGE].v) && (cpage >= 0)) {

                    button[bzone] = 1;
                    if ((tab != PAGE) || (limits_top > 0)) {
                        limits_top = 0;
                        setview(PAGE);
                    }
                    show_hint(1,"enter ocr limits: top left, botton right");
                    redraw_button = -1;
                    cl_ready = 0;
                }
            }

            /* (book)

                type - read-only button, set accordingly to the pattern
                type of the current symbol or pattern. The various letter
                sizes or styles (normal, footnote, etc) used by the book are
                numbered from 0 by Clara OCR ("type 0", "type 1", etc).

            */
            else if (i == btype) {

                /* former behaviour (per-symbol selection) */
                /*
                if (help) {
                    if (*cm_o_ftro != ' ')
                        show_hint(0,"this is the current pattern type");
                    else
                        show_hint(0,"buttons 1 or 2 change the pattern type");
                    return(1);
                }

                if ((dw[PATTERN].v) && (*cm_o_ftro == ' ')) {
                    if (++button[btype] > toppt)
                        button[btype] = 0;

                    redraw_button = btype;
                    form_changed = 1;
                    form_auto_submit();
                }
                */

                if (help) {
                    show_hint(0,"pattern type for current symbol/pattern");
                    return(1);
                }
            }

            /* (book)

                bold - read-only button, set accordingly to the bold
                status of the current symbol or pattern.

            */
            else if (i == bbold) {

                /* former behaviour (per-symbol selection) */
                /*
                if (help) {
                    show_hint(0,"click to toggle the bold flag");
                    return(1);
                }

                button[bbold] ^= 1;
                redraw_button = bbold;
                form_changed = 1;
                form_auto_submit();
                */

                if (help) {
                    show_hint(0,"bold status for current symbol/pattern");
                    return(1);
                }
            }

            /* (book)

                italic - read-only button, set accordingly to the italic
                status of the current symbol or pattern.

            */
            else if (i == bitalic) {

                /* former behaviour (per-symbol selection) */
                /*
                if (help) {
                    show_hint(0,"click to toggle the italic flag");
                    return(1);
                }

                button[bitalic] ^= 1;
                redraw_button = bitalic;
                form_changed = 1;
                form_auto_submit();
                */

                if (help) {
                    show_hint(0,"italic status for current symbol/pattern");
                    return(1);
                }
            }

            /* (book)

                bad - toggles the button state. The bad flag
                is used to identify damaged bitmaps.

            */
            else if (i == bbad) {

                if (help) {
                    show_hint(0,"click to toggle the bad flag");
                    return(1);
                }

                button[bbad] ^= 1;
                redraw_button = bbad;
                form_changed = 1;
                form_auto_submit();
            }

            /* (book)

                latin/greek/etc - read-only button, set accordingly to
                the alphabet of the current symbol or pattern.

            */
            else if (i == balpha) {

                /* former behaviour (per-symbol selection) */
                /*
                if (help) {
                    show_hint(0,"circulate alphabets and word types");
                    return(1);
                }
                if (++button[balpha] >= nalpha)
                    button[balpha] = 0;
                redraw_button = balpha;
                redraw_map = 1;
                set_alpha();
                */
                /*
                form_changed = 1;
                form_auto_submit();
                */

                if (help) {
                    show_hint(0,"alphabet of current symbol/pattern");
                    return(1);
                }
            }
        }
    }

    /* first resize area */
    else if ((tab == PAGE) && (dw[PAGE].v) && (dw[PAGE_OUTPUT].v) &&
             (DM <= x) && (x < DM+DW) &&
             (pb=dw[PAGE].dt+dw[PAGE].dh) &&
             (((pb+sb <= y) && (y < dw[PAGE_OUTPUT].dt)) /* ||
              ((pb <= y) && (y < pb+10))*/ )) {

        if (help) {
            show_hint(0,"click and drag to resize windows");
            return(1);
        }

        sliding = 3;

        /* must redraw to clear the ellipses */
        redraw_dw = 1;
    }

    /* second resize area */
    else if ((tab == PAGE) && (dw[PAGE_OUTPUT].v) && (dw[PAGE_SYMBOL].v) &&
             (DM <= x) && (x < DM+DW) &&
             (ob=dw[PAGE_OUTPUT].dt+dw[PAGE_OUTPUT].dh) &&
             (((ob+sb <= y) && (y < dw[PAGE_SYMBOL].dt)) /* ||
              ((ob <= y) && (y < ob+10)) */ )) {

        if (help) {
            show_hint(0,"click and drag to resize windows");
            return(1);
        }

        sliding = 4;

        /* must redraw to clear the ellipses */
        redraw_dw = 1;
    }

    /* vertical scrollbar */
    else if ((*cm_o_hide == ' ') &&
        (DM+DW+10 <= x) &&
        (x <= DM+DW+10+RW) &&
        (DT <= y) && (y < DT+DH)) {

        /* steppers */
        if (y <= DT+RW) {

            if (help) {
                show_hint(0,"button 1 step up, button 2 go to top");
                return(1);
            }

            step_up();
        }
        else if (y >= DT+DH-RW) {

            if (help) {
                if (DS)
                    show_hint(0,"button 1 step down, button 2 go to bottom");
                return(1);
            }

            step_down();
        }

        /* cursor */
        else {

            if (help) {
                if (DS)
                    show_hint(0,"drag or click button 1 or 2 to move just one page");
                return(1);
            }

            gettimeofday(&tclick,NULL);
            sliding = 2;
            s0 = y;
        }

        return(0);
    }

    /* horizontal scrollbar */
    else if ((DM <= x) && (x < DM+DW) &&
         (DT+DH+10 <= y) &&
         (y <= DT+DH+10+RW)) {

        /* steppers */
        if (x <= DM+RW) {

            if (help) {
                show_hint(0,"button 1 step left, button 2 go to left limit");
                return(1);
            }

            step_left();
        }
        else if (x >= DM+DW-RW) {

            if (help) {
                show_hint(0,"button 1 step right, button 2 go to right limit");
                return(1);
            }

            step_right();
        }

        /* cursor */
	else {

            if (help) {
                if (DS)
                    show_hint(0,"drag or click button 1 or 2 to move just one page");
                return(1);
            }

            gettimeofday(&tclick,NULL);
            sliding = 1;
            s0 = x;
        }

        return(0);
    }

    /*
    if (help)
        show_hint(0,"");
    */

    return(0);
}

/*

Get pointer coordinates.

*/
void get_pointer(int *x,int *y)
{
    Window root,child;
    int rx,ry,cx,cy;
    unsigned mask;

    /* get pointer coordinates */
    rx = ry = -1;
    XQueryPointer(xd,XW,&root,&child,&rx,&ry,&cx,&cy,&mask);
    *x = cx;
    *y = cy;
}

/*

Flashes a message about the graphic element currently under the
pointer.

This is the service responsible to make appear messages when
you move the pointer across the Clara X window.

In some cases (as for HTML anchors) the element changes color.

*/
void show_info(void) {
    int m,px,py;

    /* get pointer coordinates */
    get_pointer(&px,&py);

    /*
        The function mactions flashes a message for most
        elements on the application window (buttons, tabs,
        symbols on the PAGE window and others) when called
        accordingly.
    */
    m = mactions_b1(1,px,py);

    /*
        The function show_htinfo hilights the HTML element
        currently under the pointer, but that will mess with the
        active menu (if any), so we we avoid calling show_htinfo
        when a menu is active.

        The function show_htinfo also destroys
        the menu clipping. But the menu clipping is activated when
        the menu is dismissed, in order to redraw just the menu
        rectangle, and not the entire application window. So we
        avoid calling show_htinfo when mclip is active.
    */
    if ((cmenu != NULL) || (mclip))
        return;

    /*
        Flashes a message for the HTML element currently under
        the pointer.
    */
    if ((DM <= px) && (px < DM+DW) &&
       (DT <= py) && (py < DT+DH)) {

        if ((HTML) || ((CDW==PAGE) && (*cm_o_bb != ' '))) {
            XRectangle r;

            r.x = DM;
            r.y = DT;
            r.width = DW;
            r.height = DH;
            XSetClipRectangles(xd,xgc,0,0,&r,1,Unsorted);

            show_htinfo(px,py);

            r.x = 0;
            r.y = 0;
            r.width = WW;
            r.height = WH;
            XSetClipRectangles(xd,xgc,0,0,&r,1,Unsorted);

            if (curr_hti < 0)
                show_hint(0,"");
        }
    }

    else if (!m)
        show_hint(0,"");
}

/*

Mouse actions activated by mouse button 2.

*/
void mactions_b2(int help,int x,int y)
{
    int i;

    /* change to selected window */
    select_dw(x,y);

    /*

        Open window options menu.

    */
    if ((DM <= x) && (x < DM+DW) &&
        (DT <= y) && (y < DT+DH)) {

        /* PAGE context menu */
        if ((tab == PAGE) || (tab == PAGE_FATBITS)) {
            CX0 = x;
            CY0 = y;
            if (cmenu == NULL)
                form_auto_submit();
            cmenu = CM + CM_D;
            redraw_menu = 1;
        }
    }

    /* vertical scrollbar */
    else if ((*cm_o_hide == ' ') &&
             (DM+DW+10 <= x) &&
             (x <= DM+DW+10+RW) &&
             (DT <= y) && (y < DT+DH)) {

        /* steppers */
        if (y <= DT+RW) {
            Y0 = 0;
            redraw_dw = 1;
        }
        else if (y >= DT+DH-RW) {
            if ((GRY - VR) >= 0)
                Y0 = GRY - VR;
            redraw_dw = 1;
        }

        /* page down */
        else
            next();

        return;
    }

    /* horizontal scrollbar */
    else if ((*cm_o_hide == ' ') &&
             (DM <= x) && (x < DM+DW) &&
             (DT+DH+10 <= y) &&
             (y <= DT+DH+10+RW)) {

        /* steppers */
        if (x <= DM+RW) {
            X0 = 0;
            redraw_dw = 1;
        }
        else if (x >= DM+DW-RW) {
            if ((GRX - HR) >= 0)
                X0 = GRX - HR;
            redraw_dw = 1;
        }

        /* page right */
        else if ((CDW==PAGE_FATBITS) || (CDW==PAGE) || (CDW==PAGE_OUTPUT)) {
            X0 += HR;
            if (X0 > (XRES-HR))
                X0 = XRES-HR;
            redraw_dw = 1;
        }

        return;
    }

    /*

        Application Buttons
        -------------------

    */
    else if ((10 <= x) && (x <= (10+BW))) {

        i = (y-PT) / (BH+VS);
        if ((y - PT - (i*(BH+VS))) <= BH) {

            /* end */
            /*
            if (i == earrow) {
                if ((XRES - HR) >= 0)
                    X0 = XRES - HR;
                redraw_dw = 1;
            }
            */

            /* bottom */
            /*
            else if (i == barrow) {
                if ((YRES - VR) >= 0)
                    Y0 = YRES - VR;
                redraw_dw = 1;
            }
            */

            /* zoom - */
            if (i == bzoom) {

                if (CDW == PAGE) {
                    if (RF < MRF) {
                        ++RF;
                        setview(PAGE);
                        check_dlimits(1);
                    }
                }

                else if ((CDW == PAGE_FATBITS) || (CDW == TUNE_PATTERN)) {
                    if (ZPS > 1) {
                        ZPS -= 2;
                        comp_wnd_size(-1,-1);
                        setview(CDW);
                    }
                }

                else if ((CDW == PATTERN) ||
                         (CDW == PATTERN_LIST) ||
                         (CDW == PATTERN_TYPES) ||
                         (CDW == PAGE_SYMBOL)) {

                    if (plist_rf < 8) {
                        ++plist_rf;
                        dw[PATTERN].rg = 1;
                        dw[PATTERN_LIST].rg = 1;
                        dw[PATTERN_TYPES].rg = 1;
                        dw[PAGE_SYMBOL].rg = 1;
                        redraw_dw = 1;
                    }
                }
            }

            /* open OCR operations menu */
            else if (i == bocr) {
                CX0 = x;
                CY0 = y;
                if (cmenu == NULL)
                    form_auto_submit();
                cmenu = CM + CM_R;
                redraw_menu = 1;
            }

            /* change pattern type */
            /*
            else if (i == btype) {

                if (*cm_o_ftro == ' ') {
                    if (--button[btype] < 0)
                        button[btype] = 99;
                    redraw_button = btype;
                }
            }
            */

            /*

                page largest view

                Show the scanned page using a 1:1 relation
                between document pixels and display pixels.
            */
            /*
            if (i == bzp) {
                nrf = 1;
            }
            */

            /*

                page smallest view

                Show the scanned page using the largest reduction
                factor. The largest reduction factor is the minimum
                one that makes the entire document fit on its window.
            */
            /*
            else if (i == bzl) {
                nrf = MRF;
            }
            */

            /* change reduction factor */
            /*
            if (nrf != dw[PAGE].rf) {
                setview(PAGE);
                RF = nrf;
                check_dlimits(1);
                force_redraw();
            }
            */
        }
    }
}

/*

Get the menu item under the pointer.

*/
int get_item(int x,int y) {
    int j;

    if ((cmenu != NULL) &&
        (CX0 <= x) && (x <= CX0+CW) &&
        (CY0 <= y) && (y <= CY0+CH)) {
    
        /* compute action */
        j = (y - CY0 - 5) / (DFH + VS);
        if (j < 0)
            j = 0;
        else if (j >= cmenu->n)
            j = cmenu->n - 1;
        j += ((cmenu-CM) << CM_IT_WD);
        return(j);
    }
    return(-1);
}

/*

Get and process X events.

This function is continuously being called to check the existence
of X events to be processed and to process them (if any).

Obs. The flag have_s_ev must be cleared only when returning from
this service.

*/
void xevents(void)
{
    int no_ev;
    static int expose_queued=0;
    static struct timeval rt = {0,0};
    struct timeval t;
    unsigned d;

    /*
        No new event. Events are got from the Xserver or
        directly from the buffer xev. Some internal procedures
        feed xev conveniently and set the flag have_s_ev.
    */
    if ((!have_s_ev) && (XCheckWindowEvent(xd,XW,evmasks,&xev) == 0)) {

        /*
            Compute age of the expose event queued.
        */
        if (expose_queued) {
            gettimeofday(&t,NULL);
            d = (t.tv_sec - rt.tv_sec) * 1000000;
            if (d == 0)
                d += (t.tv_usec - rt.tv_usec);
            else
                d += -rt.tv_usec + t.tv_usec;
        }
        else {
    
            /* whenever absent, age is.. zero */
            d = 0;
    
            /*
                When OCRing, redraw components as requested,
                and return immediately.
            */
            if (ocring) {
                redraw();
                return;
            }
        }
    
        /*
            If there is not a 0.05 seconds old expose event
            queued, then redraw components as requested (unless
            there is an expose event queued), wait 0.1 seconds to
            not burn CPU, and return.
        */
        if (d < 50000) {
    
            /* redraw components */
            if (!expose_queued)
                redraw();
    
            /* wait 0.1 seconds */
            if (!ocring) {
                t.tv_sec = 0;
                t.tv_usec = 100000;
                select(0,NULL,NULL,NULL,&t);
            }

            /* dismiss menus that became unfocused */
            if (cmenu != NULL)
                dismiss(0);
    
            /* show local info */
            if ((!ocring) && (focused) && (!sliding))
                show_info();
    
            return;
        }

        else {

            /* attend the old expose event */
            if (expose_queued) {
                /* db("attending old expose event"); */
                force_redraw();
                redraw();
                expose_queued = 0;
            }
        }
    
        /* tell the code below to not try to process events */
        no_ev = 1;
    }
    
    /*
        Remember that we got a new event from the queue.
    */
    else {
        no_ev = 0;
        /* db("got event %d",xev.type); */
    }

    /* just got or lost focus */
    if (xev.type == FocusIn) {
        force_redraw();
        focused = 1;
        have_s_ev = 0;
        return;
    }
    else if (xev.type == FocusOut) {
        focused = 0;
        have_s_ev = 0;
        return;
    }

    /*
        Remember time of the expose event (it's the event
        being processed now). This event will be processed
        as soon as it becomes sufficiently old, in order to
        avoid expensive redrawing of the entire window.
    */
    if ((!no_ev) && (xev.type == Expose)) {

        gettimeofday(&rt,NULL);
        expose_queued = 1;
        have_s_ev = 0;
        return;
    }

    /* no new event */
    if (no_ev) {
        have_s_ev = 0;
        return;
    }

    /*
        Redraw as request by the user action that provoked the
        event. In some cases, this call should clear expose_queue,
        but it's hard to distinguish such cases, so we prefer to
        maintain the expose_queued status and redraw the window
        entirely again after some delay.
    */
    else {
        redraw();
    }

    /*
        There is an active menu. Mouse button 1 will select an
        item, ESC will dismiss the menu, and arrows will
        browse the menus.
    */
    if (cmenu != NULL) {
        bev = (XButtonEvent *) &xev;

        /* selected item */
        if ((xev.type == ButtonPress) && (bev->button == Button1)) {
            mactions_b1(0,xev.xbutton.x,xev.xbutton.y);
        }

        /* key */
        else if (xev.type == KeyPress) {
            kactions();
        }

        /* just hilight the current item */
        else if (xev.type == MotionNotify) {
            int j;

            j = get_item(xev.xbutton.x,xev.xbutton.y) & CM_IT_M;
            if (j != cmenu->c) {
                cmenu->c = j;
                redraw_menu = 1;
            }
        }
    }

    /* Application window resize */
    else if (xev.type == ConfigureNotify) {
        Window root;
        int x,y;
        unsigned w,h,b,d;

        XGetGeometry(xd,XW,&root,&x,&y,&w,&h,&b,&d);
        if ((w != WW) || (h != WH)) {
            /* db("resizing from %d %d to %d %d",WW,WH,w,h); */
            comp_wnd_size(w,h);
            set_mclip(0);
            X0 = Y0 = 0;
            setview(CDW);
        }
    }

    /* motion notify */
    else if (xev.type == MotionNotify) {
        slide();
    }

    /* Mouse button release finishes sliding modes */
    else if (xev.type == ButtonRelease) {
        unsigned d;

        if (sliding) {

            /* sliding delay */
            gettimeofday(&t,NULL);
            d = (t.tv_sec - tclick.tv_sec) * 1000000;
            if (d == 0)
                d += (t.tv_usec - tclick.tv_usec);
            else
                d += -tclick.tv_usec + t.tv_usec;

            /* a quick press-release on mode 1 means "page left" */
            if ((sliding == 1) && (d < 200000)) {
                if ((CDW==PAGE_FATBITS) || (CDW==PAGE) || (CDW==PAGE_OUTPUT)) {
                    X0 -= HR;
                    if (X0 < 0)
                        X0 = 0;
                }
            }

            /* a quick press-release on mode 2 means "page up" */
            if ((sliding == 2) && (d < 200000))
                prior();

            if (curr_mc >= 0)
                redraw_dw = 1;
        }
        sliding = 0;
    }

    /* Process other events */
    else {

        switch (xev.type) {

            case ButtonPress:

                bev = (XButtonEvent *) &xev;
                if (bev->button == Button1) {
                    mactions_b1(0,xev.xbutton.x,xev.xbutton.y);
                }
                else if (bev->button == Button3) {
                    mactions_b2(0,xev.xbutton.x,xev.xbutton.y);
                }

                break;

            case KeyPress:
                kactions();
                break;
    
            default:
                db("unattended X event %d (ok)",xev.type);
                break;
        }
    }

    have_s_ev = 0;
}

/*

Synchronize the variables alpha, alpha_sz and alpha_cols with
the current status of the alphabet selection button.

*/
void set_alpha(void)
{
    int i;

    /* get current alphabet and its size */
    i = inv_balpha[(int)(button[balpha])];
    if (i == LATIN) {
        alpha = latin;
        alpha_sz = latin_sz;
        alpha_cols = 2;
    }
    else if (i == GREEK) {
        alpha = (alpharec *) greek;
        alpha_sz = greek_sz;
        alpha_cols = 3;
    }
    else if (i == CYRILLIC) {
        alpha = cyrillic;
        alpha_sz = cyrillic_sz;
        alpha_cols = 2;
    }
    else if (i == HEBREW) {
        alpha = hebrew;
        alpha_sz = hebrew_sz;
        alpha_cols = 1;
    }
    else if (i == ARABIC) {
        alpha = arabic;
        alpha_sz = arabic_sz;
        alpha_cols = 1;
    }
    else if (i == NUMBER) {
        alpha = number;
        alpha_sz = number_sz;
        alpha_cols = 1;
    }
    else
        alpha = NULL;
}

/*

Set the application window name.

*/
void swn(char *n)
{
    char *list[2];
    XTextProperty xtp;

    list[0] = n;
    list[1] = NULL;
    XStringListToTextProperty(list,1,&xtp);
    XSetWMName(xd,XW,&xtp);
}

/*

Set the default font. Clara uses one same font for all (menus,
button labels, HTML rendering, etc. The font to be set is
obtained from the Options menu status.

*/
void set_xfont(void)
{
    XCharStruct *x;

    /* free current font */
    if (dfont != NULL)
        XFreeFont(xd,dfont);

    /* use small font */
    if (*cm_o_small == 'X') {
        if (((dfont = XLoadQueryFont(xd,"6x13")) == NULL) && verbose)
            warn("oops.. could not read font 6x13\n");
    }

    /* use medium font */
    else if (*cm_o_medium == 'X') {
        if (((dfont = XLoadQueryFont(xd,"9x15")) == NULL) && verbose)
            warn("oops.. could not read font 9x15\n");
    }

    /* use large font */
    else if (*cm_o_large == 'X') {
        if (((dfont = XLoadQueryFont(xd,"10x20")) == NULL) && verbose)
            warn("oops.. could not read font 10x20\n");
    }

    /* use the font informed as command-line parameter */
    else {
        if (((dfont = XLoadQueryFont(xd,xfont)) == NULL) && verbose)
            warn("oops.. could not read font %s\n",xfont);
    }

    /* try "fixed" */
    if ((dfont == NULL) && (strcmp(xfont,"fixed") != 0)) {
        if (((dfont = XLoadQueryFont(xd,"fixed")) == NULL) && verbose)
            warn("oops.. could not read font fixed\n");
    }

    /* no font loaded: abandon */
    if (dfont == NULL) {
        fatal(XE,"could not load a font, giving up");
    }

    /* text font height and width */
    x = &(dfont->max_bounds);
    DFH = (DFA=x->ascent) + (DFD=x->descent);
    DFW = x->width;
}

/*

Reset the parameters of all windows.

*/
void init_dws(int m)
{
    int i;

    for (i=0; i<=TOP_DW; ++i) {
        dw[i].x0 = 0;
        dw[i].y0 = 0;
        dw[i].hr = 0;
        dw[i].vr = 0;
        dw[i].grx = 0;
        dw[i].gry = 0;
        dw[i].v = 0;
        dw[i].ds = 1;
        dw[i].hm = 0;
        if (m) {
            dw[i].p_ge = NULL;
            dw[i].p_gesz = 0;
        }
        dw[i].p_topge = -1;
        dw[i].p_curr_hti = -1;
        dw[i].ulink = 1;
        dw[i].hs = 1;
        dw[i].rg = 1;
    }

    dw[WELCOME].ds = 0;

    /*
        In HTML windows that mix tables with other free elements
        we define a nonzero left margin for aesthetic.
    */
    dw[TUNE].hm = DFW;
    dw[PATTERN_TYPES].hm = DFW;
}

/*

Context menu initializer.

*/
void cmi(void)
{
    cmdesc *c;

    /* (book)

        File menu
        ---------

        This menu is activated from the menu bar on the top of the
        application X window.
    */
    c = addmenu(1,"File",0);

    /* (book)

        Load page

        Enter the page list to select a page to be loaded.
    */
    CM_F_LOAD = additem(c,"Load page",CM_TA,0);

    /* (book)

        Save session

        Save on disk the page session (file page.session), the patterns
        (file "pattern") and the revision acts (file "acts").
    */
    CM_F_SAVE = additem(c,"Save session",CM_TA,0);

    /* (book)

        Save zone

        Save on disk the current zone as the file zone.pbm.
    */
    CM_F_SZ = additem(c,"Save zone",CM_TA,0);

    /* (book)

        Write report

        Save the contents of the PAGE LIST window to the file
        report.txt on the working directory.
    */
    CM_F_REP = additem(c,"Write report",CM_TA,0);

    CM_F = c - CM;

    /* (book)

        Quit the program

        Just quit the program (asking before if the session is to
        be saved.
    */
    CM_F_QUIT = additem(c,"Quit",CM_TA,0);
    CM_F = c - CM;

    /* (book)

        Edit menu
        ---------

        This menu is activated from the menu bar on the top of the
        application X window.
    */
    c = addmenu(1,"Edit",0);

    /* (book)

        Only doubts

        When selected, the right or the left arrows used on the
        PATTERN or the PATTERN PROPS windows will move to the next
        or the previous untransliterated patterns.
    */
    CM_E_DOUBTS = additem(c,"[ ] Only doubts",CM_TB,0);

    /* (book)

        Re-scan all patterns

        When selected, the classification heuristic will retry
        all patterns for each symbol. This is required when trying
        to resolve the unclassified symbols using a second
        classification method.
    */
    CM_E_RESCAN = additem(c,"[ ] Re-scan all patterns",CM_TB,0);

    /* (book)

        Fill region

        When selected, the mouse button 1 will fill the region
        around one pixel on the pattern bitmap under edition on the
        font tab.
    */
    CM_E_FILL = additem(c,"[ ] Fill region",CM_TR,1);

    /* (book)

        Paint pixel

        When selected, the mouse button 1 will paint individual
        pixels on the pattern bitmap under edition on the font tab.
    */
    CM_E_PP = additem(c,"[ ] Paint pixel",CM_TR,0);

    /* (book)

        Clear region

        When selected, the mouse button 1 will clear the region
        around one pixel on the pattern bitmap under edition on the
        font tab.
    */
    CM_E_FILL_C = additem(c,"[ ] Clear region",CM_TR,0);

    /* (book)

        Clear pixel

        When selected, the mouse button 1 will clear individual
        pixels on the pattern bitmap under edition on the font tab.
    */
    CM_E_PP_C = additem(c,"[ ] Clear pixel",CM_TR,0);

    /* (book)

        Sort by page

        When selected, the pattern list window will divide
        the patterns in blocks accordingly to their (page)
        sources.
    */
    CM_E_SP = additem(c,"[ ] Sort by page",CM_TB,1);

    /* (book)

        Sort by matches

        When selected, the pattern list window will use as the
        first criterion when sorting the patterns, the number of
        matches of each pattern.
    */
    CM_E_SM = additem(c,"[ ] Sort by matches",CM_TB,0);

    /* (book)

        Sort by transliteration

        When selected, the pattern list window will use as the
        second criterion when sorting the patterns, their
        transliterations.
    */
    CM_E_ST = additem(c,"[ ] Sort by transliteration",CM_TB,0);

    /* (book)

        Sort by number of pixels

        When selected, the pattern list window will use as the
        third criterion when sorting the patterns, their
        number of pixels.
    */
    CM_E_SN = additem(c,"[ ] Sort by number of pixels",CM_TB,0);

    /* (book)

        Sort by width

        When selected, the pattern list window will use as the
        fourth criterion when sorting the patterns, their
        widths.
    */
    CM_E_SW = additem(c,"[ ] Sort by width",CM_TB,0);

    /* (book)

        Sort by height

        When selected, the pattern list window will use as the
        fifth criterion when sorting the patterns, their
        heights.
    */
    CM_E_SH = additem(c,"[ ] Sort by height",CM_TB,0);

    /* (book)

        Del Untransliterated patterns

        Remove from the font all untransliterated fonts.
    */
    CM_E_DU = additem(c,"Del Untransliterated patterns",CM_TA,1);

    /* (book)

        Reset match counters

        Change to zero the cumulative matches field of all
        patterns.
    */
    CM_E_CM = additem(c,"Reset match counters",CM_TA,0);

    /* (book)

        Set pattern type.

        Set the pattern type for all patterns marked as "other".
    */
    CM_E_SETPT = additem(c,"Set pattern type",CM_TA,0);

#ifdef PATT_SKEL
    /* (book)

        Reset skeleton parameters

        Reset the parameters for skeleton computation for all
        patterns.
    */
    CM_E_RSKEL = additem(c,"Reset skeleton parameters",CM_TA,0);
#endif

    CM_E = c - CM;

    /* (book)

        View menu
        ---------

        This menu is activated from the menu bar on the top of the
        application X window.
    */
    c = addmenu(1,"View",0);

    /* (book)

        Show skeletons

        Show the skeleton on the windows PAGE_FATBITS. The
        skeletons are computed on the fly.
    */
    CM_V_SKEL = additem(c,"[ ] Show skeletons",CM_TR,0);

    /* (book)

        Show border

        Show the border on the window PAGE_FATBITS. The
        border is computed on the fly.
    */
    CM_V_BORDER = additem(c,"[ ] Show border",CM_TR,0);

    /* (book)

        Show pattern skeleton

        For each symbol, will show the skeleton of its best match
        on the PAGE (fatbits) window.
    */
    CM_V_HS = additem(c,"[ ] Show pattern skeleton",CM_TR,1);

    /* (book)

        Show pattern border

        For each symbol, will show the border of its best match
        on the PAGE (fatbits) window.
    */
    CM_V_HB = additem(c,"[ ] Show pattern border",CM_TR,0);

    /* (book)

        Show HTML source

        Show the HTML source of the document, instead of the
        graphic rendering.
    */
    CM_V_VHS = additem(c,"[ ] Show HTML source",CM_TB,1);

    /* (book)

        Show web clip

        Toggle the web clip feature. When enabled, the PAGE_SYMBOL
        window will include the clip of the document around the
        current symbol that will be used through web revision.
    */
    CM_V_WCLIP = additem(c,"[ ] Show web clip",CM_TB,0);

    /* (book)

        Show alphabet map

        Toggle the alphabet map display. When enabled, a mapping
        from Latin letters to the current alphabet will be
        displayed.
    */
    CM_V_MAP = additem(c,"[ ] Show alphabet map",CM_TB,0);

    /* (book)

        Show current class

        Identify the symbols on the current class using
        a gray ellipse.
    */
    CM_V_CC = additem(c,"[ ] Show current class",CM_TB,0);

    /* (book)

        Show closures

        Identify the individual closures when displaying the
        current document.
    */
    CM_V_CLOSURES = additem(c,"[ ] Show closures",CM_TR,1);

    /* (book)

        Show symbols

        Identify the individual symbols when displaying the
        current document.
    */
    CM_V_SYMBOLS = additem(c,"[ ] Show symbols",CM_TR,0);

    /* (book)

        Show words

        Identify the individual words when displaying the
        current document.
    */
    CM_V_WORDS = additem(c,"[ ] Show words",CM_TR,0);

    /* (book)

        Show lines (geometrical)

        Identify the lines (computed using geometrical criteria)
        when displaying the current document.
    */
    CM_V_GLINES = additem(c,"[ ] Show lines (geometrical)",CM_TR,0);

    /* (book)

        Bold Only

        Restrict the show symbols or show words feature to
        bold symbols or words.
    */
    CM_V_BO = additem(c,"[ ] Bold Only",CM_TR,1);

    /* (book)

        Italic Only

        Restrict the show symbols or show words feature to
        bold symbols or words.
    */
    CM_V_IO = additem(c,"[ ] Italic Only",CM_TR,0);

    /* (book)

        Show matches

        Display bitmap matches when performing OCR.
    */
    CM_V_MAT = additem(c,"[ ] Show matches",CM_TR,1);

    /* (book)

        Show comparisons

        Display all bitmap comparisons when performing OCR.
    */
    CM_V_CMP = additem(c,"[ ] Show comparisons",CM_TR,0);

    /* (book)

        Show matches

        Display bitmap matches when performing OCR,
        waiting a key after each display.
    */
    CM_V_MAT_K = additem(c,"[ ] Show matches and wait",CM_TR,0);

    /* (book)

        Show comparisons and wait

        Display all bitmap comparisons when performing OCR,
        waiting a key after each display.
    */
    CM_V_CMP_K = additem(c,"[ ] Show comparisons and wait",CM_TR,0);

    /* (book)

        Show skeleton tuning

        Display each candidate when tuning the skeletons of the
        patterns.
    */
    CM_V_ST = additem(c,"[ ] Show skeleton tuning",CM_TB,0);

    /* (book)

        Presentation

        Perform a presentation. This item is visible on the menu
        only when the program is started with the -A option.
    */
    if (allow_pres)
        CM_V_PRES = additem(c,"Presentation",CM_TA,1);
    CM_V = c - CM;

    /* (book)

        OCR steps menu
        --------------

        This menu is activated when the mouse button 2 is pressed on
        the OCR button. It allows running specific OCR steps (all
        steps run in sequence when the OCR button is pressed).

    */
    c = addmenu(0,"OCR operations",0);

    /* (book)

        Consist structures

        All OCR data structures are submitted to consistency
        tests. This is under implementation.
    */
    CM_R_CONS = additem(c,"Consist structures",CM_TA,0);

    /* (book)

        Prepare patterns

        Compute the skeletons and analyse the patterns for
        the achievement of best results by the classifier. Not
        fully implemented yet.
    */
    CM_R_OPT = additem(c,"Prepare patterns",CM_TA,0);

    /* (book)

        Read revision data

        Revision data from the web interface is read, and
        added to the current OCR training knowledge.
    */
    CM_R_REV = additem(c,"Read revision data",CM_TA,0);

    /* (book)

        Detect blocks

        Start detecting blocks of text on the page.

    */
    CM_R_BLOCK = additem(c,"Detect blocks",CM_TA,0);

    /* (book)

        Classification

        start classifying the symbols of the current
        page or of all pages, depending on the state of the
        "Work on current page only" item of the Options menu. It
        will also build the font automatically if the
        corresponding item is selected on the Options menu.

    */
    CM_R_CLASS = additem(c,"Classification",CM_TA,0);

    /* (book)

        Geometric merging

        Merge closures on symbols depending on their
        geometry.
    */
    CM_R_GEOMERGE = additem(c,"Geometric merging",CM_TA,0);

    /* (book)

        Build words and lines

        Start building the words and lines. These heuristics will
        be applied on the
        current page or on all pages, depending on the state of
        the "Work on current page only" item of the Options menu.

    */
    CM_R_BUILD = additem(c,"Build words and lines",CM_TA,0);

    /* (book)

        Generate spelling hints

        Obs. this is not implemented yet.

        Start filtering through ispell to generate
        transliterations for unknow symbols or alternative
        transliterations for known symbols. Clara will use the
        dictionaries available for the languages selected on
        the Languages menu. Filtering will be performed on the
        current page or on all pages, depending on the state of
        the "Work on current page only" item of the Options menu.

    */
    CM_R_SPELL = additem(c,"Generate spelling hints",CM_TA,0);

    /* (book)

        Generate output

        The OCR output is generated to be displayed on the
        "PAGE (output)" window. The output is also saved to the
        file page.html.
    */
    CM_R_OUTP = additem(c,"Generate output",CM_TA,0);

    /* (book)

        Generate web doubts

        Files containing symbols to be revised through the web
        interface are created on the "doubts" subdirectory of
        the work directory. This step is performed only when
        Clara OCR is started with the -W command-line switch.
    */
    CM_R_DOUBTS = additem(c,"Generate web doubts",CM_TA,0);

    CM_R = c - CM;

    /* (book)

        Window options menu
        -------------------

        This menu is activated when the mouse button 2 is pressed on
        the window that displays the scanned document (that is, the
        PAGE or the PAGE FATBITS window).

    */
    c = addmenu(0,"Window options",0);

    /* (book)

        Use as pattern

        The pattern of the class of this symbol will be the unique
        pattern used on all subsequent symbol classifications. This
        feature is intended to be used with the "OCR this symbol"
        feature, so it becomes possible to choose two symbols to
        be compared, in order to test the classification routines.

    */
    CM_D_TS = additem(c,"Use as pattern",CM_TA,0);

    /* (book)

        OCR this symbol

        Starts classifying only the symbol under the pointer. The
        classification will re-scan all patterns even if the "re-scan
        all patterns" option is unselected.
    */
    CM_D_OS = additem(c,"OCR this symbol",CM_TA,0);

    /* (book)

        Bottom left here

        Scroll the window contents in order to the
        current pointer position become the bottom left.
    */
    CM_D_HERE = additem(c,"Bottom left here",CM_TA,0);

    /* (book)

        Merge with current symbol

        Merge this fragment with the current symbol.
    */
    CM_D_ADD = additem(c,"Merge with current symbol",CM_TA,0);

    /* (book)

        Link as next symbol

        Create a symbol link from the current symbol (the one
        identified by the graphic cursor) to this symbol.
    */
    CM_D_SLINK = additem(c,"Link as next symbol",CM_TA,0);

    /* (book)

        Disassemble

        Make the current symbol nonpreferred and each of its
        components preferred.
    */
    CM_D_DIS = additem(c,"Disassemble",CM_TA,0);

    /* (book)

        Link as accent

        Create an accent link from the current symbol (the one
        identified by the graphic cursor) to this symbol.
    */
    CM_D_ALINK = additem(c,"Link as accent",CM_TA,0);

    /* (book)

        Diagnose symbol pairing

        Run locally the symbol pairing heuristic to try to join
        this symbol to the word containing the current symbol.
        This is useful to know why the OCR is not joining two
        symbols on one same word.
    */
    CM_D_SDIAG = additem(c,"Diagnose symbol pairing",CM_TA,0);

    /* (book)

        Diagnose word pairing

        Run locally the word pairing heuristic to try to join
        this word with the word containing the current symbol
        on one same line. This is useful to know why the OCR is
        not joining two words on one same line.
    */
    CM_D_WDIAG = additem(c,"Diagnose word pairing",CM_TA,0);

    /* (book)

        Diagnose merging

        Run locally the geometrical merging heuristic to try
        to merge this piece to the current symbol.
    */
    CM_D_GM = additem(c,"Diagnose merging",CM_TA,0);
    CM_D = c - CM;

    /* (book)

        Alphabets menu
        --------------

        This item selects the alphabets that will be available on the
        alphabets button.
    */
    c = addmenu(1,"Alphabets",0);

    /* (book)

        Arabic

        This is a provision for future support of Arabic
        alphabet.
    */
    CM_A_ARABIC = additem(c,"[ ] Arabic",CM_TB,0);

    /* (book)

        Cyrillic

        This is a provision for future support of Cyrillic
        alphabet.
    */
    CM_A_CYRILLIC = additem(c,"[ ] Cyrillic",CM_TB,0);

    /* (book)

        Greek

        This is a provision for future support of Greek
        alphabet.
    */
    CM_A_GREEK = additem(c,"[ ] Greek",CM_TB,0);

    /* (book)

        Hebrew

        This is a provision for future support of Hebrew
        alphabet.
    */
    CM_A_HEBREW = additem(c,"[ ] Hebrew",CM_TB,0);

    /* (book)

        Kana

        This is a provision for future support of Kana
        alphabet.
    */
    CM_A_KANA = additem(c,"[ ] Kana",CM_TB,0);

    /* (book)

        Latin

        Words that use the Latin alphabet include those from
        the languages of most Western European countries (English,
        German, French, Spanish, Portuguese and others).
    */
    CM_A_LATIN = additem(c,"[ ] Latin",CM_TB,0);

    /* (book)

        Number

        Indo-arabic decimal numbers like
        1234, +55-11-12345678 or 2000.
    */
    CM_A_NUMBER = additem(c,"[ ] Numeric",CM_TB,1);

    /* (book)

        Ideogram

        Ideograms.
    */
    CM_A_IDEOGRAM = additem(c,"[ ] Ideograms",CM_TB,0);

    CM_A = c - CM;

#ifdef LANG_MENU
    /* (book)

        Languages menu
        --------------

        This menu selects the languages used on the book. It will
        define the dictionaries that will be applied when trying
        to generate the transliteration of unknown symbols or
        generating alternative transliterations or marking a
        word for future revision.
    */
    c = addmenu(1,"Languages",0);

    /* (book)

        English (USA)

        Toggle American English spell checking for
        each word found on the page.
    */
    CM_L_EN = additem(c,"[ ] English (USA)",CM_TB,0);

    /* (book)

        English (UK)

        Toggle British English spell checking for
        each word found on the page.
    */
    CM_L_UK = additem(c,"[ ] English (UK)",CM_TB,0);

    /* (book)

        French

        Toggle French spell checking for
        each word found on the page.
    */
    CM_L_FR = additem(c,"[ ] French",CM_TB,0);

    /* (book)

        German

        Toggle german spell checking for
        each word found on the page.
    */
    CM_L_DE = additem(c,"[ ] German",CM_TB,0);

    /* (book)

        Greek

        Toggle greek spell checking for
        each word found on the page.
    */
    CM_L_GR = additem(c,"[ ] Greek",CM_TB,0);

    /* (book)

        Portuguese (Brazil)

        Toggle brazilian protuguese spell checking for
        each word found on the page.
    */
    CM_L_BR = additem(c,"[ ] Portuguese (Brazil)",CM_TB,0);

    /* (book)

        Portuguese (Portugal)

        Toggle Portuguese spell checking for
        each word found on the page.
    */
    CM_L_PT = additem(c,"[ ] Portuguese (Portugal)",CM_TB,0);

    /* (book)

        Russian

        Toggle Russian spell checking for
        each word found on the page.
    */
    CM_L_RU = additem(c,"[ ] Russian",CM_TB,0);

    CM_L = c - CM;
#endif

    /* (book)

        Options menu
        ------------
    */
    c = addmenu(1,"Options",0);

    /* (book)

        Work on current page only

        OCR operations (classification, merge, etc) will be
        performed only on the current page.
    */
    CM_O_CURR = additem(c,"[ ] Work on current page only",CM_TR,0);

    /* (book)


        Work on all pages

        OCR operations (classification, merge, etc) will be
        performed on all pages.
    */
    CM_O_ALL = additem(c,"[ ] Work on all pages",CM_TR,0);

    /* (book)

        Small font

        Use a small X font (6x13).
    */
    CM_O_SMALL = additem(c,"[ ] Small font (6x13)",CM_TR,1);

    /* (book)

        Medium font

        Use the medium font (9x15).
    */
    CM_O_MEDIUM = additem(c,"[ ] Medium font (9x15)",CM_TR,0);

    /* (book)

        Small font

        Use a large X font (10x20).
    */
    CM_O_LARGE = additem(c,"[ ] Large font (10x20)",CM_TR,0);

    /* (book)

        Default font

        Use the default font (7x13 or "fixed" or the one informed
        on the command line).
    */
    CM_O_DEF = additem(c,"[ ] Default font",CM_TR,0);

    /* (book)

        Hide scrollbars

        Toggle the hide scrollbars flag. When active, this
        flag hides the display of scrolllbar on all windows.
    */
    CM_O_HIDE = additem(c,"[ ] Hide scrollbars",CM_TB,0);

    /* (book)

        Omit fragments

        Toggle the hide fragments flag. When active,
        fragments won't be included on the list of
        patterns.
    */
    CM_O_OF = additem(c,"[ ] Omit fragments",CM_TB,0);

    /* (book)

        Report scale

        Report the scale on the tab when the PAGE window is
        active.
    */
    CM_O_RS = additem(c,"[ ] Report scale",CM_TB,0);

    /* (book)

        Display box instead of symbol

        On the PAGE window displays the bounding boxes instead of
        the symbols themselves. This is useful when designing new
        heuristics.
    */
    CM_O_BB = additem(c,"[ ] Display box instead of symbol",CM_TB,0);

    /* (book)

        Emulate deadkeys

        Toggle the emulate deadkeys flag. Deadkeys are useful for
        generating accented characters. Deadkeys emulation are disabled
        by default The emulation of deadkeys may be set on startup
        through the -i command-line switch.
    */
    CM_O_DKEYS = additem(c,"[ ] Emulate deadkeys",CM_TB,1);

    /* (book)

        Menu auto popup

        Toggle the automenu feature. When enabled, the menus on the
        menu bar will pop automatically when the pointer reaches
        the menu bar.
    */
    CM_O_AMENU = additem(c,"[ ] Menu auto popup",CM_TB,0);

    /* (book)

        Search unexpected mismatches

        Compare all patters with same type and transliteration. Must
        be used with the "Show comparisons and wait" option on
        to diagnose symbol comparison problems.
    */
    CM_O_SUM = additem(c,"[ ] Search unexpected mismatches",CM_TB,0);

    CM_O = c - CM;

    /*

        Assign menu flags to the menu labels.

    */

    /* edit menu */
    cm_e_od      = F(CM_E_DOUBTS);
    cm_e_rescan  = F(CM_E_RESCAN);
    cm_e_pp      = F(CM_E_PP);
    cm_e_pp_c    = F(CM_E_PP_C);
    cm_e_fill    = F(CM_E_FILL);
    cm_e_fill_c  = F(CM_E_FILL_C);
    cm_e_sm      = F(CM_E_SM);
    cm_e_sp      = F(CM_E_SP);
    cm_e_st      = F(CM_E_ST);
    cm_e_sw      = F(CM_E_SW);
    cm_e_sh      = F(CM_E_SH);
    cm_e_sn      = F(CM_E_SN);

    /* view menu */
    cm_v_skel     = F(CM_V_SKEL);
    cm_v_border   = F(CM_V_BORDER);
    cm_v_wclip    = F(CM_V_WCLIP);
    cm_v_map      = F(CM_V_MAP);
    cm_v_hs       = F(CM_V_HS);
    cm_v_hb       = F(CM_V_HB);
    cm_v_vhs      = F(CM_V_VHS);
    cm_v_cc       = F(CM_V_CC);
    cm_v_closures = F(CM_V_CLOSURES);
    cm_v_symbols  = F(CM_V_SYMBOLS);
    cm_v_words    = F(CM_V_WORDS);
    cm_v_glines   = F(CM_V_GLINES);
    cm_v_mat      = F(CM_V_MAT);
    cm_v_mat_k    = F(CM_V_MAT_K);
    cm_v_cmp_k    = F(CM_V_CMP_K);
    cm_v_cmp      = F(CM_V_CMP);
    cm_v_st       = F(CM_V_ST);
    cm_v_bo       = F(CM_V_BO);
    cm_v_io       = F(CM_V_IO);

    /* options menu */
    cm_o_small   = F(CM_O_SMALL);
    cm_o_medium  = F(CM_O_MEDIUM);
    cm_o_large   = F(CM_O_LARGE);
    cm_o_def     = F(CM_O_DEF);
    cm_o_hide    = F(CM_O_HIDE);
    cm_o_of      = F(CM_O_OF);
    cm_o_rs      = F(CM_O_RS);
    cm_o_bb      = F(CM_O_BB);
    cm_o_dkeys   = F(CM_O_DKEYS);
    cm_o_amenu   = F(CM_O_AMENU);
    cm_o_curr    = F(CM_O_CURR);
    cm_o_all     = F(CM_O_ALL);
    cm_o_sum     = F(CM_O_SUM);

    /* alphabets menu */
    cm_a_arabic    = F(CM_A_ARABIC);
    cm_a_cyrillic  = F(CM_A_CYRILLIC);
    cm_a_greek     = F(CM_A_GREEK);
    cm_a_hebrew    = F(CM_A_HEBREW);
    cm_a_kana      = F(CM_A_KANA);
    cm_a_latin     = F(CM_A_LATIN);
    cm_a_ideogram  = F(CM_A_IDEOGRAM);
    cm_a_number    = F(CM_A_NUMBER);

    /* initial values */
    if (RW < 0) {
        *cm_o_hide = 'X';
        RW = - RW;
    }
    if (dkeys) {
        *cm_o_dkeys = 'X';
        RW = - RW;
    }

    /* these are on by default */
    *cm_v_cc         = 'X';

    *cm_o_curr       = 'X';
    *cm_o_of         = 'X';

    *cm_a_number     = 'X';
    *cm_a_latin      = 'X';
}

/*

GUI initialization.

*/
void xpreamble()
{
    int xscreen;
    Colormap cmap;

    /* X preamble */
    if ((batch_mode == 0) && (wnd_ready == 0)) {

        /* connect to the X server */
        if (displayname == NULL)
            xd = XOpenDisplay("");
        else
            xd = XOpenDisplay(displayname);

        if (xd == NULL) {
            if (displayname != NULL)
                fatal(XE,"cannot connect display %s",displayname);
            else
                fatal(XE,"cannot connect display");
        }
    }

    /* context menus initialization */
    cmi();

    /* set alphabets button label */
    set_bl_alpha();

    /* alloc X font */
    if (batch_mode == 0)
        set_xfont();

    /* compute minimal window size */
    comp_wnd_size(WW,WH);

    /* buffers for revision bitmaps */
    rw = c_realloc(NULL,sizeof(char)*RWX*RWY,NULL);

    /* alloc colors and create application X window */
    if ((batch_mode == 0) && (wnd_ready == 0)) {
        char *color[6];
        int i,j,n,r;
        XColor exact;

        /* get default screen */
        xscreen = DefaultScreen(xd);

        /* get colors */
        cmap = DefaultColormap(xd,xscreen);
        j = 0;
        if ((cschema!=NULL) && ((n=strlen(cschema)) > 1)) {
            color[0] = cschema;
            for (i=0; i<n; ++i) {
                if (cschema[i] == ',') {
                    cschema[i++] = 0;
                    if (j < 5)
                        color[++j] = cschema + i;
                }
            }
        }
        if ((cschema!=NULL) && (strcmp(cschema,"c") == 0)) {
            r = XAllocNamedColor(xd,cmap,"black",&black,&exact) &&
                XAllocNamedColor(xd,cmap,"pink",&gray,&exact) &&
                XAllocNamedColor(xd,cmap,"white",&white,&exact) &&
                XAllocNamedColor(xd,cmap,"violet",&darkgray,&exact) &&
                XAllocNamedColor(xd,cmap,"gray50",&vdgray,&exact);
        }
        else if (j == 4) {
            r = XAllocNamedColor(xd,cmap,color[0],&black,&exact) &&
                XAllocNamedColor(xd,cmap,color[1],&gray,&exact) &&
                XAllocNamedColor(xd,cmap,color[2],&white,&exact) &&
                XAllocNamedColor(xd,cmap,color[3],&darkgray,&exact) &&
                XAllocNamedColor(xd,cmap,color[4],&vdgray,&exact);
        }
        else {
            r = XAllocNamedColor(xd,cmap,"black",&black,&exact) &&
                XAllocNamedColor(xd,cmap,"gray80",&gray,&exact) &&
                XAllocNamedColor(xd,cmap,"white",&white,&exact) &&
                XAllocNamedColor(xd,cmap,"gray60",&darkgray,&exact) &&
                XAllocNamedColor(xd,cmap,"gray40",&vdgray,&exact);
        }

        if ((!r) && verbose) {
            warn("Clara OCR could not alloc some colors. As a\n");
            warn("result, the application window may become\n");
            warn("unreadable. As a workaround, try to close\n");
            warn("Clara OCR and some applications that alloc\n");
            warn("many colors (e.g. Netscape) and start\n");
            warn("Clara OCR again\n");
        }

        /*
            Create window at (WX,WY), border width 2. BUG: (WX,WY) is
            being ignored (why?).
        */
        XW = XCreateSimpleWindow (xd,
                DefaultRootWindow(xd),WX,WY,WW,WH,2,
                white.pixel,darkgray.pixel);
        xw = (use_xb) ? pm : XW;
        have_xw = 1;

        /* ask for events */
        XSelectInput(xd,XW,evmasks);

        /* pop this window up on the screen */
        if (XMapRaised(xd,XW) == BadWindow) {
            fatal(XE,"could not create window");
        }

        /* graphics context(s) */
        if (!use_xb) {
            xgc = XCreateGC(xd,xw,0,0);
        }

        /* window ready flag */
        wnd_ready = 1;
    }

    /* initialize document windows */
    init_dws(1);

    /* PAGE special settings */
    dw[PAGE].rf = MRF;

    /* set the window title */
    if (batch_mode == 0) {
        char s[81];

        snprintf(s,80,"Clara OCR");
        s[80] = 0;
        swn(s);
    }

}

/*

Copy the viewable region on the PAGE (fatbits) window to
cfont.

*/
#define MAXHL 10
void copychar(void)
{
    int l,r,t,b,w,h;
    int k,cs[MAXHL],dx[MAXHL],dy[MAXHL],n;
    sdesc *y;

    /* make sure we're on the correct window */
    CDW = PAGE_FATBITS;

    /* mc limits */
    l = X0;
    r = X0 + FS - 1;
    t = Y0;
    b = Y0 + FS - 1;
    w = r-l+1;
    h = b-t+1;

    /* clear cfont */
    memset(cfont,WHITE,FS*FS);

    /* look for closures that intersect the rectangle (l,r,t,b) */
    for (n=0,k=0; k<=topps; ++k) {
        int i,j;
        int ia,ib,ja,jb;

        /*
            Does the symbol k have intersection with the
            region we're interested on?
        */
        y = mc + ps[k];
        if ((intersize(y->l,y->r,l,r,NULL,NULL) > 0) &&
            (intersize(y->t,y->b,t,b,NULL,NULL) > 0)) {

            /*
                Is the symbol k entirely contained in the
                region we're interested on?
            */
            if ((n<MAXHL) && (l<=y->l) && (y->r<=r) && (t<=y->t) && (y->b<=b)) {
                dx[n] = y->l - l;
                dy[n] = y->t - t;
                cs[n] = ps[k];
                ++n;
            }

            /*

                Now we need to compute the values of i and j for which
                (l+i,t+j) is a point of the closure k and of
                the rectangle (l,r,t,b). The minimum and maximum such
                values will be ia, ib, ja and jb:

                    i = ia, ia+1, ..., ib
                    j = ja, ja+1, ..., jb

            */
            if (y->l <= l)
                ia = 0;
            else
                ia = y->l - l;
            if (y->r >= r)
                ib = FS - 1;
            else
                ib = y->r - l;
            if (y->t <= t)
                ja = 0;
            else
                ja = y->t - t;
            if (y->b >= b)
                jb = FS - 1;
            else
                jb = y->b - t;

            /* copy the sampled pixels to the buffer */
            for (i=ia; i<=ib; ++i) {
                for (j=ja; j<=jb; ++j) {

                    if (spixel(y,l+i,t+j) == BLACK)
                        cfont[i+j*FS] = BLACK;
                    else
                        cfont[i+j*FS] = WHITE;
                }
            }
        }
    }

    /* compute the skeleton on the fly */
    if ((*cm_v_skel != ' ') || (*cm_v_border != ' ')) {
        int u,v,d,e,p;

        memset(cb,WHITE,LFS*FS);
        for (u=0; u<FS; ++u)
            for (v=0; v<FS; ++v)
                cb[u+v*LFS] = cfont[u+v*FS];
        if (*cm_v_skel != ' ')
            skel(-1,-1,0,0);
        else
            cb_border(0,0);
        for (v=0; v<FS; ++v) {
            d = v*LFS;
            e = v*FS;
            for (u=0; u<FS; ++u)
                 if ((p=cb[u+d]) == GRAY)
                     cfont[u+e] = BLACK;
                 else if (p == BLACK)
                     cfont[u+e] = WHITE;
        }
    }

    for (k=0; k<n; ++k) {
        int u,v,d,e,p;

        /* Show the pattern border */
        if ((*cm_v_hb != ' ') && (mc[cs[k]].bm >= 0)) {
            {
                int i,j,h;
                unsigned char m,*p;
                pdesc *d;
    
                memset(cb,WHITE,LFS*FS);
                d = pattern + id2idx(mc[cs[k]].bm);
                for (j=0; j<d->bh; ++j) {
                    p = d->b + BLS*j;
                    m = 128;
                    h = (j+dy[k])*LFS;
                    for (i=0; i<d->bw; ++i) {
                        if ((*p & m) != 0)
                            cb[i+dx[k]+h] = BLACK;
                        if ((m >>= 1) <= 0) {
                            ++p;
                            m = 128;
                        }
                    }
                }
            }
            cb_border(0,0);
            for (v=0; v<FS; ++v) {
                d = v*LFS;
                e = v*FS;
                for (u=0; u<FS; ++u)
                    if ((p=cb[u+d]) == GRAY)
                        cfont[u+e] |= D_MASK;
            }
        }

        /* Show the pattern skeleton */
        if ((*cm_v_hs != ' ') && (mc[cs[k]].bm >= 0)) {
            int i,j,h;
            unsigned char m,*p;
            pdesc *d;

            d = pattern + id2idx(mc[cs[k]].bm);
            dx[k] += (d->bw - d->sw) / 2;
            dy[k] += (d->bh - d->sh) / 2;
            for (j=0; j<d->sh; ++j) {
                p = d->s + BLS*j;
                m = 128;
                h = (j+dy[k])*FS;
                for (i=0; i<d->sw; ++i) {
                    if ((*p & m) != 0)
                        cfont[i+dx[k]+h] = WHITE;
                    if ((m >>= 1) <= 0) {
                        ++p;
                        m = 128;
                    }
                }
            }
        }
    }

    redraw_dw = 1;
    redraw_grid = 1;
}

/*

comp_wnd_size: compute the application window size.

This routine decides about the window size. The decision process
is as follows:

1. If a window size is specified through the parameters ww
(width) and wh (height), this is mandatory, but must be
larger or equal the minimum size.

2. If the caller specify ww==0 and wh==0, then the routine
computes the minimum width and height, and set the window size to
these values.

3. If ww==-1 and wh==-1, then use the current window size, but
enlarge it if necessary.

*/
void comp_wnd_size(int ww,int wh)
{
    int i,dx,dy;
    int w,h;
    unsigned cww,cwh;
    int old_pt,old_ph;

    /* remember PT and PH */
    old_pt = PT;
    old_ph = PH;

    /* tab height is compute from the font height */
    TH = DFH + 6;

    /* menu bar height */
    MH = 2 + VS + DFH + VS + 2;

    /*
        Largest alphabet label size (currently hardcoded...)
    */
    BW = 8;

    /*
        Compute button width and height (the globals BW and BH)
        from the button labels and the font size.
    */
    for (i=0; i<BUTT_PER_COL; ++i) {
        int a,b;
        char *bl;

        bl = BL[i];
        for (a=b=0; a >= 0; ++a) {
            if ((bl[a] == ',') || (bl[a] == ':') || (bl[a] == 0)) {
                if (a-b > BW)
                    BW = a - b;
                if (bl[a] == 0)
                    a = -2;
                else
                    b = a;
            }
        }
    }
    BW += 1;
    BW *= DFW;
    BH = DFH + 4*VS;

    /*
        Compute minimum plate width and height.

        The plate width and height are computed basically from FS
        and ZPS, in order to the FSxFS grid fit in the plate.

        In order to guarantee minimum visibility for the
        pattern properties window, we add 30*DFW.

        The clearance is 10. The plate height must include the
        tab height (TH).

        Obs. these formulae are inverted at the end of this
             routine, so if you change them, please change that
             code too.
    */
    PW = 10 + (FS * (ZPS+1) + 10 + 30*DFW + 10 + RW) + 10;
    PH = TH + 10 + (FS * (ZPS+1) + 10 + RW) + 10;

    /* plate horizontal and vertical margins */
    PM = 10 + BW + (((cm_v_map == NULL) || (*cm_v_map == ' ')) ? 10 : 62);
    PT = MH + 10;

    /* get current width and height, if any */
    if (have_xw) {
        Window root;
        int x,y;
        unsigned b,d;

        XGetGeometry(xd,XW,&root,&x,&y,&cww,&cwh,&b,&d);
    }
    else {
        cww = 0;
        cwh = 0;
    }

    /*
        These are the expected differences (WH-PH) and (WW-PW)
        between the total height and width and the plate height
        and width.        
    */
    dy = MH + 10 + 10 + DFH + 10;
    dx = PM + 10;

    /* oops.. requested height is mandatory */
    if (wh > 0) {
        WH = wh;

        /* plate height */
        PH = WH - dy;
    }

    /*
        First we compute the minimum plate height, then we add
        dy to obtain the total width.
    */
    {
        int j;

        /*
            the buttons must not exceed the plate
            vertical limits.
        */
        if (PH < (h=BUTT_PER_COL * (BH+4+VS)))
            PH = h;

        /*
            the welcome window must fit vertically in the plate
            (15*DFH is an approximation to its height.
        */
        if (PH < (h=15 * (DFH+VS)))
            PH = h;

        /*
            the alphabet map must not exceed
            the plate vertical limits.
        */
        if ((cm_v_map != NULL) && (*cm_v_map != ' ') && (PH < (h=16*max_sz)))
            PH = h;

        /*
            if the caller requests to preserve the geometry,
            then try not changing it.
        */
        if ((wh < 0) && (PH < (h=cwh-dy)))
            PH = h;

        /*
           is there a large menu?
        */
        for (j=0; j<=TOP_CM; ++j) {
            if (PH < (h=CM[j].n * (DFH+VS)))
                PH = h;
        }

        /*
           add vertical space to the menu bar, status line and
           clearances
        */
        WH = PH + dy;
    }

    /* oops.. requested width is mandatory */
    if (ww > 0) {
        WW = ww;

        /* plate width */
        PW = WW - dx;
    }

    /*
        First we compute the minimum plate width, then we add
        dx to obtain the total width.
    */
    {
        int a,i,lw;

        /* must fit 80 characters horizontally (status line) */
        if (PW < (w=80*DFW-dx))
            PW = w;

        /*
            if the caller requests to preserve the geometry,
            then try not changing it.
        */
        if ((ww < 0) && (PW < (w=cww-dx)))
            PW = w;

        /* minimum tab width */
        for (i=0, lw=0; i<=TOP_DW; ++i)
            if (lw < (a=DFW*strlen(dwname(i))))
                lw = a;
        if (PW < (w=((lw+20)+20)*TABN))
            PW = w;

        /*
           add button width, alphabet map width (if visible) and
           clearances
        */
        WW = PW + dx;
    }

    /* resize the window if the current geometry differs from (WW,WH) */
    if (have_xw) {
        if ((WW != cww) || (WH != cwh)) {
            /* db("resizing window"); */
            XResizeWindow(xd,XW,WW,WH);
        }
    }

    /* alloc or reallocate the pixmap */
    if ((use_xb) && ((!have_pm) || (WW > pmw) || (WH > pmh))) {
        int d;

        /* free the current buffer (if any) */
        if (have_pm) {
            XFreeGC(xd,xgc);
            XFreePixmap(xd,pm);
        }

        /* alloc a new one */
        d = DefaultDepth(xd,DefaultScreen(xd));
        pm = XCreatePixmap(xd,DefaultRootWindow(xd),WW,WH,d);

        /* could not alloc: switch to unbuffered mode */
        if (pm == BadAlloc) {
            warn("could not alloc buffer, switching to unbuffered mode\n");
            have_pm = 0;
            use_xb = 0;
            xw = XW;
            if (have_xw) {
                xgc = XCreateGC(xd,XW,0,0);
            }
        }

        /* success */
        else {
            have_pm = 1;
            pmw = WW;
            pmh = WH;
            xw = pm;
            xgc = XCreateGC(xd,pm,0,0);
        }
    }

    /* tab width */
    TW = PW/TABN - 20;

    /* choose (the best possible)/(a sane) ZPS */
    {
        int z,zx,zy;

        zx = ((PW - 10 - 10 - 10 - 30*DFW - 10 - RW) / FS) - 1;
        if ((zx & 1) == 0)
            zx -= 1;
        zy = ((PH - TH - 10 - 10 - 10 - RW) / FS) - 1;
        if ((zy & 1) == 0)
            zy -= 1;
        if ((z=zx) > zy)
            z = zy;

        /* the best possible */
        /*
        if ((1 <= z) && (ZPS != z))
            ZPS = z;
        */

        /* a sane */
        if ((1 <= z) && (z < ZPS))
            ZPS = z;
    }

    /*
        PAGE special settings.

        When resizing the window, we try to preserve the same
        relative height for the windows PAGE, PAGE_OUTPUT and
        PAGE_SYMBOL, that share one mode of the PAGE tab.

        The height of these three windows is determined by
        the variables page_j1 and page_j2 (first and second
        junctions). So we need to recompute them.

          +------------------------------------------+
          | File Edit...                             |
          +------------------------------------------+
          | +------+   +----+ +--------+ +-------+   |
          | | zoom |   |page| |patterns| | tune  |   |
          | +------+ +-+    +-+        +-+       +-+ |
          | +------+ | +-------------------------+ | |
          | | zone | | | (PAGE)                  | | |
          | +------+ | |                         | | |
          | +------+ | +-------------------------+ | |
          | | OCR  | |                             | | - page_j1
          | +------+ | +-------------------------+ | |
          | +------+ | | (PAGE_OUTPUT)           | | |
          | | stop | | |                         | | |
          | +------+ | +-------------------------+ | |
          |     .    |                             | | - page_j2
          |     .    | +-------------------------+ | |
          |          | | (PAGE_SYMBOL)           | | |
          |          | |                         | | |
          |          | +-------------------------+ | |
          |          +-----------------------------+ |
          |                                          |
          | (status line)                            |
          +------------------------------------------+

    */
    if (page_j1 < 0) {
        opage_j1 = page_j1 = PT+TH+(PH-TH)/3;
        opage_j2 = page_j2 = PT+TH+2*(PH-TH)/3;
    }
    else {
        float a;

        a = ((float)(old_ph-TH)) / (page_j1-old_pt-TH);
        page_j1 = PT+TH+(PH-TH)/a;
        a = ((float)(old_ph-TH)) / (page_j2-old_pt-TH);
        page_j2 = PT+TH+(PH-TH)/a;

        /*
           make sure that the height of PAGE_OUTPUT is
           at least 4*RW.
        */
        if (page_j2 - page_j1 < 20+4*RW)
            page_j1 = page_j2 - 20 - 4*RW;

        /*
            make sure that the height of PAGE is at least
            4*RW.
        */
        if (page_j1-PT < TH+30+4*RW) {
            page_j1 = PT+TH+30+4*RW;
            page_j2 = page_j1+20+4*RW;
        }

        /*
            make sure that the height of PAGE_SYMBOL is
            at least 4*RW.
        */
        if (PT+PH-page_j2 <30+4*RW) {
            page_j2 = PT+PH-30-4*RW;
            page_j1 = page_j2-20-4*RW;
        }

        opage_j1 = page_j1;
        opage_j2 = page_j2;
    }
}

#ifdef HAVE_SIGNAL
/*

This is a first draft for a presentation routine. By now it
merely changes to large font, then to small font, returns to
default font and stops.

Obs. any call to new_alrm must be done just before returning from
this service.

*/
void cpresentation(int reset)
{
    /* presentation and current state */
    static int p,s;

    if ((have_s_ev) || (ocring)) {
        new_alrm(100000,2);
        return;
    }

    /* initialize presentation */
    if (reset) {
        pmode = 1;
        p = s = 0;
        xev.type = FocusIn;
        have_s_ev = 1;
        new_alrm(100000,2);
        return;
    }

    /*
        Presentation 0: change font
    */
    if (p == 0) {
        int delay;

        if (s == 0) {

            ++s;

            if (!mselect(CM_O_LARGE,&delay))
                ++s;
        }

        else if (s == 1) {
            if (!mselect(CM_O_SMALL,&delay))
                ++s;
        }

        else if (s == 2) {
            if (!mselect(CM_O_DEF,&delay))
                ++s;
        }

        else if (s == 3) {
            if (!fselect("imre.pbm",&delay))
                ++s;
        }

        else if (s == 4) {
            if (!sselect(5,&delay))
                ++s;
        }

        else if (s == 5) {
            if (!press('r',&delay))
                ++s;
        }

        else if (s == 6) {
            if (!sselect(235,&delay))
                ++s;
        }

        else if (s == 7) {
            if (!press('h',&delay))
                ++s;
        }

        else if (s == 8) {
            if (!bselect(bocr,&delay))
                ++s;
        }

        else if (s == 9) {
            pmode = 0;
            if (allow_pres == 2)
                finish = 1;
        }

        if (pmode) {
            if (allow_pres == 2)
                new_alrm(250000,2);
            else
                new_alrm(delay,2);
        }
    }
}
#endif
