package Language::INTERCAL::Parser::CLCintercal;

# Parser for CLC-INTERCAL

# This file is part of CLC-INTERCAL.

# Copyright (C) 1999 Claudio Calvelli <lunatic@assurdo.com>, all rights reserved

# WARNING - do not operate heavy machinery while using CLC-INTERCAL

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use vars qw($VERSION);
$VERSION = '0.05';

use Language::INTERCAL::Runtime::Library;
use Language::INTERCAL::Opcodes;

sub parse {
    _parse('CLC', @_);
}

sub _parse {
    my ($language, $source, $listing, $filename, $toggle) = @_;
    my %toggle = %$toggle;
    my %flags = ();
    my @seqblocks = ($filename);
    my @cblock = ();
    my %labels = ();
    my $stmtno = 0;
    my $pleaseno = 0;

    my %data = (
	'toggle' => \%toggle,
	'flags' => \%flags,
	'registers' => {},
	'come_froms' => [],
	'abstain' => [],
	'reinstate' => [],
	'fork' => [],
	'gabstain' => [],
	'greinstate' => [],
	'gfork' => [],
	'nexts' => [],
	'files' => [\@seqblocks],
	'labels' => \%labels,
	'name' => $filename,
	'language' => $language,
    );

    output($listing, "*    STMT #  (LBL)   STATEMENT\n");
    output($listing, "\n");

    $source =~ s/^\s*//;
    my $offset = length($&);

    while ($source ne '') {
	$stmtno++;
	my $stmt = roman($stmtno, $toggle->{'roman'});

	my $splat = 0;
	my $orig = $source;
	my $delta = 0;

	my $label = 0;
	my $please = 0;
	my $abstain = 1;
	my $double_oh_seven = 100;
	my $prn_double_oh_seven = '';
	my $statement = 0;
	my $prn_statement = '';
	my $end_block = 0;

	# first component - label
	if ($source =~ s/^\(\s*(\d+)\s*\)\s*//) {
	    $label = 0 + $1;
	    $splat = 458 if exists $labels{$label};
	    $splat = 72 if $label < 1 || $label > 65535;
	    if (@cblock || ! @seqblocks) {
		@cblock = ([0, 1, 100, [S_NOP], 0, 0]) if ! @cblock;
		push @seqblocks, [@cblock];
	    }
	    @cblock = ();
	    $end_block = 1;
	}

	# second component - PLEASE, DO, or PLEASE DO
	if (! $splat) {
	    $delta += length($orig) - length($source);
	    $orig = $source;
	    $please = 1 if $source =~ s/^PLEASE\s*//i;
	    $pleaseno += $please;
	    $splat = 81 if $source !~ s/^DO\s*//i && ! $please;
	}

	# third/fourth component - optional NOT, optional DOUBLE-OH-SEVEN
	if (! $splat) {
	    $abstain = 0 if $source =~ s/^NOT\s*//i
			 || $source =~ s/^\s*//
			 || $source =~ s/^N\s*'\s*T\s*//i;
	    if ($source =~ s/^%\s*(\d+)\s*//) {
		$double_oh_seven = 0 + $1;
		$prn_double_oh_seven =
		    '%' . roman($double_oh_seven, $toggle->{'roman'}) . ' ';
		$splat = 100 if $double_oh_seven > 100;
	    } elsif ($language eq 'CLC' && $source =~ s/^%\s*//) {
		($source, $splat, $double_oh_seven, $prn_double_oh_seven) =
		    _parse_expression($source, 1, \%data, 0);
		$prn_double_oh_seven = '%' . $prn_double_oh_seven . ' ';
	    }
	}

	# fifth component - the actual statement
	if (! $splat) {
	    my $eb;
	    ($source, $splat, $statement, $prn_statement, $eb) =
		_parse_statement($source, 1, \%data, scalar(@cblock), $stmtno);
	    my $end_block ||= $eb;
	}

	# recognise comments
	if ($splat) {
	    $source = substr($source, 1) if $source eq $orig;
	    if ($source =~ /PLEASE|DO|\(\d+\)/i) {
		$source = $& . $';
	    } else {
		$source = '';
	    }
	    $prn_statement = substr($orig, 0, length($orig) - length($source));
	    $prn_statement =~ s/^\s+//;
	    $prn_statement =~ s/\s+$//;
	    $prn_statement =~ s/\t/ /g;
	    my $s = $splat;
	    $s = 0 if $s < 0;
	    $statement = [S_ILLEGAL, sprintf("%03d ", $s) . $prn_statement];
	}

	# and add it to the current block, shifting the block if this must end
	if ($statement->[0] == S_COME || $statement->[0] == S_LEARN) {
	    push @seqblocks, [@cblock] if @cblock;
	    @cblock = ();
	}
	if ($statement->[0] == S_COME) {
	    my $ccf = $data{'come_froms'};
	    push @$ccf, [$statement->[1], 0, scalar(@seqblocks), 0, $stmtno];
	}
	push @cblock,
	    [$label, $abstain, $double_oh_seven, $statement, $stmtno, $offset];
	if ($label) {
	    $labels{$label} =
		[0, scalar(@seqblocks), $#cblock,
		 roman($label, $toggle->{'roman'}), $stmtno, {}];
	}
	if ($end_block) {
	    push @seqblocks, [@cblock];
	    @cblock = ();
	}

	# print statement if required
	if (defined $listing) {
	    my $lab =
		$label ? "(" . roman($label, $toggle->{'roman'}) . ")" : '';
	    my $text = ($splat ? ''
			       : (("DON'T", "DO", "PLEASE DON'T", "PLEASE")
				  [$please * 2 + $abstain]) . ' ')
		     . $prn_double_oh_seven
		     . $prn_statement;
	    if ($splat) {
		$splat = sprintf "*%03d", $splat < 0 ? 0 : $splat;
	    } else {
		$splat = '';
	    }
	    my $width = $toggle{'width'};
	    my $fmt = "%-4s %-7s %s ";
	    my $indent = sprintf($fmt, $splat, $stmt, $lab);
	    $indent .= ' ' x (21 - length($indent)) if length($indent) < 21;
	    while (length($indent . $text) > $width || $text =~ /\n/) {
		my $t = substr($text, 0, $width + 1 - length($indent));
		unless ($t =~ s/\s+$//) {
		    $t =~ s/\s+\S+$//;
		}
		$t = $` if $t =~ /\n/;
		if ($t eq '' || length($t) > $width - length($indent)) {
		    $t = substr($text, 0, $width - length($indent));
		    $t = $` if $t =~ /\n/;
		}
		output($listing, $indent . $t . "\n");
		$stmt = $lab = '';
		$text = substr($text, length($t));
		$text =~ s/^\s+//;
		$indent = sprintf($fmt, $splat, $stmt, $lab);
		$indent .= ' ' x (21 - length($indent)) if length($indent) < 21;
	    }
	    output($listing, $indent . $text . "\n");
	}
	die "Source unchanged ($orig) ($source)???\n" if $orig eq $source;
	$delta += length($orig) - length($source);
	$offset += $delta;
    }

    push @seqblocks, [@cblock] if @cblock;
    push @seqblocks, [[0, 1, 100, [S_NOP], 0, 0]] if ! @seqblocks;

    my $bug = $toggle->{'bug'};
    my $ubug = $toggle->{'ubug'};
    if ($bug || $ubug) {
	my $block = 1 + int(rand(scalar(@seqblocks) - 1));
	my $stmt = int(rand(scalar @{$seqblocks[$block]}));
	splice @{$seqblocks[$block]}, $stmt, 0,
	       [0, 1, $bug || $ubug, [$bug ? S_BUG : S_UBUG], 0, -1];
	for my $label (keys %{$data{'labels'}}) {
	    $data{'labels'}{$label}[2]++
		if $data{'labels'}{$label}[1] == $block
		&& $data{'labels'}{$label}[2] >= $stmt;
	}
	for my $name (qw(come_froms abstain reinstate fork
			 nexts gabstain greinstate gfork))
	{
	    for my $item (@{$data{$name}}) {
		$item->[3] ++ if $item->[2] == $block && $item->[3] >= $stmt;
	    }
	}
    }

    die "499 YOU NEED TO SAY \"PLEASE\" TO COMPILE THIS PROGRAM\n"
	if $stmtno > 3 && $pleaseno * 5 < $stmtno;
    die "499 THIS IS FAR TOO POLITE\n"
	if $stmtno > 3 && $pleaseno * 3 > $stmtno;

    return \%data;
}

sub _parse_statement {
    my ($source, $really, $data, $stmtid, $stmtno) = @_;

    my ($src, $splat, $stm, $stmtext, $eblock) =
	_parse_sub_statement($source, $really, $data, $stmtid, $stmtno);
    if ($splat) {
	return ($src, $splat, $stm, $stmtext, $eblock)
	    if $data->{'language'} ne 'CLC';
	my ($src, $splat, $stm, $stmtext) =
	    _parse_expression($source, $really, $data, 0);
	if (! $splat && $src =~ s/^WHILE\s*//) {
	    my ($src1, $splat1, $stm1, $stmtext1, $eblock1) =
		_parse_sub_statement($src, $really, $data, $stmtid, $stmtno);
	    if (! $splat1) {
		return ($src1, 0, [S_WHILE_E, $stm1, $stm],
			"$stmtext WHILE $stmtext1", 1);
	    }
	}
    }
    return ($src, $splat, $stm, $stmtext, $eblock) if $splat;

    if ($data->{'language'} eq 'CLC' && $src =~ /^WHILE\s*/) {
	my $s = $';
	my ($src1, $splat1, $stm1, $stmtext1, $eblock1) =
	    _parse_sub_statement($s, $really, $data, $stmtid, $stmtno);
	if (! $splat1) {
	    return ($src1, 0, [S_WHILE_CB, $stm, $stm1],
		    "$stmtext WHILE $stmtext1", 1);
	}
	($src1, $splat1, $stm1, $stmtext1) =
	    _parse_expression($s, $really, $data, 0);
	if (! $splat1) {
	    return ($src1, 0, [S_WHILE_E, $stm, $stm1],
		    "$stmtext WHILE $stmtext1", 1);
	}
    }

    return ($src, 0, $stm, $stmtext, $eblock);
}

sub _parse_sub_statement {
    my ($source, $really, $data, $stmtid, $stmtno) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [T_STATEMENT], 'STATEMENT')
	    if $source =~ s/^STATEMENT\s*//i;
	return ($source, 218, [], '');
    }
    if ($really < 0) {
	$really = 0;
    }

    return _parse_reinstate($source, $really, $data, $stmtid, $stmtno, 0)
	if $source =~ s/^ABSTAIN\s*FROM\s*//i;
    return _parse_come_from($source, $really, $data)
	if $source =~ s/^COME\s*FROM\s*//i;
    return _parse_convert($source, $really, $data, $stmtid, $stmtno)
	if $data->{'language'} eq 'CLC' && $source =~ s/^CONVERT\s*//i;
#    return _parse_create($source, $really, $data, $stmtid, $stmtno)
#	if $data->{'language'} eq 'CLC' && $source =~ s/^CREATE\s*//i;
    return _parse_enrol($source, $really, $data)
	if $data->{'language'} eq 'CLC' && $source =~ s/^ENROL\s*//i;
    return _parse_enslave($source, $really, $data)
	if $data->{'language'} eq 'CLC' && $source =~ s/^ENSLAVE\s*//i;
    if ($data->{'language'} eq 'CLC' && $source =~ s/^FINISH\s*LECTURE\s*//i) {
	return ($source, 0, [S_FINISH], 'FINISH LECTURE', 1);
    }
    return _parse_resume($source, $really, $data, 0)
	if $source =~ s/^FORGET\s*//i;
    return _parse_free($source, $really, $data)
	if $data->{'language'} eq 'CLC' && $source =~ s/^FREE\s*//i;
    if ($source =~ s/^GIVE\s*UP\s*//i) {
	return ($source, 0, [S_STOP], 'GIVE UP', 0);
    }
    return _parse_remember($source, $really, $data, 0)
	if $source =~ s/^IGNORE\s*//i;
    return _parse_read($source, $really, $data)
	if $source =~ s/^READ\s*OUT\s*//i;
    return _parse_reinstate($source, $really, $data, $stmtid, $stmtno, 1)
	if $source =~ s/^REINSTATE\s*//i;
    return _parse_remember($source, $really, $data, 1)
	if $source =~ s/^REMEMBER\s*//i;
    return _parse_resume($source, $really, $data, 1)
	if $source =~ s/^RESUME\s*//i;
    return _parse_stash($source, $really, $data, 0)
	if $source =~ s/^RETRIEVE\s*//i;
    return _parse_stash($source, $really, $data, 1)
	if $source =~ s/^STASH\s*//i;
    return _parse_study($source, $really, $data)
	if $data->{'language'} eq 'CLC' && $source =~ s/^STUDY\s*//i;
    return _parse_swap($source, $really, $data, $stmtid, $stmtno)
	if $data->{'language'} eq 'CLC' && $source =~ s/^SWAP\s*//i;
    return _parse_write($source, $really, $data)
	if $source =~ s/^WRITE\s*IN\s*//i;

    my ($src, $splat, $reg, $regtext) =
	_parse_register($source, $really, $data, 0, 0);
    if (! $splat) {
	return ($src, 0, [S_GRADUATES, $reg], "$regtext GRADUATES", 0)
	    if $data->{'language'} eq 'CLC' && $src =~ s/^GRADUATES\s*//i;
	return _parse_learns($src, $really, $data, $reg, $regtext)
	    if $data->{'language'} eq 'CLC' && $src =~ s/^LEARNS\s*//i;
	return _parse_sub_assign($src, $really, $data, $reg, $regtext)
	    if $src =~ s/^SUB\s*//i;
	return _parse_reg_assign($src, $really, $data, $reg, $regtext)
	    if $src =~ /^<-\s*/i;
    }

    ($src, $splat, $reg, $regtext) =
	_parse_label($source, $really, $data);
    if (! $splat) {
	if ($really) {
	    my $ab = $data->{'nexts'};
	    my $block_id = @{$data->{'files'}[0]};
	    push @$ab, [$reg, 0, $block_id, $stmtid, $stmtno];
	}
	return ($src, 0, [S_NEXT, $reg], "$regtext NEXT", 1)
	    if $src =~ s/^NEXT\s*//i;
    }

    return ($source, -1, [], '', 0);
}

#sub _parse_statement_list {
#    my ($source, $really, $data, $stmtid, $stmtno) = @_;
#
#    if (! $really) {
#	# fake parse
#	return ($source, 0, [T_STATEMENT_LIST], 'STATEMENT LIST')
#	    if $source =~ s/^STATEMENT\s*LIST\s*//i;
#	return ($source, 218, [], '');
#    }
#    if ($really < 0) {
#	$really = 0;
#    }
#
#    my ($src, $splat, $stm, $stmtext, $eblock) =
#	_parse_statement($source, $really, $data, $stmtid, $stmtno);
#    return ($src, $splat, $stm, $stmtext, $eblock) if $splat;
#    my @stm = ($stm);
#    my $text = $stmtext;
#    while ($src =~ s/^\+\s*//) {
#	($src, $splat, $stm, $stmtext, $eblock) =
#	    _parse_statement($src, $really, $data, $stmtid, $stmtno);
#	return ($src, $splat, $stm, $stmtext, 0) if $splat;
#	push @stm, $stm;
#	$text .= ' + ' . $stmtext;
#    }
#
#    return ($src, 0, [@stm], $text, $eblock);
#}

sub _parse_sub_assign {
    my ($source, $really, $data, $reg, $regtext) = @_;
    my ($src, $splat, $sub, $subtext) =
	_add_subscripts($source, $really, $data, $reg, $regtext, 0);
    return ($source, $splat, [], '', 0) if $splat;
    return _parse_reg_assign($src, $really, $data, $sub, $subtext);
}

sub _parse_reg_assign {
    my ($source, $really, $data, $reg, $regtext) = @_;
    return ($source, 93, [], '', 0) if $source !~ s/^<-\s*//;
    my ($src, $splat, $exp, $exptext) =
	_parse_expression($source, $really, $data, 0);
    return ($src, $splat, $exp, $exptext, 0) if $splat;
    my $text = $regtext . ' <- ' . $exptext;
    return ($src, 0, [S_ASSIGN, $reg, $exp], $text, 0) if $src !~ /^BY/i;
    return ($src, 241, [], '', 0) if ! _can_subscript($reg);
    my @subs = ($exp);
    while ($src =~ s/^BY\s*//i) {
	($src, $splat, $exp, $exptext) =
	    _parse_expression($src, $really, $data, 0);
	return ($src, $splat, $exp, $exptext, 0) if $splat;
	$text .= ' BY ' . $exptext;
	push @subs, $exp;
    }
    return ($src, 0, [S_ASSIGN, $reg, @subs], $text, 0);
}

sub _add_subscripts {
    my ($source, $really, $data, $reg, $regtext, $at0) = @_;
    return ($source, 241, [], '') if ! _can_subscript($reg);
    my @subs = ();
    $regtext .= ' SUB';
    while (1) {
	my ($src, $splat, $exp, $exptext) =
	    _parse_expression($source, $really, $data, $at0);
	return ($source, $splat, [], '', 0) if $splat && ! @subs;
	return ($source, 0, _join_subscripts($reg, @subs), $regtext) if $splat;
	$regtext .= ' ' . $exptext;
	push @subs, $exp;
	$source = $src;
    }
}

sub _can_subscript {
    my ($reg) = @_;
    $reg = $reg->[1] while $reg->[0] == E_AND ||
			   $reg->[0] == E_OR ||
			   $reg->[0] == E_XOR ||
			   $reg->[0] == E_INDIRECT;
    return 1 if $reg->[0] == E_OWNER; # find out at runtime
    return 0 if $reg->[0] != E_REGISTER;
    return 1 if $reg->[1] eq ';' || $reg->[1] eq ',';
    return 0;
}

sub _join_subscripts {
    my ($reg, @subs) = @_;
    my $prev = undef;
    while ($reg->[0] == E_AND || $reg->[0] == E_OR || $reg->[0] == E_XOR) {
	$prev = $reg;
	$reg = $reg->[1];
    }
    if ($reg->[0] == E_OWNER ||
	$reg->[0] == E_REGISTER ||
	$reg->[0] == E_INDIRECT)
    {
	my $r = [E_SUBSCRIPT, $reg, @subs];
	if (defined $prev) {
	    $prev->[2] = $r;
	} else {
	    $_[0] = $r;
	}
    }
}

sub _can_be_class {
    my ($reg) = @_;
    $reg = $reg->[1] while $reg->[0] == E_AND ||
			   $reg->[0] == E_OR ||
			   $reg->[0] == E_XOR ||
			   $reg->[0] == E_INDIRECT;
    return 1 if $reg->[0] == E_OWNER; # find out at runtime
    return 0 if $reg->[0] != E_REGISTER;
    return 1 if $reg->[1] eq '@';
    return 0;
}

sub _can_be_overloaded {
    my ($reg) = @_;
    $reg = $reg->[1] while $reg->[0] == E_AND ||
			   $reg->[0] == E_OR ||
			   $reg->[0] == E_XOR ||
			   $reg->[0] == E_INDIRECT;
    return 1 if $reg->[0] == E_OWNER; # find out at runtime
    return 1 if $reg->[0] == E_SUBSCRIPT;
    return 0 if $reg->[0] != E_REGISTER;
    return 1 if $reg->[1] eq '.' || $reg->[1] eq ':';
    return 0;
}

sub _parse_expression {
    my ($source, $really, $data, $at0) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [T_EXPRESSION], 'EXPRESSION')
	    if $source =~ s/^EXPRESSION\s*//i;
	return ($source, 218, [], '');
    }

    # get an initial subexpression
    my ($src, $splat, $exp, $exptext);
    if ($source =~ s/^['"!]//) {
	my $ear = $&;
	$source =~ s/^\s+//;
	if ($ear eq '!') {
	    $ear = "'";
	    $source =~ s/^/./;
	}
	($src, $splat, $exp, $exptext) =
	    _parse_expression($source, $really, $data, $at0);
	if (! $splat && $src !~ s/^$ear\s*//) {
	    return ($src, $ear eq '"' ? 399 : 398, [], '');
	}
	$exptext = $ear . $exptext . $ear;
    } else {
	($src, $splat, $exp, $exptext) =
	    _parse_subexpression($source, $really, $data, $at0);
    }
    return ($src, $splat, $exp, $exptext) if $splat;

    while ($src =~ s!^([~]|[cC]\010/)\s*!! ||
    	  ($data->{'language'} eq 'C' && $src =~ s!^(\$)\s*!!) ||
    	  ($data->{'language'} eq 'CLC' && $src =~ s!^([/\\]|[cC]\010\|)\s*!!))
    {
	my $prn = $1;
	$prn = '' if length($prn) > 1 || $prn eq '$';
	my $op = {''  => E_INTERLEAVE,
		  '~'  => E_SELECT,
		  '/'  => E_OVERLOAD_REGISTER,
		  '\\' => E_OVERLOAD_RANGE}->{$prn};
	$prn = "C\010/" if $prn eq '' && $data->{'toggle'}{'mingle'};
	if ($op == E_OVERLOAD_REGISTER) {
	    return ($src, 514, [], '') if ! _can_be_overloaded($exp);
	    $data->{'flags'}{'overload'} = 1;
	} elsif ($op == E_OVERLOAD_RANGE) {
	    $data->{'flags'}{'overload'} = 1;
	}
	my ($e, $t);
	($src, $splat, $e, $t) =
	    _parse_expression($src, $really, $data,
			      $at0 || $op == E_OVERLOAD_RANGE
				   || $op == E_OVERLOAD_REGISTER);
	return ($src, $splat, $e, $t) if $splat;
	$exp = [$op, $exp, $e];
	$exptext .= ' ' . $prn . ' ' . $t;
    }

    return ($src, 0, $exp, $exptext);
}

sub _parse_expression_list {
    my ($source, $really, $data) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [[T_EXPRESSION]], 'EXPRESSION LIST')
	    if $source =~ s/^EXPRESSION\s*(:?LIST\s*)?//i;
	return ($source, 218, [], '');
    }

    my ($src, $splat, $exp, $exptext) =
	_parse_expression($source, $really, $data, 0);
    return ($src, $splat, $exp, $exptext) if $splat;
    my @exps = ($exp);
    my $text = $exptext;
    while ($src =~ s/^\+\s*//) {
	($src, $splat, $exp, $exptext) =
	    _parse_expression($src, $really, $data, 0);
	return ($src, $splat, $exp, $exptext, 0) if $splat;
	push @exps, $exp;
	$text .= ' + ' . $exptext;
    }

    return ($src, 0, [@exps], $text);
}

sub _parse_subexpression {
    my ($source, $really, $data, $at0) = @_;

    # rabbit ears or sparks
    if ($source =~ /^['"!]/) {
	return _parse_expression($source, $really, $data, $at0);
    }

    # unary operators
    if ($source =~ s/^([vV]\010-|[&V])\s*// ||
	($data->{'language'} eq 'C' && $source =~ s/^(\?)\s*//) ||
	($data->{'language'} eq 'CLC' && $source =~ s/^()\s*//))
    {
	my $unary = $1;
	my ($src, $splat, $exp, $exptext) =
	    _parse_expression($source, $really, $data, $at0);
	return ($src, $splat, $exp, $exptext) if $splat;
	return _fix_unary($src, $unary, $exp, $exptext, $data->{'toggle'});
    }

    # constant
    if ($source =~ /^#/) {
	return _parse_constant($source, $really, $data, 1);
    }

    # "numberof"
    if ($data->{'language'} eq 'CLC' && $source =~ s/^\-\s*//) {
	my ($src, $splat, $reg, $regtext) =
	    _parse_register($source, $really, $data, 0, $at0);
	return ($src, $splat, $reg, $regtext) if $splat;
	while ($reg->[0] == E_SUBSCRIPT || $reg->[0] == E_INDIRECT) {
	    $reg = $reg->[$reg->[0] == E_SUBSCRIPT ? 1 : 2];
	}
	return ($src, 0, [E_CONSTANT, pack("n", $reg->[2])], '-' . $regtext)
	    if $reg->[0] == E_REGISTER;
	return ($src, 0, [E_NUMBER, $reg], '-' . $regtext);
    }

    # splat
    if ($data->{'language'} eq 'CLC' && $source =~ s/^\*\s*//) {
	return ($source, 0, [E_SPLAT], '*');
    }

    # register?
    my ($src, $splat, $reg, $regtext) =
	_parse_register($source, $really, $data, 1, $at0);
    if (! $splat && $src =~ s/^SUB\s*//i) {
	($src, $splat, $reg, $regtext) =
	    _add_subscripts($src, $really, $data, $reg, $regtext, $at0);
    }
    return ($src, $splat, $reg, $regtext);
}

sub _fix_unary {
    my ($src, $unary, $exp, $exptext, $toggle) = @_;
    $unary = '' if length($unary) > 1 || $unary eq '?';
    my $op = $unary eq '&' ? E_AND : ($unary eq 'V' ? E_OR : E_XOR);
    $unary = "V\010-" if $unary eq '' && $toggle->{'xor'};
    return ($src, 0, [$op, $exp], $unary . $exptext);
}

sub _parse_constant {
    my ($source, $really, $data, $unaries) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [T_CONSTANT], 'CONSTANT')
	    if $source =~ s/^CONSTANT\s*//i;
	return ($source, 218, [], '');
    }

    return ($source, 199, [], '') if $source !~ s/^#\s*//;

    my @unary = ();
    while ($unaries &&
	   ($source =~ s/^([vV]\010-|[&V])\s*// ||
	    ($data->{'language'} eq 'C' && $source =~ s/^(\?)\s*//) ||
	    ($data->{'language'} eq 'CLC' && $source =~ s/^()\s*//)))
    {
	push @unary, $1;
    }

    if ($source =~ s/^(\d+)\s*//) {
	my $n = 0 + $1;
	return ($source, 199, [], '') if $n > 65535;
	my $exp = [E_CONSTANT, pack('n', $n)];
	my $exptext = '#' . roman($n, $data->{'toggle'}{'roman'});
	while (@unary) {
	    my $u = pop @unary;
	    my ($s, $t);
	    ($s, $t, $exp, $exptext) =
		_fix_unary('', $u, $exp, $exptext, $data->{'toggle'});
	}
	return ($source, 0, $exp, $exptext);
    }

    return ($source, 199, [], '');
}

sub _parse_register {
    my ($source, $really, $data, $unaries, $at0) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [T_REGISTER], 'REGISTER')
	    if $source =~ s/^REGISTER\s*//i;
	return ($source, 218, [], '');
    }

    my @belong = ();
    while ($data->{'language'} eq 'CLC' && $source =~ s/^([\$1-9])\s*//) {
	unshift @belong, $1 eq '$' ? 1 : $1;
    }

    my $regtype = 0;

    if ($data->{'language'} eq 'CLC' && $source =~ s/^([\@\+])\s*//) {
	$regtype = $1;
    } elsif ($source =~ s/^([,\.;:])\s*//) {
	$regtype = $1;
    } else {
	return ($source, 299, [], '');
    }

    my @unary = ();
    while ($unaries &&
	   ($source =~ s/^([vV]\010-|[&V])\s*// ||
	    ($data->{'language'} eq 'C' && $source =~ s/^(\?)\s*//) ||
	    ($data->{'language'} eq 'CLC' && $source =~ s/^()\s*//)))
    {
	push @unary, $1;
    }

    my $exp = 0;
    my $exptext = 0;

    if ($regtype eq '+') {
	my ($splat, $regtext, $rgt, $regnumber);
	($source, $splat, $regtype, $regtext) =
	    _parse_register($source, $really, $data, 0, $at0);
	return ($source, $splat, $regtype, $regtext) if $splat;
	return ($source, 242, [], '') if $source !~ s/^\-\s*//;
	($source, $splat, $regnumber, $rgt) =
	    _parse_register($source, $really, $data, 0, $at0);
	return ($source, $splat, $regtype, $regtext) if $splat;
	$regtype = $regtype->[1] while
	    $regtype->[0] == E_SUBSCRIPT || $regtype->[0] == E_INDIRECT;
	while ($regnumber->[0] == E_SUBSCRIPT || $regnumber->[0] == E_INDIRECT) {
	    $regnumber = $regnumber->[$regnumber->[0] == E_SUBSCRIPT ? 1 : 2];
	}
	if ($regtype->[0] == E_REGISTER && $regnumber->[0] == E_REGISTER) {
	    $exp = [E_REGISTER, $regtype->[1], $regnumber->[2]];
	} else {
	    $exp = [E_INDIRECT, $regtype, $regnumber];
	}
	$exptext = '+' . $regtext . '-' . $rgt;
    } elsif ($source =~ s/^(\d+)\s*//) {
	my $n = 0 + $1;
	return ($source, 299, [], '') if $n > 65535
				      || ($n == 0 && $regtype ne '@')
				      || ($n == 0 && ! $at0);
	$exp = [E_REGISTER, $regtype, $n];
	$exptext = $regtype . roman($n, $data->{'toggle'}{'roman'});
    } else {
	return ($source, 299, [], '');
    }

    while (@belong) {
	my $b = pop @belong;
	$exptext = ($b eq '1' || $b eq '$'
		    ? '$'
		    : roman($b, $data->{'toggle'}{'roman'}))
		 . ' ' . $exptext;
	$exp = [E_OWNER, $b eq '$' ? 1 : $b, $exp];
    }
    while (@unary) {
	my $u = pop @unary;
	my ($s, $t);
	($s, $t, $exp, $exptext) =
	    _fix_unary('', $u, $exp, $exptext, $data->{'toggle'});
    }
    return ($source, 0, $exp, $exptext);
}

sub _parse_register_list {
    my ($source, $really, $data, $subs) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [[T_REGISTER]], 'REGISTER')
	    if $source =~ s/^REGISTER\s*(?:LIST\s*)?//i;
	return ($source, 218, [], '');
    }

    my ($src, $splat, $reg, $regtext) =
	_parse_register($source, $really, $data, 0, 0);
    if ($subs && ! $splat && $src =~ s/^SUB\s*//i) {
	($src, $splat, $reg, $regtext) =
	    _add_subscripts($src, $really, $data, $reg, $regtext, 0);
    }
    return ($src, $splat, $reg, $regtext) if $splat;
    my @regs = ($reg);
    my $text = $regtext;
    while ($src =~ s/^\+\s*//) {
	($src, $splat, $reg, $regtext) =
	    _parse_register($src, $really, $data, 0, 0);
	return ($src, $splat, $reg, $regtext) if $splat;
	if ($subs && ! $splat && $src =~ s/^SUB\s*//i) {
	    ($src, $splat, $reg, $regtext) =
		_add_subscripts($src, $really, $data, $reg, $regtext, 0);
	}
	return ($src, $splat, $reg, $regtext) if $splat;
	push @regs, $reg;
	$text .= ' + ' . $regtext;
    }

    return ($src, 0, [@regs], $text);
}

sub _parse_label {
    my ($source, $really, $data) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [T_LABEL], 'LABEL')
	    if $source =~ s/^LABEL\s*//i;
	return ($source, 218, [], '');
    }

    if ($source =~ s/^\(\s*(\d+)\)\s*\s*//) {
	my $n = 0 + $1;
	return ($source, 72, [], '') if $n > 65535;
	my $exp = [E_CONSTANT, pack('n', $n)];
	my $exptext = '(' . roman($n, $data->{'toggle'}{'roman'}) . ')';
	return ($source, 0, $exp, $exptext);
    }

    return ($source, 72, [], '') if $data->{'language'} eq 'C';

    return _parse_expression($source, $really, $data, 0);
}

sub _parse_come_from {
    my ($source, $really, $data) = @_;
    my ($src, $splat, $lab, $labtext) =
	_parse_label($source, $really, $data);
    return ($src, $splat, $lab, $labtext, 0) if $splat;
    $labtext = "COME FROM " . $labtext;
    return ($src, 0, [S_COME, $lab], $labtext, 0);
}

sub _parse_reinstate {
    my ($source, $really, $data, $stmtid, $stmtno, $reinstate) = @_;
    my $opcode = $reinstate ? S_REINSTATE : S_ABSTAIN;
    my $gopcode = $reinstate ? S_GREINSTATE : S_GABSTAIN;
    my $prncode = $reinstate ? 'REINSTATE' : 'ABSTAIN FROM';
    my $record = $reinstate ? 'reinstate' : 'abstain';
    my ($src, $splat, $lab, $labtext) =
	_parse_label($source, $really, $data);
    my $gerund = 0;
    if ($splat) {
	# not ABSTAIN/REINSTATE label, try gerund
	($src, $splat, $lab, $labtext) =
	    _parse_gerund_list($source, $really, $data, $stmtid, $stmtno);
	return ($src, $splat, $lab, $labtext, 0) if $splat;
	$lab = [$gopcode, @$lab];
	$gerund = 1;
    } else {
	$lab = [$opcode, $lab];
    }
    $labtext = $prncode . ' ' . $labtext;
    my $end_block = 0;
    my $o = $reinstate ? 'ABSTAINING\s*FROM' : 'REINSTATING';
    if ($data->{'language'} eq 'CLC' &&
	$src =~ s/^WHILE\s*$o\s*(?:IT|THEM)\s*//i)
    {
	$labtext .= ' WHILE';
	if ($reinstate) {
	    $labtext .= ' ABSTAINING FROM';
	} else {
	    $labtext .= ' REINSTATING';
	}
	$lab->[0] = $lab->[0] == $gopcode ? S_GFORK : S_FORK;
	$labtext .= @$lab > 2 ? ' THEM' : ' IT';
	$end_block = 1;
	$data->{'flags'}{'quantum'} = 1;
	$record = 'fork';
    }
    if ($really) {
	# record abstain/reinstate for optimiser and back end
	if (! $gerund) {
	    my $ab = $data->{$record};
	    my $block_id = @{$data->{'files'}[0]};
	    push @$ab, [$lab->[1], 0, $block_id, $stmtid, $stmtno];
	} else {
	    my $ab = $data->{'g' . $record};
	    my $block_id = @{$data->{'files'}[0]};
	    push @$ab, [$lab, 0, $block_id, $stmtid, $stmtno];
	}
    }
    return ($src, 0, $lab, $labtext, $end_block);
}

sub _parse_gerund {
    my ($source, $really, $data, $stmtid, $stmtno) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [T_GERUND], 'GERUND')
	    if $source =~ s/^GERUND\s*//i;
	return ($source, 218, [], '');
    }

    return ($source, 0, [S_ABSTAIN, S_GABSTAIN, S_FORK, S_GFORK], 'ABSTAINING')
	if $source =~ s/^ABSTAINING\s*(?:FROM\s*)?//i;
    return ($source, 0, [S_ASSIGN], 'CALCULATING')
	if $source =~ s/^CALCULATING\s*//i;
    return ($source, 0, [S_COME], 'COMING FROM')
	if $source =~ s/^COMING\s*(?:FROM\s*)//i;
    return ($source, 0, [S_CONVERT, S_CFORK], 'CONVERTING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^CONVERTING\s*//i;
#    return ($source, 0, [S_CREATE, S_CRFORK], 'CREATING')
#	if $data->{'language'} eq 'CLC' && $source =~ s/^CREATING\s*//i;
    return ($source, 0, [S_ENROL], 'ENROLLING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^ENROLLING\s*//i;
    return ($source, 0, [S_ENSLAVE], 'ENSLAVING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^ENSLAVING\s*//i;
    return ($source, 0, [S_FINISH], 'FINISHING LECTURE')
	if $data->{'language'} eq 'CLC' && $source =~ s/^FINISHING\s*(?:LECTURE\s*)//i;
    return ($source, 0, [S_FORGET], 'FORGETTING')
	if $source =~ s/^FORGETTING\s*//i;
    return ($source, 0, [S_FREE], 'FREEING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^(?:FREEING|ESCAPING)\s*//i;
    return ($source, 0, [S_STOP], 'GIVING UP')
	if $source =~ s/^GIVING\s*(?:UP\s*)?//i;
    return ($source, 0, [S_GRADUATES], 'GRADUATING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^GRADUATING\s*//i;
    return ($source, 0, [S_IGNORE, S_VFORK], 'IGNORING')
	if $source =~ s/^IGNORING\s*//i;
    return ($source, 0, [S_LEARN], 'LEARNING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^LEARNING\s*//i;
    return ($source, 0, [S_NEXT], 'NEXTING')
	if $source =~ s/^NEXTING\s*//i;
    return ($source, 0, [S_READ], 'READING OUT')
	if $source =~ s/^READING\s*(?:OUT\s*)?//i;
    return ($source, 0, [S_REINSTATE, S_GREINSTATE, S_FORK, S_GFORK],
			'REINSTATING')
	if $source =~ s/^REINSTATING\s*//i;
    return ($source, 0, [S_REMEMBER, S_VFORK], 'REMEMBERING')
	if $source =~ s/^REMEMBERING\s*//i;
    return ($source, 0, [S_RESUME], 'RESUMING')
	if $source =~ s/^RESUMING\s*//i;
    return ($source, 0, [S_RETRIEVE], 'RETRIEVING')
	if $source =~ s/^RETRIEVING\s*//i;
    return ($source, 0, [S_STASH], 'STASHING')
	if $source =~ s/^STASHING\s*//i;
    return ($source, 0, [S_STUDY], 'STUDYING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^STUDYING\s*//i;
    return ($source, 0, [S_SWAP, S_SFORK], 'SWAPPING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^SWAPPING\s*//i;
    return ($source, 0, [S_WHILE_CB, S_WHILE_BC], 'LOOPING')
	if $data->{'language'} eq 'CLC' && $source =~ s/^LOOPING\s*//i;
    return ($source, 0, [S_WHILE_E], 'HANDLING EVENTS')
	if $data->{'language'} eq 'CLC' &&
	   $source =~ s/^HANDLING\s*(?:EVENTS\s*)?//i;
    return ($source, 0, [S_WRITE], 'WRITING IN')
	if $source =~ s/^WRITING\s*(?:IN\s*)?//i;

    return ($source, 84, [], '') if $data->{'language'} eq 'C';

    my ($src, $splat, $stm, $stmtext) =
	_parse_statement($source, -1, $data, $stmtid, $stmtno);
    return ($src, 0, $stm->[0] == S_STOP ? [] : [$stm->[0]], $stmtext)
	if ! $splat;

    return ($source, 84, [], '');
}

sub _parse_gerund_list {
    my ($source, $really, $data, $stmtid, $stmtno) = @_;

    if (! $really) {
	# fake parse
	return ($source, 0, [T_GERUND_LIST], 'GERUND LIST')
	    if $source =~ s/^GERUND\s*LIST\s*//i;
	return ($source, 218, [], '');
    }

    my ($src, $splat, $ger, $gertext) =
	_parse_gerund($source, $really, $data, $stmtid, $stmtno);
    return ($src, $splat, $ger, $gertext) if $splat;
    my @gers = @$ger;
    my $text = $gertext;
    while ($src =~ s/^\+\s*//) {
	($src, $splat, $ger, $gertext) =
	    _parse_gerund($src, $really, $data, $stmtid, $stmtno);
	return ($src, $splat, $ger, $gertext, 0) if $splat;
	push @gers, @$ger;
	$text .= ' + ' . $gertext;
    }

    return ($src, 0, [@gers], $text);
}

sub _parse_enrol {
    my ($source, $really, $data) = @_;
    return ($source, 201, [], '', 0) if $source =~ /^L/i;
    my ($src, $splat, $reg, $regtext) =
	_parse_register($source, $really, $data, 0, 0);
    return ($src, $splat, $reg, $regtext) if $splat;
    return ($src, 202, [], '', 0) if $src !~ s/^TO\s*LEARN\s*//i;
    my ($src1, $splat1, $num, $numtext) =
	_parse_expression_list($src, $really, $data);
    return ($src1, $splat1, $num, $numtext, 0) if $splat1;
    return ($src1, 0, [S_ENROL, $reg, @$num],
	    "ENROL $regtext TO LEARN $numtext", 0);
}

sub _parse_enslave {
    my ($source, $really, $data) = @_;
    my ($src, $splat, $reg, $regtext) =
	_parse_register($source, $really, $data, 0, 0);
    return ($src, $splat, $reg, $regtext, 0) if $splat;
    return ($src, 112, [], '', 0) if $src !~ s/^TO\s*//i;
    my ($src1, $splat1, $reg1, $regtext1) =
	_parse_register($src, $really, $data, 0, 0);
    return ($src1, $splat1, $reg1, $regtext1, 0) if $splat1;
    return ($src1, 0, [S_ENSLAVE, $reg, $reg1],
	    "ENSLAVE $regtext TO $regtext1", 0);
}

sub _parse_free {
    my ($source, $really, $data) = @_;
    my ($src, $splat, $reg, $regtext) =
	_parse_register($source, $really, $data, 0, 0);
    return ($src, $splat, $reg, $regtext, 0) if $splat;
    return ($src, 116, [], '', 0) if $src !~ s/^FROM\s*//i;
    my ($src1, $splat1, $reg1, $regtext1) =
	_parse_register($src, $really, $data, 0, 0);
    return ($src1, $splat1, $reg1, $regtext1, 0) if $splat1;
    return ($src1, 0, [S_FREE, $reg, $reg1],
	    "FREE $regtext FROM $regtext1", 0);
}

sub _parse_read {
    my ($source, $really, $data) = @_;
    my ($src, $splat, $reg, $regtext) =
	_parse_register_list($source, $really, $data, 1);
    return ($src, $splat, $reg, $regtext, 0) if $splat;
    return ($src, 0, [S_READ, @$reg], "READ OUT $regtext", 0);
}

sub _parse_write {
    my ($source, $really, $data) = @_;
    my ($src, $splat, $reg, $regtext) =
	_parse_register_list($source, $really, $data, 1);
    return ($src, $splat, $reg, $regtext, 0) if $splat;
    return ($src, 0, [S_WRITE, @$reg], "WRITE IN $regtext", 0);
}

sub _parse_remember {
    my ($source, $really, $data, $remember) = @_;
    my $opcode = $remember ? S_REMEMBER : S_IGNORE;
    my $prncode = $remember ? 'REMEMBER' : 'IGNORE';
    my ($src, $splat, $reg, $regtext) =
	_parse_register_list($source, $really, $data, 0);
    return ($src, $splat, $reg, $regtext, 0) if $splat;
    my $text = $prncode . ' ' . $regtext;
    $data->{'flags'}{'ignore'} = 1 if $really;
    my $end_block = 0;
    my $o = $remember ? 'IGNORING' : 'REMEMBERING';
    if ($src =~ s/^WHILE\s*$o\s*(?:IT|THEM)\s*//i) {
	$text .= " WHILE $o ";
	$text .= @$reg > 1 ? 'THEM' : 'IT';
	$end_block = 1;
	$opcode = S_VFORK;
	$data->{'flags'}{'quantum'} = 1;
    }
    return ($src, 0, [$opcode, @$reg], $text, $end_block);
}

sub _parse_stash {
    my ($source, $really, $data, $stash) = @_;
    my $opcode = $stash ? S_STASH : S_RETRIEVE;
    my $prncode = $stash ? 'STASH' : 'RETRIEVE';
    my ($src, $splat, $reg, $regtext) =
	_parse_register_list($source, $really, $data, 0);
    return ($src, $splat, $reg, $regtext, 0) if $splat;
    my $text = $prncode . ' ' . $regtext;
    return ($src, 0, [$opcode, @$reg], $text, 0);
}

sub _parse_resume {
    my ($source, $really, $data, $resume) = @_;
    my $opcode = $resume ? S_RESUME : S_FORGET;
    my $prncode = $resume ? 'RESUME' : 'FORGET';
    my ($src, $splat, $exp, $exptext) =
	_parse_expression($source, $really, $data, 0);
    return ($src, $splat, $exp, $exptext, 0) if $splat;
    return ($src, 0, [$opcode, $exp], $prncode . ' ' . $exptext, $resume);
}

sub _parse_learns {
    my ($source, $really, $data, $reg, $regtext) = @_;
    my ($src, $splat, $exp, $exptext) =
	_parse_expression($source, $really, $data, 0);
    return ($src, $splat, $exp, $exptext, 0) if $splat;
    return ($src, 0, [S_LEARN, $reg, $exp],
	    "$regtext LEARNS $exptext", 1);
}

sub _parse_study {
    my ($source, $really, $data) = @_;
    my ($src, $splat, $exp, $exptext) =
	_parse_expression($source, $really, $data, 0);
    return ($src, $splat, $exp, $exptext, 0) if $splat;
    return ($src, 252, [], '', 0) if $src !~ s/^AT\s*//i;
    my ($src1, $splat1, $lab, $labtext) =
	_parse_label($src, $really, $data);
    return ($src1, $splat1, $lab, $labtext, 0) if $splat1;
    return ($src1, 252, [], '', 0) if $src1 !~ s/^IN\s*CLASS\s*//i;
    my ($src2, $splat2, $reg, $regtext) =
	_parse_register($src1, $really, $data, 0, 0);
    return ($src2, $splat2, $reg, $regtext, 0) if $splat2;
    return ($src2, 252, [], '', 0) if ! _can_be_class($reg);
    return ($src2, 999, [], '', 0)
	if $lab->[0] == E_CONSTANT && atoi($lab->[1]) < 1000;
    return ($src2, 0, [S_STUDY, $exp, $lab, $reg],
	    "STUDY $exptext AT $labtext IN CLASS $regtext", 0);
}

sub _parse_swap {
    my ($source, $really, $data, $stmtid, $stmtno) = @_;
    my ($src, $splat, $stm, $stmtext, $ispost) =
	_parse_m_statement($source, $really ? -1 : 0, $data, $stmtid, $stmtno);
    return ($src, $splat, $stm, $stmtext, 0) if $splat;
    return ($src, 218, [], '', 0) if $src !~ s/^AND\s*//i;
    my ($src1, $splat1, $stm1, $stmtext1, $ispost1) =
	_parse_m_statement($src, $really ? -1 : 0, $data, $stmtid, $stmtno);
    return ($src1, $splat1, $stm1, $stmtext1, 0) if $splat1;
    return ($src1, 218, [], '', 0) if @$stm != @$stm1;
    my $i;
    for ($i = 1; $i < @$stm; $i++) {
	$stm->[$i] = $stm->[$i][0];
	$stm1->[$i] = $stm1->[$i][0];
    }
    my @template = sort (@$stm[1..$#$stm]);
    my @template1 = sort (@$stm1[1..$#$stm1]);
    while (@template) {
	my $e = shift @template;
	my $e1 = shift @template1;
	return ($src1, 218, [], '', 0) if $e ne $e1;
    }
    $data->{'flags'}{'postprocess'} = 1 if $ispost || $ispost1;
    $data->{'flags'}{'loop_swap'} = 1 if ! $ispost && ! $ispost1;
    $stm = [S_SWAP, $stm, $stm1];
    $stmtext = "SWAP $stmtext AND $stmtext1";
    return ($src1, 0, $stm, $stmtext, 0)
	if $src1 !~ s/^WHILE\s*LEAVING\s*THEM\s*UNCHANGED\s*//i;
    $stmtext .= ' WHILE LEAVING THEM UNCHANGED';
    $data->{'flags'}{'quantum'} = 1;
    $stm->[0] = S_SFORK;
    return ($src1, 0, $stm, $stmtext, 1);
}

sub _parse_convert {
    my ($source, $really, $data, $stmtid, $stmtno) = @_;
    my ($src, $splat, $stm, $stmtext, $ispost) =
	_parse_m_statement($source, $really ? -1 : 0, $data, $stmtid, $stmtno);
    return ($src, $splat, $stm, $stmtext, 0) if $splat;
    return ($src, 218, [], '', 0) if $src !~ s/^TO\s*//i;
    my ($src1, $splat1, $stm1, $stmtext1, $ispost1) =
	_parse_m_statement($src, $really ? -1 : 0, $data, $stmtid, $stmtno);
    return ($src1, $splat1, $stm1, $stmtext1, 0) if $splat1;
    return ($src1, 218, [], '', 0) if @$stm != @$stm1;
    my $i;
    for ($i = 1; $i < @$stm; $i++) {
	$stm->[$i] = $stm->[$i][0];
	$stm1->[$i] = $stm1->[$i][0];
    }
    my @template = sort (@$stm[1..$#$stm]);
    my @template1 = sort (@$stm1[1..$#$stm1]);
    while (@template) {
	my $e = shift @template;
	my $e1 = shift @template1;
	return ($src1, 218, [], '', 0) if $e ne $e1;
    }
    $data->{'flags'}{'postprocess'} = 1 if $ispost || $ispost1;
    $data->{'flags'}{'loop_swap'} = 1 if ! $ispost && ! $ispost1;
    $stm = [S_CONVERT, $stm, $stm1];
    $stmtext = "CONVERT $stmtext TO $stmtext1";
    return ($src1, 0, $stm, $stmtext, 0)
	if $src1 !~ s/^WHILE\s*LEAVING\s*IT\s*UNCHANGED\s*//i;
    $stmtext .= ' WHILE LEAVING IT UNCHANGED';
    $data->{'flags'}{'quantum'} = 1;
    $stm->[0] = S_CFORK;
    return ($src1, 0, $stm, $stmtext, 1);
}

#sub _parse_create {
#    my ($source, $really, $data, $stmtid, $stmtno) = @_;
#    my ($src, $splat, $stm, $stmtext, $ispost) =
#	_parse_m_statement($source, $really ? -1 : 0, $data, $stmtid, $stmtno);
#    return ($src, $splat, $stm, $stmtext, 0) if $splat;
#    return ($src, 218, [], '', 0) if $src !~ s/^TO\s*//i;
#    my ($src1, $splat1, $stm1, $stmtext1, $eblock1) =
#	_parse_statement_list($src, $really ? -1 : 0, $data, $stmtid, $stmtno);
#    return ($src1, $splat1, $stm1, $stmtext1, 0) if $splat1;
#    return ($src1, 218, [], '', 0) if @$stm != @$stm1;
#    my $i;
#    for ($i = 1; $i < @$stm; $i++) {
#	$stm->[$i] = $stm->[$i][0];
#	$stm1->[$i] = $stm1->[$i][0];
#    }
#    my @template = sort (@$stm[1..$#$stm]);
#    my @template1 = sort (@$stm1[1..$#$stm1]);
#    while (@template) {
#	my $e = shift @template;
#	my $e1 = shift @template1;
#	return ($src1, 218, [], '', 0) if $e ne $e1;
#    }
#    $data->{'flags'}{'postprocess'} = 1;
#    $stm = [S_CREATE, $stm, @$stm1];
#    $stmtext = "CREATE $stmtext AS $stmtext1";
#    return ($src1, 0, $stm, $stmtext, 0)
#	if $src1 !~ s/^WHILE\s*NOT\s*CREATING\s*IT\s*//i;
#    $stmtext .= ' WHILE NOT CREATING IT';
#    $data->{'flags'}{'quantum'} = 1;
#    $stm->[0] = S_CRFORK;
#    return ($src1, 0, $stm, $stmtext, 1);
#}

sub _parse_m_statement {
    my ($source, $really, $data, $stmtid, $stmtno) = @_;
    my ($src, $splat, $stm, $stmtext, $eblock) =
	_parse_statement($source, $really, $data, $stmtid, $stmtno);
    return ($src, $splat, $stm, $stmtext, 1) if ! $splat;
    return ($source, 0, [S_WHILE_BC, [-1], [-1]],
		'BODY WHILE CONDITION', 0)
	if $source =~ s/^BODY\s*WHILE\s*CONDITION\s*//i;
    return ($source, 0, [S_WHILE_CB, [-1], [-1]],
		'CONDITION WHILE BODY', 0)
	if $source =~ s/^CONDITION\s*WHILE\s*BODY\s*//i;
    return ($src, $splat, $stm, $stmtext, 0);
}

1;
