package Language::INTERCAL::ByteCode;

# Definitions of bytecode symbols etc

# This file is part of CLC-INTERCAL

# Copyright (c) 2006 Claudio Calvelli, all rights reserved.

# CLC-INTERCAL is copyrighted software. However, permission to use, modify,
# and distribute it is granted provided that the conditions set out in the
# licence agreement are met. See files README and COPYING in the distribution.

@@DATA ByteCode@@

use strict;
use vars qw($PERVERSION);
$PERVERSION = "CLC-INTERCAL INTERCAL/ByteCode.pm 1.-94.-4";

use Carp;

use Language::INTERCAL::Exporter '1.-94.-4';
use Language::INTERCAL::Splats '1.-94.-4', qw(:SP);
use Language::INTERCAL::Numbers '1.-94.-4';
use Language::INTERCAL::DoubleOhSeven '1.-94.-4';
use Language::INTERCAL::SharkFin '1.-94.-4';
use Language::INTERCAL::Arrays '1.-94.-4';
use Language::INTERCAL::Whirlpool '1.-94.-4';
use Language::INTERCAL::GenericIO '1.-94.-4',
	qw($stdwrite $stdread $stdsplat $devnull);

use vars qw(@EXPORT @EXPORT_OK %EXPORT_TAGS);

use constant BYTE_SIZE     => 8;      # number of bits per byte (must be == 8)
use constant NUM_OPCODES   => 0x80;   # number of virtual opcodes
use constant OPCODE_RANGE  => 1 << BYTE_SIZE;
use constant BC_MASK       => OPCODE_RANGE - 1;
use constant BIGNUM_SHIFT  => BYTE_SIZE - 1;
use constant BIGNUM_RANGE  => 1 << BIGNUM_SHIFT;
use constant BIGNUM_MASK   => (BIGNUM_RANGE - 1) << 1;
use constant BYTE_SHIFT    => OPCODE_RANGE - NUM_OPCODES;

@EXPORT_OK = qw(
    bytecode bytedecode bc_list BC BCget bc_bytype bc_match BC_MASK
    BC_constants is_constant is_multibyte bc_xtype bc_skip bc_forall
    @@FILL OPCODES BC_ NAME '' 76 ' '@@
    reg_list reg_name reg_create reg_codetype reg_decode
    reg_code
);

%EXPORT_TAGS = (
    BC => [qw(
	BC BCget BC_MASK bytecode bytedecode
	@@FILL OPCODES BC_ NAME '' 76 ' '@@
    )],
);

my %bytecodes = (
    @@ALL OPCODES NAME@@ => ['@@'DESCR'@@', '@@TYPE@@', '@@NUMBER@@', '@@ARGS@@', @@CONST@@, @@ASSIGNABLE@@],
);

my %bytedecode = (
    @@ALL OPCODES NUMBER@@ => '@@'NAME'@@',
);

my @bc_list = qw(
    @@FILL OPCODES '' NAME '' 76 ' '@@
);

sub BC_@@ALL OPCODES NAME@@ () { @@NUMBER@@; }

my @reg_list = qw(
    @@FILL SPECIAL '' NAME '' 76 ' '@@
);

my %reg_list = (
    @@ALL DOUBLE_OH_SEVEN NAME@@ => ['@@'CODE'@@', @@DEFAULT@@, BC_DOS, '%', @@NUMBER@@],
    @@ALL SHARK_FIN NAME@@ => ['@@'CODE'@@', @@DEFAULT@@, BC_SHF, '^', @@NUMBER@@],
    @@ALL WHIRLPOOL NAME@@ => ['@@'CODE'@@', @@DEFAULT@@, BC_WHP, '@', @@NUMBER@@],
);

my %reg_names = (
    '%@@ALL DOUBLE_OH_SEVEN NUMBER@@' => '@@NAME@@',
    '^@@ALL SHARK_FIN NUMBER@@' => '@@NAME@@',
    '@@@ALL WHIRLPOOL NUMBER@@' => '@@NAME@@',
);

my %mulmap = map { ( $_ => 1 ) } BC_MUL, BC_STR;

sub bc_list () {
    @bc_list;
}

sub BC {
    @_ == 1 || croak "Usage: BC(value)";
    my ($val) = @_;
    croak "Invalid undefined value" unless defined $val;
    my $orig = $val;
    $val < BYTE_SHIFT
	and return ($val + NUM_OPCODES);
    $val < OPCODE_RANGE
	and return (BC_HSN, $val);
    my $div = int($val / OPCODE_RANGE);
    $div < OPCODE_RANGE
	and return (BC_OSN, $div, $val % OPCODE_RANGE);
    croak "Invalid value $orig: does not fit in one spot";
}

sub bytecode ($) {
    my ($name) = @_;
    $name =~ /^\d+$/ && $name < BYTE_SHIFT ? ($name + NUM_OPCODES)
					   : $bytecodes{$name}[2];
}

sub bytedecode ($) {
    my ($b) = @_;
    if ($b >= NUM_OPCODES) {
	my $n = $b - NUM_OPCODES;
	return () if $n >= BYTE_SHIFT;
	return "#$n" unless wantarray;
	return ("#$n", 'Constant', '#', $b, '', 1, 1);
    } else {
	return () unless exists $bytedecode{$b};
	return $bytedecode{$b} unless wantarray;
	return ($bytedecode{$b}, @{$bytecodes{$bytedecode{$b}}});
    }
}

sub BCget {
    @_ == 3 or croak "Usage: BCget(CODE, \\POSITION, END)";
    my ($code, $cp, $ep) = @_;
    $$cp >= $ep and faint(SP_INVALID, "end of code", "BCget");
    my $byte = ord(substr($code, $$cp, 1));
    $$cp++;
    if ($byte >= NUM_OPCODES) {
	return $byte - NUM_OPCODES;
    }
    if ($byte == BC_HSN) {
confess "HSN $$cp >= $ep" if $$cp >= $ep;
	$$cp >= $ep and faint(SP_INVALID, "end of code", "BCget/HSN");
	return ord(substr($code, $$cp++, 1));
    }
    if ($byte == BC_OSN) {
confess "OSN $$cp + 1 >= $ep" if $$cp + 1 >= $ep;
	$$cp + 1 >= $ep and faint(SP_INVALID, "end of code", "BCget/OSN");
	my $nx = unpack('n', substr($code, $$cp, 2));
	$$cp += 2;
	return $nx;
    }
confess sprintf("Invalid bytecode 0x%02X", $byte); # XXX
    faint(SP_INVALID, sprintf("0x%02x", $byte), "BCget")
}

sub BC_constants () {
    (NUM_OPCODES..BC_MASK);
}

sub is_constant ($) {
    my ($byte) = @_;
    return 1 if $byte >= NUM_OPCODES ||
		$byte == BC_HSN ||
		$byte == BC_OSN;
    return 0;
}

sub is_multibyte ($) {
    my ($byte) = @_;
    return 1 if $byte == BC_HSN;
    return 2 if $byte == BC_OSN;
    0;
}

sub bc_bytype {
    @_ or croak "Usage: bc_bytype(TYPES)";
    my %types = ();
    for my $type (@_) {
	if ($type eq 'R' || $type eq 'S') {
	    $types{$type} = 0;
	    next;
	}
	if ($type =~ /^[CEP<>L\[\]]$/) {
	    $types{E} = $types{R} = $types{'#'} = 0;
	    next;
	}
	if ($type eq 'V') {
	    $types{R} = $types{V} = 0;
	    next;
	}
	if ($type eq 'O') {
	    $types{S} = 0;
	    next;
	}
    }
    my %values = exists $types{V} ? %mulmap : ();
    map {
	my ($desc, $type, $value, $args, $function) = @{$bytecodes{$_}};
	if (exists $types{$type} || exists $values{$value}) {
	    $value;
	} else {
	    ();
	}
    } keys %bytecodes;
}

sub bc_match {
    @_ >= 2 && @_ <= 4
	or croak "Usage: bc_match(PATTERN, CODE [,START [,END]])";
    my ($pattern, $code, $start, $end) = @_;
    $start ||= 0;
    $end = length($code) if not defined $end;
    _match($pattern, $code, $start, $end, undef);
}

sub bc_skip {
    @_ >= 1 && @_ <= 3
	or croak "Usage: bc_skip(CODE [,START [,END]])";
    my ($code, $start, $end) = @_;
    $start ||= 0;
    $end = length($code) if not defined $end;
    return undef if $start >= $end || $start < 0;
    my $byte = ord(substr($code, $start, 1));
    return 1 if $byte >= NUM_OPCODES;
    return undef if ! exists $bytedecode{$byte};
    my $name = $bytedecode{$byte};
    my $pattern = $bytecodes{$name}[1];
    _match($pattern, $code, $start, $end, undef);
}

sub bc_forall {
    @_ == 5
	or croak "Usage: bc_forall(PATTERN, CODE, START, END, CLOSURE)";
    my ($pattern, $code, $start, $end, $closure) = @_;
    $start ||= 0;
    $end = length($code) if not defined $end;
    return undef if $start >= $end || $start < 0;
    my $np = '';
    while ($pattern =~ s/^(.*?)C\(/(/) {
	my $a = $1;
	$a =~ s/(.)/$1\x01/g;
	$np .= $a . 'C';
	$np .= '(' . _args('forall', \$pattern) . ')';
	$np .= "\01";
    }
    $pattern =~ s/(.)/$1\x01/g;
    $pattern = "\x01" if $pattern eq '';
    $np .= $pattern;
    _match($np, $code, $start, $end, $closure);
}

sub bc_xtype {
    @_ == 1 or croak "Usage: bc_xtype(\\PATTERN)";
    my ($pattern) = @_;
    _args('xtype', $pattern);
}

my %typemap = (
    'S' => { 'S' => 0 },
    'O' => { 'S' => 0 },
    'E' => { 'E' => 0, 'R' => 0, '#' => 0 },
    'A' => { 'E' => 0, 'R' => 0, '#' => 0 },
    'R' => { 'R' => 0 },
    'V' => { 'R' => 0, 'V' => 0 },
    '#' => { '#' => 0 },
    'C' => { '#' => 0 },
    'Z' => { 'S' => 0, 'E' => 0, 'R' => 0, '#' => 0 },
    '*' => { 'S' => 0, 'E' => 0, 'R' => 0, '#' => 0 },
);

sub _args {
    my ($name, $pattern) = @_;
    faint(SP_BCMATCH, $name, 'Missing (') if $$pattern !~ s/^\(//;
    my $count = 1;
    my $result = '';
    while ($count > 0) {
	$$pattern =~ s/^([^\(\)]*)([\(\)])//
	    or faint(SP_BCMATCH, $name, 'Missing )');
	$count++ if $2 eq '(';
	$count-- if $2 eq ')';
	$result .= $1 . ($count ? $2 : '');
    }
    $result;
}

sub _match {
    my ($pattern, $code, $sc, $ep, $closure) = @_;
    my $osc = $sc;
    MATCH: while ($pattern ne '') {
	my $e = substr($pattern, 0, 1, '');
	if ($e eq "\x00") {
	    $closure->(undef, '>') if $closure;
	    next MATCH;
	}
	if ($e eq "\x01") {
	    $closure->($sc, undef) if $closure;
	    next MATCH;
	}
if ($sc >= $ep) { $pattern =~ s/\x00/\\x00/g; confess "_match/0 $sc >= $ep ($pattern)" }
	faint(SP_INVALID, 'end of code', '_match') if $sc >= $ep;
	my $v = ord(substr($code, $sc, 1));
	if (exists $typemap{$e}) {
	    # check next opcode is correct type
	    my ($op, $desc, $type, $value, $args, $function) = bytedecode($v);
confess "_match/1 $v $e $pattern" unless defined $type;
	    faint(SP_INVALID, $v, "_match: $e")
		unless defined $type;
confess "_match/2 $v($op) $e $type $pattern (sc=$sc)" unless exists $typemap{$e}{$type} || (exists $mulmap{$v} && exists $typemap{$e}{V});
	    faint(SP_INVALID, $type, "_match: $e")
		unless exists $typemap{$e}{$type} ||
		       (exists $mulmap{$v} && exists $typemap{$e}{V});
	    if ($type eq '#' && $e ne '*') {
		my $num = BCget($code, \$sc, $ep);
		$closure->($v, "#$num") if $closure;
		if ($e eq 'C') {
		    $args = _args('count', \$pattern) x $num;
		    $args .= "\x00";
		    $closure->(undef, '<') if $closure;
		} else {
		    $args = '';
		}
	    } else {
		$sc++;
		$args = '' if $e eq 'O' || $e eq '*';
		$closure->($v, $op) if $closure;
	    }
	    $pattern = $args . $pattern;
	    next MATCH;
	} elsif ($e eq 'N') {
	    # any nonzero number
	    return undef if $v == 0;
	    $closure->($v, "N$v") if $closure;
	    $sc++;
	} elsif ($e eq '<') {
	    # left grammar element
	    my $count = BCget($code, \$sc, $ep);
	    my $num = BCget($code, \$sc, $ep);
	    if ($num == 0) {
		$closure->(undef, '?<') if $closure;
	    } elsif ($num == 1 || $num == 2) {
		$closure->(undef, ',<') if $closure;
	    } else {
		$closure->(undef, ',!<') if $closure;
	    }
	    if ($count && $closure) {
		$closure->(undef, $count == 65535 ? '*' : $count);
	    }
	    $pattern = "E\x00" . $pattern;
	    next MATCH;
	} elsif ($e eq '>') {
	    # right grammar element
	    my $num = BCget($code, \$sc, $ep);
	    if ($num == 0 || $num == 6) {
		my $count = BCget($code, \$sc, $ep);
		if ($count && $closure) {
		    $closure->(undef, $count);
		}
		$closure->($v, $num ? '!<' : '?<') if $closure;
		$pattern = "E\x00" . $pattern;
		next MATCH;
	    }
	    if ($num == 1 || $num == 2) {
		$closure->($v, ',<') if $closure;
		my $count = BCget($code, \$sc, $ep);
		if ($count && $closure) {
		    $closure->(undef, $count);
		}
		$pattern = "E\x00" . $pattern;
		next MATCH;
	    }
	    if ($num == 3 || $num == 7) {
		$closure->($v, ',!<') if $closure;
		my $count = BCget($code, \$sc, $ep);
		if ($count && $closure) {
		    $closure->(undef, $count);
		}
		$pattern = "E\x00" . $pattern;
		next MATCH;
	    }
	    if ($num == 4) {
		$num = BCget($code, \$sc, $ep);
		my $se = $sc + $num;
		$se <= $ep
		    or faint(SP_INVALID, '???', '_match: >');
		if ($closure) {
		    $closure->(undef, '=<');
		    while ($sc < $se) {
			$sc += _match('*', $code, $sc, $se, $closure);
		    }
		    $closure->(undef, '>');
		} else {
		    $sc = $se;
		}
		next MATCH;
	    }
	    if ($num == 15) {
		$closure->($v, '*') if $closure;
		next MATCH;
	    }
	    faint(SP_INVALID, $num, "_match: >");
	} elsif ($e eq '[') {
	    # XXX left optimise element
	    faint(SP_TODO, 'match on [');
	} elsif ($e eq ']') {
	    # XXX right optimise element
	    faint(SP_TODO, 'match on ]');
	} else {
	    faint(SP_BCMATCH, 'type', $e);
	}
    }
    $sc - $osc;
}

sub reg_list () {
    @reg_list;
}

sub reg_create {
    @_ == 2 || @_ == 3
	or croak "Usage: reg_create(REGISTER, OBJECT [, VALUE])";
    my ($rn, $object, @value) = @_;
    $rn = $reg_names{$rn} if exists $reg_names{$rn};
    if (exists $reg_list{$rn}) {
	@value = $reg_list{$rn}[1] if ! @value;
	my $rt = $reg_list{$rn}[3];
	my $dt = $reg_list{$rn}[0];
	return Language::INTERCAL::DoubleOhSeven->new($dt, $object, @value)
	    if $rt eq '%';
	return Language::INTERCAL::SharkFin->new($dt, $object, @value)
	    if $rt eq '^';
	return Language::INTERCAL::Whirlpool->new(@value)
	    if $rt eq '@';
    }
    $rn =~ /^\./
	and return Language::INTERCAL::Numbers::Spot->new(@value || 0);
    $rn =~ /^:/
	and return Language::INTERCAL::Numbers::Twospot->new(@value || 0);
    $rn =~ /^,/
	and return Language::INTERCAL::Arrays::Tail->new(@value || []);
    $rn =~ /^;/
	and return Language::INTERCAL::Arrays::Hybrid->new(@value || []);
    $rn =~ /^\@/
	and return Language::INTERCAL::Whirlpool->new();
    faint(SP_SPECIAL, $rn);
}

sub reg_codetype {
    @_ == 1 or croak "Usage: reg_codetype(REGISTER)";
    my ($rn) = @_;
    exists $reg_list{$rn} and return $reg_list{$rn}[0];
    if (exists $reg_names{$rn}) {
	$rn = $reg_names{$rn};
	return $reg_list{$rn}[0];
    }
    $rn =~ /^\./ and return 'spot';
    $rn =~ /^:/ and return 'twospot';
    $rn =~ /^,/ and return 'tail';
    $rn =~ /^;/ and return 'hybrid';
    $rn =~ /^\@/ and return 'whirlpool';
    faint(SP_SPECIAL, $rn);
}

sub reg_name {
    @_ == 1 or croak "Usage: reg_name(REGISTER)";
    my ($rn) = @_;
    exists $reg_list{$rn}
	and return $reg_list{$rn}[3] . $reg_list{$rn}[4];
    if (exists $reg_names{$rn}) {
	$rn = $reg_names{$rn};
	return $reg_list{$rn}[3] . $reg_list{$rn}[4];
    }
    $rn =~ /^([%^\@])(.*)$/ && exists $reg_list{$2} && $reg_list{$2}[3] eq $1
	and return $reg_list{$2}[3] . $reg_list{$2}[4];
    $rn =~ s/^([\.:,;\@^%])0*(\d+)$/$1$2/ and return $rn;
    undef;
}

sub reg_code {
    @_ == 1 or croak "Usage: reg_code(REGISTER)";
    my ($rn) = @_;
    exists $reg_list{$rn}
	and return ($reg_list{$rn}[2], BC($reg_list{$rn}[4]));
    if (exists $reg_names{$rn}) {
	$rn = $reg_names{$rn};
	return ($reg_list{$rn}[2], BC($reg_list{$rn}[4]));
    }
    $rn =~ /^([%^\@])(.*)$/ && exists $reg_list{$2} && $reg_list{$2}[3] eq $1
	and return ($reg_list{$2}[2], BC($reg_list{$2}[4]));
    $rn =~ /^\.(\d+)$/ and return (BC_SPO, BC($1));
    $rn =~ /^:(\d+)$/ and return (BC_TSP, BC($1));
    $rn =~ /^,(\d+)$/ and return (BC_TAI, BC($1));
    $rn =~ /^;(\d+)$/ and return (BC_HYB, BC($1));
    $rn =~ /^\@(\d+)$/ and return (BC_WHP, BC($1));
    $rn =~ /^\%(\d+)$/ and return (BC_DOS, BC($1));
    $rn =~ /^\^(\d+)$/ and return (BC_SHF, BC($1));
    undef;
}

sub reg_decode {
    @_ == 1 or croak "Usage: reg_name(REGISTER)";
    my ($rn) = @_;
    return $rn if $rn =~ /^[.,:;\@_]/;
    if ($rn =~ /^[%^]\d+$/) {
	return undef unless exists $reg_names{$rn};
	$rn = $reg_names{$rn};
    } elsif ($rn =~ s/^([%^])//) {
	return undef unless exists $reg_list{$rn};
	return undef if $1 ne $reg_list{$rn}[3];
    } else {
	return undef unless exists $reg_list{$rn};
    }
    $reg_list{$rn}[3] . $rn;
}

1;

__END__

=pod

=head1 TITLE

Language::INTERCAL::Bytecode - definitions for the ICBM

=head1 DESCRIPTION

The ICBM (B<I>NTERCAL B<C>ommon B<B>ytecode B<M>angler) is the part of
CLC-INTERCAL which defines the INTERCAL virtual machine, that is the
execution environment for interpreted INTERCAL programs, as well as the
intermediate language used to compile programs.

The compiler produces bytecode from program source; this bytecode can
be interpreted to execute the program immediately; alternatively, a
backend can produce something else from the bytecode, for example C
or Perl source code which can then be compiled to your computer's
native object format.

The compiler itself is just some more bytecode. Thus, to produce the
compiler you need a compiler compiler, and to produce that you need
a compiler compiler compiler; to produce the latter you would need
a compiler compiler compiler compiler, and so on to infinity. To
simplify the programmer's life (eh?), the compiler compiler is able
to compile itself, and is therefore identical to the compiler compiler
compiler (etcetera).

The programmer can start the process because a pre-compiled compiler
compiler, in the form of bytecode, is provided with the CLC-INTERCAL
distribution; this compiler compiler then is able to compile all
other compilers, as well as to rebuild itself if need be.

This document describes the bytecode used by all user programs, compilers,
and the compiler compiler. Other documentation, if and when it will be
written, describes the compilation process in more detail as well as the
rest of the execution environment.

A program, in its executable form, consists of a list of statements. Each
statement can contain alternative execution paths. Execution starts at
the first statement in the list and proceed in sequence unless the bytecode
has some form of jump, next or is subject to a come from. The choice of a
particular execution path amongst the various possibilities will be discussed
later.

=head1 BYTECODE DEFINITIONS

Each byte in the bytecode can be part of a statement, an expression or a
register. In addition, a subset of expressions can be assigned to: these
are called assignable expressions. For example, a constant is an assignable
expression. When assigned to, it changes the value of the constant. This
is necessary to implement overloading and is also a great obfuscation
mechanism.

The rest of this document uses the three-letter abbreviations for bytes in
the bytecode. To see the corresponding numeric code, use:

    perl -lMLanguage::INTERCAL::ByteCode=:BC -e 'print BC_???'

where I<???> is the three letter abbreviation used in the text. To see the
one-line description type:

    perl -lMLanguage::INTERCAL::ByteCode=:BC,bytedecode -e \
    'print ((bytedecode(BC_???))[1])'

If you see the number anywhere and want to have the three-letter abbreviation
type:

    perl -lMLanguage::INTERCAL::ByteCode=:bytedecode -e \
    'print scalar bytedecode(nnn)'

Your shell or command interpreter might have its own idea about the above
commands and the use of quotes, as usual.

=head2 CONSTANTS

Constants can be specified in four ways:

=over 4

=item ###

Byte larger than maximum opcode.

Any byte with value greather than the maximum opcode is interpreted as a
16 bit (spot) constant by subtracting the number of opcodes from the byte.
For example, since there are 128 opcodes, byte 131 is equivalent to #3,
and byte 255 (the maximum value) is #127

@@MULTI CONSTANTS NAME@@
=item @@NAME@@

@@DESCR@@

@@DOC 76@@

@@MULTI@@
=back

The exportable function I<BC> converts a number into bytecode, returning
a list of numbers: use C<pack('C*', BC($n)> to convert a number to a
bytecode string.

The exportable function I<BCget> converts a bytecode string to a number.
The function takes three parameters: the string containing the bytecode,
a scalar reference indicating the starting point (this will be updated
to the first opcode after the constant) and the end of the bytecode
block one wants to examine.

As the number of opcodes can change, it is recommended to use I<BC> and
I<BCget> whenever possible to ensure compatibility with future versions
of CLC-INTERCAL.

=head2 REGISTERS

Registers can be any number of register prefixes, followed by a type and
a constant. There are limitations in the useful combination of prefixes.

The register types are:

=over 4

@@MULTI REGISTERS NAME@@
=item @@NAME@@

@@DESCR@@

@@DOC 76@@

@@MULTI@@
=back

Crawling horror (_) registers are not explicitely specified; rather, the
operations which take a crawling horror accept the register number and
form the complete register name internally.

The prefixes which can applied to registers are:

=over 4

@@MULTI PREFIXES NAME@@
=item @@NAME@@

@@DESCR@@

@@DOC 76@@

@@MULTI@@
=back

=head2 ASSIGNABLE EXPRESSIONS

Assignable expressions are sequences of bytecode which can used as the
target of an assignment. Of course, all registers are assignable;
all constants are also assignable, which makes then really variables.
Instead of describing the assignable expressions separately, we describe
all expressions and mention which ones are assignable. Assigning to
an expression means assigning appropriate values to its subexpressions such
that the expression, if evaluated, would result in the value being assigned.
This is not always possible, so it can generate runtime errors.

=head2 EXPRESSIONS

In addition to registers and constants, the following are valid expressions:

=over 4

@@MULTI EXPRESSIONS NAME@@
=item @@NAME@@

@@DESCR@@

@@DOC 76@@

@@MULTI@@
=back

=head2 STATEMENTS

The following opcodes are valid statements:

=over 4

@@MULTI STATEMENTS NAME@@
=item @@NAME@@

@@DESCR@@

@@DOC 76@@

@@MULTI@@
=back

=head1 SPECIAL REGISTERS

Bytecode operation is controlled by a number of special registers;
registers prefixed by a double-oh-seven (%) are special spot registers;
registers prefixed by a shark fin (^) are special tail registers;
whirlpool (@) registers are normal class registers, but also contain
special values;
these registers cannot be accessed directly by user programs (unless
one adds the corresponding syntax to the I<sick.io> compiler), but
are necessary for writing compilers, and are accessible to programs
written in assembler.

The double-oh-seven and shark fin registers are often referred to by
symbolic name, rather than number. The assembler recognises such names
and maps them to numbers. For this reason, the listing below uses the
names. Since trace outputs may show numbers rather than names, the
register number is also shown in parentheses.

=head2 DOUBLE-OH-SEVEN REGISTERS

Double-oh-seven registers hold one-spot numbers which can be useful for
compilers and interpreters.

=over 4

@@MULTI DOUBLE_OH_SEVEN NAME@@
=item @@TYPE@@@@NAME@@

@@DESCR@@ (@@TYPE@@@@NUMBER@@)

@@DOC 76@@

@@MULTI@@
=back

=head2 SHARK FIN REGISTER

Shark fin registers hold tail arrays which can be useful for the runtime
and the compilers.

=over 4

@@MULTI SHARK_FIN NAME@@
=item @@TYPE@@@@NAME@@

@@DESCR@@ (@@TYPE@@@@NUMBER@@)

@@DOC 76@@

@@MULTI@@
=back

=head2 EXTRA MEANING FOR WHIRLPOOL REGISTERS

Selected whirlpool registers contain some extra data determining a filehandle.
For example, to send some output to the standard splat filehandle and then
restore normal behaviour (output to standard read) one could say:

    DO READ OUT @3 + ,1 + @1

Similarly, if such registers appear in a WRITE IN they change the filehandle
for input operations.

Using of whirlpool registers with other numbers is currently an error: in
future, they will enable virtual I/O using classes.

=over 4

@@MULTI WHIRLPOOL NUMBER@@
=item @@TYPE@@@@NUMBER@@ (@@TYPE@@@@NAME@@)

@@DESCR@@

@@DOC 76@@

@@MULTI@@
=back

=head1 SEE ALSO

A qualified psychiatrist

=head1 AUTHOR

Claudio Calvelli - intercal (at) sdf.lonestar.org

