package Language::INTERCAL::GenericIO;

# Write/read data

# This file is part of CLC-INTERCAL

# Copyright (c) 2006 Claudio Calvelli, all rights reserved.

# CLC-INTERCAL is copyrighted software. However, permission to use, modify,
# and distribute it is granted provided that the conditions set out in the
# licence agreement are met. See files README and COPYING in the distribution.

use strict;
use vars qw($PERVERSION);
$PERVERSION = "CLC-INTERCAL INTERCAL/GenericIO.pm 1.-94.-4";

use Carp;
use Cwd;
use File::Spec;
use IO::Handle;
use IO::File;

use Language::INTERCAL::Exporter '1.-94.-4';
use Language::INTERCAL::Charset '1.-94.-4', qw(toascii fromascii);

use vars qw(@EXPORT @EXPORT_OK @EXPORT_TAGS
	    $stdread $stdwrite $stdsplat $devnull);

@EXPORT = ();
@EXPORT_OK = qw($stdread $stdwrite $stdsplat $devnull);
@EXPORT_TAGS = (files => [qw($stdread $stdwrite $stdsplat $devnull)]);

$stdread = new Language::INTERCAL::GenericIO('FILE', 'r', '-');
$stdwrite = new Language::INTERCAL::GenericIO('FILE', 'w', '-');
$stdsplat = new Language::INTERCAL::GenericIO('FILE', 'r', '-2');
$devnull = new Language::INTERCAL::GenericIO('TEE', 'r', []);

sub new {
    @_ == 4 or croak
	"Usage: new Language::INTERCAL::GenericIO(TYPE, MODE, DATA)";
    my ($class, $type, $mode, $data) = @_;
    $mode =~ /^[rwa]\+?$/ or croak "Invalid mode \"$mode\"";
    $type = uc($type);
    my $filemode = $mode;
    $filemode =~ tr/rw/wr/;
    my %object = (
	type => $type,
	mode => $mode,
	data => $data,
	buffer => '',
	read_convert => sub { shift },
	write_convert => sub { shift },
	write_charset => 'ASCII',
	text_newline => "\n",
    );
    if ($type eq 'FILE' || $type eq 'UFILE') {
	my $fh;
	if (ref $data && ref $data eq 'GLOB') {
	    $fh = $data;
	} elsif ($data eq '-' || $data eq '-1') {
	    $fh = $mode =~ /r/ ? \*STDOUT : \*STDIN;
	} elsif ($data eq '-2') {
	    $fh = \*STDERR;
	} else {
	    # need absolute paths for use with checkpoint/restart; if
	    # File::Spec is too old and can't do it, tough
	    if (File::Spec->can('rel2abs')) {
		$data = File::Spec->rel2abs($data);
	    }
	    $fh = new IO::File($data, $filemode) or die "$data: $!\n";
	    # $fh->autoflush(1);
	    $object{close_code} = sub { close $fh };
	}
	croak "Cannot open $data: $!" if ! defined $fh;
	if ($type eq 'FILE') {
	    $object{read_code} = sub { print $fh $_[0] };
	    $object{write_code} = sub {
		my ($size) = @_;
		my $b = '';
		read $fh, $b, $size;
		$b;
	    };
	    $object{tell_code} = sub { tell $fh };
	    $object{seek_code} = sub { seek $fh, $_[0], 0 };
	} else {
	    $object{read_code} = sub {
		my ($line) = @_;
		syswrite $fh, $line, length($line);
	    };
	    $object{write_code} = sub {
		my ($size) = @_;
		my $b = '';
		sysread $fh, $b, $size;
		$b;
	    };
	    $object{tell_code} = sub { sysseek $fh, 0, 1 };
	    $object{seek_code} = sub { sysseek $fh, $_[0], 0 };
	}
    } elsif ($type eq 'TEE') {
	$mode =~ /[ar]/ or croak "MODE must be \"read\" when TYPE is TEE";
	ref $data && 'ARRAY' eq ref $data or croak "DATA must be a array ref";
	$object{read_code} = sub {
	    my ($line) = @_;
	    for (@$data) { $_->read_binary($line) }
	};
	# object is not seekable and of course not writable
    } elsif ($type eq 'ARRAY') {
	ref $data && 'ARRAY' eq ref $data or croak "DATA must be a array ref";
	$object{read_code} = sub {
	    my ($line) = @_;
	    push @$data, $line;
	};
	$object{write_code} = sub {
	    my ($size) = @_;
	    return '' unless @$data;
	    my $line = shift @$data;
	    while (@$data && length($line) < $size) {
		$line .= shift @$data;
	    }
	    if (length($line) > $size) {
		unshift @$data, substr($line, $size);
		$line = substr($line, 0, $size);
	    }
	    $line;
	};
	# object is (currently) not seekable
    } elsif ($type eq 'STRING') {
	ref $data && 'SCALAR' eq ref $data or croak "DATA must be a scalar ref";
	$object{read_code} = sub {
	    my ($line) = @_;
	    $$data .= $line;
	};
	$object{write_code} = sub {
	    my ($size) = @_;
	    substr($$data, 0, $size, '');
	};
	# string is (currently) not seekable
    } elsif ($type eq 'OBJECT') {
	ref $data or croak "DATA must be a reference";
	UNIVERSAL::isa($data, 'UNIVERSAL')
	    or croak "DATA must be an object";
	$object{read_code} = sub { $data->read(@_); }
	     if $data->can('read');
	$object{write_code} = sub { $data->write(@_); }
	     if $data->can('write');
	# object is (currently) not seekable
    } elsif ($type eq 'COUNT') {
	$mode =~ /[ar]/ or croak "MODE must be \"read\" when TYPE is COUNT";
	ref $data && 'SCALAR' eq ref $data or croak "DATA must be a scalar ref";
	$object{read_code} = sub { $$data += length($_[0]) };
	# object is (currently) not seekable
    } else {
	# TODO (1.-90) - $type eq 'LECTURE'
	croak "Invalid type \"$type\"";
    }
    bless \%object, $class;
}

sub DESTROY {
    my ($fh) = @_;
    &{$fh->{close_code}} if exists $fh->{close_code};
}

sub can_tell {
    @_ == 1 or croak "Usage: IO->can_tell";
    my ($fh) = @_;
    return exists $fh->{tell_code};
}

sub tell {
    @_ == 1 or croak "Usage: IO->tell";
    my ($fh) = @_;
    croak "Non seekable" unless exists $fh->{tell_code};
    &{$fh->{tell_code}};
}

sub can_seek {
    @_ == 1 or croak "Usage: IO->can_seek";
    my ($fh) = @_;
    return exists $fh->{seek_code};
}

sub reset {
    @_ == 1 or croak "Usage: IO->reset";
    my ($fh) = @_;
    croak "Non seekable" unless exists $fh->{seek_code};
    &{$fh->{seek_code}}(0);
    $fh->{buffer} = '';
    $fh;
}

sub seek {
    @_ == 2 or croak "Usage: IO->seek(POS)";
    my ($fh, $pos) = @_;
    croak "Non seekable" unless exists $fh->{seek_code};
    &{$fh->{seek_code}}($pos);
    $fh->{buffer} = '';
    $fh;
}

sub data {
    @_ == 1 or croak "Usage: IO->data";
    my ($fh) = @_;
    return $fh->{data};
}

sub can_read {
    @_ == 1 or croak "Usage: IO->can_read";
    my ($fh) = @_;
    return exists $fh->{read_code};
}

sub read_binary {
    @_ == 2 or croak "Usage: IO->read_binary(DATA)";
    my ($fh, $string) = @_;
    croak "Filehandle not open for reading" if ! exists $fh->{read_code};
    &{$fh->{read_code}}($string);
    $fh;
}

sub read_text {
    @_ == 2 or croak "Usage: IO->read_text(DATA)";
    my ($fh, $string) = @_;
    croak "Filehandle not open for reading" if ! exists $fh->{read_code};
    croak "Filehandle not set up for text reading"
	if ! exists $fh->{read_convert};
    $string = &{$fh->{read_convert}}($string);
    &{$fh->{read_code}}($string);
    $fh;
}

sub read_charset {
    @_ == 2 or croak "Usage: IO->read_charset(CHARSET)";
    my ($fh, $charset) = @_;
    $fh->{read_convert} = fromascii($charset);
    $fh;
}

sub describe {
    @_ == 1 or croak "Usage: IO->describe";
    my ($fh) = @_;
    my $type = $fh->{type};
    my $mode = $fh->{mode};
    my $data = $fh->{data};
    if ($type eq 'FILE' || $type eq 'UFILE') {
	return "FILE($mode, $data)";
    } elsif ($type eq 'TEE') {
	return "TEE(" . join(',', map { describe($_) } @$data) . ")";
    } elsif ($type eq 'ARRAY' || $type eq 'STRING' || $type eq 'OBJECT') {
	return $type;
    } elsif ($type eq 'COUNT') {
	return "COUNT($$data)";
    }
    # TODO (1.-90) - $type eq 'LECTURE'
    $fh;
}

sub can_write {
    @_ == 1 or croak "Usage: IO->can_write";
    my ($fh) = @_;
    return exists $fh->{write_code};
}

sub write_binary {
    @_ == 2 or croak "Usage: IO->write_binary(SIZE)";
    my ($fh, $size) = @_;
    croak "Filehandle not open for writing"
	if ! exists $fh->{write_code};
confess "size is undef" if ! defined $size;
    if (length($fh->{buffer}) >= $size) {
	return substr($fh->{buffer}, 0, $size, '');
    }
    my $data = '';
    if ($fh->{buffer} ne '') {
	$data = $fh->{buffer};
	$fh->{buffer} = '';
    }
    $data . &{$fh->{write_code}}($size - length($data));
}

sub write_text {
    @_ == 1 or @_ == 2 or croak "Usage: IO->write_text [(NEWLINE)]";
    my ($fh, $newline) = @_;
    croak "Filehandle not open for writing"
	if ! exists $fh->{write_code};
    croak "Filehandle not open for writing text"
	if ! exists $fh->{write_convert};
    if (defined $newline) {
	if ($newline ne '') {
	    eval { $newline = &{ fromascii($fh->{write_charset}) }($newline) };
	    $newline = "\n" if $@;
	}
    } else {
	$newline = $fh->{text_newline};
    }
    my $nlpos = $newline eq '' ? -1 : index $fh->{buffer}, $newline;
    # we must read one at a time, even though it's painfully slow,
    # otherwise we may have the user typing one line and not knowing
    # why the program is stopped dead.
    while ($nlpos < 0) {
	my $data = &{$fh->{write_code}}(1);
	last if $data eq '';
	$fh->{buffer} .= $data;
	$nlpos = $newline eq '' ? -1 : index $fh->{buffer}, $newline;
    }
    if ($nlpos < 0) {
	$nlpos = length($fh->{buffer});
    } else {
	$nlpos += length($newline);
    }
    my $line = substr($fh->{buffer}, 0, $nlpos, '');
    &{$fh->{write_convert}}($line);
}

sub write_charset {
    @_ == 2 or croak "Usage: IO->write_charset(CHARSET)";
    my ($fh, $charset) = @_;
    $fh->{write_charset} = $charset;
    $fh->{write_convert} = toascii($charset);
    eval { $fh->{text_newline} = &{ fromascii($charset) }("\n") };
    $fh->{text_newline} = "\n" if $@;
    $fh;
}

1;
