############################################################################
# Usage:
#  SIM_AC_CHECK_OPENGL([ACTION-IF-FOUND [, ACTION-IF-NOT-FOUND]])
#
#  Try to find an OpenGL development system, either a native
#  implementation or the OpenGL-compatible Mesa library. If
#  it is found, these shell variables are set:
#
#    $sim_ac_gl_cppflags (extra flags the preprocessor needs for OpenGL/Mesa)
#    $sim_ac_gl_ldflags  (extra flags the linker needs for OpenGL/Mesa)
#    $sim_ac_gl_libs     (link libraries the linker needs for OpenGL/Mesa)
#
#  In addition, the variable $sim_ac_gl_avail is set to "yes" if an
#  OpenGL-compatible development system is found.
#
# TODO:
#   these macros are ripe for a complete rewrite...
#
# Authors:
#   Morten Eriksen <mortene@sim.no>
#   Lars J. Aas <larsa@sim.no>

# **************************************************************************

AC_DEFUN([SIM_AC_CHECK_HEADERS_GL],
[sim_ac_gl_header_avail=false
if test x"$with_opengl" != xno; then
  sim_ac_gl_save_CPPFLAGS=$CPPFLAGS
  if test x"$with_opengl" != xyes; then
    sim_ac_gl_cppflags="-I${with_opengl}/include"
    CPPFLAGS="$CPPFLAGS $sim_ac_gl_cppflags"
  fi

  AC_CHECK_HEADER([GL/gl.h], [
    sim_ac_gl_header_avail=true
    AC_DEFINE([HAVE_GL_GL_H],1 , [define if the GL header should be included as GL/gl.h])
  ], [
    AC_CHECK_HEADER([OpenGL/gl.h], [
      sim_ac_gl_header_avail=true
      AC_DEFINE([HAVE_OPENGL_GL_H],1 , [define if the GL header should be included as OpenGL/gl.h])
    ])
  ])

  sim_ac_gl_hpux=/opt/graphics/OpenGL
  if test x$sim_ac_gl_header_avail = xfalse && test -d $sim_ac_gl_hpux; then
    sim_ac_gl_cppflags=-I$sim_ac_gl_hpux/include
    CPPFLAGS="$CPPFLAGS $sim_ac_gl_cppflags"
    AC_CHECK_HEADER([GL/gl.h], [
      sim_ac_gl_header_avail=true
      AC_DEFINE([HAVE_GL_GL_H],1 , [define if the GL header should be included as GL/gl.h])
    ], [
      AC_CHECK_HEADER([OpenGL/gl.h], [
        sim_ac_gl_header_avail=true
        AC_DEFINE([HAVE_OPENGL_GL_H],1 , [define if the GL header should be included as OpenGL/gl.h])
      ])
    ])
  fi

  CPPFLAGS="$sim_ac_gl_save_CPPFLAGS"
fi

if $sim_ac_gl_header_avail; then
  :
  $1
else
  :
  $2
fi])

# **************************************************************************

AC_DEFUN(SIM_AC_CHECK_OPENGL, [

sim_ac_gl_cppflags=
sim_ac_gl_ldflags=
sim_ac_gl_libs=
sim_ac_gl_avail=no

AC_ARG_WITH(
  [mesa],
  AC_HELP_STRING([--with-mesa],
                 [prefer MesaGL (if found) over OpenGL [[default=yes]]]),
  [],
  [with_mesa=yes])

## Mac OS X uses some weird "framework" options.

## It's usually libGL.so on UNIX systems and opengl32.lib on MSWindows.
sim_ac_gl_glnames="-lGL -lopengl32"
sim_ac_gl_mesaglnames=-lMesaGL

if test "x$with_mesa" = "xyes"; then
  sim_ac_gl_first=$sim_ac_gl_mesaglnames
  sim_ac_gl_second=$sim_ac_gl_glnames
else
  sim_ac_gl_first=$sim_ac_gl_glnames
  sim_ac_gl_second=$sim_ac_gl_mesaglnames
fi

AC_ARG_WITH(
  [opengl],
  AC_HELP_STRING([--with-opengl=DIR],
                 [OpenGL/Mesa installation directory]),
  [],
  [with_opengl=yes])

if test x"$with_opengl" != xno; then
  sim_ac_use_framework_option=false;
  case $host_os in
  darwin*)
    if test x"$GCC" = x"yes"; then
      SIM_AC_CC_COMPILER_OPTION([-framework OpenGL], [sim_ac_use_framework_option=true])
    fi
    ;;
  esac

  if test x"$with_opengl" != xyes; then
    sim_ac_gl_cppflags="-I${with_opengl}/include"
    sim_ac_gl_ldflags="-L${with_opengl}/lib"
  else
    if $sim_ac_use_framework_option; then
      # hopefully, this is the default behavior and not needed. 20011005 larsa
      # sim_ac_gl_cppflags="-F/System/Library/Frameworks/OpenGL.framework/"
      sim_ac_gl_ldflags="-Wl,-framework,OpenGL"
    else
      ## This is a common location for the OpenGL library on HPUX.
      sim_ac_gl_hpux=/opt/graphics/OpenGL
      if test -d $sim_ac_gl_hpux; then
        sim_ac_gl_cppflags=-I$sim_ac_gl_hpux/include
        sim_ac_gl_ldflags=-L$sim_ac_gl_hpux/lib
      fi
    fi
  fi

  sim_ac_save_cppflags=$CPPFLAGS
  sim_ac_save_ldflags=$LDFLAGS
  sim_ac_save_libs=$LIBS

  CPPFLAGS="$CPPFLAGS $sim_ac_gl_cppflags"
  LDFLAGS="$LDFLAGS $sim_ac_gl_ldflags"

  ## This must be done after include-paths have been set up for CPPFLAGS.
  AC_CHECK_HEADERS([GL/gl.h OpenGL/gl.h])

  AC_CACHE_CHECK(
    [whether OpenGL library is available],
    sim_cv_lib_gl,
    [sim_cv_lib_gl=UNRESOLVED

    # Mac OS X uses nada, which is why "" was set first
    for sim_ac_gl_libcheck in "" $sim_ac_gl_first $sim_ac_gl_second; do
      if test "x$sim_cv_lib_gl" = "xUNRESOLVED"; then
        LIBS="$sim_ac_gl_libcheck $sim_ac_save_libs"
        AC_TRY_LINK([
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#endif /* HAVE_WINDOWS_H */
#ifdef HAVE_GL_GL_H
#include <GL/gl.h>
#else
#ifdef HAVE_OPENGL_GL_H
/* Mac OS X */
#include <OpenGL/gl.h>
#endif
#endif
],
                    [
glPointSize(1.0f);
], [
          if test x"$sim_ac_gl_libcheck" = x""; then
            sim_cv_lib_gl="$sim_ac_gl_ldflags"
          else
            sim_cv_lib_gl="$sim_ac_gl_libcheck"
          fi])
      fi
    done
  ])

  LIBS="$sim_ac_save_libs"

  case $sim_cv_lib_gl in
  -Wl,-framework,OpenGL)
    sim_ac_gl_libs=
    sim_ac_gl_ldflags="$sim_cv_lib_gl"
    ;;
  -l*)
    sim_ac_gl_libs="$sim_cv_lib_gl"
    ;;
  *)
    AC_MSG_WARN([couldn't compile or link with OpenGL library -- trying with pthread library in place...])

    SIM_AC_CHECK_PTHREAD([
      sim_ac_gl_cppflags="$sim_ac_gl_cppflags $sim_ac_pthread_cppflags"
      sim_ac_gl_ldflags="$sim_ac_gl_ldflags $sim_ac_pthread_ldflags"],
      [AC_MSG_WARN([couldn't compile or link with pthread library])])

    if test "x$sim_ac_pthread_avail" = "xyes"; then
      AC_CACHE_CHECK(
        [whether OpenGL library can be linked with pthread library],
        sim_cv_lib_gl_pthread,
        [sim_cv_lib_gl_pthread=UNRESOLVED

        for sim_ac_gl_libcheck in $sim_ac_gl_first $sim_ac_gl_second; do
          if test "x$sim_cv_lib_gl_pthread" = "xUNRESOLVED"; then
            LIBS="$sim_ac_gl_libcheck $sim_ac_pthread_libs $sim_ac_save_libs"
            AC_TRY_LINK([
#ifdef HAVE_GL_GL_H
#include <GL/gl.h>
#else
#ifdef HAVE_OPENGL_GL_H
#include <OpenGL/gl.h>
#endif
#endif
],
                        [
glPointSize(1.0f);
],
                        [sim_cv_lib_gl_pthread="$sim_ac_gl_libcheck"])
          fi
        done
      ])

      if test "x$sim_cv_lib_gl_pthread" != "xUNRESOLVED"; then
        sim_ac_gl_libs="$sim_cv_lib_gl_pthread $sim_ac_pthread_libs"
      fi
    fi
    ;;
  esac


  # MacOS will have empty sim_ac_gl_libs, so don't check if it is empty...
  if test x"$sim_cv_gl_libs" != x"UNRESOLVED"; then
    sim_ac_gl_avail=yes
    $1
  else
    CPPFLAGS=$sim_ac_save_cppflags
    LDFLAGS=$sim_ac_save_ldflags
    LIBS=$sim_ac_save_libs
    $2
  fi
fi
])


# **************************************************************************
# SIM_AC_GLU_READY_IFELSE( [ACTION-IF-TRUE], [ACTION-IF-FALSE] )

AC_DEFUN([SIM_AC_GLU_READY_IFELSE], [
AC_CHECK_HEADERS([GL/glu.h OpenGL/glu.h])
AC_CACHE_CHECK(
  [if GLU is available as part of GL library],
  [sim_cv_glu_ready],
  [AC_TRY_LINK(
    [
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#endif /* HAVE_WINDOWS_H */
#ifdef HAVE_GL_GL_H
#include <GL/gl.h>
#else
#ifdef HAVE_OPENGL_GL_H
#include <OpenGL/gl.h>
#endif
#endif
#ifdef HAVE_GL_GLU_H
#include <GL/glu.h>
#else
#ifdef HAVE_OPENGL_GLU_H
#include <OpenGL/glu.h>
#endif
#endif
],
    [
gluSphere(0L, 1.0, 1, 1);
/* Defect JAGad01283 of HP's aCC compiler causes a link failure unless
   there is at least one "pure" OpenGL call along with GLU calls. */
glEnd();
],
    [sim_cv_glu_ready=true],
    [sim_cv_glu_ready=false])])
if ${sim_cv_glu_ready}; then
  ifelse([$1], , :, [$1])
else
  ifelse([$2], , :, [$2])
fi
]) # SIM_AC_GLU_READY_IFELSE()


############################################################################
# Usage:
#  SIM_AC_CHECK_GLU([ACTION-IF-FOUND [, ACTION-IF-NOT-FOUND]])
#
#  Try to use the OpenGL utility library; GLU. If it is found,
#  these shell variables are set:
#
#    $sim_ac_glu_cppflags (extra flags the compiler needs for GLU)
#    $sim_ac_glu_ldflags  (extra flags the linker needs for GLU)
#    $sim_ac_glu_libs     (link libraries the linker needs for GLU)
#
#  The CPPFLAGS, LDFLAGS and LIBS flags will also be modified accordingly.
#  In addition, the variable $sim_ac_gly_avail is set to "yes" if GLU
#  is found.
#
#
# Author: Morten Eriksen, <mortene@sim.no>.

AC_DEFUN([SIM_AC_CHECK_GLU], [
AC_CHECK_HEADERS([GL/glu.h OpenGL/glu.h])

unset sim_ac_glu_cppflags
unset sim_ac_glu_ldflags
unset sim_ac_glu_libs
sim_ac_glu_avail=no

# It's usually libGLU.so on UNIX systems and glu32.lib on MSWindows.
sim_ac_glu_names="-lGLU -lglu32"
sim_ac_glu_mesanames=-lMesaGLU

# with_mesa is set from the SIM_AC_CHECK_OPENGL macro.
if test "x$with_mesa" = "xyes"; then
  sim_ac_glu_first=$sim_ac_glu_mesanames
  sim_ac_glu_second=$sim_ac_glu_names
else
  sim_ac_glu_first=$sim_ac_glu_names
  sim_ac_glu_second=$sim_ac_glu_mesanames
fi

AC_ARG_WITH(
  [glu],
  AC_HELP_STRING([--with-glu=DIR],
                 [use the OpenGL utility library [[default=yes]]]),
  [],
  [with_glu=yes])

if test x"$with_glu" != xno; then
  if test x"$with_glu" != xyes; then
    sim_ac_glu_cppflags="-I${with_glu}/include"
    sim_ac_glu_ldflags="-L${with_glu}/lib"
  fi

  sim_ac_save_cppflags=$CPPFLAGS
  sim_ac_save_ldflags=$LDFLAGS
  sim_ac_save_libs=$LIBS

  CPPFLAGS="$CPPFLAGS $sim_ac_glu_cppflags"
  LDFLAGS="$LDFLAGS $sim_ac_glu_ldflags"

  AC_CACHE_CHECK(
    [whether GLU is available],
    sim_cv_lib_glu,
    [sim_cv_lib_glu=UNRESOLVED

    # Some platforms (like BeOS) have the GLU functionality in the GL
    # library (and no GLU library present).
    for sim_ac_glu_libcheck in "" $sim_ac_glu_first $sim_ac_glu_second; do
      if test "x$sim_cv_lib_glu" = "xUNRESOLVED"; then
        LIBS="$sim_ac_glu_libcheck $sim_ac_save_libs"
        AC_TRY_LINK([
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#endif /* HAVE_WINDOWS_H */
#ifdef HAVE_GL_GL_H
#include <GL/gl.h>
#else
#ifdef HAVE_OPENGL_GL_H
#include <OpenGL/gl.h>
#endif
#endif
#ifdef HAVE_GL_GLU_H
#include <GL/glu.h>
#else
#ifdef HAVE_OPENGL_GLU_H
#include <OpenGL/glu.h>
#endif
#endif
],
                    [
gluSphere(0L, 1.0, 1, 1);
/* Defect JAGad01283 of HP's aCC compiler causes a link failure unless
   there is at least one "pure" OpenGL call along with GLU calls. */
glEnd();
],
                    [sim_cv_lib_glu="$sim_ac_glu_libcheck"])
      fi
    done
    if test x"$sim_cv_lib_glu" = x"" &&
       test x`echo $LDFLAGS | grep -c -- "-Wl,-framework,OpenGL"` = x1; then
      # just for the visual representation on Mac OS X
      sim_cv_lib_glu="-Wl,-framework,OpenGL"
    fi
  ])

  LIBS="$sim_ac_save_libs"

  if test "x$sim_cv_lib_glu" != "xUNRESOLVED"; then
    if test x"$sim_cv_lib_glu" = x"-Wl,-framework,OpenGL"; then
      sim_ac_glu_libs=""
    else
      sim_ac_glu_libs="$sim_cv_lib_glu"
    fi
    LIBS="$sim_ac_glu_libs $sim_ac_save_libs"
    sim_ac_glu_avail=yes
    $1
  else
    CPPFLAGS=$sim_ac_save_cppflags
    LDFLAGS=$sim_ac_save_ldflags
    LIBS=$sim_ac_save_libs
    $2
  fi
fi
])


############################################################################
# Usage:
#  SIM_AC_GLU_NURBSOBJECT([ACTION-IF-FOUND [, ACTION-IF-NOT-FOUND]])
#
#  Try to find out whether the interface struct against the GLU
#  library NURBS functions is called "GLUnurbs" or "GLUnurbsObj".
#  (This seems to have changed somewhere between release 1.1 and
#  release 1.3 of GLU).
#
#  The variable $sim_ac_glu_nurbsobject is set to the correct name
#  if the nurbs structure is found.
#
# Author: Morten Eriksen, <mortene@sim.no>.

AC_DEFUN(SIM_AC_GLU_NURBSOBJECT, [
AC_CACHE_CHECK(
  [what structure to use in the GLU NURBS interface],
  sim_cv_func_glu_nurbsobject,
  [sim_cv_func_glu_nurbsobject=NONE
   for sim_ac_glu_structname in GLUnurbs GLUnurbsObj; do
    if test "$sim_cv_func_glu_nurbsobject" = NONE; then
      AC_TRY_LINK([
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#endif /* HAVE_WINDOWS_H */
#ifdef HAVE_GL_GL_H
#include <GL/gl.h>
#else
#ifdef HAVE_OPENGL_GL_H
#include <OpenGL/gl.h>
#endif
#endif
#ifdef HAVE_GL_GLU_H
#include <GL/glu.h>
#else
#ifdef HAVE_OPENGL_GLU_H
#include <OpenGL/glu.h>
#endif
#endif
],
                  [
$sim_ac_glu_structname * hepp = gluNewNurbsRenderer();
gluDeleteNurbsRenderer(hepp);
/* Defect JAGad01283 of HP's aCC compiler causes a link failure unless
   there is at least one "pure" OpenGL call along with GLU calls. */
glEnd();
],
                  [sim_cv_func_glu_nurbsobject=$sim_ac_glu_structname])
    fi
  done
])

if test $sim_cv_func_glu_nurbsobject = NONE; then
  sim_ac_glu_nurbsobject=
  $2
else
  sim_ac_glu_nurbsobject=$sim_cv_func_glu_nurbsobject
  $1
fi
])

# **************************************************************************
# SIM_AC_HAVE_GLX_IFELSE( IF-FOUND, IF-NOT-FOUND )
#
# Check whether GLX is on the system.

AC_DEFUN([SIM_AC_HAVE_GLX_IFELSE], [
AC_CACHE_CHECK(
  [whether GLX is on the system],
  sim_cv_have_glx,
  AC_TRY_LINK(
    [
#include <GL/glx.h>
#include <GL/gl.h>
],
    [
(void)glXChooseVisual(0L, 0, 0L);
/* Defect JAGad01283 of HP's aCC compiler causes a link failure unless
   there is at least one "pure" OpenGL call along with GLU calls. */
glEnd();
],
    [sim_cv_have_glx=true],
    [sim_cv_have_glx=false]))

if ${sim_cv_have_glx=false}; then
  ifelse([$1], , :, [$1])
else
  ifelse([$2], , :, [$2])
fi
]) # SIM_AC_HAVE_GLX_IFELSE()


# **************************************************************************
# SIM_AC_HAVE_WGL_IFELSE( IF-FOUND, IF-NOT-FOUND )
#
# Check whether WGL is on the system.
#
# This macro has one important side-effect: the variable
# sim_ac_wgl_libs will be set to the list of libraries
# needed to link with wgl*() functions.

AC_DEFUN([SIM_AC_HAVE_WGL_IFELSE], [
sim_ac_save_libs=$LIBS
## Not directly needed by the wgl*() calls, but to create a
## context we need functions from this library.
sim_ac_wgl_libs="-lgdi32"
LIBS="$LIBS $sim_ac_wgl_libs"

AC_CACHE_CHECK(
  [whether WGL is on the system],
  sim_cv_have_wgl,
  AC_TRY_LINK(
    [
#include <windows.h>
#include <GL/gl.h>
],
    [(void)wglCreateContext(0L);],
    [sim_cv_have_wgl=true],
    [sim_cv_have_wgl=false]))

LIBS=$sim_ac_save_libs
if ${sim_cv_have_wgl=false}; then
  ifelse([$1], , :, [$1])
else
  ifelse([$2], , :, [$2])
fi
]) # SIM_AC_HAVE_WGL_IFELSE()
