/*
 * mini_map.c
 *
 * crafted - a pud editor for the freecraft project.
 * 
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "crafted.h"
#include "craftedintl.h"

#include "mini_map.h"


static GtkWidget *minimap_preview = NULL;
static gint minimap_size = 128;

/*************************************************************************
*  update_minimap
**************************************************************************/
void update_minimap(void)
{
  guchar  *buf = NULL;
  gint     i, j, ip, jp;
  gushort  tile;
  guchar   color[3];

  if (!minimap_preview)
    return;
  
  buf = g_malloc0(minimap_size * minimap_size * 3);

  for (i = 0 ; i < minimap_size ; i++)
    for (j = 0 ; j < minimap_size ; j++)
    {
      ip = (i*Pud.width)/minimap_size;
      jp = (j*Pud.width)/minimap_size;
      tile = Pud.fields[ip*Pud.width+jp];
      switch (tile >> 8)
      {
        case 0: /* plain tile */
          switch (tile >> 4)
          {
            case 1: /* light water */
              color[0] = 0; color[1] = 128; color[2] = 255;
              break;

            case 2: /* dark water */
              color[0] = 0; color[1] = 0; color[2] = 255;
              break;

            case 3: /* light coast */
              color[0] = 187; color[1] = 140; color[2] = 102;
              break;

            case 4: /* dark coast */
              color[0] = 121; color[1] = 75; color[2] = 38;
              break;

            case 5: /* light grass */
              color[0] = 32; color[1] = 213; color[2] = 41;
              break;

            case 6: /* dark grass */
              color[0] = 23; color[1] = 174; color[2] = 30;
              break;

            case 7: /* forest */
              color[0] = 55; color[1] = 101; color[2] = 57;
              break;

            case 8: /* rocks */
              color[0] = 192; color[1] = 192; color[2] = 192;
              break;

            default:
              g_print("FIXME!!!\ttile:%X\n", tile);
          }
          break;

        case 1: /* dark water/light water */
          color[0] = 0; color[1] = 128; color[2] = 255;
          break;

        case 2: /* light water/light coast */
          color[0] = 187; color[1] = 140; color[2] = 102;
          break;

        case 3: /* dark coast/light coast */
          color[0] = 121; color[1] = 75; color[2] = 38;
          break;

        case 4: /* rocks/light coast */
          color[0] = 192; color[1] = 192; color[2] = 192;
          break;

        case 5: /* light coast/light grass */
          color[0] = 32; color[1] = 213; color[2] = 41;
          break;

        case 6: /* dark grass/light grass */
          color[0] = 23; color[1] = 174; color[2] = 30;
          break;

        case 7: /* forest/light grass */
          color[0] = 55; color[1] = 101; color[2] = 57;
          break;

        default:
          g_print("FIXME!!!\ttile:%X\n", tile);

      }
      buf[3*(i*minimap_size+j)+0] = color[0];
      buf[3*(i*minimap_size+j)+1] = color[1];
      buf[3*(i*minimap_size+j)+2] = color[2];
    }
  for (i = 0;  i < minimap_size ; i++)
  {
    gtk_preview_draw_row(GTK_PREVIEW(minimap_preview), &buf[3*i*minimap_size],
                         0, i, minimap_size);
  }
  gtk_widget_queue_draw(minimap_preview);
  g_free(buf);
}

/*************************************************************************
*  minimap_size_changed
**************************************************************************/
static void minimap_size_changed(GtkAdjustment *adj)
{
  minimap_size = adj->value / 32;
  minimap_size *= 32;
  gtk_preview_size(GTK_PREVIEW(minimap_preview), minimap_size, minimap_size);
  gtk_widget_queue_resize(minimap_preview);
  update_minimap();
}

/*************************************************************************
*  click_on_minimap
**************************************************************************/
static gboolean click_on_minimap(GtkWidget *preview, GdkEventButton *ev)
{
  GtkAdjustment *adj;
  gint           new_off;

  adj = gtk_range_get_adjustment(GTK_RANGE(hscroll));
  new_off = ev->x*32*Pud.width/minimap_size-adj->page_size/2;
  new_off = CLAMP(new_off, 0, adj->upper - adj->page_size);
  gtk_adjustment_set_value(adj, new_off);

  adj = gtk_range_get_adjustment(GTK_RANGE(vscroll));
  new_off = ev->y*32*Pud.height/minimap_size-adj->page_size/2;
  new_off = CLAMP(new_off, 0, adj->upper - adj->page_size);
  gtk_adjustment_set_value(adj, new_off);

  return FALSE;
}

/*************************************************************************
*  show_mini_map
**************************************************************************/
void show_mini_map(void)
{
  static GtkWidget *mini_map = NULL;
  GtkWidget     *main_vbox, *button, *label, *frame, *hbox, *scale, *scr;
  GtkAdjustment *adj;
  guchar        *buf;  
  gint           i, j;

  if (mini_map)
  {
    gtk_widget_show(mini_map);
    return; /* FIXME show or raise */
  }

  if (Pud.is_loaded)
  {
    minimap_size = 128;
    mini_map = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_signal_connect(GTK_OBJECT(mini_map), "destroy",
                       (GtkSignalFunc)gtk_widget_destroyed,
                       &mini_map);
    gtk_signal_connect(GTK_OBJECT(mini_map), "delete_event",
                       (GtkSignalFunc)gtk_widget_hide, NULL);

    main_vbox = gtk_vbox_new(FALSE,0);
    gtk_container_border_width(GTK_CONTAINER(main_vbox), 1);
    gtk_container_add(GTK_CONTAINER(mini_map), main_vbox);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_label_align(GTK_FRAME(frame), 0.5, 0.5);
    gtk_box_pack_start(GTK_BOX(main_vbox), frame, TRUE, TRUE, 3); 
  
    scr = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_set_usize(scr, 140, 140);
    gtk_container_add(GTK_CONTAINER(frame), scr);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scr),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    minimap_preview = gtk_preview_new(GTK_PREVIEW_COLOR);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scr), minimap_preview);
    gtk_preview_size(GTK_PREVIEW(minimap_preview),
                     minimap_size, minimap_size);
    gtk_widget_add_events(minimap_preview, GDK_BUTTON_PRESS_MASK);
    gtk_signal_connect(GTK_OBJECT(minimap_preview), "button_press_event",
                       (GtkSignalFunc)click_on_minimap, NULL);
 
    /* a scale to change the size of the mini_map */
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
    label = gtk_label_new("size: ");
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    adj = (GtkAdjustment *)gtk_adjustment_new(minimap_size, 32, 1024, 32, 256, 0);
    gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                       (GtkSignalFunc)minimap_size_changed, NULL);
    scale = gtk_hscale_new(adj);
    gtk_scale_set_draw_value(GTK_SCALE(scale), FALSE);
    gtk_box_pack_start_defaults(GTK_BOX(hbox), scale);

    update_minimap();

  } else
  {
    GtkWidget *dialog, *label, *button;

    dialog = gtk_dialog_new();
    gtk_widget_show(dialog);
    label = gtk_label_new(_("No map loaded,\n  no mini map available  "));
    gtk_widget_show(label);
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), label);
    button = gtk_button_new_with_label(" OK ");
    gtk_widget_show(button);
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			      (GtkSignalFunc)gtk_widget_destroy, 
			      GTK_OBJECT(dialog));
    gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->action_area), 
				button);
  }
  gtk_widget_show_all(mini_map);
}

